#replication code for "Expected market returns:SVIX, realized volatility, and the role of dividends" 
#Matthijs Lof
#January 7th, 2019

rm(list=ls())

library(xts)
library(vars)
library(sandwich)
library(reshape)
library(xtable)
library(car)



EWMA <- function(x,lambda)
{
  Tt <- length(x)
  y  <- rep(NA,Tt)
  y[1:20] <- var(na.omit(x[1:20]))
  for (t in 21:Tt){y[t] <- ifelse(is.na(x[t]),y[t-1],lambda*y[t-1]+(1-lambda)*x[t]^2)} 
  return(y)
}


#load data
sp500.data   <- read.csv("SP500.csv", header = TRUE,sep=";",dec=".")
sp500TR.data <- read.csv("SP500TR.csv", header = TRUE,sep=",",dec=".")
CRSP.data    <- read.csv("CRSP.csv", header = TRUE,sep=";",dec=".")
rf.data      <- read.csv("RF.csv", header = TRUE,sep=";",dec=".")
svix.data    <- read.csv("SVIX.csv", header = TRUE,sep=";",dec=".")
vix.data     <- read.csv("VIX.csv", header = TRUE,sep=",",dec=".")



#convert to daily xts
sp500.D   <- xts(sp500.data[,-1], order.by=as.Date(sp500.data$Date, '%d.%m.%Y'))
sp500TR.D <- xts(sp500TR.data[,-1], order.by=as.Date(sp500TR.data$Date, '%Y-%m-%d'))
CRSP.D    <- xts(CRSP.data[,-1], order.by=as.Date(CRSP.data$date, '%d.%m.%Y'))
rf.D      <- xts(rf.data[,-1], order.by=as.Date(rf.data$date, '%d.%m.%Y'))
svix.D    <- xts(svix.data[,-1], order.by=as.Date(svix.data$date, '%d.%m.%Y'))
vix.D     <- xts(vix.data[,-1], order.by=as.Date(vix.data$Date, '%Y-%m-%d'))



#add risk-free rates to stock returns
sp500.D$rf    <- rf.D[index(sp500.D)]
sp500.D$rf    <- na.locf(sp500.D$rf)
sp500TR.D$rf  <- rf.D[index(sp500TR.D)]
sp500TR.D$rf  <- na.locf(sp500TR.D$rf)
CRSP.D$rf     <- rf.D[index(CRSP.D)]
CRSP.D$rf     <- na.locf(CRSP.D$rf)

#Daily returns
sp500.D$R    <- sp500.D$Close/lag(sp500.D$Close,1)-1
sp500.D$Re   <- sp500.D$R-sp500.D$rf

sp500TR.D$R  <- sp500TR.D$Close/lag(sp500TR.D$Close,1)-1
sp500TR.D$Re <- sp500TR.D$R-sp500TR.D$rf

CRSP.D$R     <- CRSP.D$vwretd
CRSP.D$Re    <- CRSP.D$vwretd-CRSP.D$rf

CRSP.D$Rx     <- CRSP.D$vwretx
CRSP.D$Rxe    <- CRSP.D$vwretx-CRSP.D$rf

#Daily ewma
decay <- 0.94
sp500.D$ewma   <- EWMA(sp500.D$R,decay)
sp500TR.D$ewma <- EWMA(sp500TR.D$R,decay)
CRSP.D$ewma    <- EWMA(CRSP.D$R,decay)
CRSP.D$ewmax   <- EWMA(CRSP.D$Rx,decay)


#monthly returns
sp500.M     <- exp(apply.monthly(log(1+sp500.D$Re),sum,na.rm=TRUE))-1
sp500.M$rf  <- exp(apply.monthly(log(1+sp500.D$rf),sum,na.rm=TRUE))-1

sp500TR.M    <- exp(apply.monthly(log(1+sp500TR.D$Re),sum,na.rm=TRUE))-1 
sp500TR.M$rf <- exp(apply.monthly(log(1+sp500TR.D$rf),sum,na.rm=TRUE))-1 

CRSP.M      <- exp(apply.monthly(log(1+CRSP.D$Re),sum,na.rm=TRUE))-1
CRSP.M$rf   <- exp(apply.monthly(log(1+CRSP.D$rf),sum,na.rm=TRUE))-1
CRSP.M$Rxe  <- exp(apply.monthly(log(1+CRSP.D$Rxe),sum,na.rm=TRUE))-1

svix.M      <- apply.monthly(svix.D,tail,1)

vix.M       <- apply.monthly(vix.D,tail,1)


#monthly realized variance
sp500.M$var     <- apply.monthly(sp500.D$R,var,na.rm=TRUE)*240 
sp500TR.M$var   <- apply.monthly(sp500TR.D$R,var,na.rm=TRUE)*240 
CRSP.M$var      <- apply.monthly(CRSP.D$R,var,na.rm=TRUE)*240
CRSP.M$varx      <- apply.monthly(CRSP.D$Rx,var,na.rm=TRUE)*240



#Monthly ewma
sp500.M$ewma     <- apply.monthly(sp500.D$ewma,tail,1)*240 
sp500TR.M$ewma   <- apply.monthly(sp500TR.D$ewma,tail,1)*240 
CRSP.M$ewma      <- apply.monthly(CRSP.D$ewma,tail,1)*240 
CRSP.M$ewmax      <- apply.monthly(CRSP.D$ewmax,tail,1)*240 


#cumulative forward looking returns
sp500.M$R1  <- 12*(lag(sp500.M$Re,-1))
sp500.M$R2  <- 6*(lag(exp(rollsum(log(1+sp500.M$Re),2,align="left",fill = NA)),-1)-1)
sp500.M$R3  <- 4*(lag(exp(rollsum(log(1+sp500.M$Re),3,align="left",fill = NA)),-1)-1)
sp500.M$R6  <- 2*(lag(exp(rollsum(log(1+sp500.M$Re),6,align="left",fill = NA)),-1)-1)
sp500.M$R12 <- (lag(exp(rollsum(log(1+sp500.M$Re),12,align="left",fill = NA)),-1)-1)

sp500TR.M$R1  <- 12*(lag(sp500TR.M$Re,-1))
sp500TR.M$R2  <- 6*(lag(exp(rollsum(log(1+sp500TR.M$Re),2,align="left",fill = NA)),-1)-1)
sp500TR.M$R3  <- 4*(lag(exp(rollsum(log(1+sp500TR.M$Re),3,align="left",fill = NA)),-1)-1)
sp500TR.M$R6  <- 2*(lag(exp(rollsum(log(1+sp500TR.M$Re),6,align="left",fill = NA)),-1)-1)
sp500TR.M$R12 <- (lag(exp(rollsum(log(1+sp500TR.M$Re),12,align="left",fill = NA)),-1)-1)

CRSP.M$R1  <- 12*(lag(CRSP.M$Re,-1))
CRSP.M$R2  <- 6*(lag(exp(rollsum(log(1+CRSP.M$Re),2,align="left",fill = NA)),-1)-1)
CRSP.M$R3  <- 4*(lag(exp(rollsum(log(1+CRSP.M$Re),3,align="left",fill = NA)),-1)-1)
CRSP.M$R6  <- 2*(lag(exp(rollsum(log(1+CRSP.M$Re),6,align="left",fill = NA)),-1)-1)
CRSP.M$R12 <- (lag(exp(rollsum(log(1+CRSP.M$Re),12,align="left",fill = NA)),-1)-1)

CRSP.M$Rx1  <- 12*(lag(CRSP.M$Rxe,-1))
CRSP.M$Rx2  <- 6*(lag(exp(rollsum(log(1+CRSP.M$Rxe),2,align="left",fill = NA)),-1)-1)
CRSP.M$Rx3  <- 4*(lag(exp(rollsum(log(1+CRSP.M$Rxe),3,align="left",fill = NA)),-1)-1)
CRSP.M$Rx6  <- 2*(lag(exp(rollsum(log(1+CRSP.M$Rxe),6,align="left",fill = NA)),-1)-1)
CRSP.M$Rx12 <- (lag(exp(rollsum(log(1+CRSP.M$Rxe),12,align="left",fill = NA)),-1)-1)



#SVIX to return objects
sp500.M$RftSVIX2m1  <- svix.M$RftSVIX2m1
sp500.M$RftSVIX2m2  <- svix.M$RftSVIX2m2
sp500.M$RftSVIX2m3  <- svix.M$RftSVIX2m3
sp500.M$RftSVIX2m6  <- svix.M$RftSVIX2m6
sp500.M$RftSVIX2m12 <- svix.M$RftSVIX2m12
sp500.M$vix         <- (vix.M$Close/100)^2



sp500TR.M$RftSVIX2m1  <- svix.M$RftSVIX2m1
sp500TR.M$RftSVIX2m2  <- svix.M$RftSVIX2m2
sp500TR.M$RftSVIX2m3  <- svix.M$RftSVIX2m3
sp500TR.M$RftSVIX2m6  <- svix.M$RftSVIX2m6
sp500TR.M$RftSVIX2m12 <- svix.M$RftSVIX2m12
sp500TR.M$vix         <- (vix.M$Close/100)^2



CRSP.M$RftSVIX2m1  <- svix.M$RftSVIX2m1
CRSP.M$RftSVIX2m2  <- svix.M$RftSVIX2m2
CRSP.M$RftSVIX2m3  <- svix.M$RftSVIX2m3
CRSP.M$RftSVIX2m6  <- svix.M$RftSVIX2m6
CRSP.M$RftSVIX2m12 <- svix.M$RftSVIX2m12
CRSP.M$vix         <- (vix.M$Close/100)^2


sp500.M$Rftm1  <- svix.M$Rftm1
sp500.M$Rftm2  <- svix.M$Rftm2
sp500.M$Rftm3  <- svix.M$Rftm3
sp500.M$Rftm6  <- svix.M$Rftm6
sp500.M$Rftm12 <- svix.M$Rftm12

sp500TR.M$Rftm1  <- svix.M$Rftm1
sp500TR.M$Rftm2  <- svix.M$Rftm2
sp500TR.M$Rftm3  <- svix.M$Rftm3
sp500TR.M$Rftm6  <- svix.M$Rftm6
sp500TR.M$Rftm12 <- svix.M$Rftm12

CRSP.M$Rftm1  <- svix.M$Rftm1
CRSP.M$Rftm2  <- svix.M$Rftm2
CRSP.M$Rftm3  <- svix.M$Rftm3
CRSP.M$Rftm6  <- svix.M$Rftm6
CRSP.M$Rftm12 <- svix.M$Rftm12

REG <- function(y,x,H)
{
  Tt <- nrow(na.omit(x))
  reg <- lm(ts(y)~ts(x))

  ##HANSEN HODRICK
  X <- cbind(1,na.omit(x))

  u0 <- matrix(reg$residuals,Tt,1)  
  u  <- u0%*%matrix(1,1,ncol(X))
  err <- X*u  
    
  V <- t(err)%*%err/Tt
  
  if(H>1)
  {
    for (i in 1:H)
    {
      S <- t(err[1:(Tt-i),])%*%(err[(1+i):Tt,])/Tt
      V <- V+(S+t(S))
    }
  }
  
  D    <- solve(t(X)%*%X/Tt)
  VChh <- (D%*%V%*%D/Tt)
  
  colnames(VChh) <- NULL
  rownames(VChh) <- NULL
  
  se <- diag(VChh)
  sqrt(se)
  
  #VChac <-  vcovHAC(reg,prewhite = TRUE)
  #VCnw  <- NeweyWest(reg,lag=H)
  
  VC    <- VChh 
  ct    <-  coeftest(reg,vcov.=VC)
  
  
  yrm <- y
  
  for (t in 2:length(y)){yrm[t] <- mean(y[1:(t-1)])}
  if(ncol(x)==1)
     {
     pval  <-  linearHypothesis(reg,diag(2),c(0,1),test="F",vcov.=VC)$`Pr(>F)`[2]
     oosR2_r <-  1-sum(na.omit((y-1*x)^2))/sum(na.omit(y-yrm-0*x)^2)   #restriced 
     oosR2_u  <-  1-sum(na.omit((resid(reg))^2))/sum(na.omit(y-yrm-0*x)^2)  #unrestriced
  }
  if(ncol(x)==2)
     {
     pval  <-  linearHypothesis(reg,diag(3),c(0,1,0),test="F",vcov.=VC)$`Pr(>F)`[2]
     oosR2_r <-  1-sum(na.omit((y-x[,1])^2))/sum(na.omit(y-yrm-0*x[,1])^2)   #restriced 
     oosR2_u  <-  1-sum(na.omit((resid(reg))^2))/sum(na.omit(y-yrm-0*x[,1])^2)  #unrestriced
  }

  res <-(c(t(ct[,1:2]),pval,summary(reg)$r.squared,oosR2_r))
  if (ncol(x)==1){names(res) <- c("alpha","","beta","","p","R2","oosr2r")}
  if (ncol(x)==2){names(res) <- c("alpha","","beta","","gamma","","p","R2","oosr2r")}
  return(res)
}



####TABLE 1
xtable(rbind(cbind(
#SP500
cbind(
REG(sp500.M$R1,sp500.M$RftSVIX2m1,1),
REG(sp500.M$R2,sp500.M$RftSVIX2m2,2),
REG(sp500.M$R3,sp500.M$RftSVIX2m3,3),
REG(sp500.M$R6,sp500.M$RftSVIX2m6,6),
REG(sp500.M$R12,sp500.M$RftSVIX2m12,12))
,
#SP500TR
cbind(
  REG(sp500TR.M$R1,sp500TR.M$RftSVIX2m1,1),
  REG(sp500TR.M$R2,sp500TR.M$RftSVIX2m2,2),
  REG(sp500TR.M$R3,sp500TR.M$RftSVIX2m3,3),
  REG(sp500TR.M$R6,sp500TR.M$RftSVIX2m6,6),
  REG(sp500TR.M$R12,sp500TR.M$RftSVIX2m12,12))
)
,
cbind(
  #CRSP RX 
  cbind(
    REG(CRSP.M$Rx1,CRSP.M$RftSVIX2m1,1),
    REG(CRSP.M$Rx2,CRSP.M$RftSVIX2m2,2),
    REG(CRSP.M$Rx3,CRSP.M$RftSVIX2m3,3),
    REG(CRSP.M$Rx6,CRSP.M$RftSVIX2m6,6),
    REG(CRSP.M$Rx12,CRSP.M$RftSVIX2m12,12))
,
#CRSP 
cbind(
  REG(CRSP.M$R1,CRSP.M$RftSVIX2m1,1),
  REG(CRSP.M$R2,CRSP.M$RftSVIX2m2,2),
  REG(CRSP.M$R3,CRSP.M$RftSVIX2m3,3),
  REG(CRSP.M$R6,CRSP.M$RftSVIX2m6,6),
  REG(CRSP.M$R12,CRSP.M$RftSVIX2m12,12))
)))



####TABLE 2
##EWMA same sample
#SP500
xtable(rbind(cbind(
cbind(
  REG(sp500.M['1996/2012']$R1,sp500.M['1996/2012']$ewma*sp500.M['1996/2012']$Rftm1,1),
  REG(sp500.M['1996/2012']$R2,sp500.M['1996/2012']$ewma*sp500.M['1996/2012']$Rftm2,2),
  REG(sp500.M['1996/2012']$R3,sp500.M['1996/2012']$ewma*sp500.M['1996/2012']$Rftm3,3),
  REG(sp500.M['1996/2012']$R6,sp500.M['1996/2012']$ewma*sp500.M['1996/2012']$Rftm6,6),
  REG(sp500.M['1996/2012']$R12,sp500.M['1996/2012']$ewma*sp500.M['1996/2012']$Rftm12,12))
,
#SP500TR
cbind(
  REG(sp500TR.M['1996/2012']$R1,sp500TR.M['1996/2012']$ewma*sp500TR.M['1996/2012']$Rftm1,1),
  REG(sp500TR.M['1996/2012']$R2,sp500TR.M['1996/2012']$ewma*sp500TR.M['1996/2012']$Rftm2,2),
  REG(sp500TR.M['1996/2012']$R3,sp500TR.M['1996/2012']$ewma*sp500TR.M['1996/2012']$Rftm3,3),
  REG(sp500TR.M['1996/2012']$R6,sp500TR.M['1996/2012']$ewma*sp500TR.M['1996/2012']$Rftm6,6),
  REG(sp500TR.M['1996/2012']$R12,sp500TR.M['1996/2012']$ewma*sp500TR.M['1996/2012']$Rftm12,12))
),cbind(
#CRSP 
cbind(
  REG(CRSP.M['1996/2012']$R1,CRSP.M['1996/2012']$ewma*CRSP.M['1996/2012']$Rftm1,1),
  REG(CRSP.M['1996/2012']$R2,CRSP.M['1996/2012']$ewma*CRSP.M['1996/2012']$Rftm2,2),
  REG(CRSP.M['1996/2012']$R3,CRSP.M['1996/2012']$ewma*CRSP.M['1996/2012']$Rftm3,3),
  REG(CRSP.M['1996/2012']$R6,CRSP.M['1996/2012']$ewma*CRSP.M['1996/2012']$Rftm6,6),
  REG(CRSP.M['1996/2012']$R12,CRSP.M['1996/2012']$ewma*CRSP.M['1996/2012']$Rftm12,12))
,
#CRSP RX 
cbind(
  REG(CRSP.M['1996/2012']$Rx1,CRSP.M['1996/2012']$ewmax*CRSP.M['1996/2012']$Rftm1,1),
  REG(CRSP.M['1996/2012']$Rx2,CRSP.M['1996/2012']$ewmax*CRSP.M['1996/2012']$Rftm2,2),
  REG(CRSP.M['1996/2012']$Rx3,CRSP.M['1996/2012']$ewmax*CRSP.M['1996/2012']$Rftm3,3),
  REG(CRSP.M['1996/2012']$Rx6,CRSP.M['1996/2012']$ewmax*CRSP.M['1996/2012']$Rftm6,6),
  REG(CRSP.M['1996/2012']$Rx12,CRSP.M['1996/2012']$ewmax*CRSP.M['1996/2012']$Rftm12,12))
)))


#TABLE A.1
##vix same sample
#SP500
xtable(rbind(cbind(
  cbind(
    REG(sp500.M['1996/2012']$R1,sp500.M['1996/2012']$vix*sp500.M['1996/2012']$Rftm1,1),
    REG(sp500.M['1996/2012']$R2,sp500.M['1996/2012']$vix*sp500.M['1996/2012']$Rftm2,2),
    REG(sp500.M['1996/2012']$R3,sp500.M['1996/2012']$vix*sp500.M['1996/2012']$Rftm3,3),
    REG(sp500.M['1996/2012']$R6,sp500.M['1996/2012']$vix*sp500.M['1996/2012']$Rftm6,6),
    REG(sp500.M['1996/2012']$R12,sp500.M['1996/2012']$vix*sp500.M['1996/2012']$Rftm12,12))
  ,
  #SP500TR
  cbind(
    REG(sp500TR.M['1996/2012']$R1,sp500TR.M['1996/2012']$vix*sp500TR.M['1996/2012']$Rftm1,1),
    REG(sp500TR.M['1996/2012']$R2,sp500TR.M['1996/2012']$vix*sp500TR.M['1996/2012']$Rftm2,2),
    REG(sp500TR.M['1996/2012']$R3,sp500TR.M['1996/2012']$vix*sp500TR.M['1996/2012']$Rftm3,3),
    REG(sp500TR.M['1996/2012']$R6,sp500TR.M['1996/2012']$vix*sp500TR.M['1996/2012']$Rftm6,6),
    REG(sp500TR.M['1996/2012']$R12,sp500TR.M['1996/2012']$vix*sp500TR.M['1996/2012']$Rftm12,12))
),cbind(
  #CRSP 
  cbind(
    REG(CRSP.M['1996/2012']$R1,CRSP.M['1996/2012']$vix*CRSP.M['1996/2012']$Rftm1,1),
    REG(CRSP.M['1996/2012']$R2,CRSP.M['1996/2012']$vix*CRSP.M['1996/2012']$Rftm2,2),
    REG(CRSP.M['1996/2012']$R3,CRSP.M['1996/2012']$vix*CRSP.M['1996/2012']$Rftm3,3),
    REG(CRSP.M['1996/2012']$R6,CRSP.M['1996/2012']$vix*CRSP.M['1996/2012']$Rftm6,6),
    REG(CRSP.M['1996/2012']$R12,CRSP.M['1996/2012']$vix*CRSP.M['1996/2012']$Rftm12,12))
  ,
  #CRSP RX 
  cbind(
    REG(CRSP.M['1996/2012']$Rx1,CRSP.M['1996/2012']$vix*CRSP.M['1996/2012']$Rftm1,1),
    REG(CRSP.M['1996/2012']$Rx2,CRSP.M['1996/2012']$vix*CRSP.M['1996/2012']$Rftm2,2),
    REG(CRSP.M['1996/2012']$Rx3,CRSP.M['1996/2012']$vix*CRSP.M['1996/2012']$Rftm3,3),
    REG(CRSP.M['1996/2012']$Rx6,CRSP.M['1996/2012']$vix*CRSP.M['1996/2012']$Rftm6,6),
    REG(CRSP.M['1996/2012']$Rx12,CRSP.M['1996/2012']$vix*CRSP.M['1996/2012']$Rftm12,12))
)))





#TABLE A.2
##VAR same sample
#SP500
xtable(rbind(cbind(
cbind(
  REG(sp500.M['1996/2012']$R1,sp500.M['1996/2012']$var*sp500.M['1996/2012']$Rftm1,1),
  REG(sp500.M['1996/2012']$R2,sp500.M['1996/2012']$var*sp500.M['1996/2012']$Rftm2,2),
  REG(sp500.M['1996/2012']$R3,sp500.M['1996/2012']$var*sp500.M['1996/2012']$Rftm3,3),
  REG(sp500.M['1996/2012']$R6,sp500.M['1996/2012']$var*sp500.M['1996/2012']$Rftm6,6),
  REG(sp500.M['1996/2012']$R12,sp500.M['1996/2012']$var*sp500.M['1996/2012']$Rftm12,12))
,
#SP500TR
cbind(
  REG(sp500TR.M['1996/2012']$R1,sp500TR.M['1996/2012']$var*sp500TR.M['1996/2012']$Rftm1,1),
  REG(sp500TR.M['1996/2012']$R2,sp500TR.M['1996/2012']$var*sp500TR.M['1996/2012']$Rftm2,2),
  REG(sp500TR.M['1996/2012']$R3,sp500TR.M['1996/2012']$var*sp500TR.M['1996/2012']$Rftm3,3),
  REG(sp500TR.M['1996/2012']$R6,sp500TR.M['1996/2012']$var*sp500TR.M['1996/2012']$Rftm6,6),
  REG(sp500TR.M['1996/2012']$R12,sp500TR.M['1996/2012']$var*sp500TR.M['1996/2012']$Rftm12,12))
),cbind(
#CRSP 
cbind(
  REG(CRSP.M['1996/2012']$R1,CRSP.M['1996/2012']$var*CRSP.M['1996/2012']$Rftm1,1),
  REG(CRSP.M['1996/2012']$R2,CRSP.M['1996/2012']$var*CRSP.M['1996/2012']$Rftm2,2),
  REG(CRSP.M['1996/2012']$R3,CRSP.M['1996/2012']$var*CRSP.M['1996/2012']$Rftm3,3),
  REG(CRSP.M['1996/2012']$R6,CRSP.M['1996/2012']$var*CRSP.M['1996/2012']$Rftm6,6),
  REG(CRSP.M['1996/2012']$R12,CRSP.M['1996/2012']$var*CRSP.M['1996/2012']$Rftm12,12))
,
#CRSP RX 
cbind(
  REG(CRSP.M['1996/2012']$Rx1,CRSP.M['1996/2012']$varx*CRSP.M['1996/2012']$Rftm1,1),
  REG(CRSP.M['1996/2012']$Rx2,CRSP.M['1996/2012']$varx*CRSP.M['1996/2012']$Rftm2,2),
  REG(CRSP.M['1996/2012']$Rx3,CRSP.M['1996/2012']$varx*CRSP.M['1996/2012']$Rftm3,3),
  REG(CRSP.M['1996/2012']$Rx6,CRSP.M['1996/2012']$varx*CRSP.M['1996/2012']$Rftm6,6),
  REG(CRSP.M['1996/2012']$Rx12,CRSP.M['1996/2012']$varx*CRSP.M['1996/2012']$Rftm12,12))
)))

#TABLE 3
##BOTH SVIX AND EWMA
xtable(rbind(cbind(
#SP500
cbind(
  REG(sp500.M$R1,cbind(sp500.M$RftSVIX2m1,sp500.M['1996/2012']$ewma*sp500.M['1996/2012']$Rftm1),1),
  REG(sp500.M$R2,cbind(sp500.M$RftSVIX2m2,sp500.M['1996/2012']$ewma*sp500.M['1996/2012']$Rftm2),2),
  REG(sp500.M$R3,cbind(sp500.M$RftSVIX2m3,sp500.M['1996/2012']$ewma*sp500.M['1996/2012']$Rftm3),3),
  REG(sp500.M$R6,cbind(sp500.M$RftSVIX2m6,sp500.M['1996/2012']$ewma*sp500.M['1996/2012']$Rftm6),6),
  REG(sp500.M$R12,cbind(sp500.M$RftSVIX2m12,sp500.M['1996/2012']$ewma*sp500.M['1996/2012']$Rftm12),12))
,
#SP500TR
cbind(
  REG(sp500TR.M$R1,cbind(sp500TR.M$RftSVIX2m1,sp500TR.M['1996/2012']$ewma*sp500TR.M['1996/2012']$Rftm1),1),
  REG(sp500TR.M$R2,cbind(sp500TR.M$RftSVIX2m2,sp500TR.M['1996/2012']$ewma*sp500TR.M['1996/2012']$Rftm2),2),
  REG(sp500TR.M$R3,cbind(sp500TR.M$RftSVIX2m3,sp500TR.M['1996/2012']$ewma*sp500TR.M['1996/2012']$Rftm3),3),
  REG(sp500TR.M$R6,cbind(sp500TR.M$RftSVIX2m6,sp500TR.M['1996/2012']$ewma*sp500TR.M['1996/2012']$Rftm6),6),
  REG(sp500TR.M$R12,cbind(sp500TR.M$RftSVIX2m12,sp500TR.M['1996/2012']$ewma*sp500TR.M['1996/2012']$Rftm12),12))
),
cbind(
#CRSP 
cbind(
  REG(CRSP.M$R1,cbind(CRSP.M$RftSVIX2m1,CRSP.M['1996/2012']$ewma*CRSP.M['1996/2012']$Rftm1),1),
  REG(CRSP.M$R2,cbind(CRSP.M$RftSVIX2m2,CRSP.M['1996/2012']$ewma*CRSP.M['1996/2012']$Rftm2),2),
  REG(CRSP.M$R3,cbind(CRSP.M$RftSVIX2m3,CRSP.M['1996/2012']$ewma*CRSP.M['1996/2012']$Rftm3),3),
  REG(CRSP.M$R6,cbind(CRSP.M$RftSVIX2m6,CRSP.M['1996/2012']$ewma*CRSP.M['1996/2012']$Rftm6),6),
  REG(CRSP.M$R12,cbind(CRSP.M$RftSVIX2m12,CRSP.M['1996/2012']$ewma*CRSP.M['1996/2012']$Rftm12),12))
,
#CRSP RX 
cbind(
  REG(CRSP.M$Rx1,cbind(CRSP.M$RftSVIX2m1,CRSP.M['1996/2012']$ewmax*CRSP.M['1996/2012']$Rftm1),1),
  REG(CRSP.M$Rx2,cbind(CRSP.M$RftSVIX2m2,CRSP.M['1996/2012']$ewmax*CRSP.M['1996/2012']$Rftm2),2),
  REG(CRSP.M$Rx3,cbind(CRSP.M$RftSVIX2m3,CRSP.M['1996/2012']$ewmax*CRSP.M['1996/2012']$Rftm3),3),
  REG(CRSP.M$Rx6,cbind(CRSP.M$RftSVIX2m6,CRSP.M['1996/2012']$ewmax*CRSP.M['1996/2012']$Rftm6),6),
  REG(CRSP.M$Rx12,cbind(CRSP.M$RftSVIX2m12,CRSP.M['1996/2012']$ewmax*CRSP.M['1996/2012']$Rftm12),12))
)))


#Table A.3
##BOTH VIX AND VAR
xtable(rbind(cbind(
  #SP500
  cbind(
    REG(sp500.M$R1['1996/2012'],cbind(sp500TR.M['1996/2012']$vix*sp500TR.M['1996/2012']$Rftm1,sp500.M['1996/2012']$var*sp500.M['1996/2012']$Rftm1),1),
    REG(sp500.M$R2['1996/2012'],cbind(sp500TR.M['1996/2012']$vix*sp500TR.M['1996/2012']$Rftm2,sp500.M['1996/2012']$var*sp500.M['1996/2012']$Rftm2),2),
    REG(sp500.M$R3['1996/2012'],cbind(sp500TR.M['1996/2012']$vix*sp500TR.M['1996/2012']$Rftm3,sp500.M['1996/2012']$var*sp500.M['1996/2012']$Rftm3),3),
    REG(sp500.M$R6['1996/2012'],cbind(sp500TR.M['1996/2012']$vix*sp500TR.M['1996/2012']$Rftm6,sp500.M['1996/2012']$var*sp500.M['1996/2012']$Rftm6),6),
    REG(sp500.M$R12['1996/2012'],cbind(sp500TR.M['1996/2012']$vix*sp500TR.M['1996/2012']$Rftm12,sp500.M['1996/2012']$var*sp500.M['1996/2012']$Rftm12),12))
  ,
  #SP500TR
  cbind(
    REG(sp500TR.M$R1['1996/2012'],cbind(sp500TR.M['1996/2012']$vix*sp500TR.M['1996/2012']$Rftm1,sp500TR.M['1996/2012']$var*sp500TR.M['1996/2012']$Rftm1),1),
    REG(sp500TR.M$R2['1996/2012'],cbind(sp500TR.M['1996/2012']$vix*sp500TR.M['1996/2012']$Rftm2,sp500TR.M['1996/2012']$var*sp500TR.M['1996/2012']$Rftm2),2),
    REG(sp500TR.M$R3['1996/2012'],cbind(sp500TR.M['1996/2012']$vix*sp500TR.M['1996/2012']$Rftm3,sp500TR.M['1996/2012']$var*sp500TR.M['1996/2012']$Rftm3),3),
    REG(sp500TR.M$R6['1996/2012'],cbind(sp500TR.M['1996/2012']$vix*sp500TR.M['1996/2012']$Rftm6,sp500TR.M['1996/2012']$var*sp500TR.M['1996/2012']$Rftm6),6),
    REG(sp500TR.M$R12['1996/2012'],cbind(sp500TR.M['1996/2012']$vix*sp500TR.M['1996/2012']$Rftm12,sp500TR.M['1996/2012']$var*sp500TR.M['1996/2012']$Rftm12),12))
),
cbind(
  #CRSP 
  cbind(
    REG(CRSP.M$R1['1996/2012'],cbind(sp500TR.M['1996/2012']$vix*sp500TR.M['1996/2012']$Rftm1,CRSP.M['1996/2012']$var*CRSP.M['1996/2012']$Rftm1),1),
    REG(CRSP.M$R2['1996/2012'],cbind(sp500TR.M['1996/2012']$vix*sp500TR.M['1996/2012']$Rftm2,CRSP.M['1996/2012']$var*CRSP.M['1996/2012']$Rftm2),2),
    REG(CRSP.M$R3['1996/2012'],cbind(sp500TR.M['1996/2012']$vix*sp500TR.M['1996/2012']$Rftm3,CRSP.M['1996/2012']$var*CRSP.M['1996/2012']$Rftm3),3),
    REG(CRSP.M$R6['1996/2012'],cbind(sp500TR.M['1996/2012']$vix*sp500TR.M['1996/2012']$Rftm6,CRSP.M['1996/2012']$var*CRSP.M['1996/2012']$Rftm6),6),
    REG(CRSP.M$R12['1996/2012'],cbind(sp500TR.M['1996/2012']$vix*sp500TR.M['1996/2012']$Rftm12,CRSP.M['1996/2012']$var*CRSP.M['1996/2012']$Rftm12),12))
  ,
  #CRSP RX 
  cbind(
    REG(CRSP.M$Rx1['1996/2012'],cbind(sp500TR.M['1996/2012']$vix*sp500TR.M['1996/2012']$Rftm1,CRSP.M['1996/2012']$varx*CRSP.M['1996/2012']$Rftm1),1),
    REG(CRSP.M$Rx2['1996/2012'],cbind(sp500TR.M['1996/2012']$vix*sp500TR.M['1996/2012']$Rftm2,CRSP.M['1996/2012']$varx*CRSP.M['1996/2012']$Rftm2),2),
    REG(CRSP.M$Rx3['1996/2012'],cbind(sp500TR.M['1996/2012']$vix*sp500TR.M['1996/2012']$Rftm3,CRSP.M['1996/2012']$varx*CRSP.M['1996/2012']$Rftm3),3),
    REG(CRSP.M$Rx6['1996/2012'],cbind(sp500TR.M['1996/2012']$vix*sp500TR.M['1996/2012']$Rftm6,CRSP.M['1996/2012']$varx*CRSP.M['1996/2012']$Rftm6),6),
    REG(CRSP.M$Rx12['1996/2012'],cbind(sp500TR.M['1996/2012']$vix*sp500TR.M['1996/2012']$Rftm12,CRSP.M['1996/2012']$varx*CRSP.M['1996/2012']$Rftm12),12))
)))



### MONTHLY VARIANCE COMPARISON

svix.M$ewma <- sp500.M$ewma[index(svix.M)]
svix.M$ewmaTR <- sp500TR.M$ewma[index(svix.M)]
svix.M$vix <- sp500.M$vix[index(svix.M)]
svix.M$var <- sp500TR.M$var[index(svix.M)]

svix.M$R1 <- sp500.M$R1[index(svix.M)]
svix.M$R12 <- sp500.M$R12[index(svix.M)]


svix.M$ewmaD <- diff(svix.M$ewma,1)
svix.M$ewmaTRD <- diff(svix.M$ewmaTR,1)
svix.M$vixD <- diff(svix.M$vix,1)
svix.M$SVIX2m1D <- diff(svix.M$SVIX2m1,1)
svix.M$SVIX2m2D <- diff(svix.M$SVIX2m2,1)
svix.M$SVIX2m3D <- diff(svix.M$SVIX2m3,1)
svix.M$SVIX2m6D <- diff(svix.M$SVIX2m6,1)
svix.M$SVIX2m12D <- diff(svix.M$SVIX2m12,1)
svix.M$R1D       <- diff(svix.M$R1,1)
svix.M$R12D       <- diff(svix.M$R12,1)
svix.M$varD <- diff(svix.M$var,1)



svix.M$ewmaDL    <- lag(svix.M$ewmaD,1)
svix.M$ewmaTRDL  <- lag(svix.M$ewmaTRD,1)
svix.M$vixDL    <- lag(svix.M$vixD,1)
svix.M$SVIX2m1DL <- lag(svix.M$SVIX2m1D,1)
svix.M$SVIX2m2DL <- lag(svix.M$SVIX2m2D,1)
svix.M$SVIX2m3DL <- lag(svix.M$SVIX2m3D,1)
svix.M$SVIX2m6DL <- lag(svix.M$SVIX2m6D,1)
svix.M$SVIX2m12DL <- lag(svix.M$SVIX2m12D,1)
svix.M$varDL    <- lag(svix.M$varD,1)



C1 <- cor(svix.M[,c("SVIX2m1","SVIX2m2","SVIX2m3","SVIX2m6","SVIX2m12","ewma","R1")],use="complete.obs")
C2 <- cor(svix.M[,c("SVIX2m1D","SVIX2m2D","SVIX2m3D","SVIX2m6D","SVIX2m12D","ewmaD","R1D")],use="complete.obs")

C1[lower.tri(C1,diag=TRUE)]<-NA
C2[lower.tri(C2,diag=TRUE)]<-NA

#TABLE 4
xtable(rbind(C1[-nrow(C1),-1],C2[-nrow(C1),-1]))

C1_IA <- cor(svix.M[,c("SVIX2m1","vix","ewma","var","R1")],use="complete.obs")
C2_IA <- cor(svix.M[,c("SVIX2m1D","vixD","ewmaD","varD","R1D")],use="complete.obs")

C1_IA[lower.tri(C1_IA,diag=TRUE)]<-NA
C2_IA[lower.tri(C2_IA,diag=TRUE)]<-NA

#TABLE A4
xtable(rbind(C1_IA[-nrow(C1_IA),-1],C2_IA[-nrow(C1_IA),-1]))


reg1M <- (lm(SVIX2m1D~SVIX2m1DL,svix.M))
reg2M <- (lm(SVIX2m1D~ewmaDL,svix.M))
reg3M <- (lm(SVIX2m1D~SVIX2m1DL+ewmaDL,svix.M))


reg4M <- (lm(ewmaD~SVIX2m1DL,svix.M))
reg5M <- (lm(ewmaD~ewmaDL,svix.M))
reg6M <- (lm(ewmaD~SVIX2m1DL+ewmaDL,svix.M))


#Table 5
xtable(cbind(
(c(t(coeftest(reg1M,vcov.=NeweyWest(reg1M))[,1:2]),NA,NA,summary(reg1M)$r.squared)),
(c(t(coeftest(reg2M,vcov.=NeweyWest(reg2M))[1,1:2]),NA,NA,t(coeftest(reg2M,vcov.=NeweyWest(reg2M))[2,1:2]),summary(reg2M)$r.squared)),
(c(t(coeftest(reg3M,vcov.=NeweyWest(reg3M))[,1:2]),summary(reg3M)$r.squared)),
(c(t(coeftest(reg4M,vcov.=NeweyWest(reg4M))[,1:2]),NA,NA,summary(reg4M)$r.squared)),
(c(t(coeftest(reg5M,vcov.=NeweyWest(reg5M))[1,1:2]),NA,NA,t(coeftest(reg5M,vcov.=NeweyWest(reg5M))[2,1:2]),summary(reg5M)$r.squared)),
(c(t(coeftest(reg6M,vcov.=NeweyWest(reg6M))[,1:2]),summary(reg6M)$r.squared))),digits=4)



reg1MA <- (lm(vixD~vixDL,svix.M))
reg2MA <- (lm(vixD~varDL,svix.M))
reg3MA <- (lm(vixD~vixDL+varDL,svix.M))


reg4MA <- (lm(varD~vixDL,svix.M))
reg5MA <- (lm(varD~varDL,svix.M))
reg6MA <- (lm(varD~vixDL+varDL,svix.M))


#Table A5
xtable(cbind(
  (c(t(coeftest(reg1MA,vcov.=NeweyWest(reg1MA))[,1:2]),NA,NA,summary(reg1MA)$r.squared)),
  (c(t(coeftest(reg2MA,vcov.=NeweyWest(reg2MA))[1,1:2]),NA,NA,t(coeftest(reg2MA,vcov.=NeweyWest(reg2MA))[2,1:2]),summary(reg2MA)$r.squared)),
  (c(t(coeftest(reg3MA,vcov.=NeweyWest(reg3MA))[,1:2]),summary(reg3MA)$r.squared)),
  (c(t(coeftest(reg4MA,vcov.=NeweyWest(reg4MA))[,1:2]),NA,NA,summary(reg4MA)$r.squared)),
  (c(t(coeftest(reg5MA,vcov.=NeweyWest(reg5MA))[1,1:2]),NA,NA,t(coeftest(reg5MA,vcov.=NeweyWest(reg5MA))[2,1:2]),summary(reg5MA)$r.squared)),
  (c(t(coeftest(reg6MA,vcov.=NeweyWest(reg6MA))[,1:2]),summary(reg6MA)$r.squared))),digits=4)

