//CGG New Keynesian model with fully non-linear vs. linear equilibrium equations
//Consumption demand or discount factor shocks take economy to ZLB; Fiscal multiplier analysis
//Kimball (1995) aggregator to get slope of PC and micro-evidence of price stickiness jointly

%addpath('Dynare424');
close all;

//Dynare version: 4.2.5
//Mathias Trabandt, mathias.trabandt@gmail.com, April 2013  


//MT changes per March 2013:
//1. Taylor rule coefficients set to 1.5 and 0.125
//2. Make labor income taxes a true dynamic variable (in actual and pot econ.)
//3. Put in a new fiscal rule for labor income taxes (parameter phiN)
//4. Renamed equation numbering due to new variables
//5. Plot more variables (Debt, Lump-sum taxes, dist. labor tax)
//6. Deleted all code after figure 2 in mod file
//7. Changed ordering of variables in Dynare var statement!
//8. Made lots of changes to plot_results.m file (variable creation, plotting etc)
//9. Per new switch, allow government spending shock only when ZLB is binding (gAtZLBonly)
//10.Re-specified the Taylor rule in logs rather than in levels (to get rid of the SS expressions in the linear case)

//MT changes per April 2013:
//11.Kimball (1995) aggregator introduced
//12.Discount factor shock added
//13.More variables in var statement due to 11 and 12.
//14.Fixed error in steady state computations
//15.General (non-) indexation scheme available in non-linear model. Yet not available 
//in linearized model and steady state. I.e. stick to indexation for the time being! 

 
// Endogenous variables 
var 
xV piV RV rrV rrVpot yV pstarV bGV tauV tauNV gV deltaV                                              //these 12 variables are plotted
xVBREVE piVHAT RVHAT rrVHAT rrVpotHAT yVHAT pstarVHAT bGVBREVE tauVBREVE tauNVBREVE gVHAT deltaVHAT  //these 12 variables are plotted
cV nV wV sV mcV fV varthetaV aV ptildeV delta2V delta1V delta3V nuV 
cVpot nVpot wVpot yVpot bGVpot tauVpot lamV tauNVpot nVHAT 
cVHAT wVHAT mcVHAT cVpotHAT wVpotHAT yVpotHAT bGVpotBREVE 
tauVpotBREVE nVpotHAT tauNVpotBREVE nuVBREVE
;  
 
// Shocks 
varexo nuV_eps gV_eps deltaV_eps;  
 
// Parameters 
parameters betta tauC tauN cV_SS sigma chi thetap k alfa xip piV_SS
imposeZLB RV_SS gampi gamx yV_SS tauV_SS phiT bGV_SS rhoG rhonu sigG 
gV_SS nuV_SS signu pibar bGy gy tauy nu wV_SS nV_SS rrV_SS pstarV_SS
sV_SS fV_SS mcV_SS phiN tauNV_SS gAtZLBonly psie pitilde rhodelta sigdelta deltaV_SS; 
 

//initialize parameters and imposed steady states
pibar=1.005;  %steady state gross inflation rate
thetap=0.1;   %net price markup in steady state
xip=0.9;     %Calvo sticky prices parameter
psie=0;      %Kimball aggregator parameter 
pitilde=pibar;%Indexation to ss inflation

if abs(pitilde-pibar)>0,
    error('pitilde needs to equal pibar for now!');
end

gampi=1.5;   %Taylor rule coef. on inflation
gamx=0.125;  %Taylor rule coef. on output gap

alfa=0.3;    %capital share in production
betta=0.995; %discount factor of households
sigma=1;     %inverse intertemporal elasticity of substitution
             %code needs fixing for sigma different from 1
chi=1/0.4;   %inverse Frisch elasticity of labor supply
nu=0.01;     %steady state of cons. demand shock

gy=0.2;      %gov. consumption to output ratio in steady state
tauC=0;      %steady state cons. tax
bGy=2.4;     %steady state debt to GDP ratio
tauy=0;      %steady state transfer to GDP ratio

phiT=1*(1/betta-1)*2; %fiscal rule for lump-sum taxes: coef. on debt
phiN=0*(1/betta-1)*2; %fiscal rule for labor taxes: coef. on debt

rhoG=0.95;    %AR(1) gov. consumption shock
rhonu=0.95;   %AR(1) cons. demand shock
rhodelta=0.95;%AR(1) discount factor shock

sigG=0.05;   %Std. gov. consumption shock  %rig such that d gy 1%
signu=0.01;  %Std. cons. demand shock
sigdelta=0.005;%Std. discount factor shock

imposeZLB=1; %impose ZLB in Taylor rule if =1; if =0 ZLB not imposed
gAtZLBonly=0;%if=0, gov. spending follows AR(1); if=1 gov. spending positive at ZLB only


//available figures
%fig1: Cons. demand shock; with and without ZLB
%fig2: Discount factor shock; with and without ZLB
%fig3: Cons. demand shock vs. cons. demand plus gov. cons. shock; with ZLB
%4 - find sigdeltas
%6 - repeated simulation of model with sigdelta shocks
%7 - find signus (SHER)
%8 - repeated simulation of model with signu shocks (SHER)

do_fig=8;
do_show_unconstained=0; %if =1, ZLB not imposed results are plotted

//endogenous steady states are calculated in _steadystate.m file  
model; 
///////////////////////////////////////////////////////
//non-linear economy                                 //
///////////////////////////////////////////////////////
//set some fiscal instruments to their steady states
#tauCV=tauC;
#tauCV_tp1=tauC;

//n1 marg. utility
lamV=(cV-cV_SS*nuV)^(-1/sigma)/(1+tauCV);  
  
//n2 leisure-labor tradeoff 
nV^chi =(1-tauNV)*lamV*wV; 
 
//n3 euler equation
lamV=betta*deltaV*RV/piV(+1)*lamV(+1); 

//n4 resource constraint
cV+gV=yV;

//n5 production
yV=pstarV^(-1)*k^alfa*nV^(1-alfa); 
 
//n6 non-lin. pricing 1 
sV=(1+thetap)*(1+psie)/(1+psie+thetap*psie)*lamV*yV*mcV*varthetaV^((1+thetap)/thetap*(1+psie))+xip*betta*deltaV*(pitilde/piV(+1))^(-(1+thetap)/thetap*(1+psie))*sV(+1);  

//n7 non-lin. pricing 2 
fV=lamV*yV*varthetaV^((1+thetap)/thetap*(1+psie))+xip*betta*deltaV*(pitilde/piV(+1))^(-(1+psie+psie*thetap)/thetap)*fV(+1); 

//n8 non-lin. pricing 3
aV=psie*thetap/(1+psie+psie*thetap)*yV*lamV+xip*betta*deltaV*(pitilde/piV(+1))*aV(+1);

//n9 non-lin. pricing 4
sV=fV*ptildeV-aV*ptildeV^(1+(1+psie)*(1+thetap)/thetap);

//n10 zero profit condition
varthetaV=1+psie-psie*delta2V;

//n11 aggregate price index
varthetaV=delta3V;

//n12 overall price dispersion index 
pstarV=1/(1+psie)*(varthetaV/delta1V)^((1+thetap)*(1+psie)/thetap)+psie/(1+psie);

//n13 dispersion index 
delta1V^(-(1+thetap)*(1+psie)/thetap)=(1-xip)*ptildeV^(-(1+thetap)*(1+psie)/thetap)+xip*(pitilde/piV*delta1V(-1))^(-(1+thetap)*(1+psie)/thetap); 

//n14 dispersion index 
delta2V=(1-xip)*ptildeV+xip*pitilde/piV*delta2V(-1); 

//n15 dispersion index 
delta3V^(-(1+psie+psie*thetap)/thetap)=(1-xip)*ptildeV^(-(1+psie+psie*thetap)/thetap)+xip*(pitilde/piV*delta3V(-1))^(-(1+psie+psie*thetap)/thetap); 

//n16 marginal cost
mcV=wV/( (1-alfa)*k^alfa*nV^(-alfa) ); 

//n17 Taylor rule
#RVnot=RV_SS*(piV/piV_SS)^gampi*(yV/yVpot)^gamx;
RV=max(1.0000-1e6*(1-imposeZLB),RVnot);

//n18 gov. budget
bGV=RV(-1)/piV*bGV(-1)+gV-tauCV*cV-tauNV*wV*nV-tauV;

//n19 fiscal rule for lump-sum taxes
tauV/yV_SS=tauV_SS/yV_SS+phiT*(bGV(-1)/yV_SS-bGV_SS/yV_SS);

//n20 fiscal rule for distortionary labor income taxes
tauNV/yV_SS=tauN/yV_SS+phiN*(bGV(-1)/yV_SS-bGV_SS/yV_SS);
 
//n21 flex-price: euler equation 
(cVpot-cV_SS*nuV)^(-1/sigma)/(1+tauCV)=betta*deltaV*rrVpot*(cVpot(+1)-cV_SS*nuV(+1))^(-1/sigma)/(1+tauCV_tp1); 

//n22 flex-price: leisure-labor tradeoff 
nVpot^chi/(cVpot-cV_SS*nuV)^(-1/sigma) =(1-tauNVpot)/(1+tauCV)*wVpot; 

//n23 flex-price: wage
wVpot=(1-alfa)/(1+thetap)*k^alfa*nVpot^(-alfa);

//n24 flex-price: resource constraint
cVpot+gV=yVpot;

//n25 flex-price: production
yVpot=k^alfa*nVpot^(1-alfa); 

//n26 flex-price: gov. budget
bGVpot=rrVpot(-1)*bGVpot(-1)+gV-tauCV*cVpot-tauNVpot*wVpot*nVpot-tauVpot;

//n27 flex-price: fiscal rule for lump-sum taxes
tauVpot/yV_SS=tauV_SS/yV_SS+phiT*(bGVpot(-1)/yV_SS-bGV_SS/yV_SS);

//n28 flex-price: fiscal rule for distortionary labor income taxes
tauNVpot/yV_SS=tauN/yV_SS+phiN*(bGVpot(-1)/yV_SS-bGV_SS/yV_SS);

//n29 (extra equation, not listed in manuscript) output gap
xV=log(yV/yVpot);

//n30 (extra equation, not listed in manuscript) real rate
rrV=RV/piV(+1);

//n31 (extra equation, not listed in manuscript) gov. consumption shock
//try adding shocks while at the ZLB instead of unit root....perhaps more stable numerically
(gV-gV_SS)/gV_SS=(1-gAtZLBonly)*rhoG*(gV(-1)-gV_SS)/gV_SS+gAtZLBonly*(gV(-1)-gV_SS)/gV_SS*(RVnot<1.0001)+sigG*gV_eps;

//n32 (extra equation, not listed in manuscript)  consumption demand shock
(nuV-nuV_SS)=rhonu*(nuV(-1)-nuV_SS)+signu*nuV_eps;

//n32 (extra equation, not listed in manuscript)  discount factor shock
(deltaV-1)/1=rhodelta*(deltaV(-1)-1)/1+sigdelta*deltaV_eps;


///////////////////////////////////////////////////////
///linear economy                                    //
///////////////////////////////////////////////////////
//set some fiscal shocks not needed to zero
#tauCVBREVE=0;
#tauCVBREVE_tp1=0;

//l1 labor-leisure tradeoff
chi*nVHAT+(cVHAT-nuVBREVE)/(sigma*(1-nu))=wVHAT-tauNVBREVE/(1-tauN)-tauCVBREVE/(1+tauC);

//l2 euler equation
0=deltaVHAT+RVHAT-piVHAT(+1)-(cVHAT(+1)-nuVBREVE(+1))/(sigma*(1-nu))+(cVHAT-nuVBREVE)/(sigma*(1-nu))-(tauCVBREVE_tp1-tauCVBREVE)/(1+tauC);

//l3 resource constraint
cV_SS*cVHAT+gV_SS*gVHAT=yV_SS*yVHAT;

//l4 production
yVHAT=(1-alfa)*nVHAT;

//l5 Phillips curve
piVHAT=betta*piVHAT(+1)+(1-xip)*(1-xip*betta)/xip/(1-(1+thetap)*psie)*mcVHAT;

//l6 marginal cost
mcVHAT=wVHAT+alfa*nVHAT;

//l7 Taylor rule
#RVHATnot=gampi*piVHAT+gamx*(yVHAT-yVpotHAT);
RVHAT=max( (1.0000-RV_SS)/RV_SS-1e6*(1-imposeZLB),RVHATnot);

//l8 gov. budget
bGVBREVE=RV_SS/piV_SS*bGV_SS*(RVHAT(-1)-piVHAT)+RV_SS/piV_SS*bGVBREVE(-1)
        +gV_SS*gVHAT-tauC*cV_SS*cVHAT-cV_SS*tauCVBREVE-wV_SS*nV_SS*tauNVBREVE
        -tauN*wV_SS*nV_SS*(wVHAT+nVHAT)-tauVBREVE;

//l9 fiscal rule for lump-sum taxes
tauVBREVE=phiT*bGVBREVE(-1);

//l10 fiscal rule for distortionary labor income taxes
tauNVBREVE=phiN*bGVBREVE(-1);

//l11 flex-price, euler equation
0=deltaVHAT+rrVpotHAT-(cVpotHAT(+1)-nuVBREVE(+1))/(sigma*(1-nu))+(cVpotHAT-nuVBREVE)/(sigma*(1-nu))-(tauCVBREVE_tp1-tauCVBREVE)/(1+tauC);

//l12 flex-price, labor/leisure tradeoff
chi*nVpotHAT+(cVpotHAT-nuVBREVE)/(sigma*(1-nu))=wVpotHAT-tauNVpotBREVE/(1-tauN)-tauCVBREVE/(1+tauC);

//l13 flex-price, real wage
wVpotHAT=-alfa*nVpotHAT;

//l4 flex-price, resource constraint
cV_SS*cVpotHAT+gV_SS*gVHAT=yV_SS*yVpotHAT;

//l5 flex-price, production
yVpotHAT=(1-alfa)*nVpotHAT;

//l6 flex-price, gov. budget
bGVpotBREVE=RV_SS/piV_SS*bGV_SS*rrVpotHAT(-1)+RV_SS/piV_SS*bGVpotBREVE(-1)
        +gV_SS*gVHAT-tauC*cV_SS*cVpotHAT-cV_SS*tauCVBREVE-wV_SS*nV_SS*tauNVpotBREVE
        -tauN*wV_SS*nV_SS*(wVpotHAT+nVpotHAT)-tauVpotBREVE;

//l7 flex-price, fiscal rule for lump-sum taxes
tauVpotBREVE=phiT*bGVpotBREVE(-1);

//l8 flex-price, fiscal rule for distortionary labor income taxes
tauNVpotBREVE=phiN*bGVpotBREVE(-1);

//l19 (extra equation, not listed in manuscript)  output gap
xVBREVE=yVHAT-yVpotHAT;

//l20 (extra equation, not listed in manuscript) real rate
rrVHAT=RVHAT-piVHAT(+1);

//l21 (extra equation, not listed in manuscript) gov. consumption shock
gVHAT=(1-gAtZLBonly)*rhoG*gVHAT(-1)+gAtZLBonly*gVHAT(-1)*(RVHATnot<-0.00993)+sigG*gV_eps;

//l22 (extra equation, not listed in manuscript) consumption demand shock
nuVBREVE=rhonu*nuVBREVE(-1)+signu*nuV_eps;

//l23 (extra equation, not listed in manuscript) price dispersion
pstarVHAT=xip*pstarVHAT(-1);

//l24 (extra equation, not listed in manuscript)  discount factor shock
deltaVHAT=rhodelta*deltaVHAT(-1)+sigdelta*deltaV_eps;

end; 
 
//some checks  
resid; steady; check; 

//prepare options for det. simul 
options_.slowc=.25; options_.maxit_=1000; warning off;  
 

//create results
if do_fig==1, //fig1: Cons. demand shock; with and without ZLB
    shocks;   
    var nuV_eps;  %cons. demand   
    periods 1:1; 
    values -1;  
    end;

    %baseline
    simul(periods=250,stack_solve_algo=0); 
    oo_endo_simul_base=oo_.endo_simul; 
    oo_steady_state_base=oo_.steady_state; 

    %perturbation
    imposeZLB_idx=loc(M_.param_names(:,:),'imposeZLB'); 
    M_.params(imposeZLB_idx)=0; 
    steady;resid;check; model_steadystate;    
    simul(periods=250,stack_solve_algo=0); 
    if do_show_unconstained==1,
        oo_endo_simul_perturb=oo_.endo_simul;
    else
        oo_endo_simul_perturb=NaN*oo_.endo_simul; 
    end
    oo_steady_state_perturb=oo_.steady_state; 
     
    %Plotting 
    figtitle=['Impulse Responses to a Consumption Demand Shock - With and Without ZLB']; 
    if do_show_unconstained==1,
        legstring=[{'Nonlinear Model: ZLB imposed'},{'Linear Model: ZLB imposed'},{'Nonlinear Model: ZLB not imposed'},{'Linear Model: ZLB not imposed'}]; 
    else
        legstring=[{'Nonlinear Model'},{'Linear Model'}]; 
    end
    skip_ini_period=0;
    plot_results; 
end


if do_fig==2, //fig2: Discount factor shock; with and without ZLB
    shocks;   
    var deltaV_eps;  %discount factor shock
    periods 1:1; 
    values 1;  
    end;

    %baseline
    simul(periods=250,stack_solve_algo=0); 
    oo_endo_simul_base=oo_.endo_simul; 
    oo_steady_state_base=oo_.steady_state; 

    %perturbation
    imposeZLB_idx=loc(M_.param_names(:,:),'imposeZLB'); 
    M_.params(imposeZLB_idx)=0; 
    steady;resid;check; model_steadystate;    
    simul(periods=250,stack_solve_algo=0); 
    if do_show_unconstained==1,
        oo_endo_simul_perturb=oo_.endo_simul;
    else
        oo_endo_simul_perturb=NaN*oo_.endo_simul; 
    end
    oo_steady_state_perturb=oo_.steady_state; 
     
    %Plotting 
    figtitle=['Impulse Responses to a Discount Factor Shock - With and Without ZLB']; 
    if do_show_unconstained==1,
        legstring=[{'Nonlinear Model: ZLB imposed'},{'Linear Model: ZLB imposed'},{'Nonlinear Model: ZLB not imposed'},{'Linear Model: ZLB not imposed'}]; 
    else
        legstring=[{'Nonlinear Model'},{'Linear Model'}]; 
    end
    skip_ini_period=0;
    plot_results; 
end




if do_fig==3, //fig2: Cons. demand shock vs. cons. demand plus gov. cons. shock; with ZLB
    shocks;   
    var nuV_eps;  %cons. demand   
    periods 1:1; 
    values -1;  
    end;

    %baseline
    simul(periods=250,stack_solve_algo=0); 
    oo_endo_simul_base=oo_.endo_simul; 
    oo_steady_state_base=oo_.steady_state; 


    %perturbation
    shocks;   
    var nuV_eps;  %cons. demand   
    periods 1:1; 
    values -1;  
    var gV_eps;  %gov. consumption  
    periods 1:1; 
    values 1; 
    end;
    
    steady;resid;check; model_steadystate;    
    simul(periods=250,stack_solve_algo=0); 
    oo_endo_simul_perturb=oo_.endo_simul; 
    oo_steady_state_perturb=oo_.steady_state; 
     
    %Plotting 
    figtitle=['Impulse Responses to a Consumption Demand Shock - With and Without Gov. Spending Stimulus']; 
    legstring=[{'Nonlin. Model: Cons.Dem.Shk'},{'Lin. Model: Cons.Dem.Shk'},{'Nonlin. Model: Cons.Dem. & Gov.Spend.Shk'},{'Lin. Model: Cons.Dem. & Gov.Shk'}]; 
    skip_ini_period=0;
    plot_results; 
end

 
if do_fig==4,

    %values of sigdelta to loop through
    sigdeltas = 0.003:.0001:.0066;
    
    %only relevant if you also want to include a government spending shock
    %this is how you change the size of said shock, or any other shock:
    sigG_idx=loc(M_.param_names(:,:),'sigG'); 
    M_.params(sigG_idx)=.0001;

    %Calvo sticky price parameter
    xip_idx=loc(M_.param_names(:,:),'xip'); 
    M_.params(xip_idx)=.667;

    %initialize vectors to count # of ZLB-constrained quarters for each signu
    ZLB_periods_nl = zeros(length(sigdeltas),1);
    ZLB_periods_lin = zeros(length(sigdeltas),1);

    %%%%%%
    %Taylor rule parameters
    gampi_idx=loc(M_.param_names(:,:),'gampi'); 
    M_.params(gampi_idx)=1.5;
    gamx_idx=loc(M_.param_names(:,:),'gamx'); 
    M_.params(gamx_idx)=.125;
    %%%%%%

    %FOR LOOP START
    for i=1:length(sigdeltas);

        %change value of sigdelta for the simulation in this loop:
        sigdelta_idx=loc(M_.param_names(:,:),'sigdelta'); 
        M_.params(sigdelta_idx)=sigdeltas(i);

        shocks;   
            var deltaV_eps;
            periods 1:1; 
            values 1;
            var nuV_eps;  %cons. demand   
            periods 1:1; 
            values 0;  
            var gV_eps;  %gov. consumption  
            periods 1:1; 
            values 0; 
        end;

        %make sure ZLB is imposed
        imposeZLB_idx=loc(M_.param_names(:,:),'imposeZLB');
        M_.params(imposeZLB_idx)=1;

        steady;resid;check; model_steadystate;

        if i > 1,
            %start simulating next time from previous solution; this is here so the model converges quicker.
            oo_.endo_simul=oo_endo_simul_lastiter;
        end;

        %if signus(i) < .2,
        %    options_.slowc = 1;
        %elseif (signus(i) >= .2 && signus(i) < .3),
            options_.slowc = .2;
        %else,
        %    options_.slowc = .1; options_.maxit_=10000;
        %end;

        simul(periods=250,stack_solve_algo=0);

        %save values at last simulation to use as starting point for next iteration
        oo_endo_simul_lastiter=oo_.endo_simul;

        %save values of RV (non-linear nom. int. rate)
        oo_endo_simul_nlIR=oo_.endo_simul(3,1:end);

        %save values of RVHAT (linear nom. int. rate)
        oo_endo_simul_linIR=oo_.endo_simul(15,1:end);

        iii = 0;
        jjj = 0;

        for k = 1:length(oo_endo_simul_nlIR);
            if (abs(oo_endo_simul_nlIR(k) - 1) < .0001),
                iii = iii + 1;
            end;
        end;
        ZLB_periods_nl(i) = iii;

        for k = 1:length(oo_endo_simul_linIR);
            if (abs(oo_endo_simul_linIR(k) + .0100) < .0001),
                jjj = jjj + 1;
            end;
        end;
        ZLB_periods_lin(i) = jjj;

    %FOR LOOP END
    end;

/////THIS VARIABLE DISPLAYS THE NUMBER OF ZLB PERIODS THE LIN AND NONLIN VERSIONS OF THE MODEL HIT GIVEN EACH SIGNU SHOCK.
alltogether = zeros(length(sigdeltas),3);
alltogether(1:end, 1) = ZLB_periods_lin;
alltogether(1:end, 2) = ZLB_periods_nl;
alltogether(1:end, 3) = transpose(sigdeltas);

/////FUNCTION BELOW PRODUCES AN NX2 ARRAY OF ZLB DURATIONS AND CORRESPONDING MAX SIGNU SHOCKS FOR LIN AND NONLIN VERSIONS
/////OF THE MODEL. (FOR DOCUMENTATION, SEE FUNCTION FILE OR TYPE "help getsignus".) USE THESE VALUES OF SIGNU TO REPEATEDLY
/////ESTIMATE THE MODEL, BELOW. THESE VALUES NEED TO BE ENTERED INTO THE do_fig=6 CODE MANUALLY.

[lin_sigdeltas nonlin_sigdeltas] = getsignus(alltogether);

end;


if do_fig==6,

    j = 1;
    for linear = [1 0];

        %% SET PARAMETER VALUES %%%%%%%%%%%%%%%%%%%%%%%%%%%%

        %Calvo sticky price parameter
        xip_idx=loc(M_.param_names(:,:),'xip'); 
        M_.params(xip_idx)=.667;

        %Taylor rule parameters
        gampi_idx=loc(M_.param_names(:,:),'gampi'); 
        M_.params(gampi_idx)=1.5;
        gamx_idx=loc(M_.param_names(:,:),'gamx'); 
        M_.params(gamx_idx)=.125;

        %Government shock parameter
        sigG_idx=loc(M_.param_names(:,:),'sigG'); 
        M_.params(sigG_idx)=.0001;

        %make sure ZLB is imposed
        imposeZLB_idx=loc(M_.param_names(:,:),'imposeZLB');
        M_.params(imposeZLB_idx)=1;


        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

        %values of signu to loop through
        if M_.params(xip_idx) == .667,
            if linear == 1,
                %for psie = -12.2:
                sigdeltas = [0.00100 0.00450 0.00475 0.00500 0.00525 0.00550 0.00575 0.00625 0.00650 0.00675 0.00725 0.00775 0.00800];
                %for psie = 0:
                %sigdeltas = [0.0010 0.0038 0.0040 0.0043 0.0045 0.0047 0.0050 0.0052 0.0055];

            else,
                %Nonlinear model 'breaking points'
                %for psie = -12.2:
                %sigdeltas = [0.00100 0.00600 0.00750 0.00900 0.01050 0.01200 0.01325 0.01450 0.01525 0.01675 0.01775 0.01900 0.02000];%0.01575 max value for 8ZLB
                %this is just the linear 8-quarter trap shock size:
                sigdeltas = [0.00650 0.00650 0.00650 0.00650 0.00650 0.00650 0.00650 0.00650 0.00650];
                %for psie = 0:
                %sigdeltas = [0.0010 0.0039 0.0042 0.0045 0.0048 0.0051 0.0054 0.0057 0.0060];

            end;
            %initialize files in which will save the above vars
            %for psie = -12.2:
            file1 = 'linear_vars667_sd_v2.mat';
            %file2 = 'nonlinear_vars667_sd_v2.mat';
            %for the linear 8ZLB shock size nonlin values:
            file2 = 'nonlinear_vars667_ONEsd_v2.mat';
            %for psie = 0:
            %file1 = 'linear_vars667_sd_psie0_v2.mat';
            %file2 = 'nonlinear_vars667_sd_psie0_v2.mat';
        end;

        %initialize cells in which model's variable values will be stored for each set of parameters
        var_values_lin = cell(length(sigdeltas),1);
        var_values_nonlin = cell(length(sigdeltas),1);
        %with government shock
        var_values_linG = cell(length(sigdeltas),1);
        var_values_nonlinG = cell(length(sigdeltas),1);


        %FOR LOOP START
        for i=1:length(sigdeltas);

            %set value of sigdelta for this iteration
            sigdelta_idx=loc(M_.param_names(:,:),'sigdelta'); 
            M_.params(sigdelta_idx)=sigdeltas(i);

            shocks;   
                var deltaV_eps;
                periods 1:1; 
                values 1;
                var nuV_eps;  %cons. demand   
                periods 1:1; 
                values 0;  
                var gV_eps;  %gov. consumption  
                periods 1:1; 
                values 0; 
            end;

            steady;resid;check; model_steadystate;
            
            options_.slowc = .5;
            
            %start simulating next time from previous solution
            if i > 1,
                oo_.endo_simul=oo_endo_simul_lastiter1;
            end;    

            %reset starting point for simulation when going from big shocks back to small shocks (lin to nonlin)
            if i == 1 & j > 1
                oo_.endo_simul = oo_endo_simul_first;
            end;

            simul(periods=250,stack_solve_algo=0);

            %save values at last simulation
            oo_endo_simul_lastiter1=oo_.endo_simul;

            %after first estimation, save values
            if i == 1 & j == 1
                oo_endo_simul_first = oo_.endo_simul;
            end;

            if linear == 1,
                %save LINEAR output, NO government shock
                oo_endo_simul_lin=oo_.endo_simul([13:24,52,49],1:end);
                var_values_lin{i} = oo_endo_simul_lin;

            else,
                %save NON-LINEAR output, NO government shock
                oo_endo_simul_nl=oo_.endo_simul([1:12,41,29],1:end);
                var_values_nonlin{i} = oo_endo_simul_nl;
            end;

            //////////

            %add in a government shock
            shocks;   
                var deltaV_eps;
                periods 1:1; 
                values 1;
                var nuV_eps;  %cons. demand   
                periods 1:1; 
                values 0;  
                var gV_eps;  %gov. consumption  
                periods 1:1; 
                values 1; 
            end;

            steady;resid;check; model_steadystate;    

            if i > 1,
                %start simulating next time from previous solution
                oo_.endo_simul=oo_endo_simul_lastiter2;
            end;

            %reset starting point for simulation when going from big shocks back to small shocks (lin to nonlin)
            if i == 1 & j > 1
                oo_.endo_simul = oo_endo_simul_first2;
            end;

            simul(periods=250,stack_solve_algo=0); 

            %save values at last simulation
            oo_endo_simul_lastiter2=oo_.endo_simul;

            %after first estimation, save values
            if i == 1 & j == 1
                oo_endo_simul_first2 = oo_.endo_simul;
            end;

            if linear == 1,
                %save LINEAR output, WITH government shock
                oo_endo_simul_linG=oo_.endo_simul([13:24,52,49],1:end);
                var_values_linG{i} = oo_endo_simul_linG;   

            else,
                %save NON-LINEAR output, WITH government shock
                oo_endo_simul_nlG=oo_.endo_simul([1:12,41,29],1:end);
                var_values_nonlinG{i} = oo_endo_simul_nlG;
            end;

        %FOR LOOP END
        end;

        if linear == 1,
            save(file1, 'var_values_lin');
            save(file1, 'var_values_linG', '-append');
            save(file1, 'gy', '-append');
            save(file1, 'bGy', '-append');
        else,
            save(file2, 'var_values_nonlin');
            save(file2, 'var_values_nonlinG', '-append');
            save(file2, 'gy', '-append');
            save(file2, 'bGy', '-append');
        end; 

    j = j+1;
    end;

end;



if do_fig==7,

    %values of signu to loop through
    signus = 0.5100:0.0010:0.5400;
    
    %only relevant if you also want to include a government spending shock
    %this is how you change the size of said shock, or any other shock:
    sigG_idx=loc(M_.param_names(:,:),'sigG'); 
    M_.params(sigG_idx)=.0001;

    %Calvo sticky price parameter
    xip_idx=loc(M_.param_names(:,:),'xip'); 
    M_.params(xip_idx)=.667;

    %initialize vectors to count # of ZLB-constrained quarters for each signu
    ZLB_periods_nl = zeros(length(signus),1);
    ZLB_periods_lin = zeros(length(signus),1);

    %%%%%%
    %Taylor rule parameters
    gampi_idx=loc(M_.param_names(:,:),'gampi'); 
    M_.params(gampi_idx)=1.5;
    gamx_idx=loc(M_.param_names(:,:),'gamx'); 
    M_.params(gamx_idx)=.125;
    %%%%%%

    %FOR LOOP START
    for i=1:length(signus);

        %change value of signu for the simulation in this loop:
        signu_idx=loc(M_.param_names(:,:),'signu'); 
        M_.params(signu_idx)=signus(i);

        shocks;   
            var deltaV_eps;
            periods 1:1; 
            values 0;
            var nuV_eps;  %cons. demand   
            periods 1:1; 
            values -1;  
            var gV_eps;  %gov. consumption  
            periods 1:1; 
            values 0; 
        end;

        %make sure ZLB is imposed
        imposeZLB_idx=loc(M_.param_names(:,:),'imposeZLB');
        M_.params(imposeZLB_idx)=1;

        steady;resid;check; model_steadystate;

        if i > 1,
            %start simulating next time from previous solution; this is here so the model converges quicker.
            oo_.endo_simul=oo_endo_simul_lastiter;
        end;

        %if signus(i) < .2,
        %    options_.slowc = 1;
        %elseif (signus(i) >= .2 && signus(i) < .3),
            options_.slowc = .2;
        %else,
        %    options_.slowc = .1; options_.maxit_=10000;
        %end;

        simul(periods=250,stack_solve_algo=0);

        %save values at last simulation to use as starting point for next iteration
        oo_endo_simul_lastiter=oo_.endo_simul;

        %save values of RV (non-linear nom. int. rate)
        oo_endo_simul_nlIR=oo_.endo_simul(3,1:end);

        %save values of RVHAT (linear nom. int. rate)
        oo_endo_simul_linIR=oo_.endo_simul(15,1:end);

        iii = 0;
        jjj = 0;

        for k = 1:length(oo_endo_simul_nlIR);
            if (abs(oo_endo_simul_nlIR(k) - 1) < .0001),
                iii = iii + 1;
            end;
        end;
        ZLB_periods_nl(i) = iii;

        for k = 1:length(oo_endo_simul_linIR);
            if (abs(oo_endo_simul_linIR(k) + .0100) < .0001),
                jjj = jjj + 1;
            end;
        end;
        ZLB_periods_lin(i) = jjj;

    %FOR LOOP END
    end;



/////THIS VARIABLE DISPLAYS THE NUMBER OF ZLB PERIODS THE LIN AND NONLIN VERSIONS OF THE MODEL HIT GIVEN EACH SIGNU SHOCK.
alltogether = zeros(length(signus),3);
alltogether(1:end, 1) = ZLB_periods_lin;
alltogether(1:end, 2) = ZLB_periods_nl;
alltogether(1:end, 3) = transpose(signus);

/////FUNCTION BELOW PRODUCES AN NX2 ARRAY OF ZLB DURATIONS AND CORRESPONDING MAX SIGNU SHOCKS FOR LIN AND NONLIN VERSIONS
/////OF THE MODEL. (FOR DOCUMENTATION, SEE FUNCTION FILE OR TYPE "help getsignus".) USE THESE VALUES OF SIGNU TO REPEATEDLY
/////ESTIMATE THE MODEL, BELOW. THESE VALUES NEED TO BE ENTERED INTO THE do_fig=6 CODE MANUALLY.

[lin_signus nonlin_signus] = getsignus(alltogether);
end;



if do_fig==8,

    j = 1;
    for linear = [0];

        %% SET PARAMETER VALUES %%%%%%%%%%%%%%%%%%%%%%%%%%%%

        %Calvo sticky price parameter
        xip_idx=loc(M_.param_names(:,:),'xip'); 
        M_.params(xip_idx)=.9;

        %Taylor rule parameters
        gampi_idx=loc(M_.param_names(:,:),'gampi'); 
        M_.params(gampi_idx)=1.5;
        gamx_idx=loc(M_.param_names(:,:),'gamx'); 
        M_.params(gamx_idx)=.125;

        %Government shock parameter
        sigG_idx=loc(M_.param_names(:,:),'sigG'); 
        M_.params(sigG_idx)=.0001;

        %make sure ZLB is imposed
        imposeZLB_idx=loc(M_.param_names(:,:),'imposeZLB');
        M_.params(imposeZLB_idx)=1;


        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

        %values of signu to loop through
        %%% Dixit Stiglitz psie = 0 and xip = 0.9
        if M_.params(xip_idx) == .9,
            if linear == 1,
                %for psie = -12.2:
                %signus = [0.001 0.1215 0.1280 0.1345 0.1420 0.1495 0.1570 0.1655 0.1740 0.1835 0.1930 0.2030 0.2140];
                %for psie = 0:
                signus = [0.001 0.1215 0.1280 0.1350 0.1420 0.1495 0.1570 0.1655 0.1740 0.1835 0.1930 0.2030 0.2140];

            else,
                %Nonlinear model 'breaking points'
                %for psie = -12.2:
                %signus = [0.001 0.1650 0.2045 0.2450 0.2840 0.3205 0.3545 0.3865 0.4165 0.4445 0.4740 0.5030 0.5325];
                %this is just the linear 8-quarter trap shock size:
                %signus = [0.1740 0.1740 0.1740 0.1740 0.1740 0.1740 0.1740 0.1740 0.1740];
                %for psie = 0:
                signus = [0.001 0.1270 0.1385 0.1510 0.1650 0.1810 0.1970 0.2150 0.2330 0.2500 0.2680 0.2850 0.3030];

            end;
            %initialize files in which will save the above vars
            %for psie = -12.2:
            %file1 = 'linear_vars9_sn_v2.mat';
            %file2 = 'nonlinear_vars9_sn_v2.mat';
            %for the linear 8ZLB shock size nonlin values:
            %file2 = 'nonlinear_vars9_ONEsn_v2.mat';
            %for psie = 0:
            %file1 = 'linear_vars9_sn_psie0_v2.mat';
            file2 = 'nonlinear_vars9_sn_psie0_v2.mat';
        end;


        %initialize cells in which model's variable values will be stored for each set of parameters
        var_values_lin = cell(length(signus),1);
        var_values_nonlin = cell(length(signus),1);
        %with government shock
        var_values_linG = cell(length(signus),1);
        var_values_nonlinG = cell(length(signus),1);


        %FOR LOOP START
        for i=1:length(signus);

            %set value of signu for this iteration
            signu_idx=loc(M_.param_names(:,:),'signu'); 
            M_.params(signu_idx)=signus(i);

            shocks;   
                var deltaV_eps;
                periods 1:1; 
                values 0;
                var nuV_eps;  %cons. demand   
                periods 1:1; 
                values -1;  
                var gV_eps;  %gov. consumption  
                periods 1:1; 
                values 0; 
            end;

            steady;resid;check; model_steadystate;
            
            options_.slowc = .5;
            
            %start simulating next time from previous solution
            if i > 1,
                oo_.endo_simul=oo_endo_simul_lastiter1;
            end;    

            %reset starting point for simulation when going from big shocks back to small shocks (lin to nonlin)
            if i == 1 & j > 1
                oo_.endo_simul = oo_endo_simul_first;
            end;

            simul(periods=250,stack_solve_algo=0);
            %save values at last simulation
            oo_endo_simul_lastiter1=oo_.endo_simul;

            %after first estimation, save values
            if i == 1 & j == 1
                oo_endo_simul_first = oo_.endo_simul;
            end;

            if linear == 1,
                %save LINEAR output, NO government shock
                oo_endo_simul_lin=oo_.endo_simul([13:23,57,52,49],1:end);
                var_values_lin{i} = oo_endo_simul_lin;

            else,
                %save NON-LINEAR output, NO government shock
                oo_endo_simul_nl=oo_.endo_simul([1:11,37,41,29],1:end);
                var_values_nonlin{i} = oo_endo_simul_nl;
            end;

            //////////

            %add in a government shock
            shocks;   
                var deltaV_eps;
                periods 1:1; 
                values 0;
                var nuV_eps;  %cons. demand   
                periods 1:1; 
                values -1;  
                var gV_eps;  %gov. consumption  
                periods 1:1; 
                values 1; 
            end;

            steady;resid;check; model_steadystate;    

            if i > 1,
                %start simulating next time from previous solution
                oo_.endo_simul=oo_endo_simul_lastiter2;
            end;

            %reset starting point for simulation when going from big shocks back to small shocks (lin to nonlin)
            if i == 1 & j > 1
                oo_.endo_simul = oo_endo_simul_first2;
            end;

            simul(periods=250,stack_solve_algo=0); 

            %save values at last simulation
            oo_endo_simul_lastiter2=oo_.endo_simul;

            %after first estimation, save values
            if i == 1 & j == 1
                oo_endo_simul_first2 = oo_.endo_simul;
            end;

            if linear == 1,
                %save LINEAR output, WITH government shock
                oo_endo_simul_linG=oo_.endo_simul([13:23,57,52,49],1:end);
                var_values_linG{i} = oo_endo_simul_linG;   

            else,
                %save NON-LINEAR output, WITH government shock
                oo_endo_simul_nlG=oo_.endo_simul([1:11,37,41,29],1:end);
                var_values_nonlinG{i} = oo_endo_simul_nlG;
            end;

        %FOR LOOP END
        end;

        if linear == 1,
            save(file1, 'var_values_lin');
            save(file1, 'var_values_linG', '-append');
            save(file1, 'gy', '-append');
            save(file1, 'bGy', '-append');
        else,
            save(file2, 'var_values_nonlin');
            save(file2, 'var_values_nonlinG', '-append');
            save(file2, 'gy', '-append');
            save(file2, 'bGy', '-append');
        end; 

    j = j+1;
    end;
end