##============================================================================##
# Jianhao Lin, Jiacheng Fan, Yifan Zhang, Liangyuan Chen, "Real-time 
#   Macroeconomic Projection Using Narrative Central Bank Communication", 
#   Journal of Applied Econometrics, forthcoming
##----------------------------------------------------------------------------##
  
# Forecast GDP in real time using PBC communication indices (PCIs) and 
# macro factors.

# Note: 
#   1) Estimate numerous individual models with different lags of predictors.
#   2) Select the best individual model with the smallest out-of-sample (OOS) RMSE.
#   3) Use the best individual model to obtain the combination model.
#   4) Use the combination model to generate the final OOS forecasts.

##----------------------------------------------------------------------------##
# Jiacheng Fan, 2022 (fanjch7@mail2.sysu.edu.cn)
##============================================================================##
library(readxl)
library(midasr)
library(openxlsx)
rm(list = ls())

# Adjust this path to where you stored the files
path = "~/lin-fan-zhang-chen-files"

# Set up paths for reading data and storing results
rawdatpath = paste0(path, "/Data/Raw_Data/")
tmpdatpath = paste0(path, "/Data/Temp_Data/")

setwd(path)


#=============================   Step (1) & (2)  ==============================#
#===============================================================================
# Nowcasting
#===============================================================================
#-------------------------------------
# Load Data
#-------------------------------------
# Target
gdp <- na.omit(read.csv(paste0(rawdatpath, 'Target_GDP.csv'))[,2])

# PCIs: GDP
hdmr_gdp <- read.csv(paste0(tmpdatpath, 'PCIs_GDP_Realtime_Nowcast.csv'), header = FALSE)
zp1 <- hdmr_gdp[, seq(1, (length(hdmr_gdp)-3), 4)]
zo1 <- hdmr_gdp[, seq(2, (length(hdmr_gdp)-2), 4)]

# PCIs: CPI
hdmr_cpi <- read.csv(paste0(tmpdatpath, 'PCIs_CPI_Realtime_Nowcast.csv'), header = FALSE)
zp2 <- hdmr_cpi[, seq(1, (length(hdmr_cpi)-3), 4)]
zo2 <- hdmr_cpi[, seq(2, (length(hdmr_cpi)-2), 4)]

# PCIs: IVA
hdmr_iva <- read.csv(paste0(tmpdatpath, 'PCIs_IVA_Realtime_Nowcast.csv'), header = FALSE)
zp3 <- hdmr_iva[, seq(1, (length(hdmr_iva)-3), 4)]
zo3 <- hdmr_iva[, seq(2, (length(hdmr_iva)-2), 4)]

# PCIs: FAI
hdmr_fai <- read.csv(paste0(tmpdatpath, 'PCIs_FAI_Realtime_Nowcast.csv'), header = FALSE)
zp4 <- hdmr_fai[, seq(1, (length(hdmr_fai)-3), 4)]
zo4 <- hdmr_fai[, seq(2, (length(hdmr_fai)-2), 4)]

# PCIs: M2
hdmr_m2 <- read.csv(paste0(tmpdatpath, 'PCIs_M2_Realtime_Nowcast.csv'), header = FALSE)
zp5 <- hdmr_m2[, seq(1, (length(hdmr_m2)-3), 4)]
zo5 <- hdmr_m2[, seq(2, (length(hdmr_m2)-2), 4)]

# PCIs: PPI
hdmr_ppi <- read.csv(paste0(tmpdatpath, 'PCIs_PPI_Realtime_Nowcast.csv'), header = FALSE)
zp6 <- hdmr_ppi[, seq(1, (length(hdmr_ppi)-3), 4)]
zo6 <- hdmr_ppi[, seq(2, (length(hdmr_ppi)-2), 4)]

# Macro Factors
factor <- read.csv(paste0(tmpdatpath, 'MFactors_Realtime.csv'), header = F)
df1 <- data.frame(factor[, seq(1, (length(factor)-4) ,5)])
df2 <- data.frame(factor[, seq(2, (length(factor)-3), 5)])
df3 <- data.frame(factor[, seq(3, (length(factor)-2), 5)])
df4 <- data.frame(factor[, seq(4, (length(factor)-1), 5)])
df5 <- data.frame(factor[, seq(5, (length(factor)), 5)])


#-------------------------------------
# Set Estimating Windows & Variables
#-------------------------------------
len <- 7            # Initial window size = 7 years
leny <- 4 * len
lenx <- 12 * len

yy = ts(c(gdp[1:leny], gdp[leny+1]), start = c(2003,1), end = c(2003,1)+c(len,0), frequency = 4)
xx1 = ts(c(zp1[1:lenx, 1], zp1[lenx+1, 1], zp1[lenx+2, 2], zp1[lenx+3, 3]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx2 = ts(c(zo1[1:lenx, 1], zo1[lenx+1, 1], zo1[lenx+2, 2], zo1[lenx+3, 3]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx3 = ts(c(zp2[1:lenx, 1], zp2[lenx+1, 1], zp2[lenx+2, 2], zp2[lenx+3, 3]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx4 = ts(c(zo2[1:lenx, 1], zo2[lenx+1, 1], zo2[lenx+2, 2], zo2[lenx+3, 3]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx5 = ts(c(zp3[1:lenx, 1], zp3[lenx+1, 1], zp3[lenx+2, 2], zp3[lenx+3, 3]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx6 = ts(c(zo3[1:lenx, 1], zo3[lenx+1, 1], zo3[lenx+2, 2], zo3[lenx+3, 3]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx7 = ts(c(zp4[1:lenx, 1], zp4[lenx+1, 1], zp4[lenx+2, 2], zp4[lenx+3, 3]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx8 = ts(c(zo4[1:lenx, 1], zo4[lenx+1, 1], zo4[lenx+2, 2], zo4[lenx+3, 3]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx9 = ts(c(zp5[1:lenx, 1], zp5[lenx+1, 1], zp5[lenx+2, 2], zp5[lenx+3, 3]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx10 = ts(c(zo5[1:lenx, 1], zo5[lenx+1, 1], zo5[lenx+2, 2], zo5[lenx+3, 3]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx11 = ts(c(zp6[1:lenx, 1], zp6[lenx+1, 1], zp6[lenx+2, 2], zp6[lenx+3, 3]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx12 = ts(c(zo6[1:lenx, 1], zo6[lenx+1, 1], zo6[lenx+2, 2], zo6[lenx+3, 3]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
zz1 = ts(c(df1[1:lenx, 1], df1[lenx+1, 2], df1[lenx+2, 3], df1[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
zz2 = ts(c(df2[1:lenx, 1], df2[lenx+1, 2], df2[lenx+2, 3], df2[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
zz3 = ts(c(df3[1:lenx, 1], df3[lenx+1, 2], df3[lenx+2, 3], df3[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
zz4 = ts(c(df4[1:lenx, 1], df4[lenx+1, 2], df4[lenx+2, 3], df4[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
zz5 = ts(c(df5[1:lenx, 1], df5[lenx+1, 2], df5[lenx+2, 3], df5[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)


#-------------------------------------
# Estimate and Nowcast (h=0)
#-------------------------------------
Weightfunction <- c()
Ylag <- c()
Xlag <- c()
Leads <- c()

result_rec_now <- list(Weightfunction = c(),
                       Ylag = c(),
                       Xlag = c(),
                       Leads = c(),
                       Recu_df1 = c(),
                       Recu_df2 = c(),
                       Recu_df3 = c(),
                       Recu_df4 = c(),
                       Recu_df5 = c(),
                       Recu_zpos_gdp = c(),
                       Recu_zzero_gdp = c(),
                       Recu_zpos_cpi = c(),
                       Recu_zzero_cpi = c(),
                       Recu_zpos_iva = c(),
                       Recu_zzero_iva = c(),
                       Recu_zpos_fai = c(),
                       Recu_zzero_fai = c(),
                       Recu_zpos_m2 = c(),
                       Recu_zzero_m2 = c(),
                       Recu_zpos_ppi = c(),
                       Recu_zzero_ppi = c())

Ylag.max <- 4
Xlag.max <- 12

for (ylag in seq(1, Ylag.max, 1)) {
  for (xlag in seq(3, Xlag.max, 3)) {
    for (leads in seq(0, 3, 1)) {
      for (weight in c("Exp Almon", "Almon", "Beta", "BetaNN")) {
        result_rec_now$Weightfunction <- append(result_rec_now$Weightfunction, weight)
        result_rec_now$Ylag <- append(result_rec_now$Ylag, ylag)
        result_rec_now$Xlag <- append(result_rec_now$Xlag, xlag)
        result_rec_now$Leads <- append(result_rec_now$Leads, leads)
        
        # Setting
        lagx <- 2 + xlag
        lead <- 3 - leads
        
        # Training
        ## DF1
        model <- try({mdz1 <- if(weight == "Exp Almon") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz1, lead:lagx, 3, nealmon),
                  start = list(zz1=c(1, 1, -0.5)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "Almon") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz1, lead:lagx, 3, almonp),
                  start = list(zz1=c(1, 0, 0, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "Beta") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz1, lead:lagx, 3, nbeta),
                  start = list(zz1=c(1, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "BetaNN") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz1, lead:lagx, 3, nbetaMT),
                  start = list(zz1=c(1, 0, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        }
        ## DF2
        mdz2 <- if(weight == "Exp Almon") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz2, lead:lagx, 3, nealmon),
                  start = list(zz2=c(1, 1, -0.5)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "Almon") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz2, lead:lagx, 3, almonp),
                  start = list(zz2=c(1, 0, 0, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "Beta") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz2, lead:lagx, 3, nbeta),
                  start = list(zz2=c(1, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "BetaNN") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz2, lead:lagx, 3, nbetaMT),
                  start = list(zz2=c(1, 0, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        }
        ## DF3
        mdz3 <- if(weight == "Exp Almon") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz3, lead:lagx, 3, nealmon),
                  start = list(zz3=c(1, 1, -0.5)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "Almon") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz3, lead:lagx, 3, almonp),
                  start = list(zz3=c(1, 0, 0, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "Beta") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz3, lead:lagx, 3, nbeta),
                  start = list(zz3=c(1, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "BetaNN") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz3, lead:lagx, 3, nbetaMT),
                  start = list(zz3=c(1, 0, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        }    
        ## DF4
        mdz4 <- if(weight == "Exp Almon") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz4, lead:lagx, 3, nealmon),
                  start = list(zz4=c(1, 1, -0.5)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "Almon") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz4, lead:lagx, 3, almonp),
                  start = list(zz4=c(1, 0, 0, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "Beta") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz4, lead:lagx, 3, nbeta),
                  start = list(zz4=c(1, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "BetaNN"){
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz4, lead:lagx, 3, nbetaMT),
                  start = list(zz4=c(1, 0, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        }
        ## DF5
        mdz5 <- if(weight == "Exp Almon") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz5, lead:lagx, 3, nealmon),
                  start = list(zz5=c(1, 1, -0.5)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "Almon") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz5, lead:lagx, 3, almonp),
                  start = list(zz5=c(1, 0, 0, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "Beta") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz5, lead:lagx, 3, nbeta),
                  start = list(zz5=c(1, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "BetaNN") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz5, lead:lagx, 3, nbetaMT),
                  start = list(zz5=c(1, 0, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        }
        ## zp1
        mdx1 <- if(weight == "Exp Almon") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx1, lead:lagx, 3, nealmon),
                  start = list(xx1=c(1, 1, -0.5)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "Almon") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx1, lead:lagx, 3, almonp),
                  start = list(xx1=c(1, 0, 0, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "Beta") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx1, lead:lagx, 3, nbeta),
                  start = list(xx1=c(1, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "BetaNN") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx1, lead:lagx, 3, nbetaMT),
                  start = list(xx1=c(1, 0, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        }
        ## zo1
        mdx2 <- if(weight == "Exp Almon") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx2, lead:lagx, 3, nealmon),
                  start = list(xx2=c(1, 1, -0.5)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "Almon") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx2, lead:lagx, 3, almonp),
                  start = list(xx2=c(1, 0, 0, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "Beta") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx2, lead:lagx, 3, nbeta),
                  start = list(xx2=c(1, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "BetaNN") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx2, lead:lagx, 3, nbetaMT),
                  start = list(xx2=c(1, 0, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        }
        ## zp2
        mdx3 <- if(weight == "Exp Almon") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx3, lead:lagx, 3, nealmon),
                  start = list(xx3=c(1, 1, -0.5)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "Almon") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx3, lead:lagx, 3, almonp),
                  start = list(xx3=c(1, 0, 0, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "Beta") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx3, lead:lagx, 3, nbeta),
                  start = list(xx3=c(1, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "BetaNN") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx3, lead:lagx, 3, nbetaMT),
                  start = list(xx3=c(1, 0, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        }
        ## zo2
        mdx4 <- if(weight == "Exp Almon") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx4, lead:lagx, 3, nealmon),
                  start = list(xx4=c(1, 1, -0.5)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "Almon") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx4, lead:lagx, 3, almonp),
                  start = list(xx4=c(1, 0, 0, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "Beta") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx4, lead:lagx, 3, nbeta),
                  start = list(xx4=c(1, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "BetaNN") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx4, lead:lagx, 3, nbetaMT),
                  start = list(xx4=c(1, 0, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        }
        ## zp3
        mdx5 <- if(weight == "Exp Almon") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx5, lead:lagx, 3, nealmon),
                  start = list(xx5=c(1, 1, -0.5)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "Almon") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx5, lead:lagx, 3, almonp),
                  start = list(xx5=c(1, 0, 0, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "Beta") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx5, lead:lagx, 3, nbeta),
                  start = list(xx5=c(1, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "BetaNN") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx5, lead:lagx, 3, nbetaMT),
                  start = list(xx5=c(1, 0, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        }
        ## zo3
        mdx6 <- if(weight == "Exp Almon") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx6, lead:lagx, 3, nealmon),
                  start = list(xx6=c(1, 1, -0.5)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "Almon") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx6, lead:lagx, 3, almonp),
                  start = list(xx6=c(1, 0, 0, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "Beta") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx6, lead:lagx, 3, nbeta),
                  start = list(xx6=c(1, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "BetaNN") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx6, lead:lagx, 3, nbetaMT),
                  start = list(xx6=c(1, 0, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        }
        ## zp4
        mdx7 <- if(weight == "Exp Almon") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx7, lead:lagx, 3, nealmon),
                  start = list(xx7=c(1, 1, -0.5)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "Almon") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx7, lead:lagx, 3, almonp),
                  start = list(xx7=c(1, 0, 0, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "Beta") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx7, lead:lagx, 3, nbeta),
                  start = list(xx7=c(1, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "BetaNN") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx7, lead:lagx, 3, nbetaMT),
                  start = list(xx7=c(1, 0, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        }
        ## zo4
        mdx8 <- if(weight == "Exp Almon") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx8, lead:lagx, 3, nealmon),
                  start = list(xx8=c(1, 1, -0.5)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "Almon") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx8, lead:lagx, 3, almonp),
                  start = list(xx8=c(1, 0, 0, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "Beta") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx8, lead:lagx, 3, nbeta),
                  start = list(xx8=c(1, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "BetaNN") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx8, lead:lagx, 3, nbetaMT),
                  start = list(xx8=c(1, 0, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        }
        ## zp5
        mdx9 <- if(weight == "Exp Almon") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx9, lead:lagx, 3, nealmon),
                  start = list(xx9=c(1, 1, -0.5)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "Almon") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx9, lead:lagx, 3, almonp),
                  start = list(xx9=c(1, 0, 0, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "Beta") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx9, lead:lagx, 3, nbeta),
                  start = list(xx9=c(1, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "BetaNN") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx9, lead:lagx, 3, nbetaMT),
                  start = list(xx9=c(1, 0, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        }
        ## zo5
        mdx10 <- if(weight == "Exp Almon") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx10, lead:lagx, 3, nealmon),
                  start = list(xx10=c(1, 1, -0.5)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "Almon") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx10, lead:lagx, 3, almonp),
                  start = list(xx10=c(1, 0, 0, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "Beta") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx10, lead:lagx, 3, nbeta),
                  start = list(xx10=c(1, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "BetaNN") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx10, lead:lagx, 3, nbetaMT),
                  start = list(xx10=c(1, 0, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        }
        ## zp6
        mdx11 <- if(weight == "Exp Almon") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx11, lead:lagx, 3, nealmon),
                  start = list(xx11=c(1, 1, -0.5)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "Almon") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx11, lead:lagx, 3, almonp),
                  start = list(xx11=c(1, 0, 0, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "Beta") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx11, lead:lagx, 3, nbeta),
                  start = list(xx11=c(1, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "BetaNN") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx11, lead:lagx, 3, nbetaMT),
                  start = list(xx11=c(1, 0, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        }
        ## zo6
        mdx12 <- if(weight == "Exp Almon") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx12, lead:lagx, 3, nealmon),
                  start = list(xx12=c(1, 1, -0.5)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "Almon") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx12, lead:lagx, 3, almonp),
                  start = list(xx12=c(1, 0, 0, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "Beta") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx12, lead:lagx, 3, nbeta),
                  start = list(xx12=c(1, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        } else if (weight == "BetaNN") {
          midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx12, lead:lagx, 3, nbetaMT),
                  start = list(xx12=c(1, 0, 0, 0)),
                  Ofunction = "optim", 
                  method = "Nelder-Mead")
        }
        
        # Predicting
        Out_recu_df_1 <- c()
        Out_recu_df_2 <- c()
        Out_recu_df_3 <- c()
        Out_recu_df_4 <- c()
        Out_recu_df_5 <- c()
        Out_recu_zp1 <- c()
        Out_recu_zo1 <- c()
        Out_recu_zp2 <- c()
        Out_recu_zo2 <- c()
        Out_recu_zp3 <- c()
        Out_recu_zo3 <- c()
        Out_recu_zp4 <- c()
        Out_recu_zo4 <- c()
        Out_recu_zp5 <- c()
        Out_recu_zo5 <- c()
        Out_recu_zp6 <- c()
        Out_recu_zo6 <- c()
        
        for (i in seq(1, length(zp2), 3)) {
          # Sampling
          p = floor((i-1)/3)
          l = ceiling(i/3)
          l1 = l + 1
          m1 = l + leny - 1
          m2 = m1 + 1
          j = i + lenx - 1
          k1 = j + 1
          k2 = j + 2
          k3 = j + 3
          i1 = i + 1
          i2 = i + 2
          i3 = i + 3
          
          # Start & End
          starty = c(2003,1)
          endy = starty + c(len, 0) + c(0,p)
          startx = c(2003,1)
          endx = startx + c(len, 1+i)
          fulldata <- list(
            yy = ts(na.omit(c(gdp[1:m1], gdp[m2])), start = starty, end = endy, frequency = 4), 
            xx1 = ts(c(zp1[1:j, i], zp1[k1, i], zp1[k2, i1], zp1[k3, i2]), start = startx, end = endx, frequency = 12),
            xx2 = ts(c(zo1[1:j, i], zo1[k1, i], zo1[k2, i1], zo1[k3, i2]), start = startx, end = endx, frequency = 12),
            xx3 = ts(c(zp2[1:j, i], zp2[k1, i], zp2[k2, i1], zp2[k3, i2]), start = startx, end = endx, frequency = 12),
            xx4 = ts(c(zo2[1:j, i], zo2[k1, i], zo2[k2, i1], zo2[k3, i2]), start = startx, end = endx, frequency = 12),
            xx5 = ts(c(zp3[1:j, i], zp3[k1, i], zp3[k2, i1], zp3[k3, i2]), start = startx, end = endx, frequency = 12),
            xx6 = ts(c(zo3[1:j, i], zo3[k1, i], zo3[k2, i1], zo3[k3, i2]), start = startx, end = endx, frequency = 12),
            xx7 = ts(c(zp4[1:j, i], zp4[k1, i], zp4[k2, i1], zp4[k3, i2]), start = startx, end = endx, frequency = 12),
            xx8 = ts(c(zo4[1:j, i], zo4[k1, i], zo4[k2, i1], zo4[k3, i2]), start = startx, end = endx, frequency = 12),
            xx9 = ts(c(zp5[1:j, i], zp5[k1, i], zp5[k2, i1], zp5[k3, i2]), start = startx, end = endx, frequency = 12),
            xx10 = ts(c(zo5[1:j, i], zo5[k1, i], zo5[k2, i1], zo5[k3, i2]), start = startx, end = endx, frequency = 12),
            xx11 = ts(c(zp6[1:j, i], zp6[k1, i], zp6[k2, i1], zp6[k3, i2]), start = startx, end = endx, frequency = 12),
            xx12 = ts(c(zo6[1:j, i], zo6[k1, i], zo6[k2, i1], zo6[k3, i2]), start = startx, end = endx, frequency = 12),
            zz1 = ts(c(df1[1:j, i], df1[k1, i1], df1[k2, i2], df1[k3, i3]), start = startx, end = endx, frequency = 12),
            zz2 = ts(c(df2[1:j, i], df2[k1, i1], df2[k2, i2], df2[k3, i3]), start = startx, end = endx, frequency = 12),
            zz3 = ts(c(df3[1:j, i], df3[k1, i1], df3[k2, i2], df3[k3, i3]), start = startx, end = endx, frequency = 12),
            zz4 = ts(c(df4[1:j, i], df4[k1, i1], df4[k2, i2], df4[k3, i3]), start = startx, end = endx, frequency = 12),
            zz5 = ts(c(df5[1:j, i], df5[k1, i1], df5[k2, i2], df5[k3, i3]), start = startx, end = endx, frequency = 12)
          )
          insample <- 1:m1
          outsample <- (1:length(fulldata$yy))[-insample]
          
          ## 3. Combine MF & CB Recuing window
          recu_combmdxz <- try({
            average_forecast(list(mdz1, mdz2, mdz3, mdz4, mdz5, 
                                  mdx1, mdx2, mdx3, mdx4, mdx5, mdx6,
                                  mdx7, mdx8, mdx9, mdx10, mdx11, mdx12), 
                             data = fulldata, insample = insample, outsample = outsample, 
                             type = "recursive",
                             measures = c("MSE"),
                             fweights = c("EW", "BICW", "MSFE", "DMSFE"), 
                             show_progress = FALSE)}, silent = T) 
          if ('try-error' %in% class(recu_combmdxz)) {
            recu_forexz <- rep(NA, 17)
          } else {
            recu_forexz <- recu_combmdxz[["forecast"]]
          }
          
          # Collecting
          Out_recu_df_1 <- append(Out_recu_df_1, recu_forexz[1])
          Out_recu_df_2 <- append(Out_recu_df_2, recu_forexz[2])
          Out_recu_df_3 <- append(Out_recu_df_3, recu_forexz[3])
          Out_recu_df_4 <- append(Out_recu_df_4, recu_forexz[4])
          Out_recu_df_5 <- append(Out_recu_df_5, recu_forexz[5])
          Out_recu_zp1 <- append(Out_recu_zp1, recu_forexz[6])
          Out_recu_zo1 <- append(Out_recu_zo1, recu_forexz[7])
          Out_recu_zp2 <- append(Out_recu_zp2, recu_forexz[8])
          Out_recu_zo2 <- append(Out_recu_zo2, recu_forexz[9])
          Out_recu_zp3 <- append(Out_recu_zp3, recu_forexz[10])
          Out_recu_zo3 <- append(Out_recu_zo3, recu_forexz[11])
          Out_recu_zp4 <- append(Out_recu_zp4, recu_forexz[12])
          Out_recu_zo4 <- append(Out_recu_zo4, recu_forexz[13])
          Out_recu_zp5 <- append(Out_recu_zp5, recu_forexz[14])
          Out_recu_zo5 <- append(Out_recu_zo5, recu_forexz[15])
          Out_recu_zp6 <- append(Out_recu_zp6, recu_forexz[16])
          Out_recu_zo6 <- append(Out_recu_zo6, recu_forexz[17])
        }
        }, silent = T)
        if ('try-error' %in% class(model)) {
          cat("wrong:", ylag, xlag, lead, sep=" ", "\n")
          Rmse_df1 <- NA
          Rmse_df2 <- NA
          Rmse_df3 <- NA
          Rmse_df4 <- NA
          Rmse_df5 <- NA
          Rmse_zpos_gdp <- NA
          Rmse_zzero_gdp <-NA
          Rmse_zpos_cpi <- NA
          Rmse_zzero_cpi <-NA
          Rmse_zpos_iva <- NA
          Rmse_zzero_iva <-NA
          Rmse_zpos_fai <- NA
          Rmse_zzero_fai <-NA
          Rmse_zpos_m2 <- NA
          Rmse_zzero_m2 <- NA
          Rmse_zpos_ppi <- NA
          Rmse_zzero_ppi <-NA
        } else {
          Rmse_df1 <- sqrt(sum((Out_recu_df_1 - gdp[-c(1:leny)])^2) / length(Out_recu_df_1))
          Rmse_df2 <- sqrt(sum((Out_recu_df_2 - gdp[-c(1:leny)])^2) / length(Out_recu_df_2))
          Rmse_df3 <- sqrt(sum((Out_recu_df_3 - gdp[-c(1:leny)])^2) / length(Out_recu_df_3))
          Rmse_df4 <- sqrt(sum((Out_recu_df_4 - gdp[-c(1:leny)])^2) / length(Out_recu_df_4))
          Rmse_df5 <- sqrt(sum((Out_recu_df_5 - gdp[-c(1:leny)])^2) / length(Out_recu_df_5))
          Rmse_zpos_gdp <- sqrt(sum((Out_recu_zp1 - gdp[-c(1:leny)])^2) / length(Out_recu_zp1))
          Rmse_zzero_gdp <- sqrt(sum((Out_recu_zo1 - gdp[-c(1:leny)])^2) / length(Out_recu_zo1))
          Rmse_zpos_cpi <- sqrt(sum((Out_recu_zp2 - gdp[-c(1:leny)])^2) / length(Out_recu_zp2))
          Rmse_zzero_cpi <- sqrt(sum((Out_recu_zo2 - gdp[-c(1:leny)])^2) / length(Out_recu_zo2))
          Rmse_zpos_iva <- sqrt(sum((Out_recu_zp3 - gdp[-c(1:leny)])^2) / length(Out_recu_zp3))
          Rmse_zzero_iva <- sqrt(sum((Out_recu_zo3 - gdp[-c(1:leny)])^2) / length(Out_recu_zo3))
          Rmse_zpos_fai <- sqrt(sum((Out_recu_zp4 - gdp[-c(1:leny)])^2) / length(Out_recu_zp4))
          Rmse_zzero_fai <- sqrt(sum((Out_recu_zo4 - gdp[-c(1:leny)])^2) / length(Out_recu_zo4))
          Rmse_zpos_m2 <- sqrt(sum((Out_recu_zp5 - gdp[-c(1:leny)])^2) / length(Out_recu_zp5))
          Rmse_zzero_m2 <- sqrt(sum((Out_recu_zo5 - gdp[-c(1:leny)])^2) / length(Out_recu_zo5))
          Rmse_zpos_ppi <- sqrt(sum((Out_recu_zp6 - gdp[-c(1:leny)])^2) / length(Out_recu_zp6))
          Rmse_zzero_ppi <- sqrt(sum((Out_recu_zo6 - gdp[-c(1:leny)])^2) / length(Out_recu_zo6))
        }
        result_rec_now$Recu_df1 <- append(result_rec_now$Recu_df1, Rmse_df1)
        result_rec_now$Recu_df2 <- append(result_rec_now$Recu_df2, Rmse_df2)
        result_rec_now$Recu_df3 <- append(result_rec_now$Recu_df3, Rmse_df3)
        result_rec_now$Recu_df4 <- append(result_rec_now$Recu_df4, Rmse_df4)
        result_rec_now$Recu_df5 <- append(result_rec_now$Recu_df5, Rmse_df5)
        result_rec_now$Recu_zpos_gdp <- append(result_rec_now$Recu_zpos_gdp, Rmse_zpos_gdp)
        result_rec_now$Recu_zzero_gdp <- append(result_rec_now$Recu_zzero_gdp, Rmse_zzero_gdp)
        result_rec_now$Recu_zpos_cpi <- append(result_rec_now$Recu_zpos_cpi, Rmse_zpos_cpi)
        result_rec_now$Recu_zzero_cpi <- append(result_rec_now$Recu_zzero_cpi, Rmse_zzero_cpi)
        result_rec_now$Recu_zpos_iva <- append(result_rec_now$Recu_zpos_iva, Rmse_zpos_iva)
        result_rec_now$Recu_zzero_iva <- append(result_rec_now$Recu_zzero_iva, Rmse_zzero_iva)
        result_rec_now$Recu_zpos_fai <- append(result_rec_now$Recu_zpos_fai, Rmse_zpos_fai)
        result_rec_now$Recu_zzero_fai <- append(result_rec_now$Recu_zzero_fai, Rmse_zzero_fai)
        result_rec_now$Recu_zpos_m2 <- append(result_rec_now$Recu_zpos_m2, Rmse_zpos_m2)
        result_rec_now$Recu_zzero_m2 <- append(result_rec_now$Recu_zzero_m2, Rmse_zzero_m2)
        result_rec_now$Recu_zpos_ppi <- append(result_rec_now$Recu_zpos_ppi, Rmse_zpos_ppi)
        result_rec_now$Recu_zzero_ppi <- append(result_rec_now$Recu_zzero_ppi, Rmse_zzero_ppi)
      }
    }
  }
}


#===============================================================================
# Forecasting
#===============================================================================
#-------------------------------------
# Load Data
#-------------------------------------
# Target
gdp <- na.omit(read.csv(paste0(rawdatpath, 'Target_GDP.csv'))[,2])

# PCIs: GDP
hdmr_gdp <- read.csv(paste0(tmpdatpath, 'PCIs_GDP_Realtime.csv'), header = FALSE)
zp1 <- hdmr_gdp[, seq(1, (length(hdmr_gdp)-3), 4)]
zo1 <- hdmr_gdp[, seq(2, (length(hdmr_gdp)-2), 4)]

# PCIs: CPI
hdmr_cpi <- read.csv(paste0(tmpdatpath, 'PCIs_CPI_Realtime.csv'), header = FALSE)
zp2 <- hdmr_cpi[, seq(1, (length(hdmr_cpi)-3), 4)]
zo2 <- hdmr_cpi[, seq(2, (length(hdmr_cpi)-2), 4)]

# PCIs: IVA
hdmr_iva <- read.csv(paste0(tmpdatpath, 'PCIs_IVA_Realtime.csv'), header = FALSE)
zp3 <- hdmr_iva[, seq(1, (length(hdmr_iva)-3), 4)]
zo3 <- hdmr_iva[, seq(2, (length(hdmr_iva)-2), 4)]

# PCIs: FAI
hdmr_fai <- read.csv(paste0(tmpdatpath, 'PCIs_FAI_Realtime.csv'), header = FALSE)
zp4 <- hdmr_fai[, seq(1, (length(hdmr_fai)-3), 4)]
zo4 <- hdmr_fai[, seq(2, (length(hdmr_fai)-2), 4)]

# PCIs: M2
hdmr_m2 <- read.csv(paste0(tmpdatpath, 'PCIs_M2_Realtime.csv'), header = FALSE)
zp5 <- hdmr_m2[, seq(1, (length(hdmr_m2)-3), 4)]
zo5 <- hdmr_m2[, seq(2, (length(hdmr_m2)-2), 4)]

# PCIs: PPI
hdmr_ppi <- read.csv(paste0(tmpdatpath, 'PCIs_PPI_Realtime.csv'), header = FALSE)
zp6 <- hdmr_ppi[, seq(1, (length(hdmr_ppi)-3), 4)]
zo6 <- hdmr_ppi[, seq(2, (length(hdmr_ppi)-2), 4)]

# Macro Factors
factor <- read.csv(paste0(tmpdatpath, 'MFactors_Realtime.csv'), header = F)
df1 <- data.frame(factor[, seq(1, (length(factor)-4) ,5)])
df2 <- data.frame(factor[, seq(2, (length(factor)-3), 5)])
df3 <- data.frame(factor[, seq(3, (length(factor)-2), 5)])
df4 <- data.frame(factor[, seq(4, (length(factor)-1), 5)])
df5 <- data.frame(factor[, seq(5, (length(factor)), 5)])


#-------------------------------------
# Set Estimating Windows & Variables
#-------------------------------------
len <- 7            # Initial window size = 7 years
leny <- 4 * len
lenx <- 12 * len

yy = ts(c(gdp[1:leny], gdp[leny+1]), start = c(2003,1), end = c(2003,1)+c(len,0), frequency = 4)
xx1 = ts(c(zp1[1:lenx, 1], zp1[lenx+1, 2], zp1[lenx+2, 3], zp1[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx2 = ts(c(zo1[1:lenx, 1], zo1[lenx+1, 2], zo1[lenx+2, 3], zo1[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx3 = ts(c(zp2[1:lenx, 1], zp2[lenx+1, 2], zp2[lenx+2, 3], zp2[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx4 = ts(c(zo2[1:lenx, 1], zo2[lenx+1, 2], zo2[lenx+2, 3], zo2[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx5 = ts(c(zp3[1:lenx, 1], zp3[lenx+1, 2], zp3[lenx+2, 3], zp3[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx6 = ts(c(zo3[1:lenx, 1], zo3[lenx+1, 2], zo3[lenx+2, 3], zo3[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx7 = ts(c(zp4[1:lenx, 1], zp4[lenx+1, 2], zp4[lenx+2, 3], zp4[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx8 = ts(c(zo4[1:lenx, 1], zo4[lenx+1, 2], zo4[lenx+2, 3], zo4[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx9 = ts(c(zp5[1:lenx, 1], zp5[lenx+1, 2], zp5[lenx+2, 3], zp5[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx10 = ts(c(zo5[1:lenx, 1], zo5[lenx+1, 2], zo5[lenx+2, 3], zo5[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx11 = ts(c(zp6[1:lenx, 1], zp6[lenx+1, 2], zp6[lenx+2, 3], zp6[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx12 = ts(c(zo6[1:lenx, 1], zo6[lenx+1, 2], zo6[lenx+2, 3], zo6[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
zz1 = ts(c(df1[1:lenx, 1], df1[lenx+1, 2], df1[lenx+2, 3], df1[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
zz2 = ts(c(df2[1:lenx, 1], df2[lenx+1, 2], df2[lenx+2, 3], df2[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
zz3 = ts(c(df3[1:lenx, 1], df3[lenx+1, 2], df3[lenx+2, 3], df3[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
zz4 = ts(c(df4[1:lenx, 1], df4[lenx+1, 2], df4[lenx+2, 3], df4[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
zz5 = ts(c(df5[1:lenx, 1], df5[lenx+1, 2], df5[lenx+2, 3], df5[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)


#-------------------------------------
# Estimate and Forecast (h=1)
#-------------------------------------
Weightfunction <- c()
Ylag <- c()
Xlag <- c()
Tau <- c()
result_rec_for1 <- list(Weightfunction = c(),
                        Ylag = c(),
                        Xlag = c(),
                        Tau = c(),
                        Recu_df1 = c(),
                        Recu_df2 = c(),
                        Recu_df3 = c(),
                        Recu_df4 = c(),
                        Recu_df5 = c(),
                        Recu_zpos_gdp = c(),
                        Recu_zzero_gdp = c(),
                        Recu_zpos_cpi = c(),
                        Recu_zzero_cpi = c(),
                        Recu_zpos_iva = c(),
                        Recu_zzero_iva = c(),
                        Recu_zpos_fai = c(),
                        Recu_zzero_fai = c(),
                        Recu_zpos_m2 = c(),
                        Recu_zzero_m2 = c(),
                        Recu_zpos_ppi = c(),
                        Recu_zzero_ppi = c())

Ylag.max <- 4
Xlag.max <- 12

for (ylag in seq(1, Ylag.max, 1)) {
  for (xlag in seq(3, Xlag.max, 3)) {
    for (weight in c("Exp Almon", "Almon", "Beta", "BetaNN")) {
      result_rec_for1$Weightfunction <- append(result_rec_for1$Weightfunction, weight)
      result_rec_for1$Ylag <- append(result_rec_for1$Ylag, ylag)
      result_rec_for1$Xlag <- append(result_rec_for1$Xlag, xlag)
      result_rec_for1$Tau <- append(result_rec_for1$Tau, 1)
      
      lagx <- 2 + xlag
      lead <- 3
      
      # Training
      ## DF1 (Mon.Freq)
      mdz1 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz1, lead:lagx, 3, nealmon),
                start = list(zz1=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz1, lead:lagx, 3, almonp),
                start = list(zz1=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz1, lead:lagx, 3, nbeta),
                start = list(zz1=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz1, lead:lagx, 3, nbetaMT),
                start = list(zz1=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## DF2 (Mon.Freq)
      mdz2 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz2, lead:lagx, 3, nealmon),
                start = list(zz2=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz2, lead:lagx, 3, almonp),
                start = list(zz2=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz2, lead:lagx, 3, nbeta),
                start = list(zz2=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz2, lead:lagx, 3, nbetaMT),
                start = list(zz2=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## DF3 (Mon.Freq)
      mdz3 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz3, lead:lagx, 3, nealmon),
                start = list(zz3=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz3, lead:lagx, 3, almonp),
                start = list(zz3=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz3, lead:lagx, 3, nbeta),
                start = list(zz3=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz3, lead:lagx, 3, nbetaMT),
                start = list(zz3=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }    
      ## DF4 (Mon.Freq)
      mdz4 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz4, lead:lagx, 3, nealmon),
                start = list(zz4=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz4, lead:lagx, 3, almonp),
                start = list(zz4=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz4, lead:lagx, 3, nbeta),
                start = list(zz4=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz4, lead:lagx, 3, nbetaMT),
                start = list(zz4=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## DF5 (Mon.Freq)
      mdz5 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz5, lead:lagx, 3, nealmon),
                start = list(zz5=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz5, lead:lagx, 3, almonp),
                start = list(zz5=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz5, lead:lagx, 3, nbeta),
                start = list(zz5=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(zz5, lead:lagx, 3, nbetaMT),
                start = list(zz5=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zp1 (Mon.Freq)
      mdx1 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx1, lead:lagx, 3, nealmon),
                start = list(xx1=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx1, lead:lagx, 3, almonp),
                start = list(xx1=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx1, lead:lagx, 3, nbeta),
                start = list(xx1=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx1, lead:lagx, 3, nbetaMT),
                start = list(xx1=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zo1 (Mon.Freq)
      mdx2 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx2, lead:lagx, 3, nealmon),
                start = list(xx2=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx2, lead:lagx, 3, almonp),
                start = list(xx2=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx2, lead:lagx, 3, nbeta),
                start = list(xx2=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx2, lead:lagx, 3, nbetaMT),
                start = list(xx2=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## z_pos_cpi (Mon.Freq)
      mdx3 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx3, lead:lagx, 3, nealmon),
                start = list(xx3=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx3, lead:lagx, 3, almonp),
                start = list(xx3=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx3, lead:lagx, 3, nbeta),
                start = list(xx3=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx3, lead:lagx, 3, nbetaMT),
                start = list(xx3=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## z_zero_cpi (Mon.Freq)
      mdx4 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx4, lead:lagx, 3, nealmon),
                start = list(xx4=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx4, lead:lagx, 3, almonp),
                start = list(xx4=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx4, lead:lagx, 3, nbeta),
                start = list(xx4=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx4, lead:lagx, 3, nbetaMT),
                start = list(xx4=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zp3
      mdx5 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx5, lead:lagx, 3, nealmon),
                start = list(xx5=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx5, lead:lagx, 3, almonp),
                start = list(xx5=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx5, lead:lagx, 3, nbeta),
                start = list(xx5=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx5, lead:lagx, 3, nbetaMT),
                start = list(xx5=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zo3
      mdx6 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx6, lead:lagx, 3, nealmon),
                start = list(xx6=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx6, lead:lagx, 3, almonp),
                start = list(xx6=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx6, lead:lagx, 3, nbeta),
                start = list(xx6=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx6, lead:lagx, 3, nbetaMT),
                start = list(xx6=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zp4
      mdx7 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx7, lead:lagx, 3, nealmon),
                start = list(xx7=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx7, lead:lagx, 3, almonp),
                start = list(xx7=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx7, lead:lagx, 3, nbeta),
                start = list(xx7=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx7, lead:lagx, 3, nbetaMT),
                start = list(xx7=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zo4
      mdx8 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx8, lead:lagx, 3, nealmon),
                start = list(xx8=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx8, lead:lagx, 3, almonp),
                start = list(xx8=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx8, lead:lagx, 3, nbeta),
                start = list(xx8=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx8, lead:lagx, 3, nbetaMT),
                start = list(xx8=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zp5
      mdx9 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx9, lead:lagx, 3, nealmon),
                start = list(xx9=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx9, lead:lagx, 3, almonp),
                start = list(xx9=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx9, lead:lagx, 3, nbeta),
                start = list(xx9=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx9, lead:lagx, 3, nbetaMT),
                start = list(xx9=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zo5
      mdx10 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx10, lead:lagx, 3, nealmon),
                start = list(xx10=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx10, lead:lagx, 3, almonp),
                start = list(xx10=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx10, lead:lagx, 3, nbeta),
                start = list(xx10=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx10, lead:lagx, 3, nbetaMT),
                start = list(xx10=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zp6
      mdx11 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx11, lead:lagx, 3, nealmon),
                start = list(xx11=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx11, lead:lagx, 3, almonp),
                start = list(xx11=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx11, lead:lagx, 3, nbeta),
                start = list(xx11=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx11, lead:lagx, 3, nbetaMT),
                start = list(xx11=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zo6
      mdx12 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx12, lead:lagx, 3, nealmon),
                start = list(xx12=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx12, lead:lagx, 3, almonp),
                start = list(xx12=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx12, lead:lagx, 3, nbeta),
                start = list(xx12=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 1:ylag, 1) +mls(xx12, lead:lagx, 3, nbetaMT),
                start = list(xx12=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      
      # Predicting
      Out_recu_df_1 <- c()
      Out_recu_df_2 <- c()
      Out_recu_df_3 <- c()
      Out_recu_df_4 <- c()
      Out_recu_df_5 <- c()
      Out_recu_zp1 <- c()
      Out_recu_zo1 <- c()
      Out_recu_zp2 <- c()
      Out_recu_zo2 <- c()
      Out_recu_zp3 <- c()
      Out_recu_zo3 <- c()
      Out_recu_zp4 <- c()
      Out_recu_zo4 <- c()
      Out_recu_zp5 <- c()
      Out_recu_zo5 <- c()
      Out_recu_zp6 <- c()
      Out_recu_zo6 <- c()
      
      for (i in seq(1, length(zp2)-3, 3)) {
        # Sampling
        p = floor((i-1)/3)
        l = ceiling(i/3)
        l1 = l + 1
        m1 = l + leny - 1
        m2 = m1 + 1
        j = i + lenx - 1
        k1 = j + 1
        k2 = j + 2
        k3 = j + 3
        i1 = i + 1
        i2 = i + 2
        i3 = i + 3
        
        # Start & End
        starty = c(2003,1)
        endy = starty + c(len, 0) + c(0,p)
        startx = c(2003,1)
        endx = startx + c(len, 1+i)
        fulldata <- list(
          yy = ts(na.omit(c(gdp[1:m1], gdp[m2])), start = starty, end = endy, frequency = 4), 
          xx1 = ts(c(zp1[1:j, i], zp1[k1, i1], zp1[k2, i2], zp1[k3, i3]), start = startx, end = endx, frequency = 12),
          xx2 = ts(c(zo1[1:j, i], zo1[k1, i1], zo1[k2, i2], zo1[k3, i3]), start = startx, end = endx, frequency = 12),
          xx3 = ts(c(zp2[1:j, i], zp2[k1, i1], zp2[k2, i2], zp2[k3, i3]), start = startx, end = endx, frequency = 12),
          xx4 = ts(c(zo2[1:j, i], zo2[k1, i1], zo2[k2, i2], zo2[k3, i3]), start = startx, end = endx, frequency = 12),
          xx5 = ts(c(zp3[1:j, i], zp3[k1, i1], zp3[k2, i2], zp3[k3, i3]), start = startx, end = endx, frequency = 12),
          xx6 = ts(c(zo3[1:j, i], zo3[k1, i1], zo3[k2, i2], zo3[k3, i3]), start = startx, end = endx, frequency = 12),
          xx7 = ts(c(zp4[1:j, i], zp4[k1, i1], zp4[k2, i2], zp4[k3, i3]), start = startx, end = endx, frequency = 12),
          xx8 = ts(c(zo4[1:j, i], zo4[k1, i1], zo4[k2, i2], zo4[k3, i3]), start = startx, end = endx, frequency = 12),
          xx9 = ts(c(zp5[1:j, i], zp5[k1, i1], zp5[k2, i2], zp5[k3, i3]), start = startx, end = endx, frequency = 12),
          xx10 = ts(c(zo5[1:j, i], zo5[k1, i1], zo5[k2, i2], zo5[k3, i3]), start = startx, end = endx, frequency = 12),
          xx11 = ts(c(zp6[1:j, i], zp6[k1, i1], zp6[k2, i2], zp6[k3, i3]), start = startx, end = endx, frequency = 12),
          xx12 = ts(c(zo6[1:j, i], zo6[k1, i1], zo6[k2, i2], zo6[k3, i3]), start = startx, end = endx, frequency = 12),
          zz1 = ts(c(df1[1:j, i], df1[k1, i1], df1[k2, i2], df1[k3, i3]), start = startx, end = endx, frequency = 12),
          zz2 = ts(c(df2[1:j, i], df2[k1, i1], df2[k2, i2], df2[k3, i3]), start = startx, end = endx, frequency = 12),
          zz3 = ts(c(df3[1:j, i], df3[k1, i1], df3[k2, i2], df3[k3, i3]), start = startx, end = endx, frequency = 12),
          zz4 = ts(c(df4[1:j, i], df4[k1, i1], df4[k2, i2], df4[k3, i3]), start = startx, end = endx, frequency = 12),
          zz5 = ts(c(df5[1:j, i], df5[k1, i1], df5[k2, i2], df5[k3, i3]), start = startx, end = endx, frequency = 12)
        )
        insample <- 1:m1
        outsample <- (1:length(fulldata$yy))[-insample]
        
        ## 3. Combine MF & CB
        ### Recuing window
        recu_combmdxz <- try({
          average_forecast(list(mdz1, mdz2, mdz3, mdz4, mdz5, 
                                mdx1, mdx2, mdx3, mdx4, mdx5, mdx6,
                                mdx7, mdx8, mdx9, mdx10, mdx11, mdx12), 
                           data = fulldata, insample = insample, outsample = outsample, 
                           type = "recursive",
                           measures = c("MSE"),
                           fweights = c("EW", "BICW", "MSFE", "DMSFE"), 
                           show_progress = FALSE)}, silent = T) 
        if ('try-error' %in% class(recu_combmdxz)) {
          recu_forexz <- rep(NA, 17)
        } else {
          recu_forexz <- recu_combmdxz[["forecast"]]
        }
        
        # Collecting
        Out_recu_df_1 <- append(Out_recu_df_1, recu_forexz[1])
        Out_recu_df_2 <- append(Out_recu_df_2, recu_forexz[2])
        Out_recu_df_3 <- append(Out_recu_df_3, recu_forexz[3])
        Out_recu_df_4 <- append(Out_recu_df_4, recu_forexz[4])
        Out_recu_df_5 <- append(Out_recu_df_5, recu_forexz[5])
        Out_recu_zp1 <- append(Out_recu_zp1, recu_forexz[6])
        Out_recu_zo1 <- append(Out_recu_zo1, recu_forexz[7])
        Out_recu_zp2 <- append(Out_recu_zp2, recu_forexz[8])
        Out_recu_zo2 <- append(Out_recu_zo2, recu_forexz[9])
        Out_recu_zp3 <- append(Out_recu_zp3, recu_forexz[10])
        Out_recu_zo3 <- append(Out_recu_zo3, recu_forexz[11])
        Out_recu_zp4 <- append(Out_recu_zp4, recu_forexz[12])
        Out_recu_zo4 <- append(Out_recu_zo4, recu_forexz[13])
        Out_recu_zp5 <- append(Out_recu_zp5, recu_forexz[14])
        Out_recu_zo5 <- append(Out_recu_zo5, recu_forexz[15])
        Out_recu_zp6 <- append(Out_recu_zp6, recu_forexz[16])
        Out_recu_zo6 <- append(Out_recu_zo6, recu_forexz[17])
      }
      
      Rmse_df1 <- sqrt(sum((Out_recu_df_1 - gdp[-c(1:leny)])^2) / length(Out_recu_df_1))
      Rmse_df2 <- sqrt(sum((Out_recu_df_2 - gdp[-c(1:leny)])^2) / length(Out_recu_df_2))
      Rmse_df3 <- sqrt(sum((Out_recu_df_3 - gdp[-c(1:leny)])^2) / length(Out_recu_df_3))
      Rmse_df4 <- sqrt(sum((Out_recu_df_4 - gdp[-c(1:leny)])^2) / length(Out_recu_df_4))
      Rmse_df5 <- sqrt(sum((Out_recu_df_5 - gdp[-c(1:leny)])^2) / length(Out_recu_df_5))
      Rmse_zpos_gdp <- sqrt(sum((Out_recu_zp1 - gdp[-c(1:leny)])^2) / length(Out_recu_zp1))
      Rmse_zzero_gdp <- sqrt(sum((Out_recu_zo1 - gdp[-c(1:leny)])^2) / length(Out_recu_zo1))
      Rmse_zpos_cpi <- sqrt(sum((Out_recu_zp2 - gdp[-c(1:leny)])^2) / length(Out_recu_zp2))
      Rmse_zzero_cpi <- sqrt(sum((Out_recu_zo2 - gdp[-c(1:leny)])^2) / length(Out_recu_zo2))
      Rmse_zpos_iva <- sqrt(sum((Out_recu_zp3 - gdp[-c(1:leny)])^2) / length(Out_recu_zp3))
      Rmse_zzero_iva <- sqrt(sum((Out_recu_zo3 - gdp[-c(1:leny)])^2) / length(Out_recu_zo3))
      Rmse_zpos_fai <- sqrt(sum((Out_recu_zp4 - gdp[-c(1:leny)])^2) / length(Out_recu_zp4))
      Rmse_zzero_fai <- sqrt(sum((Out_recu_zo4 - gdp[-c(1:leny)])^2) / length(Out_recu_zo4))
      Rmse_zpos_m2 <- sqrt(sum((Out_recu_zp5 - gdp[-c(1:leny)])^2) / length(Out_recu_zp5))
      Rmse_zzero_m2 <- sqrt(sum((Out_recu_zo5 - gdp[-c(1:leny)])^2) / length(Out_recu_zo5))
      Rmse_zpos_ppi <- sqrt(sum((Out_recu_zp6 - gdp[-c(1:leny)])^2) / length(Out_recu_zp6))
      Rmse_zzero_ppi <- sqrt(sum((Out_recu_zo6 - gdp[-c(1:leny)])^2) / length(Out_recu_zo6))
      
      result_rec_for1$Recu_df1 <- append(result_rec_for1$Recu_df1, Rmse_df1)
      result_rec_for1$Recu_df2 <- append(result_rec_for1$Recu_df2, Rmse_df2)
      result_rec_for1$Recu_df3 <- append(result_rec_for1$Recu_df3, Rmse_df3)
      result_rec_for1$Recu_df4 <- append(result_rec_for1$Recu_df4, Rmse_df4)
      result_rec_for1$Recu_df5 <- append(result_rec_for1$Recu_df5, Rmse_df5)
      result_rec_for1$Recu_zpos_gdp <- append(result_rec_for1$Recu_zpos_gdp, Rmse_zpos_gdp)
      result_rec_for1$Recu_zzero_gdp <- append(result_rec_for1$Recu_zzero_gdp, Rmse_zzero_gdp)
      result_rec_for1$Recu_zpos_cpi <- append(result_rec_for1$Recu_zpos_cpi, Rmse_zpos_cpi)
      result_rec_for1$Recu_zzero_cpi <- append(result_rec_for1$Recu_zzero_cpi, Rmse_zzero_cpi)
      result_rec_for1$Recu_zpos_iva <- append(result_rec_for1$Recu_zpos_iva, Rmse_zpos_iva)
      result_rec_for1$Recu_zzero_iva <- append(result_rec_for1$Recu_zzero_iva, Rmse_zzero_iva)
      result_rec_for1$Recu_zpos_fai <- append(result_rec_for1$Recu_zpos_fai, Rmse_zpos_fai)
      result_rec_for1$Recu_zzero_fai <- append(result_rec_for1$Recu_zzero_fai, Rmse_zzero_fai)
      result_rec_for1$Recu_zpos_m2 <- append(result_rec_for1$Recu_zpos_m2, Rmse_zpos_m2)
      result_rec_for1$Recu_zzero_m2 <- append(result_rec_for1$Recu_zzero_m2, Rmse_zzero_m2)
      result_rec_for1$Recu_zpos_ppi <- append(result_rec_for1$Recu_zpos_ppi, Rmse_zpos_ppi)
      result_rec_for1$Recu_zzero_ppi <- append(result_rec_for1$Recu_zzero_ppi, Rmse_zzero_ppi)
    }
  }
}


#-------------------------------------
# Estimate and Forecast (h=2)
#-------------------------------------
Weightfunction <- c()
Ylag <- c()
Xlag <- c()
Tau <- c()
result_rec_for2 <- list(Weightfunction = c(),
                        Ylag = c(),
                        Xlag = c(),
                        Tau = c(),
                        Recu_df1 = c(),
                        Recu_df2 = c(),
                        Recu_df3 = c(),
                        Recu_df4 = c(),
                        Recu_df5 = c(),
                        Recu_zpos_gdp = c(),
                        Recu_zzero_gdp = c(),
                        Recu_zpos_cpi = c(),
                        Recu_zzero_cpi = c(),
                        Recu_zpos_iva = c(),
                        Recu_zzero_iva = c(),
                        Recu_zpos_fai = c(),
                        Recu_zzero_fai = c(),
                        Recu_zpos_m2 = c(),
                        Recu_zzero_m2 = c(),
                        Recu_zpos_ppi = c(),
                        Recu_zzero_ppi = c())

Ylag.max <- 4
Xlag.max <- 12

for (ylag in seq(1, Ylag.max, 1)) {
  for (xlag in seq(3, Xlag.max, 3)) {
    for (weight in c("Exp Almon", "Almon", "Beta", "BetaNN")) {
      result_rec_for2$Weightfunction <- append(result_rec_for2$Weightfunction, weight)
      result_rec_for2$Ylag <- append(result_rec_for2$Ylag, ylag)
      result_rec_for2$Xlag <- append(result_rec_for2$Xlag, xlag)
      result_rec_for2$Tau <- append(result_rec_for2$Tau, 2)
      
      lagy <- 1 + ylag
      lagx <- 5 + xlag
      lead <- 6
      
      # Training
      ## DF1 (Mon.Freq)
      mdz1 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(zz1, lead:lagx, 3, nealmon),
                start = list(zz1=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(zz1, lead:lagx, 3, almonp),
                start = list(zz1=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(zz1, lead:lagx, 3, nbeta),
                start = list(zz1=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(zz1, lead:lagx, 3, nbetaMT),
                start = list(zz1=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## DF2 (Mon.Freq)
      mdz2 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(zz2, lead:lagx, 3, nealmon),
                start = list(zz2=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(zz2, lead:lagx, 3, almonp),
                start = list(zz2=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(zz2, lead:lagx, 3, nbeta),
                start = list(zz2=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(zz2, lead:lagx, 3, nbetaMT),
                start = list(zz2=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## DF3 (Mon.Freq)
      mdz3 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(zz3, lead:lagx, 3, nealmon),
                start = list(zz3=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(zz3, lead:lagx, 3, almonp),
                start = list(zz3=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(zz3, lead:lagx, 3, nbeta),
                start = list(zz3=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(zz3, lead:lagx, 3, nbetaMT),
                start = list(zz3=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }    
      ## DF4 (Mon.Freq)
      mdz4 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(zz4, lead:lagx, 3, nealmon),
                start = list(zz4=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(zz4, lead:lagx, 3, almonp),
                start = list(zz4=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(zz4, lead:lagx, 3, nbeta),
                start = list(zz4=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(zz4, lead:lagx, 3, nbetaMT),
                start = list(zz4=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## DF5 (Mon.Freq)
      mdz5 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(zz5, lead:lagx, 3, nealmon),
                start = list(zz5=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(zz5, lead:lagx, 3, almonp),
                start = list(zz5=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(zz5, lead:lagx, 3, nbeta),
                start = list(zz5=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(zz5, lead:lagx, 3, nbetaMT),
                start = list(zz5=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## z_pos_gdp (Mon.Freq) 
      mdx1 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx1, lead:lagx, 3, nealmon),
                start = list(xx1=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx1, lead:lagx, 3, almonp),
                start = list(xx1=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx1, lead:lagx, 3, nbeta),
                start = list(xx1=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx1, lead:lagx, 3, nbetaMT),
                start = list(xx1=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## z_zero_gdp (Mon.Freq)
      mdx2 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx2, lead:lagx, 3, nealmon),
                start = list(xx2=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx2, lead:lagx, 3, almonp),
                start = list(xx2=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx2, lead:lagx, 3, nbeta),
                start = list(xx2=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 2:ylag, 1) +mls(xx2, lead:lagx, 3, nbetaMT),
                start = list(xx2=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zp2 (Mon.Freq)
      mdx3 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx3, lead:lagx, 3, nealmon),
                start = list(xx3=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx3, lead:lagx, 3, almonp),
                start = list(xx3=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx3, lead:lagx, 3, nbeta),
                start = list(xx3=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx3, lead:lagx, 3, nbetaMT),
                start = list(xx3=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zo2 (Mon.Freq)
      mdx4 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx4, lead:lagx, 3, nealmon),
                start = list(xx4=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx4, lead:lagx, 3, almonp),
                start = list(xx4=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx4, lead:lagx, 3, nbeta),
                start = list(xx4=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx4, lead:lagx, 3, nbetaMT),
                start = list(xx4=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zp3
      mdx5 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx5, lead:lagx, 3, nealmon),
                start = list(xx5=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx5, lead:lagx, 3, almonp),
                start = list(xx5=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx5, lead:lagx, 3, nbeta),
                start = list(xx5=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx5, lead:lagx, 3, nbetaMT),
                start = list(xx5=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zo3
      mdx6 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx6, lead:lagx, 3, nealmon),
                start = list(xx6=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx6, lead:lagx, 3, almonp),
                start = list(xx6=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx6, lead:lagx, 3, nbeta),
                start = list(xx6=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx6, lead:lagx, 3, nbetaMT),
                start = list(xx6=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zp4
      mdx7 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx7, lead:lagx, 3, nealmon),
                start = list(xx7=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx7, lead:lagx, 3, almonp),
                start = list(xx7=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx7, lead:lagx, 3, nbeta),
                start = list(xx7=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx7, lead:lagx, 3, nbetaMT),
                start = list(xx7=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zo4
      mdx8 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx8, lead:lagx, 3, nealmon),
                start = list(xx8=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx8, lead:lagx, 3, almonp),
                start = list(xx8=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx8, lead:lagx, 3, nbeta),
                start = list(xx8=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx8, lead:lagx, 3, nbetaMT),
                start = list(xx8=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zp5
      mdx9 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx9, lead:lagx, 3, nealmon),
                start = list(xx9=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx9, lead:lagx, 3, almonp),
                start = list(xx9=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx9, lead:lagx, 3, nbeta),
                start = list(xx9=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx9, lead:lagx, 3, nbetaMT),
                start = list(xx9=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zo5
      mdx10 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx10, lead:lagx, 3, nealmon),
                start = list(xx10=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx10, lead:lagx, 3, almonp),
                start = list(xx10=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx10, lead:lagx, 3, nbeta),
                start = list(xx10=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx10, lead:lagx, 3, nbetaMT),
                start = list(xx10=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zp6
      mdx11 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx11, lead:lagx, 3, nealmon),
                start = list(xx11=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx11, lead:lagx, 3, almonp),
                start = list(xx11=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx11, lead:lagx, 3, nbeta),
                start = list(xx11=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx11, lead:lagx, 3, nbetaMT),
                start = list(xx11=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zo6
      mdx12 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx12, lead:lagx, 3, nealmon),
                start = list(xx12=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx12, lead:lagx, 3, almonp),
                start = list(xx12=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 2:lagy, 1) +mls(xx12, lead:lagx, 3, nbeta),
                start = list(xx12=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 2:ylag, 1) +mls(xx12, lead:lagx, 3, nbetaMT),
                start = list(xx12=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      
      # Predicting
      Out_recu_df_1 <- c()
      Out_recu_df_2 <- c()
      Out_recu_df_3 <- c()
      Out_recu_df_4 <- c()
      Out_recu_df_5 <- c()
      Out_recu_zp1 <- c()
      Out_recu_zo1 <- c()
      Out_recu_zp2 <- c()
      Out_recu_zo2 <- c()
      Out_recu_zp3 <- c()
      Out_recu_zo3 <- c()
      Out_recu_zp4 <- c()
      Out_recu_zo4 <- c()
      Out_recu_zp5 <- c()
      Out_recu_zo5 <- c()
      Out_recu_zp6 <- c()
      Out_recu_zo6 <- c()
      
      for (i in seq(1, length(zp2)-3, 3)) {
        # Sampling
        p = floor((i-1)/3)
        l = ceiling(i/3)
        l1 = l + 1
        m1 = l + leny - 1
        m2 = m1 + 1
        j = i + lenx - 1
        k1 = j + 1
        k2 = j + 2
        k3 = j + 3
        i1 = i + 1
        i2 = i + 2
        i3 = i + 3
        
        # Start & End
        starty = c(2003,1)
        endy = starty + c(len, 0) + c(0,p)
        startx = c(2003,1)
        endx = startx + c(len, 1+i)
        fulldata <- list(
          yy = ts(na.omit(c(gdp[1:m1], gdp[m2])), start = starty, end = endy, frequency = 4), 
          xx1 = ts(c(zp1[1:j, i], zp1[k1, i1], zp1[k2, i2], zp1[k3, i3]), start = startx, end = endx, frequency = 12),
          xx2 = ts(c(zo1[1:j, i], zo1[k1, i1], zo1[k2, i2], zo1[k3, i3]), start = startx, end = endx, frequency = 12),
          xx3 = ts(c(zp2[1:j, i], zp2[k1, i1], zp2[k2, i2], zp2[k3, i3]), start = startx, end = endx, frequency = 12),
          xx4 = ts(c(zo2[1:j, i], zo2[k1, i1], zo2[k2, i2], zo2[k3, i3]), start = startx, end = endx, frequency = 12),
          xx5 = ts(c(zp3[1:j, i], zp3[k1, i1], zp3[k2, i2], zp3[k3, i3]), start = startx, end = endx, frequency = 12),
          xx6 = ts(c(zo3[1:j, i], zo3[k1, i1], zo3[k2, i2], zo3[k3, i3]), start = startx, end = endx, frequency = 12),
          xx7 = ts(c(zp4[1:j, i], zp4[k1, i1], zp4[k2, i2], zp4[k3, i3]), start = startx, end = endx, frequency = 12),
          xx8 = ts(c(zo4[1:j, i], zo4[k1, i1], zo4[k2, i2], zo4[k3, i3]), start = startx, end = endx, frequency = 12),
          xx9 = ts(c(zp5[1:j, i], zp5[k1, i1], zp5[k2, i2], zp5[k3, i3]), start = startx, end = endx, frequency = 12),
          xx10 = ts(c(zo5[1:j, i], zo5[k1, i1], zo5[k2, i2], zo5[k3, i3]), start = startx, end = endx, frequency = 12),
          xx11 = ts(c(zp6[1:j, i], zp6[k1, i1], zp6[k2, i2], zp6[k3, i3]), start = startx, end = endx, frequency = 12),
          xx12 = ts(c(zo6[1:j, i], zo6[k1, i1], zo6[k2, i2], zo6[k3, i3]), start = startx, end = endx, frequency = 12),
          zz1 = ts(c(df1[1:j, i], df1[k1, i1], df1[k2, i2], df1[k3, i3]), start = startx, end = endx, frequency = 12),
          zz2 = ts(c(df2[1:j, i], df2[k1, i1], df2[k2, i2], df2[k3, i3]), start = startx, end = endx, frequency = 12),
          zz3 = ts(c(df3[1:j, i], df3[k1, i1], df3[k2, i2], df3[k3, i3]), start = startx, end = endx, frequency = 12),
          zz4 = ts(c(df4[1:j, i], df4[k1, i1], df4[k2, i2], df4[k3, i3]), start = startx, end = endx, frequency = 12),
          zz5 = ts(c(df5[1:j, i], df5[k1, i1], df5[k2, i2], df5[k3, i3]), start = startx, end = endx, frequency = 12)
        )
        insample <- 1:m1
        outsample <- (1:length(fulldata$yy))[-insample]
        
        ## 3. Combine MF & CB
        ### Recuing window
        recu_combmdxz <- try({
          average_forecast(list(mdz1, mdz2, mdz3, mdz4, mdz5, 
                                mdx1, mdx2, mdx3, mdx4, mdx5, mdx6,
                                mdx7, mdx8, mdx9, mdx10, mdx11, mdx12), 
                           data = fulldata, insample = insample, outsample = outsample, 
                           type = "recursive",
                           measures = c("MSE"),
                           fweights = c("EW", "BICW", "MSFE", "DMSFE"), 
                           show_progress = FALSE)}, silent = T) 
        if ('try-error' %in% class(recu_combmdxz)) {
          recu_forexz <- rep(NA, 17)
        } else {
          recu_forexz <- recu_combmdxz[["forecast"]]
        }
        
        # Collecting
        Out_recu_df_1 <- append(Out_recu_df_1, recu_forexz[1])
        Out_recu_df_2 <- append(Out_recu_df_2, recu_forexz[2])
        Out_recu_df_3 <- append(Out_recu_df_3, recu_forexz[3])
        Out_recu_df_4 <- append(Out_recu_df_4, recu_forexz[4])
        Out_recu_df_5 <- append(Out_recu_df_5, recu_forexz[5])
        Out_recu_zp1 <- append(Out_recu_zp1, recu_forexz[6])
        Out_recu_zo1 <- append(Out_recu_zo1, recu_forexz[7])
        Out_recu_zp2 <- append(Out_recu_zp2, recu_forexz[8])
        Out_recu_zo2 <- append(Out_recu_zo2, recu_forexz[9])
        Out_recu_zp3 <- append(Out_recu_zp3, recu_forexz[10])
        Out_recu_zo3 <- append(Out_recu_zo3, recu_forexz[11])
        Out_recu_zp4 <- append(Out_recu_zp4, recu_forexz[12])
        Out_recu_zo4 <- append(Out_recu_zo4, recu_forexz[13])
        Out_recu_zp5 <- append(Out_recu_zp5, recu_forexz[14])
        Out_recu_zo5 <- append(Out_recu_zo5, recu_forexz[15])
        Out_recu_zp6 <- append(Out_recu_zp6, recu_forexz[16])
        Out_recu_zo6 <- append(Out_recu_zo6, recu_forexz[17])
      }
      
      Rmse_df1 <- sqrt(sum((Out_recu_df_1 - gdp[-c(1:leny)])^2) / length(Out_recu_df_1))
      Rmse_df2 <- sqrt(sum((Out_recu_df_2 - gdp[-c(1:leny)])^2) / length(Out_recu_df_2))
      Rmse_df3 <- sqrt(sum((Out_recu_df_3 - gdp[-c(1:leny)])^2) / length(Out_recu_df_3))
      Rmse_df4 <- sqrt(sum((Out_recu_df_4 - gdp[-c(1:leny)])^2) / length(Out_recu_df_4))
      Rmse_df5 <- sqrt(sum((Out_recu_df_5 - gdp[-c(1:leny)])^2) / length(Out_recu_df_5))
      Rmse_zpos_gdp <- sqrt(sum((Out_recu_zp1 - gdp[-c(1:leny)])^2) / length(Out_recu_zp1))
      Rmse_zzero_gdp <- sqrt(sum((Out_recu_zo1 - gdp[-c(1:leny)])^2) / length(Out_recu_zo1))
      Rmse_zpos_cpi <- sqrt(sum((Out_recu_zp2 - gdp[-c(1:leny)])^2) / length(Out_recu_zp2))
      Rmse_zzero_cpi <- sqrt(sum((Out_recu_zo2 - gdp[-c(1:leny)])^2) / length(Out_recu_zo2))
      Rmse_zpos_iva <- sqrt(sum((Out_recu_zp3 - gdp[-c(1:leny)])^2) / length(Out_recu_zp3))
      Rmse_zzero_iva <- sqrt(sum((Out_recu_zo3 - gdp[-c(1:leny)])^2) / length(Out_recu_zo3))
      Rmse_zpos_fai <- sqrt(sum((Out_recu_zp4 - gdp[-c(1:leny)])^2) / length(Out_recu_zp4))
      Rmse_zzero_fai <- sqrt(sum((Out_recu_zo4 - gdp[-c(1:leny)])^2) / length(Out_recu_zo4))
      Rmse_zpos_m2 <- sqrt(sum((Out_recu_zp5 - gdp[-c(1:leny)])^2) / length(Out_recu_zp5))
      Rmse_zzero_m2 <- sqrt(sum((Out_recu_zo5 - gdp[-c(1:leny)])^2) / length(Out_recu_zo5))
      Rmse_zpos_ppi <- sqrt(sum((Out_recu_zp6 - gdp[-c(1:leny)])^2) / length(Out_recu_zp6))
      Rmse_zzero_ppi <- sqrt(sum((Out_recu_zo6 - gdp[-c(1:leny)])^2) / length(Out_recu_zo6))
      
      result_rec_for2$Recu_df1 <- append(result_rec_for2$Recu_df1, Rmse_df1)
      result_rec_for2$Recu_df2 <- append(result_rec_for2$Recu_df2, Rmse_df2)
      result_rec_for2$Recu_df3 <- append(result_rec_for2$Recu_df3, Rmse_df3)
      result_rec_for2$Recu_df4 <- append(result_rec_for2$Recu_df4, Rmse_df4)
      result_rec_for2$Recu_df5 <- append(result_rec_for2$Recu_df5, Rmse_df5)
      result_rec_for2$Recu_zpos_gdp <- append(result_rec_for2$Recu_zpos_gdp, Rmse_zpos_gdp)
      result_rec_for2$Recu_zzero_gdp <- append(result_rec_for2$Recu_zzero_gdp, Rmse_zzero_gdp)
      result_rec_for2$Recu_zpos_cpi <- append(result_rec_for2$Recu_zpos_cpi, Rmse_zpos_cpi)
      result_rec_for2$Recu_zzero_cpi <- append(result_rec_for2$Recu_zzero_cpi, Rmse_zzero_cpi)
      result_rec_for2$Recu_zpos_iva <- append(result_rec_for2$Recu_zpos_iva, Rmse_zpos_iva)
      result_rec_for2$Recu_zzero_iva <- append(result_rec_for2$Recu_zzero_iva, Rmse_zzero_iva)
      result_rec_for2$Recu_zpos_fai <- append(result_rec_for2$Recu_zpos_fai, Rmse_zpos_fai)
      result_rec_for2$Recu_zzero_fai <- append(result_rec_for2$Recu_zzero_fai, Rmse_zzero_fai)
      result_rec_for2$Recu_zpos_m2 <- append(result_rec_for2$Recu_zpos_m2, Rmse_zpos_m2)
      result_rec_for2$Recu_zzero_m2 <- append(result_rec_for2$Recu_zzero_m2, Rmse_zzero_m2)
      result_rec_for2$Recu_zpos_ppi <- append(result_rec_for2$Recu_zpos_ppi, Rmse_zpos_ppi)
      result_rec_for2$Recu_zzero_ppi <- append(result_rec_for2$Recu_zzero_ppi, Rmse_zzero_ppi)
    }
  }
}


#-------------------------------------
# Estimate and Forecast (h=3)
#-------------------------------------
Weightfunction <- c()
Ylag <- c()
Xlag <- c()
Tau <- c()
result_rec_for3 <- list(Weightfunction = c(),
                        Ylag = c(),
                        Xlag = c(),
                        Tau = c(),
                        Recu_df1 = c(),
                        Recu_df2 = c(),
                        Recu_df3 = c(),
                        Recu_df4 = c(),
                        Recu_df5 = c(),
                        Recu_zpos_gdp = c(),
                        Recu_zzero_gdp = c(),
                        Recu_zpos_cpi = c(),
                        Recu_zzero_cpi = c(),
                        Recu_zpos_iva = c(),
                        Recu_zzero_iva = c(),
                        Recu_zpos_fai = c(),
                        Recu_zzero_fai = c(),
                        Recu_zpos_m2 = c(),
                        Recu_zzero_m2 = c(),
                        Recu_zpos_ppi = c(),
                        Recu_zzero_ppi = c())

Ylag.max <- 4
Xlag.max <- 12

for (ylag in seq(1, Ylag.max, 1)) {
  for (xlag in seq(3, Xlag.max, 3)) {
    for (weight in c("Exp Almon", "Almon", "Beta", "BetaNN")) {
      result_rec_for3$Weightfunction <- append(result_rec_for3$Weightfunction, weight)
      result_rec_for3$Ylag <- append(result_rec_for3$Ylag, ylag)
      result_rec_for3$Xlag <- append(result_rec_for3$Xlag, xlag)
      result_rec_for3$Tau <- append(result_rec_for3$Tau, 3)
      
      lagy <- 2 + ylag
      lagx <- 8 + xlag
      lead <- 9
      
      # Training
      ## DF1 (Mon.Freq)
      mdz1 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(zz1, lead:lagx, 3, nealmon),
                start = list(zz1=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(zz1, lead:lagx, 3, almonp),
                start = list(zz1=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(zz1, lead:lagx, 3, nbeta),
                start = list(zz1=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(zz1, lead:lagx, 3, nbetaMT),
                start = list(zz1=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## DF2 (Mon.Freq)
      mdz2 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(zz2, lead:lagx, 3, nealmon),
                start = list(zz2=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(zz2, lead:lagx, 3, almonp),
                start = list(zz2=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(zz2, lead:lagx, 3, nbeta),
                start = list(zz2=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(zz2, lead:lagx, 3, nbetaMT),
                start = list(zz2=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## DF3 (Mon.Freq)
      mdz3 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(zz3, lead:lagx, 3, nealmon),
                start = list(zz3=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(zz3, lead:lagx, 3, almonp),
                start = list(zz3=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(zz3, lead:lagx, 3, nbeta),
                start = list(zz3=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(zz3, lead:lagx, 3, nbetaMT),
                start = list(zz3=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }    
      ## DF4 (Mon.Freq)
      mdz4 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(zz4, lead:lagx, 3, nealmon),
                start = list(zz4=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(zz4, lead:lagx, 3, almonp),
                start = list(zz4=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(zz4, lead:lagx, 3, nbeta),
                start = list(zz4=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(zz4, lead:lagx, 3, nbetaMT),
                start = list(zz4=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## DF5 (Mon.Freq)
      mdz5 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(zz5, lead:lagx, 3, nealmon),
                start = list(zz5=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(zz5, lead:lagx, 3, almonp),
                start = list(zz5=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(zz5, lead:lagx, 3, nbeta),
                start = list(zz5=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(zz5, lead:lagx, 3, nbetaMT),
                start = list(zz5=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## z_pos_gdp (Mon.Freq)
      mdx1 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx1, lead:lagx, 3, nealmon),
                start = list(xx1=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx1, lead:lagx, 3, almonp),
                start = list(xx1=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx1, lead:lagx, 3, nbeta),
                start = list(xx1=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx1, lead:lagx, 3, nbetaMT),
                start = list(xx1=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## z_zero_gdp (Mon.Freq)
      mdx2 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx2, lead:lagx, 3, nealmon),
                start = list(xx2=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx2, lead:lagx, 3, almonp),
                start = list(xx2=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx2, lead:lagx, 3, nbeta),
                start = list(xx2=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx2, lead:lagx, 3, nbetaMT),
                start = list(xx2=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zp2 (Mon.Freq)
      mdx3 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx3, lead:lagx, 3, nealmon),
                start = list(xx3=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx3, lead:lagx, 3, almonp),
                start = list(xx3=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx3, lead:lagx, 3, nbeta),
                start = list(xx3=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx3, lead:lagx, 3, nbetaMT),
                start = list(xx3=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zo2 (Mon.Freq)
      mdx4 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx4, lead:lagx, 3, nealmon),
                start = list(xx4=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx4, lead:lagx, 3, almonp),
                start = list(xx4=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx4, lead:lagx, 3, nbeta),
                start = list(xx4=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx4, lead:lagx, 3, nbetaMT),
                start = list(xx4=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zp3
      mdx5 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx5, lead:lagx, 3, nealmon),
                start = list(xx5=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx5, lead:lagx, 3, almonp),
                start = list(xx5=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx5, lead:lagx, 3, nbeta),
                start = list(xx5=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx5, lead:lagx, 3, nbetaMT),
                start = list(xx5=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zo3
      mdx6 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx6, lead:lagx, 3, nealmon),
                start = list(xx6=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx6, lead:lagx, 3, almonp),
                start = list(xx6=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx6, lead:lagx, 3, nbeta),
                start = list(xx6=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx6, lead:lagx, 3, nbetaMT),
                start = list(xx6=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zp4
      mdx7 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx7, lead:lagx, 3, nealmon),
                start = list(xx7=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx7, lead:lagx, 3, almonp),
                start = list(xx7=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx7, lead:lagx, 3, nbeta),
                start = list(xx7=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx7, lead:lagx, 3, nbetaMT),
                start = list(xx7=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zo4
      mdx8 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx8, lead:lagx, 3, nealmon),
                start = list(xx8=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx8, lead:lagx, 3, almonp),
                start = list(xx8=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx8, lead:lagx, 3, nbeta),
                start = list(xx8=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx8, lead:lagx, 3, nbetaMT),
                start = list(xx8=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zp5
      mdx9 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx9, lead:lagx, 3, nealmon),
                start = list(xx9=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx9, lead:lagx, 3, almonp),
                start = list(xx9=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx9, lead:lagx, 3, nbeta),
                start = list(xx9=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx9, lead:lagx, 3, nbetaMT),
                start = list(xx9=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zo5
      mdx10 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx10, lead:lagx, 3, nealmon),
                start = list(xx10=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx10, lead:lagx, 3, almonp),
                start = list(xx10=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx10, lead:lagx, 3, nbeta),
                start = list(xx10=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx10, lead:lagx, 3, nbetaMT),
                start = list(xx10=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zp6
      mdx11 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx11, lead:lagx, 3, nealmon),
                start = list(xx11=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx11, lead:lagx, 3, almonp),
                start = list(xx11=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx11, lead:lagx, 3, nbeta),
                start = list(xx11=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx11, lead:lagx, 3, nbetaMT),
                start = list(xx11=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zo6
      mdx12 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx12, lead:lagx, 3, nealmon),
                start = list(xx12=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx12, lead:lagx, 3, almonp),
                start = list(xx12=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 3:lagy, 1) +mls(xx12, lead:lagx, 3, nbeta),
                start = list(xx12=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 2:ylag, 1) +mls(xx12, lead:lagx, 3, nbetaMT),
                start = list(xx12=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      
      # Predicting
      Out_recu_df_1 <- c()
      Out_recu_df_2 <- c()
      Out_recu_df_3 <- c()
      Out_recu_df_4 <- c()
      Out_recu_df_5 <- c()
      Out_recu_zp1 <- c()
      Out_recu_zo1 <- c()
      Out_recu_zp2 <- c()
      Out_recu_zo2 <- c()
      Out_recu_zp3 <- c()
      Out_recu_zo3 <- c()
      Out_recu_zp4 <- c()
      Out_recu_zo4 <- c()
      Out_recu_zp5 <- c()
      Out_recu_zo5 <- c()
      Out_recu_zp6 <- c()
      Out_recu_zo6 <- c()
      
      for (i in seq(1, length(zp2)-3, 3)) {
        # Sampling
        p = floor((i-1)/3)
        l = ceiling(i/3)
        l1 = l + 1
        m1 = l + leny - 1
        m2 = m1 + 1
        j = i + lenx - 1
        k1 = j + 1
        k2 = j + 2
        k3 = j + 3
        i1 = i + 1
        i2 = i + 2
        i3 = i + 3
        
        # Start & End
        starty = c(2003,1)
        endy = starty + c(len, 0) + c(0,p)
        startx = c(2003,1)
        endx = startx + c(len, 1+i)
        fulldata <- list(
          yy = ts(na.omit(c(gdp[1:m1], gdp[m2])), start = starty, end = endy, frequency = 4), 
          xx1 = ts(c(zp1[1:j, i], zp1[k1, i1], zp1[k2, i2], zp1[k3, i3]), start = startx, end = endx, frequency = 12),
          xx2 = ts(c(zo1[1:j, i], zo1[k1, i1], zo1[k2, i2], zo1[k3, i3]), start = startx, end = endx, frequency = 12),
          xx3 = ts(c(zp2[1:j, i], zp2[k1, i1], zp2[k2, i2], zp2[k3, i3]), start = startx, end = endx, frequency = 12),
          xx4 = ts(c(zo2[1:j, i], zo2[k1, i1], zo2[k2, i2], zo2[k3, i3]), start = startx, end = endx, frequency = 12),
          xx5 = ts(c(zp3[1:j, i], zp3[k1, i1], zp3[k2, i2], zp3[k3, i3]), start = startx, end = endx, frequency = 12),
          xx6 = ts(c(zo3[1:j, i], zo3[k1, i1], zo3[k2, i2], zo3[k3, i3]), start = startx, end = endx, frequency = 12),
          xx7 = ts(c(zp4[1:j, i], zp4[k1, i1], zp4[k2, i2], zp4[k3, i3]), start = startx, end = endx, frequency = 12),
          xx8 = ts(c(zo4[1:j, i], zo4[k1, i1], zo4[k2, i2], zo4[k3, i3]), start = startx, end = endx, frequency = 12),
          xx9 = ts(c(zp5[1:j, i], zp5[k1, i1], zp5[k2, i2], zp5[k3, i3]), start = startx, end = endx, frequency = 12),
          xx10 = ts(c(zo5[1:j, i], zo5[k1, i1], zo5[k2, i2], zo5[k3, i3]), start = startx, end = endx, frequency = 12),
          xx11 = ts(c(zp6[1:j, i], zp6[k1, i1], zp6[k2, i2], zp6[k3, i3]), start = startx, end = endx, frequency = 12),
          xx12 = ts(c(zo6[1:j, i], zo6[k1, i1], zo6[k2, i2], zo6[k3, i3]), start = startx, end = endx, frequency = 12),
          zz1 = ts(c(df1[1:j, i], df1[k1, i1], df1[k2, i2], df1[k3, i3]), start = startx, end = endx, frequency = 12),
          zz2 = ts(c(df2[1:j, i], df2[k1, i1], df2[k2, i2], df2[k3, i3]), start = startx, end = endx, frequency = 12),
          zz3 = ts(c(df3[1:j, i], df3[k1, i1], df3[k2, i2], df3[k3, i3]), start = startx, end = endx, frequency = 12),
          zz4 = ts(c(df4[1:j, i], df4[k1, i1], df4[k2, i2], df4[k3, i3]), start = startx, end = endx, frequency = 12),
          zz5 = ts(c(df5[1:j, i], df5[k1, i1], df5[k2, i2], df5[k3, i3]), start = startx, end = endx, frequency = 12)
        )
        insample <- 1:m1
        outsample <- (1:length(fulldata$yy))[-insample]
        
        ## 3. Combine MF & CB
        ### Recuing window
        recu_combmdxz <- try({
          average_forecast(list(mdz1, mdz2, mdz3, mdz4, mdz5, 
                                mdx1, mdx2, mdx3, mdx4, mdx5, mdx6,
                                mdx7, mdx8, mdx9, mdx10, mdx11, mdx12), 
                           data = fulldata, insample = insample, outsample = outsample, 
                           type = "recursive",
                           measures = c("MSE"),
                           fweights = c("EW", "BICW", "MSFE", "DMSFE"), 
                           show_progress = FALSE)}, silent = T) 
        if ('try-error' %in% class(recu_combmdxz)) {
          recu_forexz <- rep(NA, 17)
        } else {
          recu_forexz <- recu_combmdxz[["forecast"]]
        }
        
        # Collecting
        Out_recu_df_1 <- append(Out_recu_df_1, recu_forexz[1])
        Out_recu_df_2 <- append(Out_recu_df_2, recu_forexz[2])
        Out_recu_df_3 <- append(Out_recu_df_3, recu_forexz[3])
        Out_recu_df_4 <- append(Out_recu_df_4, recu_forexz[4])
        Out_recu_df_5 <- append(Out_recu_df_5, recu_forexz[5])
        Out_recu_zp1 <- append(Out_recu_zp1, recu_forexz[6])
        Out_recu_zo1 <- append(Out_recu_zo1, recu_forexz[7])
        Out_recu_zp2 <- append(Out_recu_zp2, recu_forexz[8])
        Out_recu_zo2 <- append(Out_recu_zo2, recu_forexz[9])
        Out_recu_zp3 <- append(Out_recu_zp3, recu_forexz[10])
        Out_recu_zo3 <- append(Out_recu_zo3, recu_forexz[11])
        Out_recu_zp4 <- append(Out_recu_zp4, recu_forexz[12])
        Out_recu_zo4 <- append(Out_recu_zo4, recu_forexz[13])
        Out_recu_zp5 <- append(Out_recu_zp5, recu_forexz[14])
        Out_recu_zo5 <- append(Out_recu_zo5, recu_forexz[15])
        Out_recu_zp6 <- append(Out_recu_zp6, recu_forexz[16])
        Out_recu_zo6 <- append(Out_recu_zo6, recu_forexz[17])
      }
      
      Rmse_df1 <- sqrt(sum((Out_recu_df_1 - gdp[-c(1:leny)])^2) / length(Out_recu_df_1))
      Rmse_df2 <- sqrt(sum((Out_recu_df_2 - gdp[-c(1:leny)])^2) / length(Out_recu_df_2))
      Rmse_df3 <- sqrt(sum((Out_recu_df_3 - gdp[-c(1:leny)])^2) / length(Out_recu_df_3))
      Rmse_df4 <- sqrt(sum((Out_recu_df_4 - gdp[-c(1:leny)])^2) / length(Out_recu_df_4))
      Rmse_df5 <- sqrt(sum((Out_recu_df_5 - gdp[-c(1:leny)])^2) / length(Out_recu_df_5))
      Rmse_zpos_gdp <- sqrt(sum((Out_recu_zp1 - gdp[-c(1:leny)])^2) / length(Out_recu_zp1))
      Rmse_zzero_gdp <- sqrt(sum((Out_recu_zo1 - gdp[-c(1:leny)])^2) / length(Out_recu_zo1))
      Rmse_zpos_cpi <- sqrt(sum((Out_recu_zp2 - gdp[-c(1:leny)])^2) / length(Out_recu_zp2))
      Rmse_zzero_cpi <- sqrt(sum((Out_recu_zo2 - gdp[-c(1:leny)])^2) / length(Out_recu_zo2))
      Rmse_zpos_iva <- sqrt(sum((Out_recu_zp3 - gdp[-c(1:leny)])^2) / length(Out_recu_zp3))
      Rmse_zzero_iva <- sqrt(sum((Out_recu_zo3 - gdp[-c(1:leny)])^2) / length(Out_recu_zo3))
      Rmse_zpos_fai <- sqrt(sum((Out_recu_zp4 - gdp[-c(1:leny)])^2) / length(Out_recu_zp4))
      Rmse_zzero_fai <- sqrt(sum((Out_recu_zo4 - gdp[-c(1:leny)])^2) / length(Out_recu_zo4))
      Rmse_zpos_m2 <- sqrt(sum((Out_recu_zp5 - gdp[-c(1:leny)])^2) / length(Out_recu_zp5))
      Rmse_zzero_m2 <- sqrt(sum((Out_recu_zo5 - gdp[-c(1:leny)])^2) / length(Out_recu_zo5))
      Rmse_zpos_ppi <- sqrt(sum((Out_recu_zp6 - gdp[-c(1:leny)])^2) / length(Out_recu_zp6))
      Rmse_zzero_ppi <- sqrt(sum((Out_recu_zo6 - gdp[-c(1:leny)])^2) / length(Out_recu_zo6))
      
      result_rec_for3$Recu_df1 <- append(result_rec_for3$Recu_df1, Rmse_df1)
      result_rec_for3$Recu_df2 <- append(result_rec_for3$Recu_df2, Rmse_df2)
      result_rec_for3$Recu_df3 <- append(result_rec_for3$Recu_df3, Rmse_df3)
      result_rec_for3$Recu_df4 <- append(result_rec_for3$Recu_df4, Rmse_df4)
      result_rec_for3$Recu_df5 <- append(result_rec_for3$Recu_df5, Rmse_df5)
      result_rec_for3$Recu_zpos_gdp <- append(result_rec_for3$Recu_zpos_gdp, Rmse_zpos_gdp)
      result_rec_for3$Recu_zzero_gdp <- append(result_rec_for3$Recu_zzero_gdp, Rmse_zzero_gdp)
      result_rec_for3$Recu_zpos_cpi <- append(result_rec_for3$Recu_zpos_cpi, Rmse_zpos_cpi)
      result_rec_for3$Recu_zzero_cpi <- append(result_rec_for3$Recu_zzero_cpi, Rmse_zzero_cpi)
      result_rec_for3$Recu_zpos_iva <- append(result_rec_for3$Recu_zpos_iva, Rmse_zpos_iva)
      result_rec_for3$Recu_zzero_iva <- append(result_rec_for3$Recu_zzero_iva, Rmse_zzero_iva)
      result_rec_for3$Recu_zpos_fai <- append(result_rec_for3$Recu_zpos_fai, Rmse_zpos_fai)
      result_rec_for3$Recu_zzero_fai <- append(result_rec_for3$Recu_zzero_fai, Rmse_zzero_fai)
      result_rec_for3$Recu_zpos_m2 <- append(result_rec_for3$Recu_zpos_m2, Rmse_zpos_m2)
      result_rec_for3$Recu_zzero_m2 <- append(result_rec_for3$Recu_zzero_m2, Rmse_zzero_m2)
      result_rec_for3$Recu_zpos_ppi <- append(result_rec_for3$Recu_zpos_ppi, Rmse_zpos_ppi)
      result_rec_for3$Recu_zzero_ppi <- append(result_rec_for3$Recu_zzero_ppi, Rmse_zzero_ppi)
    }
  }
}


#-------------------------------------
# Estimate and Forecast (h=4)
#-------------------------------------
Weightfunction <- c()
Ylag <- c()
Xlag <- c()
Tau <- c()
result_rec_for4 <- list(Weightfunction = c(),
                        Ylag = c(),
                        Xlag = c(),
                        Tau = c(),
                        Recu_df1 = c(),
                        Recu_df2 = c(),
                        Recu_df3 = c(),
                        Recu_df4 = c(),
                        Recu_df5 = c(),
                        Recu_zpos_gdp = c(),
                        Recu_zzero_gdp = c(),
                        Recu_zpos_cpi = c(),
                        Recu_zzero_cpi = c(),
                        Recu_zpos_iva = c(),
                        Recu_zzero_iva = c(),
                        Recu_zpos_fai = c(),
                        Recu_zzero_fai = c(),
                        Recu_zpos_m2 = c(),
                        Recu_zzero_m2 = c(),
                        Recu_zpos_ppi = c(),
                        Recu_zzero_ppi = c())

Ylag.max <- 4
Xlag.max <- 12

for (ylag in seq(1, Ylag.max, 1)) {
  for (xlag in seq(3, Xlag.max, 3)) {
    for (weight in c("Exp Almon", "Almon", "Beta", "BetaNN")) {
      result_rec_for4$Weightfunction <- append(result_rec_for4$Weightfunction, weight)
      result_rec_for4$Ylag <- append(result_rec_for4$Ylag, ylag)
      result_rec_for4$Xlag <- append(result_rec_for4$Xlag, xlag)
      result_rec_for4$Tau <- append(result_rec_for4$Tau, 4)
      
      lagy <- 3 + ylag
      lagx <- 11 + xlag
      lead <- 12
      
      # Training
      ## DF1 (Mon.Freq)
      mdz1 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(zz1, lead:lagx, 3, nealmon),
                start = list(zz1=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(zz1, lead:lagx, 3, almonp),
                start = list(zz1=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(zz1, lead:lagx, 3, nbeta),
                start = list(zz1=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(zz1, lead:lagx, 3, nbetaMT),
                start = list(zz1=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## DF2 (Mon.Freq)
      mdz2 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(zz2, lead:lagx, 3, nealmon),
                start = list(zz2=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(zz2, lead:lagx, 3, almonp),
                start = list(zz2=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(zz2, lead:lagx, 3, nbeta),
                start = list(zz2=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(zz2, lead:lagx, 3, nbetaMT),
                start = list(zz2=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## DF3 (Mon.Freq)
      mdz3 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(zz3, lead:lagx, 3, nealmon),
                start = list(zz3=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(zz3, lead:lagx, 3, almonp),
                start = list(zz3=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(zz3, lead:lagx, 3, nbeta),
                start = list(zz3=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(zz3, lead:lagx, 3, nbetaMT),
                start = list(zz3=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }    
      ## DF4 (Mon.Freq)
      mdz4 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(zz4, lead:lagx, 3, nealmon),
                start = list(zz4=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(zz4, lead:lagx, 3, almonp),
                start = list(zz4=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(zz4, lead:lagx, 3, nbeta),
                start = list(zz4=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(zz4, lead:lagx, 3, nbetaMT),
                start = list(zz4=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## DF5 (Mon.Freq)
      mdz5 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(zz5, lead:lagx, 3, nealmon),
                start = list(zz5=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(zz5, lead:lagx, 3, almonp),
                start = list(zz5=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(zz5, lead:lagx, 3, nbeta),
                start = list(zz5=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(zz5, lead:lagx, 3, nbetaMT),
                start = list(zz5=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## z_pos_gdp (Mon.Freq) 
      mdx1 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 4:ylag, 1) +mls(xx1, lead:lagx, 3, nealmon),
                start = list(xx1=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 4:ylag, 1) +mls(xx1, lead:lagx, 3, almonp),
                start = list(xx1=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 4:ylag, 1) +mls(xx1, lead:lagx, 3, nbeta),
                start = list(xx1=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 4:ylag, 1) +mls(xx1, lead:lagx, 3, nbetaMT),
                start = list(xx1=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## z_zero_gdp (Mon.Freq)
      mdx2 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 4:ylag, 1) +mls(xx2, lead:lagx, 3, nealmon),
                start = list(xx2=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 4:ylag, 1) +mls(xx2, lead:lagx, 3, almonp),
                start = list(xx2=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 4:ylag, 1) +mls(xx2, lead:lagx, 3, nbeta),
                start = list(xx2=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 4:ylag, 1) +mls(xx2, lead:lagx, 3, nbetaMT),
                start = list(xx2=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zp2 (Mon.Freq)
      mdx3 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx3, lead:lagx, 3, nealmon),
                start = list(xx3=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx3, lead:lagx, 3, almonp),
                start = list(xx3=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx3, lead:lagx, 3, nbeta),
                start = list(xx3=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx3, lead:lagx, 3, nbetaMT),
                start = list(xx3=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zo2 (Mon.Freq)
      mdx4 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx4, lead:lagx, 3, nealmon),
                start = list(xx4=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx4, lead:lagx, 3, almonp),
                start = list(xx4=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx4, lead:lagx, 3, nbeta),
                start = list(xx4=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx4, lead:lagx, 3, nbetaMT),
                start = list(xx4=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zp3
      mdx5 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx5, lead:lagx, 3, nealmon),
                start = list(xx5=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx5, lead:lagx, 3, almonp),
                start = list(xx5=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx5, lead:lagx, 3, nbeta),
                start = list(xx5=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx5, lead:lagx, 3, nbetaMT),
                start = list(xx5=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zo3
      mdx6 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx6, lead:lagx, 3, nealmon),
                start = list(xx6=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx6, lead:lagx, 3, almonp),
                start = list(xx6=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx6, lead:lagx, 3, nbeta),
                start = list(xx6=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx6, lead:lagx, 3, nbetaMT),
                start = list(xx6=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zp4
      mdx7 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx7, lead:lagx, 3, nealmon),
                start = list(xx7=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx7, lead:lagx, 3, almonp),
                start = list(xx7=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx7, lead:lagx, 3, nbeta),
                start = list(xx7=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx7, lead:lagx, 3, nbetaMT),
                start = list(xx7=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zo4
      mdx8 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx8, lead:lagx, 3, nealmon),
                start = list(xx8=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx8, lead:lagx, 3, almonp),
                start = list(xx8=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx8, lead:lagx, 3, nbeta),
                start = list(xx8=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx8, lead:lagx, 3, nbetaMT),
                start = list(xx8=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zp5
      mdx9 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx9, lead:lagx, 3, nealmon),
                start = list(xx9=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx9, lead:lagx, 3, almonp),
                start = list(xx9=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx9, lead:lagx, 3, nbeta),
                start = list(xx9=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx9, lead:lagx, 3, nbetaMT),
                start = list(xx9=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zo5
      mdx10 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx10, lead:lagx, 3, nealmon),
                start = list(xx10=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx10, lead:lagx, 3, almonp),
                start = list(xx10=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx10, lead:lagx, 3, nbeta),
                start = list(xx10=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx10, lead:lagx, 3, nbetaMT),
                start = list(xx10=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zp6
      mdx11 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx11, lead:lagx, 3, nealmon),
                start = list(xx11=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx11, lead:lagx, 3, almonp),
                start = list(xx11=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx11, lead:lagx, 3, nbeta),
                start = list(xx11=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx11, lead:lagx, 3, nbetaMT),
                start = list(xx11=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      ## zo6
      mdx12 <- if(weight == "Exp Almon") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx12, lead:lagx, 3, nealmon),
                start = list(xx12=c(1, 1, -0.5)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Almon") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx12, lead:lagx, 3, almonp),
                start = list(xx12=c(1, 0, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "Beta") {
        midas_r(yy ~ mls(yy, 4:lagy, 1) +mls(xx12, lead:lagx, 3, nbeta),
                start = list(xx12=c(1, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      } else if (weight == "BetaNN") {
        midas_r(yy ~ mls(yy, 2:ylag, 1) +mls(xx12, lead:lagx, 3, nbetaMT),
                start = list(xx12=c(1, 0, 0, 0)),
                Ofunction = "optim", 
                method = "Nelder-Mead")
      }
      
      # Predicting
      Out_recu_df_1 <- c()
      Out_recu_df_2 <- c()
      Out_recu_df_3 <- c()
      Out_recu_df_4 <- c()
      Out_recu_df_5 <- c()
      Out_recu_zp1 <- c()
      Out_recu_zo1 <- c()
      Out_recu_zp2 <- c()
      Out_recu_zo2 <- c()
      Out_recu_zp3 <- c()
      Out_recu_zo3 <- c()
      Out_recu_zp4 <- c()
      Out_recu_zo4 <- c()
      Out_recu_zp5 <- c()
      Out_recu_zo5 <- c()
      Out_recu_zp6 <- c()
      Out_recu_zo6 <- c()
      
      for (i in seq(1, length(zp2)-3, 3)) {
        # Sampling
        p = floor((i-1)/3)
        l = ceiling(i/3)
        l1 = l + 1
        m1 = l + leny - 1
        m2 = m1 + 1
        j = i + lenx - 1
        k1 = j + 1
        k2 = j + 2
        k3 = j + 3
        i1 = i + 1
        i2 = i + 2
        i3 = i + 3
        
        # Start & End
        starty = c(2003,1)
        endy = starty + c(len, 0) + c(0,p)
        startx = c(2003,1)
        endx = startx + c(len, 1+i)
        fulldata <- list(
          yy = ts(na.omit(c(gdp[1:m1], gdp[m2])), start = starty, end = endy, frequency = 4), 
          xx1 = ts(c(zp1[1:j, i], zp1[k1, i1], zp1[k2, i2], zp1[k3, i3]), start = startx, end = endx, frequency = 12),
          xx2 = ts(c(zo1[1:j, i], zo1[k1, i1], zo1[k2, i2], zo1[k3, i3]), start = startx, end = endx, frequency = 12),
          xx3 = ts(c(zp2[1:j, i], zp2[k1, i1], zp2[k2, i2], zp2[k3, i3]), start = startx, end = endx, frequency = 12),
          xx4 = ts(c(zo2[1:j, i], zo2[k1, i1], zo2[k2, i2], zo2[k3, i3]), start = startx, end = endx, frequency = 12),
          xx5 = ts(c(zp3[1:j, i], zp3[k1, i1], zp3[k2, i2], zp3[k3, i3]), start = startx, end = endx, frequency = 12),
          xx6 = ts(c(zo3[1:j, i], zo3[k1, i1], zo3[k2, i2], zo3[k3, i3]), start = startx, end = endx, frequency = 12),
          xx7 = ts(c(zp4[1:j, i], zp4[k1, i1], zp4[k2, i2], zp4[k3, i3]), start = startx, end = endx, frequency = 12),
          xx8 = ts(c(zo4[1:j, i], zo4[k1, i1], zo4[k2, i2], zo4[k3, i3]), start = startx, end = endx, frequency = 12),
          xx9 = ts(c(zp5[1:j, i], zp5[k1, i1], zp5[k2, i2], zp5[k3, i3]), start = startx, end = endx, frequency = 12),
          xx10 = ts(c(zo5[1:j, i], zo5[k1, i1], zo5[k2, i2], zo5[k3, i3]), start = startx, end = endx, frequency = 12),
          xx11 = ts(c(zp6[1:j, i], zp6[k1, i1], zp6[k2, i2], zp6[k3, i3]), start = startx, end = endx, frequency = 12),
          xx12 = ts(c(zo6[1:j, i], zo6[k1, i1], zo6[k2, i2], zo6[k3, i3]), start = startx, end = endx, frequency = 12),
          zz1 = ts(c(df1[1:j, i], df1[k1, i1], df1[k2, i2], df1[k3, i3]), start = startx, end = endx, frequency = 12),
          zz2 = ts(c(df2[1:j, i], df2[k1, i1], df2[k2, i2], df2[k3, i3]), start = startx, end = endx, frequency = 12),
          zz3 = ts(c(df3[1:j, i], df3[k1, i1], df3[k2, i2], df3[k3, i3]), start = startx, end = endx, frequency = 12),
          zz4 = ts(c(df4[1:j, i], df4[k1, i1], df4[k2, i2], df4[k3, i3]), start = startx, end = endx, frequency = 12),
          zz5 = ts(c(df5[1:j, i], df5[k1, i1], df5[k2, i2], df5[k3, i3]), start = startx, end = endx, frequency = 12)
        )
        insample <- 1:m1
        outsample <- (1:length(fulldata$yy))[-insample]
        
        ## 3. Combine MF & CB
        ### Recuing window
        recu_combmdxz <- try({
          average_forecast(list(mdz1, mdz2, mdz3, mdz4, mdz5, 
                                mdx1, mdx2, mdx3, mdx4, mdx5, mdx6,
                                mdx7, mdx8, mdx9, mdx10, mdx11, mdx12), 
                           data = fulldata, insample = insample, outsample = outsample, 
                           type = "recursive",
                           measures = c("MSE"),
                           fweights = c("EW", "BICW", "MSFE", "DMSFE"), 
                           show_progress = FALSE)}, silent = T) 
        if ('try-error' %in% class(recu_combmdxz)) {
          recu_forexz <- rep(NA, 17)
        } else {
          recu_forexz <- recu_combmdxz[["forecast"]]
        }
        
        # Collecting
        Out_recu_df_1 <- append(Out_recu_df_1, recu_forexz[1])
        Out_recu_df_2 <- append(Out_recu_df_2, recu_forexz[2])
        Out_recu_df_3 <- append(Out_recu_df_3, recu_forexz[3])
        Out_recu_df_4 <- append(Out_recu_df_4, recu_forexz[4])
        Out_recu_df_5 <- append(Out_recu_df_5, recu_forexz[5])
        Out_recu_zp1 <- append(Out_recu_zp1, recu_forexz[6])
        Out_recu_zo1 <- append(Out_recu_zo1, recu_forexz[7])
        Out_recu_zp2 <- append(Out_recu_zp2, recu_forexz[8])
        Out_recu_zo2 <- append(Out_recu_zo2, recu_forexz[9])
        Out_recu_zp3 <- append(Out_recu_zp3, recu_forexz[10])
        Out_recu_zo3 <- append(Out_recu_zo3, recu_forexz[11])
        Out_recu_zp4 <- append(Out_recu_zp4, recu_forexz[12])
        Out_recu_zo4 <- append(Out_recu_zo4, recu_forexz[13])
        Out_recu_zp5 <- append(Out_recu_zp5, recu_forexz[14])
        Out_recu_zo5 <- append(Out_recu_zo5, recu_forexz[15])
        Out_recu_zp6 <- append(Out_recu_zp6, recu_forexz[16])
        Out_recu_zo6 <- append(Out_recu_zo6, recu_forexz[17])
      }
      
      Rmse_df1 <- sqrt(sum((Out_recu_df_1 - gdp[-c(1:leny)])^2) / length(Out_recu_df_1))
      Rmse_df2 <- sqrt(sum((Out_recu_df_2 - gdp[-c(1:leny)])^2) / length(Out_recu_df_2))
      Rmse_df3 <- sqrt(sum((Out_recu_df_3 - gdp[-c(1:leny)])^2) / length(Out_recu_df_3))
      Rmse_df4 <- sqrt(sum((Out_recu_df_4 - gdp[-c(1:leny)])^2) / length(Out_recu_df_4))
      Rmse_df5 <- sqrt(sum((Out_recu_df_5 - gdp[-c(1:leny)])^2) / length(Out_recu_df_5))
      Rmse_zpos_gdp <- sqrt(sum((Out_recu_zp1 - gdp[-c(1:leny)])^2) / length(Out_recu_zp1))
      Rmse_zzero_gdp <- sqrt(sum((Out_recu_zo1 - gdp[-c(1:leny)])^2) / length(Out_recu_zo1))
      Rmse_zpos_cpi <- sqrt(sum((Out_recu_zp2 - gdp[-c(1:leny)])^2) / length(Out_recu_zp2))
      Rmse_zzero_cpi <- sqrt(sum((Out_recu_zo2 - gdp[-c(1:leny)])^2) / length(Out_recu_zo2))
      Rmse_zpos_iva <- sqrt(sum((Out_recu_zp3 - gdp[-c(1:leny)])^2) / length(Out_recu_zp3))
      Rmse_zzero_iva <- sqrt(sum((Out_recu_zo3 - gdp[-c(1:leny)])^2) / length(Out_recu_zo3))
      Rmse_zpos_fai <- sqrt(sum((Out_recu_zp4 - gdp[-c(1:leny)])^2) / length(Out_recu_zp4))
      Rmse_zzero_fai <- sqrt(sum((Out_recu_zo4 - gdp[-c(1:leny)])^2) / length(Out_recu_zo4))
      Rmse_zpos_m2 <- sqrt(sum((Out_recu_zp5 - gdp[-c(1:leny)])^2) / length(Out_recu_zp5))
      Rmse_zzero_m2 <- sqrt(sum((Out_recu_zo5 - gdp[-c(1:leny)])^2) / length(Out_recu_zo5))
      Rmse_zpos_ppi <- sqrt(sum((Out_recu_zp6 - gdp[-c(1:leny)])^2) / length(Out_recu_zp6))
      Rmse_zzero_ppi <- sqrt(sum((Out_recu_zo6 - gdp[-c(1:leny)])^2) / length(Out_recu_zo6))
      
      result_rec_for4$Recu_df1 <- append(result_rec_for4$Recu_df1, Rmse_df1)
      result_rec_for4$Recu_df2 <- append(result_rec_for4$Recu_df2, Rmse_df2)
      result_rec_for4$Recu_df3 <- append(result_rec_for4$Recu_df3, Rmse_df3)
      result_rec_for4$Recu_df4 <- append(result_rec_for4$Recu_df4, Rmse_df4)
      result_rec_for4$Recu_df5 <- append(result_rec_for4$Recu_df5, Rmse_df5)
      result_rec_for4$Recu_zpos_gdp <- append(result_rec_for4$Recu_zpos_gdp, Rmse_zpos_gdp)
      result_rec_for4$Recu_zzero_gdp <- append(result_rec_for4$Recu_zzero_gdp, Rmse_zzero_gdp)
      result_rec_for4$Recu_zpos_cpi <- append(result_rec_for4$Recu_zpos_cpi, Rmse_zpos_cpi)
      result_rec_for4$Recu_zzero_cpi <- append(result_rec_for4$Recu_zzero_cpi, Rmse_zzero_cpi)
      result_rec_for4$Recu_zpos_iva <- append(result_rec_for4$Recu_zpos_iva, Rmse_zpos_iva)
      result_rec_for4$Recu_zzero_iva <- append(result_rec_for4$Recu_zzero_iva, Rmse_zzero_iva)
      result_rec_for4$Recu_zpos_fai <- append(result_rec_for4$Recu_zpos_fai, Rmse_zpos_fai)
      result_rec_for4$Recu_zzero_fai <- append(result_rec_for4$Recu_zzero_fai, Rmse_zzero_fai)
      result_rec_for4$Recu_zpos_m2 <- append(result_rec_for4$Recu_zpos_m2, Rmse_zpos_m2)
      result_rec_for4$Recu_zzero_m2 <- append(result_rec_for4$Recu_zzero_m2, Rmse_zzero_m2)
      result_rec_for4$Recu_zpos_ppi <- append(result_rec_for4$Recu_zpos_ppi, Rmse_zpos_ppi)
      result_rec_for4$Recu_zzero_ppi <- append(result_rec_for4$Recu_zzero_ppi, Rmse_zzero_ppi)
    }
  }
}


#-------------------------------------
# Collect Results
#-------------------------------------
result_rec_now  <- data.frame(result_rec_now)
result_rec_for1 <- data.frame(result_rec_for1)
result_rec_for2 <- data.frame(result_rec_for2)
result_rec_for3 <- data.frame(result_rec_for3)
result_rec_for4 <- data.frame(result_rec_for4)
sheets <- list("sheet1" = result_rec_now, "sheet2" = result_rec_for1, "sheet3" = result_rec_for2,
               "sheet4" = result_rec_for3, "sheet5" = result_rec_for4)
write.xlsx(sheets, paste0(tmpdatpath, "Forecast_GDP_Realtime.xlsx"), sheetName=c("rec_now", "rec_for1", "rec_for2", "rec_for3", "rec_for4"))


#================================   Step (3)  =================================#
#===============================================================================
# Nowcasting
#===============================================================================
#-------------------------------------
# Load Data
#-------------------------------------
# Target
gdp <- na.omit(read.csv(paste0(rawdatpath, 'Target_GDP.csv'))[,2])

# PCIs: GDP
hdmr_gdp <- read.csv(paste0(tmpdatpath, 'PCIs_GDP_Realtime_Nowcast.csv'), header = FALSE)
zp1 <- hdmr_gdp[, seq(1, (length(hdmr_gdp)-3), 4)]
zo1 <- hdmr_gdp[, seq(2, (length(hdmr_gdp)-2), 4)]

# PCIs: CPI
hdmr_cpi <- read.csv(paste0(tmpdatpath, 'PCIs_CPI_Realtime_Nowcast.csv'), header = FALSE)
zp2 <- hdmr_cpi[, seq(1, (length(hdmr_cpi)-3), 4)]
zo2 <- hdmr_cpi[, seq(2, (length(hdmr_cpi)-2), 4)]

# PCIs: IVA
hdmr_iva <- read.csv(paste0(tmpdatpath, 'PCIs_IVA_Realtime_Nowcast.csv'), header = FALSE)
zp3 <- hdmr_iva[, seq(1, (length(hdmr_iva)-3), 4)]
zo3 <- hdmr_iva[, seq(2, (length(hdmr_iva)-2), 4)]

# PCIs: FAI
hdmr_fai <- read.csv(paste0(tmpdatpath, 'PCIs_FAI_Realtime_Nowcast.csv'), header = FALSE)
zp4 <- hdmr_fai[, seq(1, (length(hdmr_fai)-3), 4)]
zo4 <- hdmr_fai[, seq(2, (length(hdmr_fai)-2), 4)]

# PCIs: M2
hdmr_m2 <- read.csv(paste0(tmpdatpath, 'PCIs_M2_Realtime_Nowcast.csv'), header = FALSE)
zp5 <- hdmr_m2[, seq(1, (length(hdmr_m2)-3), 4)]
zo5 <- hdmr_m2[, seq(2, (length(hdmr_m2)-2), 4)]

# PCIs: PPI
hdmr_ppi <- read.csv(paste0(tmpdatpath, 'PCIs_PPI_Realtime_Nowcast.csv'), header = FALSE)
zp6 <- hdmr_ppi[, seq(1, (length(hdmr_ppi)-3), 4)]
zo6 <- hdmr_ppi[, seq(2, (length(hdmr_ppi)-2), 4)]

# Macro Factors
factor <- read.csv(paste0(tmpdatpath, 'MFactors_Realtime.csv'), header = F)
df1 <- data.frame(factor[, seq(1, (length(factor)-4) ,5)])
df2 <- data.frame(factor[, seq(2, (length(factor)-3), 5)])
df3 <- data.frame(factor[, seq(3, (length(factor)-2), 5)])
df4 <- data.frame(factor[, seq(4, (length(factor)-1), 5)])
df5 <- data.frame(factor[, seq(5, (length(factor)), 5)])


#-------------------------------------
# Set Estimating Windows & Variables
#-------------------------------------
len <- 7            # Initial window size = 7 years
leny <- 4 * len
lenx <- 12 * len
qdate <- format(seq.Date(from = as.Date("2003/03/01",format = "%Y/%m/%d"), by = "quarter", length.out = length(gdp)), format = "%Y/%m")

yy   <- ts(c(gdp[1:leny], gdp[leny+1]), start = c(2003,1), end = c(2003,1)+c(len,0), frequency = 4)
xx1  <- ts(c(zp1[1:lenx, 1], zp1[lenx+1, 1], zp1[lenx+2, 2], zp1[lenx+3, 3]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx2  <- ts(c(zo1[1:lenx, 1], zo1[lenx+1, 1], zo1[lenx+2, 2], zo1[lenx+3, 3]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx3  <- ts(c(zp2[1:lenx, 1], zp2[lenx+1, 1], zp2[lenx+2, 2], zp2[lenx+3, 3]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx4  <- ts(c(zo2[1:lenx, 1], zo2[lenx+1, 1], zo2[lenx+2, 2], zo2[lenx+3, 3]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx5  <- ts(c(zp3[1:lenx, 1], zp3[lenx+1, 1], zp3[lenx+2, 2], zp3[lenx+3, 3]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx6  <- ts(c(zo3[1:lenx, 1], zo3[lenx+1, 1], zo3[lenx+2, 2], zo3[lenx+3, 3]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx7  <- ts(c(zp4[1:lenx, 1], zp4[lenx+1, 1], zp4[lenx+2, 2], zp4[lenx+3, 3]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx8  <- ts(c(zo4[1:lenx, 1], zo4[lenx+1, 1], zo4[lenx+2, 2], zo4[lenx+3, 3]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx9  <- ts(c(zp5[1:lenx, 1], zp5[lenx+1, 1], zp5[lenx+2, 2], zp5[lenx+3, 3]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx10 <- ts(c(zo5[1:lenx, 1], zo5[lenx+1, 1], zo5[lenx+2, 2], zo5[lenx+3, 3]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx11 <- ts(c(zp6[1:lenx, 1], zp6[lenx+1, 1], zp6[lenx+2, 2], zp6[lenx+3, 3]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx12 <- ts(c(zo6[1:lenx, 1], zo6[lenx+1, 1], zo6[lenx+2, 2], zo6[lenx+3, 3]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
zz1  <- ts(c(df1[1:lenx, 1], df1[lenx+1, 2], df1[lenx+2, 3], df1[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
zz2  <- ts(c(df2[1:lenx, 1], df2[lenx+1, 2], df2[lenx+2, 3], df2[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
zz3  <- ts(c(df3[1:lenx, 1], df3[lenx+1, 2], df3[lenx+2, 3], df3[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
zz4  <- ts(c(df4[1:lenx, 1], df4[lenx+1, 2], df4[lenx+2, 3], df4[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
zz5  <- ts(c(df5[1:lenx, 1], df5[lenx+1, 2], df5[lenx+2, 3], df5[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)


#-------------------------------------
# Model Combination (h=0)
#-------------------------------------
result_rec_now <- na.omit(read_xlsx(paste0(tmpdatpath, "Forecast_GDP_Realtime.xlsx"), sheet = 1, col_names = T))

# Select best individual MIDAS model
best_recu_df1 <- result_rec_now[which(result_rec_now$Recu_df1 == min(result_rec_now$Recu_df1)), ]
best_recu_df2 <- result_rec_now[which(result_rec_now$Recu_df2 == min(result_rec_now$Recu_df2)), ]
best_recu_df3 <- result_rec_now[which(result_rec_now$Recu_df3 == min(result_rec_now$Recu_df3)), ]
best_recu_df4 <- result_rec_now[which(result_rec_now$Recu_df4 == min(result_rec_now$Recu_df4)), ]
best_recu_df5 <- result_rec_now[which(result_rec_now$Recu_df5 == min(result_rec_now$Recu_df5)), ]
best_recu_z_pos_gdp <- result_rec_now[which(result_rec_now$Recu_zpos_gdp == min(result_rec_now$Recu_zpos_gdp)), ][1,]
best_recu_z_zero_gdp <- result_rec_now[which(result_rec_now$Recu_zzero_gdp == min(result_rec_now$Recu_zzero_gdp)), ][1,]
best_recu_z_pos_cpi <- result_rec_now[which(result_rec_now$Recu_zpos_cpi == min(result_rec_now$Recu_zpos_cpi)), ]
best_recu_z_zero_cpi <- result_rec_now[which(result_rec_now$Recu_zzero_cpi == min(result_rec_now$Recu_zzero_cpi)), ]
best_recu_z_pos_iva <- result_rec_now[which(result_rec_now$Recu_zpos_iva == min(result_rec_now$Recu_zpos_iva)), ]
best_recu_z_zero_iva <- result_rec_now[which(result_rec_now$Recu_zzero_iva == min(result_rec_now$Recu_zzero_iva)), ]
best_recu_z_pos_fai <- result_rec_now[which(result_rec_now$Recu_zpos_fai == min(result_rec_now$Recu_zpos_fai)), ]
best_recu_z_zero_fai <- result_rec_now[which(result_rec_now$Recu_zzero_fai == min(result_rec_now$Recu_zzero_fai)), ]
best_recu_z_pos_m2 <- result_rec_now[which(result_rec_now$Recu_zpos_m2 == min(result_rec_now$Recu_zpos_m2)), ]
best_recu_z_zero_m2 <- result_rec_now[which(result_rec_now$Recu_zzero_m2 == min(result_rec_now$Recu_zzero_m2)), ]
best_recu_z_pos_ppi <- result_rec_now[which(result_rec_now$Recu_zpos_ppi == min(result_rec_now$Recu_zpos_ppi)), ]
best_recu_z_zero_ppi <- result_rec_now[which(result_rec_now$Recu_zzero_ppi == min(result_rec_now$Recu_zzero_ppi)), ]

## DF1
best_recu_df1_weight <- best_recu_df1$Weightfunction
best_recu_df1_ylag <- best_recu_df1$Ylag
best_recu_df1_xlag <- best_recu_df1$Xlag
best_recu_df1_lead <- best_recu_df1$Leads
best_recu_df1_tau <- "0"
## DF2
best_recu_df2_weight <- best_recu_df2$Weightfunction
best_recu_df2_ylag <- best_recu_df2$Ylag
best_recu_df2_xlag <- best_recu_df2$Xlag
best_recu_df2_lead <- best_recu_df2$Leads
best_recu_df2_tau <- "0"
## DF3
best_recu_df3_weight <- best_recu_df3$Weightfunction
best_recu_df3_ylag <- best_recu_df3$Ylag
best_recu_df3_xlag <- best_recu_df3$Xlag
best_recu_df3_lead <- best_recu_df3$Leads
best_recu_df3_tau <- "0"
## DF4
best_recu_df4_weight <- best_recu_df4$Weightfunction
best_recu_df4_ylag <- best_recu_df4$Ylag
best_recu_df4_xlag <- best_recu_df4$Xlag
best_recu_df4_lead <- best_recu_df4$Leads
best_recu_df4_tau <- "0"
## DF5
best_recu_df5_weight <- best_recu_df5$Weightfunction
best_recu_df5_ylag <- best_recu_df5$Ylag
best_recu_df5_xlag <- best_recu_df5$Xlag
best_recu_df5_lead <- best_recu_df5$Leads
best_recu_df5_tau <- "0"
## z_pos_gdp
best_recu_zpos_gdp_weight <- best_recu_z_pos_gdp$Weightfunction
best_recu_zpos_gdp_ylag <- best_recu_z_pos_gdp$Ylag
best_recu_zpos_gdp_xlag <- best_recu_z_pos_gdp$Xlag
best_recu_zpos_gdp_lead <- best_recu_z_pos_gdp$Leads
best_recu_zpos_gdp_tau <- "0"
## z_zero_gdp
best_recu_zzero_gdp_weight <- best_recu_z_zero_gdp$Weightfunction
best_recu_zzero_gdp_ylag <- best_recu_z_zero_gdp$Ylag
best_recu_zzero_gdp_xlag <- best_recu_z_zero_gdp$Xlag
best_recu_zzero_gdp_lead <- best_recu_z_zero_gdp$Leads
best_recu_zzero_gdp_tau <- "0"
## z_pos_cpi
best_recu_zpos_cpi_weight <- best_recu_z_pos_cpi$Weightfunction
best_recu_zpos_cpi_ylag <- best_recu_z_pos_cpi$Ylag
best_recu_zpos_cpi_xlag <- best_recu_z_pos_cpi$Xlag
best_recu_zpos_cpi_lead <- best_recu_z_pos_cpi$Leads
best_recu_zpos_cpi_tau <- "0"
## z_zero_cpi
best_recu_zzero_cpi_weight <- best_recu_z_zero_cpi$Weightfunction
best_recu_zzero_cpi_ylag <- best_recu_z_zero_cpi$Ylag
best_recu_zzero_cpi_xlag <- best_recu_z_zero_cpi$Xlag
best_recu_zzero_cpi_lead <- best_recu_z_zero_cpi$Leads
best_recu_zzero_cpi_tau <- "0"
## z_pos_iva
best_recu_zpos_iva_weight <- best_recu_z_pos_iva$Weightfunction
best_recu_zpos_iva_ylag <- best_recu_z_pos_iva$Ylag
best_recu_zpos_iva_xlag <- best_recu_z_pos_iva$Xlag
best_recu_zpos_iva_lead <- best_recu_z_pos_iva$Leads
best_recu_zpos_iva_tau <- "0"
## z_zero_iva
best_recu_zzero_iva_weight <- best_recu_z_zero_iva$Weightfunction
best_recu_zzero_iva_ylag <- best_recu_z_zero_iva$Ylag
best_recu_zzero_iva_xlag <- best_recu_z_zero_iva$Xlag
best_recu_zzero_iva_lead <- best_recu_z_zero_iva$Leads
best_recu_zzero_iva_tau <- "0"
## z_pos_fai
best_recu_zpos_fai_weight <- best_recu_z_pos_fai$Weightfunction
best_recu_zpos_fai_ylag <- best_recu_z_pos_fai$Ylag
best_recu_zpos_fai_xlag <- best_recu_z_pos_fai$Xlag
best_recu_zpos_fai_lead <- best_recu_z_pos_fai$Leads
best_recu_zpos_fai_tau <- "0"
## z_zero_fai
best_recu_zzero_fai_weight <- best_recu_z_zero_fai$Weightfunction
best_recu_zzero_fai_ylag <- best_recu_z_zero_fai$Ylag
best_recu_zzero_fai_xlag <- best_recu_z_zero_fai$Xlag
best_recu_zzero_fai_lead <- best_recu_z_zero_fai$Leads
best_recu_zzero_fai_tau <- "0"
## z_pos_m2
best_recu_zpos_m2_weight <- best_recu_z_pos_m2$Weightfunction
best_recu_zpos_m2_ylag <- best_recu_z_pos_m2$Ylag
best_recu_zpos_m2_xlag <- best_recu_z_pos_m2$Xlag
best_recu_zpos_m2_lead <- best_recu_z_pos_m2$Leads
best_recu_zpos_m2_tau <- "0"
## z_zero_m2
best_recu_zzero_m2_weight <- best_recu_z_zero_m2$Weightfunction
best_recu_zzero_m2_ylag <- best_recu_z_zero_m2$Ylag
best_recu_zzero_m2_xlag <- best_recu_z_zero_m2$Xlag
best_recu_zzero_m2_lead <- best_recu_z_zero_m2$Leads
best_recu_zzero_m2_tau <- "0"
## z_pos_ppi
best_recu_zpos_ppi_weight <- best_recu_z_pos_ppi$Weightfunction
best_recu_zpos_ppi_ylag <- best_recu_z_pos_ppi$Ylag
best_recu_zpos_ppi_xlag <- best_recu_z_pos_ppi$Xlag
best_recu_zpos_ppi_lead <- best_recu_z_pos_ppi$Leads
best_recu_zpos_ppi_tau <- "0"
## z_zero_ppi
best_recu_zzero_ppi_weight <- best_recu_z_zero_ppi$Weightfunction
best_recu_zzero_ppi_ylag <- best_recu_z_zero_ppi$Ylag
best_recu_zzero_ppi_xlag <- best_recu_z_zero_ppi$Xlag
best_recu_zzero_ppi_lead <- best_recu_z_zero_ppi$Leads
best_recu_zzero_ppi_tau <- "0"

# Training and combine best individual models to get combination MIDAS
## DF1
best_recu_df1_xlag <- 2 + best_recu_df1_xlag
best_recu_df1_lead <- 3 - best_recu_df1_lead
if (best_recu_df1_weight == "Almon") {
  best_md1 <- midas_r(yy ~ mls(yy, 1:best_recu_df1_ylag, 1) + mls(zz1, best_recu_df1_lead:best_recu_df1_xlag, 3, almonp),
                      start = list(zz1=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df1_weight == "Beta") {
  best_md1 <- midas_r(yy ~ mls(yy, 1:best_recu_df1_ylag, 1) + mls(zz1, best_recu_df1_lead:best_recu_df1_xlag, 3, nbeta),
                      start = list(zz1=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df1_weight == "BetaNN") {
  best_md1 <- midas_r(yy ~ mls(yy, 1:best_recu_df1_ylag, 1) + mls(zz1, best_recu_df1_lead:best_recu_df1_xlag, 3, nbetaMT),
                      start = list(zz1=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md1 <- midas_r(yy ~ mls(yy, 1:best_recu_df1_ylag, 1) + mls(zz1, best_recu_df1_lead:best_recu_df1_xlag, 3, nealmon),
                      start = list(zz1=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## DF2
best_recu_df2_xlag <- 2 + best_recu_df2_xlag
best_recu_df2_lead <- 3 - best_recu_df2_lead
if (best_recu_df2_weight == "Almon") {
  best_md2 <- midas_r(yy ~ mls(yy, 1:best_recu_df2_ylag, 1) + mls(zz2, best_recu_df2_lead:best_recu_df2_xlag, 3, almonp),
                      start = list(zz2=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df2_weight == "Beta") {
  best_md2 <- midas_r(yy ~ mls(yy, 1:best_recu_df2_ylag, 1) + mls(zz2, best_recu_df2_lead:best_recu_df2_xlag, 3, nbeta),
                      start = list(zz2=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df2_weight == "BetaNN") {
  best_md2 <- midas_r(yy ~ mls(yy, 1:best_recu_df2_ylag, 1) + mls(zz2, best_recu_df2_lead:best_recu_df2_xlag, 3, nbetaMT),
                      start = list(zz2=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md2 <- midas_r(yy ~ mls(yy, 1:best_recu_df2_ylag, 1) + mls(zz2, best_recu_df2_lead:best_recu_df2_xlag, 3, nealmon),
                      start = list(zz2=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## DF3
best_recu_df3_xlag <- 2 + best_recu_df3_xlag
best_recu_df3_lead <- 3 - best_recu_df3_lead
if (best_recu_df3_weight == "Almon") {
  best_md3 <- midas_r(yy ~ mls(yy, 1:best_recu_df3_ylag, 1) + mls(zz3, best_recu_df3_lead:best_recu_df3_xlag, 3, almonp),
                      start = list(zz3=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df3_weight == "Beta") {
  best_md3 <- midas_r(yy ~ mls(yy, 1:best_recu_df3_ylag, 1) + mls(zz3, best_recu_df3_lead:best_recu_df3_xlag, 3, nbeta),
                      start = list(zz3=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df3_weight == "BetaNN") {
  best_md3 <- midas_r(yy ~ mls(yy, 1:best_recu_df3_ylag, 1) + mls(zz3, best_recu_df3_lead:best_recu_df3_xlag, 3, nbetaMT),
                      start = list(zz3=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md3 <- midas_r(yy ~ mls(yy, 1:best_recu_df3_ylag, 1) + mls(zz3, best_recu_df3_lead:best_recu_df3_xlag, 3, nealmon),
                      start = list(zz3=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## DF4
best_recu_df4_xlag <- 2 + best_recu_df4_xlag
best_recu_df4_lead <- 3 - best_recu_df4_lead
if (best_recu_df4_weight == "Almon") {
  best_md4 <- midas_r(yy ~ mls(yy, 1:best_recu_df4_ylag, 1) + mls(zz4, best_recu_df4_lead:best_recu_df4_xlag, 3, almonp),
                      start = list(zz4=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df4_weight == "Beta") {
  best_md4 <- midas_r(yy ~ mls(yy, 1:best_recu_df4_ylag, 1) + mls(zz4, best_recu_df4_lead:best_recu_df4_xlag, 3, nbeta),
                      start = list(zz4=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df4_weight == "BetaNN") {
  best_md4 <- midas_r(yy ~ mls(yy, 1:best_recu_df4_ylag, 1) + mls(zz4, best_recu_df4_lead:best_recu_df4_xlag, 3, nbetaMT),
                      start = list(zz4=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md4 <- midas_r(yy ~ mls(yy, 1:best_recu_df4_ylag, 1) + mls(zz4, best_recu_df4_lead:best_recu_df4_xlag, 3, nealmon),
                      start = list(zz4=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## DF5
best_recu_df5_xlag <- 2 + best_recu_df5_xlag
best_recu_df5_lead <- 3 - best_recu_df5_lead
if (best_recu_df2_weight == "Almon") {
  best_md5 <- midas_r(yy ~ mls(yy, 1:best_recu_df5_ylag, 1) + mls(zz5, best_recu_df5_lead:best_recu_df5_xlag, 3, almonp),
                      start = list(zz5=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df5_weight == "Beta") {
  best_md5 <- midas_r(yy ~ mls(yy, 1:best_recu_df5_ylag, 1) + mls(zz5, best_recu_df5_lead:best_recu_df5_xlag, 3, nbeta),
                      start = list(zz5=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df5_weight == "BetaNN") {
  best_md5 <- midas_r(yy ~ mls(yy, 1:best_recu_df5_ylag, 1) + mls(zz5, best_recu_df5_lead:best_recu_df5_xlag, 3, nbetaMT),
                      start = list(zz5=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md5 <- midas_r(yy ~ mls(yy, 1:best_recu_df5_ylag, 1) + mls(zz5, best_recu_df5_lead:best_recu_df5_xlag, 3, nealmon),
                      start = list(zz5=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## z_pos_gdp
best_recu_zpos_gdp_xlag <- 2 + best_recu_zpos_gdp_xlag
best_recu_zpos_gdp_lead <- 3 - best_recu_zpos_gdp_lead
if (best_recu_zpos_gdp_weight == "Almon") {
  best_md6 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_gdp_ylag, 1) + mls(xx1, best_recu_zpos_gdp_lead:best_recu_zpos_gdp_xlag, 3, almonp),
                      start = list(xx1=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zpos_gdp_weight == "Beta") {
  best_md6 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_gdp_ylag, 1) + mls(xx1, best_recu_zpos_gdp_lead:best_recu_zpos_gdp_xlag, 3, nbeta),
                      start = list(xx1=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zpos_gdp_weight == "BetaNN") {
  best_md6 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_gdp_ylag, 1) + mls(xx1, best_recu_zpos_gdp_lead:best_recu_zpos_gdp_xlag, 3, nbetaMT),
                      start = list(xx1=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md6 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_gdp_ylag, 1) + mls(xx1, best_recu_zpos_gdp_lead:best_recu_zpos_gdp_xlag, 3, nealmon),
                      start = list(xx1=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## z_zero_gdp
best_recu_zzero_gdp_xlag <- 2 + best_recu_zzero_gdp_xlag
best_recu_zzero_gdp_lead <- 3 - best_recu_zzero_gdp_lead
if (best_recu_zzero_gdp_weight == "Almon") {
  best_md7 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_gdp_ylag, 1) + mls(xx2, best_recu_zzero_gdp_lead:best_recu_zzero_gdp_xlag, 3, almonp),
                      start = list(xx2=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zzero_gdp_weight == "Beta") {
  best_md7 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_gdp_ylag, 1) + mls(xx2, best_recu_zzero_gdp_lead:best_recu_zzero_gdp_xlag, 3, nbeta),
                      start = list(xx2=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zzero_gdp_weight == "BetaNN") {
  best_md7 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_gdp_ylag, 1) + mls(xx2, best_recu_zzero_gdp_lead:best_recu_zzero_gdp_xlag, 3, nbetaMT),
                      start = list(xx2=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md7 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_gdp_ylag, 1) + mls(xx2, best_recu_zzero_gdp_lead:best_recu_zzero_gdp_xlag, 3, nealmon),
                      start = list(xx2=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## z_pos_cpi
best_recu_zpos_cpi_xlag <- 2 + best_recu_zpos_cpi_xlag
best_recu_zpos_cpi_lead <- 3 - best_recu_zpos_cpi_lead
if (best_recu_zpos_cpi_weight == "Almon") {
  best_md8 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_cpi_ylag, 1) + mls(xx3, best_recu_zpos_cpi_lead:best_recu_zpos_cpi_xlag, 3, almonp),
                      start = list(xx3=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zpos_cpi_weight == "Beta") {
  best_md8 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_cpi_ylag, 1) + mls(xx3, best_recu_zpos_cpi_lead:best_recu_zpos_cpi_xlag, 3, nbeta),
                      start = list(xx3=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zpos_cpi_weight == "BetaNN") {
  best_md8 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_cpi_ylag, 1) + mls(xx3, best_recu_zpos_cpi_lead:best_recu_zpos_cpi_xlag, 3, nbetaMT),
                      start = list(xx3=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md8 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_cpi_ylag, 1) + mls(xx3, best_recu_zpos_cpi_lead:best_recu_zpos_cpi_xlag, 3, nealmon),
                      start = list(xx3=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## z_zero_cpi
best_recu_zzero_cpi_xlag <- 2 + best_recu_zzero_cpi_xlag
best_recu_zzero_cpi_lead <- 3 - best_recu_zzero_cpi_lead
if (best_recu_zzero_cpi_weight == "Almon") {
  best_md9 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_cpi_ylag, 1) + mls(xx4, best_recu_zzero_cpi_lead:best_recu_zzero_cpi_xlag, 3, almonp),
                      start = list(xx4=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zzero_cpi_weight == "Beta") {
  best_md9 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_cpi_ylag, 1) + mls(xx4, best_recu_zzero_cpi_lead:best_recu_zzero_cpi_xlag, 3, nbeta),
                      start = list(xx4=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zzero_cpi_weight == "BetaNN") {
  best_md9 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_cpi_ylag, 1) + mls(xx4, best_recu_zzero_cpi_lead:best_recu_zzero_cpi_xlag, 3, nbetaMT),
                      start = list(xx4=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md9 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_cpi_ylag, 1) + mls(xx4, best_recu_zzero_cpi_lead:best_recu_zzero_cpi_xlag, 3, nealmon),
                      start = list(xx4=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## z_pos_iva
best_recu_zpos_iva_xlag <- 2 + best_recu_zpos_iva_xlag
best_recu_zpos_iva_lead <- 3 - best_recu_zpos_iva_lead
if (best_recu_zpos_iva_weight == "Almon") {
  best_md10 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_iva_ylag, 1) + mls(xx5, best_recu_zpos_iva_lead:best_recu_zpos_iva_xlag, 3, almonp),
                       start = list(xx5=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_iva_weight == "Beta") {
  best_md10 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_iva_ylag, 1) + mls(xx5, best_recu_zpos_iva_lead:best_recu_zpos_iva_xlag, 3, nbeta),
                       start = list(xx5=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_iva_weight == "BetaNN") {
  best_md10 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_iva_ylag, 1) + mls(xx5, best_recu_zpos_iva_lead:best_recu_zpos_iva_xlag, 3, nbetaMT),
                       start = list(xx5=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md10 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_iva_ylag, 1) + mls(xx5, best_recu_zpos_iva_lead:best_recu_zpos_iva_xlag, 3, nealmon),
                       start = list(xx5=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}
### z_zero_iva
best_recu_zzero_iva_xlag <- 2 + best_recu_zzero_iva_xlag
best_recu_zzero_iva_lead <- 3 - best_recu_zzero_iva_lead
if (best_recu_zzero_iva_weight == "Almon") {
  best_md11 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_iva_ylag, 1) + mls(xx6, best_recu_zzero_iva_lead:best_recu_zzero_iva_xlag, 3, almonp),
                       start = list(xx6=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_iva_weight == "Beta") {
  best_md11 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_iva_ylag, 1) + mls(xx6, best_recu_zzero_iva_lead:best_recu_zzero_iva_xlag, 3, nbeta),
                       start = list(xx6=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_iva_weight == "BetaNN") {
  best_md11 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_iva_ylag, 1) + mls(xx6, best_recu_zzero_iva_lead:best_recu_zzero_iva_xlag, 3, nbetaMT),
                       start = list(xx6=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md11 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_iva_ylag, 1) + mls(xx6, best_recu_zzero_iva_lead:best_recu_zzero_iva_xlag, 3, nealmon),
                       start = list(xx6=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}
## z_pos_fai
best_recu_zpos_fai_xlag <- 2 + best_recu_zpos_fai_xlag
best_recu_zpos_fai_lead <- 3 - best_recu_zpos_fai_lead
if (best_recu_zpos_fai_weight == "Almon") {
  best_md12 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_fai_ylag, 1) + mls(xx7, best_recu_zpos_fai_lead:best_recu_zpos_fai_xlag, 3, almonp),
                       start = list(xx7=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_fai_weight == "Beta") {
  best_md12 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_fai_ylag, 1) + mls(xx7, best_recu_zpos_fai_lead:best_recu_zpos_fai_xlag, 3, nbeta),
                       start = list(xx7=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_fai_weight == "BetaNN") {
  best_md12 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_fai_ylag, 1) + mls(xx7, best_recu_zpos_fai_lead:best_recu_zpos_fai_xlag, 3, nbetaMT),
                       start = list(xx7=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md12 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_fai_ylag, 1) + mls(xx7, best_recu_zpos_fai_lead:best_recu_zpos_fai_xlag, 3, nealmon),
                       start = list(xx7=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}
### z_zero_fai
best_recu_zzero_fai_xlag <- 2 + best_recu_zzero_fai_xlag
best_recu_zzero_fai_lead <- 3 - best_recu_zzero_fai_lead
if (best_recu_zzero_fai_weight == "Almon") {
  best_md13 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_fai_ylag, 1) + mls(xx8, best_recu_zzero_fai_lead:best_recu_zzero_fai_xlag, 3, almonp),
                       start = list(xx8=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_fai_weight == "Beta") {
  best_md13 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_fai_ylag, 1) + mls(xx8, best_recu_zzero_fai_lead:best_recu_zzero_fai_xlag, 3, nbeta),
                       start = list(xx8=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_fai_weight == "BetaNN") {
  best_md13 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_fai_ylag, 1) + mls(xx8, best_recu_zzero_fai_lead:best_recu_zzero_fai_xlag, 3, nbetaMT),
                       start = list(xx8=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md13 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_fai_ylag, 1) + mls(xx8, best_recu_zzero_fai_lead:best_recu_zzero_fai_xlag, 3, nealmon),
                       start = list(xx8=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}
## z_pos_m2
best_recu_zpos_m2_xlag <- 2 + best_recu_zpos_m2_xlag
best_recu_zpos_m2_lead <- 3 - best_recu_zpos_m2_lead
if (best_recu_zpos_m2_weight == "Almon") {
  best_md14 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_m2_ylag, 1) + mls(xx9, best_recu_zpos_m2_lead:best_recu_zpos_m2_xlag, 3, almonp),
                       start = list(xx9=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_m2_weight == "Beta") {
  best_md14 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_m2_ylag, 1) + mls(xx9, best_recu_zpos_m2_lead:best_recu_zpos_m2_xlag, 3, nbeta),
                       start = list(xx9=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_m2_weight == "BetaNN") {
  best_md14 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_m2_ylag, 1) + mls(xx9, best_recu_zpos_m2_lead:best_recu_zpos_m2_xlag, 3, nbetaMT),
                       start = list(xx9=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md14 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_m2_ylag, 1) + mls(xx9, best_recu_zpos_m2_lead:best_recu_zpos_m2_xlag, 3, nealmon),
                       start = list(xx9=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}
### z_zero_m2
best_recu_zzero_m2_xlag <- 2 + best_recu_zzero_m2_xlag
best_recu_zzero_m2_lead <- 3 - best_recu_zzero_m2_lead
if (best_recu_zzero_m2_weight == "Almon") {
  best_md15 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_m2_ylag, 1) + mls(xx10, best_recu_zzero_m2_lead:best_recu_zzero_m2_xlag, 3, almonp),
                       start = list(xx10=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_m2_weight == "Beta") {
  best_md15 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_m2_ylag, 1) + mls(xx10, best_recu_zzero_m2_lead:best_recu_zzero_m2_xlag, 3, nbeta),
                       start = list(xx10=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_m2_weight == "BetaNN") {
  best_md15 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_m2_ylag, 1) + mls(xx10, best_recu_zzero_m2_lead:best_recu_zzero_m2_xlag, 3, nbetaMT),
                       start = list(xx10=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md15 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_m2_ylag, 1) + mls(xx10, best_recu_zzero_m2_lead:best_recu_zzero_m2_xlag, 3, nealmon),
                       start = list(xx10=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}
## z_pos_ppi
best_recu_zpos_ppi_xlag <- 2 + best_recu_zpos_ppi_xlag
best_recu_zpos_ppi_lead <- 3 - best_recu_zpos_ppi_lead
if (best_recu_zpos_ppi_weight == "Almon") {
  best_md16 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_ppi_ylag, 1) + mls(xx11, best_recu_zpos_ppi_lead:best_recu_zpos_ppi_xlag, 3, almonp),
                       start = list(xx11=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_ppi_weight == "Beta") {
  best_md16 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_ppi_ylag, 1) + mls(xx11, best_recu_zpos_ppi_lead:best_recu_zpos_ppi_xlag, 3, nbeta),
                       start = list(xx11=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_ppi_weight == "BetaNN") {
  best_md16 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_ppi_ylag, 1) + mls(xx11, best_recu_zpos_ppi_lead:best_recu_zpos_ppi_xlag, 3, nbetaMT),
                       start = list(xx11=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md16 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_ppi_ylag, 1) + mls(xx11, best_recu_zpos_ppi_lead:best_recu_zpos_ppi_xlag, 3, nealmon),
                       start = list(xx11=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}
### z_zero_ppi
best_recu_zzero_ppi_xlag <- 2 + best_recu_zzero_ppi_xlag
best_recu_zzero_ppi_lead <- 3 - best_recu_zzero_ppi_lead
if (best_recu_zzero_ppi_weight == "Almon") {
  best_md17 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_ppi_ylag, 1) + mls(xx12, best_recu_zzero_ppi_lead:best_recu_zzero_ppi_xlag, 3, almonp),
                       start = list(xx12=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_ppi_weight == "Beta") {
  best_md17 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_ppi_ylag, 1) + mls(xx12, best_recu_zzero_ppi_lead:best_recu_zzero_ppi_xlag, 3, nbeta),
                       start = list(xx12=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_ppi_weight == "BetaNN") {
  best_md17 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_ppi_ylag, 1) + mls(xx12, best_recu_zzero_ppi_lead:best_recu_zzero_ppi_xlag, 3, nbetaMT),
                       start = list(xx12=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md17 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_ppi_ylag, 1) + mls(xx12, best_recu_zzero_ppi_lead:best_recu_zzero_ppi_xlag, 3, nealmon),
                       start = list(xx12=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}

# Estimating & Forecasting
Recu_mf_ew = c()
Recu_mf_bicw = c()
Recu_mf_msfew = c()
Recu_mf_dmsfew = c()
Recu_zp_ew = c()
Recu_zp_bicw = c()
Recu_zp_msfew = c()
Recu_zp_dmsfew = c()
Recu_zz_ew = c()
Recu_zz_bicw = c()
Recu_zz_msfew = c()
Recu_zz_dmsfew = c()
Recu_cb_ew = c()
Recu_cb_bicw = c()
Recu_cb_msfew = c()
Recu_cb_dmsfew = c()
Recu_mfcb_ew = c()
Recu_mfcb_bicw = c()
Recu_mfcb_msfew = c()
Recu_mfcb_dmsfew = c()

for (i in seq(1, length(zp2), 3)) {
  # Sampling
  p = floor((i-1)/3)
  l = ceiling(i/3)
  l1 = l + 1
  m1 = l + leny - 1
  m2 = m1 + 1
  j = i + lenx - 1
  k1 = j + 1
  k2 = j + 2
  k3 = j + 3
  i1 = i + 1
  i2 = i + 2
  i3 = i + 3
  
  # Start & End
  starty = c(2003,1)
  endy = starty + c(len, 0) + c(0,p)
  startx = c(2003,1)
  endx = startx + c(len, 1+i)
  fulldata <- list(
    yy = ts(na.omit(c(gdp[1:m1], gdp[m2])), start = starty, end = endy, frequency = 4), 
    xx1 = ts(c(zp1[1:j, i], zp1[k1, i], zp1[k2, i1], zp1[k3, i2]), start = startx, end = endx, frequency = 12),
    xx2 = ts(c(zo1[1:j, i], zo1[k1, i], zo1[k2, i1], zo1[k3, i2]), start = startx, end = endx, frequency = 12),
    xx3 = ts(c(zp2[1:j, i], zp2[k1, i], zp2[k2, i1], zp2[k3, i2]), start = startx, end = endx, frequency = 12),
    xx4 = ts(c(zo2[1:j, i], zo2[k1, i], zo2[k2, i1], zo2[k3, i2]), start = startx, end = endx, frequency = 12),
    xx5 = ts(c(zp3[1:j, i], zp3[k1, i], zp3[k2, i1], zp3[k3, i2]), start = startx, end = endx, frequency = 12),
    xx6 = ts(c(zo3[1:j, i], zo3[k1, i], zo3[k2, i1], zo3[k3, i2]), start = startx, end = endx, frequency = 12),
    xx7 = ts(c(zp4[1:j, i], zp4[k1, i], zp4[k2, i1], zp4[k3, i2]), start = startx, end = endx, frequency = 12),
    xx8 = ts(c(zo4[1:j, i], zo4[k1, i], zo4[k2, i1], zo4[k3, i2]), start = startx, end = endx, frequency = 12),
    xx9 = ts(c(zp5[1:j, i], zp5[k1, i], zp5[k2, i1], zp5[k3, i2]), start = startx, end = endx, frequency = 12),
    xx10 = ts(c(zo5[1:j, i], zo5[k1, i], zo5[k2, i1], zo5[k3, i2]), start = startx, end = endx, frequency = 12),
    xx11 = ts(c(zp6[1:j, i], zp6[k1, i], zp6[k2, i1], zp6[k3, i2]), start = startx, end = endx, frequency = 12),
    xx12 = ts(c(zo6[1:j, i], zo6[k1, i], zo6[k2, i1], zo6[k3, i2]), start = startx, end = endx, frequency = 12),
    zz1 = ts(c(df1[1:j, i], df1[k1, i1], df1[k2, i2], df1[k3, i3]), start = startx, end = endx, frequency = 12),
    zz2 = ts(c(df2[1:j, i], df2[k1, i1], df2[k2, i2], df2[k3, i3]), start = startx, end = endx, frequency = 12),
    zz3 = ts(c(df3[1:j, i], df3[k1, i1], df3[k2, i2], df3[k3, i3]), start = startx, end = endx, frequency = 12),
    zz4 = ts(c(df4[1:j, i], df4[k1, i1], df4[k2, i2], df4[k3, i3]), start = startx, end = endx, frequency = 12),
    zz5 = ts(c(df5[1:j, i], df5[k1, i1], df5[k2, i2], df5[k3, i3]), start = startx, end = endx, frequency = 12)
  )
  insample <- 1:m1
  outsample <- (1:length(fulldata$yy))[-insample]
  
  # Combination Model
  ## MF
  best_nowmdzz <- average_forecast(list(best_md1, best_md2, best_md3, best_md4, best_md5),
                                   data = fulldata, insample = insample, outsample = outsample,
                                   type = "recursive", 
                                   measures = c("MSE"),
                                   fweights = c("EW", "BICW", "MSFE", "DMSFE"),
                                   show_progress = FALSE)
  Recu_mf_ew <- append(Recu_mf_ew, best_nowmdzz[["avgforecast"]][1])
  Recu_mf_bicw <- append(Recu_mf_bicw, best_nowmdzz[["avgforecast"]][2])
  Recu_mf_msfew <- append(Recu_mf_msfew, best_nowmdzz[["avgforecast"]][3])
  Recu_mf_dmsfew <- append(Recu_mf_dmsfew, best_nowmdzz[["avgforecast"]][4])
  ## MF+CB_+
  best_nowmdxzp <- average_forecast(list(best_md1, best_md2, best_md3, best_md4, best_md5, best_md6),
                                    data = fulldata, insample = insample, outsample = outsample,
                                    type = "recursive", 
                                    measures = c("MSE"),
                                    fweights = c("EW", "BICW", "MSFE", "DMSFE"),
                                    show_progress = FALSE)
  Recu_zp_ew <- append(Recu_zp_ew, best_nowmdxzp[["avgforecast"]][1])
  Recu_zp_bicw <- append(Recu_zp_bicw, best_nowmdxzp[["avgforecast"]][2])
  Recu_zp_msfew <- append(Recu_zp_msfew, best_nowmdxzp[["avgforecast"]][3])
  Recu_zp_dmsfew <- append(Recu_zp_dmsfew, best_nowmdxzp[["avgforecast"]][4])  
  ## MF+CB_0
  best_nowmdxzz <- average_forecast(list(best_md1, best_md2, best_md3, best_md4, best_md5, best_md7),
                                    data = fulldata, insample = insample, outsample = outsample,
                                    type = "recursive", 
                                    measures = c("MSE"),
                                    fweights = c("EW", "BICW", "MSFE", "DMSFE"),
                                    show_progress = FALSE)
  Recu_zz_ew <- append(Recu_zz_ew, best_nowmdxzz[["avgforecast"]][1])
  Recu_zz_bicw <- append(Recu_zz_bicw, best_nowmdxzz[["avgforecast"]][2])
  Recu_zz_msfew <- append(Recu_zz_msfew, best_nowmdxzz[["avgforecast"]][3])
  Recu_zz_dmsfew <- append(Recu_zz_dmsfew, best_nowmdxzz[["avgforecast"]][4])    
  ## MF+CB_2
  best_nowmdxz_2 <- average_forecast(list(best_md1, best_md2, best_md3, best_md4,
                                          best_md5, best_md6, best_md7),
                                     data = fulldata, insample = insample, outsample = outsample,
                                     type = "recursive", 
                                     measures = c("MSE"),
                                     fweights = c("EW", "BICW", "MSFE", "DMSFE"),
                                     show_progress = FALSE)
  Recu_cb_ew <- append(Recu_cb_ew, best_nowmdxz_2[["avgforecast"]][1])
  Recu_cb_bicw <- append(Recu_cb_bicw, best_nowmdxz_2[["avgforecast"]][2])
  Recu_cb_msfew <- append(Recu_cb_msfew, best_nowmdxz_2[["avgforecast"]][3])
  Recu_cb_dmsfew <- append(Recu_cb_dmsfew, best_nowmdxz_2[["avgforecast"]][4])  
  ## MF+CB_12
  best_nowmdxz_12 <- average_forecast(list(best_md1, best_md2, best_md3, best_md4,
                                           best_md5, best_md6, best_md7, best_md8, best_md9,
                                           best_md10, best_md11, best_md12, best_md13,
                                           best_md14, best_md15, best_md16, best_md17),
                                      data = fulldata, insample = insample, outsample = outsample,
                                      type = "recursive", 
                                      measures = c("MSE"),
                                      fweights = c("EW", "BICW", "MSFE", "DMSFE"),
                                      show_progress = FALSE)
  Recu_mfcb_ew <- append(Recu_mfcb_ew, best_nowmdxz_12[["avgforecast"]][1])
  Recu_mfcb_bicw <- append(Recu_mfcb_bicw, best_nowmdxz_12[["avgforecast"]][2])
  Recu_mfcb_msfew <- append(Recu_mfcb_msfew, best_nowmdxz_12[["avgforecast"]][3])
  Recu_mfcb_dmsfew <- append(Recu_mfcb_dmsfew, best_nowmdxz_12[["avgforecast"]][4])
}

# Saving predicting results
best_result_rec_now <- data.frame(qdate = qdate,
                                  real = gdp,
                                  mf_ew = append(rep(0, leny), Recu_mf_ew),
                                  mf_bicw = append(rep(0, leny), Recu_mf_bicw),
                                  mf_msfew = append(rep(0, leny), Recu_mf_msfew),
                                  mf_dmsfew = append(rep(0, leny), Recu_mf_dmsfew),
                                  zp_ew = append(rep(0, leny), Recu_zp_ew),
                                  zp_bicw = append(rep(0, leny), Recu_zp_bicw),
                                  zp_msfew = append(rep(0, leny), Recu_zp_msfew),
                                  zp_dmsfew = append(rep(0, leny), Recu_zp_dmsfew),
                                  zz_ew = append(rep(0, leny), Recu_zz_ew),
                                  zz_bicw = append(rep(0, leny), Recu_zz_bicw),
                                  zz_msfew = append(rep(0, leny), Recu_zz_msfew),
                                  zz_dmsfew = append(rep(0, leny), Recu_zz_dmsfew),                         
                                  cb_ew = append(rep(0, leny), Recu_cb_ew),
                                  cb_bicw = append(rep(0, leny), Recu_cb_bicw),
                                  cb_msfew = append(rep(0, leny), Recu_cb_msfew),
                                  cb_dmsfew = append(rep(0, leny), Recu_cb_dmsfew),
                                  mfcb_ew = append(rep(0, leny), Recu_mfcb_ew),
                                  mfcb_bicw = append(rep(0, leny), Recu_mfcb_bicw),
                                  mfcb_msfew = append(rep(0, leny), Recu_mfcb_msfew),
                                  mfcb_dmsfew = append(rep(0, leny), Recu_mfcb_dmsfew))


#===============================================================================
# Forecasting
#===============================================================================
#-------------------------------------
# Load Data
#-------------------------------------
# Target
gdp <- na.omit(read.csv(paste0(rawdatpath, 'Target_GDP.csv'))[,2])

# PCIs: GDP
hdmr_gdp <- read.csv(paste0(tmpdatpath, 'PCIs_GDP_Realtime.csv'), header = FALSE)
zp1 <- hdmr_gdp[, seq(1, (length(hdmr_gdp)-3), 4)]
zo1 <- hdmr_gdp[, seq(2, (length(hdmr_gdp)-2), 4)]

# PCIs: CPI
hdmr_cpi <- read.csv(paste0(tmpdatpath, 'PCIs_CPI_Realtime.csv'), header = FALSE)
zp2 <- hdmr_cpi[, seq(1, (length(hdmr_cpi)-3), 4)]
zo2 <- hdmr_cpi[, seq(2, (length(hdmr_cpi)-2), 4)]

# PCIs: IVA
hdmr_iva <- read.csv(paste0(tmpdatpath, 'PCIs_IVA_Realtime.csv'), header = FALSE)
zp3 <- hdmr_iva[, seq(1, (length(hdmr_iva)-3), 4)]
zo3 <- hdmr_iva[, seq(2, (length(hdmr_iva)-2), 4)]

# PCIs: FAI
hdmr_fai <- read.csv(paste0(tmpdatpath, 'PCIs_FAI_Realtime.csv'), header = FALSE)
zp4 <- hdmr_fai[, seq(1, (length(hdmr_fai)-3), 4)]
zo4 <- hdmr_fai[, seq(2, (length(hdmr_fai)-2), 4)]

# PCIs: M2
hdmr_m2 <- read.csv(paste0(tmpdatpath, 'PCIs_M2_Realtime.csv'), header = FALSE)
zp5 <- hdmr_m2[, seq(1, (length(hdmr_m2)-3), 4)]
zo5 <- hdmr_m2[, seq(2, (length(hdmr_m2)-2), 4)]

# PCIs: PPI
hdmr_ppi <- read.csv(paste0(tmpdatpath, 'PCIs_PPI_Realtime.csv'), header = FALSE)
zp6 <- hdmr_ppi[, seq(1, (length(hdmr_ppi)-3), 4)]
zo6 <- hdmr_ppi[, seq(2, (length(hdmr_ppi)-2), 4)]

# Macro Factors
factor <- read.csv(paste0(tmpdatpath, 'MFactors_Realtime.csv'), header = F)
df1 <- data.frame(factor[, seq(1, (length(factor)-4) ,5)])
df2 <- data.frame(factor[, seq(2, (length(factor)-3), 5)])
df3 <- data.frame(factor[, seq(3, (length(factor)-2), 5)])
df4 <- data.frame(factor[, seq(4, (length(factor)-1), 5)])
df5 <- data.frame(factor[, seq(5, (length(factor)), 5)])


#-------------------------------------
# Set Estimating Windows & Variables
#-------------------------------------
len <- 7            # Initial window size = 7 years
leny <- 4 * len
lenx <- 12 * len
qdate <- format(seq.Date(from = as.Date("2003/03/01",format = "%Y/%m/%d"), by = "quarter", length.out = length(gdp)), format = "%Y/%m")

yy   <- ts(c(gdp[1:leny], gdp[leny+1]), start = c(2003,1), end = c(2003,1)+c(len,0), frequency = 4)
xx1  <- ts(c(zp1[1:lenx, 1], zp1[lenx+1, 2], zp1[lenx+2, 3], zp1[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx2  <- ts(c(zo1[1:lenx, 1], zo1[lenx+1, 2], zo1[lenx+2, 3], zo1[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx3  <- ts(c(zp2[1:lenx, 1], zp2[lenx+1, 2], zp2[lenx+2, 3], zp2[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx4  <- ts(c(zo2[1:lenx, 1], zo2[lenx+1, 2], zo2[lenx+2, 3], zo2[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx5  <- ts(c(zp3[1:lenx, 1], zp3[lenx+1, 2], zp3[lenx+2, 3], zp3[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx6  <- ts(c(zo3[1:lenx, 1], zo3[lenx+1, 2], zo3[lenx+2, 3], zo3[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx7  <- ts(c(zp4[1:lenx, 1], zp4[lenx+1, 2], zp4[lenx+2, 3], zp4[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx8  <- ts(c(zo4[1:lenx, 1], zo4[lenx+1, 2], zo4[lenx+2, 3], zo4[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx9  <- ts(c(zp5[1:lenx, 1], zp5[lenx+1, 2], zp5[lenx+2, 3], zp5[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx10 <- ts(c(zo5[1:lenx, 1], zo5[lenx+1, 2], zo5[lenx+2, 3], zo5[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx11 <- ts(c(zp6[1:lenx, 1], zp6[lenx+1, 2], zp6[lenx+2, 3], zp6[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
xx12 <- ts(c(zo6[1:lenx, 1], zo6[lenx+1, 2], zo6[lenx+2, 3], zo6[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
zz1  <- ts(c(df1[1:lenx, 1], df1[lenx+1, 2], df1[lenx+2, 3], df1[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
zz2  <- ts(c(df2[1:lenx, 1], df2[lenx+1, 2], df2[lenx+2, 3], df2[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
zz3  <- ts(c(df3[1:lenx, 1], df3[lenx+1, 2], df3[lenx+2, 3], df3[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
zz4  <- ts(c(df4[1:lenx, 1], df4[lenx+1, 2], df4[lenx+2, 3], df4[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)
zz5  <- ts(c(df5[1:lenx, 1], df5[lenx+1, 2], df5[lenx+2, 3], df5[lenx+3, 4]), start = c(2003,1), end = c(2003,1)+c(len,2), frequency = 12)


#-------------------------------------
# Model Combination (h=1)
#-------------------------------------
result_rec_for1 <- na.omit(read_xlsx(paste0(tmpdatpath, "Forecast_GDP_Realtime.xlsx"), sheet = 2, col_names = T))

# Select best individual MIDAS model
best_recu_df1 <- result_rec_for1[which(result_rec_for1$Recu_df1 == min(result_rec_for1$Recu_df1)), ]
best_recu_df2 <- result_rec_for1[which(result_rec_for1$Recu_df2 == min(result_rec_for1$Recu_df2)), ]
best_recu_df3 <- result_rec_for1[which(result_rec_for1$Recu_df3 == min(result_rec_for1$Recu_df3)), ]
best_recu_df4 <- result_rec_for1[which(result_rec_for1$Recu_df4 == min(result_rec_for1$Recu_df4)), ]
best_recu_df5 <- result_rec_for1[which(result_rec_for1$Recu_df5 == min(result_rec_for1$Recu_df5)), ]
best_recu_z_pos_gdp <- result_rec_for1[which(result_rec_for1$Recu_zpos_gdp == min(result_rec_for1$Recu_zpos_gdp)), ][1,]
best_recu_z_zero_gdp <- result_rec_for1[which(result_rec_for1$Recu_zzero_gdp == min(result_rec_for1$Recu_zzero_gdp)), ][1,]
best_recu_z_pos_cpi <- result_rec_for1[which(result_rec_for1$Recu_zpos_cpi == min(result_rec_for1$Recu_zpos_cpi)), ]
best_recu_z_zero_cpi <- result_rec_for1[which(result_rec_for1$Recu_zzero_cpi == min(result_rec_for1$Recu_zzero_cpi)), ]
best_recu_z_pos_iva <- result_rec_for1[which(result_rec_for1$Recu_zpos_iva == min(result_rec_for1$Recu_zpos_iva)), ]
best_recu_z_zero_iva <- result_rec_for1[which(result_rec_for1$Recu_zzero_iva == min(result_rec_for1$Recu_zzero_iva)), ]
best_recu_z_pos_fai <- result_rec_for1[which(result_rec_for1$Recu_zpos_fai == min(result_rec_for1$Recu_zpos_fai)), ]
best_recu_z_zero_fai <- result_rec_for1[which(result_rec_for1$Recu_zzero_fai == min(result_rec_for1$Recu_zzero_fai)), ]
best_recu_z_pos_m2 <- result_rec_for1[which(result_rec_for1$Recu_zpos_m2 == min(result_rec_for1$Recu_zpos_m2)), ]
best_recu_z_zero_m2 <- result_rec_for1[which(result_rec_for1$Recu_zzero_m2 == min(result_rec_for1$Recu_zzero_m2)), ]
best_recu_z_pos_ppi <- result_rec_for1[which(result_rec_for1$Recu_zpos_ppi == min(result_rec_for1$Recu_zpos_ppi)), ]
best_recu_z_zero_ppi <- result_rec_for1[which(result_rec_for1$Recu_zzero_ppi == min(result_rec_for1$Recu_zzero_ppi)), ]

## DF1
best_recu_df1_weight <- best_recu_df1$Weightfunction
best_recu_df1_ylag <- best_recu_df1$Ylag
best_recu_df1_xlag <- best_recu_df1$Xlag
best_recu_df1_lead <- 3
best_recu_df1_tau <- "1"
## DF2
best_recu_df2_weight <- best_recu_df2$Weightfunction
best_recu_df2_ylag <- best_recu_df2$Ylag
best_recu_df2_xlag <- best_recu_df2$Xlag
best_recu_df2_lead <- 3
best_recu_df2_tau <- "1"
## DF3
best_recu_df3_weight <- best_recu_df3$Weightfunction
best_recu_df3_ylag <- best_recu_df3$Ylag
best_recu_df3_xlag <- best_recu_df3$Xlag
best_recu_df3_lead <- 3
best_recu_df3_tau <- "1"
## DF4
best_recu_df4_weight <- best_recu_df4$Weightfunction
best_recu_df4_ylag <- best_recu_df4$Ylag
best_recu_df4_xlag <- best_recu_df4$Xlag
best_recu_df4_lead <- 3
best_recu_df4_tau <- "1"
## DF5
best_recu_df5_weight <- best_recu_df5$Weightfunction
best_recu_df5_ylag <- best_recu_df5$Ylag
best_recu_df5_xlag <- best_recu_df5$Xlag
best_recu_df5_lead <- 3
best_recu_df5_tau <- "1"
## z_pos_gdp
best_recu_zpos_gdp_weight <- best_recu_z_pos_gdp$Weightfunction
best_recu_zpos_gdp_ylag <- best_recu_z_pos_gdp$Ylag
best_recu_zpos_gdp_xlag <- best_recu_z_pos_gdp$Xlag
best_recu_zpos_gdp_lead <- 3
best_recu_zpos_gdp_tau <- "1"
## z_zero_gdp
best_recu_zzero_gdp_weight <- best_recu_z_zero_gdp$Weightfunction
best_recu_zzero_gdp_ylag <- best_recu_z_zero_gdp$Ylag
best_recu_zzero_gdp_xlag <- best_recu_z_zero_gdp$Xlag
best_recu_zzero_gdp_lead <- 3
best_recu_zzero_gdp_tau <- "1"
## z_pos_cpi
best_recu_zpos_cpi_weight <- best_recu_z_pos_cpi$Weightfunction
best_recu_zpos_cpi_ylag <- best_recu_z_pos_cpi$Ylag
best_recu_zpos_cpi_xlag <- best_recu_z_pos_cpi$Xlag
best_recu_zpos_cpi_lead <- 3
best_recu_zpos_cpi_tau <- "1"
## z_zero_cpi
best_recu_zzero_cpi_weight <- best_recu_z_zero_cpi$Weightfunction
best_recu_zzero_cpi_ylag <- best_recu_z_zero_cpi$Ylag
best_recu_zzero_cpi_xlag <- best_recu_z_zero_cpi$Xlag
best_recu_zzero_cpi_lead <- 3
best_recu_zzero_cpi_tau <- "1"
## z_pos_iva
best_recu_zpos_iva_weight <- best_recu_z_pos_iva$Weightfunction
best_recu_zpos_iva_ylag <- best_recu_z_pos_iva$Ylag
best_recu_zpos_iva_xlag <- best_recu_z_pos_iva$Xlag
best_recu_zpos_iva_lead <- 3
best_recu_zpos_iva_tau <- "1"
## z_zero_iva
best_recu_zzero_iva_weight <- best_recu_z_zero_iva$Weightfunction
best_recu_zzero_iva_ylag <- best_recu_z_zero_iva$Ylag
best_recu_zzero_iva_xlag <- best_recu_z_zero_iva$Xlag
best_recu_zzero_iva_lead <- 3
best_recu_zzero_iva_tau <- "1"
## z_pos_fai
best_recu_zpos_fai_weight <- best_recu_z_pos_fai$Weightfunction
best_recu_zpos_fai_ylag <- best_recu_z_pos_fai$Ylag
best_recu_zpos_fai_xlag <- best_recu_z_pos_fai$Xlag
best_recu_zpos_fai_lead <- 3
best_recu_zpos_fai_tau <- "1"
## z_zero_fai
best_recu_zzero_fai_weight <- best_recu_z_zero_fai$Weightfunction
best_recu_zzero_fai_ylag <- best_recu_z_zero_fai$Ylag
best_recu_zzero_fai_xlag <- best_recu_z_zero_fai$Xlag
best_recu_zzero_fai_lead <- 3
best_recu_zzero_fai_tau <- "1"
## z_pos_m2
best_recu_zpos_m2_weight <- best_recu_z_pos_m2$Weightfunction
best_recu_zpos_m2_ylag <- best_recu_z_pos_m2$Ylag
best_recu_zpos_m2_xlag <- best_recu_z_pos_m2$Xlag
best_recu_zpos_m2_lead <- 3
best_recu_zpos_m2_tau <- "1"
## z_zero_m2
best_recu_zzero_m2_weight <- best_recu_z_zero_m2$Weightfunction
best_recu_zzero_m2_ylag <- best_recu_z_zero_m2$Ylag
best_recu_zzero_m2_xlag <- best_recu_z_zero_m2$Xlag
best_recu_zzero_m2_lead <- 3
best_recu_zzero_m2_tau <- "1"
## z_pos_ppi
best_recu_zpos_ppi_weight <- best_recu_z_pos_ppi$Weightfunction
best_recu_zpos_ppi_ylag <- best_recu_z_pos_ppi$Ylag
best_recu_zpos_ppi_xlag <- best_recu_z_pos_ppi$Xlag
best_recu_zpos_ppi_lead <- 3
best_recu_zpos_ppi_tau <- "1"
## z_zero_ppi
best_recu_zzero_ppi_weight <- best_recu_z_zero_ppi$Weightfunction
best_recu_zzero_ppi_ylag <- best_recu_z_zero_ppi$Ylag
best_recu_zzero_ppi_xlag <- best_recu_z_zero_ppi$Xlag
best_recu_zzero_ppi_lead <- 3
best_recu_zzero_ppi_tau <- "1"

# Training and combine best individual models to get combination MIDAS
## DF1
best_recu_df1_ylag <- 0 + best_recu_df1_ylag
best_recu_df1_xlag <- 2 + best_recu_df1_xlag
if (best_recu_df1_weight == "Almon") {
  best_md1 <- midas_r(yy ~ mls(yy, 1:best_recu_df1_ylag, 1) + mls(zz1, best_recu_df1_lead:best_recu_df1_xlag, 3, almonp),
                      start = list(zz1=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df1_weight == "Beta") {
  best_md1 <- midas_r(yy ~ mls(yy, 1:best_recu_df1_ylag, 1) + mls(zz1, best_recu_df1_lead:best_recu_df1_xlag, 3, nbeta),
                      start = list(zz1=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df1_weight == "BetaNN") {
  best_md1 <- midas_r(yy ~ mls(yy, 1:best_recu_df1_ylag, 1) + mls(zz1, best_recu_df1_lead:best_recu_df1_xlag, 3, nbetaMT),
                      start = list(zz1=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md1 <- midas_r(yy ~ mls(yy, 1:best_recu_df1_ylag, 1) + mls(zz1, best_recu_df1_lead:best_recu_df1_xlag, 3, nealmon),
                      start = list(zz1=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## DF2
best_recu_df2_ylag <- 0 + best_recu_df2_ylag
best_recu_df2_xlag <- 2 + best_recu_df2_xlag
if (best_recu_df2_weight == "Almon") {
  best_md2 <- midas_r(yy ~ mls(yy, 1:best_recu_df2_ylag, 1) + mls(zz2, best_recu_df2_lead:best_recu_df2_xlag, 3, almonp),
                      start = list(zz2=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df2_weight == "Beta") {
  best_md2 <- midas_r(yy ~ mls(yy, 1:best_recu_df2_ylag, 1) + mls(zz2, best_recu_df2_lead:best_recu_df2_xlag, 3, nbeta),
                      start = list(zz2=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df2_weight == "BetaNN") {
  best_md2 <- midas_r(yy ~ mls(yy, 1:best_recu_df2_ylag, 1) + mls(zz2, best_recu_df2_lead:best_recu_df2_xlag, 3, nbetaMT),
                      start = list(zz2=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md2 <- midas_r(yy ~ mls(yy, 1:best_recu_df2_ylag, 1) + mls(zz2, best_recu_df2_lead:best_recu_df2_xlag, 3, nealmon),
                      start = list(zz2=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## DF3
best_recu_df3_ylag <- 0 + best_recu_df3_ylag
best_recu_df3_xlag <- 2 + best_recu_df3_xlag
if (best_recu_df3_weight == "Almon") {
  best_md3 <- midas_r(yy ~ mls(yy, 1:best_recu_df3_ylag, 1) + mls(zz3, best_recu_df3_lead:best_recu_df3_xlag, 3, almonp),
                      start = list(zz3=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df3_weight == "Beta") {
  best_md3 <- midas_r(yy ~ mls(yy, 1:best_recu_df3_ylag, 1) + mls(zz3, best_recu_df3_lead:best_recu_df3_xlag, 3, nbeta),
                      start = list(zz3=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df3_weight == "BetaNN") {
  best_md3 <- midas_r(yy ~ mls(yy, 1:best_recu_df3_ylag, 1) + mls(zz3, best_recu_df3_lead:best_recu_df3_xlag, 3, nbetaMT),
                      start = list(zz3=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md3 <- midas_r(yy ~ mls(yy, 1:best_recu_df3_ylag, 1) + mls(zz3, best_recu_df3_lead:best_recu_df3_xlag, 3, nealmon),
                      start = list(zz3=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## DF4
best_recu_df4_ylag <- 0 + best_recu_df4_ylag
best_recu_df4_xlag <- 2 + best_recu_df4_xlag
if (best_recu_df4_weight == "Almon") {
  best_md4 <- midas_r(yy ~ mls(yy, 1:best_recu_df4_ylag, 1) + mls(zz4, best_recu_df4_lead:best_recu_df4_xlag, 3, almonp),
                      start = list(zz4=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df4_weight == "Beta") {
  best_md4 <- midas_r(yy ~ mls(yy, 1:best_recu_df4_ylag, 1) + mls(zz4, best_recu_df4_lead:best_recu_df4_xlag, 3, nbeta),
                      start = list(zz4=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df4_weight == "BetaNN") {
  best_md4 <- midas_r(yy ~ mls(yy, 1:best_recu_df4_ylag, 1) + mls(zz4, best_recu_df4_lead:best_recu_df4_xlag, 3, nbetaMT),
                      start = list(zz4=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md4 <- midas_r(yy ~ mls(yy, 1:best_recu_df4_ylag, 1) + mls(zz4, best_recu_df4_lead:best_recu_df4_xlag, 3, nealmon),
                      start = list(zz4=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## DF5
best_recu_df5_ylag <- 0 + best_recu_df5_ylag
best_recu_df5_xlag <- 2 + best_recu_df5_xlag
if (best_recu_df2_weight == "Almon") {
  best_md5 <- midas_r(yy ~ mls(yy, 1:best_recu_df5_ylag, 1) + mls(zz5, best_recu_df5_lead:best_recu_df5_xlag, 3, almonp),
                      start = list(zz5=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df5_weight == "Beta") {
  best_md5 <- midas_r(yy ~ mls(yy, 1:best_recu_df5_ylag, 1) + mls(zz5, best_recu_df5_lead:best_recu_df5_xlag, 3, nbeta),
                      start = list(zz5=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df5_weight == "BetaNN") {
  best_md5 <- midas_r(yy ~ mls(yy, 1:best_recu_df5_ylag, 1) + mls(zz5, best_recu_df5_lead:best_recu_df5_xlag, 3, nbetaMT),
                      start = list(zz5=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md5 <- midas_r(yy ~ mls(yy, 1:best_recu_df5_ylag, 1) + mls(zz5, best_recu_df5_lead:best_recu_df5_xlag, 3, nealmon),
                      start = list(zz5=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## z_pos_gdp
best_recu_zpos_gdp_xlag <- 0 + best_recu_zpos_gdp_xlag
best_recu_zpos_gdp_lead <- 2 + best_recu_zpos_gdp_lead
if (best_recu_zpos_gdp_weight == "Almon") {
  best_md6 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_gdp_ylag, 1) + mls(xx1, best_recu_zpos_gdp_lead:best_recu_zpos_gdp_xlag, 3, almonp),
                      start = list(xx1=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zpos_gdp_weight == "Beta") {
  best_md6 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_gdp_ylag, 1) + mls(xx1, best_recu_zpos_gdp_lead:best_recu_zpos_gdp_xlag, 3, nbeta),
                      start = list(xx1=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zpos_gdp_weight == "BetaNN") {
  best_md6 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_gdp_ylag, 1) + mls(xx1, best_recu_zpos_gdp_lead:best_recu_zpos_gdp_xlag, 3, nbetaMT),
                      start = list(xx1=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md6 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_gdp_ylag, 1) + mls(xx1, best_recu_zpos_gdp_lead:best_recu_zpos_gdp_xlag, 3, nealmon),
                      start = list(xx1=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## z_zero_gdp
best_recu_zzero_gdp_xlag <- 0 + best_recu_zzero_gdp_xlag
best_recu_zzero_gdp_lead <- 2 + best_recu_zzero_gdp_lead
if (best_recu_zzero_gdp_weight == "Almon") {
  best_md7 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_gdp_ylag, 1) + mls(xx2, best_recu_zzero_gdp_lead:best_recu_zzero_gdp_xlag, 3, almonp),
                      start = list(xx2=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zzero_gdp_weight == "Beta") {
  best_md7 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_gdp_ylag, 1) + mls(xx2, best_recu_zzero_gdp_lead:best_recu_zzero_gdp_xlag, 3, nbeta),
                      start = list(xx2=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zzero_gdp_weight == "BetaNN") {
  best_md7 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_gdp_ylag, 1) + mls(xx2, best_recu_zzero_gdp_lead:best_recu_zzero_gdp_xlag, 3, nbetaMT),
                      start = list(xx2=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md7 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_gdp_ylag, 1) + mls(xx2, best_recu_zzero_gdp_lead:best_recu_zzero_gdp_xlag, 3, nealmon),
                      start = list(xx2=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## z_pos_cpi
best_recu_zpos_cpi_ylag <- 0 + best_recu_zpos_cpi_ylag
best_recu_zpos_cpi_xlag <- 2 + best_recu_zpos_cpi_xlag
if (best_recu_zpos_cpi_weight == "Almon") {
  best_md8 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_cpi_ylag, 1) + mls(xx3, best_recu_zpos_cpi_lead:best_recu_zpos_cpi_xlag, 3, almonp),
                      start = list(xx3=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zpos_cpi_weight == "Beta") {
  best_md8 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_cpi_ylag, 1) + mls(xx3, best_recu_zpos_cpi_lead:best_recu_zpos_cpi_xlag, 3, nbeta),
                      start = list(xx3=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zpos_cpi_weight == "BetaNN") {
  best_md8 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_cpi_ylag, 1) + mls(xx3, best_recu_zpos_cpi_lead:best_recu_zpos_cpi_xlag, 3, nbetaMT),
                      start = list(xx3=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md8 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_cpi_ylag, 1) + mls(xx3, best_recu_zpos_cpi_lead:best_recu_zpos_cpi_xlag, 3, nealmon),
                      start = list(xx3=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## z_zero_cpi
best_recu_zzero_cpi_ylag <- 0 + best_recu_zzero_cpi_ylag
best_recu_zzero_cpi_xlag <- 2 + best_recu_zzero_cpi_xlag
if (best_recu_zzero_cpi_weight == "Almon") {
  best_md9 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_cpi_ylag, 1) + mls(xx4, best_recu_zzero_cpi_lead:best_recu_zzero_cpi_xlag, 3, almonp),
                      start = list(xx4=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zzero_cpi_weight == "Beta") {
  best_md9 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_cpi_ylag, 1) + mls(xx4, best_recu_zzero_cpi_lead:best_recu_zzero_cpi_xlag, 3, nbeta),
                      start = list(xx4=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zzero_cpi_weight == "BetaNN") {
  best_md9 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_cpi_ylag, 1) + mls(xx4, best_recu_zzero_cpi_lead:best_recu_zzero_cpi_xlag, 3, nbetaMT),
                      start = list(xx4=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md9 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_cpi_ylag, 1) + mls(xx4, best_recu_zzero_cpi_lead:best_recu_zzero_cpi_xlag, 3, nealmon),
                      start = list(xx4=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## z_pos_iva
best_recu_zpos_iva_ylag <- 0 + best_recu_zpos_iva_ylag
best_recu_zpos_iva_xlag <- 2 + best_recu_zpos_iva_xlag
if (best_recu_zpos_iva_weight == "Almon") {
  best_md10 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_iva_ylag, 1) + mls(xx5, best_recu_zpos_iva_lead:best_recu_zpos_iva_xlag, 3, almonp),
                       start = list(xx5=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_iva_weight == "Beta") {
  best_md10 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_iva_ylag, 1) + mls(xx5, best_recu_zpos_iva_lead:best_recu_zpos_iva_xlag, 3, nbeta),
                       start = list(xx5=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_iva_weight == "BetaNN") {
  best_md10 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_iva_ylag, 1) + mls(xx5, best_recu_zpos_iva_lead:best_recu_zpos_iva_xlag, 3, nbetaMT),
                       start = list(xx5=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md10 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_iva_ylag, 1) + mls(xx5, best_recu_zpos_iva_lead:best_recu_zpos_iva_xlag, 3, nealmon),
                       start = list(xx5=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}
### z_zero_iva
best_recu_zzero_iva_ylag <- 0 + best_recu_zzero_iva_ylag
best_recu_zzero_iva_xlag <- 2 + best_recu_zzero_iva_xlag
if (best_recu_zzero_iva_weight == "Almon") {
  best_md11 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_iva_ylag, 1) + mls(xx6, best_recu_zzero_iva_lead:best_recu_zzero_iva_xlag, 3, almonp),
                       start = list(xx6=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_iva_weight == "Beta") {
  best_md11 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_iva_ylag, 1) + mls(xx6, best_recu_zzero_iva_lead:best_recu_zzero_iva_xlag, 3, nbeta),
                       start = list(xx6=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_iva_weight == "BetaNN") {
  best_md11 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_iva_ylag, 1) + mls(xx6, best_recu_zzero_iva_lead:best_recu_zzero_iva_xlag, 3, nbetaMT),
                       start = list(xx6=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md11 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_iva_ylag, 1) + mls(xx6, best_recu_zzero_iva_lead:best_recu_zzero_iva_xlag, 3, nealmon),
                       start = list(xx6=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}
## z_pos_fai
best_recu_zpos_fai_ylag <- 0 + best_recu_zpos_fai_ylag
best_recu_zpos_fai_xlag <- 2 + best_recu_zpos_fai_xlag
if (best_recu_zpos_fai_weight == "Almon") {
  best_md12 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_fai_ylag, 1) + mls(xx7, best_recu_zpos_fai_lead:best_recu_zpos_fai_xlag, 3, almonp),
                       start = list(xx7=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_fai_weight == "Beta") {
  best_md12 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_fai_ylag, 1) + mls(xx7, best_recu_zpos_fai_lead:best_recu_zpos_fai_xlag, 3, nbeta),
                       start = list(xx7=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_fai_weight == "BetaNN") {
  best_md12 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_fai_ylag, 1) + mls(xx7, best_recu_zpos_fai_lead:best_recu_zpos_fai_xlag, 3, nbetaMT),
                       start = list(xx7=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md12 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_fai_ylag, 1) + mls(xx7, best_recu_zpos_fai_lead:best_recu_zpos_fai_xlag, 3, nealmon),
                       start = list(xx7=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}
### z_zero_fai
best_recu_zzero_fai_ylag <- 0 + best_recu_zzero_fai_ylag
best_recu_zzero_fai_xlag <- 2 + best_recu_zzero_fai_xlag
if (best_recu_zzero_fai_weight == "Almon") {
  best_md13 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_fai_ylag, 1) + mls(xx8, best_recu_zzero_fai_lead:best_recu_zzero_fai_xlag, 3, almonp),
                       start = list(xx8=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_fai_weight == "Beta") {
  best_md13 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_fai_ylag, 1) + mls(xx8, best_recu_zzero_fai_lead:best_recu_zzero_fai_xlag, 3, nbeta),
                       start = list(xx8=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_fai_weight == "BetaNN") {
  best_md13 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_fai_ylag, 1) + mls(xx8, best_recu_zzero_fai_lead:best_recu_zzero_fai_xlag, 3, nbetaMT),
                       start = list(xx8=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md13 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_fai_ylag, 1) + mls(xx8, best_recu_zzero_fai_lead:best_recu_zzero_fai_xlag, 3, nealmon),
                       start = list(xx8=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}
## z_pos_m2
best_recu_zpos_m2_ylag <- 0 + best_recu_zpos_m2_ylag
best_recu_zpos_m2_xlag <- 2 + best_recu_zpos_m2_xlag
if (best_recu_zpos_m2_weight == "Almon") {
  best_md14 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_m2_ylag, 1) + mls(xx9, best_recu_zpos_m2_lead:best_recu_zpos_m2_xlag, 3, almonp),
                       start = list(xx9=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_m2_weight == "Beta") {
  best_md14 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_m2_ylag, 1) + mls(xx9, best_recu_zpos_m2_lead:best_recu_zpos_m2_xlag, 3, nbeta),
                       start = list(xx9=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_m2_weight == "BetaNN") {
  best_md14 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_m2_ylag, 1) + mls(xx9, best_recu_zpos_m2_lead:best_recu_zpos_m2_xlag, 3, nbetaMT),
                       start = list(xx9=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md14 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_m2_ylag, 1) + mls(xx9, best_recu_zpos_m2_lead:best_recu_zpos_m2_xlag, 3, nealmon),
                       start = list(xx9=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}
### z_zero_m2
best_recu_zzero_m2_ylag <- 0 + best_recu_zzero_m2_ylag
best_recu_zzero_m2_xlag <- 2 + best_recu_zzero_m2_xlag
if (best_recu_zzero_m2_weight == "Almon") {
  best_md15 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_m2_ylag, 1) + mls(xx10, best_recu_zzero_m2_lead:best_recu_zzero_m2_xlag, 3, almonp),
                       start = list(xx10=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_m2_weight == "Beta") {
  best_md15 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_m2_ylag, 1) + mls(xx10, best_recu_zzero_m2_lead:best_recu_zzero_m2_xlag, 3, nbeta),
                       start = list(xx10=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_m2_weight == "BetaNN") {
  best_md15 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_m2_ylag, 1) + mls(xx10, best_recu_zzero_m2_lead:best_recu_zzero_m2_xlag, 3, nbetaMT),
                       start = list(xx10=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md15 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_m2_ylag, 1) + mls(xx10, best_recu_zzero_m2_lead:best_recu_zzero_m2_xlag, 3, nealmon),
                       start = list(xx10=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}
## z_pos_ppi
best_recu_zpos_ppi_ylag <- 0 + best_recu_zpos_ppi_ylag
best_recu_zpos_ppi_xlag <- 2 + best_recu_zpos_ppi_xlag
if (best_recu_zpos_ppi_weight == "Almon") {
  best_md16 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_ppi_ylag, 1) + mls(xx11, best_recu_zpos_ppi_lead:best_recu_zpos_ppi_xlag, 3, almonp),
                       start = list(xx11=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_ppi_weight == "Beta") {
  best_md16 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_ppi_ylag, 1) + mls(xx11, best_recu_zpos_ppi_lead:best_recu_zpos_ppi_xlag, 3, nbeta),
                       start = list(xx11=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_ppi_weight == "BetaNN") {
  best_md16 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_ppi_ylag, 1) + mls(xx11, best_recu_zpos_ppi_lead:best_recu_zpos_ppi_xlag, 3, nbetaMT),
                       start = list(xx11=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md16 <- midas_r(yy ~ mls(yy, 1:best_recu_zpos_ppi_ylag, 1) + mls(xx11, best_recu_zpos_ppi_lead:best_recu_zpos_ppi_xlag, 3, nealmon),
                       start = list(xx11=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}
### z_zero_ppi
best_recu_zzero_ppi_ylag <- 0 + best_recu_zzero_ppi_ylag
best_recu_zzero_ppi_xlag <- 2 + best_recu_zzero_ppi_xlag
if (best_recu_zzero_ppi_weight == "Almon") {
  best_md17 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_ppi_ylag, 1) + mls(xx12, best_recu_zzero_ppi_lead:best_recu_zzero_ppi_xlag, 3, almonp),
                       start = list(xx12=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_ppi_weight == "Beta") {
  best_md17 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_ppi_ylag, 1) + mls(xx12, best_recu_zzero_ppi_lead:best_recu_zzero_ppi_xlag, 3, nbeta),
                       start = list(xx12=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_ppi_weight == "BetaNN") {
  best_md17 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_ppi_ylag, 1) + mls(xx12, best_recu_zzero_ppi_lead:best_recu_zzero_ppi_xlag, 3, nbetaMT),
                       start = list(xx12=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md17 <- midas_r(yy ~ mls(yy, 1:best_recu_zzero_ppi_ylag, 1) + mls(xx12, best_recu_zzero_ppi_lead:best_recu_zzero_ppi_xlag, 3, nealmon),
                       start = list(xx12=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}

# Estimating & Forecasting
Recu_mf_ew = c()
Recu_mf_bicw = c()
Recu_mf_msfew = c()
Recu_mf_dmsfew = c()
Recu_zp_ew = c()
Recu_zp_bicw = c()
Recu_zp_msfew = c()
Recu_zp_dmsfew = c()
Recu_zz_ew = c()
Recu_zz_bicw = c()
Recu_zz_msfew = c()
Recu_zz_dmsfew = c()
Recu_cb_ew = c()
Recu_cb_bicw = c()
Recu_cb_msfew = c()
Recu_cb_dmsfew = c()
Recu_mfcb_ew = c()
Recu_mfcb_bicw = c()
Recu_mfcb_msfew = c()
Recu_mfcb_dmsfew = c()

for (i in seq(1, length(zp2)-3, 3)) {
  # Sampling
  p = floor((i-1)/3)
  l = ceiling(i/3)
  l1 = l + 1
  m1 = l + leny - 1
  m2 = m1 + 1
  j = i + lenx - 1
  k1 = j + 1
  k2 = j + 2
  k3 = j + 3
  i1 = i + 1
  i2 = i + 2
  i3 = i + 3
  
  # Start & End
  starty = c(2003,1)
  endy = starty + c(len, 0) + c(0,p)
  startx = c(2003,1)
  endx = startx + c(len, 1+i)
  fulldata <- list(
    yy = ts(na.omit(c(gdp[1:m1], gdp[m2])), start = starty, end = endy, frequency = 4), 
    xx1 = ts(c(zp1[1:j, i], zp1[k1, i1], zp1[k2, i2], zp1[k3, i3]), start = startx, end = endx, frequency = 12),
    xx2 = ts(c(zo1[1:j, i], zo1[k1, i1], zo1[k2, i2], zo1[k3, i3]), start = startx, end = endx, frequency = 12),
    xx3 = ts(c(zp2[1:j, i], zp2[k1, i1], zp2[k2, i2], zp2[k3, i3]), start = startx, end = endx, frequency = 12),
    xx4 = ts(c(zo2[1:j, i], zo2[k1, i1], zo2[k2, i2], zo2[k3, i3]), start = startx, end = endx, frequency = 12),
    xx5 = ts(c(zp3[1:j, i], zp3[k1, i1], zp3[k2, i2], zp3[k3, i3]), start = startx, end = endx, frequency = 12),
    xx6 = ts(c(zo3[1:j, i], zo3[k1, i1], zo3[k2, i2], zo3[k3, i3]), start = startx, end = endx, frequency = 12),
    xx7 = ts(c(zp4[1:j, i], zp4[k1, i1], zp4[k2, i2], zp4[k3, i3]), start = startx, end = endx, frequency = 12),
    xx8 = ts(c(zo4[1:j, i], zo4[k1, i1], zo4[k2, i2], zo4[k3, i3]), start = startx, end = endx, frequency = 12),
    xx9 = ts(c(zp5[1:j, i], zp5[k1, i1], zp5[k2, i2], zp5[k3, i3]), start = startx, end = endx, frequency = 12),
    xx10 = ts(c(zo5[1:j, i], zo5[k1, i1], zo5[k2, i2], zo5[k3, i3]), start = startx, end = endx, frequency = 12),
    xx11 = ts(c(zp6[1:j, i], zp6[k1, i1], zp6[k2, i2], zp6[k3, i3]), start = startx, end = endx, frequency = 12),
    xx12 = ts(c(zo6[1:j, i], zo6[k1, i1], zo6[k2, i2], zo6[k3, i3]), start = startx, end = endx, frequency = 12),
    zz1 = ts(c(df1[1:j, i], df1[k1, i1], df1[k2, i2], df1[k3, i3]), start = startx, end = endx, frequency = 12),
    zz2 = ts(c(df2[1:j, i], df2[k1, i1], df2[k2, i2], df2[k3, i3]), start = startx, end = endx, frequency = 12),
    zz3 = ts(c(df3[1:j, i], df3[k1, i1], df3[k2, i2], df3[k3, i3]), start = startx, end = endx, frequency = 12),
    zz4 = ts(c(df4[1:j, i], df4[k1, i1], df4[k2, i2], df4[k3, i3]), start = startx, end = endx, frequency = 12),
    zz5 = ts(c(df5[1:j, i], df5[k1, i1], df5[k2, i2], df5[k3, i3]), start = startx, end = endx, frequency = 12)
  )
  insample <- 1:m1
  outsample <- (1:length(fulldata$yy))[-insample]
  
  # Combination Model
  ## MF
  best_nowmdzz <- average_forecast(list(best_md1, best_md2, best_md3, best_md4, best_md5),
                                   data = fulldata, insample = insample, outsample = outsample,
                                   type = "recursive", 
                                   measures = c("MSE"),
                                   fweights = c("EW", "BICW", "MSFE", "DMSFE"),
                                   show_progress = FALSE)
  Recu_mf_ew <- append(Recu_mf_ew, best_nowmdzz[["avgforecast"]][1])
  Recu_mf_bicw <- append(Recu_mf_bicw, best_nowmdzz[["avgforecast"]][2])
  Recu_mf_msfew <- append(Recu_mf_msfew, best_nowmdzz[["avgforecast"]][3])
  Recu_mf_dmsfew <- append(Recu_mf_dmsfew, best_nowmdzz[["avgforecast"]][4])
  ## MF+CB_+
  best_nowmdxzp <- average_forecast(list(best_md1, best_md2, best_md3, best_md4, best_md5, best_md6),
                                    data = fulldata, insample = insample, outsample = outsample,
                                    type = "recursive", 
                                    measures = c("MSE"),
                                    fweights = c("EW", "BICW", "MSFE", "DMSFE"),
                                    show_progress = FALSE)
  Recu_zp_ew <- append(Recu_zp_ew, best_nowmdxzp[["avgforecast"]][1])
  Recu_zp_bicw <- append(Recu_zp_bicw, best_nowmdxzp[["avgforecast"]][2])
  Recu_zp_msfew <- append(Recu_zp_msfew, best_nowmdxzp[["avgforecast"]][3])
  Recu_zp_dmsfew <- append(Recu_zp_dmsfew, best_nowmdxzp[["avgforecast"]][4])  
  ## MF+CB_0
  best_nowmdxzz <- average_forecast(list(best_md1, best_md2, best_md3, best_md4, best_md5, best_md7),
                                    data = fulldata, insample = insample, outsample = outsample,
                                    type = "recursive", 
                                    measures = c("MSE"),
                                    fweights = c("EW", "BICW", "MSFE", "DMSFE"),
                                    show_progress = FALSE)
  Recu_zz_ew <- append(Recu_zz_ew, best_nowmdxzz[["avgforecast"]][1])
  Recu_zz_bicw <- append(Recu_zz_bicw, best_nowmdxzz[["avgforecast"]][2])
  Recu_zz_msfew <- append(Recu_zz_msfew, best_nowmdxzz[["avgforecast"]][3])
  Recu_zz_dmsfew <- append(Recu_zz_dmsfew, best_nowmdxzz[["avgforecast"]][4])      
  ## MF+CB_2
  best_nowmdxz_2 <- average_forecast(list(best_md1, best_md2, best_md3, best_md4,
                                          best_md5, best_md6, best_md7),
                                     data = fulldata, insample = insample, outsample = outsample,
                                     type = "recursive", 
                                     measures = c("MSE"),
                                     fweights = c("EW", "BICW", "MSFE", "DMSFE"),
                                     show_progress = FALSE)
  Recu_cb_ew <- append(Recu_cb_ew, best_nowmdxz_2[["avgforecast"]][1])
  Recu_cb_bicw <- append(Recu_cb_bicw, best_nowmdxz_2[["avgforecast"]][2])
  Recu_cb_msfew <- append(Recu_cb_msfew, best_nowmdxz_2[["avgforecast"]][3])
  Recu_cb_dmsfew <- append(Recu_cb_dmsfew, best_nowmdxz_2[["avgforecast"]][4])  
  ## MF+CB_12
  best_nowmdxz_12 <- average_forecast(list(best_md1, best_md2, best_md3, best_md4,
                                           best_md5, best_md6, best_md7, best_md8, best_md9,
                                           best_md10, best_md11, best_md12, best_md13,
                                           best_md14, best_md15, best_md16, best_md17),
                                      data = fulldata, insample = insample, outsample = outsample,
                                      type = "recursive", 
                                      measures = c("MSE"),
                                      fweights = c("EW", "BICW", "MSFE", "DMSFE"),
                                      show_progress = FALSE)
  Recu_mfcb_ew <- append(Recu_mfcb_ew, best_nowmdxz_12[["avgforecast"]][1])
  Recu_mfcb_bicw <- append(Recu_mfcb_bicw, best_nowmdxz_12[["avgforecast"]][2])
  Recu_mfcb_msfew <- append(Recu_mfcb_msfew, best_nowmdxz_12[["avgforecast"]][3])
  Recu_mfcb_dmsfew <- append(Recu_mfcb_dmsfew, best_nowmdxz_12[["avgforecast"]][4])
}

# Saving predicting results
best_result_rec_for1 <- data.frame(qdate = qdate,
                                   real = gdp,
                                   mf_ew = append(rep(0, leny), Recu_mf_ew),
                                   mf_bicw = append(rep(0, leny), Recu_mf_bicw),
                                   mf_msfew = append(rep(0, leny), Recu_mf_msfew),
                                   mf_dmsfew = append(rep(0, leny), Recu_mf_dmsfew),
                                   zp_ew = append(rep(0, leny), Recu_zp_ew),
                                   zp_bicw = append(rep(0, leny), Recu_zp_bicw),
                                   zp_msfew = append(rep(0, leny), Recu_zp_msfew),
                                   zp_dmsfew = append(rep(0, leny), Recu_zp_dmsfew),
                                   zz_ew = append(rep(0, leny), Recu_zz_ew),
                                   zz_bicw = append(rep(0, leny), Recu_zz_bicw),
                                   zz_msfew = append(rep(0, leny), Recu_zz_msfew),
                                   zz_dmsfew = append(rep(0, leny), Recu_zz_dmsfew),                              
                                   cb_ew = append(rep(0, leny), Recu_cb_ew),
                                   cb_bicw = append(rep(0, leny), Recu_cb_bicw),
                                   cb_msfew = append(rep(0, leny), Recu_cb_msfew),
                                   cb_dmsfew = append(rep(0, leny), Recu_cb_dmsfew),
                                   mfcb_ew = append(rep(0, leny), Recu_mfcb_ew),
                                   mfcb_bicw = append(rep(0, leny), Recu_mfcb_bicw),
                                   mfcb_msfew = append(rep(0, leny), Recu_mfcb_msfew),
                                   mfcb_dmsfew = append(rep(0, leny), Recu_mfcb_dmsfew))


#-------------------------------------
# Model Combination (h=2)
#-------------------------------------
result_rec_for2 <- na.omit(read_xlsx(paste0(tmpdatpath, "Forecast_GDP_Realtime.xlsx"), sheet = 3, col_names = T))

# Select best individual MIDAS model
best_recu_df1 <- result_rec_for2[which(result_rec_for2$Recu_df1 == min(result_rec_for2$Recu_df1)), ]
best_recu_df2 <- result_rec_for2[which(result_rec_for2$Recu_df2 == min(result_rec_for2$Recu_df2)), ]
best_recu_df3 <- result_rec_for2[which(result_rec_for2$Recu_df3 == min(result_rec_for2$Recu_df3)), ]
best_recu_df4 <- result_rec_for2[which(result_rec_for2$Recu_df4 == min(result_rec_for2$Recu_df4)), ]
best_recu_df5 <- result_rec_for2[which(result_rec_for2$Recu_df5 == min(result_rec_for2$Recu_df5)), ]
best_recu_z_pos_gdp <- result_rec_for2[which(result_rec_for2$Recu_zpos_gdp == min(result_rec_for2$Recu_zpos_gdp)), ][1,]
best_recu_z_zero_gdp <- result_rec_for2[which(result_rec_for2$Recu_zzero_gdp == min(result_rec_for2$Recu_zzero_gdp)), ][1,]
best_recu_z_pos_cpi <- result_rec_for2[which(result_rec_for2$Recu_zpos_cpi == min(result_rec_for2$Recu_zpos_cpi)), ]
best_recu_z_zero_cpi <- result_rec_for2[which(result_rec_for2$Recu_zzero_cpi == min(result_rec_for2$Recu_zzero_cpi)), ]
best_recu_z_pos_iva <- result_rec_for2[which(result_rec_for2$Recu_zpos_iva == min(result_rec_for2$Recu_zpos_iva)), ]
best_recu_z_zero_iva <- result_rec_for2[which(result_rec_for2$Recu_zzero_iva == min(result_rec_for2$Recu_zzero_iva)), ]
best_recu_z_pos_fai <- result_rec_for2[which(result_rec_for2$Recu_zpos_fai == min(result_rec_for2$Recu_zpos_fai)), ]
best_recu_z_zero_fai <- result_rec_for2[which(result_rec_for2$Recu_zzero_fai == min(result_rec_for2$Recu_zzero_fai)), ]
best_recu_z_pos_m2 <- result_rec_for2[which(result_rec_for2$Recu_zpos_m2 == min(result_rec_for2$Recu_zpos_m2)), ]
best_recu_z_zero_m2 <- result_rec_for2[which(result_rec_for2$Recu_zzero_m2 == min(result_rec_for2$Recu_zzero_m2)), ]
best_recu_z_pos_ppi <- result_rec_for2[which(result_rec_for2$Recu_zpos_ppi == min(result_rec_for2$Recu_zpos_ppi)), ]
best_recu_z_zero_ppi <- result_rec_for2[which(result_rec_for2$Recu_zzero_ppi == min(result_rec_for2$Recu_zzero_ppi)), ]

## DF1
best_recu_df1_weight <- best_recu_df1$Weightfunction
best_recu_df1_ylag <- best_recu_df1$Ylag
best_recu_df1_xlag <- best_recu_df1$Xlag
best_recu_df1_lead <- 6
best_recu_df1_tau <- "2"
## DF2
best_recu_df2_weight <- best_recu_df2$Weightfunction
best_recu_df2_ylag <- best_recu_df2$Ylag
best_recu_df2_xlag <- best_recu_df2$Xlag
best_recu_df2_lead <- 6
best_recu_df2_tau <- "2"
## DF3
best_recu_df3_weight <- best_recu_df3$Weightfunction
best_recu_df3_ylag <- best_recu_df3$Ylag
best_recu_df3_xlag <- best_recu_df3$Xlag
best_recu_df3_lead <- 6
best_recu_df3_tau <- "2"
## DF4
best_recu_df4_weight <- best_recu_df4$Weightfunction
best_recu_df4_ylag <- best_recu_df4$Ylag
best_recu_df4_xlag <- best_recu_df4$Xlag
best_recu_df4_lead <- 6
best_recu_df4_tau <- "2"
## DF5
best_recu_df5_weight <- best_recu_df5$Weightfunction
best_recu_df5_ylag <- best_recu_df5$Ylag
best_recu_df5_xlag <- best_recu_df5$Xlag
best_recu_df5_lead <- 6
best_recu_df5_tau <- "2"
## z_pos_gdp
best_recu_zpos_gdp_weight <- best_recu_z_pos_gdp$Weightfunction
best_recu_zpos_gdp_ylag <- best_recu_z_pos_gdp$Ylag
best_recu_zpos_gdp_xlag <- best_recu_z_pos_gdp$Xlag
best_recu_zpos_gdp_lead <- 6
best_recu_zpos_gdp_tau <- "2"
## z_zero_gdp
best_recu_zzero_gdp_weight <- best_recu_z_zero_gdp$Weightfunction
best_recu_zzero_gdp_ylag <- best_recu_z_zero_gdp$Ylag
best_recu_zzero_gdp_xlag <- best_recu_z_zero_gdp$Xlag
best_recu_zzero_gdp_lead <- 6
best_recu_zzero_gdp_tau <- "2"
## z_pos_cpi
best_recu_zpos_cpi_weight <- best_recu_z_pos_cpi$Weightfunction
best_recu_zpos_cpi_ylag <- best_recu_z_pos_cpi$Ylag
best_recu_zpos_cpi_xlag <- best_recu_z_pos_cpi$Xlag
best_recu_zpos_cpi_lead <- 6
best_recu_zpos_cpi_tau <- "2"
## z_zero_cpi
best_recu_zzero_cpi_weight <- best_recu_z_zero_cpi$Weightfunction
best_recu_zzero_cpi_ylag <- best_recu_z_zero_cpi$Ylag
best_recu_zzero_cpi_xlag <- best_recu_z_zero_cpi$Xlag
best_recu_zzero_cpi_lead <- 6
best_recu_zzero_cpi_tau <- "2"
## z_pos_iva
best_recu_zpos_iva_weight <- best_recu_z_pos_iva$Weightfunction
best_recu_zpos_iva_ylag <- best_recu_z_pos_iva$Ylag
best_recu_zpos_iva_xlag <- best_recu_z_pos_iva$Xlag
best_recu_zpos_iva_lead <- 6
best_recu_zpos_iva_tau <- "2"
## z_zero_iva
best_recu_zzero_iva_weight <- best_recu_z_zero_iva$Weightfunction
best_recu_zzero_iva_ylag <- best_recu_z_zero_iva$Ylag
best_recu_zzero_iva_xlag <- best_recu_z_zero_iva$Xlag
best_recu_zzero_iva_lead <- 6
best_recu_zzero_iva_tau <- "2"
## z_pos_fai
best_recu_zpos_fai_weight <- best_recu_z_pos_fai$Weightfunction
best_recu_zpos_fai_ylag <- best_recu_z_pos_fai$Ylag
best_recu_zpos_fai_xlag <- best_recu_z_pos_fai$Xlag
best_recu_zpos_fai_lead <- 6
best_recu_zpos_fai_tau <- "2"
## z_zero_fai
best_recu_zzero_fai_weight <- best_recu_z_zero_fai$Weightfunction
best_recu_zzero_fai_ylag <- best_recu_z_zero_fai$Ylag
best_recu_zzero_fai_xlag <- best_recu_z_zero_fai$Xlag
best_recu_zzero_fai_lead <- 6
best_recu_zzero_fai_tau <- "2"
## z_pos_m2
best_recu_zpos_m2_weight <- best_recu_z_pos_m2$Weightfunction
best_recu_zpos_m2_ylag <- best_recu_z_pos_m2$Ylag
best_recu_zpos_m2_xlag <- best_recu_z_pos_m2$Xlag
best_recu_zpos_m2_lead <- 6
best_recu_zpos_m2_tau <- "2"
## z_zero_m2
best_recu_zzero_m2_weight <- best_recu_z_zero_m2$Weightfunction
best_recu_zzero_m2_ylag <- best_recu_z_zero_m2$Ylag
best_recu_zzero_m2_xlag <- best_recu_z_zero_m2$Xlag
best_recu_zzero_m2_lead <- 6
best_recu_zzero_m2_tau <- "2"
## z_pos_ppi
best_recu_zpos_ppi_weight <- best_recu_z_pos_ppi$Weightfunction
best_recu_zpos_ppi_ylag <- best_recu_z_pos_ppi$Ylag
best_recu_zpos_ppi_xlag <- best_recu_z_pos_ppi$Xlag
best_recu_zpos_ppi_lead <- 6
best_recu_zpos_ppi_tau <- "2"
## z_zero_ppi
best_recu_zzero_ppi_weight <- best_recu_z_zero_ppi$Weightfunction
best_recu_zzero_ppi_ylag <- best_recu_z_zero_ppi$Ylag
best_recu_zzero_ppi_xlag <- best_recu_z_zero_ppi$Xlag
best_recu_zzero_ppi_lead <- 6
best_recu_zzero_ppi_tau <- "2"

# Training and combine best individual models to get combination MIDAS
## DF1
best_recu_df1_ylag <- 1 + best_recu_df1_ylag
best_recu_df1_xlag <- 5 + best_recu_df1_xlag
if (best_recu_df1_weight == "Almon") {
  best_md1 <- midas_r(yy ~ mls(yy, 2:best_recu_df1_ylag, 1) + mls(zz1, best_recu_df1_lead:best_recu_df1_xlag, 3, almonp),
                      start = list(zz1=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df1_weight == "Beta") {
  best_md1 <- midas_r(yy ~ mls(yy, 2:best_recu_df1_ylag, 1) + mls(zz1, best_recu_df1_lead:best_recu_df1_xlag, 3, nbeta),
                      start = list(zz1=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df1_weight == "BetaNN") {
  best_md1 <- midas_r(yy ~ mls(yy, 2:best_recu_df1_ylag, 1) + mls(zz1, best_recu_df1_lead:best_recu_df1_xlag, 3, nbetaMT),
                      start = list(zz1=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md1 <- midas_r(yy ~ mls(yy, 2:best_recu_df1_ylag, 1) + mls(zz1, best_recu_df1_lead:best_recu_df1_xlag, 3, nealmon),
                      start = list(zz1=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## DF2
best_recu_df2_ylag <- 1 + best_recu_df2_ylag
best_recu_df2_xlag <- 5 + best_recu_df2_xlag
if (best_recu_df2_weight == "Almon") {
  best_md2 <- midas_r(yy ~ mls(yy, 2:best_recu_df2_ylag, 1) + mls(zz2, best_recu_df2_lead:best_recu_df2_xlag, 3, almonp),
                      start = list(zz2=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df2_weight == "Beta") {
  best_md2 <- midas_r(yy ~ mls(yy, 2:best_recu_df2_ylag, 1) + mls(zz2, best_recu_df2_lead:best_recu_df2_xlag, 3, nbeta),
                      start = list(zz2=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df2_weight == "BetaNN") {
  best_md2 <- midas_r(yy ~ mls(yy, 2:best_recu_df2_ylag, 1) + mls(zz2, best_recu_df2_lead:best_recu_df2_xlag, 3, nbetaMT),
                      start = list(zz2=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md2 <- midas_r(yy ~ mls(yy, 2:best_recu_df2_ylag, 1) + mls(zz2, best_recu_df2_lead:best_recu_df2_xlag, 3, nealmon),
                      start = list(zz2=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## DF3
best_recu_df3_ylag <- 1 + best_recu_df3_ylag
best_recu_df3_xlag <- 5 + best_recu_df3_xlag
if (best_recu_df3_weight == "Almon") {
  best_md3 <- midas_r(yy ~ mls(yy, 2:best_recu_df3_ylag, 1) + mls(zz3, best_recu_df3_lead:best_recu_df3_xlag, 3, almonp),
                      start = list(zz3=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df3_weight == "Beta") {
  best_md3 <- midas_r(yy ~ mls(yy, 2:best_recu_df3_ylag, 1) + mls(zz3, best_recu_df3_lead:best_recu_df3_xlag, 3, nbeta),
                      start = list(zz3=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df3_weight == "BetaNN") {
  best_md3 <- midas_r(yy ~ mls(yy, 2:best_recu_df3_ylag, 1) + mls(zz3, best_recu_df3_lead:best_recu_df3_xlag, 3, nbetaMT),
                      start = list(zz3=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md3 <- midas_r(yy ~ mls(yy, 2:best_recu_df3_ylag, 1) + mls(zz3, best_recu_df3_lead:best_recu_df3_xlag, 3, nealmon),
                      start = list(zz3=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## DF4
best_recu_df4_ylag <- 1 + best_recu_df4_ylag
best_recu_df4_xlag <- 5 + best_recu_df4_xlag
if (best_recu_df4_weight == "Almon") {
  best_md4 <- midas_r(yy ~ mls(yy, 2:best_recu_df4_ylag, 1) + mls(zz4, best_recu_df4_lead:best_recu_df4_xlag, 3, almonp),
                      start = list(zz4=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df4_weight == "Beta") {
  best_md4 <- midas_r(yy ~ mls(yy, 2:best_recu_df4_ylag, 1) + mls(zz4, best_recu_df4_lead:best_recu_df4_xlag, 3, nbeta),
                      start = list(zz4=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df4_weight == "BetaNN") {
  best_md4 <- midas_r(yy ~ mls(yy, 2:best_recu_df4_ylag, 1) + mls(zz4, best_recu_df4_lead:best_recu_df4_xlag, 3, nbetaMT),
                      start = list(zz4=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md4 <- midas_r(yy ~ mls(yy, 2:best_recu_df4_ylag, 1) + mls(zz4, best_recu_df4_lead:best_recu_df4_xlag, 3, nealmon),
                      start = list(zz4=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## DF5
best_recu_df5_ylag <- 1 + best_recu_df5_ylag
best_recu_df5_xlag <- 5 + best_recu_df5_xlag
if (best_recu_df2_weight == "Almon") {
  best_md5 <- midas_r(yy ~ mls(yy, 2:best_recu_df5_ylag, 1) + mls(zz5, best_recu_df5_lead:best_recu_df5_xlag, 3, almonp),
                      start = list(zz5=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df5_weight == "Beta") {
  best_md5 <- midas_r(yy ~ mls(yy, 2:best_recu_df5_ylag, 1) + mls(zz5, best_recu_df5_lead:best_recu_df5_xlag, 3, nbeta),
                      start = list(zz5=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df5_weight == "BetaNN") {
  best_md5 <- midas_r(yy ~ mls(yy, 2:best_recu_df5_ylag, 1) + mls(zz5, best_recu_df5_lead:best_recu_df5_xlag, 3, nbetaMT),
                      start = list(zz5=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md5 <- midas_r(yy ~ mls(yy, 2:best_recu_df5_ylag, 1) + mls(zz5, best_recu_df5_lead:best_recu_df5_xlag, 3, nealmon),
                      start = list(zz5=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## z_pos_gdp
best_recu_zpos_gdp_xlag <- 1 + best_recu_zpos_gdp_xlag
best_recu_zpos_gdp_lead <- 5 + best_recu_zpos_gdp_lead
if (best_recu_zpos_gdp_weight == "Almon") {
  best_md6 <- midas_r(yy ~ mls(yy, 2:best_recu_zpos_gdp_ylag, 1) + mls(xx1, best_recu_zpos_gdp_lead:best_recu_zpos_gdp_xlag, 3, almonp),
                      start = list(xx1=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zpos_gdp_weight == "Beta") {
  best_md6 <- midas_r(yy ~ mls(yy, 2:best_recu_zpos_gdp_ylag, 1) + mls(xx1, best_recu_zpos_gdp_lead:best_recu_zpos_gdp_xlag, 3, nbeta),
                      start = list(xx1=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zpos_gdp_weight == "BetaNN") {
  best_md6 <- midas_r(yy ~ mls(yy, 2:best_recu_zpos_gdp_ylag, 1) + mls(xx1, best_recu_zpos_gdp_lead:best_recu_zpos_gdp_xlag, 3, nbetaMT),
                      start = list(xx1=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md6 <- midas_r(yy ~ mls(yy, 2:best_recu_zpos_gdp_ylag, 1) + mls(xx1, best_recu_zpos_gdp_lead:best_recu_zpos_gdp_xlag, 3, nealmon),
                      start = list(xx1=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## z_zero_gdp
best_recu_zzero_gdp_xlag <- 1 + best_recu_zzero_gdp_xlag
best_recu_zzero_gdp_lead <- 5 + best_recu_zzero_gdp_lead
if (best_recu_zzero_gdp_weight == "Almon") {
  best_md7 <- midas_r(yy ~ mls(yy, 2:best_recu_zzero_gdp_ylag, 1) + mls(xx2, best_recu_zzero_gdp_lead:best_recu_zzero_gdp_xlag, 3, almonp),
                      start = list(xx2=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zzero_gdp_weight == "Beta") {
  best_md7 <- midas_r(yy ~ mls(yy, 2:best_recu_zzero_gdp_ylag, 1) + mls(xx2, best_recu_zzero_gdp_lead:best_recu_zzero_gdp_xlag, 3, nbeta),
                      start = list(xx2=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zzero_gdp_weight == "BetaNN") {
  best_md7 <- midas_r(yy ~ mls(yy, 2:best_recu_zzero_gdp_ylag, 1) + mls(xx2, best_recu_zzero_gdp_lead:best_recu_zzero_gdp_xlag, 3, nbetaMT),
                      start = list(xx2=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md7 <- midas_r(yy ~ mls(yy, 2:best_recu_zzero_gdp_ylag, 1) + mls(xx2, best_recu_zzero_gdp_lead:best_recu_zzero_gdp_xlag, 3, nealmon),
                      start = list(xx2=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## z_pos_cpi
best_recu_zpos_cpi_ylag <- 1 + best_recu_zpos_cpi_ylag
best_recu_zpos_cpi_xlag <- 5 + best_recu_zpos_cpi_xlag
if (best_recu_zpos_cpi_weight == "Almon") {
  best_md8 <- midas_r(yy ~ mls(yy, 2:best_recu_zpos_cpi_ylag, 1) + mls(xx3, best_recu_zpos_cpi_lead:best_recu_zpos_cpi_xlag, 3, almonp),
                      start = list(xx3=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zpos_cpi_weight == "Beta") {
  best_md8 <- midas_r(yy ~ mls(yy, 2:best_recu_zpos_cpi_ylag, 1) + mls(xx3, best_recu_zpos_cpi_lead:best_recu_zpos_cpi_xlag, 3, nbeta),
                      start = list(xx3=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zpos_cpi_weight == "BetaNN") {
  best_md8 <- midas_r(yy ~ mls(yy, 2:best_recu_zpos_cpi_ylag, 1) + mls(xx3, best_recu_zpos_cpi_lead:best_recu_zpos_cpi_xlag, 3, nbetaMT),
                      start = list(xx3=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md8 <- midas_r(yy ~ mls(yy, 2:best_recu_zpos_cpi_ylag, 1) + mls(xx3, best_recu_zpos_cpi_lead:best_recu_zpos_cpi_xlag, 3, nealmon),
                      start = list(xx3=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## z_zero_cpi
best_recu_zzero_cpi_ylag <- 1 + best_recu_zzero_cpi_ylag
best_recu_zzero_cpi_xlag <- 5 + best_recu_zzero_cpi_xlag
if (best_recu_zzero_cpi_weight == "Almon") {
  best_md9 <- midas_r(yy ~ mls(yy, 2:best_recu_zzero_cpi_ylag, 1) + mls(xx4, best_recu_zzero_cpi_lead:best_recu_zzero_cpi_xlag, 3, almonp),
                      start = list(xx4=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zzero_cpi_weight == "Beta") {
  best_md9 <- midas_r(yy ~ mls(yy, 2:best_recu_zzero_cpi_ylag, 1) + mls(xx4, best_recu_zzero_cpi_lead:best_recu_zzero_cpi_xlag, 3, nbeta),
                      start = list(xx4=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zzero_cpi_weight == "BetaNN") {
  best_md9 <- midas_r(yy ~ mls(yy, 2:best_recu_zzero_cpi_ylag, 1) + mls(xx4, best_recu_zzero_cpi_lead:best_recu_zzero_cpi_xlag, 3, nbetaMT),
                      start = list(xx4=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md9 <- midas_r(yy ~ mls(yy, 2:best_recu_zzero_cpi_ylag, 1) + mls(xx4, best_recu_zzero_cpi_lead:best_recu_zzero_cpi_xlag, 3, nealmon),
                      start = list(xx4=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## z_pos_iva
best_recu_zpos_iva_ylag <- 1 + best_recu_zpos_iva_ylag
best_recu_zpos_iva_xlag <- 5 + best_recu_zpos_iva_xlag
if (best_recu_zpos_iva_weight == "Almon") {
  best_md10 <- midas_r(yy ~ mls(yy, 2:best_recu_zpos_iva_ylag, 1) + mls(xx5, best_recu_zpos_iva_lead:best_recu_zpos_iva_xlag, 3, almonp),
                       start = list(xx5=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_iva_weight == "Beta") {
  best_md10 <- midas_r(yy ~ mls(yy, 2:best_recu_zpos_iva_ylag, 1) + mls(xx5, best_recu_zpos_iva_lead:best_recu_zpos_iva_xlag, 3, nbeta),
                       start = list(xx5=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_iva_weight == "BetaNN") {
  best_md10 <- midas_r(yy ~ mls(yy, 2:best_recu_zpos_iva_ylag, 1) + mls(xx5, best_recu_zpos_iva_lead:best_recu_zpos_iva_xlag, 3, nbetaMT),
                       start = list(xx5=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md10 <- midas_r(yy ~ mls(yy, 2:best_recu_zpos_iva_ylag, 1) + mls(xx5, best_recu_zpos_iva_lead:best_recu_zpos_iva_xlag, 3, nealmon),
                       start = list(xx5=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}
### z_zero_iva
best_recu_zzero_iva_ylag <- 1 + best_recu_zzero_iva_ylag
best_recu_zzero_iva_xlag <- 5 + best_recu_zzero_iva_xlag
if (best_recu_zzero_iva_weight == "Almon") {
  best_md11 <- midas_r(yy ~ mls(yy, 2:best_recu_zzero_iva_ylag, 1) + mls(xx6, best_recu_zzero_iva_lead:best_recu_zzero_iva_xlag, 3, almonp),
                       start = list(xx6=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_iva_weight == "Beta") {
  best_md11 <- midas_r(yy ~ mls(yy, 2:best_recu_zzero_iva_ylag, 1) + mls(xx6, best_recu_zzero_iva_lead:best_recu_zzero_iva_xlag, 3, nbeta),
                       start = list(xx6=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_iva_weight == "BetaNN") {
  best_md11 <- midas_r(yy ~ mls(yy, 2:best_recu_zzero_iva_ylag, 1) + mls(xx6, best_recu_zzero_iva_lead:best_recu_zzero_iva_xlag, 3, nbetaMT),
                       start = list(xx6=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md11 <- midas_r(yy ~ mls(yy, 2:best_recu_zzero_iva_ylag, 1) + mls(xx6, best_recu_zzero_iva_lead:best_recu_zzero_iva_xlag, 3, nealmon),
                       start = list(xx6=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}
## z_pos_fai
best_recu_zpos_fai_ylag <- 1 + best_recu_zpos_fai_ylag
best_recu_zpos_fai_xlag <- 5 + best_recu_zpos_fai_xlag
if (best_recu_zpos_fai_weight == "Almon") {
  best_md12 <- midas_r(yy ~ mls(yy, 2:best_recu_zpos_fai_ylag, 1) + mls(xx7, best_recu_zpos_fai_lead:best_recu_zpos_fai_xlag, 3, almonp),
                       start = list(xx7=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_fai_weight == "Beta") {
  best_md12 <- midas_r(yy ~ mls(yy, 2:best_recu_zpos_fai_ylag, 1) + mls(xx7, best_recu_zpos_fai_lead:best_recu_zpos_fai_xlag, 3, nbeta),
                       start = list(xx7=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_fai_weight == "BetaNN") {
  best_md12 <- midas_r(yy ~ mls(yy, 2:best_recu_zpos_fai_ylag, 1) + mls(xx7, best_recu_zpos_fai_lead:best_recu_zpos_fai_xlag, 3, nbetaMT),
                       start = list(xx7=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md12 <- midas_r(yy ~ mls(yy, 2:best_recu_zpos_fai_ylag, 1) + mls(xx7, best_recu_zpos_fai_lead:best_recu_zpos_fai_xlag, 3, nealmon),
                       start = list(xx7=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}
### z_zero_fai
best_recu_zzero_fai_ylag <- 1 + best_recu_zzero_fai_ylag
best_recu_zzero_fai_xlag <- 5 + best_recu_zzero_fai_xlag
if (best_recu_zzero_fai_weight == "Almon") {
  best_md13 <- midas_r(yy ~ mls(yy, 2:best_recu_zzero_fai_ylag, 1) + mls(xx8, best_recu_zzero_fai_lead:best_recu_zzero_fai_xlag, 3, almonp),
                       start = list(xx8=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_fai_weight == "Beta") {
  best_md13 <- midas_r(yy ~ mls(yy, 2:best_recu_zzero_fai_ylag, 1) + mls(xx8, best_recu_zzero_fai_lead:best_recu_zzero_fai_xlag, 3, nbeta),
                       start = list(xx8=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_fai_weight == "BetaNN") {
  best_md13 <- midas_r(yy ~ mls(yy, 2:best_recu_zzero_fai_ylag, 1) + mls(xx8, best_recu_zzero_fai_lead:best_recu_zzero_fai_xlag, 3, nbetaMT),
                       start = list(xx8=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md13 <- midas_r(yy ~ mls(yy, 2:best_recu_zzero_fai_ylag, 1) + mls(xx8, best_recu_zzero_fai_lead:best_recu_zzero_fai_xlag, 3, nealmon),
                       start = list(xx8=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}
## z_pos_m2
best_recu_zpos_m2_ylag <- 1 + best_recu_zpos_m2_ylag
best_recu_zpos_m2_xlag <- 5 + best_recu_zpos_m2_xlag
if (best_recu_zpos_m2_weight == "Almon") {
  best_md14 <- midas_r(yy ~ mls(yy, 2:best_recu_zpos_m2_ylag, 1) + mls(xx9, best_recu_zpos_m2_lead:best_recu_zpos_m2_xlag, 3, almonp),
                       start = list(xx9=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_m2_weight == "Beta") {
  best_md14 <- midas_r(yy ~ mls(yy, 2:best_recu_zpos_m2_ylag, 1) + mls(xx9, best_recu_zpos_m2_lead:best_recu_zpos_m2_xlag, 3, nbeta),
                       start = list(xx9=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_m2_weight == "BetaNN") {
  best_md14 <- midas_r(yy ~ mls(yy, 2:best_recu_zpos_m2_ylag, 1) + mls(xx9, best_recu_zpos_m2_lead:best_recu_zpos_m2_xlag, 3, nbetaMT),
                       start = list(xx9=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md14 <- midas_r(yy ~ mls(yy, 2:best_recu_zpos_m2_ylag, 1) + mls(xx9, best_recu_zpos_m2_lead:best_recu_zpos_m2_xlag, 3, nealmon),
                       start = list(xx9=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}
### z_zero_m2
best_recu_zzero_m2_ylag <- 1 + best_recu_zzero_m2_ylag
best_recu_zzero_m2_xlag <- 5 + best_recu_zzero_m2_xlag
if (best_recu_zzero_m2_weight == "Almon") {
  best_md15 <- midas_r(yy ~ mls(yy, 2:best_recu_zzero_m2_ylag, 1) + mls(xx10, best_recu_zzero_m2_lead:best_recu_zzero_m2_xlag, 3, almonp),
                       start = list(xx10=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_m2_weight == "Beta") {
  best_md15 <- midas_r(yy ~ mls(yy, 2:best_recu_zzero_m2_ylag, 1) + mls(xx10, best_recu_zzero_m2_lead:best_recu_zzero_m2_xlag, 3, nbeta),
                       start = list(xx10=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_m2_weight == "BetaNN") {
  best_md15 <- midas_r(yy ~ mls(yy, 2:best_recu_zzero_m2_ylag, 1) + mls(xx10, best_recu_zzero_m2_lead:best_recu_zzero_m2_xlag, 3, nbetaMT),
                       start = list(xx10=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md15 <- midas_r(yy ~ mls(yy, 2:best_recu_zzero_m2_ylag, 1) + mls(xx10, best_recu_zzero_m2_lead:best_recu_zzero_m2_xlag, 3, nealmon),
                       start = list(xx10=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}
## z_pos_ppi
best_recu_zpos_ppi_ylag <- 1 + best_recu_zpos_ppi_ylag
best_recu_zpos_ppi_xlag <- 5 + best_recu_zpos_ppi_xlag
if (best_recu_zpos_ppi_weight == "Almon") {
  best_md16 <- midas_r(yy ~ mls(yy, 2:best_recu_zpos_ppi_ylag, 1) + mls(xx11, best_recu_zpos_ppi_lead:best_recu_zpos_ppi_xlag, 3, almonp),
                       start = list(xx11=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_ppi_weight == "Beta") {
  best_md16 <- midas_r(yy ~ mls(yy, 2:best_recu_zpos_ppi_ylag, 1) + mls(xx11, best_recu_zpos_ppi_lead:best_recu_zpos_ppi_xlag, 3, nbeta),
                       start = list(xx11=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_ppi_weight == "BetaNN") {
  best_md16 <- midas_r(yy ~ mls(yy, 2:best_recu_zpos_ppi_ylag, 1) + mls(xx11, best_recu_zpos_ppi_lead:best_recu_zpos_ppi_xlag, 3, nbetaMT),
                       start = list(xx11=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md16 <- midas_r(yy ~ mls(yy, 2:best_recu_zpos_ppi_ylag, 1) + mls(xx11, best_recu_zpos_ppi_lead:best_recu_zpos_ppi_xlag, 3, nealmon),
                       start = list(xx11=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}
### z_zero_ppi
best_recu_zzero_ppi_ylag <- 1 + best_recu_zzero_ppi_ylag
best_recu_zzero_ppi_xlag <- 5 + best_recu_zzero_ppi_xlag
if (best_recu_zzero_ppi_weight == "Almon") {
  best_md17 <- midas_r(yy ~ mls(yy, 2:best_recu_zzero_ppi_ylag, 1) + mls(xx12, best_recu_zzero_ppi_lead:best_recu_zzero_ppi_xlag, 3, almonp),
                       start = list(xx12=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_ppi_weight == "Beta") {
  best_md17 <- midas_r(yy ~ mls(yy, 2:best_recu_zzero_ppi_ylag, 1) + mls(xx12, best_recu_zzero_ppi_lead:best_recu_zzero_ppi_xlag, 3, nbeta),
                       start = list(xx12=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_ppi_weight == "BetaNN") {
  best_md17 <- midas_r(yy ~ mls(yy, 2:best_recu_zzero_ppi_ylag, 1) + mls(xx12, best_recu_zzero_ppi_lead:best_recu_zzero_ppi_xlag, 3, nbetaMT),
                       start = list(xx12=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md17 <- midas_r(yy ~ mls(yy, 2:best_recu_zzero_ppi_ylag, 1) + mls(xx12, best_recu_zzero_ppi_lead:best_recu_zzero_ppi_xlag, 3, nealmon),
                       start = list(xx12=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}

# Estimating & Forecasting
Recu_mf_ew = c()
Recu_mf_bicw = c()
Recu_mf_msfew = c()
Recu_mf_dmsfew = c()
Recu_zp_ew = c()
Recu_zp_bicw = c()
Recu_zp_msfew = c()
Recu_zp_dmsfew = c()
Recu_zz_ew = c()
Recu_zz_bicw = c()
Recu_zz_msfew = c()
Recu_zz_dmsfew = c()
Recu_cb_ew = c()
Recu_cb_bicw = c()
Recu_cb_msfew = c()
Recu_cb_dmsfew = c()
Recu_mfcb_ew = c()
Recu_mfcb_bicw = c()
Recu_mfcb_msfew = c()
Recu_mfcb_dmsfew = c()

for (i in seq(1, length(zp2)-3, 3)) {
  # Sampling
  p = floor((i-1)/3)
  l = ceiling(i/3)
  l1 = l + 1
  m1 = l + leny - 1
  m2 = m1 + 1
  j = i + lenx - 1
  k1 = j + 1
  k2 = j + 2
  k3 = j + 3
  i1 = i + 1
  i2 = i + 2
  i3 = i + 3
  
  # Start & End
  starty = c(2003,1)
  endy = starty + c(len, 0) + c(0,p)
  startx = c(2003,1)
  endx = startx + c(len, 1+i)
  fulldata <- list(
    yy = ts(na.omit(c(gdp[1:m1], gdp[m2])), start = starty, end = endy, frequency = 4), 
    xx1 = ts(c(zp1[1:j, i], zp1[k1, i1], zp1[k2, i2], zp1[k3, i3]), start = startx, end = endx, frequency = 12),
    xx2 = ts(c(zo1[1:j, i], zo1[k1, i1], zo1[k2, i2], zo1[k3, i3]), start = startx, end = endx, frequency = 12),
    xx3 = ts(c(zp2[1:j, i], zp2[k1, i1], zp2[k2, i2], zp2[k3, i3]), start = startx, end = endx, frequency = 12),
    xx4 = ts(c(zo2[1:j, i], zo2[k1, i1], zo2[k2, i2], zo2[k3, i3]), start = startx, end = endx, frequency = 12),
    xx5 = ts(c(zp3[1:j, i], zp3[k1, i1], zp3[k2, i2], zp3[k3, i3]), start = startx, end = endx, frequency = 12),
    xx6 = ts(c(zo3[1:j, i], zo3[k1, i1], zo3[k2, i2], zo3[k3, i3]), start = startx, end = endx, frequency = 12),
    xx7 = ts(c(zp4[1:j, i], zp4[k1, i1], zp4[k2, i2], zp4[k3, i3]), start = startx, end = endx, frequency = 12),
    xx8 = ts(c(zo4[1:j, i], zo4[k1, i1], zo4[k2, i2], zo4[k3, i3]), start = startx, end = endx, frequency = 12),
    xx9 = ts(c(zp5[1:j, i], zp5[k1, i1], zp5[k2, i2], zp5[k3, i3]), start = startx, end = endx, frequency = 12),
    xx10 = ts(c(zo5[1:j, i], zo5[k1, i1], zo5[k2, i2], zo5[k3, i3]), start = startx, end = endx, frequency = 12),
    xx11 = ts(c(zp6[1:j, i], zp6[k1, i1], zp6[k2, i2], zp6[k3, i3]), start = startx, end = endx, frequency = 12),
    xx12 = ts(c(zo6[1:j, i], zo6[k1, i1], zo6[k2, i2], zo6[k3, i3]), start = startx, end = endx, frequency = 12),
    zz1 = ts(c(df1[1:j, i], df1[k1, i1], df1[k2, i2], df1[k3, i3]), start = startx, end = endx, frequency = 12),
    zz2 = ts(c(df2[1:j, i], df2[k1, i1], df2[k2, i2], df2[k3, i3]), start = startx, end = endx, frequency = 12),
    zz3 = ts(c(df3[1:j, i], df3[k1, i1], df3[k2, i2], df3[k3, i3]), start = startx, end = endx, frequency = 12),
    zz4 = ts(c(df4[1:j, i], df4[k1, i1], df4[k2, i2], df4[k3, i3]), start = startx, end = endx, frequency = 12),
    zz5 = ts(c(df5[1:j, i], df5[k1, i1], df5[k2, i2], df5[k3, i3]), start = startx, end = endx, frequency = 12)
  )
  insample <- 1:m1
  outsample <- (1:length(fulldata$yy))[-insample]
  
  # Combination Model
  ## MF
  best_nowmdzz <- average_forecast(list(best_md1, best_md2, best_md3, best_md4, best_md5),
                                   data = fulldata, insample = insample, outsample = outsample,
                                   type = "recursive", 
                                   measures = c("MSE"),
                                   fweights = c("EW", "BICW", "MSFE", "DMSFE"),
                                   show_progress = FALSE)
  Recu_mf_ew <- append(Recu_mf_ew, best_nowmdzz[["avgforecast"]][1])
  Recu_mf_bicw <- append(Recu_mf_bicw, best_nowmdzz[["avgforecast"]][2])
  Recu_mf_msfew <- append(Recu_mf_msfew, best_nowmdzz[["avgforecast"]][3])
  Recu_mf_dmsfew <- append(Recu_mf_dmsfew, best_nowmdzz[["avgforecast"]][4])
  ## MF+CB_+
  best_nowmdxzp <- average_forecast(list(best_md1, best_md2, best_md3, best_md4, best_md5, best_md6),
                                    data = fulldata, insample = insample, outsample = outsample,
                                    type = "recursive", 
                                    measures = c("MSE"),
                                    fweights = c("EW", "BICW", "MSFE", "DMSFE"),
                                    show_progress = FALSE)
  Recu_zp_ew <- append(Recu_zp_ew, best_nowmdxzp[["avgforecast"]][1])
  Recu_zp_bicw <- append(Recu_zp_bicw, best_nowmdxzp[["avgforecast"]][2])
  Recu_zp_msfew <- append(Recu_zp_msfew, best_nowmdxzp[["avgforecast"]][3])
  Recu_zp_dmsfew <- append(Recu_zp_dmsfew, best_nowmdxzp[["avgforecast"]][4])  
  ## MF+CB_0
  best_nowmdxzz <- average_forecast(list(best_md1, best_md2, best_md3, best_md4, best_md5, best_md7),
                                    data = fulldata, insample = insample, outsample = outsample,
                                    type = "recursive", 
                                    measures = c("MSE"),
                                    fweights = c("EW", "BICW", "MSFE", "DMSFE"),
                                    show_progress = FALSE)
  Recu_zz_ew <- append(Recu_zz_ew, best_nowmdxzz[["avgforecast"]][1])
  Recu_zz_bicw <- append(Recu_zz_bicw, best_nowmdxzz[["avgforecast"]][2])
  Recu_zz_msfew <- append(Recu_zz_msfew, best_nowmdxzz[["avgforecast"]][3])
  Recu_zz_dmsfew <- append(Recu_zz_dmsfew, best_nowmdxzz[["avgforecast"]][4])        
  ## MF+CB_2
  best_nowmdxz_2 <- average_forecast(list(best_md1, best_md2, best_md3, best_md4,
                                          best_md5, best_md6, best_md7),
                                     data = fulldata, insample = insample, outsample = outsample,
                                     type = "recursive", 
                                     measures = c("MSE"),
                                     fweights = c("EW", "BICW", "MSFE", "DMSFE"),
                                     show_progress = FALSE)
  Recu_cb_ew <- append(Recu_cb_ew, best_nowmdxz_2[["avgforecast"]][1])
  Recu_cb_bicw <- append(Recu_cb_bicw, best_nowmdxz_2[["avgforecast"]][2])
  Recu_cb_msfew <- append(Recu_cb_msfew, best_nowmdxz_2[["avgforecast"]][3])
  Recu_cb_dmsfew <- append(Recu_cb_dmsfew, best_nowmdxz_2[["avgforecast"]][4])  
  ## MF+CB_12
  best_nowmdxz_12 <- average_forecast(list(best_md1, best_md2, best_md3, best_md4,
                                           best_md5, best_md6, best_md7, best_md8, best_md9,
                                           best_md10, best_md11, best_md12, best_md13,
                                           best_md14, best_md15, best_md16, best_md17),
                                      data = fulldata, insample = insample, outsample = outsample,
                                      type = "recursive", 
                                      measures = c("MSE"),
                                      fweights = c("EW", "BICW", "MSFE", "DMSFE"),
                                      show_progress = FALSE)
  Recu_mfcb_ew <- append(Recu_mfcb_ew, best_nowmdxz_12[["avgforecast"]][1])
  Recu_mfcb_bicw <- append(Recu_mfcb_bicw, best_nowmdxz_12[["avgforecast"]][2])
  Recu_mfcb_msfew <- append(Recu_mfcb_msfew, best_nowmdxz_12[["avgforecast"]][3])
  Recu_mfcb_dmsfew <- append(Recu_mfcb_dmsfew, best_nowmdxz_12[["avgforecast"]][4])
}

# Saving predicting results
best_result_rec_for2 <- data.frame(qdate = qdate,
                                   real = gdp,
                                   mf_ew = append(rep(0, leny), Recu_mf_ew),
                                   mf_bicw = append(rep(0, leny), Recu_mf_bicw),
                                   mf_msfew = append(rep(0, leny), Recu_mf_msfew),
                                   mf_dmsfew = append(rep(0, leny), Recu_mf_dmsfew),
                                   zp_ew = append(rep(0, leny), Recu_zp_ew),
                                   zp_bicw = append(rep(0, leny), Recu_zp_bicw),
                                   zp_msfew = append(rep(0, leny), Recu_zp_msfew),
                                   zp_dmsfew = append(rep(0, leny), Recu_zp_dmsfew),
                                   zz_ew = append(rep(0, leny), Recu_zz_ew),
                                   zz_bicw = append(rep(0, leny), Recu_zz_bicw),
                                   zz_msfew = append(rep(0, leny), Recu_zz_msfew),
                                   zz_dmsfew = append(rep(0, leny), Recu_zz_dmsfew),                                  
                                   cb_ew = append(rep(0, leny), Recu_cb_ew),
                                   cb_bicw = append(rep(0, leny), Recu_cb_bicw),
                                   cb_msfew = append(rep(0, leny), Recu_cb_msfew),
                                   cb_dmsfew = append(rep(0, leny), Recu_cb_dmsfew),
                                   mfcb_ew = append(rep(0, leny), Recu_mfcb_ew),
                                   mfcb_bicw = append(rep(0, leny), Recu_mfcb_bicw),
                                   mfcb_msfew = append(rep(0, leny), Recu_mfcb_msfew),
                                   mfcb_dmsfew = append(rep(0, leny), Recu_mfcb_dmsfew))


#-------------------------------------
# Model Combination (h=3)
#-------------------------------------
result_rec_for3 <- na.omit(read_xlsx(paste0(tmpdatpath, "Forecast_GDP_Realtime.xlsx"), sheet = 4, col_names = T))

# Select best individual MIDAS model
best_recu_df1 <- result_rec_for3[which(result_rec_for3$Recu_df1 == min(result_rec_for3$Recu_df1)), ]
best_recu_df2 <- result_rec_for3[which(result_rec_for3$Recu_df2 == min(result_rec_for3$Recu_df2)), ]
best_recu_df3 <- result_rec_for3[which(result_rec_for3$Recu_df3 == min(result_rec_for3$Recu_df3)), ]
best_recu_df4 <- result_rec_for3[which(result_rec_for3$Recu_df4 == min(result_rec_for3$Recu_df4)), ]
best_recu_df5 <- result_rec_for3[which(result_rec_for3$Recu_df5 == min(result_rec_for3$Recu_df5)), ]
best_recu_z_pos_gdp <- result_rec_for3[which(result_rec_for3$Recu_zpos_gdp == min(result_rec_for3$Recu_zpos_gdp)), ][1,]
best_recu_z_zero_gdp <- result_rec_for3[which(result_rec_for3$Recu_zzero_gdp == min(result_rec_for3$Recu_zzero_gdp)), ][1,]
best_recu_z_pos_cpi <- result_rec_for3[which(result_rec_for3$Recu_zpos_cpi == min(result_rec_for3$Recu_zpos_cpi)), ]
best_recu_z_zero_cpi <- result_rec_for3[which(result_rec_for3$Recu_zzero_cpi == min(result_rec_for3$Recu_zzero_cpi)), ]
best_recu_z_pos_iva <- result_rec_for3[which(result_rec_for3$Recu_zpos_iva == min(result_rec_for3$Recu_zpos_iva)), ]
best_recu_z_zero_iva <- result_rec_for3[which(result_rec_for3$Recu_zzero_iva == min(result_rec_for3$Recu_zzero_iva)), ]
best_recu_z_pos_fai <- result_rec_for3[which(result_rec_for3$Recu_zpos_fai == min(result_rec_for3$Recu_zpos_fai)), ]
best_recu_z_zero_fai <- result_rec_for3[which(result_rec_for3$Recu_zzero_fai == min(result_rec_for3$Recu_zzero_fai)), ]
best_recu_z_pos_m2 <- result_rec_for3[which(result_rec_for3$Recu_zpos_m2 == min(result_rec_for3$Recu_zpos_m2)), ]
best_recu_z_zero_m2 <- result_rec_for3[which(result_rec_for3$Recu_zzero_m2 == min(result_rec_for3$Recu_zzero_m2)), ]
best_recu_z_pos_ppi <- result_rec_for3[which(result_rec_for3$Recu_zpos_ppi == min(result_rec_for3$Recu_zpos_ppi)), ]
best_recu_z_zero_ppi <- result_rec_for3[which(result_rec_for3$Recu_zzero_ppi == min(result_rec_for3$Recu_zzero_ppi)), ]

## DF1
best_recu_df1_weight <- best_recu_df1$Weightfunction
best_recu_df1_ylag <- best_recu_df1$Ylag
best_recu_df1_xlag <- best_recu_df1$Xlag
best_recu_df1_lead <- 9
best_recu_df1_tau <- "3"
## DF2
best_recu_df2_weight <- best_recu_df2$Weightfunction
best_recu_df2_ylag <- best_recu_df2$Ylag
best_recu_df2_xlag <- best_recu_df2$Xlag
best_recu_df2_lead <- 9
best_recu_df2_tau <- "3"
## DF3
best_recu_df3_weight <- best_recu_df3$Weightfunction
best_recu_df3_ylag <- best_recu_df3$Ylag
best_recu_df3_xlag <- best_recu_df3$Xlag
best_recu_df3_lead <- 9
best_recu_df3_tau <- "3"
## DF4
best_recu_df4_weight <- best_recu_df4$Weightfunction
best_recu_df4_ylag <- best_recu_df4$Ylag
best_recu_df4_xlag <- best_recu_df4$Xlag
best_recu_df4_lead <- 9
best_recu_df4_tau <- "3"
## DF5
best_recu_df5_weight <- best_recu_df5$Weightfunction
best_recu_df5_ylag <- best_recu_df5$Ylag
best_recu_df5_xlag <- best_recu_df5$Xlag
best_recu_df5_lead <- 9
best_recu_df5_tau <- "3"
## z_pos_gdp
best_recu_zpos_gdp_weight <- best_recu_z_pos_gdp$Weightfunction
best_recu_zpos_gdp_ylag <- best_recu_z_pos_gdp$Ylag
best_recu_zpos_gdp_xlag <- best_recu_z_pos_gdp$Xlag
best_recu_zpos_gdp_lead <- 9
best_recu_zpos_gdp_tau <- "3"
## z_zero_gdp
best_recu_zzero_gdp_weight <- best_recu_z_zero_gdp$Weightfunction
best_recu_zzero_gdp_ylag <- best_recu_z_zero_gdp$Ylag
best_recu_zzero_gdp_xlag <- best_recu_z_zero_gdp$Xlag
best_recu_zzero_gdp_lead <- 9
best_recu_zzero_gdp_tau <- "3"
## z_pos_cpi
best_recu_zpos_cpi_weight <- best_recu_z_pos_cpi$Weightfunction
best_recu_zpos_cpi_ylag <- best_recu_z_pos_cpi$Ylag
best_recu_zpos_cpi_xlag <- best_recu_z_pos_cpi$Xlag
best_recu_zpos_cpi_lead <- 9
best_recu_zpos_cpi_tau <- "3"
## z_zero_cpi
best_recu_zzero_cpi_weight <- best_recu_z_zero_cpi$Weightfunction
best_recu_zzero_cpi_ylag <- best_recu_z_zero_cpi$Ylag
best_recu_zzero_cpi_xlag <- best_recu_z_zero_cpi$Xlag
best_recu_zzero_cpi_lead <- 9
best_recu_zzero_cpi_tau <- "3"
## z_pos_iva
best_recu_zpos_iva_weight <- best_recu_z_pos_iva$Weightfunction
best_recu_zpos_iva_ylag <- best_recu_z_pos_iva$Ylag
best_recu_zpos_iva_xlag <- best_recu_z_pos_iva$Xlag
best_recu_zpos_iva_lead <- 9
best_recu_zpos_iva_tau <- "3"
## z_zero_iva
best_recu_zzero_iva_weight <- best_recu_z_zero_iva$Weightfunction
best_recu_zzero_iva_ylag <- best_recu_z_zero_iva$Ylag
best_recu_zzero_iva_xlag <- best_recu_z_zero_iva$Xlag
best_recu_zzero_iva_lead <- 9
best_recu_zzero_iva_tau <- "3"
## z_pos_fai
best_recu_zpos_fai_weight <- best_recu_z_pos_fai$Weightfunction
best_recu_zpos_fai_ylag <- best_recu_z_pos_fai$Ylag
best_recu_zpos_fai_xlag <- best_recu_z_pos_fai$Xlag
best_recu_zpos_fai_lead <- 9
best_recu_zpos_fai_tau <- "3"
## z_zero_fai
best_recu_zzero_fai_weight <- best_recu_z_zero_fai$Weightfunction
best_recu_zzero_fai_ylag <- best_recu_z_zero_fai$Ylag
best_recu_zzero_fai_xlag <- best_recu_z_zero_fai$Xlag
best_recu_zzero_fai_lead <- 9
best_recu_zzero_fai_tau <- "3"
## z_pos_m2
best_recu_zpos_m2_weight <- best_recu_z_pos_m2$Weightfunction
best_recu_zpos_m2_ylag <- best_recu_z_pos_m2$Ylag
best_recu_zpos_m2_xlag <- best_recu_z_pos_m2$Xlag
best_recu_zpos_m2_lead <- 9
best_recu_zpos_m2_tau <- "3"
## z_zero_m2
best_recu_zzero_m2_weight <- best_recu_z_zero_m2$Weightfunction
best_recu_zzero_m2_ylag <- best_recu_z_zero_m2$Ylag
best_recu_zzero_m2_xlag <- best_recu_z_zero_m2$Xlag
best_recu_zzero_m2_lead <- 9
best_recu_zzero_m2_tau <- "3"
## z_pos_ppi
best_recu_zpos_ppi_weight <- best_recu_z_pos_ppi$Weightfunction
best_recu_zpos_ppi_ylag <- best_recu_z_pos_ppi$Ylag
best_recu_zpos_ppi_xlag <- best_recu_z_pos_ppi$Xlag
best_recu_zpos_ppi_lead <- 9
best_recu_zpos_ppi_tau <- "3"
## z_zero_ppi
best_recu_zzero_ppi_weight <- best_recu_z_zero_ppi$Weightfunction
best_recu_zzero_ppi_ylag <- best_recu_z_zero_ppi$Ylag
best_recu_zzero_ppi_xlag <- best_recu_z_zero_ppi$Xlag
best_recu_zzero_ppi_lead <- 9
best_recu_zzero_ppi_tau <- "3"

# Training and combine best individual models to get combination MIDAS
## DF1
best_recu_df1_ylag <- 2 + best_recu_df1_ylag
best_recu_df1_xlag <- 8 + best_recu_df1_xlag
if (best_recu_df1_weight == "Almon") {
  best_md1 <- midas_r(yy ~ mls(yy, 3:best_recu_df1_ylag, 1) + mls(zz1, best_recu_df1_lead:best_recu_df1_xlag, 3, almonp),
                      start = list(zz1=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df1_weight == "Beta") {
  best_md1 <- midas_r(yy ~ mls(yy, 3:best_recu_df1_ylag, 1) + mls(zz1, best_recu_df1_lead:best_recu_df1_xlag, 3, nbeta),
                      start = list(zz1=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df1_weight == "BetaNN") {
  best_md1 <- midas_r(yy ~ mls(yy, 3:best_recu_df1_ylag, 1) + mls(zz1, best_recu_df1_lead:best_recu_df1_xlag, 3, nbetaMT),
                      start = list(zz1=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md1 <- midas_r(yy ~ mls(yy, 3:best_recu_df1_ylag, 1) + mls(zz1, best_recu_df1_lead:best_recu_df1_xlag, 3, nealmon),
                      start = list(zz1=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## DF2
best_recu_df2_ylag <- 2 + best_recu_df2_ylag
best_recu_df2_xlag <- 8 + best_recu_df2_xlag
if (best_recu_df2_weight == "Almon") {
  best_md2 <- midas_r(yy ~ mls(yy, 3:best_recu_df2_ylag, 1) + mls(zz2, best_recu_df2_lead:best_recu_df2_xlag, 3, almonp),
                      start = list(zz2=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df2_weight == "Beta") {
  best_md2 <- midas_r(yy ~ mls(yy, 3:best_recu_df2_ylag, 1) + mls(zz2, best_recu_df2_lead:best_recu_df2_xlag, 3, nbeta),
                      start = list(zz2=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df2_weight == "BetaNN") {
  best_md2 <- midas_r(yy ~ mls(yy, 3:best_recu_df2_ylag, 1) + mls(zz2, best_recu_df2_lead:best_recu_df2_xlag, 3, nbetaMT),
                      start = list(zz2=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md2 <- midas_r(yy ~ mls(yy, 3:best_recu_df2_ylag, 1) + mls(zz2, best_recu_df2_lead:best_recu_df2_xlag, 3, nealmon),
                      start = list(zz2=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## DF3
best_recu_df3_ylag <- 2 + best_recu_df3_ylag
best_recu_df3_xlag <- 8 + best_recu_df3_xlag
if (best_recu_df3_weight == "Almon") {
  best_md3 <- midas_r(yy ~ mls(yy, 3:best_recu_df3_ylag, 1) + mls(zz3, best_recu_df3_lead:best_recu_df3_xlag, 3, almonp),
                      start = list(zz3=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df3_weight == "Beta") {
  best_md3 <- midas_r(yy ~ mls(yy, 3:best_recu_df3_ylag, 1) + mls(zz3, best_recu_df3_lead:best_recu_df3_xlag, 3, nbeta),
                      start = list(zz3=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df3_weight == "BetaNN") {
  best_md3 <- midas_r(yy ~ mls(yy, 3:best_recu_df3_ylag, 1) + mls(zz3, best_recu_df3_lead:best_recu_df3_xlag, 3, nbetaMT),
                      start = list(zz3=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md3 <- midas_r(yy ~ mls(yy, 3:best_recu_df3_ylag, 1) + mls(zz3, best_recu_df3_lead:best_recu_df3_xlag, 3, nealmon),
                      start = list(zz3=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## DF4
best_recu_df4_ylag <- 2 + best_recu_df4_ylag
best_recu_df4_xlag <- 8 + best_recu_df4_xlag
if (best_recu_df4_weight == "Almon") {
  best_md4 <- midas_r(yy ~ mls(yy, 3:best_recu_df4_ylag, 1) + mls(zz4, best_recu_df4_lead:best_recu_df4_xlag, 3, almonp),
                      start = list(zz4=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df4_weight == "Beta") {
  best_md4 <- midas_r(yy ~ mls(yy, 3:best_recu_df4_ylag, 1) + mls(zz4, best_recu_df4_lead:best_recu_df4_xlag, 3, nbeta),
                      start = list(zz4=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df4_weight == "BetaNN") {
  best_md4 <- midas_r(yy ~ mls(yy, 3:best_recu_df4_ylag, 1) + mls(zz4, best_recu_df4_lead:best_recu_df4_xlag, 3, nbetaMT),
                      start = list(zz4=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md4 <- midas_r(yy ~ mls(yy, 3:best_recu_df4_ylag, 1) + mls(zz4, best_recu_df4_lead:best_recu_df4_xlag, 3, nealmon),
                      start = list(zz4=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## DF5
best_recu_df5_ylag <- 2 + best_recu_df5_ylag
best_recu_df5_xlag <- 8 + best_recu_df5_xlag
if (best_recu_df2_weight == "Almon") {
  best_md5 <- midas_r(yy ~ mls(yy, 3:best_recu_df5_ylag, 1) + mls(zz5, best_recu_df5_lead:best_recu_df5_xlag, 3, almonp),
                      start = list(zz5=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df5_weight == "Beta") {
  best_md5 <- midas_r(yy ~ mls(yy, 3:best_recu_df5_ylag, 1) + mls(zz5, best_recu_df5_lead:best_recu_df5_xlag, 3, nbeta),
                      start = list(zz5=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df5_weight == "BetaNN") {
  best_md5 <- midas_r(yy ~ mls(yy, 3:best_recu_df5_ylag, 1) + mls(zz5, best_recu_df5_lead:best_recu_df5_xlag, 3, nbetaMT),
                      start = list(zz5=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md5 <- midas_r(yy ~ mls(yy, 3:best_recu_df5_ylag, 1) + mls(zz5, best_recu_df5_lead:best_recu_df5_xlag, 3, nealmon),
                      start = list(zz5=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## z_pos_gdp
best_recu_zpos_gdp_xlag <- 2 + best_recu_zpos_gdp_xlag
best_recu_zpos_gdp_lead <- 8 + best_recu_zpos_gdp_lead
if (best_recu_zpos_gdp_weight == "Almon") {
  best_md6 <- midas_r(yy ~ mls(yy, 3:best_recu_zpos_gdp_ylag, 1) + mls(xx1, best_recu_zpos_gdp_lead:best_recu_zpos_gdp_xlag, 3, almonp),
                      start = list(xx1=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zpos_gdp_weight == "Beta") {
  best_md6 <- midas_r(yy ~ mls(yy, 3:best_recu_zpos_gdp_ylag, 1) + mls(xx1, best_recu_zpos_gdp_lead:best_recu_zpos_gdp_xlag, 3, nbeta),
                      start = list(xx1=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zpos_gdp_weight == "BetaNN") {
  best_md6 <- midas_r(yy ~ mls(yy, 3:best_recu_zpos_gdp_ylag, 1) + mls(xx1, best_recu_zpos_gdp_lead:best_recu_zpos_gdp_xlag, 3, nbetaMT),
                      start = list(xx1=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md6 <- midas_r(yy ~ mls(yy, 3:best_recu_zpos_gdp_ylag, 1) + mls(xx1, best_recu_zpos_gdp_lead:best_recu_zpos_gdp_xlag, 3, nealmon),
                      start = list(xx1=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## z_zero_gdp
best_recu_zzero_gdp_xlag <- 2 + best_recu_zzero_gdp_xlag
best_recu_zzero_gdp_lead <- 8 + best_recu_zzero_gdp_lead
if (best_recu_zzero_gdp_weight == "Almon") {
  best_md7 <- midas_r(yy ~ mls(yy, 3:best_recu_zzero_gdp_ylag, 1) + mls(xx2, best_recu_zzero_gdp_lead:best_recu_zzero_gdp_xlag, 3, almonp),
                      start = list(xx2=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zzero_gdp_weight == "Beta") {
  best_md7 <- midas_r(yy ~ mls(yy, 3:best_recu_zzero_gdp_ylag, 1) + mls(xx2, best_recu_zzero_gdp_lead:best_recu_zzero_gdp_xlag, 3, nbeta),
                      start = list(xx2=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zzero_gdp_weight == "BetaNN") {
  best_md7 <- midas_r(yy ~ mls(yy, 3:best_recu_zzero_gdp_ylag, 1) + mls(xx2, best_recu_zzero_gdp_lead:best_recu_zzero_gdp_xlag, 3, nbetaMT),
                      start = list(xx2=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md7 <- midas_r(yy ~ mls(yy, 3:best_recu_zzero_gdp_ylag, 1) + mls(xx2, best_recu_zzero_gdp_lead:best_recu_zzero_gdp_xlag, 3, nealmon),
                      start = list(xx2=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## z_pos_cpi
best_recu_zpos_cpi_ylag <- 2 + best_recu_zpos_cpi_ylag
best_recu_zpos_cpi_xlag <- 8 + best_recu_zpos_cpi_xlag
if (best_recu_zpos_cpi_weight == "Almon") {
  best_md8 <- midas_r(yy ~ mls(yy, 3:best_recu_zpos_cpi_ylag, 1) + mls(xx3, best_recu_zpos_cpi_lead:best_recu_zpos_cpi_xlag, 3, almonp),
                      start = list(xx3=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zpos_cpi_weight == "Beta") {
  best_md8 <- midas_r(yy ~ mls(yy, 3:best_recu_zpos_cpi_ylag, 1) + mls(xx3, best_recu_zpos_cpi_lead:best_recu_zpos_cpi_xlag, 3, nbeta),
                      start = list(xx3=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zpos_cpi_weight == "BetaNN") {
  best_md8 <- midas_r(yy ~ mls(yy, 3:best_recu_zpos_cpi_ylag, 1) + mls(xx3, best_recu_zpos_cpi_lead:best_recu_zpos_cpi_xlag, 3, nbetaMT),
                      start = list(xx3=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md8 <- midas_r(yy ~ mls(yy, 3:best_recu_zpos_cpi_ylag, 1) + mls(xx3, best_recu_zpos_cpi_lead:best_recu_zpos_cpi_xlag, 3, nealmon),
                      start = list(xx3=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## z_zero_cpi
best_recu_zzero_cpi_ylag <- 2 + best_recu_zzero_cpi_ylag
best_recu_zzero_cpi_xlag <- 8 + best_recu_zzero_cpi_xlag
if (best_recu_zzero_cpi_weight == "Almon") {
  best_md9 <- midas_r(yy ~ mls(yy, 3:best_recu_zzero_cpi_ylag, 1) + mls(xx4, best_recu_zzero_cpi_lead:best_recu_zzero_cpi_xlag, 3, almonp),
                      start = list(xx4=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zzero_cpi_weight == "Beta") {
  best_md9 <- midas_r(yy ~ mls(yy, 3:best_recu_zzero_cpi_ylag, 1) + mls(xx4, best_recu_zzero_cpi_lead:best_recu_zzero_cpi_xlag, 3, nbeta),
                      start = list(xx4=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zzero_cpi_weight == "BetaNN") {
  best_md9 <- midas_r(yy ~ mls(yy, 3:best_recu_zzero_cpi_ylag, 1) + mls(xx4, best_recu_zzero_cpi_lead:best_recu_zzero_cpi_xlag, 3, nbetaMT),
                      start = list(xx4=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md9 <- midas_r(yy ~ mls(yy, 3:best_recu_zzero_cpi_ylag, 1) + mls(xx4, best_recu_zzero_cpi_lead:best_recu_zzero_cpi_xlag, 3, nealmon),
                      start = list(xx4=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## z_pos_iva
best_recu_zpos_iva_ylag <- 2 + best_recu_zpos_iva_ylag
best_recu_zpos_iva_xlag <- 8 + best_recu_zpos_iva_xlag
if (best_recu_zpos_iva_weight == "Almon") {
  best_md10 <- midas_r(yy ~ mls(yy, 3:best_recu_zpos_iva_ylag, 1) + mls(xx5, best_recu_zpos_iva_lead:best_recu_zpos_iva_xlag, 3, almonp),
                       start = list(xx5=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_iva_weight == "Beta") {
  best_md10 <- midas_r(yy ~ mls(yy, 3:best_recu_zpos_iva_ylag, 1) + mls(xx5, best_recu_zpos_iva_lead:best_recu_zpos_iva_xlag, 3, nbeta),
                       start = list(xx5=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_iva_weight == "BetaNN") {
  best_md10 <- midas_r(yy ~ mls(yy, 3:best_recu_zpos_iva_ylag, 1) + mls(xx5, best_recu_zpos_iva_lead:best_recu_zpos_iva_xlag, 3, nbetaMT),
                       start = list(xx5=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md10 <- midas_r(yy ~ mls(yy, 3:best_recu_zpos_iva_ylag, 1) + mls(xx5, best_recu_zpos_iva_lead:best_recu_zpos_iva_xlag, 3, nealmon),
                       start = list(xx5=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}
### z_zero_iva
best_recu_zzero_iva_ylag <- 2 + best_recu_zzero_iva_ylag
best_recu_zzero_iva_xlag <- 8 + best_recu_zzero_iva_xlag
if (best_recu_zzero_iva_weight == "Almon") {
  best_md11 <- midas_r(yy ~ mls(yy, 3:best_recu_zzero_iva_ylag, 1) + mls(xx6, best_recu_zzero_iva_lead:best_recu_zzero_iva_xlag, 3, almonp),
                       start = list(xx6=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_iva_weight == "Beta") {
  best_md11 <- midas_r(yy ~ mls(yy, 3:best_recu_zzero_iva_ylag, 1) + mls(xx6, best_recu_zzero_iva_lead:best_recu_zzero_iva_xlag, 3, nbeta),
                       start = list(xx6=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_iva_weight == "BetaNN") {
  best_md11 <- midas_r(yy ~ mls(yy, 3:best_recu_zzero_iva_ylag, 1) + mls(xx6, best_recu_zzero_iva_lead:best_recu_zzero_iva_xlag, 3, nbetaMT),
                       start = list(xx6=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md11 <- midas_r(yy ~ mls(yy, 3:best_recu_zzero_iva_ylag, 1) + mls(xx6, best_recu_zzero_iva_lead:best_recu_zzero_iva_xlag, 3, nealmon),
                       start = list(xx6=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}
## z_pos_fai
best_recu_zpos_fai_ylag <- 2 + best_recu_zpos_fai_ylag
best_recu_zpos_fai_xlag <- 8 + best_recu_zpos_fai_xlag
if (best_recu_zpos_fai_weight == "Almon") {
  best_md12 <- midas_r(yy ~ mls(yy, 3:best_recu_zpos_fai_ylag, 1) + mls(xx7, best_recu_zpos_fai_lead:best_recu_zpos_fai_xlag, 3, almonp),
                       start = list(xx7=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_fai_weight == "Beta") {
  best_md12 <- midas_r(yy ~ mls(yy, 3:best_recu_zpos_fai_ylag, 1) + mls(xx7, best_recu_zpos_fai_lead:best_recu_zpos_fai_xlag, 3, nbeta),
                       start = list(xx7=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_fai_weight == "BetaNN") {
  best_md12 <- midas_r(yy ~ mls(yy, 3:best_recu_zpos_fai_ylag, 1) + mls(xx7, best_recu_zpos_fai_lead:best_recu_zpos_fai_xlag, 3, nbetaMT),
                       start = list(xx7=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md12 <- midas_r(yy ~ mls(yy, 3:best_recu_zpos_fai_ylag, 1) + mls(xx7, best_recu_zpos_fai_lead:best_recu_zpos_fai_xlag, 3, nealmon),
                       start = list(xx7=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}
### z_zero_fai
best_recu_zzero_fai_ylag <- 2 + best_recu_zzero_fai_ylag
best_recu_zzero_fai_xlag <- 8 + best_recu_zzero_fai_xlag
if (best_recu_zzero_fai_weight == "Almon") {
  best_md13 <- midas_r(yy ~ mls(yy, 3:best_recu_zzero_fai_ylag, 1) + mls(xx8, best_recu_zzero_fai_lead:best_recu_zzero_fai_xlag, 3, almonp),
                       start = list(xx8=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_fai_weight == "Beta") {
  best_md13 <- midas_r(yy ~ mls(yy, 3:best_recu_zzero_fai_ylag, 1) + mls(xx8, best_recu_zzero_fai_lead:best_recu_zzero_fai_xlag, 3, nbeta),
                       start = list(xx8=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_fai_weight == "BetaNN") {
  best_md13 <- midas_r(yy ~ mls(yy, 3:best_recu_zzero_fai_ylag, 1) + mls(xx8, best_recu_zzero_fai_lead:best_recu_zzero_fai_xlag, 3, nbetaMT),
                       start = list(xx8=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md13 <- midas_r(yy ~ mls(yy, 3:best_recu_zzero_fai_ylag, 1) + mls(xx8, best_recu_zzero_fai_lead:best_recu_zzero_fai_xlag, 3, nealmon),
                       start = list(xx8=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}
## z_pos_m2
best_recu_zpos_m2_ylag <- 2 + best_recu_zpos_m2_ylag
best_recu_zpos_m2_xlag <- 8 + best_recu_zpos_m2_xlag
if (best_recu_zpos_m2_weight == "Almon") {
  best_md14 <- midas_r(yy ~ mls(yy, 3:best_recu_zpos_m2_ylag, 1) + mls(xx9, best_recu_zpos_m2_lead:best_recu_zpos_m2_xlag, 3, almonp),
                       start = list(xx9=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_m2_weight == "Beta") {
  best_md14 <- midas_r(yy ~ mls(yy, 3:best_recu_zpos_m2_ylag, 1) + mls(xx9, best_recu_zpos_m2_lead:best_recu_zpos_m2_xlag, 3, nbeta),
                       start = list(xx9=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_m2_weight == "BetaNN") {
  best_md14 <- midas_r(yy ~ mls(yy, 3:best_recu_zpos_m2_ylag, 1) + mls(xx9, best_recu_zpos_m2_lead:best_recu_zpos_m2_xlag, 3, nbetaMT),
                       start = list(xx9=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md14 <- midas_r(yy ~ mls(yy, 3:best_recu_zpos_m2_ylag, 1) + mls(xx9, best_recu_zpos_m2_lead:best_recu_zpos_m2_xlag, 3, nealmon),
                       start = list(xx9=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}
### z_zero_m2
best_recu_zzero_m2_ylag <- 2 + best_recu_zzero_m2_ylag
best_recu_zzero_m2_xlag <- 8 + best_recu_zzero_m2_xlag
if (best_recu_zzero_m2_weight == "Almon") {
  best_md15 <- midas_r(yy ~ mls(yy, 3:best_recu_zzero_m2_ylag, 1) + mls(xx10, best_recu_zzero_m2_lead:best_recu_zzero_m2_xlag, 3, almonp),
                       start = list(xx10=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_m2_weight == "Beta") {
  best_md15 <- midas_r(yy ~ mls(yy, 3:best_recu_zzero_m2_ylag, 1) + mls(xx10, best_recu_zzero_m2_lead:best_recu_zzero_m2_xlag, 3, nbeta),
                       start = list(xx10=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_m2_weight == "BetaNN") {
  best_md15 <- midas_r(yy ~ mls(yy, 3:best_recu_zzero_m2_ylag, 1) + mls(xx10, best_recu_zzero_m2_lead:best_recu_zzero_m2_xlag, 3, nbetaMT),
                       start = list(xx10=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md15 <- midas_r(yy ~ mls(yy, 3:best_recu_zzero_m2_ylag, 1) + mls(xx10, best_recu_zzero_m2_lead:best_recu_zzero_m2_xlag, 3, nealmon),
                       start = list(xx10=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}
## z_pos_ppi
best_recu_zpos_ppi_ylag <- 2 + best_recu_zpos_ppi_ylag
best_recu_zpos_ppi_xlag <- 8 + best_recu_zpos_ppi_xlag
if (best_recu_zpos_ppi_weight == "Almon") {
  best_md16 <- midas_r(yy ~ mls(yy, 3:best_recu_zpos_ppi_ylag, 1) + mls(xx11, best_recu_zpos_ppi_lead:best_recu_zpos_ppi_xlag, 3, almonp),
                       start = list(xx11=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_ppi_weight == "Beta") {
  best_md16 <- midas_r(yy ~ mls(yy, 3:best_recu_zpos_ppi_ylag, 1) + mls(xx11, best_recu_zpos_ppi_lead:best_recu_zpos_ppi_xlag, 3, nbeta),
                       start = list(xx11=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_ppi_weight == "BetaNN") {
  best_md16 <- midas_r(yy ~ mls(yy, 3:best_recu_zpos_ppi_ylag, 1) + mls(xx11, best_recu_zpos_ppi_lead:best_recu_zpos_ppi_xlag, 3, nbetaMT),
                       start = list(xx11=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md16 <- midas_r(yy ~ mls(yy, 3:best_recu_zpos_ppi_ylag, 1) + mls(xx11, best_recu_zpos_ppi_lead:best_recu_zpos_ppi_xlag, 3, nealmon),
                       start = list(xx11=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}
### z_zero_ppi
best_recu_zzero_ppi_ylag <- 2 + best_recu_zzero_ppi_ylag
best_recu_zzero_ppi_xlag <- 8 + best_recu_zzero_ppi_xlag
if (best_recu_zzero_ppi_weight == "Almon") {
  best_md17 <- midas_r(yy ~ mls(yy, 3:best_recu_zzero_ppi_ylag, 1) + mls(xx12, best_recu_zzero_ppi_lead:best_recu_zzero_ppi_xlag, 3, almonp),
                       start = list(xx12=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_ppi_weight == "Beta") {
  best_md17 <- midas_r(yy ~ mls(yy, 3:best_recu_zzero_ppi_ylag, 1) + mls(xx12, best_recu_zzero_ppi_lead:best_recu_zzero_ppi_xlag, 3, nbeta),
                       start = list(xx12=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_ppi_weight == "BetaNN") {
  best_md17 <- midas_r(yy ~ mls(yy, 3:best_recu_zzero_ppi_ylag, 1) + mls(xx12, best_recu_zzero_ppi_lead:best_recu_zzero_ppi_xlag, 3, nbetaMT),
                       start = list(xx12=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md17 <- midas_r(yy ~ mls(yy, 3:best_recu_zzero_ppi_ylag, 1) + mls(xx12, best_recu_zzero_ppi_lead:best_recu_zzero_ppi_xlag, 3, nealmon),
                       start = list(xx12=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}

# Estimating & Forecasting
Recu_mf_ew = c()
Recu_mf_bicw = c()
Recu_mf_msfew = c()
Recu_mf_dmsfew = c()
Recu_zp_ew = c()
Recu_zp_bicw = c()
Recu_zp_msfew = c()
Recu_zp_dmsfew = c()
Recu_zz_ew = c()
Recu_zz_bicw = c()
Recu_zz_msfew = c()
Recu_zz_dmsfew = c()
Recu_cb_ew = c()
Recu_cb_bicw = c()
Recu_cb_msfew = c()
Recu_cb_dmsfew = c()
Recu_mfcb_ew = c()
Recu_mfcb_bicw = c()
Recu_mfcb_msfew = c()
Recu_mfcb_dmsfew = c()

for (i in seq(1, length(zp2)-3, 3)) {
  # Sampling
  p = floor((i-1)/3)
  l = ceiling(i/3)
  l1 = l + 1
  m1 = l + leny - 1
  m2 = m1 + 1
  j = i + lenx - 1
  k1 = j + 1
  k2 = j + 2
  k3 = j + 3
  i1 = i + 1
  i2 = i + 2
  i3 = i + 3
  
  # Start & End
  starty = c(2003,1)
  endy = starty + c(len, 0) + c(0,p)
  startx = c(2003,1)
  endx = startx + c(len, 1+i)
  fulldata <- list(
    yy = ts(na.omit(c(gdp[1:m1], gdp[m2])), start = starty, end = endy, frequency = 4), 
    xx1 = ts(c(zp1[1:j, i], zp1[k1, i1], zp1[k2, i2], zp1[k3, i3]), start = startx, end = endx, frequency = 12),
    xx2 = ts(c(zo1[1:j, i], zo1[k1, i1], zo1[k2, i2], zo1[k3, i3]), start = startx, end = endx, frequency = 12),
    xx3 = ts(c(zp2[1:j, i], zp2[k1, i1], zp2[k2, i2], zp2[k3, i3]), start = startx, end = endx, frequency = 12),
    xx4 = ts(c(zo2[1:j, i], zo2[k1, i1], zo2[k2, i2], zo2[k3, i3]), start = startx, end = endx, frequency = 12),
    xx5 = ts(c(zp3[1:j, i], zp3[k1, i1], zp3[k2, i2], zp3[k3, i3]), start = startx, end = endx, frequency = 12),
    xx6 = ts(c(zo3[1:j, i], zo3[k1, i1], zo3[k2, i2], zo3[k3, i3]), start = startx, end = endx, frequency = 12),
    xx7 = ts(c(zp4[1:j, i], zp4[k1, i1], zp4[k2, i2], zp4[k3, i3]), start = startx, end = endx, frequency = 12),
    xx8 = ts(c(zo4[1:j, i], zo4[k1, i1], zo4[k2, i2], zo4[k3, i3]), start = startx, end = endx, frequency = 12),
    xx9 = ts(c(zp5[1:j, i], zp5[k1, i1], zp5[k2, i2], zp5[k3, i3]), start = startx, end = endx, frequency = 12),
    xx10 = ts(c(zo5[1:j, i], zo5[k1, i1], zo5[k2, i2], zo5[k3, i3]), start = startx, end = endx, frequency = 12),
    xx11 = ts(c(zp6[1:j, i], zp6[k1, i1], zp6[k2, i2], zp6[k3, i3]), start = startx, end = endx, frequency = 12),
    xx12 = ts(c(zo6[1:j, i], zo6[k1, i1], zo6[k2, i2], zo6[k3, i3]), start = startx, end = endx, frequency = 12),
    zz1 = ts(c(df1[1:j, i], df1[k1, i1], df1[k2, i2], df1[k3, i3]), start = startx, end = endx, frequency = 12),
    zz2 = ts(c(df2[1:j, i], df2[k1, i1], df2[k2, i2], df2[k3, i3]), start = startx, end = endx, frequency = 12),
    zz3 = ts(c(df3[1:j, i], df3[k1, i1], df3[k2, i2], df3[k3, i3]), start = startx, end = endx, frequency = 12),
    zz4 = ts(c(df4[1:j, i], df4[k1, i1], df4[k2, i2], df4[k3, i3]), start = startx, end = endx, frequency = 12),
    zz5 = ts(c(df5[1:j, i], df5[k1, i1], df5[k2, i2], df5[k3, i3]), start = startx, end = endx, frequency = 12)
  )
  insample <- 1:m1
  outsample <- (1:length(fulldata$yy))[-insample]
  
  # Combination Model
  ## MF
  best_nowmdzz <- average_forecast(list(best_md1, best_md2, best_md3, best_md4, best_md5),
                                   data = fulldata, insample = insample, outsample = outsample,
                                   type = "recursive", 
                                   measures = c("MSE"),
                                   fweights = c("EW", "BICW", "MSFE", "DMSFE"),
                                   show_progress = FALSE)
  Recu_mf_ew <- append(Recu_mf_ew, best_nowmdzz[["avgforecast"]][1])
  Recu_mf_bicw <- append(Recu_mf_bicw, best_nowmdzz[["avgforecast"]][2])
  Recu_mf_msfew <- append(Recu_mf_msfew, best_nowmdzz[["avgforecast"]][3])
  Recu_mf_dmsfew <- append(Recu_mf_dmsfew, best_nowmdzz[["avgforecast"]][4])
  ## MF+CB_+
  best_nowmdxzp <- average_forecast(list(best_md1, best_md2, best_md3, best_md4, best_md5, best_md6),
                                    data = fulldata, insample = insample, outsample = outsample,
                                    type = "recursive", 
                                    measures = c("MSE"),
                                    fweights = c("EW", "BICW", "MSFE", "DMSFE"),
                                    show_progress = FALSE)
  Recu_zp_ew <- append(Recu_zp_ew, best_nowmdxzp[["avgforecast"]][1])
  Recu_zp_bicw <- append(Recu_zp_bicw, best_nowmdxzp[["avgforecast"]][2])
  Recu_zp_msfew <- append(Recu_zp_msfew, best_nowmdxzp[["avgforecast"]][3])
  Recu_zp_dmsfew <- append(Recu_zp_dmsfew, best_nowmdxzp[["avgforecast"]][4])  
  ## MF+CB_0
  best_nowmdxzz <- average_forecast(list(best_md1, best_md2, best_md3, best_md4, best_md5, best_md7),
                                    data = fulldata, insample = insample, outsample = outsample,
                                    type = "recursive", 
                                    measures = c("MSE"),
                                    fweights = c("EW", "BICW", "MSFE", "DMSFE"),
                                    show_progress = FALSE)
  Recu_zz_ew <- append(Recu_zz_ew, best_nowmdxzz[["avgforecast"]][1])
  Recu_zz_bicw <- append(Recu_zz_bicw, best_nowmdxzz[["avgforecast"]][2])
  Recu_zz_msfew <- append(Recu_zz_msfew, best_nowmdxzz[["avgforecast"]][3])
  Recu_zz_dmsfew <- append(Recu_zz_dmsfew, best_nowmdxzz[["avgforecast"]][4])        
  ## MF+CB_2
  best_nowmdxz_2 <- average_forecast(list(best_md1, best_md2, best_md3, best_md4,
                                          best_md5, best_md6, best_md7),
                                     data = fulldata, insample = insample, outsample = outsample,
                                     type = "recursive", 
                                     measures = c("MSE"),
                                     fweights = c("EW", "BICW", "MSFE", "DMSFE"),
                                     show_progress = FALSE)
  Recu_cb_ew <- append(Recu_cb_ew, best_nowmdxz_2[["avgforecast"]][1])
  Recu_cb_bicw <- append(Recu_cb_bicw, best_nowmdxz_2[["avgforecast"]][2])
  Recu_cb_msfew <- append(Recu_cb_msfew, best_nowmdxz_2[["avgforecast"]][3])
  Recu_cb_dmsfew <- append(Recu_cb_dmsfew, best_nowmdxz_2[["avgforecast"]][4])  
  ## MF+CB_12
  best_nowmdxz_12 <- average_forecast(list(best_md1, best_md2, best_md3, best_md4,
                                           best_md5, best_md6, best_md7, best_md8, best_md9,
                                           best_md10, best_md11, best_md12, best_md13,
                                           best_md14, best_md15, best_md16, best_md17),
                                      data = fulldata, insample = insample, outsample = outsample,
                                      type = "recursive", 
                                      measures = c("MSE"),
                                      fweights = c("EW", "BICW", "MSFE", "DMSFE"),
                                      show_progress = FALSE)
  Recu_mfcb_ew <- append(Recu_mfcb_ew, best_nowmdxz_12[["avgforecast"]][1])
  Recu_mfcb_bicw <- append(Recu_mfcb_bicw, best_nowmdxz_12[["avgforecast"]][2])
  Recu_mfcb_msfew <- append(Recu_mfcb_msfew, best_nowmdxz_12[["avgforecast"]][3])
  Recu_mfcb_dmsfew <- append(Recu_mfcb_dmsfew, best_nowmdxz_12[["avgforecast"]][4])
}

# Saving predicting results
best_result_rec_for3 <- data.frame(qdate = qdate,
                                   real = gdp,
                                   mf_ew = append(rep(0, leny), Recu_mf_ew),
                                   mf_bicw = append(rep(0, leny), Recu_mf_bicw),
                                   mf_msfew = append(rep(0, leny), Recu_mf_msfew),
                                   mf_dmsfew = append(rep(0, leny), Recu_mf_dmsfew),
                                   zp_ew = append(rep(0, leny), Recu_zp_ew),
                                   zp_bicw = append(rep(0, leny), Recu_zp_bicw),
                                   zp_msfew = append(rep(0, leny), Recu_zp_msfew),
                                   zp_dmsfew = append(rep(0, leny), Recu_zp_dmsfew),
                                   zz_ew = append(rep(0, leny), Recu_zz_ew),
                                   zz_bicw = append(rep(0, leny), Recu_zz_bicw),
                                   zz_msfew = append(rep(0, leny), Recu_zz_msfew),
                                   zz_dmsfew = append(rep(0, leny), Recu_zz_dmsfew),                                 
                                   cb_ew = append(rep(0, leny), Recu_cb_ew),
                                   cb_bicw = append(rep(0, leny), Recu_cb_bicw),
                                   cb_msfew = append(rep(0, leny), Recu_cb_msfew),
                                   cb_dmsfew = append(rep(0, leny), Recu_cb_dmsfew),
                                   mfcb_ew = append(rep(0, leny), Recu_mfcb_ew),
                                   mfcb_bicw = append(rep(0, leny), Recu_mfcb_bicw),
                                   mfcb_msfew = append(rep(0, leny), Recu_mfcb_msfew),
                                   mfcb_dmsfew = append(rep(0, leny), Recu_mfcb_dmsfew))


#-------------------------------------
# Model Combination (h=4)
#-------------------------------------
result_rec_for4 <- na.omit(read_xlsx(paste0(tmpdatpath, "Forecast_GDP_Realtime.xlsx"), sheet = 5, col_names = T))

# Select best individual MIDAS model
best_recu_df1 <- result_rec_for4[which(result_rec_for4$Recu_df1 == min(result_rec_for4$Recu_df1)), ]
best_recu_df2 <- result_rec_for4[which(result_rec_for4$Recu_df2 == min(result_rec_for4$Recu_df2)), ]
best_recu_df3 <- result_rec_for4[which(result_rec_for4$Recu_df3 == min(result_rec_for4$Recu_df3)), ]
best_recu_df4 <- result_rec_for4[which(result_rec_for4$Recu_df4 == min(result_rec_for4$Recu_df4)), ]
best_recu_df5 <- result_rec_for4[which(result_rec_for4$Recu_df5 == min(result_rec_for4$Recu_df5)), ]
best_recu_z_pos_gdp <- result_rec_for4[which(result_rec_for4$Recu_zpos_gdp == min(result_rec_for4$Recu_zpos_gdp)), ][1,]
best_recu_z_zero_gdp <- result_rec_for4[which(result_rec_for4$Recu_zzero_gdp == min(result_rec_for4$Recu_zzero_gdp)), ][1,]
best_recu_z_pos_cpi <- result_rec_for4[which(result_rec_for4$Recu_zpos_cpi == min(result_rec_for4$Recu_zpos_cpi)), ]
best_recu_z_zero_cpi <- result_rec_for4[which(result_rec_for4$Recu_zzero_cpi == min(result_rec_for4$Recu_zzero_cpi)), ]
best_recu_z_pos_iva <- result_rec_for4[which(result_rec_for4$Recu_zpos_iva == min(result_rec_for4$Recu_zpos_iva)), ]
best_recu_z_zero_iva <- result_rec_for4[which(result_rec_for4$Recu_zzero_iva == min(result_rec_for4$Recu_zzero_iva)), ]
best_recu_z_pos_fai <- result_rec_for4[which(result_rec_for4$Recu_zpos_fai == min(result_rec_for4$Recu_zpos_fai)), ]
best_recu_z_zero_fai <- result_rec_for4[which(result_rec_for4$Recu_zzero_fai == min(result_rec_for4$Recu_zzero_fai)), ]
best_recu_z_pos_m2 <- result_rec_for4[which(result_rec_for4$Recu_zpos_m2 == min(result_rec_for4$Recu_zpos_m2)), ]
best_recu_z_zero_m2 <- result_rec_for4[which(result_rec_for4$Recu_zzero_m2 == min(result_rec_for4$Recu_zzero_m2)), ]
best_recu_z_pos_ppi <- result_rec_for4[which(result_rec_for4$Recu_zpos_ppi == min(result_rec_for4$Recu_zpos_ppi)), ]
best_recu_z_zero_ppi <- result_rec_for4[which(result_rec_for4$Recu_zzero_ppi == min(result_rec_for4$Recu_zzero_ppi)), ]

## DF1
best_recu_df1_weight <- best_recu_df1$Weightfunction
best_recu_df1_ylag <- best_recu_df1$Ylag
best_recu_df1_xlag <- best_recu_df1$Xlag
best_recu_df1_lead <- 12
best_recu_df1_tau <- "4"
## DF2
best_recu_df2_weight <- best_recu_df2$Weightfunction
best_recu_df2_ylag <- best_recu_df2$Ylag
best_recu_df2_xlag <- best_recu_df2$Xlag
best_recu_df2_lead <- 12
best_recu_df2_tau <- "4"
## DF3
best_recu_df3_weight <- best_recu_df3$Weightfunction
best_recu_df3_ylag <- best_recu_df3$Ylag
best_recu_df3_xlag <- best_recu_df3$Xlag
best_recu_df3_lead <- 12
best_recu_df3_tau <- "4"
## DF4
best_recu_df4_weight <- best_recu_df4$Weightfunction
best_recu_df4_ylag <- best_recu_df4$Ylag
best_recu_df4_xlag <- best_recu_df4$Xlag
best_recu_df4_lead <- 12
best_recu_df4_tau <- "4"
## DF5
best_recu_df5_weight <- best_recu_df5$Weightfunction
best_recu_df5_ylag <- best_recu_df5$Ylag
best_recu_df5_xlag <- best_recu_df5$Xlag
best_recu_df5_lead <- 12
best_recu_df5_tau <- "4"
## z_pos_gdp
best_recu_zpos_gdp_weight <- best_recu_z_pos_gdp$Weightfunction
best_recu_zpos_gdp_ylag <- best_recu_z_pos_gdp$Ylag
best_recu_zpos_gdp_xlag <- best_recu_z_pos_gdp$Xlag
best_recu_zpos_gdp_lead <- 12
best_recu_zpos_gdp_tau <- "4"
## z_zero_gdp
best_recu_zzero_gdp_weight <- best_recu_z_zero_gdp$Weightfunction
best_recu_zzero_gdp_ylag <- best_recu_z_zero_gdp$Ylag
best_recu_zzero_gdp_xlag <- best_recu_z_zero_gdp$Xlag
best_recu_zzero_gdp_lead <- 12
best_recu_zzero_gdp_tau <- "4"
## z_pos_cpi
best_recu_zpos_cpi_weight <- best_recu_z_pos_cpi$Weightfunction
best_recu_zpos_cpi_ylag <- best_recu_z_pos_cpi$Ylag
best_recu_zpos_cpi_xlag <- best_recu_z_pos_cpi$Xlag
best_recu_zpos_cpi_lead <- 12
best_recu_zpos_cpi_tau <- "4"
## z_zero_cpi
best_recu_zzero_cpi_weight <- best_recu_z_zero_cpi$Weightfunction
best_recu_zzero_cpi_ylag <- best_recu_z_zero_cpi$Ylag
best_recu_zzero_cpi_xlag <- best_recu_z_zero_cpi$Xlag
best_recu_zzero_cpi_lead <- 12
best_recu_zzero_cpi_tau <- "4"
## z_pos_iva
best_recu_zpos_iva_weight <- best_recu_z_pos_iva$Weightfunction
best_recu_zpos_iva_ylag <- best_recu_z_pos_iva$Ylag
best_recu_zpos_iva_xlag <- best_recu_z_pos_iva$Xlag
best_recu_zpos_iva_lead <- 12
best_recu_zpos_iva_tau <- "4"
## z_zero_iva
best_recu_zzero_iva_weight <- best_recu_z_zero_iva$Weightfunction
best_recu_zzero_iva_ylag <- best_recu_z_zero_iva$Ylag
best_recu_zzero_iva_xlag <- best_recu_z_zero_iva$Xlag
best_recu_zzero_iva_lead <- 12
best_recu_zzero_iva_tau <- "4"
## z_pos_fai
best_recu_zpos_fai_weight <- best_recu_z_pos_fai$Weightfunction
best_recu_zpos_fai_ylag <- best_recu_z_pos_fai$Ylag
best_recu_zpos_fai_xlag <- best_recu_z_pos_fai$Xlag
best_recu_zpos_fai_lead <- 12
best_recu_zpos_fai_tau <- "4"
## z_zero_fai
best_recu_zzero_fai_weight <- best_recu_z_zero_fai$Weightfunction
best_recu_zzero_fai_ylag <- best_recu_z_zero_fai$Ylag
best_recu_zzero_fai_xlag <- best_recu_z_zero_fai$Xlag
best_recu_zzero_fai_lead <- 12
best_recu_zzero_fai_tau <- "4"
## z_pos_m2
best_recu_zpos_m2_weight <- best_recu_z_pos_m2$Weightfunction
best_recu_zpos_m2_ylag <- best_recu_z_pos_m2$Ylag
best_recu_zpos_m2_xlag <- best_recu_z_pos_m2$Xlag
best_recu_zpos_m2_lead <- 12
best_recu_zpos_m2_tau <- "4"
## z_zero_m2
best_recu_zzero_m2_weight <- best_recu_z_zero_m2$Weightfunction
best_recu_zzero_m2_ylag <- best_recu_z_zero_m2$Ylag
best_recu_zzero_m2_xlag <- best_recu_z_zero_m2$Xlag
best_recu_zzero_m2_lead <- 12
best_recu_zzero_m2_tau <- "4"
## z_pos_ppi
best_recu_zpos_ppi_weight <- best_recu_z_pos_ppi$Weightfunction
best_recu_zpos_ppi_ylag <- best_recu_z_pos_ppi$Ylag
best_recu_zpos_ppi_xlag <- best_recu_z_pos_ppi$Xlag
best_recu_zpos_ppi_lead <- 12
best_recu_zpos_ppi_tau <- "4"
## z_zero_ppi
best_recu_zzero_ppi_weight <- best_recu_z_zero_ppi$Weightfunction
best_recu_zzero_ppi_ylag <- best_recu_z_zero_ppi$Ylag
best_recu_zzero_ppi_xlag <- best_recu_z_zero_ppi$Xlag
best_recu_zzero_ppi_lead <- 12
best_recu_zzero_ppi_tau <- "4"

# Training and combine best individual models to get combination MIDAS
## DF1
best_recu_df1_ylag <- 3 + best_recu_df1_ylag
best_recu_df1_xlag <- 11 + best_recu_df1_xlag
if (best_recu_df1_weight == "Almon") {
  best_md1 <- midas_r(yy ~ mls(yy, 4:best_recu_df1_ylag, 1) + mls(zz1, best_recu_df1_lead:best_recu_df1_xlag, 3, almonp),
                      start = list(zz1=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df1_weight == "Beta") {
  best_md1 <- midas_r(yy ~ mls(yy, 4:best_recu_df1_ylag, 1) + mls(zz1, best_recu_df1_lead:best_recu_df1_xlag, 3, nbeta),
                      start = list(zz1=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df1_weight == "BetaNN") {
  best_md1 <- midas_r(yy ~ mls(yy, 4:best_recu_df1_ylag, 1) + mls(zz1, best_recu_df1_lead:best_recu_df1_xlag, 3, nbetaMT),
                      start = list(zz1=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md1 <- midas_r(yy ~ mls(yy, 4:best_recu_df1_ylag, 1) + mls(zz1, best_recu_df1_lead:best_recu_df1_xlag, 3, nealmon),
                      start = list(zz1=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## DF2
best_recu_df2_ylag <- 3 + best_recu_df2_ylag
best_recu_df2_xlag <- 11 + best_recu_df2_xlag
if (best_recu_df2_weight == "Almon") {
  best_md2 <- midas_r(yy ~ mls(yy, 4:best_recu_df2_ylag, 1) + mls(zz2, best_recu_df2_lead:best_recu_df2_xlag, 3, almonp),
                      start = list(zz2=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df2_weight == "Beta") {
  best_md2 <- midas_r(yy ~ mls(yy, 4:best_recu_df2_ylag, 1) + mls(zz2, best_recu_df2_lead:best_recu_df2_xlag, 3, nbeta),
                      start = list(zz2=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df2_weight == "BetaNN") {
  best_md2 <- midas_r(yy ~ mls(yy, 4:best_recu_df2_ylag, 1) + mls(zz2, best_recu_df2_lead:best_recu_df2_xlag, 3, nbetaMT),
                      start = list(zz2=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md2 <- midas_r(yy ~ mls(yy, 4:best_recu_df2_ylag, 1) + mls(zz2, best_recu_df2_lead:best_recu_df2_xlag, 3, nealmon),
                      start = list(zz2=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## DF3
best_recu_df3_ylag <- 3 + best_recu_df3_ylag
best_recu_df3_xlag <- 11 + best_recu_df3_xlag
if (best_recu_df3_weight == "Almon") {
  best_md3 <- midas_r(yy ~ mls(yy, 4:best_recu_df3_ylag, 1) + mls(zz3, best_recu_df3_lead:best_recu_df3_xlag, 3, almonp),
                      start = list(zz3=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df3_weight == "Beta") {
  best_md3 <- midas_r(yy ~ mls(yy, 4:best_recu_df3_ylag, 1) + mls(zz3, best_recu_df3_lead:best_recu_df3_xlag, 3, nbeta),
                      start = list(zz3=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df3_weight == "BetaNN") {
  best_md3 <- midas_r(yy ~ mls(yy, 4:best_recu_df3_ylag, 1) + mls(zz3, best_recu_df3_lead:best_recu_df3_xlag, 3, nbetaMT),
                      start = list(zz3=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md3 <- midas_r(yy ~ mls(yy, 4:best_recu_df3_ylag, 1) + mls(zz3, best_recu_df3_lead:best_recu_df3_xlag, 3, nealmon),
                      start = list(zz3=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## DF4
best_recu_df4_ylag <- 3 + best_recu_df4_ylag
best_recu_df4_xlag <- 11 + best_recu_df4_xlag
if (best_recu_df4_weight == "Almon") {
  best_md4 <- midas_r(yy ~ mls(yy, 4:best_recu_df4_ylag, 1) + mls(zz4, best_recu_df4_lead:best_recu_df4_xlag, 3, almonp),
                      start = list(zz4=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df4_weight == "Beta") {
  best_md4 <- midas_r(yy ~ mls(yy, 4:best_recu_df4_ylag, 1) + mls(zz4, best_recu_df4_lead:best_recu_df4_xlag, 3, nbeta),
                      start = list(zz4=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df4_weight == "BetaNN") {
  best_md4 <- midas_r(yy ~ mls(yy, 4:best_recu_df4_ylag, 1) + mls(zz4, best_recu_df4_lead:best_recu_df4_xlag, 3, nbetaMT),
                      start = list(zz4=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md4 <- midas_r(yy ~ mls(yy, 4:best_recu_df4_ylag, 1) + mls(zz4, best_recu_df4_lead:best_recu_df4_xlag, 3, nealmon),
                      start = list(zz4=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## DF5
best_recu_df5_ylag <- 3 + best_recu_df5_ylag
best_recu_df5_xlag <- 11 + best_recu_df5_xlag
if (best_recu_df2_weight == "Almon") {
  best_md5 <- midas_r(yy ~ mls(yy, 4:best_recu_df5_ylag, 1) + mls(zz5, best_recu_df5_lead:best_recu_df5_xlag, 3, almonp),
                      start = list(zz5=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df5_weight == "Beta") {
  best_md5 <- midas_r(yy ~ mls(yy, 4:best_recu_df5_ylag, 1) + mls(zz5, best_recu_df5_lead:best_recu_df5_xlag, 3, nbeta),
                      start = list(zz5=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_df5_weight == "BetaNN") {
  best_md5 <- midas_r(yy ~ mls(yy, 4:best_recu_df5_ylag, 1) + mls(zz5, best_recu_df5_lead:best_recu_df5_xlag, 3, nbetaMT),
                      start = list(zz5=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md5 <- midas_r(yy ~ mls(yy, 4:best_recu_df5_ylag, 1) + mls(zz5, best_recu_df5_lead:best_recu_df5_xlag, 3, nealmon),
                      start = list(zz5=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## z_pos_gdp
best_recu_zpos_gdp_xlag <- 3 + best_recu_zpos_gdp_xlag
best_recu_zpos_gdp_lead <- 11 + best_recu_zpos_gdp_lead
if (best_recu_zpos_gdp_weight == "Almon") {
  best_md6 <- midas_r(yy ~ mls(yy, 4:best_recu_zpos_gdp_ylag, 1) + mls(xx1, best_recu_zpos_gdp_lead:best_recu_zpos_gdp_xlag, 3, almonp),
                      start = list(xx1=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zpos_gdp_weight == "Beta") {
  best_md6 <- midas_r(yy ~ mls(yy, 4:best_recu_zpos_gdp_ylag, 1) + mls(xx1, best_recu_zpos_gdp_lead:best_recu_zpos_gdp_xlag, 3, nbeta),
                      start = list(xx1=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zpos_gdp_weight == "BetaNN") {
  best_md6 <- midas_r(yy ~ mls(yy, 4:best_recu_zpos_gdp_ylag, 1) + mls(xx1, best_recu_zpos_gdp_lead:best_recu_zpos_gdp_xlag, 3, nbetaMT),
                      start = list(xx1=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md6 <- midas_r(yy ~ mls(yy, 4:best_recu_zpos_gdp_ylag, 1) + mls(xx1, best_recu_zpos_gdp_lead:best_recu_zpos_gdp_xlag, 3, nealmon),
                      start = list(xx1=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## z_zero_gdp
best_recu_zzero_gdp_xlag <- 3 + best_recu_zzero_gdp_xlag
best_recu_zzero_gdp_lead <- 11 + best_recu_zzero_gdp_lead
if (best_recu_zzero_gdp_weight == "Almon") {
  best_md7 <- midas_r(yy ~ mls(yy, 4:best_recu_zzero_gdp_ylag, 1) + mls(xx2, best_recu_zzero_gdp_lead:best_recu_zzero_gdp_xlag, 3, almonp),
                      start = list(xx2=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zzero_gdp_weight == "Beta") {
  best_md7 <- midas_r(yy ~ mls(yy, 4:best_recu_zzero_gdp_ylag, 1) + mls(xx2, best_recu_zzero_gdp_lead:best_recu_zzero_gdp_xlag, 3, nbeta),
                      start = list(xx2=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zzero_gdp_weight == "BetaNN") {
  best_md7 <- midas_r(yy ~ mls(yy, 4:best_recu_zzero_gdp_ylag, 1) + mls(xx2, best_recu_zzero_gdp_lead:best_recu_zzero_gdp_xlag, 3, nbetaMT),
                      start = list(xx2=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md7 <- midas_r(yy ~ mls(yy, 4:best_recu_zzero_gdp_ylag, 1) + mls(xx2, best_recu_zzero_gdp_lead:best_recu_zzero_gdp_xlag, 3, nealmon),
                      start = list(xx2=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## z_pos_cpi
best_recu_zpos_cpi_ylag <- 3 + best_recu_zpos_cpi_ylag
best_recu_zpos_cpi_xlag <- 11 + best_recu_zpos_cpi_xlag
if (best_recu_zpos_cpi_weight == "Almon") {
  best_md8 <- midas_r(yy ~ mls(yy, 4:best_recu_zpos_cpi_ylag, 1) + mls(xx3, best_recu_zpos_cpi_lead:best_recu_zpos_cpi_xlag, 3, almonp),
                      start = list(xx3=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zpos_cpi_weight == "Beta") {
  best_md8 <- midas_r(yy ~ mls(yy, 4:best_recu_zpos_cpi_ylag, 1) + mls(xx3, best_recu_zpos_cpi_lead:best_recu_zpos_cpi_xlag, 3, nbeta),
                      start = list(xx3=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zpos_cpi_weight == "BetaNN") {
  best_md8 <- midas_r(yy ~ mls(yy, 4:best_recu_zpos_cpi_ylag, 1) + mls(xx3, best_recu_zpos_cpi_lead:best_recu_zpos_cpi_xlag, 3, nbetaMT),
                      start = list(xx3=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md8 <- midas_r(yy ~ mls(yy, 4:best_recu_zpos_cpi_ylag, 1) + mls(xx3, best_recu_zpos_cpi_lead:best_recu_zpos_cpi_xlag, 3, nealmon),
                      start = list(xx3=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## z_zero_cpi
best_recu_zzero_cpi_ylag <- 3 + best_recu_zzero_cpi_ylag
best_recu_zzero_cpi_xlag <- 11 + best_recu_zzero_cpi_xlag
if (best_recu_zzero_cpi_weight == "Almon") {
  best_md9 <- midas_r(yy ~ mls(yy, 4:best_recu_zzero_cpi_ylag, 1) + mls(xx4, best_recu_zzero_cpi_lead:best_recu_zzero_cpi_xlag, 3, almonp),
                      start = list(xx4=c(1, 0, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zzero_cpi_weight == "Beta") {
  best_md9 <- midas_r(yy ~ mls(yy, 4:best_recu_zzero_cpi_ylag, 1) + mls(xx4, best_recu_zzero_cpi_lead:best_recu_zzero_cpi_xlag, 3, nbeta),
                      start = list(xx4=c(1, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else if (best_recu_zzero_cpi_weight == "BetaNN") {
  best_md9 <- midas_r(yy ~ mls(yy, 4:best_recu_zzero_cpi_ylag, 1) + mls(xx4, best_recu_zzero_cpi_lead:best_recu_zzero_cpi_xlag, 3, nbetaMT),
                      start = list(xx4=c(1, 0, 0, 0)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
} else {
  best_md9 <- midas_r(yy ~ mls(yy, 4:best_recu_zzero_cpi_ylag, 1) + mls(xx4, best_recu_zzero_cpi_lead:best_recu_zzero_cpi_xlag, 3, nealmon),
                      start = list(xx4=c(1, 1, -0.5)),
                      Ofunction = "optim", 
                      method = "Nelder-Mead")
}
## z_pos_iva
best_recu_zpos_iva_ylag <- 3 + best_recu_zpos_iva_ylag
best_recu_zpos_iva_xlag <- 11 + best_recu_zpos_iva_xlag
if (best_recu_zpos_iva_weight == "Almon") {
  best_md10 <- midas_r(yy ~ mls(yy, 4:best_recu_zpos_iva_ylag, 1) + mls(xx5, best_recu_zpos_iva_lead:best_recu_zpos_iva_xlag, 3, almonp),
                       start = list(xx5=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_iva_weight == "Beta") {
  best_md10 <- midas_r(yy ~ mls(yy, 4:best_recu_zpos_iva_ylag, 1) + mls(xx5, best_recu_zpos_iva_lead:best_recu_zpos_iva_xlag, 3, nbeta),
                       start = list(xx5=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_iva_weight == "BetaNN") {
  best_md10 <- midas_r(yy ~ mls(yy, 4:best_recu_zpos_iva_ylag, 1) + mls(xx5, best_recu_zpos_iva_lead:best_recu_zpos_iva_xlag, 3, nbetaMT),
                       start = list(xx5=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md10 <- midas_r(yy ~ mls(yy, 4:best_recu_zpos_iva_ylag, 1) + mls(xx5, best_recu_zpos_iva_lead:best_recu_zpos_iva_xlag, 3, nealmon),
                       start = list(xx5=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}
### z_zero_iva
best_recu_zzero_iva_ylag <- 3 + best_recu_zzero_iva_ylag
best_recu_zzero_iva_xlag <- 11 + best_recu_zzero_iva_xlag
if (best_recu_zzero_iva_weight == "Almon") {
  best_md11 <- midas_r(yy ~ mls(yy, 4:best_recu_zzero_iva_ylag, 1) + mls(xx6, best_recu_zzero_iva_lead:best_recu_zzero_iva_xlag, 3, almonp),
                       start = list(xx6=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_iva_weight == "Beta") {
  best_md11 <- midas_r(yy ~ mls(yy, 4:best_recu_zzero_iva_ylag, 1) + mls(xx6, best_recu_zzero_iva_lead:best_recu_zzero_iva_xlag, 3, nbeta),
                       start = list(xx6=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_iva_weight == "BetaNN") {
  best_md11 <- midas_r(yy ~ mls(yy, 4:best_recu_zzero_iva_ylag, 1) + mls(xx6, best_recu_zzero_iva_lead:best_recu_zzero_iva_xlag, 3, nbetaMT),
                       start = list(xx6=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md11 <- midas_r(yy ~ mls(yy, 4:best_recu_zzero_iva_ylag, 1) + mls(xx6, best_recu_zzero_iva_lead:best_recu_zzero_iva_xlag, 3, nealmon),
                       start = list(xx6=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}
## z_pos_fai
best_recu_zpos_fai_ylag <- 3 + best_recu_zpos_fai_ylag
best_recu_zpos_fai_xlag <- 11 + best_recu_zpos_fai_xlag
if (best_recu_zpos_fai_weight == "Almon") {
  best_md12 <- midas_r(yy ~ mls(yy, 4:best_recu_zpos_fai_ylag, 1) + mls(xx7, best_recu_zpos_fai_lead:best_recu_zpos_fai_xlag, 3, almonp),
                       start = list(xx7=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_fai_weight == "Beta") {
  best_md12 <- midas_r(yy ~ mls(yy, 4:best_recu_zpos_fai_ylag, 1) + mls(xx7, best_recu_zpos_fai_lead:best_recu_zpos_fai_xlag, 3, nbeta),
                       start = list(xx7=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_fai_weight == "BetaNN") {
  best_md12 <- midas_r(yy ~ mls(yy, 4:best_recu_zpos_fai_ylag, 1) + mls(xx7, best_recu_zpos_fai_lead:best_recu_zpos_fai_xlag, 3, nbetaMT),
                       start = list(xx7=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md12 <- midas_r(yy ~ mls(yy, 4:best_recu_zpos_fai_ylag, 1) + mls(xx7, best_recu_zpos_fai_lead:best_recu_zpos_fai_xlag, 3, nealmon),
                       start = list(xx7=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}
### z_zero_fai
best_recu_zzero_fai_ylag <- 3 + best_recu_zzero_fai_ylag
best_recu_zzero_fai_xlag <- 11 + best_recu_zzero_fai_xlag
if (best_recu_zzero_fai_weight == "Almon") {
  best_md13 <- midas_r(yy ~ mls(yy, 4:best_recu_zzero_fai_ylag, 1) + mls(xx8, best_recu_zzero_fai_lead:best_recu_zzero_fai_xlag, 3, almonp),
                       start = list(xx8=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_fai_weight == "Beta") {
  best_md13 <- midas_r(yy ~ mls(yy, 4:best_recu_zzero_fai_ylag, 1) + mls(xx8, best_recu_zzero_fai_lead:best_recu_zzero_fai_xlag, 3, nbeta),
                       start = list(xx8=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_fai_weight == "BetaNN") {
  best_md13 <- midas_r(yy ~ mls(yy, 4:best_recu_zzero_fai_ylag, 1) + mls(xx8, best_recu_zzero_fai_lead:best_recu_zzero_fai_xlag, 3, nbetaMT),
                       start = list(xx8=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md13 <- midas_r(yy ~ mls(yy, 4:best_recu_zzero_fai_ylag, 1) + mls(xx8, best_recu_zzero_fai_lead:best_recu_zzero_fai_xlag, 3, nealmon),
                       start = list(xx8=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}
## z_pos_m2
best_recu_zpos_m2_ylag <- 3 + best_recu_zpos_m2_ylag
best_recu_zpos_m2_xlag <- 11 + best_recu_zpos_m2_xlag
if (best_recu_zpos_m2_weight == "Almon") {
  best_md14 <- midas_r(yy ~ mls(yy, 4:best_recu_zpos_m2_ylag, 1) + mls(xx9, best_recu_zpos_m2_lead:best_recu_zpos_m2_xlag, 3, almonp),
                       start = list(xx9=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_m2_weight == "Beta") {
  best_md14 <- midas_r(yy ~ mls(yy, 4:best_recu_zpos_m2_ylag, 1) + mls(xx9, best_recu_zpos_m2_lead:best_recu_zpos_m2_xlag, 3, nbeta),
                       start = list(xx9=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_m2_weight == "BetaNN") {
  best_md14 <- midas_r(yy ~ mls(yy, 4:best_recu_zpos_m2_ylag, 1) + mls(xx9, best_recu_zpos_m2_lead:best_recu_zpos_m2_xlag, 3, nbetaMT),
                       start = list(xx9=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md14 <- midas_r(yy ~ mls(yy, 4:best_recu_zpos_m2_ylag, 1) + mls(xx9, best_recu_zpos_m2_lead:best_recu_zpos_m2_xlag, 3, nealmon),
                       start = list(xx9=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}
### z_zero_m2
best_recu_zzero_m2_ylag <- 3 + best_recu_zzero_m2_ylag
best_recu_zzero_m2_xlag <- 11 + best_recu_zzero_m2_xlag
if (best_recu_zzero_m2_weight == "Almon") {
  best_md15 <- midas_r(yy ~ mls(yy, 4:best_recu_zzero_m2_ylag, 1) + mls(xx10, best_recu_zzero_m2_lead:best_recu_zzero_m2_xlag, 3, almonp),
                       start = list(xx10=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_m2_weight == "Beta") {
  best_md15 <- midas_r(yy ~ mls(yy, 4:best_recu_zzero_m2_ylag, 1) + mls(xx10, best_recu_zzero_m2_lead:best_recu_zzero_m2_xlag, 3, nbeta),
                       start = list(xx10=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_m2_weight == "BetaNN") {
  best_md15 <- midas_r(yy ~ mls(yy, 4:best_recu_zzero_m2_ylag, 1) + mls(xx10, best_recu_zzero_m2_lead:best_recu_zzero_m2_xlag, 3, nbetaMT),
                       start = list(xx10=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md15 <- midas_r(yy ~ mls(yy, 4:best_recu_zzero_m2_ylag, 1) + mls(xx10, best_recu_zzero_m2_lead:best_recu_zzero_m2_xlag, 3, nealmon),
                       start = list(xx10=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}
## z_pos_ppi
best_recu_zpos_ppi_ylag <- 3 + best_recu_zpos_ppi_ylag
best_recu_zpos_ppi_xlag <- 11 + best_recu_zpos_ppi_xlag
if (best_recu_zpos_ppi_weight == "Almon") {
  best_md16 <- midas_r(yy ~ mls(yy, 4:best_recu_zpos_ppi_ylag, 1) + mls(xx11, best_recu_zpos_ppi_lead:best_recu_zpos_ppi_xlag, 3, almonp),
                       start = list(xx11=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_ppi_weight == "Beta") {
  best_md16 <- midas_r(yy ~ mls(yy, 4:best_recu_zpos_ppi_ylag, 1) + mls(xx11, best_recu_zpos_ppi_lead:best_recu_zpos_ppi_xlag, 3, nbeta),
                       start = list(xx11=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zpos_ppi_weight == "BetaNN") {
  best_md16 <- midas_r(yy ~ mls(yy, 4:best_recu_zpos_ppi_ylag, 1) + mls(xx11, best_recu_zpos_ppi_lead:best_recu_zpos_ppi_xlag, 3, nbetaMT),
                       start = list(xx11=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md16 <- midas_r(yy ~ mls(yy, 4:best_recu_zpos_ppi_ylag, 1) + mls(xx11, best_recu_zpos_ppi_lead:best_recu_zpos_ppi_xlag, 3, nealmon),
                       start = list(xx11=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}
### z_zero_ppi
best_recu_zzero_ppi_ylag <- 3 + best_recu_zzero_ppi_ylag
best_recu_zzero_ppi_xlag <- 11 + best_recu_zzero_ppi_xlag
if (best_recu_zzero_ppi_weight == "Almon") {
  best_md17 <- midas_r(yy ~ mls(yy, 4:best_recu_zzero_ppi_ylag, 1) + mls(xx12, best_recu_zzero_ppi_lead:best_recu_zzero_ppi_xlag, 3, almonp),
                       start = list(xx12=c(1, 0, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_ppi_weight == "Beta") {
  best_md17 <- midas_r(yy ~ mls(yy, 4:best_recu_zzero_ppi_ylag, 1) + mls(xx12, best_recu_zzero_ppi_lead:best_recu_zzero_ppi_xlag, 3, nbeta),
                       start = list(xx12=c(1, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else if (best_recu_zzero_ppi_weight == "BetaNN") {
  best_md17 <- midas_r(yy ~ mls(yy, 4:best_recu_zzero_ppi_ylag, 1) + mls(xx12, best_recu_zzero_ppi_lead:best_recu_zzero_ppi_xlag, 3, nbetaMT),
                       start = list(xx12=c(1, 0, 0, 0)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
} else {
  best_md17 <- midas_r(yy ~ mls(yy, 4:best_recu_zzero_ppi_ylag, 1) + mls(xx12, best_recu_zzero_ppi_lead:best_recu_zzero_ppi_xlag, 3, nealmon),
                       start = list(xx12=c(1, 1, -0.5)),
                       Ofunction = "optim", 
                       method = "Nelder-Mead")
}

# Estimating & Forecasting
Recu_mf_ew = c()
Recu_mf_bicw = c()
Recu_mf_msfew = c()
Recu_mf_dmsfew = c()
Recu_zp_ew = c()
Recu_zp_bicw = c()
Recu_zp_msfew = c()
Recu_zp_dmsfew = c()
Recu_zz_ew = c()
Recu_zz_bicw = c()
Recu_zz_msfew = c()
Recu_zz_dmsfew = c()
Recu_cb_ew = c()
Recu_cb_bicw = c()
Recu_cb_msfew = c()
Recu_cb_dmsfew = c()
Recu_mfcb_ew = c()
Recu_mfcb_bicw = c()
Recu_mfcb_msfew = c()
Recu_mfcb_dmsfew = c()


for (i in seq(1, length(zp2)-3, 3)) {
  # Sampling
  p = floor((i-1)/3)
  l = ceiling(i/3)
  l1 = l + 1
  m1 = l + leny - 1
  m2 = m1 + 1
  j = i + lenx - 1
  k1 = j + 1
  k2 = j + 2
  k3 = j + 3
  i1 = i + 1
  i2 = i + 2
  i3 = i + 3
  
  # Start & End
  starty = c(2003,1)
  endy = starty + c(len, 0) + c(0,p)
  startx = c(2003,1)
  endx = startx + c(len, 1+i)
  fulldata <- list(
    yy = ts(na.omit(c(gdp[1:m1], gdp[m2])), start = starty, end = endy, frequency = 4), 
    xx1 = ts(c(zp1[1:j, i], zp1[k1, i1], zp1[k2, i2], zp1[k3, i3]), start = startx, end = endx, frequency = 12),
    xx2 = ts(c(zo1[1:j, i], zo1[k1, i1], zo1[k2, i2], zo1[k3, i3]), start = startx, end = endx, frequency = 12),
    xx3 = ts(c(zp2[1:j, i], zp2[k1, i1], zp2[k2, i2], zp2[k3, i3]), start = startx, end = endx, frequency = 12),
    xx4 = ts(c(zo2[1:j, i], zo2[k1, i1], zo2[k2, i2], zo2[k3, i3]), start = startx, end = endx, frequency = 12),
    xx5 = ts(c(zp3[1:j, i], zp3[k1, i1], zp3[k2, i2], zp3[k3, i3]), start = startx, end = endx, frequency = 12),
    xx6 = ts(c(zo3[1:j, i], zo3[k1, i1], zo3[k2, i2], zo3[k3, i3]), start = startx, end = endx, frequency = 12),
    xx7 = ts(c(zp4[1:j, i], zp4[k1, i1], zp4[k2, i2], zp4[k3, i3]), start = startx, end = endx, frequency = 12),
    xx8 = ts(c(zo4[1:j, i], zo4[k1, i1], zo4[k2, i2], zo4[k3, i3]), start = startx, end = endx, frequency = 12),
    xx9 = ts(c(zp5[1:j, i], zp5[k1, i1], zp5[k2, i2], zp5[k3, i3]), start = startx, end = endx, frequency = 12),
    xx10 = ts(c(zo5[1:j, i], zo5[k1, i1], zo5[k2, i2], zo5[k3, i3]), start = startx, end = endx, frequency = 12),
    xx11 = ts(c(zp6[1:j, i], zp6[k1, i1], zp6[k2, i2], zp6[k3, i3]), start = startx, end = endx, frequency = 12),
    xx12 = ts(c(zo6[1:j, i], zo6[k1, i1], zo6[k2, i2], zo6[k3, i3]), start = startx, end = endx, frequency = 12),
    zz1 = ts(c(df1[1:j, i], df1[k1, i1], df1[k2, i2], df1[k3, i3]), start = startx, end = endx, frequency = 12),
    zz2 = ts(c(df2[1:j, i], df2[k1, i1], df2[k2, i2], df2[k3, i3]), start = startx, end = endx, frequency = 12),
    zz3 = ts(c(df3[1:j, i], df3[k1, i1], df3[k2, i2], df3[k3, i3]), start = startx, end = endx, frequency = 12),
    zz4 = ts(c(df4[1:j, i], df4[k1, i1], df4[k2, i2], df4[k3, i3]), start = startx, end = endx, frequency = 12),
    zz5 = ts(c(df5[1:j, i], df5[k1, i1], df5[k2, i2], df5[k3, i3]), start = startx, end = endx, frequency = 12)
  )
  insample <- 1:m1
  outsample <- (1:length(fulldata$yy))[-insample]
  
  # Combination Model
  ## MF
  best_nowmdzz <- average_forecast(list(best_md1, best_md2, best_md3, best_md4, best_md5),
                                   data = fulldata, insample = insample, outsample = outsample,
                                   type = "recursive", 
                                   measures = c("MSE"),
                                   fweights = c("EW", "BICW", "MSFE", "DMSFE"),
                                   show_progress = FALSE)
  Recu_mf_ew <- append(Recu_mf_ew, best_nowmdzz[["avgforecast"]][1])
  Recu_mf_bicw <- append(Recu_mf_bicw, best_nowmdzz[["avgforecast"]][2])
  Recu_mf_msfew <- append(Recu_mf_msfew, best_nowmdzz[["avgforecast"]][3])
  Recu_mf_dmsfew <- append(Recu_mf_dmsfew, best_nowmdzz[["avgforecast"]][4])
  ## MF+CB_+
  best_nowmdxzp <- average_forecast(list(best_md1, best_md2, best_md3, best_md4, best_md5, best_md6),
                                    data = fulldata, insample = insample, outsample = outsample,
                                    type = "recursive", 
                                    measures = c("MSE"),
                                    fweights = c("EW", "BICW", "MSFE", "DMSFE"),
                                    show_progress = FALSE)
  Recu_zp_ew <- append(Recu_zp_ew, best_nowmdxzp[["avgforecast"]][1])
  Recu_zp_bicw <- append(Recu_zp_bicw, best_nowmdxzp[["avgforecast"]][2])
  Recu_zp_msfew <- append(Recu_zp_msfew, best_nowmdxzp[["avgforecast"]][3])
  Recu_zp_dmsfew <- append(Recu_zp_dmsfew, best_nowmdxzp[["avgforecast"]][4])  
  ## MF+CB_0
  best_nowmdxzz <- average_forecast(list(best_md1, best_md2, best_md3, best_md4, best_md5, best_md7),
                                    data = fulldata, insample = insample, outsample = outsample,
                                    type = "recursive", 
                                    measures = c("MSE"),
                                    fweights = c("EW", "BICW", "MSFE", "DMSFE"),
                                    show_progress = FALSE)
  Recu_zz_ew <- append(Recu_zz_ew, best_nowmdxzz[["avgforecast"]][1])
  Recu_zz_bicw <- append(Recu_zz_bicw, best_nowmdxzz[["avgforecast"]][2])
  Recu_zz_msfew <- append(Recu_zz_msfew, best_nowmdxzz[["avgforecast"]][3])
  Recu_zz_dmsfew <- append(Recu_zz_dmsfew, best_nowmdxzz[["avgforecast"]][4])        
  ## MF+CB_2
  best_nowmdxz_2 <- average_forecast(list(best_md1, best_md2, best_md3, best_md4,
                                          best_md5, best_md6, best_md7),
                                     data = fulldata, insample = insample, outsample = outsample,
                                     type = "recursive", 
                                     measures = c("MSE"),
                                     fweights = c("EW", "BICW", "MSFE", "DMSFE"),
                                     show_progress = FALSE)
  Recu_cb_ew <- append(Recu_cb_ew, best_nowmdxz_2[["avgforecast"]][1])
  Recu_cb_bicw <- append(Recu_cb_bicw, best_nowmdxz_2[["avgforecast"]][2])
  Recu_cb_msfew <- append(Recu_cb_msfew, best_nowmdxz_2[["avgforecast"]][3])
  Recu_cb_dmsfew <- append(Recu_cb_dmsfew, best_nowmdxz_2[["avgforecast"]][4])  
  ## MF+CB_12
  best_nowmdxz_12 <- average_forecast(list(best_md1, best_md2, best_md3, best_md4,
                                           best_md5, best_md6, best_md7, best_md8, best_md9,
                                           best_md10, best_md11, best_md12, best_md13,
                                           best_md14, best_md15, best_md16, best_md17),
                                      data = fulldata, insample = insample, outsample = outsample,
                                      type = "recursive", 
                                      measures = c("MSE"),
                                      fweights = c("EW", "BICW", "MSFE", "DMSFE"),
                                      show_progress = FALSE)
  Recu_mfcb_ew <- append(Recu_mfcb_ew, best_nowmdxz_12[["avgforecast"]][1])
  Recu_mfcb_bicw <- append(Recu_mfcb_bicw, best_nowmdxz_12[["avgforecast"]][2])
  Recu_mfcb_msfew <- append(Recu_mfcb_msfew, best_nowmdxz_12[["avgforecast"]][3])
  Recu_mfcb_dmsfew <- append(Recu_mfcb_dmsfew, best_nowmdxz_12[["avgforecast"]][4])
}

# Saving predicting results
best_result_rec_for4 <- data.frame(qdate = qdate,
                                   real = gdp,
                                   mf_ew = append(rep(0, leny), Recu_mf_ew),
                                   mf_bicw = append(rep(0, leny), Recu_mf_bicw),
                                   mf_msfew = append(rep(0, leny), Recu_mf_msfew),
                                   mf_dmsfew = append(rep(0, leny), Recu_mf_dmsfew),
                                   zp_ew = append(rep(0, leny), Recu_zp_ew),
                                   zp_bicw = append(rep(0, leny), Recu_zp_bicw),
                                   zp_msfew = append(rep(0, leny), Recu_zp_msfew),
                                   zp_dmsfew = append(rep(0, leny), Recu_zp_dmsfew),
                                   zz_ew = append(rep(0, leny), Recu_zz_ew),
                                   zz_bicw = append(rep(0, leny), Recu_zz_bicw),
                                   zz_msfew = append(rep(0, leny), Recu_zz_msfew),
                                   zz_dmsfew = append(rep(0, leny), Recu_zz_dmsfew),                                  
                                   cb_ew = append(rep(0, leny), Recu_cb_ew),
                                   cb_bicw = append(rep(0, leny), Recu_cb_bicw),
                                   cb_msfew = append(rep(0, leny), Recu_cb_msfew),
                                   cb_dmsfew = append(rep(0, leny), Recu_cb_dmsfew),
                                   mfcb_ew = append(rep(0, leny), Recu_mfcb_ew),
                                   mfcb_bicw = append(rep(0, leny), Recu_mfcb_bicw),
                                   mfcb_msfew = append(rep(0, leny), Recu_mfcb_msfew),
                                   mfcb_dmsfew = append(rep(0, leny), Recu_mfcb_dmsfew))


#-------------------------------------
# Save Results
#-------------------------------------
sheets <- list("sheet1" = best_result_rec_now,
               "sheet2" = best_result_rec_for1,
               "sheet3" = best_result_rec_for2,
               "sheet4" = best_result_rec_for3,
               "sheet5" = best_result_rec_for4)

write.xlsx(sheets, paste0(path, "/Data/Forecast_GDP_Best_Realtime.xlsx"), 
           sheetName=c("rec_now", "rec_for1", "rec_for2", "rec_for3", "rec_for4"))