% RESTRICT.M
% Lutz Kilian
% University of Michigan
% January 1997
%
% This program calculates the EGLS estimate for the Mark model restricted under
% the null of no predictability and cointegration.  Reference: Lutekepohl (1991),
% secton 5.2

function [U,A0,H,A1,A2,A3,A4,SIGMA]=restrict(y)

k=sicegls(y);
[t,q]=size(y);
z=y(:,1)-y(:,2);

% Drop one observation to account for first-differencing
t=t-1;

% Define lagged z and first-differenced y
z_1  =z(1:t,:);
ydiff=y(2:t+1,:)-y(1:t,:);

% Define the Y and X for EGLS

	% Define matrix of regressors as constant, time trend, lagged y, 
	% plus k lagged differences of the endogenous variable

	if k==0
		X=[ones(t-k,1) z_1(k+1:t,:)]';

	elseif k==1
		X=[ones(t-k,1) z_1(k+1:t,:) ydiff(k:t-1,:)]';

	elseif k==2
		X=[ones(t-k,1) z_1(k+1:t,:) ydiff(k:t-1,:) ydiff(k-1:t-2,:)]';

	elseif k==3
		X=[ones(t-k,1) z_1(k+1:t,:) ydiff(k:t-1,:)   ydiff(k-1:t-2,:) ydiff(k-2:t-3,:)]';

	elseif k==4
		X=[ones(t-k,1) z_1(k+1:t,:) ydiff(k:t-1,:)   ydiff(k-1:t-2,:) ydiff(k-2:t-3,:) ydiff(k-3:t-4,:)]';
	else
		disp('Error: k must not exceed 4.')
		break
	end;

	% Make endogenous variable conformable with X matrix
	Y = ydiff(k+1:t,:)';       

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Luetkepohl (1991), pp. 168:  Let beta=R*gamma+r such that only the intercept is 
% unrestricted in equation 1 and all coefficients are unrestricted in equation 2

[v,t]=size(X);
R=[eye(2) zeros(2,v-1)];
for i=0:q*k
	R=[R; [zeros(1,v+1); [zeros(1,2+i) 1 zeros(1,v-2-i)]]];
end;

r=zeros(2*v,1);
y=vec(Y);
x=y-kron(X',eye(q,q))*r;

% Now estimate the unrestricted LS model to obtain a consistent estimate of the error 
% covariance matrix
% Bhat=Y*X'*inv(X*X')        
% SIGMAr=(Y-Bhat*X)*(Y-Bhat*X)'/(t-v);
% OR: Estimate SIGMA by restriced LS as recommended on pp. 171/172 in Luetkepohl (1991)
gammar=inv(R'*kron(X*X',eye(q))*R)*R'*kron(X,eye(q))*x;
betar=R*gammar+r;
Br=reshape(betar,q,v);
SIGMAr=(Y-Br*X)*(Y-Br*X)'/t;

% Estimate restricted parameter vector
gamma=inv(R'*kron(X*X',inv(SIGMAr))*R)*R'*kron(X,inv(SIGMAr))*x;
np=length(gamma);

% Estimate full parameter vector
beta=R*gamma+r;
B=reshape(beta,q,v);
U=Y-B*X;

% Recalculate innovation variance to match with residuals U. (We do not iterate on the 
% EGLS estimator, however, as all stages are consistent.) 
SIGMA=(Y-B*X)*(Y-B*X)'/t;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if B(2,2)<=0 
R=[[eye(2) zeros(2,v-2)]; zeros(2,v)];
if k>0
	for i=1:q*k
		R=[R; [zeros(1,v); [zeros(1,1+i) 1 zeros(1,v-2-i)]]];
	end;
end;

r=zeros(2*v,1);
r(4,1)=0.000000001;

y=vec(Y);
x=y-kron(X',eye(q,q))*r;

% Now estimate the unrestricted LS model to obtain a consistent estimate of the error 
% covariance matrix
% Bhat=Y*X'*inv(X*X')        
% SIGMA=(Y-Bhat*X)*(Y-Bhat*X)'/(t-v);

% Estimate SIGMA by restriced LS as recommended on pp. 171/172 in Luetkepohl (1991)
gammar=inv(R'*kron(X*X',eye(q))*R)*R'*kron(X,eye(q))*x;
betar=R*gammar+r;
Br=reshape(betar,q,v);
SIGMAr=(Y-Br*X)*(Y-Br*X)'/t;

% Estimate restricted parameter vector
gamma=inv(R'*kron(X*X',inv(SIGMAr))*R)*R'*kron(X,inv(SIGMAr))*x;

% Estimate full parameter vector
beta=R*gamma+r;
B=reshape(beta,q,v);
U=Y-B*X;
SIGMA=(Y-B*X)*(Y-B*X)'/t;


end;

A0=B(:,1); H=B(:,2); 

if k==0
	A1=zeros(2,2); 	A2=zeros(2,2); 	A3=zeros(2,2); 	A4=zeros(2,2);
elseif k==1
	A1=B(:,3:4); 	A2=zeros(2,2); 	A3=zeros(2,2); 	A4=zeros(2,2);
elseif k==2
	A1=B(:,3:4); 	A2=B(:,5:6); 	A3=zeros(2,2); 	A4=zeros(2,2);
elseif k==3
	A1=B(:,3:4); 	A2=B(:,5:6); 	A3=B(:,7:8); 	A4=zeros(2,2);
elseif k==4
	A1=B(:,3:4); 	A2=B(:,5:6); 	A3=B(:,7:8); 	A4=B(:,9:10);
else
	disp('k too large')
	break
end;



