function varma_obj=estimate_vecm(data,foptions);
% Estimation of VECM
%
%      DY = PI Y_t-1 + GAMMA X + U 
%
%
%
% INPUT: 
%	y   ( K x T )       series
%   r   ( 1 x 1 )       cointegrating rank
%   p   ( 1 x 1 )       lag length in levels
%   mu  ( 1 x 1 )       intercept dummy:
%                          0 - no intercept 1 - unrestr. intercept
%                          2 - intercept restr. to coint relations
%
% Outputs: 
%	varma_obj
%        
%		.beta
%		.alfa
%       .mu
%       .gamma
%       .res
%       .eig
%       .cov
%

% ESTIMATION CHECKED!! 

[T,K]=size(data); 
if K>=T; error('data has to be of dimension (T x K)'); end; 

y=data'; 


r=foptions.CointOrder;
p=foptions.LagOrder; 
mu=foptions.InterceptDummy; 
minp=foptions.minp;
maxp=foptions.maxp; 

if (r>K)
	error('cointegration rank must be less than number of series');
elseif (r<0)
	error('cointegration rank must be a positive number');
end

if isempty(maxp)
	maxp=floor(sqrt(T/log(T)));
end

if (mu>0)
    D=1; 
else
    D=0;
end



if ischar(p)

	[aic,aic_p,hq,hq_p,sc,sc_p]=order_var(y,D,minp,maxp);
	switch p
		case{'aic'}
			p=aic_p;
		case{'hq'}
			p=hq_p;
		case{'sc'}
			p=sc_p; 
		otherwise
			error('invalid information criteria');
	end
end

k=p-1; 



%----------------------------------------------------------
% CREATE REGRESSOR MATRICES: 

dy=y(:,2:T)-y(:,1:(T-1));

Ytm1=y(:,(k+1):(T-1));
DY=dy(:,(k+1):(T-1));



if (k ~= 0)

    X=zeros(K*k,T-1-k);

    for t=1:(T-1-k);
        xx=dy(:,t:(t+k-1));
        xx=(flipud(xx'))';
        xx=vec(xx);
        X(:,t)=xx;
    end;

else; 
	X=[];  
end;

if (mu==1)
    X=[ones(1,T-1-k);
		X];    
elseif (mu==2)
    Ytm1=[Ytm1;
		ones(1,T-1-k)];
end



%----------------------------------------------------------
% REDUCED RANK REGRESSION: 
if (k==0) && ( (mu==0)||(mu==2))
    MM=eye(T-1-k);
else
    MM=eye(T-1-k)-(X')*inv(X*X')*X;
end


T_eff=size(DY,2); 
S00=DY*MM*(DY')/T_eff;
S01=DY*MM*(Ytm1')/T_eff;
S11=Ytm1*MM*(Ytm1')/T_eff;
S10=S01';

N=S11;
M=S10*inv(S00)*S01;

[g_eigen_vektors,g_eigen_values]=eig(M,N);  

ve=diag(g_eigen_values);
va=g_eigen_vektors';

[tmp,sort_idx]=sort(ve);
va=va(sort_idx,:)';




% Recover parameters: 
if (r ~= 0)
    beta_hat=va(:,(end-r+1):end);
    alfa_hat=S01*beta_hat*inv((beta_hat')*S11*beta_hat);
    Pi_hat=alfa_hat*(beta_hat'); 
else 
    beta_hat=va;
    alfa_hat=S01*beta_hat*inv((beta_hat')*S11*beta_hat);
    Pi_hat=alfa_hat*(beta_hat'); 
    if mu==2;
        Pi_hat=zeros(K,K+1); 
    else;
        Pi_hat=zeros(K,K);
    end
end


if (k ~= 0)
    gamma_hat=(DY-Pi_hat*Ytm1)*(X')*inv(X*(X'));
    u_hat=DY-Pi_hat*Ytm1-gamma_hat*X; 
else
    gamma_hat=[];
    u_hat=DY-Pi_hat*Ytm1; 
end


CovM_hat=u_hat*(u_hat')/size(u_hat,2); 
u_hat=[nan(K,k+1) u_hat];



if mu==0; 

        mu_hat=zeros(K,1); 

elseif mu==1 && k>0; 

        mu_hat=gamma_hat(:,1); 
        gamma_hat=gamma_hat(:,2:end); 

elseif mu==1 && k==0; 

        mu_hat=gamma_hat(:,1); 
        gamma_hat=[];  

elseif mu==2 && r ~= 0; 

        mu_hat=alfa_hat*(beta_hat(end,:)');
        beta_hat=beta_hat(1:end-1,:); 

elseif mu==2 && r==0;

        mu_hat=0;
        beta_hat=0; 

end; 


% get levels varma representation: 
[A_hat]=vecm2var(alfa_hat,beta_hat,gamma_hat);



varma_obj.data=data; 
varma_obj.res=u_hat;
varma_obj.CovM=CovM_hat; 
varma_obj.mu=mu_hat; 
varma_obj.A_0=eye(K); 
varma_obj.A=A_hat; 
varma_obj.M=[]; 
varma_obj.beta=beta_hat;
varma_obj.alfa=alfa_hat;
varma_obj.gamma=gamma_hat; 




function A=vecm2var(alfa,beta,gammma); 
%  
%      TRANSFORMS VECM 
%
%      DY = mu + PI Y_t-1 + GAMMA X + U,    PI:= alfa * (beta')
%
%      TO VAR IN LEVELS
%
%      Y=mu_var + BETA Y -1 + U
%
%
% CHECKED

    K=size(alfa,1); 
    k=size(gammma,2)/K;
    
    if (k==0)

        A=alfa*(beta')+eye(K); 
    
    else;

        A_1=alfa*(beta')+eye(K)+gammma(:,1:K); 
        A=A_1; 
    
        if (k >= 2)
 
            for i=2:k; 
    
                A_i=gammma(:,((i-1)*K+1):(i*K))-gammma(:,((i-2)*K+1):(i-1)*K);     
                A=[A A_i]; 

            end; 

        end; 

        A_k=-gammma(:,((k-1)*K+1):k*K);
        
		A=[A A_k]; 

    end;  
    
























