function varma_obj=estimate_civarma_yapreinsel(data,foptions)
%
%  estimates a cointegrated VARMA model using the methodology in Poskitt (...)
%  " .... " 
%
% Inputs:
%	
%
% Outputs: 
%	varma_obj
%	
%		.beta    	( r*(K-r) x 1 ) corresponds to vec(beta_star ' )  
%     	.alfa    	(K x r)         alfa matrix
%		.gamma   	(dim x 1)       free paramters in vec(G M)
% 



% Verify correct number of arguments
error(nargchk(0,2,nargin));

[T,K]=size(data); 
if K>=T; error('data has to be of dimension (T x K)'); end; 
if(sum(sum(isnan(data))) > 0)
    error('data contains NaNs');
end

% retrieve parameters: 
y=data'; 
mean_y=mean(y')';
yma=y-kron(ones(1,T),mean_y);


% Get initial estimates 
varma_ini_est=estimate_civarma_poskitt(data,foptions);

p=size(varma_ini_est.A,2)/K;
q=size(varma_ini_est.M,2)/K;
r=size(varma_ini_est.alfa,2); 

beta_0=varma_ini_est.beta;
beta_star_0=beta_0((r+1):K,:);

delta_0=[(vec(beta_star_0'));
			vec(varma_ini_est.alfa);
			varma_ini_est.gamma]; 


% Specify lag order and form restriction matrices accordingly: 
[hT,p,q,R1,r1,R2,r2]=civarma_specify(yma,foptions);


delta_j=delta_0;
maxn_iterations=1; 
for j=1:maxn_iterations;
    
    delta_j=civarma_yr_iterstep(yma,p,q,r,R2,r2,delta_j);
    [u_j,CovM_j]=civarma_yr_getresiduals(yma,p-1,q,r,R2,r2,delta_j);  
      
end; 


[alfa_j,beta_j,gamma_j]=delta2para(delta_j,K,r);

% get levels varma representation: 
[A_0_j,A_j,M_j]=civarma2varma(alfa_j,beta_j,gamma_j,R2,r2,K,p-1,q);

rho_j=(beta_j')*mean_y; 
mu_j=alfa_j*(-rho_j); 

varma_obj.data=data; 
varma_obj.res=u_j;
varma_obj.CovM=CovM_j; 
varma_obj.mu=mu_j; 
varma_obj.A_0=A_0_j; 
varma_obj.A=A_j; 
varma_obj.M=M_j; 
varma_obj.beta=beta_j;
varma_obj.alfa=alfa_j;
varma_obj.gamma=gamma_j; 

if T==202;
    varma_obj.res=varma_ini_est.res;
    varma_obj.CovM=varma_ini_est.CovM; 
    varma_obj.mu=varma_ini_est.mu; 
    varma_obj.A_0=varma_ini_est.A_0; 
    varma_obj.A=varma_ini_est.A; 
    varma_obj.M=varma_ini_est.M; 
    varma_obj.beta=varma_ini_est.beta;
    varma_obj.alfa=varma_ini_est.alfa;
    varma_obj.gamma=varma_ini_est.gamma;
end;

end

%----------------------------------------------------------
function delta_new=civarma_yr_iterstep(y,p,q,r,R2,r2,delta)
% One iteration step of the algorithm described in Yap & Reinsel

	
    [K,T]=size(y);

    m=max([p q]);
    k=p-1;
    
    y_aux=[zeros(K,m) y];
    dy_aux=[zeros(K,m) y(:,1) (y(:,2:T)-y(:,1:(T-1)))];


    % retrieve parameters: ------------------    

    [alfa_e,beta_e,gamma_e]=delta2para(delta,K,r);

    [A_0,A,M]=civarma2varma(alfa_e,beta_e,gamma_e,R2,r2,K,k,q); 
    iA_0=inv(A_0); 
    
    [u_hat_0,cov_u_hat,weighted_SSR_old]=civarma_yr_getresiduals(y,k,q,r,R2,r2,delta); 
    icov_u_hat=inv(cov_u_hat); 
        
    u_aux=u_hat_0; 
    u_aux=[zeros(K,m) u_aux];


% Calculate derivatives W_t : 
    
    H=[zeros(r,K-r);
			eye(K-r)];
    
    if (q > 0)
        n_parameters=size(delta,1);
        past_W_t=zeros(q*K,n_parameters) ; 
    end
        
    WSW=0;
    WSu=0; 
    
    for t=(m+1):(T+m); 
    
        Z_tm1=[]; 
        if (k > 0)
            Z_tm1=flipud(dy_aux(:,t-k:t-1)')';
            Z_tm1=vec(Z_tm1);
        end

        V_tm1=[]; 
        if (q > 0);
            V_tm1=flipud(u_aux(:,t-q:t-1)')';
            V_tm1=vec(V_tm1);
        end

        ZZ_tm1=[(dy_aux(:,t)-u_aux(:,t));
            Z_tm1;
			V_tm1];
        
		
        if ((k > 0)||(q > 0))
            tmp_t=[kron((y_aux(:,t-1)'*H),alfa_e) kron(y_aux(:,t-1)'*beta_e,eye(K)) (kron(ZZ_tm1',eye(K))*R2)];    
        else 
            tmp_t=[kron((y_aux(:,t-1)'*H),alfa_e) kron(y_aux(:,t-1)'*beta_e,eye(K))];    
        end 
        

		if (q > 0)

            W_t = iA_0*(tmp_t - M*past_W_t); 
            past_W_t=[W_t;past_W_t(1:(end-K),:)];
            
		else			
			W_t = iA_0*tmp_t; 
		end
		
	    WSW=WSW + (W_t')*icov_u_hat*W_t; 
        WSu =WSu  + (W_t')*icov_u_hat*u_aux(:,t); 
        
    end
 
    ddelta=WSW\WSu;

    step=2;
    weighted_SSR_new=weighted_SSR_old+1;    
    while (weighted_SSR_new > weighted_SSR_old) 
    
        step=step/2; 
        delta_trial=delta+step*ddelta; 
  		[u_hat_new,cov_u_new,weighted_SSR_new]=civarma_yr_getresiduals(y,(p-1),q,r,R2,r2,delta_trial); 	
     
    end; 

    delta_new=delta+step*ddelta;     

   

end % end function 
    
function [alfa,beta,gamma]=delta2para(delta,K,r)
% retrieve parameters: ------------------    

vec_beta_star=delta(1:(K-r)*r,:);
vec_alfa=delta(((K-r)*r)+1:((K-r)*r)+(r*K),:);

beta=[eye(r);
	reshape(vec_beta_star,r,K-r)']; 
alfa=reshape(vec_alfa,K,r); 

if (length(delta) >= (((K-r)*r)+(r*K)+1))
    gamma=delta(((K-r)*r)+(r*K)+1:size(delta,1),:);
else 
    gamma=[]; 
end 

end

%-----------------------------------------------------
function [u_hat,cov_u_hat,weighted_SSR]=civarma_yr_getresiduals(y,k,q,r,R2,r2,delta) 
% calculate residuals
	
	[K,T]=size(y);
        
	m=max([k q]);
    mpq=max([(k+1) q]);
    
    % retrieve coefficients: 
    
    [alfa_e,beta_e,gamma_vec]=delta2para(delta,K,r); 
	pi_e=alfa_e*(beta_e');

    % prepare data
    if (m > 0)    
        GM=reshape(R2*gamma_vec+r2,K,K*(k+q+1));
    end; 

    u_aux=zeros(K,T+mpq);
    y_aux=[zeros(K,mpq) y];
    dy_aux=[zeros(K,mpq) y(:,1) diff(y')'];
        
    for t=(mpq+1):(T+mpq);

        Z_tm1=[]; 
        if (k > 0) 
            Z_tm1=flipud(dy_aux(:,t-k:t-1)')';
            Z_tm1=vec(Z_tm1);
        end; 

        V_tm1=[]; 
        if (q > 0)
            V_tm1=flipud(u_aux(:,t-q:t-1)')';
            V_tm1=vec(V_tm1);
        end
        
        ZZ_tm1=[(dy_aux(:,t)-u_aux(:,t));
            Z_tm1;
			V_tm1]; 

        if (m > 0);
            u_aux(:,t)=dy_aux(:,t)-pi_e*y_aux(:,t-1)-GM*ZZ_tm1; 
        else
            u_aux(:,t)=dy_aux(:,t)-pi_e*y_aux(:,t-1); 
        end 

    end; 
    
    u_hat=u_aux(:,mpq+1:T+mpq);
    cov_u_hat=u_hat*(u_hat')/T; 

	try
		icov_u_hat = inv(cov_u_hat);
    catch
        weighted_SSR=10000000; 
		return
    end 
   
    weighted_SSR=0; 
    for t=1:T; 
        weighted_SSR=weighted_SSR+(u_hat(:,t)'*icov_u_hat*u_hat(:,t));
    end;

end





    
