function [Y,Xf,Xr,mfo,parm_true,pr] = datagen(varargin)

% Create data for mixed logit model with fixed coefficients and normally
% distributed random coefficients.
%
% Data can be generated under different parameterizations
%
% Model Inputs - 
%   N - the number of individuals
%   T - the number of time periods observed for each individual
%   J - the number of choices
%   syyd - the random number generator seed for data generation
%   nfx - number of observed variables with fixed coefficients
%   nfxnz - number of non-zero fixed x
%   vfx - variance of fixed x portion of utility
%   nrx - number of x variables with random coefficients
%   nrxnz - number of non-zero random coefficient x
%   vrx - variance of random coefficient part of utility 
%   rccorr - correlation among random coefficients
%
% Inputs must be written in pairs, e.g., datagen('N',1000,'J',4)
%
% Model Outputs -
%  Y - a 'JxTxN' by '1' vector of choice indicators
%  Xf - a 'JxTxN' by 'nfx' matrix of choice charactersitcs that have fixed
%  coefficients
%  Xr - a 'JxTxN' by 'nrx' matrix of choice charactersitcs that have random
%  coefficients
%  mfo - a structure with model information
%  parm_true - a structure with the true parameter values
%  pr - a 'TxN' by 'J' matrix of choice probabilities

%% Assign model characteristics
N = 800;
T = 10; 
J = 3;
syyd = 0;
nfx = 2;
vfx = 2;
nrx = 3;
vrx = 2;
rccorr = .25;

while ~isempty(varargin)
    switch varargin{1}
         
        case {'N','T','J','syyd','nfx','nfxnz','vfx','nrx','nrxnz','vrx','rccorr'}
            eval([varargin{1} ' = varargin{2};']);
            varargin(1:2) = [];
        otherwise
            error(['Unexpected option: ' varargin{1}])
    end
end

if ~exist('nfxnz','var'), nfxnz = nfx; end;
if ~exist('nrxnz','var'), nrxnz = nrx; end;

nfxnz = min(nfxnz,nfx);
nrxnz = min(nrxnz,nrx);

%% Parameters
FC = sqrt(vfx/nfxnz)*ones(nfx,1);
RCmean = zeros(nrx,1);
RCvar = (vrx/nrxnz)*eye(nrx);
RCvar(eye(nrx)==0) = rccorr*RCvar(1,1);
parm_true = struct('FC',FC,'RCmean',RCmean,'RCvar',RCvar,'cholRCvar',chol(RCvar,'lower'));

%% Covariates
rng(syyd)
Xf = randn([J*T*N nfx]);
if nfxnz<nfx
    start_fxnz = bsxfun(@plus,randi(nfx,[J*T*N 1]),(0:nfxnz-1));
    start_fxnz(start_fxnz>nfx) = start_fxnz(start_fxnz>nfx) - nfx;
    Xf = full(Xf.*sparse((1:J*T*N)'*ones(1,nfxnz),start_fxnz,1,J*T*N,nfx));
end

rng((syyd+1)^2)
Xr = randn([J*T*N nrx]);
if nrxnz<nrx
    start_rxnz = bsxfun(@plus,randi(nrx,[J*T*N 1]),(0:nrxnz-1));
    start_rxnz(start_rxnz>nrx) = start_rxnz(start_rxnz>nrx) - nrx;
    Xr = full(Xr.*sparse((1:J*T*N)'*ones(1,nrxnz),start_rxnz,1,J*T*N,nrx));
end
    
%% Generate Data
beta_i = mvnrnd(RCmean,RCvar,N);
vFC = reshape(Xf*FC,[J T*N]);
vRC = reshape(sum(Xr.*beta_i(ones(J*T,1)*(1:N),:),2),[J T*N]);
v = vFC+vRC;
pr = bsxfun(@rdivide,exp(v),sum(exp(v),1))';
y = sum(bsxfun(@lt,cumsum(pr,2),rand([T*N 1])),2)+1;

Y = bsxfun(@eq,y,(1:J));
Y = reshape(Y',[J*T*N 1]);

mfo = struct('N',N,'T',T,'J',J);


