function [Lstar, Ustar, CI_L, CI_U] = Conf_Int_CLR_old(bL, VbL, bU, VbU, n, ...
    R, alfa)
%
%
%Conf_Int_CLR. Based on the paper by Chernozhukov, Lee and Rosen (2011), 
% this function provides: (1) half-median unbiased estimators of lower and 
% upper bounds that may involve "min/max" operators; and, (2) 1-alfa
% percent confidence intervals for the parameter of interest (like those 
% in Imbens and Manski (2004), but allowing for min/max operators). 
%
%SETUP. See my notes pages 7 and 8. Let theta_star be the parameter of 
% interest, which lies between (is bounded by) [Lstar, Ustar], where 
% Lstar=max(L1,L2,...,LkL) and Ustar=min(U1,U2,...,UkU). As mentioned 
% above, we may have kL and/or kU equal to one.
%
%INPUTS: (bL, VbL, bU, VbU, n, R, alfa). bL is the kL by 1 vector of 
% estimated LOWER bounds, and VbL is its kL by kL variance-covariance 
% matrix. bU is the kU by 1 vector of estimated UPPER bounds, and VbU is 
% its kU by kU var-cov matrix. n is the sample size used to generate those 
% estimates. R is the number of repetitions to be used to compute 
% half-unbiased estimators and confidence intervals (Recommended: at least 
% 100,000. See my notes page 15). alfa is the significance level, so that 
% 1-alfa is the confidence level, e.g., for a 95 percent CI use alfa=0.05.
%
%OUTPUT: [Lstar, Ustar, CI_L, CI_U]. Lstar and Ustar are the half-median 
% unbiased estimators of the lower and upper bounds, respectively. CI_L and 
% CI_U are the lower and upper ends of the (1-alfa) percent confidence 
% interval, respectively.

%NOTES: (1) See my notes "Implementing CLR (2011) ..." from 12/21/11, pages 
% 7,8 and 14. I refer to these notes in this m-file. (2) This program is 
% based on m-files Preliminary_122211.m and Preliminary_LB_122311.m. I drop 
% some comments from those files to save space. For detailed comments and 
% more explanations of specific steps see those two m-files. (3) Some of 
% the steps/commands below can be written in fewer lines of code. I chose 
% clarity over efficiency (program will run fast anyways). (4) This program
% also works for cases with no "min/max" operators. In that case, the half-
% median unbiased estimator should be close to the original estimator and
% the (1-alfa)% CI should be close to the one obtained from IM(04). See my
% notes pages 11 and 15. (5) For the general algorithm used here, see my
% notes pages 4 and 5. (6) This program does have detailed comments for 
% getting the confidence intervals, since these were not calculated in the
% two programs mentioned above. (7) If we want to use this m-file as an
% usual m-file rather than a "function", we just need to comment out the
% first line above. Then, we would have to define each of the inputs in the
% program: bL, VbL, bU, VbU, n, R and alfa. (8) Remember, to use it in
% matlab type "[a,b,c,d]== Conf_Int_CLR(bL, VbL, bU, VbU, n, R, alfa)", and
% a, b, c and d keep the corresponding results.



% For the steps followed in this m-file see my notes page 16.

% Carlos, 12/26/11


% **********
%Stuff used when writing m-file (BE SURE TO ERASE OR COMMENT OUT @ THE END)
%bL=[1 2]'; % Vector of coefficients. Change to [2 1] to change variance
%VbL=[0.01 .005 ; .005 0.04];  % Var-Cov matrix
%bL=0;
%VbL=0.000000001;

%bU=[4 5]' ;% Vector of coefficients
%VbU=[0.01 .005 ; .005 0.04];  % Var-Cov matrix
%bU=5;
%VbU=0.04;
%bU=[ 4 5 6]';
%VbU=[0.01 .005 .005; .005 0.10 .005; .005 .005 .04];

%n=100; % Set sample size
%R=10000; % Set number of repetitions. KEY: See my notes page 15.
%alfa=0.05; %To get a 95 percent CI.
% **********





%% ***** I. GET HALF-MEDIAN UNBIASED ESTIMATE OF LOWER BOUND, Lstar *****

% This part is based on program Preliminary_LB_122311.m

%disp(' ');
%disp('**************************************************************');
%disp('  ');
%disp('----- Consider LOWER BOUND: Lstar=max(L1,L2,...,LkL) -----');
%disp('--- I. Get half-median unbiased estimator of Lstar ---'); 
%disp(' ');

%% Note: Steps 1 to 4(a) are the SAME for lower and upper bound.

%%% STEP 1: Get random draws from normals and values of constants.
c=1-(.1/log(n)); % Set value of constant, gamma tilda in CLR (11)
kL=size(bL,1); % Set the number of possible LOWER bounds, #rows in bL.
ZL=normrnd(0,1,kL,R); %It generates R draws from a kL-variate normal 
% distribution mean zero and kL by kL identity matrix. KEY: For us, each 
% COLUMN is a draw.


%%% STEP 2: Get a consistent estimate of the asymptotic distribution of
%%% sqrt(n)[bL-bLo].
OmegaL=VbL*n;


%%% STEP 3: Here we compute some objects to be used below. 
sqrOmegaL=sqrtm(OmegaL); %The unique square root of OmegaL

% sL is a kLx1 vector containing all the s(v) for v=1,...,kL
sL=zeros(kL,1); 
for v=1:kL
    sL(v)=sqrt(dot(sqrOmegaL(v,:),sqrOmegaL(v,:)))/sqrt(n);
end
%disp('Value of the vector of lower bound estimates, bL=[L1 L2 ... LkL]');
%disp(bL);
%disp('Value of sL (std. error of each lower bound estimate)');
%disp(sL);


%%% STEP4: Get V hat. This set includes those estimators of the LOWER bound
%%% that can affect the asymptotic distribution of the estimator of the 
%%% MAXIMUM, Lstar, (e.g., see my notes page 13).

%%% 4(a): Get critical value.
%% (i): Get matrix ZstarL. 
ZstarL=sqrOmegaL*ZL;
for v=1:kL
    ZstarL(v,:)=ZstarL(v,:)/(sqrt(n)*sL(v));
end
clear ZL;% ZL can be a huge matrix and consume a lot of space.

%% (ii): Get maxZstarL, which is a 1 by R row vector.
if kL==1
    maxZstarL=ZstarL;
else
    maxZstarL=max(ZstarL);
end

%% (iii): Get the c-th quantile of maxZstarL. This is the critical value in 
%% step 4a.
cv1L=quantile(maxZstarL,c);
%disp('Critical value used to determine V hat:');
%disp(cv1L);
clear maxZstarL;

%%% 4(b): ***THIS STEP IS LOWER-BOUND SPECIFIC***. Here we get the set 
%%% V hat. See my notes page 13.

%% (i): Get MAXIMUM precision-adjusted estimator.
maxv1=max(bL-cv1L*sL);
%% (ii) Create a kLx1 vector indicating whether the estimator belongs to 
%% Vhat (=1) or not (=0).
indVhatL=bL>=maxv1-2*cv1L*sL;
%disp('Number of elements in Vhat (i.e., number of elements affecting the');
%disp('asymptotic distribution of Lstar):');
%disp(sum(indVhatL));
%disp('Elements belonging to Vhat (if =1)');
%disp(indVhatL);%Vector can be long if kL is large (usually not our case).


%%% STEP5: Get estimator of LOWER bound.

%%% 5(a): Get critical value.
%% (i): We create the matrix Zstar_VhatL. It contains the rows from ZstarL 
%% corresponding to those estimators/bounds in Vhat.
Zstar_VhatL=zeros(sum(indVhatL),R);
i=1; %Counter used to fill in the appropriate row of Zstar_VhatL.
for v=1:kL
    if indVhatL(v,1)==1
        Zstar_VhatL(i,:)=ZstarL(v,:);
        i=i+1;
    end
end
clear ZstarL;% ZstarL can be a huge matrix and consume a lot of space.

%% (ii): Get a row vector containing the maximum element for each column of
%% Zstar_VhatL (i.e., the max value over "v IN Vhat" for each replication).
%% We use an "if" because if Zstar_VhatL is a vector, then the max function 
%% returns a scalar. See my notes page 11. maxZstar_VhatL is a 1 by R row 
%% vector. See also note (3) at the beginning of the m-file 
%% Preliminary_LB_122311.m.
%% KEY: This vector is also used below to get critical values for 
%% confidence intervals.
if sum(indVhatL)==1
    maxZstar_VhatL=Zstar_VhatL;
else
    maxZstar_VhatL=max(Zstar_VhatL); 
end
clear Zstar_VhatL;% This can be a huge matrix and consume a lot of space.
    
%% (iii): Get the p-th quantile of maxZstar_VhatL. This is the critical 
%% value in step 5a.
cv2L=quantile(maxZstar_VhatL,0.5); %Use p=0.5 to get half-median unbiased 
% estimator of lower bound.
%disp('Critical value used for half-median unbiased est. of lower bound');
%disp(cv2L);

%%% 5(b): ***THIS STEP IS LOWER-BOUND SPECIFIC***. Get half-median unbiased
%%% estimator of the LOWER bound, Lstar. See my notes page 13.
Lstar=max(bL-cv2L*sL);
%disp('***** Half-Median Unbiased Estimator of LOWER BOUND:');
%disp(Lstar);
%disp(' ');





%% ***** II. GET HALF-MEDIAN UNBIASED ESTIMATE OF UPPER BOUND, Ustar *****

% This part is based on program Preliminary_122211.m

%disp('**************************************************************');
%disp('  ');
%disp('----- Consider UPPER BOUND: Ustar=min(U1,U2,...,UkU) -----');
%disp('--- II. Get half-median unbiased estimator of Ustar ---'); 
%disp(' ');

%% Note: Steps 1 to 4(a) are the SAME for lower and upper bound.

%%% STEP 1: Get random draws from normals and values of constants. Note
%%% that "c" is the same as for lower bound above.
kU=size(bU,1); % Set the number of possible UPPER bounds, #rows in bU.
ZU=normrnd(0,1,kU,R); %It generates R draws from a kU-variate normal 
% distribution mean zero and kU by kU identity matrix. KEY: For us, each 
% COLUMN is a draw.


%%% STEP 2: Get a consistent estimate of the asymptotic distribution of
%%% sqrt(n)[bU-bUo].
OmegaU=VbU*n;


%%% STEP 3: Here we compute some objects to be used below. 
sqrOmegaU=sqrtm(OmegaU); %The unique square root of Omega

% sU is a kUx1 vector containing all the s(v) for v=1,...,kU.
sU=zeros(kU,1); 
for v=1:kU
    sU(v)=sqrt(dot(sqrOmegaU(v,:),sqrOmegaU(v,:)))/sqrt(n);
end
%disp('Value of the vector of upper bound estimates, bU=[U1 U2 ... UkU]');
%disp(bU);
%disp('Value of sU (std. error of each upper bound estimate)');
%disp(sU);


%%% STEP4: Get V hat. This set includes those estimators of the UPPER bound
%%% that can affect the asymptotic distribution of the estimator of the 
%%% MINIMUM, Ustar (e.g., see my note on top of CLR(11) p.17).

%%% 4(a): Get critical value.
%% (i): Get matrix ZstarU.
ZstarU=sqrOmegaU*ZU;
for v=1:kU
    ZstarU(v,:)=ZstarU(v,:)/(sqrt(n)*sU(v));
end
clear ZU;% ZU can be a huge matrix and consume a lot of space.

%% (ii): Get maxZstarU, which is a 1 by R row vector.
if kU==1
    maxZstarU=ZstarU;
else
    maxZstarU=max(ZstarU);
end

%% (iii): Get the c-th quantile of maxZstarU. This is the critical value in 
%% step 4a.
cv1U=quantile(maxZstarU,c);
%disp('Critical value used to determine V hat:');
%disp(cv1U);
clear maxZstarU;

%%% 4(b): *** THIS STEP IS UPPER-BOUND SPECIFIC***. Here we get the set 
%%% V hat. See my notes page 11.

%% (i): Get MINIMUM precision-adjusted estimator.
minv1=min(bU+cv1U*sU);
%% (ii) Create a kUx1 vector indicating whether the estimator belongs to 
%% Vhat (=1) or not (=0).
indVhatU=bU<=minv1+2*cv1U*sU;
%disp('Number of elements in Vhat (i.e., number of elements affecting the');
%disp('asymptotic distribution of Ustar):');
%disp(sum(indVhatU));
%disp('Elements belonging to Vhat (if =1)');
%disp(indVhatU);%Vector can be long if kU is large (usually not our case).


%%% STEP5: Get estimator of UPPER bound.

%%% 5(a): Get critical value.
%% (i): We create the matrix Zstar_VhatU. It contains the rows from ZstarU 
%% corresponding to those estimators/bounds in Vhat.
Zstar_VhatU=zeros(sum(indVhatU),R);
i=1; %Counter used to fill in the appropriate row of Zstar_VhatU.
for v=1:kU
    if indVhatU(v,1)==1
        Zstar_VhatU(i,:)=ZstarU(v,:);
        i=i+1;
    end
end
clear ZstarU;% ZstarU can be a huge matrix and consume a lot of space.

%% (ii): Get a row vector containing the maximum element for each column of
%% Zstar_VhatU (i.e., the max value over "v IN Vhat" for each replication).
%% We use an "if" because if Zstar_VhatU is a vector then the max function 
%% returns a scalar. See my notes page 11. maxZstar_VhatU is a 1 by R row 
%% vector. See also note (4) at the beginning of the m-file 
%% Preliminary_122211.m.
%% KEY: This vector is also used below to get critical values for 
%% confidence intervals.
if sum(indVhatU)==1
    maxZstar_VhatU=Zstar_VhatU;
else
    maxZstar_VhatU=max(Zstar_VhatU); 
end
clear Zstar_VhatU;% This can be a huge matrix and consume a lot of space.
    
%% (iii): Get the p-th quantile of maxZstar_VhatU. This is the critical 
%% value in step 5a.
cv2U=quantile(maxZstar_VhatU,0.5); %Use p=0.5 to get half-median unbiased 
% estimator of upper bound.
%disp('Critical value used for half-median unbiased est. of upper bound');
%disp(cv2U);

%%% 5(b): ***THIS STEP IS UPPER-BOUND SPECIFIC***. Get half-median unbiased 
%%% estimator of the UPPER bound, Ustar. See my notes page 4.
Ustar=min(bU+cv2U*sU);
%disp('***** Half-Median Unbiased Estimator of UPPER BOUND:');
%disp(Ustar);
%disp(' ');





%% ***** III. GET (1-alfa) PERCENT CONFIDENCE INTERVAL FOR PARAMETER *****

%disp('**************************************************************');
%disp('  ');
%disp('--- III. Get (1-alfa) percent Confidence Interval for Parameter---');
%disp('--- Confidence Interval has the form: [CI_L, CI_U] ---'); 
%disp(' ');

%% Note: See my notes pages 17, 5 and CLR (09) pages 25 and 26.

%%% STEP 1: Get the values of the variables used to determine the critical 
%%% value to be used to construct the confidence interval.
Delta_hat_plus=max(0,Ustar-Lstar);
% Get sigma, see my notes pages 17 & 18.
U75=min(bU+(quantile(maxZstar_VhatU,0.75))*sU);
U25=min(bU+(quantile(maxZstar_VhatU,0.25))*sU);
L75=max(bL-(quantile(maxZstar_VhatL,0.75))*sL);
L25=max(bL-(quantile(maxZstar_VhatL,0.25))*sL);
sigma=max(U75-U25, L25-L75);
tao=1/(sigma*log(n));
phat=1-normcdf(real(tao*Delta_hat_plus))*alfa; % NOTE: This is the only thing 
% that changes in the program when we want to change the confidence level. 

% For reference, I compute the phat we would use if instead we used 
% tao=log(n), as discussed in CLR(09) p. 26, or as used in Nevo & Rosen 
% (2010) page 23. 
%phat_alternative=1-normcdf(real(log(n)*Delta_hat_plus))*alfa;


%disp('Quantities used to get tao & phat, see CRL(09) p. 26; my notes p.18');
%disp(' ');
%disp('Delta_hat_plus = max(0, Ustar-Lstar)');
%disp(Delta_hat_plus);
%disp('U75');
%disp(U75);
%disp('U25');
%disp(U25);
%disp('L75');
%disp(L75);
%disp('L25');
%disp(L25);
%disp('U75-U25');
%disp(U75-U25);
%disp('L25-L75');
%disp(L25-L75);
%disp('sigma');
%disp(sigma);
%disp('Value of tao when taking into account the variance of Delta hat.');
%disp('This is the preferred choice of tao. See references above');
%disp(tao);
%disp('Specified Confidence Level (for reference, 1-significance level):');
%disp(1-alfa);
%disp('Value of p_hat. This is the one we use');
%disp(phat)
%disp('For reference, this is the phat we would use if instead we used');
%disp(' tao=log(n). E.g., see CLR(09) p. 26, or Nevo & Rosen (2010) p. 23');
%disp(phat_alternative);

%%% STEP 2: Get the (1-alfa) percent confidence interval.
% Lower end of confidence interval.
cvCIL=quantile(maxZstar_VhatL,phat); %Critical value used
%disp(' ');
%disp(' ');
%disp('Critical value used for LOWER END of (1-alfa)% Confidence Interval');
%disp(cvCIL);
CI_L=max(bL-cvCIL*sL);
%disp('***** LOWER END of (1-alfa)% CONFIDENCE INTERVAL, CI_L:');
%disp(CI_L);

%Upper end of confidence interval
cvCIU=quantile(maxZstar_VhatU,phat); %Critical value used
%disp('Critical value used for UPPER END of (1-alfa)% Confidence Interval');
%disp(cvCIU);
CI_U=min(bU+cvCIU*sU);
%disp('***** UPPER END of (1-alfa)% CONFIDENCE INTERVAL, CI_U:');
%disp(CI_U);
%disp(' ');
%disp(' ');


% %% Finally, FOR REFERENCE, I report the confidence interval resulting from 
% %% using tao=log(n). See comments above.
% %disp('For reference, we provide the CI with tao=log(n). See notes above.');
% CI_L_alternative=max(bL-(quantile(maxZstar_VhatL,phat_alternative))*sL);
% %disp('Lower end of (1-alfa)% CI with NON-PREFERRED tao=log(n).');
% %disp(CI_L_alternative);
% %Upper end of confidence interval
% CI_U_alternative=min(bU+(quantile(maxZstar_VhatU,phat_alternative))*sU);
% %disp('Upper end of (1-alfa)% CI with NON-PREFERRED tao=log(n).');
% %disp(CI_U_alternative);


%%disp('Program finished running');


