clc

%This is a summary m-file that demonstrates some of the claims from the
%following paper: 
%Sharp IV bounds on average treatment effects on the treated and other populations under endogeneity and noncompliance
%Journal of Applied Econometrics
%by Martin Huber, Giovanni Mellace and Lukas Laffers
%(to run this file it takes approx ~90seconds on i7 2GHz 16GB RAM)

%On top of presenting the linear program that can replicate the bounds that
%were derived analytically. Using this framework, we will also confirm the validity of the
%proofs of a few lemmas in the paper.

longString = ['This is a summary m-file that demonstrates some of the claims from the following paper:\n Sharp IV bounds on average treatment effects on the treated and other populations under endogeneity and noncompliance\n by Martin Huber, Giovanni Mellace and Lukas Laffers\n (to run this file it takes approx ~120seconds on i7 2GHz 16GB RAM)'];
disp(sprintf(longString));

%loads data into dataset and probabilitity distribution in probDistribution
loaddata 
disp(sprintf(['\n***Data loaded.***\n']))

%%
%Outer bounds for the proportion of defiers based on distribution (D,Z)
%only.
outerBoundsForDefiers = calculateouterboundsfordefiers(probDistribution);

%Check if the outer upper bound for the proportion of defiers is
%admissible.
isUpperOuterBoundAdmissible = isintheidentifiedset(probDistribution,outerBoundsForDefiers(2),supportOfOutcomeY);

%Find sharp lower bound for the proportion of defiers based on linear
%programming.
tolerance = 10e-6; %Set the tolerance level.
lowerBoundForDefiersUnderA1A2 = findlowerboundfordefiers(outerBoundsForDefiers(1),outerBoundsForDefiers(2),tolerance,probDistribution,supportOfOutcomeY);
lowerBoundForDefiersUnderA1A2A4 = findlowerboundfordefiers(outerBoundsForDefiers(1),outerBoundsForDefiers(2),tolerance,probDistribution,supportOfOutcomeY,1);
upperBoundForDefiers = outerBoundsForDefiers(2); %The upper outer bound is sharp.
sharpBoundsForDefiers = [lowerBoundForDefiersUnderA1A2,upperBoundForDefiers];

%%
%Check that identified set is an interval. (~30seconds)
disp(sprintf(['\n***Check that identified set for the share of defiers is an interval for the dataset used in the paper.***']))
sizeGridInterval = 100;
disp(sprintf(['\nUnder Assumptions A1, A2']))
identifiedsetisaninterval(sharpBoundsForDefiers(1),sharpBoundsForDefiers(2),sizeGridInterval,probDistribution,supportOfOutcomeY,1)
disp(sprintf(['\nUnder Assumptions A1, A2, A4']))
identifiedsetisaninterval(sharpBoundsForDefiers(1),sharpBoundsForDefiers(2),sizeGridInterval,probDistribution,supportOfOutcomeY,1)

%%
%Calculate sharp bounds under A1, A2
sizeGrid = 1; %this is set just for the comparison purposes, greater number will increase the computing time
%(1) based on Linear programming tool
BoundsLP = calculateboundsonateusinglp(probDistribution,sizeGrid,supportOfOutcomeY,lowerBoundForDefiersUnderA1A2,upperBoundForDefiers);
%(2) derived in the paper
BoundsAnalytic = calculateanalyticboundsonate(dataset,sizeGrid,supportOfOutcomeY,lowerBoundForDefiersUnderA1A2,upperBoundForDefiers);

%Store the results in a structure ResultEmpirical
ResultsEmpirical.dataset = dataset;
ResultsEmpirical.probDistribution = probDistribution;
ResultsEmpirical.outerBoundsForDefiers = outerBoundsForDefiers;
ResultsEmpirical.sharpBoundsForDefiers = sharpBoundsForDefiers;
ResultsEmpirical.isUpperOuterBoundAdmissible = isUpperOuterBoundAdmissible;
ResultsEmpirical.BoundsLP = BoundsLP;
ResultsEmpirical.BoundsAnalytic = BoundsAnalytic;
ResultsEmpirical.maximalDifference = max(abs([ResultsEmpirical.BoundsLP.EntirePopulation.lowerBound-...
        ResultsEmpirical.BoundsAnalytic.EntirePopulation.lowerBound,
    BoundsLP.EntirePopulation.upperBound-...
        ResultsEmpirical.BoundsAnalytic.EntirePopulation.upperBound;
    ResultsEmpirical.BoundsLP.Alwaystakers.lowerBound-...
        ResultsEmpirical.BoundsAnalytic.Alwaystakers.lowerBound,
    ResultsEmpirical.BoundsLP.Alwaystakers.upperBound-...
        ResultsEmpirical.BoundsAnalytic.Alwaystakers.upperBound;
    ResultsEmpirical.BoundsLP.Compliers.lowerBound-...
        ResultsEmpirical.BoundsAnalytic.Compliers.lowerBound,
    ResultsEmpirical.BoundsLP.Compliers.upperBound-...
        ResultsEmpirical.BoundsAnalytic.Compliers.upperBound;
    ResultsEmpirical.BoundsLP.Nevertakers.lowerBound-...
        ResultsEmpirical.BoundsAnalytic.Nevertakers.lowerBound,
    ResultsEmpirical.BoundsLP.Nevertakers.upperBound-...
        ResultsEmpirical.BoundsAnalytic.Nevertakers.upperBound;
    ResultsEmpirical.BoundsLP.TreatedPopulation.lowerBound-...
        ResultsEmpirical.BoundsAnalytic.TreatedPopulation.lowerBound,
    ResultsEmpirical.BoundsLP.TreatedPopulation.upperBound-...
        ResultsEmpirical.BoundsAnalytic.TreatedPopulation.upperBound]));

%%
%Display the comparison of the results.
disp(sprintf(['\n***We display the comparison of LP and Analytic Bounds.***']))
displayresults(ResultsEmpirical)

%%
%Check construction of the proof of the lemma that states that the identified set is an interval. 
disp(sprintf(['\n***We check the construction of the proof of Lemma 1.***']))
sizeLambdaGrid = 100;
lambda = 0:(1/(sizeLambdaGrid-1)):1;
pi01Min = ResultsEmpirical.sharpBoundsForDefiers(1);
pi01Max = ResultsEmpirical.sharpBoundsForDefiers(2);

violationInequality = zeros(1,sizeLambdaGrid-2);
violationEquality = zeros(1,sizeLambdaGrid-2);

tic %(~20seconds)
for ilambda = 2:(sizeLambdaGrid-1) %Smallest and largest values are not considered as they lead to 0 in the denominator.
    pi01Lambda = lambda(ilambda)*pi01Min + (1-lambda(ilambda))*pi01Max;
    lambda11 = lambda(ilambda)*(P10 - pi01Min)/(lambda(ilambda)*(P10 - pi01Min) + (1-lambda(ilambda))*(P10 - pi01Max));
    lambda10 = lambda(ilambda)*(P11 - P10 + pi01Min)/(lambda(ilambda)*(P11 - P10 + pi01Min) + (1-lambda(ilambda))*(P11 - P10 + pi01Max));
    lambda01 = lambda(ilambda)*pi01Min/(lambda(ilambda)*pi01Min + (1-lambda(ilambda))*pi01Max);
    lambda00 = lambda(ilambda)*(P01 - pi01Min)/(lambda(ilambda)*(P01 - pi01Min) + (1-lambda(ilambda))*(P01 - pi01Max));

    [~,jointDistributionMin] = isintheidentifiedset(probDistribution,pi01Min,supportOfOutcomeY);
    [~,jointDistributionMax] = isintheidentifiedset(probDistribution,pi01Max,supportOfOutcomeY);

    jointDistributionMinField = reshape(jointDistributionMin,sizeOfSupportOfY,sizeOfSupportOfY,sizeOfSupportOfT,sizeOfSupportOfZ);
    jointDistributionMaxField = reshape(jointDistributionMax,sizeOfSupportOfY,sizeOfSupportOfY,sizeOfSupportOfT,sizeOfSupportOfZ);

    jointDistributionLambdaField(:,:,1,:) = lambda11*jointDistributionMinField(:,:,1,:) + (1-lambda11)*jointDistributionMaxField(:,:,1,:);
    jointDistributionLambdaField(:,:,2,:) = lambda10*jointDistributionMinField(:,:,2,:) + (1-lambda10)*jointDistributionMaxField(:,:,2,:);
    jointDistributionLambdaField(:,:,3,:) = lambda01*jointDistributionMinField(:,:,3,:) + (1-lambda01)*jointDistributionMaxField(:,:,3,:);
    jointDistributionLambdaField(:,:,4,:) = lambda00*jointDistributionMinField(:,:,4,:) + (1-lambda00)*jointDistributionMaxField(:,:,4,:);
    jointDistributionLambda = reshape(jointDistributionLambdaField,sizeOfSupportOfY*sizeOfSupportOfY*sizeOfSupportOfT*sizeOfSupportOfZ,1);

    [LinearProgrammeLambda] = createlinearprogram(probDistribution,supportOfOutcomeY,pi01Lambda);
    violationInequality(ilambda) = max(LinearProgrammeLambda.Aineq*jointDistributionLambda - LinearProgrammeLambda.bineq');
    violationEquality(ilambda) = max(abs(LinearProgrammeLambda.Aeq*jointDistributionLambda - LinearProgrammeLambda.beq));
end

violation = max([violationInequality;violationEquality]);
if max(violation) < 10e-6
    disp(sprintf(['\nThe construction of the proof of Lemma 1 works for ',num2str(sizeLambdaGrid),' equidistant values between pi01Min and pi01Max.']))
else
    disp(sprintf(['\nThe construction of the proof of Lemma 1 did not work for at least one of ',num2str(sizeLambdaGrid),' equidistant values between pi01Min and pi01Max.']))
end
toc

%%
%Check construction of the proof of sharpness for ATE bounds within
%principal strata.

%Here set random seed.
randomSeed = 6; %can be changed to see that it works

disp(sprintf(['\n\n***We check construction of the proof of sharpness for ATE bounds within principal strata***\n']));

disp(sprintf(['Random seed is ', num2str(randomSeed),'\n']));

%Create artificial dataset
[dataset,probDistribution] = createartificialdataset(randomSeed,supportOfOutcomeY,10000);

%Calculate different bounds
PYDZ = probDistribution;
PDZ = sum(PYDZ);        %probability of (D,Z)
PZ  = sum(PDZ);         %probability of Z
P11 = PDZ(1,1,1)/PZ(1); %probability of D=1 given Z=1
P10 = PDZ(1,1,2)/PZ(2); %
P01 = PDZ(1,2,1)/PZ(1); %
P00 = PDZ(1,2,2)/PZ(2); %
outerBoundsForDefiers = calculateouterboundsfordefiers(probDistribution);
lowerBoundForDefiers = findlowerboundfordefiers(outerBoundsForDefiers(1),outerBoundsForDefiers(2),tolerance,probDistribution,supportOfOutcomeY);
upperBoundForDefiers = outerBoundsForDefiers(2);
sharpBoundsForDefiers = [lowerBoundForDefiers,upperBoundForDefiers];

disp(sprintf(['Outer bounds for the share of defiers [', num2str(outerBoundsForDefiers(1)),...
    ' , ',num2str(outerBoundsForDefiers(2)),']']))

disp(sprintf(['Sharp bounds for the share of defiers [', num2str(sharpBoundsForDefiers(1)),...
    ' , ',num2str(sharpBoundsForDefiers(2)),']\n']))

%Set pi01 to maximal value.
pi01 = outerBoundsForDefiers(2);
[ConditionalMeans, ConditionalDistributions] = conditionalmeansanddistributions(dataset,supportOfOutcomeY,pi01);

%We construct h_11_1_LowAte and h_11_0_LowAte that gives minimal E[Y(0)] for
%T=11.
h_11_1_LowAte = zeros(sizeOfSupportOfY,sizeOfSupportOfY);
h_11_0_LowAte = zeros(sizeOfSupportOfY,sizeOfSupportOfY);
if ConditionalMeans.y11min > ConditionalMeans.y01min
    h_11_1_LowAte(:,sizeOfSupportOfY) = ConditionalDistributions.vec11min;
    
    alpha_11_0_low = (ConditionalMeans.y01max - ConditionalMeans.y11min)/(ConditionalMeans.y01max - ConditionalMeans.y01min);
    h_11_0_LowAte(:,sizeOfSupportOfY) = alpha_11_0_low*ConditionalDistributions.vec01min + (1-alpha_11_0_low)*ConditionalDistributions.vec01max;
else
    alpha_11_1_low = (ConditionalMeans.y11max - ConditionalMeans.y01min)/(ConditionalMeans.y11max - ConditionalMeans.y11min);
    h_11_1_LowAte(:,sizeOfSupportOfY) = alpha_11_1_low*ConditionalDistributions.vec11min + (1-alpha_11_1_low)*ConditionalDistributions.vec11max;
        
    h_11_0_LowAte(:,sizeOfSupportOfY) = ConditionalDistributions.vec01min;
end

%We construct h_00_1_LowAte and h_00_0_LowAte that gives minimal E[Y(1)] for T=00.
h_00_1_LowAte = zeros(sizeOfSupportOfY,sizeOfSupportOfY);
h_00_0_LowAte = zeros(sizeOfSupportOfY,sizeOfSupportOfY);
if ConditionalMeans.y10min > ConditionalMeans.y00min
    h_00_1_LowAte(sizeOfSupportOfY,:) = ConditionalDistributions.vec10min';
    
    alpha_00_0_low = (ConditionalMeans.y00max - ConditionalMeans.y10min)/(ConditionalMeans.y00max - ConditionalMeans.y00min);
    h_00_0_LowAte(sizeOfSupportOfY,:) = alpha_00_0_low*ConditionalDistributions.vec00min' + (1-alpha_00_0_low)*ConditionalDistributions.vec00max';

else
    alpha_00_1_low = (ConditionalMeans.y10max - ConditionalMeans.y00min)/(ConditionalMeans.y10max - ConditionalMeans.y10min);
    h_00_1_LowAte(sizeOfSupportOfY,:) = alpha_00_1_low*ConditionalDistributions.vec10min' + (1-alpha_00_1_low)*ConditionalDistributions.vec10max';
    
    h_00_0_LowAte(sizeOfSupportOfY,:) = ConditionalDistributions.vec00min';
end

%We construct h_11_1_HighAte and h_11_0_HighAte that gives maximal E[Y(0)] for T=11.
h_11_1_HighAte = zeros(sizeOfSupportOfY,sizeOfSupportOfY);
h_11_0_HighAte = zeros(sizeOfSupportOfY,sizeOfSupportOfY);
if ConditionalMeans.y11max < ConditionalMeans.y01max
     h_11_1_HighAte(:,1) = ConditionalDistributions.vec11max;
     
     alpha_11_0_high = (ConditionalMeans.y01max - ConditionalMeans.y11max)/(ConditionalMeans.y01max - ConditionalMeans.y01min);
     h_11_0_HighAte(:,1) = alpha_11_0_high*ConditionalDistributions.vec01min + (1-alpha_11_0_high)*ConditionalDistributions.vec01max;
else
     alpha_11_1_high = (ConditionalMeans.y11max - ConditionalMeans.y01max)/(ConditionalMeans.y11max - ConditionalMeans.y11min);
     h_11_1_HighAte(:,1) = alpha_11_1_high*ConditionalDistributions.vec11min + (1-alpha_11_1_high)*ConditionalDistributions.vec11max;
     
     h_11_0_HighAte(:,1) = ConditionalDistributions.vec01max;
end

%We construct h_00_1_HighAte and h_00_0_HighAte that gives maximal E[Y(1)] for T=00.
h_00_1_HighAte = zeros(sizeOfSupportOfY,sizeOfSupportOfY);
h_00_0_HighAte = zeros(sizeOfSupportOfY,sizeOfSupportOfY);
if ConditionalMeans.y10max < ConditionalMeans.y00max
     h_00_1_HighAte(1,:) = ConditionalDistributions.vec10max';
     
     alpha_00_0_high = (ConditionalMeans.y00max - ConditionalMeans.y10max)/(ConditionalMeans.y00max - ConditionalMeans.y00min);
     h_00_0_HighAte(1,:) = alpha_00_0_high*ConditionalDistributions.vec00min' + (1-alpha_00_0_high)*ConditionalDistributions.vec00max';
else
     alpha_00_1_high = (ConditionalMeans.y10max - ConditionalMeans.y00max)/(ConditionalMeans.y10max - ConditionalMeans.y10min);
     h_00_1_HighAte(1,:) = alpha_00_1_high*ConditionalDistributions.vec10min' + (1-alpha_00_1_high)*ConditionalDistributions.vec10max';
     
     h_00_0_HighAte(1,:) = ConditionalDistributions.vec00max';
end

%Max ATE for T=00, Min ATE for T=11
jointDistributionMaxAteNevertakersField = zeros(sizeOfSupportOfY,sizeOfSupportOfY,4,2);
jointDistributionMaxAteNevertakersField(:,:,1,1) = h_11_1_LowAte;
jointDistributionMaxAteNevertakersField(:,:,4,1) = h_00_1_LowAte;
jointDistributionMaxAteNevertakersField(:,:,1,2) = h_11_0_LowAte;
jointDistributionMaxAteNevertakersField(:,:,4,2) = h_00_0_LowAte;
h_11_1 = h_11_1_LowAte;
h_00_1 = h_00_1_LowAte;
h_11_0 = h_11_0_LowAte;
h_00_0 = h_00_0_LowAte;
h_10 = ((1/(P11 - P10 + pi01)^2))*(P11*PYDZ(:,1,1)/PDZ(1,1,1) -(P10 - pi01)*sum(h_11_1')')*((P00*PYDZ(:,2,2)/PDZ(1,2,2))' -(P01 - pi01)*sum(h_00_0));
h_01 = ((1/(pi01)^2))*(P10*PYDZ(:,1,2)/PDZ(1,1,2) -(P10 - pi01)*sum(h_11_0')')*((P01*PYDZ(:,2,1)/PDZ(1,2,1))' -(P01 - pi01)*sum(h_00_1));
jointDistributionMaxAteNevertakersField(:,:,2,1) = h_10;
jointDistributionMaxAteNevertakersField(:,:,3,1) = h_01;
jointDistributionMaxAteNevertakersField(:,:,2,2) = h_10;
jointDistributionMaxAteNevertakersField(:,:,3,2) = h_01;
jointDistributionMaxAteNevertakersVector = reshape(jointDistributionMaxAteNevertakersField,1,sizeOfSupportOfY*sizeOfSupportOfY*4*2);

%Calculate maximum constraint violation.
violation = [P11*PYDZ(:,1,1)/PDZ(1,1,1) -  ((P10 - pi01)*sum(h_11_1')' + (P11 - P10 + pi01)*sum(h_10')');
P10*PYDZ(:,1,2)/PDZ(1,1,2) -  ((P10 - pi01)*sum(h_11_0')' + pi01*sum(h_01')');
P01*PYDZ(:,2,1)/PDZ(1,2,1) -  (pi01*sum(h_01)' + (P01 - pi01)*sum(h_00_1)');
P00*PYDZ(:,2,2)/PDZ(1,2,2) -  ((P11 - P10 + pi01)*sum(h_10)' + (P01 - pi01)*sum(h_00_0)' );
supportOfOutcomeY*sum(h_11_1')' - supportOfOutcomeY*sum(h_11_0')';
supportOfOutcomeY*sum(h_11_1)' - supportOfOutcomeY*sum(h_11_0)';
supportOfOutcomeY*sum(h_00_1')' - supportOfOutcomeY*sum(h_00_0')';
supportOfOutcomeY*sum(h_00_1)' - supportOfOutcomeY*sum(h_00_0)';
sum(sum(h_11_1))-1;
sum(sum(h_11_0))-1;
sum(sum(h_00_1))-1;
sum(sum(h_00_0))-1;
sum(sum(h_10))-1;
sum(sum(h_01))-1];
maximumViolationMax = max(violation);


%Min ATE for T=00, Max ATE for T=11
jointDistributionMinAteNevertakersField = zeros(sizeOfSupportOfY,sizeOfSupportOfY,4,2);
jointDistributionMinAteNevertakersField(:,:,1,1) = h_11_1_HighAte;
jointDistributionMinAteNevertakersField(:,:,4,1) = h_00_1_HighAte;
jointDistributionMinAteNevertakersField(:,:,1,2) = h_11_0_HighAte;
jointDistributionMinAteNevertakersField(:,:,4,2) = h_00_0_HighAte;
h_11_1 = h_11_1_HighAte;
h_00_1 = h_00_1_HighAte;
h_11_0 = h_11_0_HighAte;
h_00_0 = h_00_0_HighAte;
h_10 = ((1/(P11 - P10 + pi01)^2))*(P11*PYDZ(:,1,1)/PDZ(1,1,1) -(P10 - pi01)*sum(h_11_1')')*((P00*PYDZ(:,2,2)/PDZ(1,2,2))' -(P01 - pi01)*sum(h_00_0));
h_01 = ((1/(pi01)^2))*(P10*PYDZ(:,1,2)/PDZ(1,1,2) -(P10 - pi01)*sum(h_11_0')')*((P01*PYDZ(:,2,1)/PDZ(1,2,1))' -(P01 - pi01)*sum(h_00_1));
jointDistributionMinAteNevertakersField(:,:,2,1) = h_10;
jointDistributionMinAteNevertakersField(:,:,3,1) = h_01;
jointDistributionMinAteNevertakersField(:,:,2,2) = h_10;
jointDistributionMinAteNevertakersField(:,:,3,2) = h_01;
jointDistributionMinAteNevertakersVector = reshape(jointDistributionMinAteNevertakersField,1,sizeOfSupportOfY*sizeOfSupportOfY*4*2);

%Calculate maximum constraint violation.
violation = [P11*PYDZ(:,1,1)/PDZ(1,1,1) -  ((P10 - pi01)*sum(h_11_1')' + (P11 - P10 + pi01)*sum(h_10')');
P10*PYDZ(:,1,2)/PDZ(1,1,2) -  ((P10 - pi01)*sum(h_11_0')' + pi01*sum(h_01')');
P01*PYDZ(:,2,1)/PDZ(1,2,1) -  (pi01*sum(h_01)' + (P01 - pi01)*sum(h_00_1)');
P00*PYDZ(:,2,2)/PDZ(1,2,2) -  ((P11 - P10 + pi01)*sum(h_10)' + (P01 - pi01)*sum(h_00_0)' );
supportOfOutcomeY*sum(h_11_1')' - supportOfOutcomeY*sum(h_11_0')';
supportOfOutcomeY*sum(h_11_1)' - supportOfOutcomeY*sum(h_11_0)';
supportOfOutcomeY*sum(h_00_1')' - supportOfOutcomeY*sum(h_00_0')';
supportOfOutcomeY*sum(h_00_1)' - supportOfOutcomeY*sum(h_00_0)';
sum(sum(h_11_1))-1;
sum(sum(h_11_0))-1;
sum(sum(h_00_1))-1;
sum(sum(h_00_0))-1;
sum(sum(h_10))-1;
sum(sum(h_01))-1];
maximumViolationMin = max(violation);
%%
%Get analytic bounds
BoundsAnalytic = calculateanalyticboundsonate(dataset,sizeGrid,supportOfOutcomeY,lowerBoundForDefiers,upperBoundForDefiers);


%Create linear program.
[LinearProgramme] = createlinearprogram(probDistribution,supportOfOutcomeY,pi01);

%Finally, compare the results.

%Check if the constructed h-functions satisfy constraints of the linear
%programme.
disp(sprintf(['...Min ATE Nevertakers and Max ATE Alwaystakers...']));
disp(sprintf(['Maximum constraint violation for h-s that maximizes ATE for Nevertakers is ', num2str(maximumViolationMax)]));
inequalitiesNotSatisfied = sum(LinearProgramme.Aineq*jointDistributionMinAteNevertakersVector' - LinearProgramme.bineq'>10e-4);
disp(sprintf(['Number of LP inequality constraints that are not met: ', num2str(inequalitiesNotSatisfied)]));
equalitiesNotSatisfied = max(LinearProgramme.Aeq*jointDistributionMinAteNevertakersVector' - LinearProgramme.beq);
disp(sprintf(['Maximum constraint violation in LP: ', num2str(equalitiesNotSatisfied)]));

%Min ATE for T=11
LinearProgramme.f = LinearProgramme.fForAlwaystakers;
[~,b,~] = linprog(LinearProgramme);
resultsCompared = [LinearProgramme.fForAlwaystakers*jointDistributionMaxAteNevertakersVector', b,...
 BoundsAnalytic.Alwaystakers.lowerBound];
disp(sprintf(['Min ATE for T=11: \n[Constructed, LP, Analytic] = [', num2str(resultsCompared),']']));

%Max ATE for T=00
LinearProgramme.f = -LinearProgramme.fForNevertakers;
[~,b,~] = linprog(LinearProgramme);
resultsCompared = [LinearProgramme.fForNevertakers*jointDistributionMaxAteNevertakersVector', -b,...
 BoundsAnalytic.Nevertakers.upperBound];
disp(sprintf(['Max ATE for T=00: \n[Constructed, LP, Analytic] = [', num2str(resultsCompared),']']));

disp(sprintf(['\n']));

disp(sprintf(['...Max ATE Nevertakers and Min ATE Alwaystakers...']));
disp(sprintf(['Maximum constraint violation for h-s that minimizes ATE for Nevertakers is ', num2str(maximumViolationMin)]));
inequalitiesNotSatisfied = sum(LinearProgramme.Aineq*jointDistributionMaxAteNevertakersVector' - LinearProgramme.bineq'>10e-4);
disp(sprintf(['Number of LP inequality constraints that are not met: ', num2str(inequalitiesNotSatisfied)]));
equalitiesNotSatisfied = max(LinearProgramme.Aeq*jointDistributionMaxAteNevertakersVector' - LinearProgramme.beq);
disp(sprintf(['Maximum constraint violation in LP: ', num2str(equalitiesNotSatisfied)]));

%Min ATE for T=00
LinearProgramme.f = LinearProgramme.fForNevertakers;
[~,b,~] = linprog(LinearProgramme); 
resultsCompared = [LinearProgramme.fForNevertakers*jointDistributionMinAteNevertakersVector', b,...
 BoundsAnalytic.Nevertakers.lowerBound];
disp(sprintf(['Min ATE for T=00: \n[Constructed, LP, Analytic] = [', num2str(resultsCompared),']']));

%Max ATE for T=11
LinearProgramme.f = -LinearProgramme.fForAlwaystakers;
[~,b,~] = linprog(LinearProgramme);
resultsCompared = [LinearProgramme.fForAlwaystakers*jointDistributionMinAteNevertakersVector', -b,...
 BoundsAnalytic.Alwaystakers.upperBound];
disp(sprintf(['Max ATE for T=11: \n[Constructed, LP, Analytic] = [', num2str(resultsCompared),']']));

disp(sprintf(['\n************Completed************']));
