function [LinearProgramme] = createlinearprogram(probDistribution,supportOfOutcomeY,pi01,assumption4)

%This function creates a linear programme for a given pi01
%share of defiers
%INPUT:  probDistribution - probability distribution of (outcome,treatment,instrument)
%        pi01 - proportion of defiers
%        supportOfOutcomeY - row vector
%        assumption4 - optional argument, is equal to 1 if assumption 4
%        holds (mean dominance).
%OUTPUT: Bounds - structure with bounds on ATE for various subpopulations

%Support of the variables. This is important to realize but not necessary
%for the code.
%supportOfTreatmentD  = [1 0];             %Treatment.
%supportOfInstrumentZ = [1 0];             %Instrument.
%supportOfTypeT       = [11 10 01 00];     %Type {always taker,complier,defier,never taker}.

if nargin<4
    assumption4 = 0;
end


%Size of the support of variables.
sizeOfSupportOfY = size(supportOfOutcomeY,2);  %Outcome.
sizeOfSupportOfD = 2;                          %Treatment.
sizeOfSupportOfZ = 2;                          %Instrument.
sizeOfSupportOfT = 4;                          %Type.

%Probability of observed variables (Y,D,Z) is probDistribution.
PDZ = sum(probDistribution); %Probability of (D,Z).
PZ  = sum(PDZ);         %Probability of Z.
P11 = PDZ(1,1,1)/PZ(1); %Probability of D=1 given Z=1.
P10 = PDZ(1,1,2)/PZ(2); %Probability of D=1 given Z=0.
P01 = PDZ(1,2,1)/PZ(1); %Probability of D=0 given Z=1.
P00 = PDZ(1,2,2)/PZ(2); %Probability of D=0 given Z=0.

%we will search in the space of joint prob distributions of (Y(1),Y(0))
%given T and Z. these prob distributions are denoted as Phi
sizeOfSupportOfYYTZ = sizeOfSupportOfY*sizeOfSupportOfY*sizeOfSupportOfT*sizeOfSupportOfZ;

%_________________________________________________________________
%Phi satisfies assumption A2i (sizeOfSupportOfT*sizeOfSupportOfD
%equalities)
%matrixAssumptionA2i?
matrixAssumptionA2i = zeros(sizeOfSupportOfT*sizeOfSupportOfD,sizeOfSupportOfYYTZ);
vectorAssumptionA2i = zeros(sizeOfSupportOfT*sizeOfSupportOfD,1);

for t = 1:sizeOfSupportOfT
    %equation (8)
    matrixHelp                    = zeros(sizeOfSupportOfY,sizeOfSupportOfY,sizeOfSupportOfT,sizeOfSupportOfZ);
    for y1 = 1:sizeOfSupportOfY
        for y0 = 1:sizeOfSupportOfY
            matrixHelp(y1,y0,t,1) = supportOfOutcomeY(y1);
            matrixHelp(y1,y0,t,2) = -supportOfOutcomeY(y1);
        end
    end
    matrixAssumptionA2i(t,:) = reshape(matrixHelp,1,sizeOfSupportOfYYTZ);

    %equation (9)
    matrixHelp                    = zeros(sizeOfSupportOfY,sizeOfSupportOfY,sizeOfSupportOfT,sizeOfSupportOfZ);
    for y1 = 1:sizeOfSupportOfY
        for y0 = 1:sizeOfSupportOfY
            matrixHelp(y1,y0,t,1) = supportOfOutcomeY(y0);
            matrixHelp(y1,y0,t,2) = -supportOfOutcomeY(y0);
        end
    end
    matrixAssumptionA2i(t+sizeOfSupportOfT,:) = reshape(matrixHelp,1,sizeOfSupportOfYYTZ);
end

%________________________________________________________________________
%Phi is compatible with the Data and with the assumption A2ii
%matrixDataAssumptionA2ii
matrixDataAssumptionA2ii = zeros(sizeOfSupportOfY*sizeOfSupportOfD*sizeOfSupportOfZ,sizeOfSupportOfYYTZ);
vectorDataAssumptionA2ii = zeros(sizeOfSupportOfY*sizeOfSupportOfD*sizeOfSupportOfZ,1);

for y = 1:sizeOfSupportOfY
    %equation (10)
    matrixHelp          = zeros(sizeOfSupportOfY,sizeOfSupportOfY,sizeOfSupportOfT,sizeOfSupportOfZ);
    matrixHelp(y,:,1,1) = (P10 - pi01)*ones(sizeOfSupportOfY,1);              
    matrixHelp(y,:,2,1) = (P11 - P10 + pi01)*ones(sizeOfSupportOfY,1);   
    matrixDataAssumptionA2ii((y-1)*4+1,:) = reshape(matrixHelp,1,sizeOfSupportOfYYTZ);
    vectorDataAssumptionA2ii((y-1)*4+1,:) = probDistribution(y,1,1)/PZ(1);

    %equation (11)
    matrixHelp          = zeros(sizeOfSupportOfY,sizeOfSupportOfY,sizeOfSupportOfT,sizeOfSupportOfZ);
    matrixHelp(y,:,1,2) = (P10 - pi01)*ones(sizeOfSupportOfY,1);              
    matrixHelp(y,:,3,2) = (pi01)*ones(sizeOfSupportOfY,1);   
    matrixDataAssumptionA2ii((y-1)*4+2,:) = reshape(matrixHelp,1,sizeOfSupportOfYYTZ);
    vectorDataAssumptionA2ii((y-1)*4+2,:) = probDistribution(y,1,2)/PZ(2);

    %equation (12)
    matrixHelp          = zeros(sizeOfSupportOfY,sizeOfSupportOfY,sizeOfSupportOfT,sizeOfSupportOfZ);
    matrixHelp(:,y,3,1) = (pi01)*ones(sizeOfSupportOfY,1);              
    matrixHelp(:,y,4,1) = (P01 - pi01)*ones(sizeOfSupportOfY,1);   
    matrixDataAssumptionA2ii((y-1)*4+3,:) = reshape(matrixHelp,1,sizeOfSupportOfYYTZ);
    vectorDataAssumptionA2ii((y-1)*4+3,:) = probDistribution(y,2,1)/PZ(1);

    %equation (13)
    matrixHelp          = zeros(sizeOfSupportOfY,sizeOfSupportOfY,sizeOfSupportOfT,sizeOfSupportOfZ);
    matrixHelp(:,y,2,2) = (P11 - P10 + pi01)*ones(sizeOfSupportOfY,1);              
    matrixHelp(:,y,4,2) = (P01 - pi01)*ones(sizeOfSupportOfY,1);   
    matrixDataAssumptionA2ii((y-1)*4+4,:) = reshape(matrixHelp,1,sizeOfSupportOfYYTZ);
    vectorDataAssumptionA2ii((y-1)*4+4,:) = probDistribution(y,2,2)/PZ(2);
end

if assumption4
    %________________________________________________________________________
    %Phi is compatible with the assumption A4
    %matrixAssumptionA4
    matrixAssumptionA4 = zeros(sizeOfSupportOfD*2,sizeOfSupportOfYYTZ);
    vectorAssumptionA4 = zeros(sizeOfSupportOfD*2,1);

    matrixHelp  = zeros(sizeOfSupportOfY,sizeOfSupportOfY,sizeOfSupportOfT,sizeOfSupportOfZ);
    for y1 = 1:sizeOfSupportOfY
        for y0 = 1:sizeOfSupportOfY    
            %for d=0 t=11
            matrixHelp(y1,y0,2,1) = supportOfOutcomeY(y0);
            matrixHelp(y1,y0,1,1) = -supportOfOutcomeY(y0);
        end
    end
    matrixAssumptionA4(1,:) = reshape(matrixHelp,1,sizeOfSupportOfYYTZ);

    matrixHelp  = zeros(sizeOfSupportOfY,sizeOfSupportOfY,sizeOfSupportOfT,sizeOfSupportOfZ);
    for y1 = 1:sizeOfSupportOfY
        for y0 = 1:sizeOfSupportOfY
            %for d=0 t=00
            matrixHelp(y1,y0,2,1) = supportOfOutcomeY(y0);
            matrixHelp(y1,y0,4,1) = -supportOfOutcomeY(y0);
            end
    end
    matrixAssumptionA4(2,:) = reshape(matrixHelp,1,sizeOfSupportOfYYTZ);

    matrixHelp  = zeros(sizeOfSupportOfY,sizeOfSupportOfY,sizeOfSupportOfT,sizeOfSupportOfZ);
    for y1 = 1:sizeOfSupportOfY
        for y0 = 1:sizeOfSupportOfY
            %for d=1 t=11
            matrixHelp(y1,y0,2,1) = supportOfOutcomeY(y1);
            matrixHelp(y1,y0,1,1) = -supportOfOutcomeY(y1);
        end
    end
    matrixAssumptionA4(3,:) = reshape(matrixHelp,1,sizeOfSupportOfYYTZ);

    matrixHelp  = zeros(sizeOfSupportOfY,sizeOfSupportOfY,sizeOfSupportOfT,sizeOfSupportOfZ);
    for y1 = 1:sizeOfSupportOfY
        for y0 = 1:sizeOfSupportOfY
            %for d=1 t=00
            matrixHelp(y1,y0,2,1) = supportOfOutcomeY(y1);
            matrixHelp(y1,y0,4,1) = -supportOfOutcomeY(y1);
        end
    end    
    matrixAssumptionA4(4,:) = reshape(matrixHelp,1,sizeOfSupportOfYYTZ);   
end %end of assumption4

%________________________________________________________________________
%Phi sums up to 1
%matrixProperDistribution
matrixProperDistribution   = zeros(sizeOfSupportOfT*sizeOfSupportOfZ,sizeOfSupportOfYYTZ);
vectorProperDistribution   = ones(sizeOfSupportOfT*sizeOfSupportOfZ,1);

for z = 1:sizeOfSupportOfZ
    for t = 1:sizeOfSupportOfT
        matrixHelp                       = zeros(sizeOfSupportOfY,sizeOfSupportOfY,sizeOfSupportOfT,sizeOfSupportOfZ);
        matrixHelp(:,:,t,z)              = ones(sizeOfSupportOfY,sizeOfSupportOfY);
        matrixProperDistribution((z-1)*sizeOfSupportOfT+t,:) = reshape(matrixHelp,1,sizeOfSupportOfYYTZ);
    end
end

%________________________________________________________________________
%Phi is nonnegative
%matrixNonNegativity
matrixNonNegativity = -1*eye(sizeOfSupportOfYYTZ);
vectorNonNegativity = zeros(1,sizeOfSupportOfYYTZ);

%________________________________________________________________________
%define matrices that define all linear equalities and inequalities
%matrixOfEqualities * Phi = vectorOfEqualities
matrixOfEqualities = [matrixAssumptionA2i; matrixDataAssumptionA2ii; matrixProperDistribution];
vectorOfEqualities = [vectorAssumptionA2i; vectorDataAssumptionA2ii; vectorProperDistribution];

%matrixOfInequalities * Phi <= vectorOfInequalities
matrixOfInequalities = [matrixNonNegativity];
vectorOfInequalities = [vectorNonNegativity];

if assumption4
    matrixOfInequalities = [matrixOfInequalities; -matrixAssumptionA4];
    vectorOfInequalities = [vectorNonNegativity; vectorAssumptionA4];
end


%________________________________________________________________________
%Define f vectors that we will optimize
%ATE for entire population
matrixHelp          = zeros(sizeOfSupportOfY,sizeOfSupportOfY,sizeOfSupportOfT,sizeOfSupportOfZ);
for y1 = 1:sizeOfSupportOfY
    for y0 = 1:sizeOfSupportOfY
        matrixHelp(y1,y0,1,1) = (P10 - pi01)*(supportOfOutcomeY(y1)-supportOfOutcomeY(y0))*PZ(1);
        matrixHelp(y1,y0,1,2) = (P10 - pi01)*(supportOfOutcomeY(y1)-supportOfOutcomeY(y0))*PZ(2);
        matrixHelp(y1,y0,2,1) = (P11 - P10 + pi01)*(supportOfOutcomeY(y1)-supportOfOutcomeY(y0))*PZ(1);
        matrixHelp(y1,y0,2,2) = (P11 - P10 + pi01)*(supportOfOutcomeY(y1)-supportOfOutcomeY(y0))*PZ(2);
        matrixHelp(y1,y0,3,1) = (pi01)*(supportOfOutcomeY(y1)-supportOfOutcomeY(y0))*PZ(1);
        matrixHelp(y1,y0,3,2) = (pi01)*(supportOfOutcomeY(y1)-supportOfOutcomeY(y0))*PZ(2);
        matrixHelp(y1,y0,4,1) = (P01 - pi01)*(supportOfOutcomeY(y1)-supportOfOutcomeY(y0))*PZ(1);
        matrixHelp(y1,y0,4,2) = (P01 - pi01)*(supportOfOutcomeY(y1)-supportOfOutcomeY(y0))*PZ(2);
    end
end
fForEntirePopulation = reshape(matrixHelp,1,sizeOfSupportOfYYTZ);

%ATE for alwaystakers
matrixHelp          = zeros(sizeOfSupportOfY,sizeOfSupportOfY,sizeOfSupportOfT,sizeOfSupportOfZ);
for y1 = 1:sizeOfSupportOfY
    for y0 = 1:sizeOfSupportOfY
        matrixHelp(y1,y0,1,1) = (supportOfOutcomeY(y1)-supportOfOutcomeY(y0))*PZ(1);
        matrixHelp(y1,y0,1,2) = (supportOfOutcomeY(y1)-supportOfOutcomeY(y0))*PZ(2);
    end
end
fForAlwaystakers = reshape(matrixHelp,1,sizeOfSupportOfYYTZ);

%ATE for compliers
matrixHelp          = zeros(sizeOfSupportOfY,sizeOfSupportOfY,sizeOfSupportOfT,sizeOfSupportOfZ);
for y1 = 1:sizeOfSupportOfY
    for y0 = 1:sizeOfSupportOfY
        matrixHelp(y1,y0,2,1) = (supportOfOutcomeY(y1)-supportOfOutcomeY(y0))*PZ(1);
        matrixHelp(y1,y0,2,2) = (supportOfOutcomeY(y1)-supportOfOutcomeY(y0))*PZ(2);
    end
end
fForCompliers = reshape(matrixHelp,1,sizeOfSupportOfYYTZ);

%ATE for nevertakers
matrixHelp          = zeros(sizeOfSupportOfY,sizeOfSupportOfY,sizeOfSupportOfT,sizeOfSupportOfZ);
for y1 = 1:sizeOfSupportOfY
    for y0 = 1:sizeOfSupportOfY
        matrixHelp(y1,y0,4,1) = (supportOfOutcomeY(y1)-supportOfOutcomeY(y0))*PZ(1);
        matrixHelp(y1,y0,4,2) = (supportOfOutcomeY(y1)-supportOfOutcomeY(y0))*PZ(2);
    end
end
fForNevertakers = reshape(matrixHelp,1,sizeOfSupportOfYYTZ);

%ATE for treated population
matrixHelp          = zeros(sizeOfSupportOfY,sizeOfSupportOfY,sizeOfSupportOfT,sizeOfSupportOfZ);
denom = (P10 - pi) + (P11 - P10 + pi)*PZ(1) + (pi)*PZ(2);
for y1 = 1:sizeOfSupportOfY
    for y0 = 1:sizeOfSupportOfY
        matrixHelp(y1,y0,1,1) = (P10 - pi01)*(supportOfOutcomeY(y1)-supportOfOutcomeY(y0))*PZ(1)/denom;
        matrixHelp(y1,y0,1,2) = (P10 - pi01)*(supportOfOutcomeY(y1)-supportOfOutcomeY(y0))*PZ(2)/denom;
        matrixHelp(y1,y0,2,1) = (P11 - P10 + pi01)*(supportOfOutcomeY(y1)-supportOfOutcomeY(y0))*PZ(1)/denom;
        matrixHelp(y1,y0,3,2) = (pi01)*(supportOfOutcomeY(y1)-supportOfOutcomeY(y0))*PZ(2)/denom;
    end
end
fForTreatedPopulation = reshape(matrixHelp,1,sizeOfSupportOfYYTZ);

%________________________________________________________________________
%Create a structure called LinearProgramme.
options = optimset('linprog');
options.LargeScale = 'off';
options.Algorithm = 'simplex';
options.Simplex = 'on';
options.Display = 'off';
LinearProgramme = struct;
LinearProgramme.Aineq = matrixOfInequalities;
LinearProgramme.bineq = vectorOfInequalities;
LinearProgramme.Aeq = matrixOfEqualities;
LinearProgramme.beq = vectorOfEqualities;
LinearProgramme.lb = [];
LinearProgramme.ub = [];
LinearProgramme.x0 = [];
LinearProgramme.solver = 'linprog';
LinearProgramme.options = options;
LinearProgramme.fForEntirePopulation = fForEntirePopulation;
LinearProgramme.fForAlwaystakers = fForAlwaystakers;
LinearProgramme.fForCompliers = fForCompliers;
LinearProgramme.fForNevertakers = fForNevertakers;
LinearProgramme.fForTreatedPopulation = fForTreatedPopulation;

%Save the linear programme.