%Notes to readers and practitioners:

%The MATLAB code that follows implements the estimation algorithm described
%in the Journal of Applied Econometrics paper entitled:
%"Bayes Estimates of Distance to Market: Transactions Costs, Cooperatives
%and Milk Market Development in the Ethiopian Highlands,"
%which is co-authored with Simeon Ehui and Amare Teklu.

%The code was written November 30, 2004.
%Annotations, designed to facilitate second-use by readers were added today
%September 12, 2006.

%Readers interested in the paper, the data, the primary-data collection
%survey instruments and, possibly, the broader objectives of the research
%project within which the data were obtained, are invited to contact me at:

%Garth Holloway
%Department of Agricultural and Food Economics,
%University of Reading,
%4 Earley Gate,
%Whiteknights Road,
%PO Box 237,
%Reading
%Berkshire
%RG66AR
%United Kingdom
%phone: (44) (118) 378 6775
%facsimile (44) (118) 975 6567
%email g.holloway@cgiar.org;

%Finally, the data used in the article are a subset of quantities sampled
%from the households, including demographic and spatial characteristcs,
%farm and off-farm employment, animal and human consumption and animal
%health status of dairy and non-dairy anmimals.  These data have been made
%available by the International Livestock Research Institute, Addis Ababa,
%and are available from the author upon request.  Readers wishing to use
%the data in their own research are kindly requested to acknowledge the
%International Livestock Reserarch Institute.  Thank you.

%The reminder of this program describes and implements the estimation
%algorithm layed out in the body of the journal article and described in
%detail in the accompanying appendix.  Readers having difficulty
%interpreting code, or its representation in the text of the article, are
%encouraged to contact me.

%Garth Holloway
%September 12, 2006.

%Setup
clear
format bank
format compact
%pause

%Time
tic
%pause

%Fetch
% load distancetomarketdata data
data=load('distancetomarketdata.ascii');
production=data(:,2);
sales=data(:,12);
crossbreed=data(:,86);
localbreed=data(:,87);
extension=data(:,90);
distance=data(:,20);
education=data(:,24);
ilukura=data(:,32);
mirti=data(:,33);
%pause

%Define
a=[sales production];
b=[crossbreed localbreed extension distance education ilukura mirti];
n=68;
for i=1:n
    t(i,:)=3;
    cst(i,:)=sum(t);
    h(cst(i,:)-t(i,:)+1:cst(i,:),:)=ones(t(i,:),1)*i;
    w(cst(i,:)-t(i,:)+1:cst(i,:),i)=ones(t(i,:),1);
    y(cst(i,:)-t(i,:)+1:cst(i,:),:)=a([68 2*68 3*68]-68+i,[1 2]);
    c(cst(i,:)-t(i,:)+1:cst(i,:),:)=b([68 2*68 3*68]-68+i,:);
    x(cst(i,:)-t(i,:)+1:cst(i,:),:)=c(cst(i,:)-t(i,:)+1:cst(i,:),[1 2 3 4 5 6 7]);
    r(i,:)=c(3*(i-1)+1,[6 7]);
    i
end
[s k]=size(x);
f=c(:,[1 2 3 4 5]);
%pause

%Size
[s k]=size(x);
[s n]=size(w);
[s m]=size(y);
[s l]=size(f);
i=find(y(:,1)==0);
j=find(y(:,1)~=0);
ni=length(i);
nj=length(j);
%pause

%Design
v=reshape(y,s*m,1);
p=[zeros(s,n);w];
q=[x zeros(s,l);zeros(s,k) f];
%pause

%Schedule
burnin=5000;
gibbs=10000;
%pause

%Start
z=y;
bs=0;
g=0;
beta=inv(q'*q)*q'*v;
gamma=inv(p'*p)*p'*v;
delta=inv(r'*r)*r'*gamma;
omega=sqrt((gamma-r*delta)'*(gamma-r*delta)/n);
mu=reshape(p*gamma+q*beta,s,m);
tau=0;
%pause

%Simulate
for h=1:burnin+gibbs

    %Inverted-Wishart
    u=mvnrnd(zeros(m,1),inv((z-mu)'*(z-mu)),s)';
    sigma=inv(u*u');
    %pause


    %Multivariate-Normal
    a=v-p*gamma;
    b=q;
    c=kron(inv(sigma),eye(s));
    betacov=inv(b'*c*b);
    betahat=betacov*(b'*c*a);
    beta=mvnrnd(betahat,betacov,1)';
    %pause

    %Truncated-Normal
    mu=reshape(p*gamma+q*beta,s,m);
    mz=mu(i,1)+(z(i,2)-mu(i,2))*inv(sigma(2,2))*sigma(2,1);
    sz=sqrt(sigma(1,1)-sigma(1,2)*inv(sigma(2,2))*sigma(2,1));
    z(i,1)=norminv(unifrnd(0,1,ni,1).*normcdf(ones(ni,1)*tau,mz,sz),mz,sz);
    %pause

    %Uniform
    tau=unifrnd(max(z(i,1)),min(z(j,1)));
    %pause

    %Truncated-Normal
    a=v-q*beta;
    b=p;
    c=kron(inv(sigma),eye(s));
    d=r*delta;
    e=omega^-2*eye(n);
    gammacov=inv(b'*c*b+e);
    gammahat=gammacov*(b'*c*a+e*d);
    mg=gammahat;
    sg=sqrt(unique(diag(gammacov)));
    gamma=norminv(unifrnd(0,1,n,1).*normcdf(0,mg,sg),mg,sg);
    %pause

    %Multivariate Normal
    delta=mvnrnd(inv(r'*r)*r'*gamma,omega^2*inv(r'*r),1)';
    %pause

    %Scaled-Inverse Chi-Squared
    omega=sqrt((gamma-r*delta)'*(gamma-r*delta)/sum(normrnd(0,1,n,1).^2));
    %pause

    %Collect
    bs=bs+1;
    if h>burnin
        bs=burnin;
        g=g+1;

        %Store
        sigmas(g,:)=reshape(sigma,m*m,1)';
        betas(g,:)=beta';
        distances(g,:)=z(i,1)';
        taus(g,:)=tau';
        gammas(g,:)=gamma';
        omegas(g,:)=omega';
        deltas(g,:)=delta';

        %Likelihood
        yj=y(j,:);
        mu=reshape(p*gamma+q*beta,s,m);
        muj=mu(j,:);
        logunc=sum(-.5*log(det(sigma))-.5*m*log(2*pi)-.5*diag((yj-muj)*inv(sigma)*(yj-muj)'));
        m12i=mu(i,1)+(y(i,2)-mu(i,2))*inv(sigma(2,2))*sigma(2,1);
        s12i=sqrt(sigma(1,1)-sigma(1,2)*inv(sigma(2,2))*sigma(2,1));
        y2i=y(i,2);
        m2i=mu(i,2);
        s2i=sqrt(sigma(2,2));
        logcens=sum(log(normcdf(tau,m12i,s12i))+log(normpdf(y2i,m2i,s2i)));
        loglikes(g,:)=logunc+logcens;
        if g==gibbs
            [maxlogl gstar]=max(loglikes);
            sigmastar=reshape(sigmas(gstar,:)',m,m);
            betastar=betas(gstar,:)';
            taustar=taus(gstar,:);
            gammastar=gammas(gstar,:)';
            omegastar=omegas(gstar,:)';
            deltastar=deltas(gstar,:)';
        end
    end

    %Track
    where=[bs g]
end
%pause

%Plot
figure(1)
plot(sigmas)
title('sigmas')
axis tight
figure(2)
plot(betas)
title('betas')
axis tight
figure(3)
plot(distances)
title('distances')
axis tight
figure(4)
plot(taus)
title('censoring point')
axis tight
figure(5)
plot(gammas)
title('gammas')
axis tight
figure(6)
plot(omegas)
title('omegas')
axis tight
figure(7)
plot(deltas)
title('deltas')
axis tight
%pause

%Look
sigmahat=[prctile(sigmas,5)' mean(sigmas)' prctile(sigmas,95)']
betahat=[prctile(betas,5)' mean(betas)' prctile(betas,95)']
tauhat=[prctile(taus,5)' mean(taus)' prctile(taus,95)']
gammahat=[prctile(gammas,5)' mean(gammas)' prctile(gammas,95)']
omegahat=[prctile(omegas,5)' mean(omegas)' prctile(omegas,95)']
deltahat=[prctile(deltas,5)' mean(deltas)' prctile(deltas,95)']
muhat=p*gammahat(:,2)+q*betahat(:,2);
yhat=reshape(muhat,s,m);
corrun=corrcoef([y(j,:) yhat(j,:)]);
rsquaredun=[corrun(1,3) corrun(2,4)].^2
pospredun=[length(find(yhat(j,1)>mean(taus))) length(find(yhat(j,2)>0))]
negpredun=[length(find(yhat(j,1)<=mean(taus))) length(find(yhat(j,2)<=0))]
corrcen=corrcoef([mean(distances)' y(i,2) yhat(i,:)]);
rsquaredcen=[corrcen(1,3) corrcen(2,4)].^2
pospredcen=[length(find(yhat(i,1)>mean(taus))) length(find(yhat(i,2)>0))]
negpredcen=[length(find(yhat(i,1)<=mean(taus))) length(find(yhat(i,2)<=0))]
maxlogl
%pause

%Look
tsigmahat=[ mean(sigmas)' mean(sigmas)'./std(sigmas)']
tbetahat=[mean(betas)' mean(betas)'./std(betas)']
ttauhat=[mean(taus) mean(taus)'./std(taus)']
tgammahat=[mean(gammas)' mean(gammas)'./std(gammas)'];
tomegahat=[mean(omegas)' mean(omegas)'./std(omegas)']
tdeltahat=[mean(deltas)' mean(deltas)'./std(deltas)']
maxlogl
cond(x)
cond(r)
%pause

%Time
time=toc
timeinhours=toc/60/60
projected10=timeinhours*10
%pause

%Store
save bayesdistancetomarketdata
