%  Brent R. Hickman and Timothy P. Hubbard
% hickmanbr@uchicago.edu and timothy.hubbard@colby.edu

% please cite our Journal of Applied Econometrics paper "Replacing Sample
% Trimming with Boundary Correction in Nonparametric Estimation of
% First-Price Auctions" if this code is helpful to you.

% compute two dimensional kernel density given data (required) and points
% to evaluate densities at (optional).  the kernel choice is also optional
% with the default being the gaussian kernel

function [fhat,hx,hy] = kspdf2dCPV(x,y,zx,zy,kernel)

Tx = length(x);
Ty = length(y);
if Tx ~= Ty
    fprintf('Error - lengths of data vectors differ\n')
end

% if points to evaluate kernels at or kernel not specified use defaults
if nargin == 2
    npts = 100;
    zx = linspace(min(x),max(x),npts);
    zy = linspace(min(y),max(y),npts);
    kernel = 'gaussian';
elseif nargin == 4
    kernel = 'gaussian';
end
nx = length(zx);
ny = length(zy);

% use Hardle's (1991) bandwidth transformation constant
c = 1;
if (isequal(kernel,'gaussian') == 1)
    c = 1;
elseif (isequal(kernel,'epanechnikov') == 1)
    c = 2.214;
elseif (isequal(kernel,'triweight') == 1)
    c = 2.978;
elseif (isequal(kernel,'uniform') == 1)
    c = 1.740;
elseif (isequal(kernel,'triangle') == 1)
    c = 2.432;
elseif (isequal(kernel,'cosinus') == 1)
    c = 2.288;
elseif (isequal(kernel,'quartic') == 1)
    c = 2.623;
end

% use Silverman's (1986) optimal bandwidth
hy = c*(1.06)*std(y)*Ty^(-1/6);
hx = hy;

% get matrix for first item in data
for i=1:nx
    u = (x - zx(i))./hx;
    if (isequal(kernel,'gaussian') == 1)
        kx(i,:) = 1/sqrt(2*pi)*exp(-(u.^2)/2);
    elseif (isequal(kernel,'epanechnikov') == 1)
        kx(i,:) = 3/4*(1 - u.^2);
        kx(i,find(abs(u) > 1)) = 0;
    elseif (isequal(kernel,'triweight') == 1)
        kx(i,:) = 35/32*(1 - u.^2).^3;
        kx(i,find(abs(u) > 1)) = 0;
    elseif (isequal(kernel,'uniform') == 1)
        kx(i,:) = ones(size(u))*1/2;
        kx(i,find(abs(u) > 1)) = 0;
    elseif (isequal(kernel,'triangle') == 1)
        kx(i,:) = 1 - abs(u);
        kx(i,find(abs(u) > 1)) = 0;
    elseif (isequal(kernel,'cosinus') == 1)
        kx(i,:) = pi/4*cos(pi/2*u);
        kx(i,find(abs(u) > 1)) = 0;
    elseif (isequal(kernel,'quartic') == 1)
        kx(i,:) = 15/16*(1 - u.^2).^2;
        kx(i,find(abs(u) > 1)) = 0;
    else
        fprintf('Kernel choice not defined or spelt wrong\n')
        break        
    end
end

% get matrix for second item in data
for i=1:ny
    v = (y - zy(i))./hy;
    if (isequal(kernel,'gaussian') == 1)
        ky(i,:) = 1/sqrt(2*pi)*exp(-(v.^2)/2);
    elseif (isequal(kernel,'epanechnikov') == 1)
        ky(i,:) = 3/4*(1 - v.^2);
        ky(i,find(abs(v) > 1)) = 0;
    elseif (isequal(kernel,'triweight') == 1)
        ky(i,:) = 35/32*(1 - v.^2).^3;
        ky(i,find(abs(v) > 1)) = 0;
    elseif (isequal(kernel,'uniform') == 1)
        ky(i,:) = ones(size(v))*1/2;
        ky(i,find(abs(v) > 1)) = 0;
    elseif (isequal(kernel,'triangle') == 1)
        ky(i,:) = 1 - abs(v);
        ky(i,find(abs(v) > 1)) = 0;
    elseif (isequal(kernel,'cosinus') == 1)
        ky(i,:) = pi/4*cos(pi/2*v);
        ky(i,find(abs(v) > 1)) = 0;
    elseif (isequal(kernel,'quartic') == 1)
        ky(i,:) = 15/16*(1 - v.^2).^2;
        ky(i,find(abs(v) > 1)) = 0;
    end
end

kprod = kx.*ky;
kprod = kprod';
fhat = sum(kprod)./(Tx*hx*hy);