%  Brent R. Hickman and Timothy P. Hubbard
% hickmanbr@uchicago.edu and timothy.hubbard@colby.edu

% please cite our Journal of Applied Econometrics paper "Replacing Sample
% Trimming with Boundary Correction in Nonparametric Estimation of
% First-Price Auctions" if this code is helpful to you.

% compute two dimensional kernel cdf given data (required) and points to
% evaluate edf at (optional).  the kernel choice is also optional with the
% default being the gaussian kernel 

function [Fhat,hy] = kscdf2dCPV(x,y,zx,zy,kernel,fixnpts)

Tx = length(x);
Ty = length(y);
if Tx ~= Ty
    fprintf('Error - lengths of data vectors differ\n')
end

% if points to evaluate kernels at or kernel not specified use defaults
if nargin == 2
    npts = 100;
    zx = linspace(min(x),max(x),npts);
    zy = linspace(min(y),max(y),npts);
    kernel = 'gaussian';
elseif nargin == 4
    kernel = 'gaussian';
end
nx = length(zx);
ny = length(zy);

if (exist('fixnpts','var'))
    Tx = fixnpts;
end

% use Hardle's (1991) bandwidth transformation constant
c = 1;
if (isequal(kernel,'gaussian') == 1)
    c = 1;
elseif (isequal(kernel,'epanechnikov') == 1)
    c = 2.214;
elseif (isequal(kernel,'triweight') == 1)
    c = 2.978;
elseif (isequal(kernel,'uniform') == 1)
    c = 1.740;
elseif (isequal(kernel,'triangle') == 1)
    c = 2.432;
elseif (isequal(kernel,'cosinus') == 1)
    c = 2.288;
elseif (isequal(kernel,'quartic') == 1)
    c = 2.623;
end

% use Silverman's (1986) optimal bandwidth
hy = c*(4/3)^(1/5)*std(y)*Ty^(-1/5);

% get matrix for first item in data
for i=1:nx
    indx(i,:) = (x <= zx(i));
end

% get matrix for second item in data
for i=1:ny
    v = (y - zy(i))./hy;
    if (isequal(kernel,'gaussian') == 1)
        ky(i,:) = 1/sqrt(2*pi)*exp(-(v.^2)/2);
    elseif (isequal(kernel,'epanechnikov') == 1)
        ky(i,:) = 3/4*(1 - v.^2);
        ky(i,find(abs(v) > 1)) = 0;
    elseif (isequal(kernel,'triweight') == 1)
        ky(i,:) = 35/32*(1 - v.^2).^3;
        ky(i,find(abs(v) > 1)) = 0;
    elseif (isequal(kernel,'uniform') == 1)
        ky(i,:) = ones(size(v))*1/2;
        ky(i,find(abs(v) > 1)) = 0;
    elseif (isequal(kernel,'triangle') == 1)
        ky(i,:) = 1 - abs(v);
        ky(i,find(abs(v) > 1)) = 0;
    elseif (isequal(kernel,'cosinus') == 1)
        ky(i,:) = pi/4*cos(pi/2*v);
        ky(i,find(abs(v) > 1)) = 0;
    elseif (isequal(kernel,'quartic') == 1)
        ky(i,:) = 15/16*(1 - v.^2).^2;
        ky(i,find(abs(v) > 1)) = 0;
    end
end

% add up product of kernels over data observations
kprod = indx.*ky;
kprod = kprod';
Fhat = sum(kprod)./(Tx*hy);