%  Brent R. Hickman and Timothy P. Hubbard
% hickmanbr@uchicago.edu and timothy.hubbard@colby.edu

% please cite our Journal of Applied Econometrics paper "Replacing Sample
% Trimming with Boundary Correction in Nonparametric Estimation of
% First-Price Auctions" if this code is helpful to you.

% Replication of "Asymmetry in First-Price Auctions" by Campo, Perrigne,
% and Vuong (JAE 2003)

function [jj,ss,evalgridjj,fjj,evalgridss,fss, ...
    jmix,smix,evalgridjmix,fjmix,evalgridsmix, ...
    fsmix] = bcappliedtoCPV2012fcn;

lwidth = 2;
fsize = 14;
set(0,'defaulttextinterpreter','latex','Defaulttextfontsize',fsize);

plotind = 0
logtrans = 1
removebads = 1

load Ocs702.dat

kernel = 'triweight';

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% STAGE 1: create necessary matrices for analysis and assign auction ids
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

NT = length(Ocs702);
T = NT/2;

% create a variable for the auction id
for i=1:NT
    if mod(i,2) == 1
        Ocs702(i,3) = ceil(i/2);
    else
        Ocs702(i,3) = ceil((i - 1)/2);
    end
end

% matrix of all data:
% col 1: bid
% col 2: dummy (1 if joint, 0 if solo)
% col 3: auction id number
% col 4: log transformation of bids
% col 5: max of rival's log-transformed bid
% col 6: pseudo values (constructed below)
% col 7: winning bid indicator
alldata = Ocs702;
idcol = 3;
if logtrans == 1
    % log transformation of data
    alldata(:,4) = log(1 + alldata(:,1));
else
    % no log transformation of data
    alldata(:,4) = alldata(:,1);
end

% obtain max of rivals' bids at each auction---just opponent's bid given
% only one other bidder at auction
append = [];
for i=1:T
    ind = alldata((alldata(:,3) == i),:);
    if size(ind,1) ~= 2
        fprintf('Error in auction ID assignemnt\n')
    end
    append = [append; ind(2,4); ind(1,4)];
end
alldata(:,5) = append;

% create matrices of all solo and all joint bids
jointdata = Ocs702(Ocs702(:,2) == 1,:);
solodata = Ocs702(Ocs702(:,2) == 0,:);

% matrices of auctions based on types---joint vs joint (jj), solo vs solo
% (ss), joint vs solo (js), solo vs joint (sj)
jj = [];
ss = [];
mix = [];
for i=1:T
    ind = alldata((alldata(:,3) == i),:);
    if size(ind,1) ~= 2
        fprintf('Error in auction ID assignemnt\n')
    end
    if ind(1,2) == 1 && ind(2,2) == 1
        jj = [jj; ind];
    elseif ind(1,2) == 0 && ind(2,2) == 0
        ss = [ss; ind];
    elseif (ind(1,2) == 1 && ind(2,2) == 0) || ...
            (ind(1,2) == 0 && ind(2,2) == 1)
        mix = [mix; ind];
    end
end
jmix = mix((mix(:,2) == 1),:);
smix = mix((mix(:,2) == 0),:);


% replicate table 1 in CPV
fprintf('Variable\t#Obs\tMean\tSTD\tMin\t\tMax\t\tWithin STD\n')
fprintf('All\t\t\t%d\t%.2f\t%.2f\t%.2f\t%.2f\t%.2f\n', ...
    length(alldata),mean(alldata(:,1)),std(alldata(:,1)), ...
    min(alldata(:,1)),max(alldata(:,1)),0)
fprintf('Joint\t\t%d\t%.2f\t%.2f\t%.2f\t%.2f\t%.2f\n', ...
    length(jointdata),mean(jointdata(:,1)),std(jointdata(:,1)), ...
    min(jointdata(:,1)),max(jointdata(:,1)),0)
fprintf('Solo\t\t%d\t%.2f\t%.2f\t%.2f\t%.2f\t%.2f\n', ...
    length(solodata),mean(solodata(:,1)),std(solodata(:,1)), ...
    min(solodata(:,1)),max(solodata(:,1)),0)
fprintf('J vs J\t\t%d\t%.2f\t%.2f\t%.2f\t%.2f\t%.2f\n', ...
    length(jj),mean(jj(:,1)),std(jj(:,1)), ...
    min(jj(:,1)),max(jj(:,1)),0)
fprintf('J vs S\t\t%d\t%.2f\t%.2f\t%.2f\t%.2f\t\t%.2f\n', ...
    length(jmix),mean(jmix(:,1)),std(jmix(:,1)), ...
    min(jmix(:,1)),max(jmix(:,1)),0)
fprintf('S vs J\t\t%d\t%.2f\t%.2f\t%.2f\t%.2f\t%.2f\n', ...
    length(smix),mean(smix(:,1)),std(smix(:,1)), ...
    min(smix(:,1)),max(smix(:,1)),0)
fprintf('S vs S\t\t%d\t%.2f\t%.2f\t%.2f\t%.2f\t\t%.2f\n', ...
    length(ss),mean(ss(:,1)),std(ss(:,1)), ...
    min(ss(:,1)),max(ss(:,1)),0)

% some info on auction with id 214
data214 = alldata(alldata(:,idcol) == 214,:);
allwout214ind = 1 - ismember(alldata(:,idcol),data214(:,idcol));
allwout214 = alldata(allwout214ind == 1,:); 
jjwout214ind = 1 - ismember(jj(:,idcol),data214(:,idcol));
jjwout214 = jj(jjwout214ind == 1,:);
fprintf('\n\nSome information on auction with id 214:\n')
fprintf('The higher bid at this auction is %.2f times the lower (rival) bid\n', ...
    data214(1,1)/data214(2,1))
fprintf('The higher bid at this auction is %.2f times the next highest bid from any joint vs joint auction\n', ...
    data214(1,1)/max(jjwout214(:,1)))
fprintf('The higher bid at this auction is %.2f standard deviations from the mean joint vs joint bid (data includes auction 214)\n', ...
    (data214(1,1) - mean(jj(:,1)))/std(jj(:,1)))
fprintf('The higher bid at this auction is %.2f standard deviations from the mean joint vs joint bid (data does not include auction 214)\n', ...
    (data214(1,1) - mean(jjwout214(:,1)))/std(jjwout214(:,1)))
if removebads == 1
    jj = jjwout214;
    alldata = allwout214;
end

% some info on auction with id 219
data219 = alldata(alldata(:,idcol) == 219,:);
allwout219ind = 1 - ismember(alldata(:,idcol),data219(:,idcol));
allwout219 = alldata(allwout219ind == 1,:); 
jjwout219ind = 1 - ismember(jj(:,idcol),data219(:,idcol));
jjwout219 = jj(jjwout219ind == 1,:);
fprintf('\n\nSome information on auction with id 219:\n')
fprintf('The higher bid at this auction is %.2f times the lower (rival) bid\n', ...
    data219(1,1)/data219(2,1))
fprintf('The higher bid at this auction is %.2f times the next highest bid from any joint vs joint auction\n', ...
    data219(1,1)/max(jjwout219(:,1)))
fprintf('The higher bid at this auction is %.2f standard deviations from the mean joint vs joint bid (data includes auction 219)\n', ...
    (data219(1,1) - mean(jj(:,1)))/std(jj(:,1)))
fprintf('The higher bid at this auction is %.2f standard deviations from the mean joint vs joint bid (data does not include auction 219)\n', ...
    (data219(1,1) - mean(jjwout219(:,1)))/std(jjwout219(:,1)))
if removebads == 1
    jj = jjwout219;
    alldata = allwout219;
end

% some info on auction with id 110
data110 = alldata(alldata(:,idcol) == 110,:);
allwout110ind = 1 - ismember(alldata(:,idcol),data110(:,idcol));
allwout110 = alldata(allwout110ind == 1,:); 
sswout110ind = 1 - ismember(ss(:,idcol),data110(:,idcol));
sswout110 = ss(sswout110ind == 1,:);
fprintf('\n\nSome information on auction with id 110:\n')
fprintf('The higher bid at this auction is %.2f times the lower (rival) bid\n', ...
    data110(1,1)/data110(2,1))
fprintf('The higher bid at this auction is %.2f times the next highest bid from any solo vs solo auction\n', ...
    data110(1,1)/max(sswout110(:,1)))
fprintf('The higher bid at this auction is %.2f standard deviations from the mean solo vs solo bid (data includes auction 110)\n', ...
    (data110(1,1) - mean(ss(:,1)))/std(ss(:,1)))
fprintf('The higher bid at this auction is %.2f standard deviations from the mean solo vs solo bid (data does not include auction 110)\n', ...
    (data110(1,1) - mean(sswout110(:,1)))/std(sswout110(:,1)))
if removebads == 1
    ss = sswout110;
    alldata = allwout110;
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% STAGE 2: step 1 of nonparametric estimation---recover pseudo valuations
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% b0 is needed for h0 bandwidth---see KZ (2008) bottom of page 501 and the
% K0 term in (4.9) of ZK (1998) which is K0 = 6 + 18t + 12t^2, t\in[-1,0];
% this will compute b0 for any kernel choice provided by user
num_t1 = quad(@evalkspdf_num,-1,1,1e-14,[],kernel);
num_t2 = 0 - (-144/5 + 432/4 - 468/3 + 216/2 - 36);
denom_t1 = 0 - (-2 + 18/4 - 12/5);
denom_t2 = quad(@evalkspdf_denom,-1,1,1e-14,[],kernel);
b0 = ((num_t1^2*num_t2)/(denom_t1^2*denom_t2))^(1/5);

% estimate cdfs nonparametrically
[Ghatjj,hGjj] = kscdf2dCPVbc(jj(:,5),jj(:,4),jj(:,4),jj(:,4),kernel,b0);
[Ghatss,hGss] = kscdf2dCPVbc(ss(:,5),ss(:,4),ss(:,4),ss(:,4),kernel,b0);
[Ghatjmix,hGjmix] = kscdf2dCPVbc(jmix(:,5),jmix(:,4),jmix(:,4),jmix(:,4), ...
    kernel,b0);
[Ghatsmix,hGsmix] = kscdf2dCPVbc(smix(:,5),smix(:,4),smix(:,4),smix(:,4), ...
    kernel,b0);

% need to impose that bids are over same range in asymmetric case as this
% is an issue in evaluating kernel involving one class' bid against all
% bids from other class
minbidasym = min(min(jmix(:,4)),min(smix(:,4)));
maxbidasym = max(max(jmix(:,4)),max(smix(:,4)));

% estimate pdfs nonparametrically
[ghatjj,hgjj] = kspdf2dCPVbc(jj(:,5),jj(:,4),jj(:,4),jj(:,4),kernel,b0);
[ghatss,hgss] = kspdf2dCPVbc(ss(:,5),ss(:,4),ss(:,4),ss(:,4),kernel,b0);
[ghatjmix,hgjmix] = kspdf2dCPVbc(jmix(:,5),jmix(:,4),jmix(:,4),jmix(:,4), ...
    kernel,b0,minbidasym,maxbidasym);
[ghatsmix,hgsmix] = kspdf2dCPVbc(smix(:,5),smix(:,4),smix(:,4),smix(:,4), ...
    kernel,b0,minbidasym,maxbidasym);

% count number of times desity is zero
nzerogjj = length(find(ghatjj == 0));
nzerogss = length(find(ghatss == 0));
nzerogjmix = length(find(ghatjmix == 0));
nzerogsmix = length(find(ghatsmix == 0));



% construct pseudo values
vcol = 6;
if logtrans == 1
    d = jj(:,4)';
    v = exp(d).*(1 + Ghatjj./ghatjj) - 1;
    jj(:,vcol) = v';
    
    d = ss(:,4)';
    v = exp(d).*(1 + Ghatss./ghatss) - 1;
    ss(:,vcol) = v';
    
    d = jmix(:,4)';
    v = exp(d).*(1 + Ghatjmix./ghatjmix) - 1;
    jmix(:,vcol) = v';
    
    d = smix(:,4)';
    v = exp(d).*(1 + Ghatsmix./ghatsmix) - 1;
    smix(:,vcol) = v';
else
    b = jj(:,4)';
    v = b + Ghatjj./ghatjj;
    jj(:,vcol) = v';
    
    b = ss(:,4)';
    v = b + Ghatss./ghatss;
    ss(:,vcol) = v';
    
    b = jmix(:,4)';
    v = b + Ghatjmix./ghatjmix;
    jmix(:,vcol) = v';
    
    b = smix(:,4)';
    v = b + Ghatsmix./ghatsmix;
    smix(:,vcol) = v';
end


% count number of bids within bandwidth of high and low bids
nhfrommaxjj = length(find(jj(:,1) > max(jj(:,1)) - hgjj)) - 1;
nhfrommaxss = length(find(ss(:,1) > max(ss(:,1)) - hgss)) - 1;
nhfrommaxjmix = length(find(jmix(:,1) > max(jmix(:,1)) - hgjmix)) - 1;
nhfrommaxsmix = length(find(smix(:,1) > max(smix(:,1)) - hgsmix)) - 1;
nhfromminjj = length(find(jj(:,1) < min(jj(:,1)) + hgjj)) - 1;
nhfromminss = length(find(ss(:,1) < min(ss(:,1)) + hgss)) - 1;
nhfromminjmix = length(find(jmix(:,1) < min(jmix(:,1)) + hgjmix)) - 1;
nhfromminsmix = length(find(smix(:,1) < min(smix(:,1)) + hgsmix)) - 1;

% print some results to screen
fprintf('\n\nFirst-stage bandwidths (see 195 and 198 of CPV to compare):\n')
fprintf('Case\th_G\t\th_g\t\t#zeros in density\n')
fprintf('J vs J\t%.2f\t%.2f\t%d\n',hGjj,hgjj,nzerogjj)
fprintf('J vs S\t%.2f\t%.2f\t%d\n',hGjmix,hgjmix,nzerogjmix)
fprintf('S vs J\t%.2f\t%.2f\t%d\n',hGsmix,hgsmix,nzerogsmix)
fprintf('S vs S\t%.2f\t%.2f\t%d\n',hGss,hgss,nzerogss)

zerocasedata = [jj((ghatjj == 0),:); ss((ghatss == 0),:); ...
    jmix((ghatjmix == 0),:); smix((ghatsmix == 0),:)];
fprintf('Zero densities in auction ids:\n')
for i=1:size(zerocasedata,1)
    fprintf('\t%d\n',zerocasedata(i,idcol))
end
% ids trimmed by CPV when replicating their work
CPVtrimmedids = [3 5 17 23 25 45 47 53 55 56 75 80 83 84 92 93 110 ...
    148 153 180 183 191 210 214 219 226 227]



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% STAGE 3: step 2 of nonparametric estimation; estimate marginal
% distribution of valuations
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

usehardle = 1;
ngrid = 500;
evalgridjj = linspace(min(jj(:,vcol)),max(jj(:,vcol)),ngrid);
Ltjj = size(jj,1)/2;
[fjj,hfjj] = kspdf_bc(jj(:,vcol),kernel,evalgridjj,b0);
evalgridss = linspace(min(ss(:,vcol)),max(ss(:,vcol)),ngrid);
Ltss = size(ss,1)/2;
[fss,hfss] = kspdf_bc(ss(:,vcol),kernel,evalgridss,b0);
evalgridjmix = linspace(min(jmix(:,vcol)),max(jmix(:,vcol)),ngrid);
Ltjmix = size(jmix,1);
[fjmix,hfjmix] = kspdf_bc(jmix(:,vcol),kernel,evalgridjmix,b0);
evalgridsmix = linspace(min(smix(:,vcol)),max(smix(:,vcol)),ngrid);
Ltsmix = size(smix,1);
[fsmix,hfsmix] = kspdf_bc(smix(:,vcol),kernel,evalgridsmix,b0);

% print some results to screen
fprintf('\n\nSecond-stage bandwidths (see 196 and 198 of CPV to compare):\n')
fprintf('Case\th_f\t\tL_t\n')
fprintf('J vs J\t%.2f\t%d\n',hfjj,Ltjj)
fprintf('J vs S\t%.2f\t%d\n',hfjmix,Ltjmix)
fprintf('S vs J\t%.2f\t%d\n',hfsmix,Ltsmix)
fprintf('S vs S\t%.2f\t%d\n',hfss,Ltss)


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% STAGE 4: replicate some figures and final table
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if plotind == 1
    % figure 1 of CPV on page 196
    figure
    set(gcf,'DefaultLineLineWidth',lwidth)
    set(gca,'FontSize',fsize)
    scatter(jj(:,1),jj(:,6),'.b')
    hold all
    scatter(ss(:,1),ss(:,6),'.r')
    title('Figure 1 of CPV on page 196: Inverse bidding strategies')
    legend('J vs J','S vs S','Location','Northwest')
    
    % figure 2 of CPV on page 197
    figure
    set(gcf,'DefaultLineLineWidth',lwidth)
    set(gca,'FontSize',fsize)
    plot(evalgridjj,fjj,':b',evalgridss,fss,'-r')
    title('Figure 2 of CPV on page 197: Marginal densities of private values')
    legend('J vs J','S vs S','Location','Northeast')
    
    % figure 3 of CPV on page 198
    figure
    set(gcf,'DefaultLineLineWidth',lwidth)
    set(gca,'FontSize',fsize)
    scatter(jmix(:,1),jmix(:,6),'.b')
    hold all
    scatter(smix(:,1),smix(:,6),'.r')
    title('Figure 3 of CPV on page 198: Inverse bidding strategies')
    legend('J vs S','S vs J','Location','Northwest')
    
    % figure 4 of CPV on page 199
    figure
    set(gcf,'DefaultLineLineWidth',lwidth)
    set(gca,'FontSize',fsize)
    plot(evalgridjmix,fjmix,':b',evalgridsmix,fsmix,'-r')
    title('Figure 4 of CPV on page 199: Marginal densities of private values')
    legend('J vs S','S vs J','Location','Northeast')
    
end

% determine winning bidder, compute rents and summarize in table
uniqueids = unique(jj(:,idcol));
finaljj = [];
for i=1:Ltjj
    aucid = uniqueids(i);
    ind = jj((jj(:,idcol) == aucid),:);
    if size(ind,1) ~= 2
        fprintf('Error not two bids at symmetric auction\n')
    end
    [C,I] = max(ind(:,1));
    ind(I,7) = 1;
    [C,I] = min(ind(:,1));
    ind(I,7) = 0;
    finaljj = [finaljj; ind];
end
uniqueids = unique(ss(:,idcol));
finalss = [];
for i=1:Ltss
    aucid = uniqueids(i);
    ind = ss((ss(:,idcol) == aucid),:);
    if size(ind,1) ~= 2
        fprintf('Error not two bids at symmetric auction\n')
    end
    [C,I] = max(ind(:,1));
    ind(I,7) = 1;
    [C,I] = min(ind(:,1));
    ind(I,7) = 0;
    finalss = [finalss; ind];
end
uniqueids = unique(jmix(:,idcol));
finaljmix = [];
finalsmix = [];
winjmix = [];
winsmix = [];
for i=1:Ltjmix
    aucid = uniqueids(i);
    indjmix = jmix((jmix(:,idcol) == aucid),:);
    if size(indjmix,1) ~= 1
        fprintf('Error not oen bid at asymmetric auction\n')
    end
    indsmix = smix((smix(:,idcol) == aucid),:);
    if size(indjmix,1) ~= 1
        fprintf('Error not oen bid at asymmetric auction\n')
    end
    if indjmix(1,1) > indsmix(1,1)
        indjmix(1,7) = 1;
        indsmix(1,7) = 0;
        winjmix = [winjmix; indjmix];
    elseif indjmix(1,1) < indsmix(1,1)
        indjmix(1,7) = 0;
        indsmix(1,7) = 1;
        winsmix = [winsmix; indsmix];
    end
    finaljmix = [finaljmix; indjmix];
    finalsmix = [finalsmix; indsmix];
end

% calculate summary statistics
winjj = finaljj((finaljj(:,7) == 1),:);
winss = finalss((finalss(:,7) == 1),:);
winjj(:,8) = (winjj(:,6) - winjj(:,1))./winjj(:,6)*100;
winss(:,8) = (winss(:,6) - winss(:,1))./winss(:,6)*100;
winjmix(:,8) = (winjmix(:,6) - winjmix(:,1))./winjmix(:,6)*100;
winsmix(:,8) = (winsmix(:,6) - winsmix(:,1))./winsmix(:,6)*100;
allwins = [winjj; winss; winjmix; winsmix];
alljointwins = [winjj; winjmix];
allsolowins = [winss; winsmix];


% replicate table 2 in CPV
fprintf('\n\nVariable\t#Obs\tMean\tSTD\t\tMin\t\tMax\n')
fprintf('All\t\t\t%d\t\t%.2f\t%.2f\t%.2f\t%.2f\n', ...
    size(allwins,1),mean(allwins(:,8)),std(allwins(:,8)), ...
    min(allwins(:,8)),max(allwins(:,8)))
fprintf('Joint\t\t%d\t\t%.2f\t%.2f\t%.2f\t%.2f\n', ...
    size(alljointwins,1),mean(alljointwins(:,8)),std(alljointwins(:,8)), ...
    min(alljointwins(:,8)),max(alljointwins(:,8)))
fprintf('Solo\t\t%d\t\t%.2f\t%.2f\t%.2f\t%.2f\n', ...
    size(allsolowins,1),mean(allsolowins(:,8)),std(allsolowins(:,8)), ...
    min(allsolowins(:,8)),max(allsolowins(:,8)))
fprintf('J vs J\t\t%d\t\t%.2f\t%.2f\t%.2f\t%.2f\n', ...
    size(winjj,1),mean(winjj(:,8)),std(winjj(:,8)), ...
    min(winjj(:,8)),max(winjj(:,8)))
fprintf('J vs S\t\t%d\t\t%.2f\t%.2f\t%.2f\t%.2f\n', ...
    size(winjmix,1),mean(winjmix(:,8)),std(winjmix(:,8)), ...
    min(winjmix(:,8)),max(winjmix(:,8)))
fprintf('S vs J\t\t%d\t\t%.2f\t%.2f\t%.2f\t%.2f\n', ...
    size(winsmix,1),mean(winsmix(:,8)),std(winsmix(:,8)), ...
    min(winsmix(:,8)),max(winsmix(:,8)))
fprintf('S vs S\t\t%d\t\t%.2f\t%.2f\t%.2f\t%.2f\n', ...
    size(winss,1),mean(winss(:,8)),std(winss(:,8)), ...
    min(winss(:,8)),max(winss(:,8)))

% replicate table 2 in CPV
fprintf('\n\nVariable & Number Obs. & Mean & St.~Dev. & Minimum & Maximum \\\\ \n')
fprintf('All & %d & %.2f & %.2f & %.2f  & %.2f \\\\\n', ...
    size(allwins,1),mean(allwins(:,8)),std(allwins(:,8)), ...
    min(allwins(:,8)),max(allwins(:,8)))
fprintf('Joint & %d & %.2f & %.2f & %.2f  & %.2f \\\\\n', ...
    size(alljointwins,1),mean(alljointwins(:,8)),std(alljointwins(:,8)), ...
    min(alljointwins(:,8)),max(alljointwins(:,8)))
fprintf('Solo & %d & %.2f & %.2f & %.2f  & %.2f \\\\\n', ...
    size(allsolowins,1),mean(allsolowins(:,8)),std(allsolowins(:,8)), ...
    min(allsolowins(:,8)),max(allsolowins(:,8)))
fprintf('J vs.~J & %d & %.2f & %.2f & %.2f  & %.2f \\\\\n', ...
    size(winjj,1),mean(winjj(:,8)),std(winjj(:,8)), ...
    min(winjj(:,8)),max(winjj(:,8)))
fprintf('J vs.~S & %d & %.2f & %.2f & %.2f  & %.2f \\\\\n', ...
    size(winjmix,1),mean(winjmix(:,8)),std(winjmix(:,8)), ...
    min(winjmix(:,8)),max(winjmix(:,8)))
fprintf('S vs.~J & %d & %.2f & %.2f & %.2f  & %.2f \\\\\n', ...
    size(winsmix,1),mean(winsmix(:,8)),std(winsmix(:,8)), ...
    min(winsmix(:,8)),max(winsmix(:,8)))
fprintf('S vs.~S & %d & %.2f & %.2f & %.2f  & %.2f \\\\\n', ...
    size(winss,1),mean(winss(:,8)),std(winss(:,8)), ...
    min(winss(:,8)),max(winss(:,8)))
