function X = tmigrator(operation, varargin)
%TMIGRATOR Migrates different data files and formats to one or more
%          datasets, cell arrays or matrices.
%
%   IMPORTANT:
%           1. Each and every spreadsheet or ASCII file table must provide
%              a headline with word column identifiers: a-z A-Z 0-9 _ not
%              beginning with a number. Otherwise, an error will occur.
%
%           2. For non spreadsheets, at least one delimiter should be provided.
%              Otherwise tmigrator searches for non-letters such as '-' or '_'
%              as delimiters.
%
%           There are two ways to configure the function's behaviour.
%
%           I.  Operations:     Meta option and file input behaviour.
%           II. Parameters:     Options and additional information.
%
%   Usage:
%
%           final_table = tmigrator(<op>, <input_tables, options>)
%
%           INPUT TABLES (many formats) --> FINAL TABLE (dataset format, N x K)
%
%           The table columns consist of pure double or string values. Missing
%           / empty values are indicated by value NaN respective string ''.
%
%   Operations:
%
%               Optiations act as a meta options. When choosing one
%               operation, it configures tmigrators options according to
%               the operation's task.
%
%               You can only set one operation mode.
%
%       cc      Column Combination
%
%                   Combines the columns of different tables by
%                   stringing the columns together to a new table.
%
%       ts    	Time Series
%                              
%                   Creates one table by joining the columns of different
%                   tables over their date column. The date column of the
%                   final table is always the first column (number 1).
%
%                   Requires:
%                               Each file must have a date column on the left.
%
%                   Set Options:
%                               -columnnames, -convertdates, -key,
%                               -join, -sortrows
%
%   Parameters:
%
%               All parameters as well as the table input can be combined
%               and their order of appearance does not matter. However,
%               their exist some impossible combinations. You'll get an
%               error in those cases.
%
%               By default, all options are turned off. When they are
%               commited, tmigrator will activate them. However, the
%               operations (such as 'ts') mentioned before influence
%               the options.
%
%       Table Input:
%
%               As all parameters, you can commit an arbitrary amount of
%               files and variables to the tmigrator. Variables are
%               passed directly and files in quotes: 'file1.txt'
%
%               Files:     Folders, ASCII (CSV, ...), spreadsheets (Excel, ...),
%                          XML, ...
%                          (Tip: The -recursive option could be useful for
%                          folders!)
%
%               Variables: Datasets, cell arrays, matrices with raw input
%                          or datasets, cell arrays, matrices with
%                          tmigrator-structure.
%                          (Tip: Generate output, to see these structures!)
%
%               Example:    tmigrator('cc', 'file1', 'path', datasetX, ...
%                                     matrixY, '-recursive')
%
%
%       Predefined elements in following description of options:
%
%           <colIds>:   Cell array which elements are column names or numbers.
%                       Usually mixed sets and column selection with
%                       replacement are allowed.
%
%                       Example:     colIds = {'Date', 3, 'GDP'}
%
%             <freq>:   Available frequency for data:
%
%                       'daily':     Every day of a year.
%                       'wdaily'     Every Mo - Fri  of year.
%                       'weekly'     Every week of a year.
%                       'monthly'    Every month of a year.
%                       'quarterly'  Every three month of a year.
%                       'yearly'     Once per year.
%
%      In the following, optional parameters are written in parentheses.
%
%      Options:
%
%       -adddates, { <freq> (, startdate, enddate, fix) }:
%
%           Operation-dependend: Only avaible in 'ts' operation mode.
%
%           Add an specific date vector beginning from startdate to enddate
%           with frequency freq to the date column. The new rows appear
%           with empty fields in the all columns accept the date column.
%
%           startdate:  Any MATLAB datenum number. Default: Begin of dates.
%           enddate:    Any MATLAB datenum number. Default: End of dates.
%           fix:        Default date beginning option. 
%
%           Tip: Could be used with -interpolate.
%
%           This option applies AFTER migration to the FINAL TABLE.
%
%           Example:
%                       '-adddates', {'daily', 725612, 730486}
%
%       -cellarray:
%           Let tmigrator return the migrated table as an cell array with
%           K columns and N + 1 rows. The first row contains the column
%           identifiers as strings.
%
%           This option applies AFTER migration to the FINAL TABLE.
%
%
%       -columnnames, <collds>, strArr:
%           Changes the column names of columns defined in <collIds>
%           to names defined in the cell array strArr.
%
%           This option applies AFTER migration to the FINAL TABLE.
%           
%           Example:
%                       '-columnnames', {'a1', 'foo'}, {'gdp', 'm0'}
%
%       -complete:
%           Checks whether the final table consists of empty fields (NaNs).
%           If at least one empty field is found, the function will
%           throw an error.
%
%           This option applies AFTER migration to the FINAL TABLE.
%
%
%       -convertdates, <colIds>
%           Converts the string dates in columns array <collIds> of each
%           input table to the MATLAB datenum format. This is important
%           for the unification process.
%
%           This option applies BEFORE migration to the INPUT TABLES.
%           
%           Example:
%                       '-columndates', {'date', 'dates'}
%           
%       -dateformat format:
%           Option to switch the definition for yearly, quarterly, monthly and
%           wdaily dates from the standard middel-of-period format to the
%           another format. E. g. Q1/2012: 15.02.2012 --> 01.01.2012.
%
%           format: Format of dates. Example Q1/2012:
%                   'begin'     01.01.2012
%                   'middle'    15.02.2012
%                   'end'       31.03.2012
%
%           Tip: Could be useful with -adddates and -freqconvert options.
%
%           This option applies BEFORE migration to the INPUT TABLES.
%
%           Example:
%                       '-dateformat', 'end'
%       
%        -datefreq <freq>:
%           Set the frequency type of data in table. The default frequency is
%           'daily'.
%
%           Tip: This option should be set for -freqconvert.
%
%           This option applies BEFORE migration to the INPUT TABLES.
%
%
%        -datestr
%           Convert all date columns specified by the '-convertdates' option or
%           while operating in time series mode into the MATLAB datestr format.
%
%           This option applies AFTER migration to the FINAL TABLE.
%
%
%        -delimiter, char:
%           Delimiter char for files provided in parameters directly afterwards.
%           If the delimiter option is placed before all file parameters,
%           the delimiter is set until the delimiter is set again.
%           Therefore, the placement of this option matters; if it is only
%           set once before all files and folders, it is set globally.
%
%           This option applies BEFORE migration to the INPUT TABLES.
%
%           Example:
%                       '-delimiter', ';', 'file1', '-delimiter', '/', 'file2', 
%                       (';' is set globally, and '/' is set for 'file2')
%
%       -denumcat (, <colIds>):
%           If string categories were converted to numbered categories with
%           option -numcat then -denumcat reverse the procedure and turns
%           the number categories back to the string categories. If
%           no <colIds> with column identifiers for columns to reverse is
%           provided, all columns were applicable are converted back.
%
%           This option applies AFTER migration to the FINAL TABLE.
%
%           Example:
%                       '-denumcat', {'colournames'}
%
%       -freqconvert, <freq>, {method (, <colIds1>)} (, {method... ):
%
%           Operation-dependend: Only avaible in 'ts' operation mode.
%
%           Changes the date frequency of a whole table to <freq>. The new
%           frequency can be lower or higher than the current one. Therefore,
%           conversion methods from high --> low or low --> high frequency
%           have to be provided. Distinct methods can be applied to different
%           columns. If no <colIds> is provided this method will be applied
%           globally.Ensure you to define a method for every column,
%           otherwise you will receive an error.
%
%           method:
%                   # High to Low Frequency Conversion Methods
%                   'first'     Take the first element of each group.
%                   'last'      Take the last element of each group.
%                   'min'       Take the smallest element of group.
%                   'mean'      Take the average of group.
%                   'max'       Take the largest element of group.
%                   'sum'       Take the sum of group.
%
%                   # Low to High Frequency Conversion Methods
%                   'after'     Clone value from vertical neighbour after.
%                   'before'    Clone value from vertical neighbour before.
%                   'clone'     Insert the same value as in lower freq.
%                   'linear'    Linearly interpolate
%                   'nearest'   Clone value from nearest vertical neighbour.
%                   'single'    Do not fill empty rows. Leave single values.
%                   --------------------------------------------------------
%                   'divafter'  Divide by number of elements, than 'after'.
%                   'divbefore' Divide by number of elements, than 'before'.
%                   'divclone'  Divide by number of elements, than 'clone'.
%                   'divlinear' Divide by number of elements, than 'linear'.
%                   'divnearest'Divide by number of elements, than 'nearest'.
%                   'divsingle' Divide by number of elements, than 'single'.
%
%           Tip: You should use -datefreq first. See also -interpolate.
%
%           This option applies AFTER migration to the FINAL TABLE.
%
%           Example:
%                       '-freqconvert', 'monthly', {'sum'}, ...
%                       {'mean', 'gold_prices'}
%
%       -fullname:
%           Replace the column names by more detailed names. Usually more
%           specific names are only chosen in case of duplicate column
%           identifiers. If for example in 'ts' mode, two table have the
%           same column 'gdp'.
%
%           Tip: Could be useful with -fullpath option.
%
%           This option applies AFTER migration to the FINAL TABLE.
%
%
%       -fullpath:
%           Identify table columns from files with full relative path.
%
%           Tip: Could be useful with -fullname option.
%
%           This option applies BEFORE migration to the INPUT TABLES.
%
%
%       -german:
%           Just switches to German language in final table. In 'ts' mode,
%           the column identifier of the date column is replaced by
%           'Datum'.
%
%           This option applies AFTER migration to the FINAL TABLE.
%
%
%       -interpolate, method (, <colIds>):
%
%           Operation-dependend: Only avaible in 'ts' operation mode.
%
%           Interpolates empty fields (NaNs) of columns <colIds>. The option
%           can be used many times, if distinct methods will be applied.
%           If no <colIds> is provided the method will be applied globally.
%
%           method:
%                   'after'     Clone value from vertical neighbour after.
%                   'before'    Clone value from vertical neighbour before.
%                   'linear'    Linearly interpolate vertically.
%                   'nearest'   Clone value from nearest vertical neighbour.
%
%           Tip: Could be used with -dateinterval. See also -freqconvert.
%
%           This option applies AFTER migration to the FINAL TABLE.
%
%           Example:
%                       '-interpolate', 'linear', {'gdp', 'foo'}, ...
%                       '-interpolate', 'nearest', {'m0'}
%
%       -join (, method):
%
%           Option-dependend: Only available after using '-key' option.
%
%           Joins the input tables by the key columns specified by using the
%           -key option. For more information about joining, see MATLAB
%           join.
%
%           method:
%                   'inner'     Inner join. Default.
%                   'outer'     Outer join.
%                   'left'      Left outer join.
%                   'right'     Right outer join.
%
%           This option applies BEFORE migration to the INPUT TABLES.
%
%           Example:
%                       '-join', 'left'
%
%       -key <colIds>:
%           Defines the key columns of the input tables for further
%           proceedings. The key column must exist in every input table.
%           The key column is automatically set to the date column in
%           time series mode.
%
%           Set Options:    -unicat : Adds <colIds>.
%
%           Tip:    Could be useful with -join or -sortrows option.
%
%           This option applies BEFORE migration to the INPUT TABLES.
%
%           Example:
%                       '-key', {'id1', 'foo'}
%
%       -lowercolumns:
%           Turns all column identifiers to lower case.
%
%           This option applies AFTER migration to the FINAL TABLE.
%
%
%       -matrix:
%           Let tmigrator return the migrated table as an struct with a
%           numerical matrix (N x K) and an cell array with the column
%           identifiers.
%
%           This option applies AFTER migration to the FINAL TABLE.
%
%
%       -nomat:
%           Do not regard MAT-Files input tables.
%
%           This option applies BEFORE migration to the INPUT TABLES.
%
%
%       -numcat (, <colIds>):
%           Converts all columns with column identifiers in <colIds> from
%           n string categories to number categories [1-n] automatically. The
%           legend is available in the 'UserData' cell array for the dataset:
%
%                       final_table.Properties.UserData
%
%           The first cell element {1} is assigned to number 1, {2} to
%           number 2 and so on.
%
%           Tip: Use -denumcat for back conversion.
%
%           This option applies AFTER migration to the FINAL TABLE.
%
%           Example:
%                       '-numcat', {'colournames'}
%
%       -pertable:
%           Applies all options and operations to each table separately. The
%           recursive procedure produces an struct with a table array
%           and a table name array for each final table.
%
%           This option applies to the INPUT TABLES.
%
%
%       -progress:
%           Displays progress of tmigrator's processing in the command window.
%
%
%       -recursive:
%           Reads every commited file folder recursively with all
%           subfolders and subsubfolders and so on.
%
%           This option applies BEFORE migration to the INPUT TABLES.          
%
%
%       -removerows, method (, <colIds>):
%
%           Removes rows from the final table. The option can be used many
%           times, if distinct methods will be applied. If no <colIds>
%           is provided the method will be applied to all columns except
%           of the '-key' columns.
%
%           method:
%                   'constant'  Remove constant rows. Removes the lower 
%                               vertical neighbour row, if all columns
%                               specified in <colIds> contain the same values.
%
%                   'empty'     Remove empty rows. Removes rows which contain 
%                               empty fields in all columns specified in
%                               <colIds>.
%
%           This option applies AFTER migration to the FINAL TABLE.
%
%           Example:
%                       '-removerows', 'constant', ...
%                       '-removerows', 'empty', {'gdp', 'm0'}
%
%       -selectbydates, datevec(, mode)
%
%           Operation-dependend: Only avaible in 'ts' operation mode.
%
%           Selects the rows of the final table by specifying a date
%           vector with unique dates in MATLAB datenum format to choose.
%           The strict mode ensures that all dates in the vector are
%           available in the final table's date column.
%
%           datevec:   Vector containing unique dates to select.
%
%           mode:
%                   'available' The available dates in the final table
%                               are chosen. Default.
%
%                   'strict'    The date vector must contain dates which
%                               are definitely included in the table.
%                               Otherwise an error will occur.
%
%           This option applies AFTER migration to the FINAL TABLE.
%
%           Example:
%                       '-selectbydates', [724827; 725612; 730486], 'strict'
%
%       -selectcolumns, <colIds>:
%           Selects the columns defined in <colIds> from the final table.
%
%           This option applies AFTER migration to the FINAL TABLE.
%
%           Example:
%                       '-selectcolumns', {1, 'gdp', 'm0'}
%
%       -sortcolumns:
%           Sorts the columns according to their identifiers alphabetically.
%
%           This option applies AFTER migration to the FINAL TABLE.
%
%
%       -sortrows:
%           Sorts the rows alphabetically by the values in their key columns.
%           If no key columns were set, the rows are sorted by all table
%           columns from left to right.
%
%           Tip:    Could be useful with -key option.
%
%           This option applies AFTER migration to the FINAL TABLE.
%
%
%       -unicat <colIds>:
%           When -numcat was applied to different input tables, there is a
%           chance of various number-string associations which can cause
%           mistakes when migrating tables in another step. This option 
%           ensures that all input tables have the same number-string
%           associations in the columns identified by <colIds>.
%
%           This option applies BEFORE migration to the INPUT TABLES.
%
%           Example:
%                       '-unicat', {'colournames'}
%
%
%   Outputs:
%
%       X:  Final table respective meta table output (see '-pertable').
%           If no other table format was chosen, X is a dataset respective
%           a collection of datasets.
%
%           Tip:    See X.Properties and especially X.Properties.VarNames for 
%                   X's column names. For addtionally information see
%                   X.Properties.UserData, e. g. when using '-numcat'.
%
%
%   Examples:
%
%           Import the two files 'file1' and 'file2' with different time
%           series to one panel table. The global delimiter is ';'.
%           Sort the columns by names:
%
%           final_table1 = tmigrator('ts', '-delimiter', ';', 'file1', ...
%                                    'file2', '-sortcolumns');
%
%           Then, transforming this table into a matrix and lower the 
%           column names:
%
%           final_table1_matrix = tmigrator('ts', final_table1, '-matrix', ...
%                                            '-lowercolumns');
%
%
%           Read all files from folder recursively and form a huge table by
%           stringing all columns. Convert string categories automatically.
%           Find all delimiters by yourself and show the progress.
%
%           final_table2 = tmigrator('cc', folder, '-recursive', ...
%                                    '-numcat', '-progress');
%
%
%   Future Improvments:
%
%
%           -lowmemory       :  Algorithm trys to reduce memory usage
%                               by higher computation costs.
%
%           -dechiffre crazy headline names
%
%           -find date column, specify datecolumn --> fix -pertable mode
%                                                     -error message-
%
%           -unicat     improvements
%
%           -->direct reference to matrix2latex for latex output
%           --> -output Option: matlab (default), file, latex
%
%           better error message handling
%
%           mode: cmp - Compare different files/tables whether they are
%                       similar. Also use -unicat for label comparison.
%
%   Author:         Fabian Raters
%   Version:        2012-06-12
%   E-mail:         fra@stat-econ.uni-kiel.de
%   Institution:    Institute for Statistics and Econometrics, CAU Kiel
%   URL:            http://www.stat-econ.uni-kiel.de/
%
%   Copyright 2012 by Fabian Raters

% supress dynamic array warning
%#ok<*AGROW>

% table data
D = {};

% table names
name = {};

%%%%% Parameter scanning %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% flags
f_adddates = 0; f_adddates_arr = {};
f_cellarray = 0;
f_colnames = 0; f_colnames_arr = {}; f_colnames_names = {};
f_complete = 0;
f_convdates = 0; f_convdates_arr = {};
f_dateformat = 0; f_dateformat_format = '';
f_datefreq = 0; f_datefreq_freq = '';
f_datestr = 0;
f_denumcat = 0; f_denumcat_arr = {};
f_freqconv = 0;f_freqconv_highlow = [];f_freqconv_freq = '';f_freqconv_arr = {};
f_fullname = 0;
f_fullpath = 0;
f_german = 0;
f_interpol = 0; f_interpol_method = {}; f_interpol_arr = {};
f_join = 0; f_join_type = '';
f_key = 0; f_key_arr = {};
f_lowercol = 0;
f_matrix = 0;
f_nomat = 0;
f_numcat = 0; f_numcat_arr = {};
f_pertable = 0;
f_progress = 0;
f_recursive = 0;
f_remrows = 0; f_remrows_method = {}; f_remrows_arr = {};
f_selectbd = 0; f_selectbd_vec = []; f_selectbd_mode = 0;
f_selectcols = 0; f_selectcols_arr = {};
f_sortcols = 0;
f_sortrows = 0;
f_unicat = 0; f_unicat_arr = {};

% temp parameters for scanning
temp = {};
tdel = {};
gdel = [];

% parameter parsing
p = 1;
parlength = size(varargin, 2);
while p <= parlength
    if ischar(varargin{p})
        % convert inputs to lowercase
        parstr = lower(varargin{p});
        switch parstr
            case '-adddates'
                f_adddates = 1;
                if p < parlength
                    f_adddates_arr = varargin{p + 1};
                    if iscell(f_adddates_arr)
                        f_adddates_arr{1} = lower(f_adddates_arr{1});
                        if isFreq(f_adddates_arr{1})
                            % skip next parameter (adddates array)
                            p = p + 1;
                        else
                            error(['Wrong ''-adddates'' usage!' ...
                                ' Unknown frequency to convert: ''' ...
                                f_adddates_arr{1} '''!']);
                        end
                    else
                        error(['Options for ''-adddates'' must be' ...
                            ' provided in an cell array! For example:' ...
                            ' ''{''daily'', 725612, 730486, 1}''.']);
                    end
                else
                    error('Wrong ''-adddates'' usage!');
                end
            case '-cellarray'
                if f_matrix
                    error(['Options ''-cellarray'' and ''-matrix''' ...
                        ' cannot be combined!']);
                else
                    f_cellarray = 1;
                end
            case '-columnnames'
                f_colnames = 1;
                if p + 1 < parlength
                    f_colnames_arr = varargin{p + 1};
                    f_colnames_names = varargin{p + 2};
                    if iscell(f_colnames_arr) && iscell(f_colnames_names)
                        % skip next parameters (column array and names)
                        p = p + 2;
                    else
                        error(['Column names or indices for renaming' ...
                            ' must be provided in two subsequent cell' ...
                            ' arrays. The format of the first' ...
                            ' (indicator) cell array can be for' ...
                            ' example: ''{''col1'', ''col2''}'' or' ...
                            ' ''{1, 2}''.']);
                    end
                else
                    error('Wrong ''-columnnames'' usage!');
                end
            case '-complete'
                f_complete = 1;
            case '-convertdates'
                f_convdates = 1;
                if p < parlength
                    f_convdates_arr = varargin{p + 1};
                    if iscell(f_convdates_arr)
                        % skip next parameter (convdate array)
                        p = p + 1;
                    else
                        error(['Column names or indices for date' ...
                            ' conversion must be provided in an cell' ...
                            ' array! For example: ''{''col1'','...
                            ' ''col2''}'' or ''{1, 2}''.']);
                    end
                else
                    error('Wrong ''-convertdates'' usage!');
                end
            case '-dateformat'
                f_dateformat = 1;
                if p < parlength && ischar(varargin{p + 1})
                    f_dateformat_format = lower(varargin{p + 1});
                    switch f_dateformat_format
                        case {'begin', 'middle', 'end'}
                            % skip next parameter (dateformat format)
                            p = p + 1;
                        otherwise
                            error(['Wrong ''-dateformat'' usage!' ...
                            ' Unknown format defined: ''' ...
                            f_dateformat_format '''!']);
                    end
                else
                    error('Wrong ''-dateformat'' usage!');
                end
            case '-datefreq'
                f_datefreq = 1;
                if p < parlength && ischar(varargin{p + 1})
                    f_datefreq_freq = lower(varargin{p + 1});
                    if isFreq(f_datefreq_freq)
                        % skip next parameter (datefreq freq)
                        p = p + 1;
                    else
                        error(['Wrong ''-datefreq'' usage!' ...
                            ' Unknown frequency defined: ''' ...
                            f_datefreq_freq '''!']);
                    end
                else
                    error('Wrong ''-datefreq'' usage!');
                end
            case '-datestr'
                f_datestr = 1;
            case '-delimiter'
                if p < parlength
                    t = regexp(varargin{p + 1}, '\W');
                    if length(t) == 1 && t == 1
                        if isempty(temp)
                            gdel = varargin{p + 1};
                        else
                            tdel{end} = varargin{p + 1};
                        end
                        
                        % skip next parameter (delimiter char)
                        p = p + 1;
                    else
                        error(['Delimiter must be single non-alpha-' ...
                            'numerical characters. For example: '';''']);
                    end
                else
                    error('Wrong ''-delimiter'' usage!');
                end
            case '-denumcat'
                f_denumcat = 1;
                if p < parlength && iscell(varargin{p + 1})
                    f_denumcat_arr = varargin{p + 1};
                    % skip next parameter (denumcat array)
                    p = p + 1;
                end
            case '-freqconvert'
                f_freqconv = 1;
                if p < parlength && ischar(varargin{p + 1})
                    f_freqconv_freq = lower(varargin{p + 1});
                    if isFreq(f_freqconv_freq)
                        % skip next parameter (freqconv freq)
                        p = p + 1;
                    else
                        error(['Wrong ''-freqconvert'' usage!' ...
                            ' Unknown frequency to convert: ''' ...
                            f_freqconv_freq '''!']);
                    end
                    while p < parlength && iscell(varargin{p + 1})
                        f_freqconv_arr{end + 1} = varargin{p + 1};
                        f_freqconv_arr{end}{1} = ...
                            lower(f_freqconv_arr{end}{1});
                        switch f_freqconv_arr{end}{1}
                            case {'first', 'last', 'max', 'mean', 'min', 'sum'}
                                if isempty(f_freqconv_highlow)
                                    f_freqconv_highlow = 1;
                                else
                                    if f_freqconv_highlow == 0
                                        error(['Wrong ''-freqconvert'' usage!' ...
                                            ' Mixing high-low and low-high'
                                            ' frequency conversion methods' ...
                                            ' does not make any sense!' ...
                                            ' Method: ''' ...
                                            f_freqconv_arr{end}{1} '''!']);
                                    end
                                end
                            case {'after', 'before', 'clone', 'divafter', ...
                                    'divbefore', 'divclone', 'divlinear', ...
                                    'divnearest', 'divsingle', 'linear', ...
                                    'nearest', 'single'}
                                if isempty(f_freqconv_highlow)
                                    f_freqconv_highlow = 0;
                                else
                                    if f_freqconv_highlow == 1
                                        error(['Wrong ''-freqconvert'' usage!' ...
                                            ' Mixing high-low and low-high'
                                            ' frequency conversion methods' ...
                                            ' does not make any sense!' ...
                                            ' Method: ''' ...
                                            f_freqconv_arr{end}{1} '''!']);
                                    end
                                end
                            otherwise
                                error(['Wrong ''-freqconvert'' usage!' ...
                                ' Unknown conversion method: ''' ...
                                f_freqconv_arr{end}{1} '''!']);
                        end
                        % skip next parameter (freqconv array)
                        p = p + 1;
                    end
                else
                    error('Wrong ''-freqconvert'' usage!');
                end
            case '-fullname'
                f_fullname = 1;
            case '-fullpath'
                f_fullpath = 1;
            case '-german'
                f_german = 1;
            case '-interpolate'
                f_interpol = 1;
                if p < parlength && ischar(varargin{p + 1})
                    f_interpol_method{end + 1} = lower(varargin{p + 1});
                    switch f_interpol_method{end}
                        case {'after', 'before', 'linear', 'nearest'}
                            % skip next parameter (interpolate method)
                            p = p + 1;
                        otherwise
                            error(['Wrong ''-interpolate'' usage!' ...
                                ' Unknown interpolation method ''' ...
                                f_interpol_method{end} '''!']);
                    end
                    if p < parlength && iscell(varargin{p + 1})
                        f_interpol_arr{end + 1} = varargin{p + 1};
                        % skip next parameter (interpolate array)
                        p = p + 1;
                    else
                        f_interpol_arr{end + 1} = {};
                    end
                else
                    error('Wrong ''-interpolate'' usage!');
                end
            case '-join'
                f_join = 1;
                if p < parlength && ischar(varargin{p + 1})
                    f_join_type = lower(varargin{p + 1});
                    switch f_join_type
                        case {'inner', 'left', 'right', 'outer'}
                            % skip next parameter (join type)
                            p = p + 1;
                        otherwise
                            f_join_type = 'inner';
                    end
                end
            case '-key'
                f_key = 1;
                if p < parlength
                    f_key_arr = varargin{p + 1};
                    if iscell(f_key_arr)
                        % skip next parameter (key array)
                        p = p + 1;
                    else
                        error(['Key names or column indices must be' ...
                            ' provided in an cell array! For example:' ...
                            ' ''{''col1'', ''col2''}'' or ''{1, 2}''.']);
                    end
                else
                    error('Wrong ''-key'' usage!');
                end
            case '-lowercolumns'
                f_lowercol = 1;
            case '-matrix'
                if f_cellarray
                    error(['Options ''-matrix'' and ''-cellarray''' ...
                        ' cannot be combined!']);
                else
                    f_matrix = 1;
                end
            case '-nomat'
                f_nomat = 1;
            case '-numcat'
                f_numcat = 1;
                if p < parlength && iscell(varargin{p + 1})
                    f_numcat_arr = varargin{p + 1};
                    % skip next parameter (numcat array)
                    p = p + 1;
                end
            case '-pertable'
                f_pertable = 1;
            case '-progress'
                f_progress = 1;
            case '-recursive'
                f_recursive = 1;
            case '-removerows'
                f_remrows = 1;
                if p < parlength && ischar(varargin{p + 1})
                    f_remrows_method{end + 1} = lower(varargin{p + 1});
                    switch f_remrows_method{end}
                        case {'constant', 'empty'}
                            % skip next parameter (remrows method)
                            p = p + 1;
                        otherwise
                            error(['Wrong ''-removerows'' usage!' ...
                                ' Unknown method for removing rows: ''' ...
                                f_remrows_method{end} '''!']);
                    end
                    if p < parlength && iscell(varargin{p + 1})
                        f_remrows_arr{end + 1} = varargin{p + 1};
                        % skip next parameter (remrows array)
                        p = p + 1;
                    else
                        f_remrows_arr{end + 1} = {};
                    end
                else
                    error('Wrong ''-removerows'' usage!');
                end
            case '-selectbydates'
                f_selectbd = 1;
                if p < parlength
                    f_selectbd_vec = varargin{p + 1};
                    if isvector(f_selectbd_vec)
                        % skip next parameter (selectbd vector)
                        p = p + 1;
                    else
                        error(['The first parameter for selection by date' ...
                            ' must be the vector with the dates to select' ...
                            ' such as [724827; 725612; 730486; ...].']);
                    end
                    if p < parlength && ischar(varargin{p + 1})
                        f_selectbd_mode = lower(varargin{p + 1});
                        switch f_selectbd_mode
                            case {'available', 'strict'}
                                % skip next parameter (selectbd mode)
                                p = p + 1;
                            otherwise
                                f_selectbd_mode = 'available';
                        end
                    end
                else
                    error('Wrong ''-selectbydates'' usage!');
                end
            case '-selectcolumns'
                f_selectcols = 1;
                if p < parlength
                    f_selectcols_arr = varargin{p + 1};
                    if iscell(f_selectcols_arr)
                        % skip next parameter (selectcols array)
                        p = p + 1;
                    else
                        error(['Column names or indices for selection must' ...
                            ' be provided in an cell array! For example:' ...
                            ' ''{''col1'', ''col2''}'' or ''{1, 2}''.']);
                    end
                else
                    error('Wrong ''-selectcolumns'' usage!');
                end
            case '-sortcolumns'
                f_sortcols = 1;
            case '-sortrows'
                f_sortrows = 1;
            case '-unicat'
                f_unicat = 1;
                if p < parlength
                    f_unicat_arr = varargin{p + 1};
                    if iscell(f_unicat_arr)
                        % skip next parameter (convdate array)
                        p = p + 1;
                    else
                        error(['Column names or indices for category' ...
                            ' unification must be provided in an cell' ...
                            ' array! For example: ''{''col1'','...
                            ' ''col2''}'' or ''{1, 2}''.']);
                    end
                else
                    error('Wrong ''-unicat'' usage!');
                end
            otherwise
                temp{end + 1} = parstr;
                tdel{end + 1} = gdel;
        end
    else
        temp{end + 1} = varargin{p};
        tdel{end + 1} = gdel;
    end
    
    % step counter
    p = p + 1;
end


%%%%% Behaviour and Predefinitions %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if f_dateformat
    dates_format = f_dateformat_format;
else
    dates_format = 'middle'; % default format: middle
end

if f_datefreq
    dates_freq = f_datefreq_freq;
else
    dates_freq = 'daily';
end

if f_german
    t_date = 'Datum';
else
    t_date = 'Date';
end


%%%%% Post Processing Operations and Options %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% operation
switch operation
    case 'cc'
        
    case 'ts'
        % set flags
        
        % add date colname
        f_colnames = 1;
        f_colnames_arr(end + 1) = {1};
        dup = findDuplicates(f_colnames_arr, 0, 1);
        if dup(end)
            f_colnames_arr(end) = [];
        else
            f_colnames_names(end + 1) = {t_date};
        end
        
        % convert date column
        f_convdates = 1;
        f_convdates_arr(end + 1) = {1};
        f_convdates_arr(findDuplicates(f_convdates_arr, 0, 1)) = [];
        
        % set datecolumn as key
%         if f_key
%             error(['You cannot specify keys in time series mode!' ...
%                 ' The operation automatically sets the date column as' ...
%                 ' single key column.']);
%         end
        f_key = 1;
        f_key_arr = {1};
        
        % set flags
        f_join = 1;
        f_join_type = 'outer';
        f_sortrows = 1;
    otherwise
        error('Operation cannot be regconized!');
end

%%% Column Names Option
if f_colnames
    f_colnames_arr = formatColumn(f_colnames_arr);
    if isempty(f_colnames_arr)
        error(['Key columns are in wrong format! The cell array must' ...
            ' contain names or indices.']);
    end
    if length(f_colnames_arr) ~= length(f_colnames_names)
        error(['The two cell arrays for renaming columns' ...
            ' have the different lengths!']);
    end
    if ~iscellstr(f_colnames_names)
        error('The new column names cell array contains non-strings!');
    end
end

%%% Convert Date Option
if f_convdates
    f_convdates_arr = formatColumn(f_convdates_arr);
    if isempty(f_convdates_arr)
        error(['Columns for dates to be converted are in wrong format!' ...
            ' The cell array must contain names or indices.']);
    end
end

%%% Date String Format Option
if f_datestr
    if ~f_convdates
        error(['Cannot perform ''-datestr''. No date columns were'...
            ' specified! Use time series mode or consider ''-convertdate''']);
    end
end

%%% Uniform Categories Option
if f_unicat
    f_unicat_arr = formatColumn(f_unicat_arr);
    if isempty(f_unicat_arr)
        error(['Columns for category unification are in wrong format!' ...
            ' The cell array must contain names or indices.']);
    end
end

%%% Join Option
if f_join
    if ~f_key
        error(['Cannot perform join operation. No key columns were'...
            ' chosen!']);
    end
end

%%% Key Option
if f_key
    [f_key_arr, mix] = formatColumn(f_key_arr);
    if isempty(f_key_arr) || mix
        error(['Key columns are in wrong format! The format must' ...
            ' either be only names or only indices.']);
    end
    
    % set category unification
    f_unicat = 1;
    f_unicat_arr(end + length(f_key_arr)) = f_key_arr;
    f_unicat_arr(findDuplicates(f_unicat_arr, 0, 1)) = [];
end

%%% Add Dates Option
if f_adddates
    if ~strcmp(operation, 'ts')
        error(['Option ''-adddates'' is only avaiable in' ...
            ' time series mode!']);  
    end
    
    l = length(f_adddates_arr);
    if l < 4
        if l < 3
            if l < 2
                f_adddates_arr{2}  = [];
            end
            f_adddates_arr{3}  = [];
        end
        f_adddates_arr{4}  = [];
    end
        
    if (~isempty(f_adddates_arr{2}) && ~isnumeric(f_adddates_arr{2})) ...
            || (~isempty(f_adddates_arr{3}) && ~isnumeric(f_adddates_arr{3})) ...
            || (~isempty(f_adddates_arr{4}) && ~isnumeric(f_adddates_arr{4}))
        error(['Option array for ''-adddates'' is in wrong format!' ...
            ' The cell array must contain a type and an optional range' ...
            ' with an fix-option.']);      
    end
end

%%% Numerical Categories Option
if f_numcat
    if ~isempty(f_numcat_arr)
        f_numcat_arr = formatColumn(f_numcat_arr);
        if isempty(f_numcat_arr)
            error(['Columns for numerical category conversion are in' ...
                ' wrong format! The cell array must contain names or' ...
                ' indices.']);
        end
    end
end

%%% Convert Numerical Categories Back Option
if f_denumcat
    if ~isempty(f_denumcat_arr)
        f_denumcat_arr = formatColumn(f_denumcat_arr);
        if isempty(f_denumcat_arr)
            error(['Columns for numerical category deconversion are in' ...
                ' wrong format! The cell array must contain names or' ...
                ' indices.']);
        end
    end
end

%%% Frequency Conversion Option
if f_freqconv
    if ~strcmp(operation, 'ts')
        error('Option ''-freqconvert'' is only avaiable in time series mode!');
    end
    
    % analyse frequency to convert
    if strcmp(f_freqconv_freq, dates_freq)
        error(['The table contains already data with ' dates_freq ...
            ' frequency! An conversion to this frequency does not make any' ...
            ' sense. You should consider to use the ''-datefreq'' option.']);
    end
    
    % analyse column array
    c_f = 0;
    for i = 1:length(f_freqconv_arr)
        if length(f_freqconv_arr{i}) == 2 && ~isempty(f_freqconv_arr{i}{2})
            f_freqconv_arr{i}{2} = formatColumn(f_freqconv_arr{i}{2});
            if isempty(f_freqconv_arr{i}{2})
                error(['Columns for frequency conversion method ' ...
                    f_freqconv_arr{i}{1} ' are in wrong format!' ...
                    ' The cell array must contain names or indices.']);
            end
        else
            if c_f
                error(['A general frequency conversion method,' ...
                   ' without explicitly setting a cell array,' ...
                   ' was set twice! It does not make any sense!']);
            end
            
            % create empty array
            f_freqconv_arr{i}{2} = [];
            
            c_f = 1;
        end
    end
end

%%% Interpolation Option
if f_interpol
    if ~strcmp(operation, 'ts')
        error(['Option ''-interpolate'' is only avaiable in' ...
            ' time series mode!']);  
    end
    c_f = 0;
    for i = 1:length(f_interpol_arr)
        if ~isempty(f_interpol_arr{i})
            f_interpol_arr{i} = formatColumn(f_interpol_arr{i});
            if isempty(f_interpol_arr{i})
                error(['Columns for interpolation method ' ...
                    f_interpol_method{i} ' are in wrong format!' ...
                    ' The cell array must contain names or indices.']);
            end
        else
            if c_f
                error(['A general interpolation method,' ...
                   ' without explicitly setting a cell array,' ...
                   ' was set twice! It does not make any sense!']);
            end
            c_f = 1;
        end
    end
end

%%% Remove Rows Option
if f_remrows
    for i = 1:length(f_remrows_arr)
        if ~isempty(f_remrows_arr{i})
            f_remrows_arr{i} = formatColumn(f_remrows_arr{i});
            if isempty(f_remrows_arr{i})
                error(['Columns for removing rows method ' ...
                    f_remrows_method{i} ' are in wrong format!' ...
                    ' The cell array must contain names or indices.']);
            end
        end
    end
end

%%% Select By Dates Option
if f_selectbd
    if ~strcmp(operation, 'ts')
        error(['Option ''-selectbydates'' is only avaiable in time series' ...
            ' mode!']);
    end
    
    if isempty(f_selectbd_vec) || ~isnumeric(f_selectbd_vec)
        error(['The dates to select are in wrong format! The dates vector' ...
            ' must contain numeric dates.']);
    end
end

%%% Select Columns Option
if f_selectcols
    f_selectcols_arr = formatColumn(f_selectcols_arr);
    if isempty(f_selectcols_arr)
        error(['The columns to select are in wrong format! The cell array' ...
            ' must contain names or indices.']);
    end
end

%%%%% File and Variable Parsing %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% files, names, extension and delimiters
files = {};
fname = {};
fext = {};
fdel = {};

% get all files recursive; pass trough variables
p = 1;
i = 0;
while i < length(temp)
    i = i + 1;
    if ischar(temp{i})
        % process files and directories
        if isdir(temp{i})
            tfiles = dir(temp{i});
            for k = 1:length(tfiles)
                if strcmp(tfiles(k).name, '.') ...
                        || strcmp(tfiles(k).name, '..')
                    continue
                end
                if tfiles(k).isdir == 0 || f_recursive
                    temp{end + 1} = fullfile(temp{i}, tfiles(k).name);
                    tdel{end + 1} = tdel{i};
                end
            end
            continue
        else
            % extract path, filename and extension
            [~, tname, text] = fileparts(temp{i});
            
            % check file existence
            if exist(temp{i}, 'file') == 0
                error(['File does not exist: ' temp{i}]);
            end
            
            %%% No-MAT-File Option
            if f_nomat
                if strcmp(text, '.mat')
                    continue
                end
            end
            
            %%% Full Path Specification Option
            if f_fullpath
                fname{end + 1} = temp{i};
            else
                fname{end + 1} = tname;
            end
        end
    else
        % proceed Matlab variables
        text = 'var';
        fname{end + 1} = ['var_' num2str(p)];
        p = p + 1;
    end
    
    fdel{end + 1} = tdel{i};
    files{end + 1} = temp{i};
    fext{end + 1} = text;
end

% check for empty input data
if isempty(files)
    % return with empty output
    X = [];
    return
end

% receive unique file identifiers
tind = findDuplicates(fname);
for i = find(tind)
    fname{i} = [fname{i} fext{i}];
end
tind = findDuplicates(fname);
for i = find(tind)
    fname{i} = files{i};
end


%%%%% File and Table Import %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

for i = 1:length(files)
    % display progress
    if f_progress
        if ~strcmp(fext{i}, 'var')
            disp(['Importing data: ' files{i}]);
        end
    end
    
    % determine format
    switch fext{i}
        case {'.xls','.xlsx', '.xlsb', '.xlsm', '.ods', '.xpt'}
            %%% Spreadsheats %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
            D{end + 1} = dataset(fext{i}(2:4), files{i});
            
            %             % catch a warning
            %             if ~isempty(lastwarn)
            %                 error(['Spreadsheat has invalid column names! File: ' ...
            %                     files{i}]);
            %             end
            
            name{end + 1} = fname{i};
            
        case {'.mat', 'var'}
            %%% MATLAB Files and Variables %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
            if ischar(files{i})
                S = load(files{i});
                finames = fieldnames(S);
                S = struct2cell(load(files{i}));
            else
                finames = [];
                S = files(i);
            end
            
            if length(files) > 1
                tname1 = fname{i};
            else
                tname1 = '';
            end
            
            % process arrays
            for j = 1:length(S)
                vararr = S(j);
                tabnames = [];
                % struct type array; '-pertable' Option
                if isstruct(vararr{1})
                    tf = fieldnames(vararr{1});
                    if strcmp(tf{1}, 'tablename')
                        tabnames = vararr{1}.tablename;
                        vararr = vararr{1}.tabledata;
                    end
                end
                
                % set table names
                if isempty(finames)
                    if length(S) > 1
                        tname2 = [tname1 '_' num2str(j)];
                    else
                        tname2 = tname1;
                    end
                else
                    tname2 = [tname1 '_' finames{j}];
                end
                
                % check single table types
                for k = 1:length(vararr)
                    if isa(vararr{k}, 'dataset')
                        % dataset type array
                        D{end + 1} = vararr{k};
                        
                    else
                        if isstruct(vararr{k})
                            tf  = fieldnames(vararr{1});
                            if strcmp(tf{1}, 'colname')
                                D_ind = size(D, 2) + 1;
                                D{D_ind} = dataset([{vararr{k}.coldata} ...
                                    vararr{k}.colname]);
                                D{D_ind}.Properties.UserData = ...
                                    vararr{k}.coldef;
                            else
                                error(['Invalid input variable of type' ...
                                    ' struct!']);
                            end
                        else
                            % get size of cell array
                            s = size(vararr{k});
                            if isnumeric(vararr{k})
                                % generate column identifiers
                                varn = genColName(1, s(2));
                                
                                % matrix type array
                                D{end + 1} = dataset([vararr(k) varn]);
                                
                            elseif iscell(vararr{k})
                                % cell type array
                                
                                % check for string identifiers in first row
                                varn = vararr{k}(1, :);
                                if iscellstr(varn) == 0
                                    % generate column identifiers
                                    varn = genColName(1, s(2));
                                end
                                
                                % generate dataset from cell array
                                D_ind = size(D, 2) + 1;
                                D{D_ind} = dataset( ...
                                    zeros(s(1) - 1, 1));
                                for h = 1:s(2)
                                    % convert numeric to double columns
                                    if all(cellfun(@isnumeric, ...
                                            vararr{k}(2:end, h)))
                                        D{D_ind}(:, h) = ...
                                            dataset(cell2mat( ...
                                            vararr{k}(2:end, h)));
                                    else
                                        D{D_ind}(:, h) = ...
                                            dataset(vararr{k}(2:end, h));
                                    end
                                end
                                
                                % set column identifiers
                                D{D_ind}.Properties.VarNames = varn;
                                
                            else
                                error(['Cannot handle input variable ' ...
                                    fname{i}])
                            end
                        end
                    end
                    
                    % set table names more precise
                    if isempty(tabnames)
                        if length(vararr) > 1
                            tname3 = [tname2 '_' num2str(k)];
                        else
                            tname3 = tname2;
                        end
                    else
                        tname3 = [tname2 '_' tabnames{k}];
                    end
                    if ~isempty(tname3) && strcmp(tname3(1), '_')
                        tname3(1) = [];
                    end
                    
                    % add unique tablename to list
                    name{end + 1} = tname3;
                end
            end
            
        otherwise
            %%% RAW ASCII %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
            
            % delimiter character
            delim = fdel{i};
            
            % check for delimiter
            if isempty(delim)
                % open file for reading
                fid = fopen(files{i}, 'r');
                % read line for determining delimiter
                hline = fgetl(fid);
                while ischar(hline)
                    z = regexp(hline, '\W', 'match');
                    if ~isempty(z) && all(strcmp(z{1}, z))
                        delim = z{1};
                        break;
                    end
                    hline = fgetl(fid);
                end
                % close file
                fclose(fid);
                
                % check whether delimiter was found
                if isempty(delim)
                    error(['File format cannot be recognized: ' files{i}]);
                end
            end
            
            % check for headlines
            h = 0;
            % open file for reading
            fid = fopen(files{i}, 'r');
            % read line for determining delimiter
            hline = fgetl(fid);
            while ischar(hline)
                z = strfind(hline, delim);
                if isempty(z)
                    h = h + 1;
                else
                    break;
                end
                hline = fgetl(fid);
            end
            % close file
            fclose(fid);
            
            % check whether number of headlines were found
            if ~ischar(hline)
                error(['Unknown number of headlines. File format' ...
                    ' cannot be recognized: ' files{i}]);
            end
            
            % check for string headlines
            temp = cellfun(@str2double, textscan(hline, ...
                '%s', 'delimiter', delim), 'UniformOutput', false);
            if all(isnan(temp{1})) == 0
                error(['File has invalid numerical column names! File: ' ...
                    files{i}]);
            end
            
            % check file size and perform split if necessary
            fid = fopen(files{i}, 'r');
            fseek(fid, 0, 'eof');
            pos = ftell(fid);
            mb = pos/(1024*1024);
            
            if mb > 20
                % display progress
                if f_progress
                    % message, that function has to split a large file
                    disp(['The file ' files{i} ' has a size of ' num2str(mb) ...
                        ' MB and will be splitted into ' ...
                        num2str(ceil(mb/20)) ' parts. Afterwards, the ' ...
                        'parts are proceed seperately.']);
                end
                
                % reset file cursor position and read first line
                frewind(fid)
                for j = 1:h+1
                    sline = fgets(fid);
                end;
                hline = sline;
                
                tfile = [files{i} '_temp0'];
                fidw = fopen(tfile, 'w');
                Dt = dataset;
                clim = 20971520;
                bcount = length(sline);
                while ischar(sline)
                    if bcount > clim
                        % close old file
                        fclose(fidw);
                        % read temp file in dataset
                        Dt = vertcat(Dt, dataset('File', tfile, ...
                            'delimiter', delim, 'headerlines', 0));
                        
                        % display progress
                        if f_progress
                            disp(['Importing data: ' files{i} ...
                                ' | ' num2str(bcount/pos*100, ...
                                '%2.2f') ' %' ]);
                        end
                        
                        % overwrite temp file with next part
                        fidw = fopen(tfile, 'w');
                        bcount = bcount + fprintf(fidw, '%s', hline);
                        clim = clim + 20971520;
                    end
                    bcount = bcount + fprintf(fidw, '%s', sline);
                    sline = fgets(fid);
                end
                fclose(fidw);
                
                D{end + 1} = vertcat(Dt, dataset('File', tfile, ...
                    'delimiter', delim, 'headerlines', 0));
            else
                % read delimiter seperated file
                D{end + 1} = dataset('File', files{i}, 'delimiter', ...
                    delim, 'headerlines', h);
            end
            
            % close file
            fclose(fid);
            
            %             % catch a warning
            %             if ~isempty(lastwarn)
            %                 error(['File has invalid column names! File: ' ...
            %                     files{i}]);
            %             end
            
            % add unique tablename to list
            name{end + 1} = fname{i};
    end
end

% determine number of tables
K = size(name, 2);

%%% Per Table Option
if f_pertable
    % determine varoption
    varoption = {};
    
    % flags
    if f_adddates
        varoption{end + 1} = '-adddates';
        varoption{end + 1} = f_adddates_arr;
    end
    if f_cellarray, varoption{end + 1} = '-cellarray'; end
    if f_colnames
        varoption{end + 1} = '-columnnames';
        varoption{end + 1} = f_colnames_arr;
        varoption{end + 1} = f_colnames_names;
    end
    if f_complete, varoption{end + 1} = '-complete'; end
    if f_convdates
        varoption{end + 1} = '-convertdates';
        varoption{end + 1} = f_convdates_arr;
    end
    if f_dateformat
        varoption{end + 1} = '-dateformat';
        varoption{end + 1} = f_dateformat_format;
    end
    if f_datefreq
        varoption{end + 1} = '-datefreq';
        varoption{end + 1} = f_datefreq_freq;
    end
    if f_datestr, varoption{end + 1} = '-datestr'; end
    if f_denumcat
        varoption{end + 1} = '-denumcat';
        if ~isempty(f_denumcat_arr)
            varoption{end + 1} = f_denumcat_arr;
        end
    end
    if f_freqconv
        varoption{end + 1} = '-freqconvert';
        varoption{end + 1} = f_freqconv_freq;
        for i = 1:length(f_freqconv_arr)
            varoption{end + 1} = f_freqconv_arr{i};
        end
    end
    if f_german, varoption{end + 1} = '-german'; end
    if f_interpol
        for i = 1:length(f_interpol_method)
            varoption{end + 1} = '-interpolate';
            varoption{end + 1} = f_interpol_method{i};
            if ~isempty(f_interpol_arr{i})
                varoption{end + 1} = f_interpol_arr{i};
            end
        end
    end
    if f_key
        varoption{end + 1} = '-key';
        varoption{end + 1} = f_key_arr;
    end
    if f_lowercol, varoption{end + 1} = '-lowercolumns'; end
    if f_matrix, varoption{end + 1} = '-matrix'; end
    if f_numcat, varoption{end + 1} = '-numcat'; end
    if f_numcat
        varoption{end + 1} = '-numcat';
        if ~isempty(f_numcat_arr)
            varoption{end + 1} = f_numcat_arr;
        end
    end
    if f_progress, varoption{end + 1} = '-progress'; end
    if f_remrows
        for i = 1:length(f_remrows_method)
            varoption{end + 1} = '-removerows';
            varoption{end + 1} = f_remrows_method{i};
            if ~isempty(f_remrows_arr{i})
                varoption{end + 1} = f_remrows_arr{i};
            end
        end
    end
    if f_selectbd
        varoption{end + 1} = '-selectbydates';
        varoption{end + 1} = f_selectbd_vec;
        varoption{end + 1} = f_selectbd_mode;
    end
    if f_selectcols
        varoption{end + 1} = '-selectcolumns';
        varoption{end + 1} = f_selectcols_arr;
    end
    if f_sortcols, varoption{end + 1} = '-sortcolumns'; end
    if f_sortrows, varoption{end + 1} = '-sortrows'; end
    if f_unicat
        varoption{end + 1} = '-unicat';
        varoption{end + 1} = f_unicat_arr;
    end
    
    % create structure
    C = struct('tablename', {cell(K, 1)}, 'tabledata', {cell(K, 1)});
    for k = 1:K
        C.tablename{k} = name{k};
        C.tabledata{k} = tmigrator(operation, varoption{:}, D{k});
    end
    
    % return output
    X = C;
    return;
end


%%%%% Properties Determining %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% size of data matrices
s = zeros(2, K);

% determine sizes
for k = 1:K
    s(:, k) = size(D{k})';
end


%%%%% Date Conversion Check %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if f_convdates
    % display progress
    if f_progress
        disp('Converting date colums.');
    end
    
    for k = 1:K
        % check whether columns exist in table
        c_ind = getColInd(f_convdates_arr, D{k});
        if isempty(c_ind)
            error(['Date column(s) do not exist! Table: ' name{k}]);
        end
        
        for j = c_ind
            % check whether date has to be converted
            if isnumeric(D{k}{1, j}) == 0
                tdat = cellstr(D{k}, j);
                
                % check for empty fields
                is_empty = strcmp(tdat, '');
                
                % convert to numbers fill empty fields with NaNs
                d_vec = NaN(size(tdat));
                d_vec(~is_empty) = dateconvert(tdat(~is_empty));
                
                % check whether dates were recognized
                if isempty(d_vec)
                    error(['Date format in column ' num2str(j) ' cannot' ...
                        ' be recognized! Table: ' name{k}])
                end
                
                D{k}(:, j) = dataset(d_vec);
            end
        end
    end
end


%%%%% Unique Key Check %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if f_key
    % display progress
    if f_progress
        disp('Checking for key columns.');
    end
    
    for k = 1:K
        % check whether columns exist in table
        c_ind = getColInd(f_key_arr, D{k});
        if isempty(c_ind)
            error(['Key column(s) do not exist! Table: ' name{k}]);
        end
        
        % check for empty fields in key columns
        [c_f, c_row, c_col] = isComplete(D{k}(:, c_ind));
        if ~c_f
            error(['No value in key column(s) ' ...
                D{k}.Properties.VarNames{c_col} ' at row(s) ' c_row '!' ...
                ' Table: ' name{k}]);
        end
        
        % check uniqueness of dates
        if size(unique(D{k}, c_ind), 1) < s(1, k)
            error(['Key column(s) are not unique. Table: ' ...
                name{k}]);
        end
    end
end

%%%%% Table Migration %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%% Uniform Categories Option
if f_unicat
    % generate category cell array
    cind = 1:length(f_unicat_arr);
    c_cats = cell(1, length(f_unicat_arr));
    for k = 1:K
        % migrating categories
        if ~isempty(D{k}.Properties.UserData)
            ind = getColInd(f_unicat_arr, D{k});
            if isempty(ind)
                error(['Category column(s) do not exist! Table: ' name{k}]);
            end
            for j = cind
                if isnumeric(D{k}{1, ind(j)})
                    if isempty(c_cats{j})
                        if ~isempty(D{k}.Properties.UserData{ind(j)})
                            c_cats{j} = D{k}.Properties.UserData{ind(j)};
                        end
                    else
                        if ~isempty(D{k}.Properties.UserData{ind(j)})
                            % verify labels and look for new categories
                            for t = 1:length(D{k}.Properties.UserData{ind(j)})
                                tf = 0;
                                for i = 1:length(c_cats{j})
                                    if strcmpi(c_cats{j}{i}, ...
                                            D{k}.Properties.UserData{ind(j)}{t})
                                        tf = 1;
                                        break;
                                    end
                                end
                                if ~tf
                                    % add new definitions
                                    c_cats{j}(end + 1) = ...
                                        D{k}.Properties.UserData{ind(j)}(t);
                                end
                            end
                        end
                    end
                end
            end
        end
    end
    
    for k = 1:K
        % redefine categorical columns
        if ~isempty(D{k}.Properties.UserData)
            ind = getColInd(f_unicat_arr, D{k});
            for j = cind
                if ~isempty(c_cats{j})
                    if isempty(D{k}.Properties.UserData{ind(j)})
                        td = double(unique(D{k}(:, ind(j))));
                        td(isnan(td)) = [];
                        if ~all(ismember(td, 1:length(c_cats{j})))
                            error(['Unrecognized values found.'...
                                ' Incomplete legend for categories!' ...
                                ' Table:' name{k}]);
                        end
                    else
                        % check whether legends/labels are the same
                        lc = length(c_cats{j});
                        lu = length(D{k}.Properties.UserData{ind(j)});
                        assoc = zeros(lu, 1);
                        for t = 1:lu
                            for i = 1:lc
                                if strcmpi(c_cats{j}{i}, ...
                                        D{k}.Properties.UserData{ind(j)}{t})
                                    if i ~= t
                                        assoc(t) = i;
                                    end
                                    break;
                                end
                            end
                        end
                        if any(assoc)
                            Dt = double(D{k}, ind(j));
                            Dtc = Dt;
                            for t = find(assoc)
                                Dt(Dtc == t) = assoc(t);
                            end
                            D{k}(:, ind(j)) = dataset(Dt);
                        end
                    end
                    % add new definitions
                    D{k}.Properties.UserData{ind(j)} = c_cats{j};
                end
            end
        end
    end
end

% determine columns with maximum number of rows
[rmax, k_max] = max(s(1, :));

%%% Join Option
if f_join
    %%% Perform Join Operation %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    
    % create start dataset with key columns
    ind = getColInd(f_key_arr, D{k_max});
    C = D{k_max}(:, ind);
    l = length(ind);
    cind = k_max*ones(1, l);
    varnames = D{k_max}.Properties.VarNames(ind);
    
    % generate category cell array
    if isempty(D{k_max}.Properties.UserData)
        c_cats = cell(1, l);
    else
        c_cats = D{k_max}.Properties.UserData(ind);
    end
    
    % determine key argument
    if iscellstr(f_key_arr)
        join_keys = f_key_arr;
    else
        % conversion into numerical array
        join_keys = cell2mat(f_key_arr);
    end
    
    for k = 1:K
        % join tables
        C = join(C, D{k}, 'key', join_keys, 'type', f_join_type, ...
            'mergekeys', true);
        
        % set properties for non-key columns
        nonind = diffInd(1:s(2,k), getColInd(f_key_arr, D{k}));
        for j = nonind
            varnames{end + 1} = D{k}.Properties.VarNames{j};
            cind(end + 1) = k;
        end
        
        % add category defintions if existent
        if isempty(D{k}.Properties.UserData)
            c_cats(end + 1:end + length(nonind)) = cell(1, length(nonind));
        else
            c_cats(end + 1:end + length(nonind)) = ...
                D{k}.Properties.UserData(nonind);
        end
    end
    
else
    %%% Perform Integration of Columns %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    
    % create empty dataset
    C = dataset;
    cind = [];
    varnames = {};
    c_cats = {};
    
    for k = 1:K
        for j = 1:s(2, k)
            % determine whether column is numeric or not
            if ischar(D{k}{1, j})
                % fill cell array with empty strings
                B = repmat(cellstr(''), rmax - s(1, k), 1);
            else
                % fill matrix with NaNs
                B = nan(rmax - s(1, k), 1);
            end
            
            % place columns in cell array
            C(:, end + 1) = [D{k}(:, j); ...
                dataset([{B} D{k}.Properties.VarNames(j)])];
            
            varnames{end + 1} = D{k}.Properties.VarNames{j};
            cind(end + 1) = k;
        end
        
        % add category defintions if existent
        if isempty(D{k}.Properties.UserData)
            c_cats(end + 1:end + s(2, k)) = cell(1, s(2, k));
        else
            c_cats(end + 1:end + s(2, k)) = D{k}.Properties.UserData;
        end
    end
end

%%% Add Dates Option
if f_adddates
    % display progress
    if f_progress
        disp('Adding date rows into the table.');
    end
    
    C = addDatesToTable(C, f_adddates_arr{1}, f_adddates_arr{2}, ...
        f_adddates_arr{3}, f_adddates_arr{4});
end

% assign numerical categories
C.Properties.UserData = c_cats;

% determine final size of dataset; values have to be updated manually
[T, K] = size(C);


%%%%% Table Columns Name Specification %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%% Column Names Option
if f_colnames
    % check whether columns exist in table
    c_ind = getColInd(f_colnames_arr, C);
    if isempty(c_ind)
        error(['At least one column for category' ...
            ' renaming does not exist!']);
    end
    varnames(c_ind) = f_colnames_names;
end

%%% Full Name Option
if f_fullname
    vind = 1:K;
else
    % check for duplicate column identifiers
    vind = findDuplicates(varnames);
end
for j = find(vind)
    varnames{j} = [regexprep(name{cind(j)}, '\W', '_') '_' varnames{j}];
end

% set column names
C.Properties.VarNames = varnames;


%%%%% Post Processing Migrated Data %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%% Select Columns Option
if f_selectcols
    % display progress
    if f_progress
        disp('Selecting columns.');
    end
    
    % check whether columns exist in table
    c_ind = getColInd(f_selectcols_arr, C);
    if isempty(c_ind)
        error('At least one column to select does not exist!');
    end
    
    % define new table by selecting the columns
    C = C(:, c_ind);
    
    % update UserData
    C.Properties.UserData = C.Properties.UserData(c_ind);
    
    % update column number
    K = size(C, 2);
end

%%% Numerical Categories Option
if f_numcat
    % display progress
    if f_progress
        disp('Converting string category columns to numbered categories.');
    end
    
    % check for columns array
    if isempty(f_numcat_arr)
        % check all columns
        c_ind = 1:K;
    else
        % check whether columns exist in table
        c_ind = getColInd(f_numcat_arr, C);
        if isempty(c_ind)
            error(['At least one column for numerical category' ...
                ' conversion does not exist!']);
        end
    end
    
    % receive category data
    c_cats = C.Properties.UserData;
    for j = c_ind
        if ischar(C{1, j})
            temp = nominal(cellstr(C(:, j)));
            c_cats{j} = getlabels(temp);
            C(:, j) = dataset(double(temp));
        end
    end
    
    % add information about original string categories
    C.Properties.UserData = c_cats;
end

%%% Convert Numerical Categories Back Option
if f_denumcat
    % display progress
    if f_progress
        disp(['Converting numerical category columns back to numbered' ...
            ' categories.']);
    end
    
    % check for columns array
    if isempty(f_denumcat_arr)
        % check all columns
        c_ind = 1:K;
    else
        % check whether columns exist in table
        c_ind = getColInd(f_denumcat_arr, C);
        if isempty(c_ind)
            error(['At least one column for converting numerical' ...
                'category back does not exist!']);
        end
    end
    
    % receive category data
    c_cats = C.Properties.UserData;
    for j = c_ind
        if ~isempty(c_cats{j})
            temp = double(C, j);
            strtemp = cell(T, 1);
            for i = 1:length(c_cats{j})
                strtemp(temp == i) = c_cats{j}(i);
            end
            % convert NaNs
            strtemp(isnan(temp)) = {''};
            
            C(:, j) = dataset(strtemp);
            
            % empty labels
            c_cats{j} = [];
        end
    end
    
    % set remained information of categories
    C.Properties.UserData = c_cats;
end

%%% Frequency Conversion Option
if f_freqconv
    % get column identifier arrays
    [c_ind, c_ind_remain] = getPreparedIndArrays(f_freqconv_arr, C, 2);
    if isempty(c_ind)
        error(['At least one column for frequency conversion does' ...
            ' not exist!']);
    end
    % check whether remaining columns are left
    if ~isempty(c_ind_remain)
        error(['Cannot convert frequency because some columns have no' ...
            ' conversion rules and therefore cannot be converted!']);
    end
    
    % convert table dates to standard work format: end
    C = convertTableDates(C, dates_freq, dates_format, 'end');
    
    % receive current frequency of data while ensuring the frequency existence
    C = addDatesToTable(C, dates_freq, [], [], 1);
    dt = double(C, 1);

    % distinguish conversion: high --> low  or  low --> high
    if f_freqconv_highlow
        dt_min = min(dt);
    else
        % take beginning interval into account
        dt_min = addtodateFreq(addtodateFreq(min(dt), 1, ...
            f_freqconv_freq), -1, dates_freq);
    end
    % transform frequency of data; fixed start date
    dates = [-Inf; genDates(f_freqconv_freq, dt_min, max(dt), 1)];
        
    % create new matrix for resulting column vectors
    dates_l = size(dates, 1);
    M = zeros(dates_l - 1, K);
    
 
    %%% Successive proceeding of column arrays %%%
    for i = 1:length(c_ind)
        % display progress
        if f_progress
            disp(['Converting numerical data columns. Method: ' ...
                f_freqconv_arr{i}{1}]);
        end
        
        % distinguish conversion: high --> low  or  low --> high
        if f_freqconv_highlow
            for j = c_ind{i}
                % receive current column
                ct = double(C, j);
                
                % get values without NaNs
                ind_nonan = ~isnan(ct);
                dt_nonan = dt(ind_nonan);
                ct_nonan = ct(ind_nonan);
                
                % sort values into an array
                ct_arr = cell(dates_l - 1, 1);
                for k = 2:dates_l
                    ct_arr{k-1} = ct_nonan((dates(k-1) < dt_nonan) ...
                        & (dt_nonan <= dates(k)));
                end
                
                % create new values column vector and apply methods
                cvalues = NaN(dates_l - 1, 1);
                switch f_freqconv_arr{i}{1}
                    case 'first'
                        for k = 1:(dates_l - 1)
                            if ~isempty(ct_arr{k})
                                cvalues(k) = ct_arr{k}(1);
                            end
                        end
                    case 'last'
                        for k = 1:(dates_l - 1)
                            if ~isempty(ct_arr{k})
                                cvalues(k) = ct_arr{k}(end);
                            end
                        end
                    case 'max'
                        for k = 1:(dates_l - 1)
                            if ~isempty(ct_arr{k})
                                cvalues(k) = max(ct_arr{k});
                            end
                        end
                    case 'mean'
                        for k = 1:(dates_l - 1)
                            if ~isempty(ct_arr{k})
                                cvalues(k) = mean(ct_arr{k});
                            end
                        end
                    case 'min'
                        for k = 1:(dates_l - 1)
                            if ~isempty(ct_arr{k})
                                cvalues(k) = min(ct_arr{k});
                            end
                        end
                    case 'sum'
                        for k = 1:(dates_l - 1)
                            if ~isempty(ct_arr{k})
                                cvalues(k) = sum(ct_arr{k});
                            end
                        end
                end
                
                % copy column vector into matrix
                M(:, j) = cvalues;
            end
            
        else
            tf_clone = any(strcmp(f_freqconv_arr{i}{1}, {'clone', ...
                'divclone'}));
            for j = c_ind{i}
                % receive current column
                ct = double(C, j);
                
                % sort values into an array
                cvalues = NaN(dates_l - 1, 1);
                divnumber = NaN(dates_l - 1, 1);
                m = 1;
                n = 1;
                for k = 2:dates_l
                    if dates(k) >= dt(m)
                        cvalues(k-1-tf_clone*(n-1):k-1) = ct(m);
                        divnumber(k-n:k-1) = n;
                        m = m + 1;
                        n = 0;
                    end
                    
                    n = n + 1;
                end
                
                if strcmp(f_freqconv_arr{i}{1}(1:3), 'div')
                    cvalues = cvalues ./ divnumber;
                    t_method = f_freqconv_arr{i}{1}(4:end);
                else
                    t_method = f_freqconv_arr{i}{1};
                end
                
                if any(strcmp(t_method, {'after', 'before', 'linear', ...
                        'nearest'}))
                    cvalues = getInterpolCol(t_method, cvalues, ...
                        dates(2:end));
                end
                
                % copy column vector into matrix
                M(:, j) = cvalues;
            end
        end
    end
    
    
    %%% Final integration of new table data %%%
    
    % insert date column
    M(:, 1) = dates(2:end);
    
    % take new matrix as dataset table
    t_props = C.Properties;
    C = dataset([{M} t_props.VarNames]);
    C.Properties.UserData = t_props.UserData;
    
    % convert dates to table's standard format
    dates_freq = f_freqconv_freq;
    C = convertTableDates(C, dates_freq, 'end', dates_format);
    
    % update row number
    T = size(C, 1);
end

%%% Interpolation Option
if f_interpol
    % get column numbers
    c_ind = getPreparedIndArrays(f_interpol_arr, C);
    if isempty(c_ind)
        error('At least one column for interpolation does not exist!');
    end
    
    % get date column
    dt = double(C, 1);
    
    for i = 1:length(c_ind)
        % display progress
        if f_progress
            disp(['Vertically interpolating numerical data columns. Method: '...
                f_interpol_method{i}]);
        end
        
        for j = c_ind{i}            
            % insert interpolated column into table
            C(:, j) = dataset(getInterpolCol(f_interpol_method{i}, ...
                double(C, j), dt));
        end
    end
end

%%% Remove Rows Option
if f_remrows
    % get column numbers
    c_ind = getIndArrays(f_remrows_arr, C);
    if isempty(c_ind)
        error('At least one column for removing rows does not exist!');
    end
    
    for i = 1:length(c_ind)
        if isempty(c_ind{i})
            if f_key
                c_ind{i} = diffInd(1:K, getColInd(f_key_arr, C));
            else
                c_ind{i} = 1:K;
            end
        end
    end
    
    % check also whether columns are numeric
    c_type = isNumericCol(C);
    
    for i = 1:length(c_ind)
        % display progress
        if f_progress
            disp(['Removing rows from table according to method: '...
                f_remrows_method{i}]);
        end
        
        % distinguish methods
        switch f_remrows_method{i}
            case 'constant'
                j = 1;
                while j < T
                    tf = 1;
                    tf_empty = 1;
                    for k = c_ind{i}
                        if c_type(k)
                            if ~isnan(C{j, k})
                                tf_empty = 0;
                                if C{j, k} ~= C{j + 1, k}
                                    tf = 0;
                                    break
                                end
                            end
                        else
                            if ~isempty(C{j, k})
                                tf_empty = 0;
                                if ~strcmp(C{j, k}, C{j + 1, k})
                                    tf = 0;
                                    break
                                end
                            end
                        end
                    end
                    if tf && ~tf_empty
                        C(j + 1, :) = [];
                        T = T - 1;
                        continue
                    end
                    
                    j = j + 1;
                end
            case 'empty'
                j = 1;
                while j <= T
                    tf = 1;
                    for k = c_ind{i}
                        if (c_type(k) && ~isnan(C{j, k})) ...
                                || (~c_type(k) && ~isempty(C{j, k}))
                            tf = 0;
                            break
                        end
                    end
                    if tf
                        C(j, :) = [];
                        T = T - 1;
                        continue
                    end
                    
                    j = j + 1;
                end
        end
    end
end

%%% Select Columns Option
if f_selectbd
    % display progress
    if f_progress
        disp('Selecting rows by dates.');
    end
    
    if length(unique(f_selectbd_vec)) ~= length(f_selectbd_vec)
        error(['The given date vector contains at least one date twice.' ...
            ' Only unique dates are allowed!']);
    end
    
    % select dates via intersect operation
    [~, ind_dates, ~] = intersect(double(C, 1), f_selectbd_vec);
    
    % check for strict mode
    if strcmp(f_selectbd_mode, 'strict')
        if length(ind_dates) ~= length(f_selectbd_vec)
            error(['Not all dates provided for date selection could be' ...
                ' found in the final table. Maybe you want to disable' ...
                ' the ''strict'' mode.']);
        end
    end
    
    % cut out the selected rows
    C = C(ind_dates, :);

    % update column number
    T = size(C, 1);
end

%%% Completeness Option
if f_complete
    % display progress
    if f_progress
        disp('Checking for completeness of data.');
    end
    
    [c_f, c_row, c_col] = isComplete(C);
    if ~c_f
        error(['No value for ' C.Properties.VarNames{c_col} ...
            ' at row(s) ' c_row '!']);
    end
end

%%% Lower Case Column Option
if f_lowercol
    C.Properties.VarNames = lower(C.Properties.VarNames);
end

%%% Sort Rows Option
if f_sortrows
    % display progress
    if f_progress
        disp('Sorting rows of data.');
    end
    
    % Key Option
    if f_key
        C = sortrows(C, getColInd(f_key_arr, C));
    else
        C = sortrows(C);
    end
end

%%% Date String Format Option
if f_datestr
    % receive column numbers
    c_ind = getColInd(f_convdates_arr, C);
   
    % successive datenum to datestr conversion
    for j = c_ind
        d_vec = double(C, j);
        
        % check for NaNs
        is_nan = isnan(d_vec);
        
        % convert to strings and fill empty fields with ''
        tdat = repmat({''}, size(d_vec));
        tdat(~is_nan) = cellstr(datestr(d_vec(~is_nan)));
        
        % integrate in table
        C(:, j) = dataset(tdat);
    end
end

%%% Sort Columns Option
if f_sortcols
    % display progress
    if f_progress
        disp('Sorting columns of data.');
    end
    
    [~, index] = sort(lower(C.Properties.VarNames));
    
    %%% Key Option
    if f_key
        % always start with key columns
        c_ind = getColInd(f_key_arr, C);
        index = [c_ind diffInd(index, c_ind)];
    end
    
    % reorder columns in dataset
    C = C(:, index);
    C.Properties.UserData = C.Properties.UserData(index);
end


%%%%% Conversions and Return Output %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% reset empty legend array
if all(cellfun(@isempty, C.Properties.UserData))
    C.Properties.UserData = [];
end

%%% Cell Array Output Option
if f_cellarray
    % display progress
    if f_progress
        disp('Preparing to return cell array.');
    end
    
    % warning of labels loss
    if ~isempty(C.Properties.UserData)
        warning(['Legends of numerical categories got lost by' ...
            ' converting to cell array. If you want to keep' ...
            ' them, use a different output format!']); %#ok<WNTAG>
    end
    
    % convert dataset to cell array
    Carr = [C.Properties.VarNames; cell(T, K)];
    for j = 1:K
        if isnumeric(C{1, j})
            Carr(2:end, j) = num2cell(double(C, j));
        else
            Carr(2:end, j) = cellstr(C, j);
        end
    end
    C = Carr;
end

%%% Matrix Output Option
if f_matrix
    % display progress
    if f_progress
        disp('Preparing to return matrix.');
    end
    
    for j = 1:K
        if ~isnumeric(C{1, j})
            error(['Table column ' num2str(j) ' is not numeric.' ...
                ' Cannot generate matrix! Maybe you want to try' ...
                ' the ''-numcat'' option for converting string' ...
                ' columns to numerical category columns.']);
        end
    end
    
    % convert dataset to structured matrix
    C = struct('colname', {C.Properties.VarNames}, ...
        'coldata', {double(C)}, 'coldef', {C.Properties.UserData});
end

% return output
X = C;
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%% HELPER FUNCTIONS %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function E = addDatesToTable(D, d_freq, d_min, d_max, d_fix)
if nargin < 5
    d_fix = [];
    if nargin < 4
        d_max = [];
        if nargin < 3
            d_min = [];
        end
    end
end
if isempty(d_min)
    d_min = min(double(D, 1));
end
if isempty(d_max)
    d_max = max(double(D, 1));
end
if isempty(d_fix)
    d_fix = 0;
end

% generate date vector
dates = genDates(d_freq, d_min, d_max, d_fix);

% integrate dates via join
E = join(D, dataset(dates), 'key', 1, 'type', 'outer', 'mergekeys', true);

% copy back date column name
E.Properties.VarNames{1} = D.Properties.VarNames{1};

% copy back UserData
E.Properties.UserData = D.Properties.UserData;
end

function d_vec = addtodateFreq(d_vec, d_quant, d_freq)
switch d_freq
    case 'daily'
        d_vec = addtodateVec(d_vec, d_quant, 'day');
    case 'wdaily'
        if d_quant < 0
            ts = -1;
        else
            ts = 1;
        end
        for i = 1:abs(d_quant)
            d_vec = addtodateVec(d_vec, ts, 'day');
            
            % step up days if new date in weekend
            w_temp = weekday(d_vec) == 1 | weekday(d_vec) == 7;
            dvec(w_temp) = addtodateVec(dvec(w_temp), 2*ts, 'day');
        end
    case 'weekly'
        d_vec = addtodateVec(d_vec, d_quant*7, 'day');
    case 'monthly'
        d_vec = addtodateVec(d_vec, d_quant, 'month');
    case 'quarterly'
        d_vec = addtodateVec(d_vec, d_quant*3, 'month');
    case 'yearly'
        d_vec = addtodateVec(d_vec, d_quant, 'year');
end
end

function d_vec = addtodateVec(d_vec, d_quant, d_freq)
for i = 1:length(d_vec)
    d_vec(i) = addtodate(d_vec(i), d_quant, d_freq);
end
end

function d_vec = convertDates(d_vec, d_freq, d_from, d_to)
%CONVERTDATES - Convert the dastes from one definition to another.

% return directly if both formats are identical
if strcmp(d_from, d_to)
    return
end

% check whether standard or individual conversion applies
if ischar(d_to)
    switch d_freq
        case 'weekly'
            switch d_from
                case 'begin'
                    if strcmp(d_to, 'middle')
                        d_vec = addtodateVec(d_vec, 2, 'day');
                    else
                        d_vec = addtodateVec(d_vec, 6, 'day');
                    end
                case 'middle'
                    if strcmp(d_to, 'end')
                        d_vec = addtodateVec(d_vec, 4, 'day');
                    else
                        d_vec = addtodateVec(d_vec, -2, 'day');
                    end
                case 'end'
                    if strcmp(d_to, 'begin')
                        d_vec = addtodateVec(d_vec, -6, 'day');
                    else
                        d_vec = addtodateVec(d_vec, -4, 'day');
                    end
            end
        case 'monthly'
            switch d_from
                case 'begin'
                    if strcmp(d_to, 'middle')
                        d_vec = addtodateVec(d_vec, 14, 'day');
                    else
                        d_vec = addtodateVec(addtodateVec(d_vec, 1, ...
                            'month'), -1, 'day');
                    end
                case 'middle'
                    if strcmp(d_to, 'end')
                        d_vec = addtodateVec(addtodateVec(d_vec, 1, ...
                            'month'), -15, 'day');
                    else
                        d_vec = addtodateVec(d_vec, -14, 'day');
                    end
                case 'end'
                    if strcmp(d_to, 'begin')
                        d_vec = addtodateVec(addtodateVec(d_vec, 1, ...
                            'day'), -1, 'month');
                    else
                        d_vec = addtodateVec(addtodateVec(d_vec, 15, ...
                            'day'), -1, 'month');
                    end
            end
        case 'quarterly'
            switch d_from
                case 'begin'
                    if strcmp(d_to, 'middle')
                        d_vec = addtodateVec(addtodateVec(d_vec, 14, ...
                            'day'), 1, 'month');
                    else
                        d_vec = addtodateVec(addtodateVec(d_vec, 3, ...
                            'month'), -1, 'day');
                    end
                case 'middle'
                    if strcmp(d_to, 'end')
                        d_vec = addtodateVec(addtodateVec(d_vec, 2, ...
                            'month'), -15, 'day');
                    else
                        d_vec = addtodateVec(addtodateVec(d_vec, -14, ...
                            'day'), -1, 'month');
                    end
                case 'end'
                    if strcmp(d_to, 'begin')
                        d_vec = addtodateVec(addtodateVec(d_vec, 1, ...
                            'day'), -3, 'month');
                    else
                        d_vec = addtodateVec(addtodateVec(d_vec, 15, ...
                            'day'), -2, 'month');
                    end
            end
        case 'yearly'
            switch d_from
                case 'begin'
                    if strcmp(d_to, 'middle')
                        d_vec = addtodateVec(d_vec, 6, 'month');
                    else
                        d_vec = addtodateVec(addtodateVec(d_vec, 1, ...
                            'year'), -1, 'day');
                    end
                case 'middle'
                    if strcmp(d_to, 'end')
                        d_vec = addtodateVec(addtodateVec(d_vec, 6, ...
                            'month'), -1, 'day');
                    else
                        d_vec = addtodateVec(d_vec, -6, 'month');
                    end
                case 'end'
                    if strcmp(d_to, 'begin')
                        d_vec = addtodateVec(addtodateVec(d_vec, 1, ...
                            'day'), -1, 'year');
                    else
                        d_vec = addtodateVec(addtodateVec(d_vec, 1, ...
                            'day'), -6, 'month');
                    end
            end
    end
else
    % if d_to is a vector with dates to add
    d_vec = addtodateVec(addtodateVec(addtodateVec(d_vec, d_to(3), 'day'), ...
        d_to(2), 'month'), d_to(1), 'year');
end
end

function D = convertTableDates(D, d_freq, str_from, str_to)
D(:, 1) = dataset(convertDates(double(D, 1), d_freq, str_from, str_to));
end

function ind = diffInd(index, subind)
j = 1;
while j <= length(index)
    if any(subind == index(j))
        index(j) = [];
    else
        j = j + 1;
    end
end

ind = index;
end

function ind = findDuplicates(arr, each, logic)
%FINDDUPLICATES - Indicate groups of identical values by integers.
% numeric conversion into string
if isnumeric(arr)
    arr = num2cell(arr);
end
nind = cellfun(@isnumeric, arr);
arr(nind) = cellfun(@num2str, arr(nind), 'UniformOutput', false);

s = size(arr);
if min(s) > 1
    error('findDuplicates only support vectors!');
end

% define standard parameters
if nargin < 3
    logic = 0;
    if nargin < 2
        each = 1;
    end
end

[arr_sorted vind] = sort(arr);
tstr = '';
tind = zeros(s);
tval = 0;
tf = 1;
for j = 1:max(s)
    if length(tstr) == length(arr_sorted{j}) && all(tstr == arr_sorted{j})
        if tf && each
            tind(vind(j - 1)) = tval;
            tf = 0;
        end
        tind(vind(j)) = tval;
    else
        tstr = arr_sorted{j};
        tf = 1;
        tval = tval + 1;
    end
end
if logic
    ind = logical(tind);
else
    ind = tind;
end
end

function [arr mix] = formatColumn(c_arr)
arr = c_arr;
mix = 0;
nind = cellfun(@isnumeric, c_arr);
if ~all(nind)
    if any(nind)
        mix = 1;
    end
    if iscellstr(c_arr(~nind))
        arr(~nind) = lower(c_arr(~nind));
    else
        arr = [];
    end
end
end

function varn = genColName(a, b)
% generate column identifiers
varn = cell(1, b - a + 1);
for i = 1:(b - a + 1)
    varn{i} = ['C' num2str(a + i - 1)];
end
end

function dates = genDates(d_freq, d_min, d_max, f_fix, shift_begin, shift_end)
% generate a date vector from min to max with freq as frequency

% optional parameter handling
if nargin < 6
    shift_end = 0;
    if nargin < 5
        shift_begin = 0;
        if nargin < 4
            f_fix = 0;
        end
    end
end

% month flag
ft_m = 0;

% frequency configutation
switch d_freq
    case 'daily'
        ftype = 'day';
        dates =  addtodate(d_min, shift_begin, ftype);
    case 'wdaily'
        ftype = 'day';
        dates = addtodate(d_min, shift_begin, ftype);
        if f_dix
            switch weekday(dates)
                case {1}
                    dates = addtodate(dates, 1, ftype);
                case {7}
                    dates = addtodate(dates, 2, ftype);
            end
        end
    case 'weekly'
        ftype = 'day';
        dates = addtodate(d_min, shift_begin*7, ftype);
        if f_fix
            dates = addtodate(dates, 8 - weekday(dates), ftype);
        end
    case 'monthly'
        ftype = 'month';
        ft_m = 1;
        dates = addtodate(d_min, shift_begin, ftype);
        if f_fix
            d_vec = datevec(dates);
            dates = datenum([d_vec(1:2) eomday(d_vec(1), d_vec(2))]);
        end
    case 'quarterly'
        ftype = 'month';
        ft_m = 1;
        dates = addtodate(d_min, shift_begin*3, ftype);
        if f_fix
            d_vec = datevec(dates);
            for i = 3:3:12
                if d_vec(2) <= i                    
                    dates = datenum([d_vec(1) i eomday(d_vec(1), i)]);
                    break
                end
            end
        end
    case 'yearly'
        ftype = 'year';
        dates = addtodate(d_min, shift_begin, ftype);
        if f_fix
            d_vec = datevec(dates);
            dates = datenum([d_vec(1) 12 31]);
        end
end

% generation of date vector
d_max = addtodate(d_max, shift_end, ftype);
while dates(end) <= d_max
    dates(end + 1) = addtodate(dates(end), 1, ftype);
end
dates(end) = [];

% transpose to column vector
dates = dates'; 

% correction of monthly data
if f_fix && ft_m
    d_vec = datevec(dates);
    dates = datenum([d_vec(:, 1:2) eomday(d_vec(:, 1), d_vec(:, 2))]);
end

% selection of dates
switch d_freq
    case 'wdaily'
        dates(weekday(dates) == 1 || weekday(dates) == 7) = [];
    case 'weekly'
        dates = dates(1:7:end);
    case 'quarterly'
        dates = dates(1:3:end);
end
end

function ind = getColInd(c_arr, D)
% receives column numbers by an array of column identifiers and a dataset table
l = length(c_arr);
s = size(D, 2);
ind = zeros(1, l);

nind = cellfun(@isnumeric, c_arr);
for i = 1:l
    if nind(i)
        ind(i) = c_arr{i};
        if ind(i) > s
            ind = [];
            return;
        end
    else
        t_ind = find(strcmpi(c_arr{i}, D.Properties.VarNames));
        if isempty(t_ind)
            ind = [];
            return
        else
            ind(i) = t_ind;
        end
    end
end
end

function [ind, ind_remain] = getIndArrays(opt_arr, D, subind)
% receives all column numbers by an option array and a dataset table
if nargin < 3
    subind = [];
end

l = length(opt_arr);
ind_remain = 2:size(D, 2);
ind = cell(1, l);
for i = 1:l
    if isempty(subind)
        temp = opt_arr{i};
    else
        temp = opt_arr{i}{subind};
    end
    if ~isempty(temp)
        ind{i} = getColInd(temp, D);
        if isempty(ind{i})
            ind = [];
            ind_remain = [];
            return
        end
        ind_remain = diffInd(ind_remain, ind{i});
    end
end
end

function c_vec = getInterpolCol(method, c_vec, d_vec)
% size of column
T = size(c_vec, 1);

% if no date vector is given, create one
if nargin < 3
    d_vec = 1:T;
end

% get NaN environment
E = getNaNEnv(c_vec);


% distinguish interpolation method
switch method
    case 'linear'
        % do not apply for border NaNs
        E(E(:, 2) < 1 | E(:, 3) > T, :) = [];
        
        % replace NaN values
        c_vec(E(:, 1)) = c_vec(E(:, 2)) + ((c_vec(E(:, 3)) - c_vec(E(:, 2))) ...
            ./ (d_vec(E(:, 3)) - d_vec(E(:, 2)))) ...
            .* (d_vec(E(:, 1)) - d_vec(E(:, 2)));
    case 'nearest'
        % upper border NaNs have lower value as neighbour
        ind = E(:, 3) > T;
        
        % do only apply to non-border NaNs
        tind = E(:, 2) > 0 & E(:, 3) <= T;
        
        % replace NaN values by nearest neighbour
        ind(tind) = (d_vec(E(tind, 1)) - d_vec(E(tind, 2))) ...
            < (d_vec(E(tind, 3)) - d_vec(E(tind, 1)));
        c_vec(E(ind, 1)) = c_vec(E(ind, 2));
        c_vec(E(~ind, 1)) = c_vec(E(~ind, 3));
    case 'before'
        % do not apply for border NaNs
        E(E(:, 2) < 1, :) = [];
        
        % replace NaN values by values before
        c_vec(E(:, 1)) = c_vec(E(:, 2));
    case 'after'
        % do not apply for border NaNs
        E(E(:, 3) > T, :) = [];
        
        % replace NaN values by values afterwards
        c_vec(E(:, 1)) = c_vec(E(:, 3));
end
end

function E = getNaNEnv(c_vec)
% get the rows of NaNs and the border rows with non-NaNs

% mark NaNs
marker = isnan(c_vec);
marker_l = size(marker, 1);

E = zeros(sum(marker), 3);
l = 0;
h = 0;
k = 1;
for i = 1:marker_l
    if marker(i)
        if h < i
            h = i + 1;
            while h <= marker_l
                if ~marker(h)
                    break;
                end
                h = h + 1;
            end
        end
        E(k, :) = [i l h];
        
        k = k + 1;
    else
        l = i;
    end
end
end

function [c_ind, c_ind_remain] = getPreparedIndArrays(opt_arr, D, subind)
if nargin < 3
    subind = [];
end

% get column numbers
[c_ind, c_ind_remain] = getIndArrays(opt_arr, D, subind);

for i = 1:length(c_ind)
    % check for empty columns array
    if isempty(c_ind{i}) && ~isempty(c_ind_remain)
        % check all remaining columns
        c_ind{i} = c_ind_remain;
        c_ind_remain = [];
    end
    
    % remove non-numerical columns
    for j = 1:length(c_ind{i})
        if ~isnumeric(D{1, c_ind{i}(j)})
            c_ind{i}(j) = [];
        end
    end
end
end

function [f row col] = isComplete(D)
c_type = isNumericCol(D);

for j = 1:size(D, 2)
    if c_type(j)
        ind = datasetfun(@isnan, D(:, j), 'UniformOutput', false);
    else
        ind = datasetfun(@isempty, D(:, j), 'UniformOutput', false);
    end
    if any(ind{1})
        f = 0;
        row = num2str(find(ind{1})');
        col = j;
        return
    end
end
f = 1;
row = [];
col = [];
end

function is_freq = isFreq(d_freq)
is_freq = any(strcmp(d_freq, {'daily', 'wdaily', 'weekly', 'monthly', ...
    'quarterly', 'yearly'}));
end

function c_type = isNumericCol(D, ind)
if nargin < 2
    ind = 1:size(D, 2);
end

c_type = datasetfun(@isnumeric, D(1, ind), 'UniformOutput', true);
end