function retval = Simulator(r,N,T,corrstruct_Theta,corrstruct_Phi,stationary,estimatestruct_Theta,estimatestruct_Phi,distribution)
%Alexander Heinemann
%Efficient Estimation of Factor Models with Time and Cross-Sectional Dependence
%Journal of Applied Econometrics
%
%Subroutine

    close all;
    tic;

    %seed
    randn('state',77); 
    
    %parameters 
    Simul = 10000;
    t_mid = round(T/2);
    i_mid = round(N/2);
    NT_max = 250;
    
    switch corrstruct_Theta
        case 'MA1'
            theta = 0.3;
            Theta = eye(N)+ diag(theta*ones(N-1,1),1) + diag(theta*ones(N-1,1),-1);
            Theta_half_prime = chol(Theta);
            
        case 'AR1'
            theta = 0.5;
            Theta = zeros(N,N);
            for i = 1:(N-1)
               Theta = Theta + diag((theta^i)*ones(N-i,1),i);    
            end;
            Theta = Theta + Theta'+diag(ones(N,1),0);
            Theta_half_prime = chol(Theta);
            
        case 'I_N'
            Theta_half_prime = eye(N);
            
         otherwise
              display('error!');
    end
    
    switch corrstruct_Phi
        case 'MA1'
            phi = 0.3;
            Phi = eye(T)+ diag(phi*ones(T-1,1),1) + diag(phi*ones(T-1,1),-1);
            Phi_half = chol(Phi)';
            
        case 'AR1'
            phi = 0.5;
            Phi = zeros(T,T);
            for t = 1:(T-1)
               Phi = Phi + diag((phi^t)*ones(T-t,1),t);    
            end;
            Phi = (Phi+Phi'+diag(ones(T,1),0));
            Phi_half = chol(Phi)';
            
        case 'I_T'
            Phi_half = eye(T);
            
         otherwise
              display('error!');
    end
        
    
    %storage space
        store_c_hat = zeros(Simul,1);       %GLS
        store_c_tilde = zeros(Simul,1);     %PC
        store_c_bar = zeros(Simul,1);       %B&T
        store_c_star = zeros(Simul,1);      %CH

    %begin of simulations
    k=1;
    while (k<=Simul)

        %generate observation
        switch stationary
           case 'Yes'
               temp = randn(NT_max,r);
               F = temp(1:T,:);
               temp = randn(NT_max,r);
               Lambda = temp(1:N,:); 
           case 'No'
               temp = cumsum(randn(NT_max,r));
               F = temp(1:T,:);
               temp = randn(NT_max,r);
               Lambda = temp(1:N,:);

           otherwise
              display('error!');
        end


        switch distribution
           case 'Normal'
                eps = randn(T,N);
                e = Phi_half*eps*Theta_half_prime;

               case 'Student'
                     df = 10;
                     invPhi = inv(Phi);
                     W = wishrnd(invPhi,df+T-1,D);
                     S_half = chol(W,'upper');
                     eps = randn(T,N);
                     e = S_half\eps*Theta_half;
                     e = sqrt(df-2)*e; 
                     
                case 'BreitungTenhofen'
                    e = randn(T,N);
                    for i=1:N
                        phi_i = 0.4*rand(1)+0.5;
                        sig_i = sqrt(1-(phi_i^2));
                        for t=2:T
                            e(t,i) = phi_i*e(t-1,i)+sig_i*e(t,i);
                        end
                    end
                    
                case 'AR2'
                e = randn(T,N);
                sig = sqrt(1-(phi^2));
                for i=1:round(N/2)
                    for t=2:T
                        e(t,i) = phi*e(t-1,i)+sig*e(t,i);
                    end
                end
                for i=(round(N/2)+1):N
                    for t=3:T
                        e(t,i) = phi*e(t-2,i)+sig*e(t,i);
                    end
                end

           otherwise
              display('error!');
        end
        C = F*Lambda';
        X = C + e;

        %PC estimation
        [U,S,V] = svd(X,'econ');
        F_tilde = sqrt(T)*U(:,1:r);
        Lambda_tilde = X'*F_tilde/T;
        C_tilde = F_tilde*Lambda_tilde';
        e_tilde = X - C_tilde;


        %covariance estimation
        sigma2_hat = var(vec(e_tilde));
        switch estimatestruct_Theta

           case 'AR1' 
                theta_tilde = max(min(sum(sum(e_tilde(:,2:N).*e_tilde(:,1:(N-1)))')/((N-1)*T*sigma2_hat),0.9999),-0.9999);
                Theta_hat = zeros(N,N);
                for i = 1:(N-1)
                   Theta_hat = Theta_hat + diag((theta_tilde^i)*ones(N-i,1),i);    
                end;
                Theta_hat = Theta_hat + Theta_hat'+diag(ones(N,1),0);
                Theta_hat = sigma2_hat* Theta_hat;
                Theta_hat_half_prime = chol(Theta_hat);

           case 'MA1' 
                theta_tilde = sum(sum(e_tilde(:,2:N).*e_tilde(:,1:(N-1)))')/((N-1)*T*sigma2_hat);
                if (abs(theta_tilde) > 0.5)
                    theta_tilde = theta_tilde/(1+theta_tilde^2);
                end      
                Theta_hat = eye(N)+ diag(theta_tilde*ones(N-1,1),1) + diag(theta_tilde*ones(N-1,1),1)';
                Theta_hat_half_prime = chol(Theta_hat);

           otherwise
                display('error!');
        end

        switch estimatestruct_Phi

           case 'AR1' 
                phi_tilde = max(min(sum(sum(e_tilde(2:T,:).*e_tilde(1:(T-1),:))')/(N*(T-1)*sigma2_hat),0.9999),-0.9999);
                Phi_hat = zeros(T,T);
                for t = 1:(T-1)
                   Phi_hat = Phi_hat + diag((phi_tilde^t)*ones(T-t,1),t);    
                end;
                Phi_hat = Phi_hat+Phi_hat'+diag(ones(T,1),0);
                Phi_hat_half = chol(Phi_hat)';

           case 'MA1' 
                phi_tilde = sum(sum(e_tilde(2:T,:).*e_tilde(1:(T-1),:))')/(N*(T-1)*sigma2_hat);
                if (abs(phi_tilde) > 0.5)
                    phi_tilde = phi_tilde/(1+phi_tilde^2);
                end
                Phi_hat = sigma2_hat*(eye(T)+ diag(phi_tilde*ones(T-1,1),1) + diag(phi_tilde*ones(T-1,1),1)');
                Phi_hat_half = chol(Phi_hat)';

           otherwise
                display('error!');
        end

        %second-stage estimation 
        [F_hat, Lambda_hat] = GLS_estimator(X,r,Phi_hat_half,Theta_hat_half_prime,Phi_hat);
        C_hat = F_hat*Lambda_hat';
        [F_star, Lambda_star] = CH_estimator(X,r,Theta_hat_half_prime);
        C_star = F_star*Lambda_star';
                
        switch distribution

            case 'BreitungTenhofen'
                [fols,F_bar,lamols,Lambda_bar] = pc2step(X,r);
                
            otherwise
                [F_bar, Lambda_bar] = BT_estimator(X,r,Phi_hat_half,Phi_hat);     
        end
        
        C_bar = F_bar*Lambda_bar';


        %store results
        store_c_hat(k)   = C_hat(t_mid,i_mid)-C(t_mid,i_mid);       %GLS
        store_c_tilde(k) = C_tilde(t_mid,i_mid)-C(t_mid,i_mid);     %PC
        store_c_bar(k)   = C_bar(t_mid,i_mid)-C(t_mid,i_mid);       %B&T
        store_c_star(k)  = C_star(t_mid,i_mid)-C(t_mid,i_mid);      %CH


        %print iterations to the screen
        k = k+1;
        if(mod(k,round(Simul/10))==0)
            display(k);
        end

    end;

    
    MSE_PCA = mean(store_c_tilde.^2);
    MSE_GLS = mean(store_c_hat.^2);
    MSE_BT = mean(store_c_bar.^2);
    MSE_CH= mean(store_c_star.^2);


    toc;
    if strcmp(stationary,'Yes')
        string2 = '_stationary_';
    else
        string2 = '_nonstationary_';
    end

    retval = [MSE_PCA;MSE_GLS;MSE_BT;MSE_CH];

end