%Alexander Heinemann
%Efficient Estimation of Factor Models with Time and Cross-Sectional Dependence
%Journal of Applied Econometrics
%
%Figure 1

clear;
close all;
clc;

%seed
randn('state',14); 

%parameters
N = 50; T= 50; 
N_max = 100; T_max = 100;
r=1; Simul = 10000; theta = 0.5; phi = 0.5; sigma2 = 1;
delta2_NT = min(N,T);
delta_NT = sqrt(delta2_NT);
sqrt_T = sqrt(T);
sqrt_N = sqrt(N);
t_mid = round(T/2);
i_mid = round(N/2);

%AR1 covariance structure
Phi = zeros(T,T);
for t = 1:(T-1)
   Phi = Phi + diag((phi^t)*ones(T-t,1),t);    
end
Phi = Phi+Phi'+diag(ones(T,1),0);
Phi_half = chol(Phi)';
Theta = zeros(N,N);
for i = 1:(N-1)
   Theta = Theta + diag((theta^i)*ones(N-i,1),i);    
end;
Theta = Theta + Theta'+diag(ones(N,1),0);
Theta_half_prime = chol(Theta); 

F=randn(T,r);
Lambda = randn(N,r);

%calculate the asymptotic variance of the GLS common component
V_f_t_hat = Phi(t_mid,t_mid)/(Lambda'*(Theta\Lambda)/N);
V_lambda_i_hat = Theta(i_mid,i_mid)/(F'*(Phi\F)/T);
V_c_it_hat = (delta2_NT/N)*Lambda(i_mid,:)'*V_f_t_hat*Lambda(i_mid,:)+(delta2_NT/T)*F(t_mid,:)'*V_lambda_i_hat*F(t_mid,:);

%calculate the asymptotic variance of the PC common component
V_f_t_tilde = Phi(t_mid,t_mid)*(Lambda'*Lambda/N)\((Lambda'*(Theta\Lambda)/N)/(Lambda'*Lambda/N));
V_lambda_i_tilde = Theta(i_mid,i_mid)*(F'*F/T)\((F'*(Phi\F)/T)/(F'*F/T));
V_c_it_tilde = (delta2_NT/N)*Lambda(i_mid,:)'*V_f_t_tilde*Lambda(i_mid,:)+(delta2_NT/T)*F(t_mid,:)'*V_lambda_i_tilde*F(t_mid,:);

%storage space
store_c_it_hat_stand = zeros(Simul,1);
store_c_it_tilde_stand = zeros(Simul,1);

%begin of simulations
k=1;
while (k<Simul)
       
    %generate observation
    eps = randn(T,N);
    e = Phi_half*eps*Theta_half_prime;  %normal
    C = F*Lambda';
    X = C + e;
    
    %PC estimation
    [U,S,V] = svd(X,'econ');
    F_tilde = sqrt_T*U(:,1:r);
    Lambda_tilde = X'*F_tilde/T;
    C_tilde = F_tilde*Lambda_tilde';
    e_tilde = X - C_tilde;
     
    %Covariance estimation
    sigma2_hat = var(vec(e_tilde));
    phi_hat = max(min(sum(sum(e_tilde(2:T,:).*e_tilde(1:(T-1),:))')/(N*(T-1)*sigma2_hat),0.9999),-0.9999);
    theta_hat = max(min(sum(sum(e_tilde(:,2:N).*e_tilde(:,1:(N-1)))')/((N-1)*T*sigma2_hat),0.9999),-0.9999);
    
    Phi_hat = zeros(T,T);
    for t = 1:(T-1)
       Phi_hat = Phi_hat + diag((phi_hat^t)*ones(T-t,1),t);    
    end;
    Phi_hat = Phi_hat+Phi_hat'+diag(ones(T,1),0);
    Phi_hat_half = chol(Phi_hat)';
    
    Theta_hat = zeros(N,N);
    for i = 1:(N-1)
       Theta_hat = Theta_hat + diag((theta_hat^i)*ones(N-i,1),i);    
    end;
    Theta_hat = Theta_hat + Theta_hat'+diag(ones(N,1),0);
    Theta_hat = sigma2_hat* Theta_hat;
    Theta_hat_half_prime = chol(Theta_hat);
        
    %GLS estimation
    Y = Phi_hat_half\X/Theta_hat_half_prime;
    [U,S,V] = svd(Y,'econ');
    F_hat = sqrt_T*Phi_hat_half*U(:,1:r);
    Lambda_hat = X'*(Phi_hat\F_hat)/T;
    C_hat =  F_hat*Lambda_hat';
    
    %store results
    store_c_it_hat_stand(k,:) = delta_NT*(C_hat(t_mid,i_mid)-C(t_mid,i_mid))/sqrt(V_c_it_hat);
        
    %print iterations to the screen
    k = k+1;
    if(mod(k,round(Simul/10))==0)
       display(k);
    end
end;

%Histograms with superimposed standard normal density
color1 = [0.1,0.1,0.1];
color2 = [0.5,0.5,0.5];
edges = (-5:0.5:5);
close all;
fig = figure(1);
h1 = histogram(store_c_it_hat_stand,edges,'Normalization','pdf');
h1.FaceColor = [1,1,1];
hold on
x = linspace(-10,10,200);
normal_density = normpdf(x);
h3 = plot(x,normal_density,'-','LineWidth',2,'Color',color1);
legend('GLS','N(0,1)')
set(fig,'Color',[1 1 1]);
set(gca,'fontname','times','fontsize',12);
set(gca,'YTick',[0.05 0.1 0.15 0.2 0.25 0.3 0.35 0.4 0.45 0.5 0.55]);
axis([-4 4 0 0.42]);