%Alexander Heinemann
%Efficient Estimation of Factor Models with Time and Cross-Sectional Dependence
%Journal of Applied Econometrics
%
%Application: Lee-Carter Revisited
close all;
clear;
clc;

% Step 1: read in the data, reshape and select sample
    filename = 'Atlantis.txt';
    fid = fopen(filename);
    buffer =fscanf(fid,'%c');
    buffer = strrep(buffer,'+',' ') ;           % Eliminate +
    buffer = strrep(buffer,' .','-1') ;         % Replace . entries with -1
    data = cell2mat(textscan(buffer,'%f %f %f %f %f', Inf,'HeaderLines',3));
    m = data(:,5);                              % 3 female, 4 male, 5 total
    datastart = data(1,1);
    samplestart = 1925;
    sampleend = 2009;
    ageend = 94;
    m = reshape(m,111,length(m)/111)';
    m = m(samplestart-datastart+1:sampleend-datastart+1,1:(ageend+1));
    [T,N] = size(m);
    logm = log(m);

    
% Step 2: estimate r, the number of common components (reference: Bai 2004)
    D = zeros(T-1,T)+([(-eye(T-1)) zeros(T-1,1)])+([zeros(T-1,1) (eye(T-1))]);
    Q = eye(T-1)-(ones(T-1,T-1)/(T-1));
    DeltaX = Q*D*logm;
    kmax = 6;                                 %max. number considered in the literature
    alpha_T = T/(4*log(log(T)));
    [Uk,Sk,Vk] = svd(DeltaX,'econ');
    Lambda_k = Vk(:,1:kmax)*sqrt(Sk(1:kmax,1:kmax));
    F_k = Uk(:,1:kmax)*sqrt(Sk(1:kmax,1:kmax));
    e_k = DeltaX- F_k*Lambda_k';
    sigma2_kmax = sum(sum(e_k.^2),2)/(T*N); 
    criterion = zeros(kmax,3);

    for k = 1:kmax

        [Uk,Sk,Vk] = svd(DeltaX,'econ');
        Lambda_k = Vk(:,1:k)*sqrt(Sk(1:k,1:k));
        F_k = Uk(:,1:k)*sqrt(Sk(1:k,1:k));

        e_k = DeltaX- F_k*Lambda_k';
        V_k = sum(sum(e_k.^2),2)/(N*T);
        criterion(k,1) = V_k+k*sigma2_kmax*alpha_T*((N+T)/(N*T))*log(min(N,T));
        criterion(k,2) = V_k+k*sigma2_kmax*alpha_T*((N+T)/(N*T))*log(N*T/(N+T));
        criterion(k,3) = V_k+k*sigma2_kmax*alpha_T*((N+T-k)/(N*T))*log(N*T);

    end
    [temp,kstar] = min(criterion);
    display(kstar);
    display(criterion);
    r = kstar(1,1);                            % we take the first criterion function for the decision
    display(r);                                % Note that all criterion functions suggest r = 1    
    
    
% Step 3: we test for nonstationarity in the pervasive series (reference:
% Bai and Ng, 2004)
    logm_restr = logm(1946-samplestart:end,:); % post-WWII period
    T_restr = size(logm_restr,1);
    N_restr = size(logm_restr,2);
    D_restr = zeros(T_restr-1,T_restr)+([(-eye(T_restr-1)) zeros(T_restr-1,1)])+([zeros(T_restr-1,1) (eye(T_restr-1))]);
    Q_restr = eye(T_restr-1)-(ones(T_restr-1,T_restr-1)/(T_restr-1));
    DeltaX_restr = Q_restr*D_restr*logm_restr;
    [U_restr,S_restr,V_restr] = svd(DeltaX_restr,'econ');    
    f_restr = U_restr(:,1:r)*sqrt(S_restr(1:r,1:r));
    F_restr = cumsum(f_restr);
    [h,pValue,stat,cValue,reg] = adftest(F_restr);  
    if (h==1)
       display('We reject the null hypothesis of nonstationarity'); 
    else
        display('We accept the null hypothesis of nonstationarity'); 
    end


% Step 4: time-series demeaned log central death 
    alpha_hat = mean(logm);
    X = logm - ones(T,1)*alpha_hat;

    
% Step 5: PCA estimation
    [U,S,V] = svd(X,'econ');
    Lambda_tilde = -V(:,1:r)*sqrt(S(1:r,1:r));
    F_tilde = -U(:,1:r)*sqrt(S(1:r,1:r));
    e_tilde = X- F_tilde*Lambda_tilde'; 

% Step 6: Figure 2
    % Subfigure 2a: central death rate in 3-D surf plot
    fig2a = figure (1);
    x = linspace(0,N-1,N)';
    y = linspace(samplestart,sampleend,T)';
    [xx,yy] = meshgrid(x,y);
    graph = surf(xx,yy,m);
    set(graph,'facecolor','none')
    brighten(0.1)
    set(gca,'fontname','times','fontsize',18);
    xlabel('Age','fontname','times','fontsize',18)
    ylabel('Time','fontname','times','fontsize',18)
    zlabel('Central Death Rate','fontname','times','fontsize',18)
    axis tight
    set(fig2a,'Color',[1 1 1])

    % Subfigure 2b: residual scatter plot in 3-D
    fig2b = figure (2);
    xx = ones(T,1)*linspace(0,N-1,N);
    xx = xx(:);
    yy = linspace(samplestart,sampleend,T)'*ones(1,N);
    yy = yy(:);
    zz = e_tilde;
    zz = zz(:);
    S = 21*ones(size(zz));
    C = yy/N;
    scatter3(xx,yy,zz,S,C,'.')
    set(gca,'fontname','times','fontsize',18);
    xlabel('Age','fontname','times','fontsize',18)
    ylabel('Time','fontname','times','fontsize',18)
    axis tight
    set(fig2b,'Color',[1 1 1])
    

% Step 7: calculate heteroskedasticity and correlation structure
    sigma2_hat = zeros(N,1);
    for i = 1:N
       sigma2_hat(i) = e_tilde(:,i)'*e_tilde(:,i)/N;  
    end
    sigma_hat = sigma2_hat.^0.5;
    e_star = e_tilde./(ones(T,1)*sqrt(sigma2_hat'));                        %heteroskedasticity adjusted errors

    phi_hat = cumprod(sum(sum(e_star(2:end,:).*e_star(1:end-1,:)),2)/(N*(T-1))*ones(T-1,1));
    theta_hat = cumprod(sum(sum(e_star(:,2:end).*e_star(:,1:end-1)),2)/(T*(N-1))*ones(N-1,1));
    Theta_hat = zeros(N,N);
    for i = 1:(N-1)
       Theta_hat = Theta_hat + diag((theta_hat(i))*ones(N-i,1),i);    
    end;
    Theta_hat = Theta_hat + Theta_hat'+diag(ones(N,1),0);
    Theta_hat_hetero = Theta_hat.*(sigma_hat*sigma_hat');
    Theta_hat_hetero_half = chol(Theta_hat_hetero)';
    Phi_hat = zeros(T,T);
    for t = 1:(T-1)	
       Phi_hat = Phi_hat + diag((phi_hat(t))*ones(T-t,1),t);    
    end;
    Phi_hat = Phi_hat + Phi_hat' + diag(ones(T,1),0);
    Phi_hat_half = chol(Phi_hat);
    varepsilon = (Phi_hat_half\e_tilde)/Theta_hat_hetero_half;
    Theta_hat_hetero_half = std(varepsilon(:))*Theta_hat_hetero_half/sqrt(1.66);     %to standardize such that varepsilon has unit variance
    varepsilon = (Phi_hat_half\e_tilde)/Theta_hat_hetero_half;
    
   
% Step 8: calculate the GLS estimator (rotated for the sake of comparison)
    Y1 = Phi_hat_half\X/Theta_hat_hetero_half;
    [U1,S1,V1] = svd(Y1','econ');
    G_hat = V1(:,1:r)*sqrt(S1(1:r,1:r));
    Gamma_hat = U1(:,1:r)*sqrt(S1(1:r,1:r));
    F_hat = Phi_hat_half*G_hat;
    Lambda_hat = Theta_hat_hetero_half'*Gamma_hat;
    e_hat = X-F_hat*Lambda_hat';
    rotation = (F_hat'*F_hat)\(F_hat'*F_tilde);
    F_hat_rotated = rotation*F_hat;
    Lambda_hat_rotated = Lambda_hat/rotation;
    
    
 % Step 9: Table 2   
    display([(1939:1:1946)' F_hat_rotated(1939-samplestart+1:1946-samplestart+1) F_tilde(1939-samplestart+1:1946-samplestart+1)]);
    
    
% Step 10: Figure 3
    % Subfigure 3a: factor estimates comparison in 2-D plot
    x = linspace(samplestart,sampleend,T)';
    fig3a = figure (3);
    figplot3a = plot(x,F_hat_rotated,'-',x,F_tilde,'--');
    set(figplot3a,'linewidth',3);
    set(figplot3a,{'color'},{[0,0,0.6];[0,0.6,0]});
    legend('GLS','PCA');
    set(gca,'fontname','times','fontsize',18);
    xlabel('Time','fontname','times','fontsize',18)
    axis([2009-T+1-1, 2009+1, 1.1*min([F_tilde; F_hat_rotated]), max([F_tilde; F_hat_rotated])*1.1]);
    set(fig3a,'Color',[1 1 1])
    
    % Subfigure 3a: factor loading estimates comparison in 2-D plot 
    x = (0:1:N-1)';
    fig3b = figure (4);
    figplot3b = plot(x,Lambda_hat_rotated,'-',x,Lambda_tilde,'--');
    set(figplot3b,'linewidth',3);
    set(figplot3b,{'color'},{[0,0,0.6];[0,0.6,0]});
    legend('GLS','PCA');
    set(gca,'fontname','times','fontsize',18);
    xlabel('Age','fontname','times','fontsize',18)
    axis([0-1, N+1, 1.1*min([Lambda_tilde; Lambda_hat_rotated]), max([Lambda_tilde; Lambda_hat_rotated])*1.1]);
    set(fig3b,'Color',[1 1 1])

    
% Step 11: Moran's I
    z = e_tilde;
    numerator=0;
    for t=1:T
       for i=1:(N-1)
           numerator = numerator+ z(t,i)*z(t,i+1);
       end
    end
    for i=1:N
       for t=1:(T-1)
           numerator = numerator+ z(t,i)*z(t+1,i);
       end
    end
    denominator = vec(z)'*vec(z);
    Morans_I = numerator/denominator;
    E_I = -(2*T*N-T-N)/(T*N*(T*N-1));
    E2_I = (2*(T^3)*(N^3)-(T^3)*(N^2)-(T^2)*(N^3)-4*(T^2)*(N^2)+2*(T^2)*N+2*T*(N^2)-2*N*T+3*T^2+3*N^2)/((T^2)*(N^2)*(T*N-1)*(T*N+1));
    var_I = E2_I-E_I^2;
    Morans_I_normalized = (Morans_I-E_I)/sqrt(var_I);
    display(Morans_I_normalized);

% Step 12: Export estimated factors to a csv-file
time = linspace(samplestart,sampleend,T)';
csvwrite('statatransfer.csv',[time,F_tilde, F_hat_rotated]);
    
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%Life expectancy forecast simulations%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Step 13: Run Stata file to generate Table 3


    %PCA: coefficients from Stata after fitting an ARIMA and Asymptotic
    %Distribution
        drift_PCA_mean = -0.0328085;         drift_PCA_std = 0.0064804;
        AR1_PCA_mean = -0.02613294;          AR1_PCA_std = 0.0635776;
        sigma_PCA_mean = 0.0590477;          sigma_PCA_std = 0.0033069;
        
    %GLS: coefficients from Stata after fitting an ARIMA and Asymptotic
    %Distribution
        drift_GLS_mean = -0.0309007;        drift_GLS_std = 0.0053038; 
        AR1_GLS_mean = -0.5026;          AR1_GLS_std = 0.0711695;
        sigma_GLS_mean = 0.064422;         sigma_GLS_std = 0.0043244;

    %set parameters        
    horizon = 151;   %forecast horizon
    Simul = 5000;    %number of simulations

    F_Tplusk_PCA = zeros(horizon+2,1);
    W_F_Tminus1_PCA = Phi_hat_half(:,T-1)'*Phi_hat_half(:,T-1)*((Lambda_tilde'*Lambda_tilde)\((Lambda_tilde'*(Theta_hat\Lambda_tilde))\(Lambda_tilde'*Lambda_tilde)))/N;
    W_F_T_PCA = Phi_hat_half(:,T)'*Phi_hat_half(:,T)*((Lambda_tilde'*Lambda_tilde)\((Lambda_tilde'*(Theta_hat\Lambda_tilde))\(Lambda_tilde'*Lambda_tilde)))/N;       
    M_hat = (Phi_hat_half'*Phi_hat_half)*G_hat;
    W_Lambda_PCA = diag(Theta_hat_hetero_half'*Theta_hat_hetero_half)*sum((M_hat(1:T-1,1).*M_hat(2:T,1)).^2)/(T^2);    %vector

    F_Tplusk_GLS = zeros(horizon+2,1);
    W_F_Tminus1_GLS = Phi_hat_half(:,T-1)'*Phi_hat_half(:,T-1)/(Lambda_hat_rotated'*(Theta_hat\Lambda_hat_rotated))/N;
    W_F_T_GLS = Phi_hat_half(:,T)'*Phi_hat_half(:,T)/(Lambda_hat_rotated'*(Theta_hat\Lambda_hat_rotated))/N;
    W_Lambda_GLS = diag(Theta_hat_hetero_half'*Theta_hat_hetero_half)*sum((G_hat(1:T-1,1).*G_hat(2:T,1)).^2)/(T^2);    %vector


    Tnew = T + horizon;
     Phi_hat_new = zeros(Tnew,Tnew);
    for t = 1:15	%we truncate since phi_hat(1)^16=2.1772e-04
       Phi_hat_new = Phi_hat_new + diag((phi_hat(t))*ones(Tnew-t,1),t);
    end;
    Phi_hat_new = Phi_hat_new+Phi_hat_new'+diag(ones(Tnew,1),0);
    Phi_hat_new_half = chol(Phi_hat_new);

    randn('state',77);  

    %storage space
    store_PCA_life_expec = zeros(Simul,Tnew,N);
    store_GLS_life_expec = zeros(Simul,Tnew,N);

    
% Step 14: simulate future death rates and calculate life expectancy   
tic;
for s = 1:Simul
    
    %print to screen the process
    if(mod(s,round(Simul/100))==0)
        display(s);
    end
    
    %PCA Simulations 
        rand_param = randn(3,1);
        drift_PCA = drift_PCA_mean + drift_PCA_std*rand_param(1,1);
        AR1_PCA = AR1_PCA_mean + AR1_PCA_std*rand_param(2,1);
        sigma_PCA = max(sigma_PCA_mean + sigma_PCA_std*rand_param(3,1),0); 
        
        F_Tplusk_PCA(1,1)  = F_tilde(T-1,1)+ randn(1,1)*sqrt(W_F_Tminus1_PCA);
        F_Tplusk_PCA(2,1)  = F_tilde(T,1)+ randn(1,1)*sqrt(W_F_T_PCA);
        for q = 3:(horizon+2)
            F_Tplusk_PCA(q,1) = drift_PCA + (1+AR1_PCA)*F_Tplusk_PCA(q-1,1)-AR1_PCA*F_Tplusk_PCA(q-2,1)+sigma_PCA*randn(1,1);
        end
        random = ceil(N*T*rand(horizon,N));
        random2 = randn(N,1);
        Lambda_tilde_PCA = Lambda_tilde + sqrt(W_Lambda_PCA).*random2;
        X_tilde = [F_tilde; F_Tplusk_PCA(3:end)]*Lambda_tilde_PCA'+ [e_tilde; e_tilde(random)];
        logm_PCA = X_tilde + ones(Tnew,1)*mean(logm);
        m_PCA = exp(logm_PCA);
        prob_survive_PCA = exp(-m_PCA);
        %life expectancy calculations
        maxTnewN = max(Tnew,N);
        temp_PCA = zeros(maxTnewN,maxTnewN);
        temp = zeros(maxTnewN,maxTnewN);
        temp(1:Tnew,1:N) = prob_survive_PCA;
        for index = (-(maxTnewN-1)):(maxTnewN-1)
            temp2 = diag(temp,index);
            temp2 = cumprod(temp2);
            temp_PCA = temp_PCA + diag(temp2,index);
        end
        temp_PCA = temp_PCA(1:Tnew,1:N);
        life_expec_PCA = zeros(Tnew,N);
        for t=1:Tnew
            for i=1:N
                temp3 = 0;
                for k = 1:min(Tnew-t,N-i)
                    temp3 = temp3 + ((1-prob_survive_PCA(t+k,i+k))/m_PCA(t+k,i+k))*temp_PCA(t+k-1,i+k-1);
                end
                life_expec_PCA(t,i) = ((1-prob_survive_PCA(t,i))/m_PCA(t,i))+temp3;
            end
        end
        store_PCA_life_expec(s,:,:) = life_expec_PCA;


            %GLS Simulations
                drift_GLS = drift_GLS_mean + drift_GLS_std*rand_param(1,1);
                AR1_GLS = AR1_GLS_mean + AR1_GLS_std*rand_param(2,1);
                sigma_GLS = max(sigma_GLS_mean + sigma_GLS_std*rand_param(3,1),0); 
                F_Tplusk_GLS(1,1)  = F_tilde(T-1,1)+ randn(1,1)*sqrt(W_F_Tminus1_GLS);
                F_Tplusk_GLS(2,1)  = F_tilde(T,1)+ randn(1,1)*sqrt(W_F_T_GLS);
                for q = 3:(horizon+2)
                    F_Tplusk_GLS(q,1) = drift_GLS + (1+AR1_GLS)*F_Tplusk_GLS(q-1,1)-AR1_GLS*F_Tplusk_GLS(q-2,1)+sigma_GLS*randn(1,1);
                end
                
                Lambda_hat_rotated_GLS = Lambda_hat_rotated + sqrt(W_Lambda_GLS).*random2;
                X_hat = [F_hat_rotated; F_Tplusk_GLS(3:end)]*Lambda_hat_rotated_GLS'+ Phi_hat_new_half*[varepsilon;varepsilon(random)]*Theta_hat_hetero_half;
                logm_GLS = X_hat + ones(Tnew,1)*mean(logm);
                m_GLS = exp(logm_GLS);
                prob_survive_GLS = exp(-m_GLS);
                %life expectancy calculations
                maxTnewN = max(Tnew,N);
                temp_GLS = zeros(maxTnewN,maxTnewN);
                temp = zeros(maxTnewN,maxTnewN);
                temp(1:Tnew,1:N) = prob_survive_GLS;
                for index = (-(maxTnewN-1)):(maxTnewN-1)
                    temp2 = diag(temp,index);
                    temp2 = cumprod(temp2);
                    temp_GLS = temp_GLS + diag(temp2,index);
                end
                temp_GLS = temp_GLS(1:Tnew,1:N);
                life_expec_GLS = zeros(Tnew,N);
                for t=1:Tnew
                    for i=1:N
                        temp3 = 0;
                        for k = 1:min(Tnew-t,N-i)
                            temp3 = temp3 + ((1-prob_survive_GLS(t+k,i+k))/m_GLS(t+k,i+k))*temp_GLS(t+k-1,i+k-1);
                        end
                        life_expec_GLS(t,i) = ((1-prob_survive_GLS(t,i))/m_GLS(t,i))+temp3;
                    end
                end
                store_GLS_life_expec(s,:,:) = life_expec_GLS;
end
toc;

%generate life expectancy tables
store_PCA_life_expec_mean = zeros(Tnew,N);
store_GLS_life_expec_mean = zeros(Tnew,N);
for t=1:Tnew
    for i=1:N
        store_PCA_life_expec_mean(t,i) = mean(store_PCA_life_expec(:,t,i));
        store_GLS_life_expec_mean(t,i) = mean(store_GLS_life_expec(:,t,i));
    end
end


% Step 15: Table 4
display(store_GLS_life_expec_mean(96:10:136,1:10:91)');
display(store_PCA_life_expec_mean(96:10:136,1:10:91)');


%Step 16: country values for Figure 4
display(store_GLS_life_expec_mean(91,1)');
display(store_PCA_life_expec_mean(91,1)');



