###--------------------------------------------------------------------------------------------------------###
###---------------------- Conjugate BVAR function ---------------------------------------------------------###
###--------------------------------------------------------------------------------------------------------###

conjVAR.func <- function(Yraw = Yraw, hyperpara = hyperpara, dummy.type = dummy.type, nsave =nsave, nburn = nburn, nthin = nthin, p, cons = cons, ML.prior = ML.prior, trans = trans)
{
  #---------------------------------------------------Data Settings---------------------------------
  
  theta.prior <- ML.prior$theta.prior
  w.1.prior <- ML.prior$w.1.prior
  w.2.prior <- ML.prior$w.2.prior

  fast <- TRUE
  ntot <- nthin*nsave+nburn
  save.set <- seq(nburn+nthin, ntot, nthin)
  save.ind <- 0
  
  if(cons){
    X <- cbind(mlag(Yraw,p),1)
    Xfull <- cbind(mlag(Yraw,p),1)
  }else{
    X <- mlag(Yraw,p)
    Xfull <- mlag(Yraw,p)
  }
  
  Y <- Yraw[(p+1):nrow(Yraw),]
  X <- X[(p+1):nrow(X),]
  
  T <- nrow(X)
  K <- ncol(X)
  M <- ncol(Y)
  n <- K*M
  v <- (M*(M-1))/2
  
  if(trans == "I0") prmean <- 0 else if(trans == "I1") prmean <- 0.9
  
  #------------------------------------------getting OLS estimates--------------------------------------------- 
  XtXinv <- MASS::ginv(crossprod(X))
  A_OLS <- A.draw <-  XtXinv%*%crossprod(X,Y)
  a_OLS <- as.vector(A_OLS) #vectorizes A_OLS, i.e. a_OLS=vec(A_OLS)
  Em  <- Em.str <- Y-X%*%A_OLS
  SSE  <-  crossprod(Em)
  if(K < T){
    Sigma.draw <- SIGMA_OLS  <-  SSE/(T-K+1) 
  }else{
    Sigma.draw <- SIGMA_OLS  <-  SSE/(T+1) # T+K-K+1
  }

  if(!OLS){
      #------------------------------------------optimise hyperparameters --------------------------------------------- 
      gamma.prior <- 0.01
      ## Prior DoF
      #n0 <- 10*M
      #n0 <- M + 2 # see GLP, 2015
      if(substr(type,1,3) %in% c("OLS")){
        n0 <- M + 2
      }else{
        #n0 <- nrow(YY) - nrow(Y) + M + 2 # see Banbura et al., 2010
        n0 <- M + 2 
      }
      
      #Start constructing a VAR prior
      #Step A: Run a set of AR(p) models
      mysigma <- matrix(NA,M,1)
      for (ii in 1:M){
        Y.i <- Yraw[,ii,drop=F]
        X.i <- mlag(Yraw[,ii],p)
        Y.i <- Y.i[(p+1):nrow(Y.i),,drop=F]
        X.i <- X.i[(p+1):nrow(X.i),,drop=F]
        
        rho.i <- ginv(crossprod(X.i))%*%crossprod(X.i,Y.i)
        er.i <- Y.i-X.i%*%rho.i
        SSE.i <- crossprod(er.i)/(nrow(X.i)-p)
        
        #  tmpar <- arima(Y[,ii],order=c(1,0,0),method="ML")
        mysigma[ii,1] <- sqrt(SSE.i)#sqrt(tmpar$sigma2)
      }  

#------------------------------------------ Create dummy matrices ------------------------------------
      print(hyperpara)
      
      #Specify hyperparameter on the minnesota prior and on the constant
      #these will the arguments of the function
      shrink.1 <- hyperpara[["shrink.1"]]
      dummies.min <- get.dum.min(shrink.1,gamma.prior,prmean,mysigma,M,p)
      Y.min <- dummies.min$Ydum
      X.min <- dummies.min$Xdum
      
      if(dummy.type == "MIN-SOC-SUR"){
        w.1 <- hyperpara[["w.1"]]
        dummies.soc <- get.dum.soc(w.1,Yraw[1:p,,drop=F],p)
        Y.soc <- dummies.soc$Ydum
        X.soc <- dummies.soc$Xdum
        
        w.2 <- hyperpara[["w.2"]]
        dummies.sur <- get.dum.sur(w.2,Yraw[1:p,,drop=F],p)
        Y.sur <- dummies.sur$Ydum
        X.sur <- dummies.sur$Xdum
        
        #Compute posterior variance
        XX <- rbind(X,X.min,X.soc,X.sur)
        YY <- rbind(Y,Y.min,Y.soc,Y.sur)
      }else{
        if(dummy.type == "MIN")
        {
          #Compute posterior variance
          XX <- rbind(X,X.min)
          YY <- rbind(Y,Y.min)
        }  
      }
      
      #------------------------------------------ Posterior moments --------------------------------------------- 
      V.post <- try(solve(crossprod(XX)), silent = TRUE)
      if(is(V.post, "try-error")){V.post <- ginv(crossprod(XX))}
      V.chol <- t(chol(V.post))
      A.post <- V.post%*%crossprod(XX,YY)
      Scale.post <- crossprod(YY-XX%*%A.post)
      #Sigma.draw <- Scale.post/(nrow(YY)-ncol(XX))
      Sigma.draw <- Scale.post/(nrow(YY)-K-1)
      

  }else{
    A.post <- A_OLS
    V_post <- try(V.post <- solve(crossprod(X) + diag(1e-4, K)), silent=TRUE)
    if (is(V_post,"try-error")) V_post <- MASS::ginv(crossprod(X) + diag(1e-4, K))
    V.chol <- t(chol(V.post))
    Scale.post <- crossprod(Y-X%*%A.post)
    lambda <- 1
    n0 <- M+2
    
  }
  if(cons){
    rownames(A.post) <- c(rep(colnames(YY), p), "1")
  }else{
    rownames(A.post) <- c(rep(colnames(YY), p))
  }
  
  # Common stochastic volatility
  if(sv == "common.sv"){
    b_0 <- 25
    b_1 <- 1.5
    #prior on the mean of the log volatilities
    h_0 <- 0
    V_h0 <- 1e-20
    Bvarsigma.h <- 0.1
    temp_sv <- list(para=c(mu=0,phi=.99,sigma=.1),latent=rep(0,T))
    normalizer <- 1/exp(latent(temp_sv)/2)
    
  }
  
  #Create storage arrays for Sigma, S (sparsified Sigma), Alpha, and A (sparsified Alpha)
  ALPHA.store <- array(0,c(nsave,K,M))
  SIGMA.store <- array(0,c(nsave,M,M))
  eht.store <- matrix(0, nsave, T)
  hyperpara.store <- matrix(NA, nsave,3)
  eig <- matrix(0,nsave,1)
  
  start <- Sys.time()
  pb <- txtProgressBar(min = 0, max = ntot, style = 3) #start progress bar
  to.plot <- seq(1,ntot,by= 50)
  
  for (irep in seq_len(ntot)){

      #Step I: Sample coefficients
      #Step Ia: Sample Alpha from a Gaussian
      if(fast){
        Alpha.draw <- matrix(as.vector(A.post)+as.vector(V.chol%*% matrix(rnorm(n),K,M)%*%chol(Sigma.draw)),K,M)
      }else{
        Alpha.draw <- matrix(as.vector(A.post)+kronecker(t(chol(Sigma.draw)),V.chol)%*%rnorm(n),K,M)
      }  #Step Ib: Sparsify the current draw of Alpha
      #This part could be written in rcpp or improved in terms of speed but allows for easy setting kappa specific to each lag

      rownames(Alpha.draw) <- rownames(A.post)
      colnames(Alpha.draw) <- colnames(A.post)
      
      #Step II: Sample error variances
      #Step IIa: Sample unsparsified Sigma
      Sigma.draw.inv <- matrix(rWishart(1,n0+T,solve(Scale.post)),M,M)
      Sigma.draw <- solve(Sigma.draw.inv)

      if(sv == "common.sv"){
        chol.Sig <- chol(Sigma.draw)
        eps <- (Y-X%*%Alpha.draw)%*%t(solve(chol.Sig))
        
        para <- temp_sv$para
        
        eps <- apply(eps,1,sum) #extract(eps, 1)[[1]]
        temp_sv <- svsample2(eps, startpara = para(temp_sv), 
                               startlatent = latent(temp_sv), 
                               priormu=c(h_0,V_h0), priorphi=c(b_0,b_1), priorsigma = Bvarsigma.h)
          
        ht <- as.numeric(latent(temp_sv))
        ht <- ht - ht[1]

        #ht <- ht - ht[1]
        Sig2.t <- as.numeric(exp(ht))
        
        #Compute posterior variance
        if(dummy.type == "MIN"){
          XX <- rbind(X*normalizer,X.min)
          YY <- rbind(Y*normalizer,Y.min)
        }
        if(dummy.type == "MIN-SOC-SUR"){
          XX <- rbind(X*normalizer,X.min,X.soc,X.sur)
          YY <- rbind(Y*normalizer,Y.min,Y.soc,Y.sur)
        }
        
        V.post <- try(solve(crossprod(XX)), silent = TRUE)
        if(is(V.post, "try-error")){V.post <- ginv(crossprod(XX))}
        V.chol <- t(chol(V.post))
        A.post <- V.post%*%crossprod(XX,YY)
        Scale.post <- crossprod(YY-XX%*%A.post)
        
      }else{
        Sig2.t <- rep(1,T)
      }
      
      normalizer <- 1/sqrt(Sig2.t)
      
    get.comp <- get.companion(Beta_=Alpha.draw,varndxv=c(M,cons,p))
    get.comp.sparse <- get.companion(Beta_=A.draw,varndxv=c(M,cons,p))
    
    MM <- get.comp$MM
    MM.sparse <- get.comp.sparse$MM
    Jm <- get.comp$Jm  
    eig.draw <- max(abs(Re(eigen(MM)$values)))
      
    if (irep %in% to.plot){
      matplot(cbind(Y[,1], (X%*%A.draw)[,1]), type = "l")
    }  
    
    if (irep %in% save.set){
      save.ind <- save.ind + 1
      ALPHA.store[save.ind,,] <- Alpha.draw
      SIGMA.store[save.ind,,] <- Sigma.draw
      hyperpara.store[save.ind,] <- as.numeric(hyperpara)
      eht.store[save.ind,] <- Sig2.t
      eig[save.ind,] <- eig.draw
    }

    setTxtProgressBar(pb, irep) 
  }
  end <- Sys.time()
  duration <- (as.numeric(end)- as.numeric(start))/60
   
  hyperpara.mean <- apply(hyperpara.store,2,mean)
  hyperpara.low <- apply(hyperpara.store,2,quantile,0.05)
  hyperpara.high <- apply(hyperpara.store,2,quantile,0.95)
  
  hyperpara.summary <- rbind(Mean = hyperpara.mean, Low = hyperpara.low, High = hyperpara.high)
  if(dummy.type == "MIN-SOC-SUR"){
    colnames(hyperpara.summary) <- names(hyperpara)
  }else{
   hyperpara.summary <- hyperpara.summary[,1, drop = F]
   colnames(hyperpara.summary) <- c("shrink.1")
  }
    coeff.list <- list(ALPHA.store = ALPHA.store, SIGMA.store = SIGMA.store, hyperpara.summary = hyperpara.summary, eht.store = eht.store, eig = eig, K = K, T=T, M = M, var.names = colnames(Y), duration = duration, cons = cons, p = p, X = X, Y=Y, Y.out = Y.out)

  return(list(coeff.list = coeff.list))
}

###--------------------------------------------------------------------------------------------------------###
###------------------- Non-conjugate BVAR function (Triangularization algo) -------------------------------###
###--------------------------------------------------------------------------------------------------------###


NGSSVS.func <- function(Yraw = Yraw, type = type, nsave =nsave, nburn = nburn, nthin = nthin, p = p, cons = cons, trans = trans)
{
  #---------------------------------------------------Data Settings---------------------------------
  
  ntot <- nsave*nthin + nburn
  ntot <- nthin*nsave+nburn
  save.set <- seq(nburn+nthin, ntot, nthin)
  save.ind <- 0
  
  
  if(cons){
    X <- cbind(mlag(Yraw,p),1)
    Xfull <- cbind(mlag(Yraw,p),1)
  }else{
    X <- mlag(Yraw,p)
    Xfull <- mlag(Yraw,p)
  }
  
  Y <- Yraw[(p+1):nrow(Yraw),]
  X <- X[(p+1):nrow(X),]
  
  T <- nrow(X)
  K <- ncol(X)
  M <- ncol(Y)
  n <- K*M
  v <- (M*(M-1))/2
  
  
  if(trans == "I0"){
    prmean <- rep(0, M)
  }
  
  if(trans == "I1"){
    prmean <- rep(1,M)
  }
  

  #------------------------------------------getting OLS estimates--------------------------------------------- 
  XtXinv <- MASS::ginv(crossprod(X))
  A_OLS <- A.draw <-  XtXinv%*%crossprod(X,Y)
  a_OLS <- as.vector(A_OLS) #vectorizes A_OLS, i.e. a_OLS=vec(A_OLS)
  Em  <- Em.str <- Y-X%*%A_OLS
  SSE  <-  crossprod(Em)
  if(K < T){
    Sigma.draw <- SIGMA_OLS  <-  SSE/(T-K+1) 
  }else{
    Sigma.draw <- SIGMA_OLS  <-  SSE/(T+M+1) # T+K-K+1
  }
  
  if(K < T){
    ssvs.scale <- as.vector(diag(XtXinv))
  }else{
  mysigma <- matrix(NA,M,1)
  for (ii in 1:M){
    Y.i <- Yraw[,ii,drop=F]
    X.i <- mlag(Yraw[,ii],p)
    Y.i <- Y.i[(p+1):nrow(Y.i),,drop=F]
    X.i <- X.i[(p+1):nrow(X.i),,drop=F]
    
    rho.i <- ginv(crossprod(X.i))%*%crossprod(X.i,Y.i)
    er.i <- Y.i-X.i%*%rho.i
    SSE.i <- crossprod(er.i)/(nrow(X.i)-p)
    
    #  tmpar <- arima(Y[,ii],order=c(1,0,0),method="ML")
    mysigma[ii,1] <- sqrt(SSE.i)#sqrt(tmpar$sigma2)
  }  
  
  ssvs.scale <- rep(as.numeric(mysigma), p)
  if(cons){
    ssvs.scale <- c(ssvs.scale, 1)
  }
  
  }
  
  ssvs.scale <- rep(ssvs.scale,M)
  
  #NG and SVSS settings
  sig_eta <-rep(0.1, M)
  eta <- list()
  a_i <- 0.01
  b_i <- 0.01
  
  V_prior <- diag(n)*10^2
  V_priorinv <- diag(1/diag(V_prior))
  
  A_prior <- matrix(0,K,M)
  A_prior[1:M,1:M] <- diag(M)*prmean
  a_prior <- as.vector(A_prior)
  
  theta <- matrix(10,K,M)
  b_draw <- lower.tri(SIGMA_OLS)*1
  Upsilon_draw <- b_draw
  
  diag(Upsilon_draw) <- 1
  normalizer <- rep(1, T)
  
  if(type == "SSVS"){
    tau0 <- 0.01
    tau1 <- 100
    
    pp00 <- tau0*ssvs.scale
    pp11 <- tau1*ssvs.scale
    qq00 <- 0.001
    qq11 <- 1
    
    # initial values
    delta.draw <- matrix(1,n,1)
    gamma.draw <- matrix(1,M,1)
    epsilon.draw <- diag(1,M)
    V_prior <- diag(as.numeric(delta.draw*pp11 + (1-delta.draw)*pp00))
    
    # store parameters
    delta.store <- matrix(NA,nsave,K)
  }
  if(substr(type,1,2) == "NG"){
    c_tau <- d_tau <- e_tau <- f_tau <- 0.01
    pi_j <- matrix(0.01,p,1)
    epsilon.draw <- diag(1,M)
    delta.draw <- rep(1,n)
    
    a_tau <- matrix(0.1,p,1)  
    b_tau <- 0.6
  }
  
  
  #Create storage arrays for Sigma, S (sparsified Sigma), Alpha, and A (sparsified Alpha)
  ALPHA.store <- array(0,c(nsave,K,M))
  SIGMA.store <- array(0,c(nsave,M,M))
  eht.store <- matrix(0,nsave, T)
  delta.store <- matrix(0,nsave,n)
  epsilon.store <- array(0, c(nsave, M,M))
  eig <- matrix(0,nsave,1)
  
  start <- Sys.time()
  pb <- txtProgressBar(min = 0, max = ntot, style = 3) #start progress bar
  to.plot <- seq(1,ntot,by= 50)
  
  for (irep in seq_len(ntot)){
    #Step I: Sample autoregressive parameters
    for (mm in 1:M){
      #Sample autoregressive coefficients equation by equation
      if (mm==1){
        Y.i <- Y[,mm]*normalizer
        X.i <- X*normalizer
        
        V_post <- try(chol2inv(chol(crossprod(X.i)+diag(1/theta[,mm]))),silent=TRUE)
        if (is(V_post,"try-error")) V_post <- ginv(crossprod(X.i)+diag(1/theta[,mm]))
        A_post <- V_post%*%(crossprod(X.i,Y.i)+diag(1/theta[,mm])%*%A_prior[,mm])
        
        A.draw.i <- try(A_post+t(chol(V_post))%*%rnorm(ncol(X.i)),silent=TRUE)
        if (is(A.draw.i,"try-error")) A.draw.i <- mvrnorm(1,A_post,V_post)
        A.draw[,mm] <- A.draw.i
        Em[,mm] <- Em.str[,mm] <- Y[,mm]-X%*%A.draw.i
      }else{
        Y.i <- Y[,mm]/sqrt(sig_eta[[mm]])
        X.i <- cbind(X,Em[,1:(mm-1)])/sqrt(sig_eta[[mm]])
        
        V_post <- try(chol2inv(chol((crossprod(X.i)+diag(1/c(theta[,mm],b_draw[mm,1:(mm-1)]))))),silent=TRUE)    #solve(crossprod(X.i)+diag(1/c(theta[,mm],b_draw[mm,1:(mm-1)])))
        if (is(V_post,"try-error")) V_post <- ginv((crossprod(X.i)+diag(1/c(theta[,mm],b_draw[mm,1:(mm-1)]))))
        A_post <- V_post%*%(crossprod(X.i,Y.i)+diag(1/c(theta[,mm],b_draw[mm,1:(mm-1)]))%*%c(A_prior[,mm],rep(0,mm-1)))
        
        A.draw.i <- try(A_post+t(chol(V_post))%*%rnorm(ncol(X.i)),silent=TRUE)
        if (is(A.draw.i,"try-error")) A.draw.i <- mvrnorm(1,A_post,V_post)
        
        A.draw[,mm] <- A.draw.i[1:ncol(X)]
        Em[,mm] <- Y[,mm]-X%*%A.draw.i[1:ncol(X)]
        Em.str[,mm] <- Y[,mm]-X%*%A.draw.i[1:ncol(X)]-Em[,1:(mm-1),drop=FALSE]%*%A.draw.i[(ncol(X)+1):ncol(X.i),drop=FALSE] 
        
        Upsilon_draw[mm,1:(mm-1)] <- eta[[mm]]<- A.draw.i[(ncol(X)+1):ncol(X.i)] 
      }
    }
    #Step II: Create variance-covariance matrix
    for (jj in 1:M){
      S_1 <- a_i/2+T/2
      S_2 <- b_i/2+crossprod(Em.str[,jj])/2
      
      sig_eta[[jj]] <- 1/rgamma(1,shape = S_1,rate = S_2)
      
    }
    Sigma.draw <- Upsilon_draw%*%diag(sig_eta)%*%t(Upsilon_draw)
    
    if(sv == "common.sv"){
      chol.Sig <- chol(Sigma.draw)
      eps <- (Y-X%*%Alpha.draw)%*%t(solve(chol.Sig))
      
      para <- temp_sv$para
      eps <- (Y-X%*%Alpha.draw)%*%t(solve(chol.Sig))
      
      eps <- apply(eps,1,sum)#extract(eps, 1)[[1]]
      temp_sv <- svsample2(eps, startpara = para(temp_sv), 
                             startlatent = latent(temp_sv), 
                             priormu=c(h_0,V_h0), priorphi=c(b_0,b_1), priorsigma = Bvarsigma.h)
      
      ht <- as.numeric(latent(temp_sv))
      ht <- ht - ht[1]
      Sig2.t <- as.numeric(exp(ht))
      
    }else{
      Sig2.t <- rep(1,T)
    }
    
    normalizer <- 1/sqrt(Sig2.t)    
    
    #Step III: Shrinkage
    if(type == "SSVS"){
      alpha <- as.vector(A.draw)
      theta <- as.vector(theta)
      
      # SSVS for coefficients
      for(jj in 1:n){
        p0 <- dnorm(alpha[jj], 0, sqrt(pp00[jj]))*0.5 + 1e-50
        p1 <- dnorm(alpha[jj], 0, sqrt(pp11[jj]))*0.5 + 1e-50
        prob.delta <- p1/(p0+p1)
        if(is.nan(prob.delta)){prob.delta <- 0}
        
        # assign value to delta
        ifelse(prob.delta > runif(1), delta.draw[jj] <- 1, delta.draw[jj] <- 0)
      }
      
      theta <- as.numeric(delta.draw*pp11 + (1-delta.draw)*pp00)
      theta <- matrix(theta,K,M)
      
      # SSVS for covariances
      eta_0 <- unlist(eta)
      M1 <- 0
      for (nn in 2:M){
        M1 <-M1+1 
        et_0 <- eta[[nn]]
        for (kk in  1:M1){
          q0 <- dnorm(et_0[kk], 0, sqrt(qq00))*0.5 + 1e-50
          q1 <- dnorm(et_0[kk], 0, sqrt(qq11))*0.5 + 1e-50
          prob.gamma <- q1/(q0+q1)
          if(is.nan(prob.gamma)){prob.gamma <- 1}
          # assign value to gamma
          ifelse(prob.gamma > runif(1), gamma.draw[kk] <- 1, gamma.draw[kk] <- 0)
          # update prior
          b_draw[nn,kk] <- as.numeric(gamma.draw[kk]*qq11 + (1-gamma.draw[kk])*qq00)
          epsilon.draw[nn,kk] <- gamma.draw[kk]
        }
      }
      
    }else{
      if(substr(type,1,2) == "NG"){
          #Step II: Sample hyperparameter lambda from G(a,b)
          pi_j[1,1] <- rgamma(1,c_tau+a_tau[[1]]*n,d_tau+a_tau[[1]]/2*sum(theta)) 
          #Step III: Sample the prior scaling factors from GIG
          for (mm in 1:M){
            for (nn in 1:K){
              theta[nn,mm] <- rgig(n=1,lambda=a_tau[[1]]-0.5,(A.draw[nn,mm]-A_prior[nn,mm])^2,a_tau[[1]]*pi_j[1,1])
            }
          }
          theta[theta<1e-7] <- 1e-7
        
        # NG for covariances
        eta_0 <- unlist(eta)
        
        ## Draw prior scaling factors from GIG for covariances
        lambda_omega <- rgamma(1,e_tau+b_tau*v,f_tau+b_tau/2*sum(b_draw[lower.tri(b_draw)]))
        #Step VI: Sample the prior scaling factors for covariances from GIG
        M1 <- 0
        for (nn in 2:M){
          M1 <-M1+1 
          et_0 <- eta[[nn]]
          for (jj in  1:M1){
            b_draw[nn,jj] <- rgig(n=1,lambda=b_tau-0.5,et_0[[jj]]^2,b_tau*lambda_omega)
          }
        }
        
      }
    }
    
    get.comp <- get.companion(Beta_=A.draw,varndxv=c(M,cons,p))
    MM <- get.comp$MM
    Jm <- get.comp$Jm
    eig.draw <- max(abs(Re(eigen(MM)$values)))
    
    if (irep %in% to.plot){
      matplot(cbind(Y[,1], (X%*%A.draw)[,1]), type = "l")
    }  
    
    if (irep %in% save.set){
      save.ind <- save.ind + 1
      
      delta.store[save.ind,] <- delta.draw
      epsilon.store[save.ind,,] <- epsilon.draw
      ALPHA.store[save.ind,,] <- A.draw
      SIGMA.store[save.ind,,] <- Sigma.draw
      eht.store[save.ind,] <- Sig2.t
      eig[save.ind,] <- eig.draw
      
    }
    
    setTxtProgressBar(pb, irep) 
  }
  end <- Sys.time()
  duration <- (as.numeric(end)- as.numeric(start))/60
  
  hyperpara.summary <- NA
  
  coeff.list <- list(ALPHA.store = ALPHA.store, SIGMA.store = SIGMA.store, hyperpara.summary = hyperpara.summary, eht.store = eht.store, eig = eig, delta.store = delta.store, epsilon.store = epsilon.store, X = X, Y = Y, K = K, T=T, M = M, Y.out = Y.out, cons = cons, var.names = colnames(Y), duration = duration)
  
  return(list(coeff.list = coeff.list))
  
}

###--------------------------------------------------------------------------------------------------------###
###------------------------------- Auxillary functions for VAR --------------------------------------------###
###--------------------------------------------------------------------------------------------------------###


#Inputs needed
extract <- function(data,k){
  t <- nrow(data);n <- ncol(data)
  xx <- crossprod(data)
  eigs <- eigen(xx)
  evec <- eigs$vectors;eval <- eigs$values
  
  lam <- sqrt(n)*evec[,1:k]
  fac <- data%*%lam/n
  
  return(list(fac,lam))
}


get.svpara <- function(y,X,b0,V0,mm,vv){
  # prior specified in terms of mean "mm" and var "vv" of IG(a0,a1)
  a0 <- (mm^2/vv)+2
  a1 <- mm*(a0-1)
  
  n <- length(y)
  k <- ncol(X)
  
  par1 <- (a0+n)/2
  var <- solve(crossprod(X,X)+diag(diag(1/V0)))
  mean <- var%*%(crossprod(X,y)+ diag(diag(1/V0))%*%b0)
  par2 <- a0*a1 + sum((y-crossprod(t(X),mean))^2)
  par2 <- (par2 + crossprod(t(crossprod(mean-b0,V0)),mean-b0))/2
  
  sig2 <- 1/rgamma(1,par1,par2)
  var <- var*sig2
  mean <- mean + crossprod(t(chol(var)),rnorm(2))
  return(c(mean,sig2))
}


get.sv <- function(y,x,ht,h0,para,m0,C0,Bsig=1,a0=25,b0=1, tt =T){   
  iC0 <- 1/C0
  iC0m0 <- iC0*m0
  
  mu <- para[1]
  phi <- para[2]
  sig2 <- para[3]
  
  mu_1 = (1-phi)/(1+phi^2)*(mu)
  phi_1 = phi/(1+phi^2)
  
  eta2 <- c(rep(sig2/(1+phi^2),tt-1),sig2)
  eta <- sqrt(eta2)
  
  # sample SV process
  for(it in 1:tt){
    if(it==tt){
      mut <- mu + phi*ht[it-1]
    }else{
      if(it==1){
        mut <-  mu_1 + phi_1*(ht[2]+h0)
      }else{
        mut <- mu_1 + phi_1*(ht[it+1]+ht[it-1])
      }
    }
    
    mut1 <- mut + 0.5*eta2[it]*(crossprod(y[it,])*exp(-mut)-1) 
    hp <- rnorm(1,mut1,eta[it])
    
    num <- dnorm(hp,mut,eta[it],log=TRUE)+ sum(dnorm(y[it,],0,exp(hp/2),log=TRUE))-dnorm(hp,mut1,eta[it],log=TRUE)
    den <- dnorm(ht[it],mut,eta[it],log=TRUE)+ sum(dnorm(y[it,],0,exp(ht[it]/2),log=TRUE))-dnorm(ht[it],mut1,eta[it],log=TRUE)
    
    if(log(runif(1))<min(0,num-den)){
      ht[it] <- hp
    }
  }
  
  # sample initial value
  C00 <- 1/(iC0+phi^2/sig2)
  m00 <- C00*(iC0m0+phi*(ht[1]-mu)/sig2)
  h0 <- rnorm(1,m00,sqrt(C00))
  
  # -------------------------------------------------
  # sample AR(1) parameters
  ht0 <- ht
  ht1 <- cbind(c(h0,ht[1:(tt-1)]))
  para <- get.svpara(ht0,cbind(1, ht1),b0=c(0,1),V0=diag(1,2),mm=0.01,vv=2)
  
  return(list(ht=ht,h0=h0,para=para))
}

gammacoef <- function(mode, sd){
  k.shape <- (2+mode^2/sd^2+sqrt((4+mode^2/sd^2)*mode^2/sd^2))/2
  theta.scale <- sqrt(sd^2/k.shape)
  return(data.frame(shape = k.shape, scale = theta.scale))
}


# Marginal likelihood (simplyfied: without constant terms)
get_ML <- function(X = X, Y = Y, Xdum = Xdum, Ydum = Ydum,T,M,p, sparse, lambda){
  X_ML <- rbind(X,Xdum)
  Y_ML <- rbind(Y,Ydum)
  
  #v_prior <- nrow(Ydum)
  v_prior <- M+2
  
  V_prior <- try(solve(crossprod(Xdum)),silent=TRUE)
  if (is(V_prior,"try-error")) V_prior <- MASS::ginv(crossprod(Xdum))
  V_post <- solve(crossprod(X_ML))
  A_post <- V_post%*%t(X_ML)%*%Y_ML
  
  S_post <- crossprod(Y_ML-X_ML%*%A_post) 
  
  S.det <- as.numeric(determinant(S_post, logarithm = TRUE)[[1]])
  X.det <- as.numeric(determinant(crossprod(X_ML), logrithm = TRUE)[[1]])
  V.det <- as.numeric(determinant(V_prior, logrithm = TRUE)[[1]])
  
  
  ml <- -M/2 * (V.det + X.det) + (-(T+v_prior-1)/2)*S.det
  
  if (is.infinite(ml)) ml <- -10^10
  return(ml)
}


get_ML_DS <- function(X = X, Y = Y, Xdum = Xdum, Ydum = Ydum,T,M,p){
  
  V.prior <- crossprod(Xdum)
  V.priorinv <- solve(V.prior)
  A.prior <- V.priorinv%*%crossprod(Xdum,Ydum)
  S_prior <- crossprod(Ydum-Xdum%*%A.prior)
  
  V.post <- solve(crossprod(X)+V.priorinv)
  A.post <- V.post%*%(V.priorinv%*%A.prior+crossprod(X,Y))
  
  SSE <- crossprod(Y-X%*%A.post)
  
  S.post <- S_prior+ SSE   + t(A.post-A.prior)%*%V.priorinv%*%(A.post-A.prior)#t(A_OLS)%*%crossprod(X)%*%A_OLS+t(A.prior)%*%V.priorinv%*%A.prior-t(A.post)%*%(V.priorinv+crossprod(X))%*%A.post#crossprod(Y-X%*%A_post)#
  
  part1 <- log(det(V.prior))-log(det(crossprod(X)+V.priorinv)) #stability check if -Inf+Inf
  part1 <- ifelse(is.nan(part1),0,part1)
  
  v.prior <- nrow(Ydum)+2
  G1 <- M*(v.prior-ncol(X))*log(2)+sum(lgamma((v.prior-seq(0,(M-1))/2)))
  G2 <- M*((T+v.prior)-ncol(X))*log(2)+sum(lgamma((T+v.prior-seq(0,M-1))/2))
  
  ml <- -M/2 * (part1)+(v.prior/2)*log(det(S_prior))-(T+v.prior)/2*log(det(S.post))-G1+G2
  
  return(ml) 
}

get_ML_stable <- function(X = X, Y = Y, Xdum = Xdum, Ydum = Ydum,T,M,p){
  X_ML <- rbind(X,Xdum)
  Y_ML <- rbind(Y,Ydum)
  
  v_prior <- nrow(Ydum)
  
  V_prior <- solve(crossprod(Xdum))
  V_post <- solve(crossprod(X_ML))
  A_post <- V_post%*%t(X_ML)%*%Y_ML
  
  S_post <- crossprod(Y_ML-X_ML%*%A_post)  
  
  S.diag <- 2*sum(log(diag(chol(S_post))))
  X.diag <- 2*sum(log(diag(chol(crossprod(X_ML)))))
  V.diag <- 2*sum(log(diag(chol(V_prior))))
  
  ml <- -M/2 * (V.diag+X.diag)+(-(T+v_prior)/2)*S.diag
  
  return(ml)
}

get.dum.min <- function(theta,gamma.prior,delta,mysigma,M,p){
  #-----------
  ydummy <- matrix(0,2*M+M*(p-1)+1,M)
  xdummy <- matrix(0,2*M+M*(p-1)+1,M*p+1)
  
  ydummy[1:M,] <- diag((as.numeric(mysigma)*delta)/theta)
  ydummy[(M*(p-1)+M+1):(M*(p-1)+2*M),] <- diag(as.numeric(mysigma))
  
  jp <- diag(1:p)
  xdummy[1:(M*p),1:(M*p)] <- kronecker(jp,diag(as.numeric(mysigma)))/theta
  xdummy[nrow(xdummy),ncol(xdummy)] <- gamma.prior
  return(list(Ydum=ydummy, Xdum=xdummy))
}

norm_vec <- function(x) sqrt(sum(x^2))

remove_outliers <- function(x, na.rm = TRUE, ...) {
  qnt <- quantile(x, probs=c(.25, .75), na.rm = na.rm, ...)
  H <- 5 * IQR(x, na.rm = na.rm)
  y <- x
  y[x < (qnt[1] - H)] <- NA
  y[x > (qnt[2] + H)] <- NA
  y
}

mlag <- function(X,lag)
{
  p <- lag
  X <- as.matrix(X)
  Traw <- nrow(X)
  N <- ncol(X)
  Xlag <- matrix(0,Traw,p*N)
  for (ii in 1:p){
    Xlag[(p+1):Traw,(N*(ii-1)+1):(N*ii)]=X[(p+1-ii):(Traw-ii),(1:N)]
  }
  return(Xlag)  
}




