# Haupt, Kagerer, Steiner (2013)
# Smooth quantile based modeling of brand sales, price
#	and promotional effects from retail scanner panels
# Journal of Applied Econometrics



##############

# load packages

##############

library(prettyR)			# 2.0-4
library(quantreg)			# 4.79
library(splines)			# R 2.15.0 (32-bit)
library(MatrixModels)		# 0.3-1
library(mgcv)			# 1.7-16



##############

# define functions for estimation

##############


# function that generates an equidistant knot sequence
knots_b <- function(p)
{
	c(min(p)-(k-1):1*((max(p)-min(p))/(m+1)), 
		seq(min(p), max(p), length.out=m+2), 
		max(p)+1:(k-1)*((max(p)-min(p))/(m+1)))
}


# function that generates a smoothing spline knot sequence
knots_s <- function(p)
{
	c(min(p), sort(unique(p)), max(p))
}



# monotonicity constrained B-spline estimation using pcls

mcbs	<- function(y, x.b, X.o, k, m, R, eval=FALSE)
{

# construct the regressor matrix X
X.b			<- splineDesign(knots=knots_b(x.b), x=x.b, ord=k)
colnames(X.b)	<- paste(deparse(substitute(x.b)), 1:(m+k), sep="")
X			<- cbind("(Intercept)"=1, X.b[,-1], X.o)

# unnecessary items for pcls
w	<- y*0+1
C	<- matrix(0,0,0)
S	<- vector("list", 0)
off	<- NULL
sp	<- NULL
p	<- c(0,-0.1*(2:(m+k)),rep(0,ncol(X.o)))

# monotonicity constraint
Ain	<- R
bin	<- rep(0, nrow(Ain))

# list for the pcls-estimation
M	<- list(y=y, w=w, X=X, C=C, S=S, off=off, sp=sp, p=p, Ain=Ain, bin=bin)

# constrained estimation
coefs			<- pcls(M)
rownames(coefs)	<- colnames(M$X)

# fitted values and residuals
fits		<- X %*% coefs
res		<- y - fits

# diagonal elements of the hat matrix
if (eval)
{
  XD	<- solve(t(X)%*%X)
  js	<- which(abs(Ain %*% coefs) < 10^{-10})
  if (length(js)>0)
  {
    C_R	<- matrix(Ain[js,], nrow=length(js))
    H		<- X %*% (diag(length(coefs)) - XD %*% t(C_R) %*% 
				solve(C_R %*% XD %*% t(C_R)) %*% C_R) %*% XD %*% t(X)
    # formula for hat matrix from draft (Kagerer, K. (2013))
  } else {
    H		<- X %*% XD %*% t(X)
  }
}

# standard errors, p-value
if (eval)
{
  set.seed(42)
  rep		<- 200
  betas	<- matrix(NA, nrow=length(coefs), ncol=rep, 
					dimnames=list(rownames(coefs),NULL))
  for (r in 1:rep)
  {
    u			<- sample(res, length(y), replace=TRUE)
    y.boot		<- fits + u
    M.boot		<- M
    M.boot$y	<- y.boot
    betas[,r]	<- pcls(M.boot)
  }
  cov		<- cov(t(betas))
  std_err	<- sqrt(diag(cov))
  t_stat	<- coefs/std_err
  p_val	<- 2*pnorm(-abs(t_stat))
  lm_out	<- cbind("Estimate" = coefs, "Std. Error" = std_err,
				"t value" = t_stat, "Pr(>|t|)" = p_val)
}

# return the following
fit	<- list(coefs = coefs)
fit$fitted		<- c(fits)
fit$residuals	<- c(res)
fit$return		<- coefs
if (eval)  
{ 
  fit$edf.Hat	<- round(sum(diag(H)), 0)
  fit$out		<- lm_out
  fit$cov		<- cov
  fit$return	<- lm_out
}
print.summary.mcbs	<<- function(x) { print(x$return) }
class(fit)	<- "summary.mcbs"
fit
}



# monotonicity constrained linear smoothing spline estimation using pcls

mcss	<- function(y, x.s, X.o, R, l, eval=FALSE)
{

# construct the regressor matrix X
k			<- 2
knots			<- knots_s(x.s)
X.s			<- splineDesign(knots=knots, x=x.s, ord=k)
m			<- ncol(X.s)-k
colnames(X.s)	<- paste(deparse(substitute(x.s)), 1:(m+k), sep="")
X			<- cbind("(Intercept)"=1, X.s[,-1], X.o)

# unnecessary items for pcls
w	<- y*0+1
C	<- matrix(0,0,0)

# penalty
d2	<- function(j, knots, k)
{
	diff.knots.k_1	<- diff(knots, lag=k-1)
	d			<- rep(0, m+k)
	d[j-2]		<- 1 / diff.knots.k_1[j-1]
	d[j-1]		<- -1/diff.knots.k_1[j-1] - 1/diff.knots.k_1[j]
	d[j]			<- 1 / diff.knots.k_1[j]
	if (j == 3) { d[j-2] <- 0 }		# intercept!
	d %*% t(d)
}
P		<- matrix(0, nrow=m+k, ncol=m+k)
for (j in 3:(m+k)) 
{
  P	<- P + d2(j, knots=knots, k=2)
}
S	<- list(P)
off	<- 0
sp	<- list(l)
p	<- c(0,-0.1*(2:(m+k)),rep(0,ncol(X.o)))

# monotonicity constraint
Ain	<- R
bin	<- rep(0, nrow(Ain))

# list for the pcls-estimation
M	<- list(y=y, w=w, X=X, C=C, S=S, off=off, sp=sp, p=p, Ain=Ain, bin=bin)

# constrained estimation
coefs			<- pcls(M)
rownames(coefs)	<- colnames(M$X)

# fitted values and residuals
fits		<- X %*% coefs
res		<- y - fits

# diagonal elements of the hat matrix
if (eval)
{
  J				<- ncol(X)
  lS				<- matrix(0, J, J)
  lS[1:(m+k),1:(m+k)]	<- P
  XD	<- solve(t(X)%*%X+ lS)
  js	<- which(abs(Ain %*% coefs) < 10^{-10})
  if (length(js)>0)
  {
    C_R	<- matrix(Ain[js,], nrow=length(js))
    H		<- X %*% (diag(length(coefs)) - XD %*% t(C_R) %*% 
				solve(C_R %*% XD %*% t(C_R)) %*% C_R) %*% XD %*% t(X)
  } else {
    H		<- X %*% XD %*% t(X)
  }
}

# standard errors, p-value
if (eval)
{
  set.seed(42)
  rep		<- 200
  betas	<- matrix(NA, nrow=length(coefs), ncol=rep, 
					dimnames=list(rownames(coefs),NULL))
  for (r in 1:rep)
  {
    u			<- sample(res, length(y), replace=TRUE)
    y.boot		<- fits + u
    M.boot		<- M
    M.boot$y	<- y.boot
    betas[,r]	<- pcls(M.boot)
  }
  cov		<- cov(t(betas))
  std_err	<- sqrt(diag(cov))
  t_stat	<- coefs/std_err
  p_val	<- 2*pnorm(-abs(t_stat))
  lm_out	<- cbind("Estimate" = coefs, "Std. Error" = std_err,
				"t value" = t_stat, "Pr(>|t|)" = p_val)
}

# return the following
fit	<- list(coefs = coefs)
fit$fitted		<- c(fits)
fit$residuals	<- c(res)
fit$return		<- coefs
if (eval)  
{ 
  fit$edf.Hat	<- round(sum(diag(H)), 0)
  fit$out		<- lm_out
  fit$cov		<- cov
  fit$return	<- lm_out
}
print.summary.mcss	<<- function(x) { print(x$return) }
class(fit)	<- "summary.mcss"
fit
}



# function for quantile regressions
# (sometimes rq does not work for e.g. 0.5, but for 0.50000000001)
# -> automatizated correction of thetas!!!

theta_try	<- function(formula, R, thetas, dataset=NULL)
{

	thetas_work	<- rep(NA, length(thetas))

	for (th in 1:length(thetas)) 		
	{
	  theta_try	<- thetas[th]
	  QR_try	<- NULL
	  corr_try	<- 10^{-10}

	  while((class(QR_try)!="rq") & (corr_try<0.1)) 
	  {
	    if (is.null(dataset))
	    {
	      QR_try 		<- try(rq(formula, R=R, r=rep(0, nrow(R)),
						method="fnc", tau=theta_try))
	    } else {
	      QR_try 		<- try(rq(formula, R=R, r=rep(0, nrow(R)),
						data=dataset, 
						method="fnc", tau=theta_try))
	    }
	    thetas_work[th]	<- theta_try
	    theta_try		<- thetas[th] + corr_try
	    corr_try		<- corr_try*10
	  }
	}

	thetas_work
}






##############

# import data

##############

dataset	<- read.table("hks-data.txt", header=TRUE, sep="\t")
attach(dataset)



##############

# summary statistics (Table 2)

##############


# combine all sales and prices

price_all	<- cbind(price_flona, price_tropu, price_mmprm, price_mmaid, 
			   price_cithi, price_flogo, price_trefr, price_tropi, 
			   price_domin)
sales_all	<- cbind(sales_flona, sales_tropu, sales_mmprm, sales_mmaid, 
			   sales_cithi, sales_flogo, sales_trefr, sales_tropi, 
			   sales_domin)
share_all	<- as.data.frame(sales_all/rowSums(sales_all)*100)
colnames(share_all) <- paste("share_",substr(colnames(sales_all),7,11),sep="")


# get weekly averages

price_per_week	<- aggregate(price_all, by=list(week=week), FUN=mean)[,-1]
share_per_week	<- aggregate(share_all, by=list(week=week), FUN=mean)[,-1]


# summary statistics

round(describe(price_per_week,num.desc=c("min","max","mean","sd"))$Numeric, 2)
round(describe(share_per_week,num.desc=c("min","max","mean","sd"))$Numeric, 2)





######################################################################

# estimation of different panel models including prediction intervals

######################################################################



##############

# preparation for estimations

##############


# data for Minute Maid estimations

sales		<- sales_mmaid
price		<- price_mmaid
display	<- display_mmaid
cross_premi	<- pmin(price_flona, price_tropu)
cross_mmprm	<- price_mmprm
cross_cithi	<- price_cithi
cross_flogo	<- price_flogo
cross_trefr	<- price_trefr
cross_tropi	<- price_tropi
cross_domin	<- price_domin
end_99	<- 1 * (substr(price, 3,4)=="99")
disp_99	<- end_99*display


# number of observations, weeks and stores; store dummies

n		<- length(sales)
W		<- length(levels(factor(week)))

Stores	<- model.matrix(~ -1 + factor(store), data=dataset)
stnos		<- levels(factor(store))[-1]
S		<- ncol(Stores)
for (s in 1:S)
{
  assign(paste("s_", levels(factor(store))[s], sep=""), Stores[,s])
}


# parameters for spline estimations

k	<- 4				# order of spline
m	<- round(n^0.2-1,0)	# number of inner knots
l	<- 1				# smoothing parameter
mono	<- "D"			# monotonicity constraint (decreasing)



# models

other_vars	<- paste("log(cross_premi) + log(cross_mmprm) + ", 
			   "log(cross_cithi) + log(cross_flogo) + ",
			   "log(cross_trefr) + log(cross_tropi) + ", 
			   "log(cross_domin) + ",
			   "display + end_99 + disp_99 + ",
			   "holiday + summer + fall + winter + ", 
			   paste(paste("s_", stnos, sep=""), collapse = " + "), 
			sep="")

model_A1	<- as.formula(paste("log(sales) ~ 1 + log(price) + ", 
				other_vars, sep=""))

model_A2	<- as.formula(paste("log(sales) ~ 1 + ", 
				"splineDesign(x=log(price), ord=k, 
					knots=knots_b(log(price)))[,-1] + ", 
				other_vars, sep=""))

model_A3_LS	<- model.matrix(as.formula(paste("~ -1 +", other_vars)))

model_A3_QR	<- model_A2

model_A4_LS	<- model_A3_LS

model_A4_QR	<- as.formula(paste("log(sales) ~ 1 + ", 
				"qss(x=log(price), constraint=mono, lambda=1) + ",  
				other_vars, sep=""))

model_names	<- c("A1", "A2", "A3", "A4")



# thetas for quantile regressions
thetas		<- c(0.1,0.25,0.5,0.75,0.9)



# vectors / matrices / lists containing k, coefs, residuals

coefs_all		<- vector("list", length=4)
names(coefs_all)	<- model_names

ks			<- matrix(NA, ncol=1+length(thetas), nrow=4,
					dimnames=list(model_names, c("LS", thetas)))

resids	<- array(NA, dim=c(n, 1+length(thetas), 4), 
			dimnames=list(NULL, c("mean", thetas), model_names) )





##############

# LS and QR estimations (Tables 5-8)

##############


# model A1 (price effect: linear)

LS_A1		<- summary(lm(model_A1))
QR_tmp	<- rq(model_A1, tau=thetas)
QR_A1		<- summary(QR_tmp, method="boot")
ks["A1",]	<- nrow(LS_A1$coef)

coefs_all[["A1"]]			<- cbind(LS_A1$coef[,1], QR_tmp$coef)
colnames(coefs_all[["A1"]])	<- c("LS", thetas)

resids[,,"A1"]			<- cbind(residuals(LS_A1), residuals(QR_tmp))



# model A2 (price effect: unconstrained B-spline)

LS_A2		<- summary(lm(model_A2))
QR_tmp	<- rq(model_A2, tau=thetas)
QR_A2		<- summary(QR_tmp, method="boot")
ks["A2",]	<- nrow(LS_A2$coef)

coefs_all[["A2"]]			<- cbind(LS_A2$coef[,1], QR_tmp$coef)
colnames(coefs_all[["A2"]])	<- c("LS", thetas)
rownames(coefs_all[["A2"]])	<- c("(Intercept)", 
						paste("log(price)", 1:(m+k-1), sep=""), 
						rownames(coefs_all[["A1"]])[-c(1,2)])

resids[,,"A2"]			<- cbind(residuals(LS_A2), residuals(QR_tmp))



# model A3 (price effect: constrained B-spline)

R_3		<- cbind(cbind(diag(c(0,rep(1,m+k-2))),0)+cbind(0,-diag(m+k-1)), 
				matrix(0, ncol=ks["A1",1]-2, nrow=m+k-1))

LS_A3			<- mcbs(log(sales), log(price), 
				model_A3_LS, k=k, m=m, R=R_3, eval=TRUE)
ks["A3", "LS"]	<- max(LS_A3$edf.Hat, ks["A1",1])
cov_A3		<- LS_A3$cov

QR_tmp		<- rq(model_A3_QR, R=R_3, r=rep(0, nrow(R_3)), 
						method="fnc", tau=thetas)
QR_A3			<- summary(QR_tmp, method="boot")
ks["A3",-c(1)]	<- pmax(colSums(abs(QR_tmp$residuals)<10^{-8}), ks["A1",1])

coefs_all[["A3"]]			<- cbind(LS_A3$coefs, QR_tmp$coef)
colnames(coefs_all[["A3"]])	<- c("LS", thetas)
rownames(coefs_all[["A3"]])	<- rownames(coefs_all[["A2"]])

resids[,,"A3"]			<- cbind(residuals(LS_A3), residuals(QR_tmp))



# model A4 (price effect: smoothing spline)

m_4		<- length(unique(price))-2
R_4		<- cbind(cbind(diag(c(0,rep(1,m_4))),0)+cbind(0,-diag(m_4+1)), 
				matrix(0, ncol=ks["A1",1]-2, nrow=m_4+1))

LS_A4			<- mcss(log(sales), log(price), 
				model_A4_LS, R=R_4, l=1, eval=TRUE)
ks["A4", "LS"]	<- max(LS_A4$edf.Hat, ks["A1",1])
cov_A4		<- LS_A4$cov
coefs_all[["A4"]]	<- cbind(LS_A4$coefs, matrix(NA, nrow=length(LS_A4$coefs), 
								ncol=length(thetas)))
colnames(coefs_all[["A4"]])	<- c("LS", thetas)
rownames(coefs_all[["A4"]])	<- c("(Intercept)", 
			paste("log(price)", 1:(length(unique(price))-1), sep=""), 
						rownames(coefs_all[["A1"]])[-c(1,2)])
resids[,1,"A4"]			<- LS_A4$residuals

QR_A4					<- vector("list", length(thetas))
for (th in 1:length(thetas))
{
  QR_tmp_rqss			<- rqss(model_A4_QR, tau=thetas[th])
  QR_A4[[th]]			<- summary(QR_tmp_rqss)
  coefs_all[["A4"]][,1+th]	<- QR_tmp_rqss$coef[rownames(coefs_all[["A4"]])] 
  resids[,1+th,"A4"]		<- QR_tmp_rqss$resid[1:n]
  ks["A4",1+th]			<- max(QR_tmp_rqss$edf, ks["A1",1])
}



# fitted values

fitted	<- log(sales) - resids




##############

# evaluation: in-sample (AIC and SIC) (Table 4)

##############


# in-sample prediction: AIC, SIC


# check function from Koenker (2005), page 5, see also code for rq
chck	<- function(u, theta) { u*(theta - 1*(u<0)) }

AICs	<- matrix(NA, ncol=1+length(thetas), nrow=4, 
			dimnames=list(model_names, c("LS",thetas)))
SICs	<- AICs

u_term_LS		<- function(resids) {log(sum((resids)^2)/n )}
u_term_QR		<- function(resids, theta) {log(sum(chck(resids,theta))/n)}

for (mo in model_names) 
{

  AICs[mo,1]	<- u_term_LS(resids[,1,mo]) + ks[mo,1]*2/n
  SICs[mo,1]	<- u_term_LS(resids[,1,mo]) + ks[mo,1]*log(n)/n

  for (th in 1:length(thetas)) 
  {
    AICs[mo,th+1]	<- u_term_QR(resids[,th+1,mo], thetas[th]) + ks[mo,th+1]/n
    SICs[mo,th+1]	<- u_term_QR(resids[,th+1,mo], thetas[th]) + 
					ks[mo,th+1]*log(n)/(2*n)
  }
}

round(AICs, 3)
round(SICs, 3)




##############

# prediction intervals (Table 11)

##############


# squared residual standard error for LS estimations

sigmasq_A1	<- LS_A1$sigma^2
sigmasq_A2	<- LS_A2$sigma^2
sigmasq_A3	<- sum(resids[,1,"A3"]^2) / (n-ks["A3",1])
sigmasq_A4	<- sum(resids[,1,"A4"]^2) / (n-ks["A4",1])


# 0.9-quantile from the standard normal distribution

c_all	<- qnorm(0.9)


# regressor matrices

X_A1			<- model.matrix(model_A1)
X_A2			<- model.matrix(model_A2)
colnames(X_A2)	<- rownames(coefs_all[["A2"]])
X_A3			<- X_A2
X_A4			<- cbind(1, splineDesign(knots=knots_s(log(price)), 
				x=log(price),ord=2)[,-1], X_A1[,-c(1,2)])
colnames(X_A4)	<- rownames(coefs_all[["A4"]])


# covariance matrices for A1 and A2

cov_A1		<- sigmasq_A1 * solve(t(X_A1)%*%X_A1)
cov_A2		<- sigmasq_A2 * solve(t(X_A2)%*%X_A2)


# prediction intervals

PIs_LS	<- array(NA, dim=c(n,2,4), 
				dimnames=list(NULL, c("lower", "upper"), model_names))
PIs_QR	<- PIs_LS

for (mo in model_names)
{
  PIs_LS[,,mo]	<- fitted[,1,mo] + c_all * as.matrix(
				sqrt(get(paste("sigmasq_", mo, sep="")) + 
				diag(get(paste("X_", mo, sep="")) %*% 
					get(paste("cov_", mo, sep="")) %*% 
					t(get(paste("X_", mo, sep="")))))) %*%
				c(-1,1)
  PIs_QR[,,mo]	<- fitted[,c(2,6),mo]
}


# evaluation of prediction performance: in-sample coverage

PI_perf	<- matrix(NA, ncol=length(model_names), nrow=2, 
			dimnames=list(c("PI_LS", "PI_QR"), model_names))

for (mo in model_names)
{
  PI_perf[1,mo]	<- mean( (PIs_LS[,1,mo]<=log(sales)) & 
							  (log(sales)<=PIs_LS[,2,mo]) )
  PI_perf[2,mo]	<- mean( (PIs_QR[,1,mo]<=log(sales)) & 
							  (log(sales)<=PIs_QR[,2,mo]) )
}

round(PI_perf,3)




##############

# size of Delta(sales) for some Delta(price)=20 (Table 9)

##############


# correction factor for the LS-models

alphas_1		<- rep(NA, 4)
names(alphas_1)	<- model_names

for (mo in model_names)
{
  alphas_1[mo]	<- coefficients(lm(sales ~ -1 + exp(fitted[,1,mo])))[1]
}


# part of X-matrix: med(price) or med(price)-0.2, other variables at med

Xo_median	<- c(log(apply(cbind(cross_premi, cross_mmprm, 
			cross_cithi, cross_flogo, cross_trefr, cross_tropi, 
			cross_domin), 2, median)), rep(0,ks["A1",1]-9))

X_med_A1	<- rbind( c(1, log(median(price)), Xo_median), 
			    c(1, log(median(price)-0.2), Xo_median) )
X_med_A2	<- rbind( c(1, splineDesign(x=log(median(price)), 
				knots=knots_b(log(price)), ord=4)[,-1], Xo_median), 
			    c(1, splineDesign(x=log(median(price)-0.2), 
				knots=knots_b(log(price)), ord=4)[,-1], Xo_median) )
X_med_A3	<- X_med_A2
X_med_A4	<- rbind( c(1, splineDesign(x=log(median(price)), 
				knots=knots_s(log(price)), ord=2)[,-1], Xo_median), 
			    c(1, splineDesign(x=log(median(price)-0.2), 
				knots=knots_s(log(price)), ord=2)[,-1], Xo_median) )


# estimated sales for med(price) and med(price)-0.2

sales_est_A1	<- exp(X_med_A1 %*% coefs_all[["A1"]]) * 
					matrix(c(rep(alphas_1["A1"],2), 
					rep(1,2*length(thetas))), nrow=2)
sales_est_A2	<- exp(X_med_A2 %*% coefs_all[["A2"]]) * 
					matrix(c(rep(alphas_1["A2"],2), 
					rep(1,2*length(thetas))), nrow=2)
sales_est_A3	<- exp(X_med_A3 %*% coefs_all[["A3"]]) * 
					matrix(c(rep(alphas_1["A3"],2), 
					rep(1,2*length(thetas))), nrow=2)
sales_est_A4	<- exp(X_med_A4 %*% coefs_all[["A4"]]) * 
					matrix(c(rep(alphas_1["A4"],2), 
					rep(1,2*length(thetas))), nrow=2)

# differences

(diffs		<- rbind(round(sales_est_A1[2,] - sales_est_A1[1,], 0), 
				   round(sales_est_A2[2,] - sales_est_A2[1,], 0), 
				   round(sales_est_A3[2,] - sales_est_A3[1,], 0), 
				   round(sales_est_A4[2,] - sales_est_A4[1,], 0) ))




##############

# price vs. log(sales): estimation results and PI

##############


plot.ests	<- function(mo, PI=FALSE)
{
	mains	<- c("(A,1) parametric", "(A,2) unconstr. B-spline", 
			"(A,3) monotone B-spline", "(A,4) smoothing spline")
	names(mains)	<- model_names


	# prepare functions for plot of estimated curves

	k_tmp		<- ncol(get(paste("X_", mo, sep=""))) - (ncol(X_A1)-2)

	pr_plot	<- seq(min(price), max(price), by=0.01)

	if (mo=="A1")
	{
	  f_plot	<- function(x, th)
	  {
	    c(coefs_all[[mo]][1,th] + log(x)*coefs_all[[mo]][2,th] +
	    c(Xo_median[1:7]%*%coefs_all[[mo]][k_tmp+1:7,th]))
	  }
	  f_pi	<- function(x)
	  {
	    X_pl	<- cbind(1, log(x), matrix(Xo_median[1:7], nrow=length(x), 
					ncol=7, byrow=TRUE) )
	    c_all * sqrt( get(paste("sigmasq_", mo, sep="")) + 
			diag( X_pl %*% get(paste("cov_", mo, sep="")
					)[1:ncol(X_pl),1:ncol(X_pl)] %*% t(X_pl) ) )
	  }
	}

	if (mo=="A2" | mo=="A3")
	{
	  f_plot	<- function(x, th)
	  {
	    c(coefs_all[[mo]][1,th] + 
	    splineDesign(knots=knots_b(log(price)), x=log(x), ord=k)[,-1]%*%
			coefs_all[[mo]][2:k_tmp,th] +
	    c(Xo_median[1:7]%*%coefs_all[[mo]][k_tmp+1:7,th]))
	  }
	  f_pi	<- function(x)
	  {
	    X_pl	<- cbind(1, splineDesign(knots=knots_b(log(price)), 
						x=log(x), ord=k)[,-1], 
				matrix(Xo_median[1:7], nrow=length(x), 
					ncol=7, byrow=TRUE) )
	    c_all * sqrt( get(paste("sigmasq_", mo, sep="")) + 
			diag( X_pl %*% get(paste("cov_", mo, sep="")
					)[1:ncol(X_pl),1:ncol(X_pl)] %*% t(X_pl) ) )
	  }
	}

	if (mo=="A4")
	{
	  f_plot	<- function(x, th)
	  {
	    c(coefs_all[[mo]][1,th] + 
	    splineDesign(knots=knots_s(log(price)), x=log(x), ord=2)[,-1]%*%
			coefs_all[[mo]][2:k_tmp,th] +
	    c(Xo_median[1:7]%*%coefs_all[[mo]][k_tmp+1:7,th]))
	  }
	  f_pi	<- function(x)
	  {
	    X_pl	<- cbind(1, splineDesign(knots=knots_s(log(price)), 
						x=log(x), ord=2)[,-1],
					matrix(Xo_median[1:7], nrow=length(x), 
						ncol=7, byrow=TRUE) )
	    c_all * sqrt( get(paste("sigmasq_", mo, sep="")) + 
			diag( X_pl %*% get(paste("cov_", mo, sep="")
					)[1:ncol(X_pl),1:ncol(X_pl)] %*% t(X_pl) ) )
	  }
	}


	# plot

	par(mai=c(0.65,0.6,0.3,0.5-0.4*PI), mgp=c(2,1,0))

	plot(price, log(sales), xlab="price", ylab="log(sales)", type="n", 
			font=2, font.lab=2, main=mains[mo])


	# plot of estimation results

	if (PI==FALSE)
	{

	  # plot frequency

	  pr_min	<- floor(min(price)*10)/10
	  pr_max	<- floor(max(price)*10)/10
	  pr_int	<- seq(floor(pr_min*10)/10, floor(pr_max*10)/10, by=0.1)
	  pr_fre	<- rep(NA, length(pr_int))
	  for (p in 1:length(pr_fre))
	  {
	    pr_fre[p]	<- sum(table(price)[as.character(
		((pr_min+(p-1)*0.1)*100):((pr_min+(p-1)*0.1+0.09)*100)/100)], 
				na.rm=TRUE)
	  }

	  tmp_0	<- min(log(sales)) - 0.04*diff(range(log(sales)))
	  tmp_m	<- max(log(sales)) + 0.04*diff(range(log(sales)))
	  tmp_1	<- tmp_0 + (tmp_m - tmp_0)/6

	  axis(4, at=tmp_0+(tmp_m - tmp_0)/3, label="freq(price)", 
				tick=FALSE, font=2, line=-0.5, cex.axis=0.8)
	  axis(4, at=seq(tmp_0, tmp_1, length.out=3), 
			label=round(seq(0, max(pr_fre), length.out=3),0), 
			cex.axis=0.8)

	  for (fr in 1:length(pr_fre))
	  {
	    polygon(c(pr_int[fr], pr_int[fr], pr_int[fr]+0.09, pr_int[fr]+0.09), 
			c(0, pr_fre[fr], pr_fre[fr], 0)/
				max(pr_fre)*(tmp_m - tmp_0)/6+tmp_0, 
		col="grey90", border="black")
	  }
	  box()


	  # legend

	  legend("topright", inset=0.02, c("mean", "median", 
			"quantiles: 0.1, 0.25, 0.75, 0.9"), 
			lty=c(2,1,1), lwd=2, col=c("black","black","grey"))


	  # estimated curves

	  for (th in c("LS",thetas))
	  {
	    plot(function(x) f_plot(x, th), 
			from=min(price), to=max(price), add=TRUE, 
			lty=ifelse(th=="LS",2,1), lwd=2, 
			col=ifelse((th=="LS"|th=="0.5"),"black","grey"))
	  }

	}


	# plot of prediction intervals

	if (PI==TRUE)
	{

	  # legend

	  legend("topright", inset=0.02, 
			c("80% PI from LS", "80% PI from QR"), 
			lty=c(2,1), lwd=2, col=c("black","black"))


	  # estimated curves

	  for (th in c("0.1","0.9"))
	  {
	    plot(function(x) f_plot(x, th), 
			from=min(price), to=max(price), add=TRUE, lty=1, lwd=2)
	  }

	  for (pm in c(-1, 1))
	  {
	    plot(function(x) f_plot(x, "LS") + pm * f_pi(x), 
			from=min(price), to=max(price), add=TRUE, lty=2, lwd=2)
	  }

	}

}

plot.ests("A1")
plot.ests("A2")
plot.ests("A3")
plot.ests("A4")

plot.ests("A1", PI=TRUE)
plot.ests("A2", PI=TRUE)
plot.ests("A3", PI=TRUE)
plot.ests("A4", PI=TRUE)





######################################################################

# prediction performance of different panel models

######################################################################


# length of estimation sample, prediction horizon, number of prediction weeks

l_est		<- 52
l_prd		<- c(1,4)
no_preds	<- W - l_est - l_prd+1


# list containing arrays of coefficients

coefs			<- vector("list", length(model_names))
names(coefs)	<- model_names

for (mo in model_names) 
{

  coefs[[mo]]			<- vector("list", max(no_preds))

  for (w in 1:max(no_preds))
  {

    if (mo %in% c("A1","A2","A3"))
    {
      coefs_tmp			<- matrix(NA, ncol=1+length(thetas), 
						nrow=nrow(coefs_all[[mo]]))
      colnames(coefs_tmp)	<- c("LS", thetas)
      rownames(coefs_tmp)	<- rownames(coefs_all[[mo]])
    }

    if (mo=="A4")	# note: different est. samples => different knot sequence
    {

      # estimation sample 
      smpl_est		<- ((week >= min(week) + w-1)*
					(week <= min(week) + w-1 + l_est-1)) == TRUE

      # length of smoothing spline component
      x_len			<- length(unique(price[smpl_est]))

      # number of columns of regressor matrix
      X_col			<- x_len-1 + nrow(coefs_all[["A1"]])-1

      # NA-matrix for coefficiens
      coefs_tmp		<- matrix(NA, ncol=1+length(thetas), nrow=X_col)
      colnames(coefs_tmp)	<- c("LS", thetas)
      rownames(coefs_tmp)	<- c("(Intercept)", 
						paste("log(price)", 1:(x_len-1), sep=""),
					rownames(coefs_all[["A1"]])[-c(1,2)])
    }

    coefs[[mo]][[w]]	<- coefs_tmp

  }
}
rm(coefs_tmp, smpl_est, x_len, X_col)



# array to save residuals and fitted values (from ESTIMATION SAMPLE)

fits	<- array(NA, dim=c(S * l_est, 1+length(thetas), 
				length(model_names), max(no_preds)), 
			dimnames=list(NULL, c("LS",thetas), model_names, 
				paste("w=", 1:max(no_preds), sep="")))
ress	<- fits




##############

# estimations with estimation samples, save coefficients, fitted values, res.

##############

tmp_store			<- model.matrix(~factor(store))[,-1]
colnames(tmp_store)	<- paste("s_", stnos, sep="")
data.all	<- data.frame(sales, price, cross_premi, cross_mmprm, 
				cross_cithi, cross_flogo, cross_trefr, cross_tropi, 
				cross_domin, display, end_99, disp_99, holiday, 
				summer, fall, winter, tmp_store)


for (w in 1:max(no_preds))
{

  # estimation sample
  smpl_est		<- ((week >= min(week) + w-1)*
					(week <= min(week) + w-1 + l_est-1)) == TRUE


  # estimation, coefficients, fitted values

  # A1
  LS		<- lm(model_A1, data=data.all[smpl_est,])
  QR		<- rq(model_A1, data=data.all[smpl_est,], tau=thetas)
  coefs[["A1"]][[w]]				<- cbind(LS$coef, QR$coef)
  ress[,1,"A1",w]					<- residuals(LS)
  ress[,2:(length(thetas)+1),"A1",w]	<- residuals(QR)
  fits[,,"A1",w]			<- log(sales[smpl_est]) - ress[,,"A1",w]

  # A2
  LS		<- lm(model_A2, data=data.all[smpl_est,])
  QR		<- rq(model_A2, data=data.all[smpl_est,], tau=thetas)
  coefs[["A2"]][[w]]				<- cbind(LS$coef, QR$coef)
  ress[,1,"A2",w]					<- residuals(LS)
  ress[,2:(length(thetas)+1),"A2",w]	<- residuals(QR)
  fits[,,"A2",w]			<- log(sales[smpl_est]) - ress[,,"A2",w]

  # A3
  LS 		<- mcbs(log(sales[smpl_est]), log(price[smpl_est]), 
				model_A3_LS[smpl_est,], k=k, m=m, R=R_3)
  thetas_tmp	<- theta_try(model_A3_QR, R=R_3, thetas=thetas, 
						dataset=data.all[smpl_est,])
  QR		<- rq(model_A3_QR, R=R_3, r=rep(0, nrow(R_3)), 
				data=data.all[smpl_est,], method="fnc",tau=thetas_tmp)
  coefs[["A3"]][[w]]				<- cbind(LS$coef, QR$coef)
  ress[,1,"A3",w]					<- residuals(LS)
  ress[,2:(length(thetas)+1),"A3",w]	<- residuals(QR)
  fits[,,"A3",w]			<- log(sales[smpl_est]) - ress[,,"A3",w]

  # A4
  m_4p	<- length(unique(price[smpl_est]))-2
  R_4p	<- cbind(cbind(diag(c(0,rep(1,m_4p))),0)+cbind(0,-diag(m_4p+1)), 
				matrix(0, ncol=ks["A1",1]-2, nrow=m_4p+1))
  LS		<- mcss(log(sales[smpl_est]), log(price[smpl_est]), 
				model_A4_LS[smpl_est,], R=R_4p, l=1)
  coefs[["A4"]][[w]][,1]	  <- LS$coef
  ress[,1,"A4",w]		  <- LS$residuals
  for (th in 1:length(thetas))
  {
    rqss_tmp	<- rqss(model_A4_QR,tau=thetas[th],data=data.all[smpl_est,])
    no_tmp		<- which(names(rqss_tmp$coef)=="log(price)1")
    le_tmp		<- length(rqss_tmp$coef)
    coefs[["A4"]][[w]][,th+1]	<- 
				rqss_tmp$coef[c(1,no_tmp:le_tmp,2:(no_tmp-1))]
    ress[,th+1,"A4",w] 		<- rqss_tmp$resid[1:rqss_tmp$n]
  }
  fits[,,"A4",w]			<- log(sales[smpl_est]) - ress[,,"A4",w]

}
rm(smpl_est, LS, QR, thetas_tmp, m_4p, R_4p, rqss_tmp, no_tmp, le_tmp)




##############

# save predicted values and errors

##############


# function that sets all values outside the range of e.g. price to the 
#	nearest min or max

within_range	<- function(variable, var_range)
				{
					r_tmp		<- range(var_range)
					var_tmp	<- variable
					var_tmp[which(var_tmp < r_tmp[1])]	<- r_tmp[1]
					var_tmp[which(var_tmp > r_tmp[2])]	<- r_tmp[2]
					var_tmp
				}


# function that calculated the median cross-price of the 5 last weeks
#	for each store

cross_B_prd		<- function(variable)
				{
					apply(matrix(variable[smpl_cross], 
						nrow=S, ncol=5, byrow=TRUE), 1, median)
				}


# model names for A and B

models_AB	<- c(model_names, "B1", "B2", "B3", "B4")




# prediction


for (p in 1:length(l_prd))
{

  # array to save errors and predicted values (from PREDICTION SAMPLE), tempr.
  prds	<- array(NA, dim=c(S, 1+length(thetas), 
					length(models_AB), no_preds[p]), 
				dimnames=list(c("s_5", paste("s_", stnos, sep="")), 
					c("LS", thetas), models_AB, 
					paste("w=", 1:no_preds[p], sep="")))
  errs	<- prds



  for (w in 1:no_preds[p])
  {

    # prediction sample 
    smpl_prd	<- (week == min(week) + w-1 + l_est + l_prd[p]-1) == TRUE
    smpl_cross	<- ((week <= min(week) + w-1 + l_est - 1)*
				(week >= min(week) + w-1 + l_est - 5)) == TRUE

    # estimation sample 
    smpl_est		<- ((week >= min(week) + w-1)*
					(week <= min(week) + w-1 + l_est-1)) == TRUE


    # for semiparametric predictions, all price-observations outside the
    # estimation sample range are set on the boundary value
    # -> predictions are constant outside the estimation sample range
    price_r_prd	<- within_range(price[smpl_prd], price[smpl_est])


    # cross prices for B-model (median of last 5 weeks for each store)
    cross_premi_prd	<- cross_B_prd(cross_premi)
    cross_mmprm_prd	<- cross_B_prd(cross_mmprm)
    cross_cithi_prd	<- cross_B_prd(cross_cithi)
    cross_flogo_prd	<- cross_B_prd(cross_flogo)
    cross_trefr_prd	<- cross_B_prd(cross_trefr)
    cross_tropi_prd	<- cross_B_prd(cross_tropi)
    cross_domin_prd	<- cross_B_prd(cross_domin)


    # predictions

    # price parts of the prediction matrix

    X_1_price	<- log(price[smpl_prd])
    X_2_price	<- splineDesign(x=log(price_r_prd), 
					knots=knots_b(log(price[smpl_est])), ord=k)[,-1]
    X_3_price	<- X_2_price
    knots.w		<- knots_s(log(price[smpl_est]))
    X_4_price	<- splineDesign(x=log(price_r_prd),knots=knots.w,ord=2)[,-1]
    colnames(X_4_price)	<- paste("log(price)", 1:ncol(X_4_price), sep="")


    # cross-price part of the prediction matrix

    X_A_cross	<- log(cbind(cross_premi, cross_mmprm, 
					cross_cithi, cross_flogo, cross_trefr, 
					cross_tropi, cross_domin)[smpl_prd,])
    X_B_cross	<- log(cbind(cross_premi_prd, cross_mmprm_prd, 
					cross_cithi_prd,cross_flogo_prd,cross_trefr_prd, 
					cross_tropi_prd, cross_domin_prd))

    # remaining part of the prediction matrix

    Xo_prd			<- X_A1[smpl_prd,-(1:9)]

    # predictions

    prds[,,"A1",w] <-cbind(1,X_1_price,X_A_cross,Xo_prd)%*%coefs[["A1"]][[w]]
    prds[,,"A2",w] <-cbind(1,X_2_price,X_A_cross,Xo_prd)%*%coefs[["A2"]][[w]]
    prds[,,"A3",w] <-cbind(1,X_3_price,X_A_cross,Xo_prd)%*%coefs[["A3"]][[w]]
    prds[,,"A4",w] <-cbind(1,X_4_price,X_A_cross,Xo_prd)%*%coefs[["A4"]][[w]]

    prds[,,"B1",w] <-cbind(1,X_1_price,X_B_cross,Xo_prd)%*%coefs[["A1"]][[w]]
    prds[,,"B2",w] <-cbind(1,X_2_price,X_B_cross,Xo_prd)%*%coefs[["A2"]][[w]]
    prds[,,"B3",w] <-cbind(1,X_3_price,X_B_cross,Xo_prd)%*%coefs[["A3"]][[w]]
    prds[,,"B4",w] <-cbind(1,X_4_price,X_B_cross,Xo_prd)%*%coefs[["A4"]][[w]]

    # prediction errors
    errs[,,,w]	<- log(sales[smpl_prd]) - prds[,,,w]


  }

  assign(paste("prds_", l_prd[p], sep=""), prds)
  assign(paste("errs_", l_prd[p], sep=""), errs)
  rm(prds, errs)

}

rm(smpl_prd, smpl_cross, smpl_est, price_r_prd, 
	cross_premi_prd, cross_mmprm_prd, cross_cithi_prd, 
	cross_flogo_prd, cross_trefr_prd, cross_tropi_prd, cross_domin_prd, 
	X_1_price, X_2_price, X_3_price, X_4_price, knots.w, 
	X_A_cross, X_B_cross, Xo_prd)




##############

# evaluate prediction errors
# transformed errors (squared errors for LS, theta-weighted (-> chck) for QR)

##############


for (p in 1:length(l_prd))
{

  errs_traf_tmp	<- get(paste("errs_", l_prd[p], sep=""))

  # squared errors
  errs_traf_tmp[,1,,]	<- (get(paste("errs_", l_prd[p], sep=""))[,1,,])^2

  # theta-weighted errors
  for (th in 1:length(thetas)) {
  errs_traf_tmp[,th+1,,]	<- chck(get(paste("errs_", l_prd[p], 
							sep=""))[,th+1,,], thetas[th]) }

  assign(paste("errs_traf_", l_prd[p], sep=""), errs_traf_tmp)

  rm(errs_traf_tmp)
}




##############

# evaluation of AEP (average error of prediction)

##############


# overall AEP, i.e. over all 33(36) x 46 obs. (Table 10)

round(apply(errs_traf_1, c(3,2), mean),3)
round(apply(errs_traf_4, c(3,2), mean),3)



# function that determines the fraction of positive observations
perc_better		<- function(x) {mean(x > 0, na.rm=TRUE)}



# percentage of cases where "first" model performs better


# percentage where "first" model has smaller EP

perc_all		<- function(p, AB, mos=4:1)
			   {

				mo_pairs	<- t(outer(mos, mos, FUN="paste"))[
					lower.tri(t(outer(mos, mos, FUN="paste")))]

				tmp	<- matrix(NA, ncol=1+length(thetas), 
						nrow=(length(mos)*(length(mos)-1))/2)
				colnames(tmp)	<- c("mean", thetas)
				rownames(tmp)	<- mo_pairs


				for (mop in mo_pairs)
				{
				  m1	<- paste(AB, strsplit(mop, " ")[[1]][1], sep="")
				  m2	<- paste(AB, strsplit(mop, " ")[[1]][2], sep="")
				  err1	<- get(paste("errs_traf_", l_prd[p], 
								sep=""))[,,m1,]
				  err2	<- get(paste("errs_traf_", l_prd[p], 
								sep=""))[,,m2,]
				  diff	<- err2 - err1
				  tmp[mop,]	<- apply(diff, 2, perc_better)
				}
				tmp
			   }


# percentage of weeks  where "first" model has smaller AEP (-> over s) or
# percentage of stores where "first" model has smaller AEP (-> over w)

perc_sw		<- function(xdim, p, AB, mos=4:1) 
			   {

				d	<- (xdim=="weeks")*3 + (xdim=="stores")*1

				mo_pairs	<- t(outer(mos, mos, FUN="paste"))[
					lower.tri(t(outer(mos, mos, FUN="paste")))]

				tmp	<- matrix(NA, ncol=1+length(thetas), 
						nrow=(length(mos)*(length(mos)-1))/2)
				colnames(tmp)	<- c("mean", thetas)
				rownames(tmp)	<- mo_pairs

				for (mop in mo_pairs)
				{
				  m1	<- paste(AB, strsplit(mop, " ")[[1]][1], sep="")
				  m2	<- paste(AB, strsplit(mop, " ")[[1]][2], sep="")
				  err1	<- apply(get(paste("errs_traf_", l_prd[p], 
								sep=""))[,,m1,], c(d,2), mean)
				  err2	<- apply(get(paste("errs_traf_", l_prd[p], 
								sep=""))[,,m2,], c(d,2), mean)
				  diff	<- err2 - err1
				  tmp[mop,]	<- apply(diff, 2, perc_better)
				}
				tmp
			   }




##############

# graphical results

##############



###

# percentage where "first" model outperforms the other (by model) (Figure 2)

###


plot.perc.al	<- function(p, AB)
{
	mo_pairs	<- t(outer(4:1, 4:1, FUN="paste"))[
					lower.tri(t(outer(4:1, 4:1, FUN="paste")))]
	n_diff	<- length(mo_pairs)

	lgds	<- c("mean", parse(text=paste("vartheta == ", thetas, sep="")))



	par(mfrow=c(3,2), mai=c(1.0,0.6,0.3,0.1)/1.5, mgp=c(2,1,0))

	for (th in 1:(1+length(thetas)))
	{

	  plot(c(-0.5,4.5+7*(n_diff-1)), c(0, 1), type="n", 
			xaxt="n", yaxt="n", 
			xlab="", 
			ylab=paste("percentage: (", AB, 
					",.), first outperforms second", sep=""), 
			main=lgds[th], font.lab=2, font=2)
	  x_pos	<- rep(0:(n_diff-1), each=3)*7 + rep(c(0,2,4),n_diff)
	  axis(1, at=x_pos, 
			labels=rep(c("all", "weeks", "stores"), n_diff), 
			line=-0.5, tick=FALSE, font=2, las=3)
	  labls	<- rep(NA, n_diff)
	  for (md in 1:n_diff)
	  {
	    m1	<- strsplit(mo_pairs[md], " ")[[1]][1]
	    m2	<- strsplit(mo_pairs[md], " ")[[1]][2]
	    labls[md]	<-	paste(m1, " vs. ", m2, sep="")
	  }
	  axis(1, at=2+0:(n_diff-1)*7, labels=labls, line=2.2,tick=FALSE,font=2)
	  axis(2, at=0:10/10, labels=paste(0:10*10, "%", sep=""), 
			font=2, cex.axis=1)
	  abline(h=c(1:4/10, 6:9/10), lty=2)

	  for (md in 1:n_diff)
	  {
	    m1	<- strsplit(mo_pairs[md], " ")[[1]][1]
	    m2	<- strsplit(mo_pairs[md], " ")[[1]][2]
	    arrows( x_pos[(md-1)*3 + 1], 0, x_pos[(md-1)*3 + 1], 
				perc_all(p=p, AB=AB, mos=c(m1,m2))[th],
				length=0, lwd=10, col="grey", lend="butt")
	    arrows( x_pos[(md-1)*3 + 2], 0, x_pos[(md-1)*3 + 2], 
				perc_sw(xdim="weeks", p=p, AB=AB, mos=c(m1,m2))[th], 
				length=0, lwd=10, col="grey", lend="butt")
	    arrows( x_pos[(md-1)*3 + 3], 0, x_pos[(md-1)*3 + 3], 
				perc_sw(xdim="stores", p=p, AB=AB, mos=c(m1,m2))[th], 
				length=0, lwd=10, col="grey", lend="butt")
	  }

	  abline(h=1)
	  abline(h=0.5, lwd=2)
	  abline(h=0)
	}

}

plot.perc.al(p=1, AB="A")




###

# atwep/asep and differences for all weeks or stores, respectively 
#	(by quantile) (Figure 3)

###


plot.diff.q <- function(xdim, p, AB, quant)
{

	d	<- (xdim=="weeks")*3 + (xdim=="stores")*1
	N	<- (xdim=="weeks")*no_preds[p] + (xdim=="stores")*S
	xl	<- if (xdim=="weeks") {"prediction weeks"} else {"stores"}

	err	<- get(paste("errs_traf_", l_prd[p], sep="")
					)[,quant,paste(AB, 4:1, sep=""),]
	err	<- apply(err, c(d,2), mean)

	mo_pairs	<- t(outer(4:1, 4:1, FUN="paste"))[
					lower.tri(t(outer(4:1, 4:1, FUN="paste")))]

	diffs		<- matrix(NA, ncol=length(mo_pairs), nrow=nrow(err))
	colnames(diffs)	<- mo_pairs

	for (md in 1:length(mo_pairs))
	{
	  m1	<- paste(AB, strsplit(mo_pairs[md], " ")[[1]][1], sep="")
	  m2	<- paste(AB, strsplit(mo_pairs[md], " ")[[1]][2], sep="")
	  diffs[,md] 	<- err[,m2] - err[,m1]
	}

	if (quant == "LS") {y_lab	<-	"ASEP"}
	if (quant != "LS") {y_lab	<-	"ATWEP"}

	par(mfrow=c(2,3), mai=c(0.7,0.6,0.3,0.1)/1.5, mgp=c(2,1,0))

	for (md in 1:length(mo_pairs))
	{
	  m1	<- paste(AB, strsplit(mo_pairs[md], " ")[[1]][1], sep="")
	  m2	<- paste(AB, strsplit(mo_pairs[md], " ")[[1]][2], sep="")

	  m1l	<- paste(AB, ",", strsplit(mo_pairs[md], " ")[[1]][1], sep="")
	  m2l	<- paste(AB, ",", strsplit(mo_pairs[md], " ")[[1]][2], sep="")

	  if (quant == "LS")
	  {
	    main_plot	<- paste(y_lab, " and difference  -  LS", sep="")
	  } else {
	    main_plot	<- bquote(paste(.(y_lab), " and difference  -  ", 
								vartheta == .(quant), sep=""))
	  }
	  plot(c(1,N), c(-1,1)*max(err, na.rm=TRUE), 
			type="n", xlab=xl, yaxt="n", ylab="", 
			main=main_plot,
			font.main=2, font.lab=2, font=2)
	  axis(2, at=c(-1,1)*max(err, na.rm=TRUE)/3, tick=FALSE, line=1, 
		labels=c(paste("(",m1l,")", sep=""), paste("(",m2l,")", sep="")), 
		font.lab=2, font=2)
	  at_tmp <- seq(0,floor(1.04*max(err, na.rm=TRUE)*10)/10,length.out=3)
	  axis(2, at=at_tmp[-1], font.lab=2, font=2)
	  axis(2, at=-at_tmp, label=format(at_tmp,digits=3),font.lab=2,font=2)
	  for (nn in 1:N)
	  {
	    arrows(nn, 0, nn, - err[nn,m1], length=0, lwd=3, 
						col="grey", lend="butt")
	    arrows(nn, 0, nn,   err[nn,m2], length=0, lwd=3, 
						col="grey", lend="butt")
	    arrows(nn, 0, nn, diffs[nn,md], length=0, lwd=3,
						col="grey40", lend="butt")
	  }
	
	  abline(h=0)
	}

}

plot.diff.q(xdim="weeks",  p=1, AB="A", quant="0.5")
plot.diff.q(xdim="stores", p=1, AB="A", quant="0.5")




###

# graphic to identify out-of-PRICE-sample predictions

###


plot.outlier	<- function(AB="A", quant, w, p)
{

	# sample, ... for plot
	smpl_est_plot	<- ( (week >= min(week)+w-1) * 
						(week <= min(week)+w-1+l_est-1) ) == TRUE
	smpl_prd_plot	<- (week == min(week) + w-1 + l_est + l_prd[p]-1
						) == TRUE
	smpl_plot		<- (smpl_est_plot==TRUE | smpl_prd_plot==TRUE) == TRUE
	smpl_cross_plot	<- ((week <= min(week) + w-1 + l_est - 1)*
					(week >= min(week) + w-1 + l_est - 5))==TRUE


	# which "quantile"

	qq	<- which(c("LS",thetas) == quant)


	# plot

	par(mfrow=c(1,1), mai=c(0.65,0.6,0.1,0.1), mgp=c(2,1,0))

	plot(price[smpl_plot], log(sales[smpl_plot]), type="n",
		xlab="price", ylab="log(sales)",
		font=2, font.lab=2)
	points(price[smpl_est_plot], log(sales[smpl_est_plot]), 
		cex=0.8, col="grey")
	points(price[smpl_prd_plot], log(sales[smpl_prd_plot]), 
		cex=0.8, col="black", pch=19)

	legend(max(price[smpl_est_plot])-0.03, max(log(sales[smpl_plot])) + 0.1, 
		bg="transparent", 
		xjust=1, yjust=1, 
		title=ifelse(quant=="LS", 
				paste("mean  -  prediction week ", w, sep=""),
			paste(quant, "-quantile  -  prediction week ", w, sep="")),
		legend=c(paste("(", AB, ",1) parametric", sep=""), 
			   paste("(", AB, ",2) unconstr. B-spline", sep=""), 
			   paste("(", AB, ",3) monotone B-spline", sep=""), 
			   paste("(", AB, ",2) smoothing spline", sep=""), 
			   "estimation sample", "prediction sample"), 
		lty=c(4,1,3,2,-1,-1), lwd=c(rep(3,4),1,1), 
		col=c(rep("black",4), "grey", "black"),
		pch=c(rep(-1,4),1,19), cex=0.8,
		seg.len=3.4)


	# cross-prices for the plot

	Xc_median	<- log(apply(cbind(cross_premi, 
						cross_mmprm, cross_cithi, cross_flogo, 
						cross_trefr, cross_tropi, cross_domin)[
						smpl_est_plot,], 2, median))

	# plot A1
	plot(function(x) coefs[["A1"]][[w]][1,qq] + 
		log(x) * coefs[["A1"]][[w]][2,qq] + 
		c(Xc_median %*% coefs[["A1"]][[w]][2+1:length(Xc_median),qq]), 
			lty=4, lwd=3, 
			from=min(price[smpl_plot]), to=max(price[smpl_plot]), 
			add=TRUE)

	# plot A2
	plot(function(x) coefs[["A2"]][[w]][1,qq] + 
		c(splineDesign(knots=knots_b(log(price[smpl_est_plot])), 
			x=within_range(log(x), range(log(price[smpl_est_plot]))), 
					ord=k)[,-1] %*% coefs[["A2"]][[w]][2:(m+k),qq])+ 
		c(Xc_median %*% coefs[["A2"]][[w]][(m+k)+1:length(Xc_median),qq]), 
			lty=1, lwd=3, 
			from=min(price[smpl_plot]), to=max(price[smpl_plot]), 
			add=TRUE)

	# plot A3
	plot(function(x) coefs[["A3"]][[w]][1,qq] + 
		c(splineDesign(knots=knots_b(log(price[smpl_est_plot])), 
			x=within_range(log(x), range(log(price[smpl_est_plot]))), 
					ord=k)[,-1] %*% coefs[["A3"]][[w]][2:(m+k),qq])+ 
		c(Xc_median %*% coefs[["A3"]][[w]][(m+k)+1:length(Xc_median),qq]), 
			lty=3, lwd=3, 
			from=min(price[smpl_plot]), to=max(price[smpl_plot]), 
			add=TRUE)

	# plot A4
	k_tmp		<- length(knots_s(log(price[smpl_est_plot])))-2
	plot(function(x) coefs[["A4"]][[w]][1,qq] + 
		c(splineDesign(knots=knots_s(log(price[smpl_est_plot])), 
			x=within_range(log(x), range(log(price[smpl_est_plot]))), 
					ord=2)[,-1] %*% coefs[["A4"]][[w]][2:k_tmp,qq])+ 
		c(Xc_median %*% coefs[["A4"]][[w]][k_tmp+1:length(Xc_median),qq]), 
			lty=2, lwd=3, 
			from=min(price[smpl_plot]), to=max(price[smpl_plot]), 
			add=TRUE)
}

plot.outlier(AB="A", quant="0.5", w=2, p=1)


