% ----------------------------------------------------------------------- %
%  Monte Carlo exercise to replicate Grassi et al. (2015) Forecasting with 
%  the Standardized Self-Perturbed Kalman Filter
%
%  This script simulates and estimates a model with 2 breaks in the
%  parametes with the same setup given by the Referee. The measurement
%  error variance  is estimated by EWMA as in Koop and Korobilis (2012, 2013). 
%  The measurement error variance  is estimated by EWMA as in Koop and Korobilis (2012, 2013). 
%  Therefore, we introduce the kappa parameter for the 
%  estimation of the EWMA part. The value of kappa is set accondingly to the tipical
%  value found in KK(2012, 2013).
%  The design can be optiomized to improve the performances
%  of the SSP-KF and KK. 
%  In calculating the APD we discard the first 10% of 
%  observations for all the online methods using the trunc parameter. 
%  Of course this can be changed but the results does not change
%  substantially.
%
%  The output reports the results for the 
%    1) The SSP-KF as describe in the paper
%    2) The SSP-KF with another modification discussed at the end of Section 2
%    3) The KK with forgetting factor
%
%   Note: 1) To be completely replicable we should set a seed, something that
%         we do not do here.
%         2) The performances of the three estimator can be improved selecting
%         dynamically the design parameters, something that we do not do
%         here but we carry out in the pape
% ----------------------------------------------------------------------- %
clear all; 
clc; 
close all;
S = 1000;
APD_SSKF_MC = [];
APD_SSKF_New_MC = [];
APD_KK_MC = [];
% Crucial parameters:
% 1) For Monte Carlo set-up, is the Signal-to-Noise ratio
nsr = 0.1;
% 2) For Forgetting factor KF, is the forgetting factor (trajectories are
% very smooth with 0.98)
lambda = 0.98;
% 3) For Self-perturbed KF, is sigma
sigma = 0.02;
% 4) setting the k value for the EWMA 
kappa = 0.96;
%rng(11111, 'twister')
for MC = 1:S
    % Simulate data
    T = 250;
    m = 2;
    Zt = randn(T,m);
    % One break
    theta_1 = [0.2, 0.8];
    theta_2 = [0.4, -0.4];
    % Break dates
    tau1 = round(0.55 * T);
    tau2 = round(0.35 * T); %% One can potentially change the braking date.
    for i = 1:T  
        signal(i,1) = Zt(i,:) * [theta_1((i > tau1) + 1); theta_2((i > tau1) + 1)];
    end
    % simulate variance
    Ht = nsr * var(signal);
    % Generate LHS variable
    y = signal + sqrt(Ht) * randn(T, 1);
    % ---------------- Calling the SSP-KF of the paper ------------------ %
    [mbetat] = funcSPx(y, Zt, sigma, kappa);
    theta_SSKF = mbetat';
    % - Calling the SSP-KF with the modification described in the referee report - %
    [mbetat_new] = funcSPx_Ft(y, Zt, sigma, kappa);
    theta_SSKF_Ft = mbetat_new';
    % -------------- Calling the KK with forgetting factor -------------- %
    [mbetat_KK] = funcKK(y, Zt, lambda, kappa);
    theta_KK = mbetat_KK';
    % ---------------------- Evaluate APD ------------------------------- %
    theta_true = [repmat([theta_1(1), theta_2(1)], tau1, 1); repmat([theta_1(2), theta_2(2)], T - tau1, 1)];
    trunc = floor(T * 0.1);  %% we truncate the first 10% observations due to adaptation of on-line method 
                             %%(results do not change substantially by varying the truncation lag)   
    APD_SSKF = mean(mean(abs(theta_SSKF(trunc + 1:end,:) - theta_true(trunc + 1:end, :))));
    APD_SSKF_New = mean(mean(abs(theta_SSKF_Ft(trunc + 1:end,:) - theta_true(trunc + 1:end,:))));
    APD_KK = mean(mean(abs(theta_KK(trunc + 1:end, :) - theta_true(trunc + 1:end, :))));
    APD_SSKF_MC = [APD_SSKF_MC; APD_SSKF];
    APD_SSKF_New_MC = [APD_SSKF_New_MC; APD_SSKF_New];
    APD_KK_MC = [APD_KK_MC; APD_KK];
end
disp('APD for SSP-KF of the paper')
disp(mean(APD_SSKF_MC))
disp('APD for SSP-KF modify as described in the EndSection 2')
disp(mean(APD_SSKF_New_MC))
disp('APD for KK with forgetting factor')
disp(mean(APD_KK))

figure(1)
plot([theta_true(trunc+1:end,1),theta_SSKF(trunc+1:end,1),theta_SSKF_Ft(trunc+1:end,1),theta_KK(trunc+1:end,1)])
legend('True','SSKF-Paper','SSF-EndSection2','KF-KK')
title('Estimates of the First Parameter')
figure(2)
plot([theta_true(trunc+1:end,2),theta_SSKF(trunc+1:end,2),theta_SSKF_Ft(trunc+1:end,2),theta_KK(trunc+1:end,2)])
legend('True','SSKF-Paper','SSF-EndSection2','KF-KK')
title('Estimates of the Second Parameter')