/* BOUNDSB
** Computes bounds for outcome censoring and regressor censoring.
** Optionally tabulates the results.
**
** INPUTS:  nam     5x1     variable names for cat|Dp|Sp|Dc|Sc (p previous, c current)
**          wnam    1x1     variable name for weight or ""
**          out     1x1     0 no output, 1 tabulate results
**
** OUTPUTS: cmat    ?x1     categories
**          xmat    13x1    table of results
**
** GLOBALS: data,dnam
**
** Note: These bounds differ from those given by Horowitz and Manski (1998),
** since outcome and regressor censoring are not necessarily "joint".
** Specifically, BOUNDSB allow for regressor censoring without outcome
** censoring.
**
** Note: Estimates of variances are provided in columns 12 and 13 of xmat. Note
** that the asymptotic variance has been scaled by the sample size.
**
*/
proc(2)=BOUNDSB(nam,wnam,out);
local tol2,dec,ncat,icat,nt,tol,ta,tz,ii,wgt,cat,Dp,Sp,Dc,Sc,cmat,xmat,tmp;
nt=101; tol=1e-6; tol2=1e-8;
if ismiss(indcv(nam,dnam));
  "BOUNDSB-ERROR  variables not found:";
  format /rd 12,3;
  $selif(nam,indcv(nam,dnam).==miss(0,0));
  end;
endif;
if wnam$/="";
  if ismiss(indcv(wnam,dnam));
    "BOUNDSB-ERROR  weight not found:";
    format /rd 12,3;
    $wnam;
    end;
  endif;
  wgt=data[.,indcv(wnam,dnam)];
  dec=1;
else;
  dec=0;
endif;
cat=data[.,indcv(nam[1],dnam)];
Dp=data[.,indcv(nam[2],dnam)];
Sp=data[.,indcv(nam[3],dnam)];
Dc=data[.,indcv(nam[4],dnam)];
Sc=data[.,indcv(nam[5],dnam)];
cmat=unique(cat,1);
ncat=rows(cmat);
xmat=miss(zeros(ncat,6+4+1+2+2),0);
for icat (1,ncat,1);
  if wnam$/="";
    tmp=(cat.==cmat[icat]).*wgt;
  else;
    tmp=(cat.==cmat[icat]);
  endif;
  xmat[icat,1]= sumc(tmp.*(Dp.==0)          .*(Dc.==0));
  xmat[icat,2]= sumc(tmp.*(Dp.==0)          .*(Dc.==1));
  xmat[icat,3]= sumc(tmp.*(Dp.==0)          .*(Dc.==1).*(Sc.==1));
  xmat[icat,4]= sumc(tmp.*(Dp.==1).*(Sp.==1));
  xmat[icat,5]= sumc(tmp.*(Dp.==1).*(Sp.==1).*(Dc.==1));
  xmat[icat,6]= sumc(tmp.*(Dp.==1).*(Sp.==1).*(Dc.==1).*(Sc.==1));
  xmat[icat,14]=sumc(tmp.*(Dp.==1)          .*(Dc.==0));
  xmat[icat,15]=sumc(tmp);
  if xmat[icat,4]==0;
    xmat[icat,7]=miss(0,0);     @ t1 indeterminable @
    xmat[icat,8]=0;             @ lower bound @
    xmat[icat,9]=miss(0,0);     @ t1 indeterminable @
    xmat[icat,10]=1;            @ upper bound @
    xmat[icat,11]=miss(0,0);    @ conditional prb given observability @
  else;
    /* lower bound */
    xmat[icat,7]=missex(1-xmat[icat,3]/(xmat[icat,2]+(xmat[icat,2]==0)),xmat[icat,2]==0);
    xmat[icat,7]=missex(xmat[icat,7],xmat[icat,6].==0);
    tmp=xmat[icat,4]+xmat[icat,2]-xmat[icat,3]+xmat[icat,1];
    xmat[icat,8]=xmat[icat,6]/tmp;
    /* asymptotic variance of lower bound (implicit div by n since xmat is sums) */
    xmat[icat,12]=xmat[icat,6]*(1-xmat[icat,6])/tmp^2
      +xmat[icat,8]^2/tmp^2*(
        xmat[icat,4]*(1-xmat[icat,4])
        +xmat[icat,2]*(1-xmat[icat,2])
        +xmat[icat,3]*(1-xmat[icat,3])
        +xmat[icat,1]*(1-xmat[icat,1])
        -2*xmat[icat,4]*xmat[icat,2]
        +2*xmat[icat,4]*xmat[icat,3]
        -2*xmat[icat,4]*xmat[icat,1]
        -2*(1-xmat[icat,2])*xmat[icat,3]
        -2*xmat[icat,2]*xmat[icat,1]
        +2*xmat[icat,3]*xmat[icat,1] )
      -2*xmat[icat,8]/tmp^2*xmat[icat,6]*(1-tmp);
    /* upper bound */
    xmat[icat,9]=missex(xmat[icat,3]/(xmat[icat,2]+(xmat[icat,2]==0)),xmat[icat,2]==0);
    tmp=xmat[icat,4]+xmat[icat,3]+xmat[icat,1];
    xmat[icat,10]=(tmp-xmat[icat,5]+xmat[icat,6])/tmp;
    /* asymptotic variance of upper bound (implicit div by n since xmat is sums) */
    xmat[icat,13]=1/tmp^2*(
        xmat[icat,5]*(1-xmat[icat,5])
        +xmat[icat,6]*(1-xmat[icat,6])
        -2*(1-xmat[icat,5])*xmat[icat,6] )
      +(1-xmat[icat,10])^2/tmp^2*(
        xmat[icat,4]*(1-xmat[icat,4])
        +xmat[icat,3]*(1-xmat[icat,3])
        +xmat[icat,1]*(1-xmat[icat,1])
        -2*xmat[icat,4]*xmat[icat,3]
        -2*xmat[icat,4]*xmat[icat,1]
        -2*xmat[icat,3]*xmat[icat,1] )
      -2*(1-xmat[icat,10])/tmp^2*(xmat[icat,5]-xmat[icat,6])*(1-tmp);
    /* conditional probability given observability */
    xmat[icat,11]=missex(xmat[icat,6]/(xmat[icat,5]+(xmat[icat,5]==0)),xmat[icat,5]==0);
  endif;
endfor;
if not out;
  retp(cmat,xmat);
endif;
    /* TABULATE */
?;
format /rd 8,1;
$nam';
if wnam$/="";
  tmp=0$+"weight"|wnam; $tmp';
endif;
format /rd 7,1;
tmp=0$+" cat"|"Dp=0"|"Dp=0"|"Dp=0"|"Dp=1"|"Dp=1"|"Dp=1"|" LB"|" LB"|" UB"|" UB"|"Prb"|" LB"|" UB"; $tmp';
tmp=0$+"    "|"    "|"    "|"    "|"Sp=1"|"Sp=1"|"Sp=1"|" t1"|"Prb"|" t1"|"Prb"|"ran"|" SD"|" SD"; $tmp';
tmp=0$+"    "|"Dc=0"|"Dc=1"|"Dc=1"|"    "|"Dc=1"|"Dc=1"|"   "|"   "|"   "|"   "|"cen"|"   "|"   "; $tmp';
tmp=0$+"    "|"    "|"    "|"Sc=1"|"    "|"    "|"Sc=1"|"   "|"   "|"   "|"   "|"   "|"   "|"   "; $tmp';
for icat (1,ncat,1);
  format /rd 7,0;
  cmat[icat];;
  format /rd 7,dec;
  xmat[icat,1:6];;
  format /rd 7,1;
  xmat[icat,7:13]*100;
endfor;
?;
format /rd 7,1;
tmp=0$+" cat"|" tot"|"Dp=0"|"Dp=0"|"Dp=1"|"Dp=0"|"Dp=0"|"Dp=1"; $tmp';
tmp=0$+"    "|"    "|"Dc=0"|"Dc=1"|"Dc=0"|"Dc=0"|"Dc=1"|"Dc=0"; $tmp';
for icat (1,ncat,1);
  format /rd 7,0;
  cmat[icat];;
  format /rd 7,dec;
  xmat[icat,15] xmat[icat,1 2 14];;
  format /rd 7,1;
  100*xmat[icat,1 2 14]./xmat[icat,15];
endfor;
retp(cmat,xmat);
endp;
