/* BENCHMARK
** Extrapolates backward the number of students in each grade, assuming fixed
** probabilities of repeating grades (eg Pr(11|11)).
**
** INPUTS:  wnam    1x1     variable name for weight or ""
**          rr      5x1     vector of repeat rates for grades 8-12
**          out     1x1     0 no output, 1 tabulate results
**
** GLOBALS: data,dnam
**
** OUTPUTS: year,lab,rlt
**  lab:     rlt column:
**  year    Year
**  p1010   Conditional probability of doing 10 given did 10
**  p1011   Conditional probability of doing 11 given did 10
**  p1111   Conditional probability of doing 11 given did 11
**  p1112   Conditional probability of doing 12 given did 11
**  p1212   Conditional probability of doing 12 given did 12
**  ar11a   Apparent Retention from 10 to 11
**  ar12a   Apparent Retention from 11 to 12
**  ar11e   Apparent Retention from 10 to 11 (adjusted for random R-censoring)
**  ar12e   Apparent Retention from 11 to 12 (adjusted for random R-censoring)
**  rq11    Ratio of doing 11 in t-1 over not doing 10 in t-1 (no adjustment for random R-censoring needed)
**  rq12    Ratio of doing 12 in t-1 over not doing 11 in t-1 (no adjustment for random R-censoring needed)
**  rp11    Ratio of doing 11 in t-1 over not doing 10 in t-1 (no adjustment for random R-censoring needed)
**  rp12    Ratio of doing 12 in t-1 over not doing 11 in t-1 (no adjustment for random R-censoring needed)
**
** Note: This procedure does the same as the Excel worksheet repeatsOct.xls,
** except that grade 8 repetition is allowed here.
**
** Note: The matrix "xmat" has same layout as the spreadsheets, except that a
** column for grade 8 is included for both "first valid record", "other valid
** records" and "added records". Similarly, "ymat" is the matrix of row totals.
**
** Note: Only observations NOT INVALID are used here (equivalent to assuming
** drp==1 is random). Technically, "TMP4" is missing iff drp==1.
**
*/
proc(3)=BENCHMARK(wnam,rr,out);
local mv,n1,n2,nb,ib,ir,ic,nn,ii,jj,ig,iy,cat1,cat2,boct,znam,zmat,xnam,xdat,xmat,tot,rep,ymat;
local year,nta,nte,ntl,ntu,nt10,nd10,n1010,n1011,nt11,nd11,n1111,n1112,nt12,nd12,n1212;
local nu10,nu11,nu12,rlt,lab,tmp;
mv=miss(0,0);
cat1=reshape(seqa(8500,100,12)+seqa(8,1,5)',60,1);
cat2=seqa(108,1,5)|198|199|seqa(208,1,5)|298|299|seqa(408,1,5)|498|499;
boct=packr(unique(data[.,indcv(0$+"BOCT",dnam)],1));
n1=rows(cat1);
n2=rows(cat2);
nb=rows(boct);
znam=0$+"Back"|"Rep"|"Prog"|"Left"|"Unkn"|"Total";
zmat=arrayinit(nb|n1|rows(znam),0);
if wnam$=="";
  @xnam=0$+"SSG2"|"TMP4";@
  xnam=0$+"SSYG"|"SSCODE";
else;
  @xnam=0$+"SSG2"|"TMP4"|wnam;@
  xnam=0$+"SSYG"|"SSCODE"|wnam;
endif;
for ib (1,nb,1);
    /* LOAD DATA INTO TABLE OF FREQUENCIES */
  xdat=selif(data[.,indcv(xnam,dnam)],data[.,indcv(0$+"BOCT",dnam)].==boct[ib]);
  xmat=zeros(n1,n2);
  if wnam$=="";
    for ir (1,n1,1);
      for ic (1,n2,1);
        xmat[ir,ic]=sumc(xdat[.,1].==cat1[ir] .and xdat[.,2].==cat2[ic]);
      endfor;
    endfor;
  else;
    for ir (1,n1,1);
      for ic (1,n2,1);
        xmat[ir,ic]=sumc(xdat[.,3].*(xdat[.,1].==cat1[ir] .and xdat[.,2].==cat2[ic]));
      endfor;
    endfor;
  endif;
    /* EXTRAPOLATE BACKWARD */
  nn=maxc(8800|(boct[ib]-1900+16)*100);
  ii=sumc(cat1.<=nn+12);
  xmat[1:ii,2*7+1:3*7]=zeros(ii,7); @ clear light blue area of spreadsheet @
  do until nn<8500;
    ir=indnv(nn+12,cat1);
    xmat[ir,indnv(412,cat2)]=maxc(0|rr[5]/(1-rr[5])*xmat[ir,indnv(198,cat2)]);
    for ii (11,8,-1);
      ir=indnv(nn+ii,cat1);
      tot=sumc(xmat[indnv(nn+ii+101,cat1),.]'); @ total next grade next year @
      tmp=sumc(xmat[indnv(nn+ii+100,cat1),.]'); @ total same grade next year @
      rep=xmat[ir+1,indnv(ii+101,cat2)]+xmat[ir+1,indnv(ii+201,cat2)]+xmat[ir+1,indnv(ii+401,cat2)]; @ total repeating next grade next year @
      xmat[ir,indnv(ii+400,cat2)]=minc(maxc(0|rr[ii-7]/(1-rr[ii-7])*(xmat[ir,indnv(198,cat2)]+tot-rep)-xmat[ir,indnv(ii+100,cat2)])|tmp);
      if ii==1;
        xmat[ir,indnv(ii+401,cat2)]=maxc(0|tot-rep-xmat[ir,indnv(ii+101,cat2)]);
      else;
        xmat[ir,indnv(ii+401,cat2)]=maxc(0|tot-rep-xmat[ir,indnv(ii+101,cat2)]-xmat[ir+2,indnv(ii+101,cat2)]);
      endif;
    endfor;
    nn=nn-100;
  endo;
    /* MANUAL EDITS (Grade 8 added records) */
  if boct[ib]==1972;
    if wnam$=="";
      xmat[indnv(8908,cat1),indnv(409,cat2)]=xmat[indnv(8908,cat1),indnv(409,cat2)]+1;
    else;
      ii=selif(xdat[.,3],(xdat[.,1].==9009).and(xdat[.,2].==110));
      if rows(ii)==1;
        xmat[indnv(8908,cat1),indnv(409,cat2)]=xmat[indnv(8908,cat1),indnv(409,cat2)]+ii;
      elseif not scalmiss(ii); @ 2005-04 bootstrapping... @
        ii=sumc(ii);
        xmat[indnv(8908,cat1),indnv(409,cat2)]=xmat[indnv(8908,cat1),indnv(409,cat2)]+ii;
      endif;
    endif;
  endif;
  if boct[ib]==1975;
    if wnam$=="";
      xmat[indnv(9208,cat1),indnv(409,cat2)]=xmat[indnv(9208,cat1),indnv(409,cat2)]+1;
    else;
      ii=selif(xdat[.,3],(xdat[.,1].==9309).and(xdat[.,2].==410));
      if rows(ii)==1;
        xmat[indnv(9208,cat1),indnv(409,cat2)]=xmat[indnv(9208,cat1),indnv(409,cat2)]+ii;
      elseif not scalmiss(ii); @ 2005-04 bootstrapping... @
        ii=sumc(ii);
        xmat[indnv(9208,cat1),indnv(409,cat2)]=xmat[indnv(9208,cat1),indnv(409,cat2)]+ii;
      endif;
    endif;
  endif;
    /* ROW TOTALS */
  ymat=zeros(n1,rows(znam));
  ymat[.,rows(znam)]=sumc(xmat');
  for nn (8500,9600,100);
    for ig (8,12,1);
      ir=indnv(nn+ig,cat1);
      if ig>8;
        ymat[ir,1]=sumc(xmat[ir,indnv(108,cat2):indnv(100+ig-1,cat2)]'+xmat[ir,indnv(208,cat2):indnv(200+ig-1,cat2)]'+xmat[ir,indnv(408,cat2):indnv(400+ig-1,cat2)]');
      endif;
      ymat[ir,2]=xmat[ir,indnv(100+ig,cat2)]+xmat[ir,indnv(200+ig,cat2)]+xmat[ir,indnv(400+ig,cat2)];
      if ig<12;
        ymat[ir,3]=sumc(xmat[ir,indnv(100+ig+1,cat2):indnv(112,cat2)]'+xmat[ir,indnv(200+ig+1,cat2):indnv(212,cat2)]'+xmat[ir,indnv(400+ig+1,cat2):indnv(412,cat2)]');
      endif;
      ymat[ir,4]=xmat[ir,indnv(198,cat2)]+xmat[ir,indnv(298,cat2)]+xmat[ir,indnv(498,cat2)];
      ymat[ir,5]=xmat[ir,indnv(199,cat2)]+xmat[ir,indnv(299,cat2)]+xmat[ir,indnv(499,cat2)];
    endfor;
  endfor;
    /* LOAD TABLE INTO ARRAY */
  setarray zmat,ib,ymat;
endfor;
    /* CONDITIONAL PROBABILITIES */
year=seqa(1985,1,12);
nta  =miss(zeros(nb,1996-1984),0); @ current added grade 8 students (watch out for grade 8 repeats!) @
nte  =     zeros(nb,1996-1984);    @ TOTAL: current enrollment (grades 8-12) @
ntu  =     zeros(nb,1996-1984);    @ TOTAL: current status UNobservable (unknown destination previous year) @
ntl  =     zeros(nb,1996-1984);    @ TOTAL: current not-in-schol (left after previous year) @
nt10 =miss(zeros(nb,1996-1984),0); @ did 10 CURRENT year @
nu10 =miss(zeros(nb,1996-1984),0); @ did 10 previous year AND CURRENT YEAR UNOBSERVABLE @
nd10 =miss(zeros(nb,1996-1984),0); @ did 10 previous year AND CURRENT YEAR OBSERVABLE @
n1010=miss(zeros(nb,1996-1984),0); @ did 10 previous year and 10 current year @
n1011=miss(zeros(nb,1996-1984),0); @ did 10 previous year and 11 current year @
nt11 =miss(zeros(nb,1996-1984),0);
nd11 =miss(zeros(nb,1996-1984),0);
nu11 =miss(zeros(nb,1996-1984),0);
n1111=miss(zeros(nb,1996-1984),0);
n1112=miss(zeros(nb,1996-1984),0);
nt12 =miss(zeros(nb,1996-1984),0);
nd12 =miss(zeros(nb,1996-1984),0);
nu12 =miss(zeros(nb,1996-1984),0);
n1212=miss(zeros(nb,1996-1984),0);
for ib (1,nb,1);
  for iy (1985,1996,1);
    ir=indnv((iy-1900)*100+8,cat1);
    nta[ib,iy-1984]=arraytomat(zmat[ib,ir,indcv(0$+"Total",znam)]);
    if iy>1985; @ subtract grade 8 repeats and grade 9 dropbacks @
      nta[ib,iy-1984]=nta[ib,iy-1984]-arraytomat(zmat[ib,indnv((iy-1900-1)*100+8,cat1),indcv(0$+"Rep",znam)]);
      nta[ib,iy-1984]=nta[ib,iy-1984]-arraytomat(zmat[ib,indnv((iy-1900-1)*100+9,cat1),indcv(0$+"Back",znam)]);
    endif;
    for ic (8,12,1);
      ir=indnv((iy-1900)*100+ic,cat1);
      nte[ib,iy-1984]=nte[ib,iy-1984]+arraytomat(zmat[ib,ir,indcv(0$+"Total",znam)]);
    endfor;
    ir=indnv((iy-1900)*100+10,cat1);
    nt10[ib,iy-1984]=arraytomat(zmat[ib,ir,indcv(0$+"Total",znam)]);
    nu10[ib,iy-1984]=arraytomat(zmat[ib,ir,indcv(0$+"Unkn",znam)]);
    ir=indnv((iy-1900)*100+11,cat1);
    nt11[ib,iy-1984]=arraytomat(zmat[ib,ir,indcv(0$+"Total",znam)]);
    nu11[ib,iy-1984]=arraytomat(zmat[ib,ir,indcv(0$+"Unkn",znam)]);
    ir=indnv((iy-1900)*100+12,cat1);
    nt12[ib,iy-1984]=arraytomat(zmat[ib,ir,indcv(0$+"Total",znam)]);
    nu12[ib,iy-1984]=arraytomat(zmat[ib,ir,indcv(0$+"Unkn",znam)]);
  endfor;
  for iy (1986,1996,1);
    for ic (8,12,1);
      ir=indnv((iy-1900-1)*100+ic,cat1);
      ntl[ib,iy-1984]=ntl[ib,iy-1984]+arraytomat(zmat[ib,ir,indcv(0$+"Left",znam)]);
      ntu[ib,iy-1984]=ntu[ib,iy-1984]+arraytomat(zmat[ib,ir,indcv(0$+"Unkn",znam)]);
    endfor;
    ir=indnv((iy-1900-1)*100+10,cat1);
    nd10[ib,iy-1984]=arraytomat(zmat[ib,ir,indcv(0$+"Total",znam)]-zmat[ib,ir,indcv(0$+"Unkn",znam)]);
    n1010[ib,iy-1984]=arraytomat(zmat[ib,ir,indcv(0$+"Rep",znam)]);
    n1011[ib,iy-1984]=arraytomat(zmat[ib,ir,indcv(0$+"Prog",znam)]);
    ir=indnv((iy-1900-1)*100+11,cat1);
    nd11[ib,iy-1984]=arraytomat(zmat[ib,ir,indcv(0$+"Total",znam)]-zmat[ib,ir,indcv(0$+"Unkn",znam)]);
    n1111[ib,iy-1984]=arraytomat(zmat[ib,ir,indcv(0$+"Rep",znam)]);
    n1112[ib,iy-1984]=arraytomat(zmat[ib,ir,indcv(0$+"Prog",znam)]);
    ir=indnv((iy-1900-1)*100+12,cat1);
    nd12[ib,iy-1984]=arraytomat(zmat[ib,ir,indcv(0$+"Total",znam)]-zmat[ib,ir,indcv(0$+"Unkn",znam)]);
    nd12[ib,iy-1984]=arraytomat(zmat[ib,ir,indcv(0$+"Total",znam)]-zmat[ib,ir,indcv(0$+"Unkn",znam)]);
    n1212[ib,iy-1984]=arraytomat(zmat[ib,ir,indcv(0$+"Rep",znam)]);
  endfor;
endfor;
    /* CHECK */
if maxc(maxc(abs(nt10[.,1:11]-nu10[.,1:11]-nd10[.,2:12])))>1e-10;
   "BENCHMARK-ERROR  nt10-nu10 and nd10 do not match";
  end;
endif;
if maxc(maxc(abs(nt11[.,1:11]-nu11[.,1:11]-nd11[.,2:12])))>1e-10;
   "BENCHMARK-ERROR  nt11-nu11 and nd11 do not match";
   end;
endif;
if maxc(maxc(abs(nt12[.,1:11]-nu12[.,1:11]-nd12[.,2:12])))>1e-10;
   "BENCHMARK-ERROR  nt12-nu12 and nd12 do not match";
   end;
endif;
    /* ESTIMATE ENROLLMENT STATUS ASSUMING RANDOM CENSORING (HAZARD APPROACH) (ignore dropbacks) */
xmat=zeros(n1,nb);
for ib (1,nb,1);
  for ic (8,12,1);
    ir=indnv((1985-1900)*100+ic,cat1); @ g,t (year 1985) @
    xmat[ir,ib]=arraytomat(zmat[ib,ir,indcv(0$+"Total",znam)]);
  endfor;
  for iy (1986,1996,1);
    ir=indnv((iy-1900)*100+08,cat1); @ g,t (grade 8) @
    xmat[ir,ib]=arraytomat(zmat[ib,ir,indcv(0$+"Total",znam)]);
    for ic (9,12,1);
      ir=indnv((iy-1900)*100+ic,cat1); @ g,t @
      xmat[ir,ib]=0;
      ii=indnv((iy-1900-1)*100+(ic-1),cat1); @ g-1,t-1 @
      tmp=arraytomat(zmat[ib,ii,indcv(0$+"Total",znam)]-zmat[ib,ii,indcv(0$+"Unkn",znam)]);
      if boct[ib]==1975 and iy==1991;
        tmp=arraytomat(zmat[ib,ii,indcv(0$+"Total",znam)]);
      endif;
      if tmp>0;
        xmat[ir,ib]=xmat[ir,ib]+arraytomat(zmat[ib,ii,indcv(0$+"Prog",znam)])/tmp*xmat[ii,ib];
      endif;
      ii=indnv((iy-1900-1)*100+ic,cat1); @ g,t-1 @
      tmp=arraytomat(zmat[ib,ii,indcv(0$+"Total",znam)]-zmat[ib,ii,indcv(0$+"Unkn",znam)]);
      if boct[ib]==1975 and iy==1991;
        tmp=arraytomat(zmat[ib,ii,indcv(0$+"Total",znam)]);
      endif;
      if tmp>0;
        xmat[ir,ib]=xmat[ir,ib]+arraytomat(zmat[ib,ii,indcv(0$+"Rep",znam)])/tmp*xmat[ii,ib];
      endif;
    endfor;
  endfor;
endfor;
    /* ENROLLMENT */
if out>=2;
  format /rd 6,1;
  ?; "Total Grade 10 Enrollment";
  tmp=(mv~year')|(boct~nt10)|(mv~sumc(nt10)');
  tmp;
  ?; "Total Grade 11 Enrollment";
  tmp=(mv~year')|(boct~nt11)|(mv~sumc(nt11)');
  tmp;
  ?; "Total Grade 12 Enrollment";
  tmp=(mv~year')|(boct~nt12)|(mv~sumc(nt12)');
  tmp;
  ?; "Grade 10 Enrollment with unknown destination";
  tmp=(mv~year')|(boct~nu10)|(mv~sumc(nu10)');
  tmp;
  ?; "Grade 11 Enrollment with unknown destination";
  tmp=(mv~year')|(boct~nu11)|(mv~sumc(nu11)');
  tmp;
  ?; "Grade 12 Enrollment with unknown destination";
  tmp=(mv~year')|(boct~nu12)|(mv~sumc(nu12)');
  tmp;
  ?; "Grade 10 Enrollment - excluding unknown destination";
  tmp=(mv~year')|(boct~nt10-nu10)|(mv~sumc(nt10-nu10)');
  tmp;
  ?; "Total Grade 11 Enrollment - excluding unknown destination";
  tmp=(mv~year')|(boct~nt11-nu11)|(mv~sumc(nt11-nu11)');
  tmp;
  ?; "Total Grade 12 Enrollment - excluding unknown destination";
  tmp=(mv~year')|(boct~nt12-nu12)|(mv~sumc(nt12-nu12)');
  tmp;
endif;
    /* CONDITIONAL PROBABILITIES ETC */
lab=0$+"year";
rlt=year;
format /rd 6,1;
lab=lab|"p1010";
rlt=rlt~sumc(n1010)./sumc(nd10);
if out==1;
  ?; "Conditional probability of doing 10 given did 10";
  tmp=n1010./nd10;
  tmp=(mv~year')|(boct~100*tmp)|(mv~100*rlt[.,cols(rlt)]');
  tmp;
endif;
lab=lab|"p1011";
rlt=rlt~sumc(n1011)./sumc(nd10);
if out==1;
  ?; "Conditional probability of doing 11 given did 10";
  tmp=n1011./nd10;
  tmp=(mv~year')|(boct~100*tmp)|(mv~100*rlt[.,cols(rlt)]');
  tmp;
endif;
lab=lab|"p1111";
rlt=rlt~sumc(n1111)./sumc(nd11);
if out==1;
  ?; "Conditional probability of doing 11 given did 11";
  tmp=n1111./nd11;
  tmp=(mv~year')|(boct~100*tmp)|(mv~100*rlt[.,cols(rlt)]');
  tmp;
endif;
lab=lab|"p1112";
rlt=rlt~sumc(n1112)./sumc(nd11);
if out==1;
  ?; "Conditional probability of doing 12 given did 11";
  tmp=n1112./nd11;
  tmp=(mv~year')|(boct~100*tmp)|(mv~100*rlt[.,cols(rlt)]');
  tmp;
endif;
lab=lab|"p1212";
rlt=rlt~sumc(n1212)./sumc(nd12);
if out==1;
  ?; "Conditional probability of doing 12 given did 12";
  tmp=n1212./nd12;
  tmp=(mv~year')|(boct~100*tmp)|(mv~100*rlt[.,cols(rlt)]');
  tmp;
endif;
    /* APPARENT RETENTION RATES */
lab=lab|"ar11a";
rlt=rlt~(mv|sumc(nt11[.,2:12])./sumc(nt10[.,1:11]));
if out==1;
  ?; "Apparent Retention from 10 to 11";
  tmp=miss(zeros(nb,1),0)~nt11[.,2:12]./nt10[.,1:11];
  tmp=(mv~year')|(boct~100*tmp)|(mv~100*rlt[.,cols(rlt)]');
  tmp;
endif;
lab=lab|"ar12a";
rlt=rlt~(mv|sumc(nt12[.,2:12])./sumc(nt11[.,1:11]));
if out==1;
  ?; "Apparent Retention from 11 to 12";
  tmp=miss(zeros(nb,1),0)~nt12[.,2:12]./nt11[.,1:11];
  tmp=(mv~year')|(boct~100*tmp)|(mv~100*rlt[.,cols(rlt)]');
  tmp;
endif;
lab=lab|"ar11e";
ymat=xmat[indnv(seqa(8500+10,100,12),cat1),.]'; @ Year 10 enrollment estimates @
zmat=xmat[indnv(seqa(8500+11,100,12),cat1),.]'; @ Year 11 enrollment estimates @
rlt=rlt~(mv|sumc(zmat[.,2:12])./sumc(ymat[.,1:11]));
if out==1;
  ?; "Apparent Retention from 10 to 11 (adjusted for random R-censoring)";
  tmp=miss(zeros(nb,1),0)~zmat[.,2:12]./ymat[.,1:11];
  tmp=(mv~year')|(boct~100*tmp)|(mv~100*rlt[.,cols(rlt)]');
  tmp;
endif;
lab=lab|"ar12e";
ymat=xmat[indnv(seqa(8500+11,100,12),cat1),.]'; @ Year 11 enrollment estimates @
zmat=xmat[indnv(seqa(8500+12,100,12),cat1),.]'; @ Year 12 enrollment estimates @
rlt=rlt~(mv|sumc(zmat[.,2:12])./sumc(ymat[.,1:11]));
if out==1;
  ?; "Apparent Retention from 11 to 12 (adjusted for random R-censoring)";
  tmp=miss(zeros(nb,1),0)~zmat[.,2:12]./ymat[.,1:11];
  tmp=(mv~year')|(boct~100*tmp)|(mv~100*rlt[.,cols(rlt)]');
  tmp;
endif;
    /* RATIO OF DOING g IN t-1 OVER DOING g-1 IN t-1 */
lab=lab|"rq11";
ymat=xmat[indnv(seqa(8500+10,100,12),cat1),.]'; @ Year 10 enrollment estimates @
zmat=xmat[indnv(seqa(8500+11,100,12),cat1),.]'; @ Year 11 enrollment estimates @
rlt=rlt~(mv|sumc(zmat[.,1:11])./sumc(ymat[.,1:11])); @ lagged ratio @
if out==1;
  ?; "Ratio of doing 11 in t-1 over not doing 10 in t-1 (no adjustment for random R-censoring needed)";
  tmp=miss(zeros(nb,1),0)~(zmat[.,2:12]./ymat[.,2:12]);
  tmp=(mv~year')|(boct~100*tmp)|(mv~100*rlt[.,cols(rlt)]');
  tmp;
endif;
lab=lab|"rq12";
ymat=xmat[indnv(seqa(8500+11,100,12),cat1),.]'; @ Year 11 enrollment estimates @
zmat=xmat[indnv(seqa(8500+12,100,12),cat1),.]'; @ Year 12 enrollment estimates @
rlt=rlt~(mv|sumc(zmat[.,1:11])./sumc(ymat[.,1:11])); @ lagged ratio @
if out==1;
  ?; "Ratio of doing 12 in t-1 over not doing 11 in t-1 (no adjustment for random R-censoring needed)";
  tmp=miss(zeros(nb,1),0)~(zmat[.,2:12]./ymat[.,2:12]);
  tmp=(mv~year')|(boct~100*tmp)|(mv~100*rlt[.,cols(rlt)]');
  tmp;
endif;
    /* RATIO OF DOING g IN t-1 OVER NOT DOING g-1 IN t-1 */
tot=rev(cumsumc(rev(nta')))'-nta+nte+cumsumc(ntl')'+cumsumc(ntu')'; @ total pop @
lab=lab|"rp11";
rlt=rlt~(mv|sumc(nt11[.,1:11])./sumc(tot[.,1:11]-nt10[.,1:11]));
if out==1;
  ?; "Ratio of doing 11 in t-1 over not doing 10 in t-1 (no adjustment for random R-censoring needed)";
  tmp=miss(zeros(nb,1),0)~(nt11[.,1:11]./(tot[.,1:11]-nt10[.,1:11]));
  tmp=(mv~year')|(boct~100*tmp)|(mv~100*rlt[.,cols(rlt)]');
  tmp;
endif;
lab=lab|"rp12";
rlt=rlt~(mv|sumc(nt12[.,1:11])./sumc(tot[.,1:11]-nt11[.,1:11]));
if out==1;
  ?; "Ratio of doing 12 in t-1 over not doing 11 in t-1 (no adjustment for random R-censoring needed)";
  tmp=miss(zeros(nb,1),0)~(nt12[.,1:11]./(tot[.,1:11]-nt11[.,1:11]));
  tmp=(mv~year')|(boct~100*tmp)|(mv~100*rlt[.,cols(rlt)]');
  tmp;
endif;
retp(year,lab,rlt);
endp;
