/*************************************************
* 	DISCRETE FACTOR LIKELIHOOD FUNCTION			     *
* Two points of support					                 *
* Model originally written for HCSUS			       *
* Mapping of variables are as follows:			     *
*	`risky' = any risky assets				             *
*	`supp'  = medigap (including employer)		     *
*	`hmo'   = hmo						                       *
* `noins' = neither supp nor hmo			           *
* Should exclude Medicaid since not in model		 *
* Normalization for model as shown below			   *
**************************************************

----------------------------------------------------
 CONSTRUCT				ARGUMENT	PARAMETERS
----------------------------------------------------
 Any risky assets     $ML_y1		
 Has supplemental ins $ML_y2
 Has Medicare HMO			$ML_y3
 RISKY index     			THETA1	
 SUPP index   			  THETA2	
 HMO index   			    THETA3	
 First POS in RISKY  	THETA4	RH011	
 First POS in SUPP		THETA5	RH012	
 First POS in HMO			THETA6	RH013	
 Probability pt 1			THETA7	D1  [Note:P1=MNLprob(D1)]


NOTES about likelihood:
(1)   The local macro verbose will print out a lot of stuff about the heterogeneity

(2)   The third POS in all equations is set so mean is zero.  

(3) 	To interpret the subscripts on rho note that the first one
	refers to the point of support, and the second one to the equation.
 	Thus, the POS for equation 1 (risky assets) are rho11, rho21, and rho31 where
	we will set rho31 so that the mean is zero.

(4) 	One problem with earlier versions of this model was that the probabilities did
    	not use an MNL spefication.  Thus, it did not guarantee that P3 was always bounded
    	by 0 and 1.  This version corrects that problem.
*/

#delimit cr
capture program drop DFLike
program define DFLike
	version 7.0
	args lnf theta1 theta2 theta3 theta4 theta5 theta6 theta7 
	local verbose = 0
quietly{

* Define probability distribution of the discrete factor
	scalar p1 = exp(`theta7')/(1+exp(`theta7')+exp(`theta7'))
	scalar p2 = 1-p1

* First point of support
	scalar rho11 = `theta4'
	scalar rho12 = `theta5'
	scalar rho13 = `theta6'

* Second point of support (normalize to mean zero)
	scalar rho21 = -1*(rho11*p1)/p2
	scalar rho22 = -1*(rho12*p1)/p2
	scalar rho23 = -1*(rho13*p1)/p2

* Dummy variables for outcomes
* Six possible: (hmo, supp, or noins) x (risky, norisky)
	tempvar risky supp hmo noins norisk
	 gen double `risky'=$ML_y1
	 gen double `supp'=$ML_y2
	 gen double `hmo'=$ML_y3
	 gen double `noins' = 1-`supp'-`hmo'
	 gen double `norisk'=1-`risky'

* Probabilities given first POS
	tempvar prrisk1 expsupp1 exphmo1 prsupp1 prhmo1 prnoins1
	 gen double `prrisk1' = normprob(`theta1'+rho11)
	 gen double `expsupp1'= exp(`theta2'+rho12)
	 gen double `exphmo1' = exp(`theta3'+rho13)
	 gen double `prsupp1' = `expsupp1'/(1+`expsupp1'+`exphmo1')
	 gen double `prhmo1'  = `exphmo1'/(1+`expsupp1'+`exphmo1')
	 gen double `prnoins1'= 1/(1+`expsupp1'+`exphmo1')
	
* Probabilities given second POS
	tempvar prrisk2 expsupp2 exphmo2 prsupp2 prhmo2 prnoins2
	 gen double `prrisk2' = normprob(`theta1'+rho21)
	 gen double `expsupp2'= exp(`theta2'+rho22)
	 gen double `exphmo2' = exp(`theta3'+rho23)
	 gen double `prsupp2' = `expsupp2'/(1+`expsupp2'+`exphmo2')
	 gen double `prhmo2'  = `exphmo2'/(1+`expsupp2'+`exphmo2')
	 gen double `prnoins2'= 1/(1+`expsupp2'+`exphmo2')

* Define the likelihood of each possible observation
* Note:  Code is likeABC where A=risky, B=supp, C=hmo
	tempvar like100 like000 like110 like010 like101 like001
	 gen double `like100'=p1*`prrisk1'*`prnoins1'+p2*`prrisk2'*`prnoins2'
	 gen double `like000'=p1*(1-`prrisk1')*`prnoins1'+p2*(1-`prrisk2')*`prnoins2'
	 gen double `like110'=p1*`prrisk1'*`prsupp1'+p2*`prrisk2'*`prsupp2'
	 gen double `like010'=p1*(1-`prrisk1')*`prsupp1'+p2*(1-`prrisk2')*`prsupp2'
	 gen double `like101'=p1*`prrisk1'*`prhmo1'+p2*`prrisk2'*`prhmo2'
	 gen double `like001'=p1*(1-`prrisk1')*`prhmo1'+p2*(1-`prrisk2')*`prhmo2'

	 replace `lnf' = weight_new*ln(`like100') if `risky' & `noins'
	 replace `lnf' = weight_new*ln(`like000') if `norisk' & `noins'
	 replace `lnf' = weight_new*ln(`like110') if `risky' & `supp'
	 replace `lnf' = weight_new*ln(`like010') if `norisk' & `supp'
	 replace `lnf' = weight_new*ln(`like101') if `risky' & `hmo'
	 replace `lnf' = weight_new*ln(`like001') if `norisk' & `hmo'

 } /* End the quietly bracket */

************ Check Likelihood Function ***************
if `verbose' {
	scalar mean_eq1 = p1*rho11 + p2*rho21
	scalar mean_eq2 = p1*rho12 + p2*rho22
	scalar mean_eq3 = p1*rho13 + p2*rho23
	scalar var_eq1  = p1*(rho11-mean_eq1)^2 + p2*(rho21-mean_eq1)^2
	scalar var_eq2  = p1*(rho12-mean_eq2)^2 + p2*(rho22-mean_eq2)^2
	scalar var_eq3  = p1*(rho13-mean_eq3)^2 + p2*(rho23-mean_eq3)^2
	#d ;
	di as text "Equation 1: " _newline(1)
	as text "             rho11 = " as result %10.5f rho11 _newline(1)
	as text "             rho21 = " as result %10.5f rho21 _newline(1)
	as text "             mean  = " as result %10.5f mean_eq1 _newline(1)
	as text "             var   = " as result %10.5f var_eq1 _newline(1)
	as text "Equation 2: " _newline(1)
	as text "             rho12 = " as result %10.5f rho12 _newline(1)
	as text "             rho22 = " as result %10.5f rho22 _newline(1)
	as text "             mean  = " as result %10.5f mean_eq2 _newline(1)
	as text "             var   = " as result %10.5f var_eq2 _newline(1)
	as text "Equation 3: " _newline(1)
	as text "             rho13 = " as result %10.5f rho13 _newline(1)
	as text "             rho23 = " as result %10.5f rho23 _newline(1)
	as text "             mean  = " as result %10.5f mean_eq3 _newline(1)
	as text "             var   = " as result %10.5f var_eq3 _newline(1)
	as text "Probabilities: " _newline(1)
	as text "             p1 = " as result %10.5f p1 _newline(1)
	as text "             p2 = " as result %10.5f p2 _newline(1);
	#d cr
} /* end check likelihood function */


end
