/********************************************************************************/
/* Program creating a dataset at the municipality level				  */
/* It contains the fixed effects and explanatory variables for the third-stage  */
/* regressions. When it has run, the output dataset town_alpha.dta should be    */
/* exported to GAUSS                                                            */
/********************************************************************************/



#delimit;

clear;

set matsize 500;
set mem 500m;

local path  "C:\dossiers\SpatialM\GMS_ANPE\publication_JAE_doc\";



/*******************************************************************************/
/* Creation of the dataset with local variables summarizing the information    */
/* of the dataset at the individual level							 */
/*******************************************************************************/

u "`path'panel", clear;

gen mat1=1-mat2;
gen edu1=1-edu2-edu3-edu4;
gen nat1=1-nat2-nat3-nat4-nat5;
gen sex1=1-sex2;
gen enf0=1-enf1-enf2-enf3-enf4-enf5;

gen te=t*e; replace te=. if e==0;
gen ta=t*a; replace ta=. if a==0;

gen nbcho=1;
gen nbcho_30=(t>=30);
gen nbcho_90=(t>=90);
gen nbcho_180=(t>=180);
gen nbcho_360=(t>=360);
gen nbcho_720=(t>=720);

sort depcom;

merge depcom using "`path'datacom.dta", keep(depcom pop90);
keep if _merge==1|_merge==3;

collapse (sum) nbcho nbcho_30 nbcho_90 nbcho_180 nbcho_360 nbcho_720
         (mean) te ta e a c age sex1 sex2 mat1 mat2 enf0 enf1 enf2 enf3 enf4 enf5 nat1 nat2 nat3 nat4 nat5 edu1 edu2 edu3 edu4 handi, by(depcom pop90);

rename depcom dc;
sort dc;
saveold "`path'town_alpha.dta", replace;



/*********************************************************/
/* Adding local variables coming from the duration model */ 
/*********************************************************/

/* Adding averages by municipality */

use "`path'alpha", clear;
sort dc;
merge dc using "`path'town_alpha.dta";
drop _merge;

/* Correctif pour les valeurs manquantes */

replace xbe=. if xbe==100000; replace xba=. if xba==100000; replace nbe=. if nbe==100000; replace nba=. if nba==100000;
replace alphae=. if alphae==100000; replace alphaa=. if alphaa==100000;
replace sme6=. if sme6==100000; replace sme12=. if sme12==100000; replace sme24=. if sme24==100000; 
replace sma6=. if sma6==100000; replace sma12=. if sma12==100000; replace sma24=. if sma24==100000; 
replace smulte6=. if smulte6==100000; replace smulte12=. if smulte12==100000; replace smulte24=. if smulte24==100000; 
replace smulta6=. if smulta6==100000; replace smulta12=. if smulta12==100000; replace smulta24=. if smulta24==100000; 
replace ske6=. if ske6==100000; replace ske12=. if ske12==100000; replace ske24=. if ske24==100000; 
replace ska6=. if ska6==100000; replace ska12=. if ska12==100000; replace ska24=. if ska24==100000; 

replace xbe4_8=. if xbe4_8==100000; replace xba4_8=. if xba4_8==100000; replace nbe4_8=. if nbe4_8==100000; replace nba4_8=. if nba4_8==100000;
replace xbe10_14=. if xbe10_14==100000; replace xba10_14=. if xba10_14==100000; replace nbe10_14=. if nbe10_14==100000; replace nba10_14=. if nba10_14==100000;
replace xbe22_26=. if xbe22_26==100000; replace xba22_26=. if xba22_26==100000; replace nbe22_26=. if nbe22_26==100000; replace nba22_26=. if nba22_26==100000;

rename dc depcom; 
sort depcom;

/* Adding census variables and accessibility variables at the municipality level */

merge depcom using "`path'datacom.dta";
drop _merge;

rename depcom dc;

/* Creating some variables and renaming some variables */

replace alphae=0 if nbe==0 & nbcho>0; 
replace alphaa=0 if nba==0 & nbcho>0; 

replace smulte6=1   if nbe==0 & nbcho>0; 
replace smulte12=1  if nbe==0 & nbcho>0; 
replace smulte24=1  if nbe==0 & nbcho>0; 

replace smulta6=1   if nba==0 & nbcho>0; 
replace smulta12=1  if nba==0 & nbcho>0; 
replace smulta24=1  if nba==0 & nbcho>0; 

gen lHme6=ln(-ln(sme6));
gen lHme12=ln(-ln(sme12));
gen lHme24=ln(-ln(sme24));

gen lHma6=ln(-ln(sma6));
gen lHma12=ln(-ln(sma12));
gen lHma24=ln(-ln(sma24));

gen lHmulte6=ln(-ln(smulte6));
gen lHmulte12=ln(-ln(smulte12));
gen lHmulte24=ln(-ln(smulte24));

gen lHmulta6=ln(-ln(smulta6));
gen lHmulta12=ln(-ln(smulta12));
gen lHmulta24=ln(-ln(smulta24));

gen lHke6=ln(-ln(ske6));
gen lHke12=ln(-ln(ske12));
gen lHke24=ln(-ln(ske24));

gen lHka6=ln(-ln(ska6));
gen lHka12=ln(-ln(ska12));
gen lHka24=ln(-ln(ska24));

gen lalphae=ln(alphae);
gen lalphaa=ln(alphaa);

sort dc;



/***********/
/* Labels  */
/***********/

label var dc "Town code";

label variable enf0	"0 child";
label variable sex1	"Male";
label variable mat1	"Single";
label variable nat1	"French";
label variable edu1	"Secondary school diploma";

label var nbcho "Number of unemployed workers";
label var pop90 "Population in 1990";
label var pop99 "Population in 1999";

label var xbe "Mean Prediction Xb (job)";
label var xba "Mean Prediction Xb (other)";

label var nbe "Number of exits (job)";
label var nba "Number of exits (other)";

label var alphae "Town fixed effect (job)";
label var alphaa "Town fixed effect (other)";

label var sme6  "S. 6 months, model (job)";
label var sme12 "S. 1 year, model (job)";
label var sme24 "S. 2 years, model (job)";

label var sma6  "S. 6 months, model (other)";
label var sma12 "S. 1 year, model (other)";
label var sma24 "S. 2 years, model (other)";

label var smulte6  "S. 6 months, model, reconstructed (job)";
label var smulte12 "S. 1 year, model, reconstructed (job)";
label var smulte24 "S. 2 years, model, reconstructed (job)";

label var smulta6  "S. 6 months, model, reconstructed (other)";
label var smulta12 "S. 1 year, model, reconstructed (other)";
label var smulta24 "S. 2 years, model, reconstructed (other)";

label var lHme6  "ln (-lnS. 6 months), model (job)";
label var lHme12 "ln (-lnS. 1 year), model (job)";
label var lHme24 "ln (-lnS. 2 years), model (job)";

label var lHma6  "ln (-lnS. 6 months), model (other)";
label var lHma12 "ln (-lnS. 1 year), model (other)";
label var lHma24 "ln (-lnS. 2 years), model (other)";

label var ske6  "S. 6 months, Kaplan (job)";
label var ske12 "S. 1 year, Kaplan (job)";
label var ske24 "S. 2 years, Kaplan (job)";

label var ska6  "S. 6 months, Kaplan (other)";
label var ska12 "S. 1 year, Kaplan (other)";
label var ska24 "S. 2 years, Kaplan (other)";

label var lHke6  "ln (-lnS. 6 months), Kaplan (job)";
label var lHke12 "ln (-lnS. 1 year), Kaplan (job)";
label var lHke24 "ln (-lnS. 2 years), Kaplan (job)";

label var lHka6  "ln (-lnS. 6 months), Kaplan (other)";
label var lHka12 "ln (-lnS. 1 year), Kaplan (other)";
label var lHka24 "ln (-lnS. 2 years), Kaplan (other)";

label variable te	"Unemployment duration if exit to job";
label variable ta	"Unemployment duration if exit to non-employment";

label var alphae "Town fixed effect (job)";
label var alphaa "Town fixed effect (other)";

label var lalphae "Log Town fixed effect (job)";
label var lalphaa "Log Town fixed effect (other)";

label variable nbcho	 "Number of unemployed workers at initial date";
label variable nbcho_30	 "Number of unemployed workers at 30 days";
label variable nbcho_90	 "Number of unemployed workers at 90 days";
label variable nbcho_180 "Number of unemployed workers at 180 days";
label variable nbcho_360 "Number of unemployed workers at 360 days";
label variable nbcho_720 "Number of unemployed workers at 720 days";

saveold "`path'town_alpha", replace;



/************************************************************************/
/* Variance analysis at the municipality level, cf. Table 5			*/
/************************************************************************/

gen lHmeXbe6 =xbe4_8+lHme6;
gen lHmeXbe12=xbe10_14+lHme12;
gen lHmeXbe24=xbe22_26+lHme24;

gen lHmueXbe6 =xbe4_8+lHmulte6;
gen lHmueXbe12=xbe10_14+lHmulte12;
gen lHmueXbe24=xbe22_26+lHmulte24;

gen lHmaXba6 =xba4_8+lHma6;
gen lHmaXba12=xba10_14+lHma12;
gen lHmaXba24=xba22_26+lHma24;

gen lHmuaXba6 =xba4_8+lHmulta6;
gen lHmuaXba12=xba10_14+lHmulta12;
gen lHmuaXba24=xba22_26+lHmulta24;

correlate xbe4_8 lHke6  lHme6  lHmeXbe6  lHmulte6  lHmueXbe6  [freq=nbe4_8], covariance;
correlate xbe10_14 lHke12 lHme12 lHmeXbe12 lHmulte12 lHmueXbe12 [freq=nbe10_14], covariance;
correlate xbe22_26 lHke24 lHme24 lHmeXbe24 lHmulte24 lHmueXbe24 [freq=nbe22_26], covariance;

correlate xbe4_8 lHke6  lHme6  lHmeXbe6  lHmulte6  lHmueXbe6  [freq=nbe4_8];
correlate xbe10_14 lHke12 lHme12 lHmeXbe12 lHmulte12 lHmueXbe12 [freq=nbe10_14];
correlate xbe22_26 lHke24 lHme24 lHmeXbe24 lHmulte24 lHmueXbe24 [freq=nbe22_26];

correlate xba4_8 lHka6  lHma6  lHmaXba6  lHmulta6  lHmuaXba6  [freq=nba4_8], covariance;
correlate xba10_14 lHka12 lHma12 lHmaXba12 lHmulta12 lHmuaXba12 [freq=nba10_14], covariance;
correlate xba22_26 lHka24 lHma24 lHmaXba24 lHmulta24 lHmuaXba24 [freq=nba22_26], covariance;

correlate xba4_8 lHka6  lHma6  lHmaXba6  lHmulta6  lHmuaXba6  [freq=nba4_8];
correlate xba10_14 lHka12 lHma12 lHmaXba12 lHmulta12 lHmuaXba12 [freq=nba10_14];
correlate xba22_26 lHka24 lHma24 lHmaXba24 lHmulta24 lHmuaXba24 [freq=nba22_26];



/************************************************************/
/* Construction of the dataset used for Graph 6			*/
/************************************************************/

keep if pop99>10000 & missing(pop90)==0; 

egen mxbe10_14=mean(xbe10_14);
egen mlHme12=mean(lHme12);

replace xbe10_14=xbe10_14-mxbe10_14;
replace lHme12=lHme12-mlHme12;

correlate xbe10_14 lHme12;

keep dc pop99 xbe10_14 lHme12;

save "`path'town_alpha_graph", replace; /* Dataset used to construct graph 6 */

