% table2.m
%
% Program for:
% Marc Giannoni "Robust Optimal Monetary Policy in a Forward-Looking Model 
% with Parameter and Shock Uncertainty" in Journal of Applied Econometrics.
%
% Program generates figures 1 and 2.
% Compute Statistics and Optimal Rules for different values of nu, rho 
% Generate numbers for Table 2.
% Requires: opl3f.m
%
% Last revision: December 2005.

% *****************************************************************
% [0] Set parameters as inputs
% *****************************************************************

% Parameters in baseline model:
b  = .99;       % beta
s0 = .1571;     % sigma
a  = .66;       % alpha
g  = .6314; 	% gamma
k0 = .0238; 	% kappa
w0  = .4729; 	% omega
t  = (1-a*b)*(1-a)*(w0+s0)/(a*k0*w0) - 1/w0;	% theta
lx = k0/t;      % LAMBDA, RW(97,98)
li = .236405;

clear a g t

% 95% intervals for sigma and kappa (Amato and Laubach)
ses = .0328;  % estimated standard error for sigma
sek = .0035;  % estimated standard error for kappa
sew = .0946;  % estimated standard error for omega (20%)

sl = max(s0-2*ses, 0);	% lower bound for s
sh = s0+2*ses;		% upper bound for s
kl = max(k0-2*sek, 0);	% lower bound for k
kh = k0+2*sek;		% upper bound for k
wl = max(w0-2*sew, 0);	% lower bound for w
wh = w0+2*sew;		% upper bound for w

rd0 = .35;
re0 = rd0;
rm0 = rd0;

rh = .8;

%disp(' ')
%disp('Parameters of candidate minmax equilibrium')
sm = sl; %input('s (sl, s0, sh) = ');
km = kh; %input('k (kl, k0, kh) = ');
wm = wl; %input('w (wl, w0, wh) = ');
rdm = rh;
rem = rh;
rmm = rh;
num = 1;

% Coefficients of optimal baseline rule
psipi0 = k0/(li*s0);
psix00 = lx/(li*s0);
psix10 = -lx/(li*s0);
psii10 = 1+k0/(b*s0)+1/b;
psii20 = -1/b;

% Coefficients of robust policy rule
psipim = km/(li*sm);
psix0m = lx/(li*sm);
psix1m = -lx/(li*sm);
psii1m = 1+km/(b*sm)+1/b;
psii2m = -1/b;

% Assume AR(1) for exog. disturbances
%	re(t)  = rho re(t-1) + e(t)
r_ser = [0 .35 .9];	% rho

nu_ser = [0 .5 1]; %nu

disp('nu     psi_pi  psi_x0 psi_x1 psi_i1 psi_i2  V[pi]  V[x]  V[i] E[L_0]') 

TABOP = zeros(size(nu_ser,2)*4, 10); % Opt. pan
TABJR = zeros(size(nu_ser,2)*2, 10); % Judd-Rudebusch rule
    

% *****************************************************************
% [1] Optimal Plan
% *****************************************************************

disp('Optimal Plan')

it = 0;

for i = 1:size(nu_ser, 2)

	nu = nu_ser(i);

	%for j = 1:size(r_ser, 2)

		%rd = r_ser(j);
        %re = rd;
        %rm = rd;
        
		% Optimal rule (psi0, theta0)    
        [Vi00, Vpi00, Vx00, EL00, NUE] ...
            = opl3f(psipi0, psix00, psix10, psii10, psii20, k0, s0, w0, rd0, re0, rm0, nu, 0);

		TABOP(it*4+1, :) = [nu, psipi0, psix00*4, psix10*4, psii10, psii20, Vpi00, Vx00, Vi00, EL00]; 
        
        
   		% Optimal rule (psi0, theta*)    
        [Vi0m, Vpi0m, Vx0m, EL0m, NUE] ...
            = opl3f(psipi0, psix00, psix10, psii10, psii20, km, sm, wm, rdm, rem, rmm, nu, 0);

		TABOP(it*4+2, :) = [nu, psipi0, psix00*4, psix10*4, psii10, psii20, Vpi0m, Vx0m, Vi0m, EL0m]; 

   		% Optimal rule (psi*, theta0)    
        [Vim0, Vpim0, Vxm0, ELm0, NUE] ...
            = opl3f(psipim, psix0m, psix1m, psii1m, psii2m, k0, s0, w0, rd0, re0, rm0, nu, 0);

		TABOP(it*4+3, :) = [nu, psipim, psix0m*4, psix1m*4, psii1m, psii2m, Vpim0, Vxm0, Vim0, ELm0]; 
        
  		% Optimal rule (psi*, theta*)    
        [Vimm, Vpimm, Vxmm, ELmm, NUE] ...
            = opl3f(psipim, psix0m, psix1m, psii1m, psii2m, km, sm, wm, rdm, rem, rmm, nu, 0);

        TABOP(it*4+4, :) = [nu, psipim, psix0m*4, psix1m*4, psii1m, psii2m, Vpimm, Vxmm, Vimm, ELmm]; 

		it = it+1;  % row of TAB
    %end
end

disp(TABOP)

% *****************************************************************
% [4] Estimated Historical Rule: Judd-Rudebusch (1998)
% *****************************************************************

disp('Estimated Historical Rule: Judd-Rudebusch (1998)')

% Estimated rule:
psipi0jr = 0.4239;
psix0jr = 0.297/4;
psix1jr = -0.0324/4;
psii1jr = 1.16;
psii2jr = -0.43;
psi0jr = 0;

it = 0;

for i = 1:size(nu_ser, 2)

	nu = nu_ser(i);

	%for j = 1:size(r_ser, 2)

		%rd = r_ser(j);
        %re = rd;
        %rm = rd;
        
		% Estimated rule (psi, theta0)    
        [Vi0, Vpi0, Vx0, EL0, NUE] ...
            = opl3f(psipi0jr, psix0jr, psix1jr, psii1jr, psii2jr, k0, s0, w0, rd0, re0, rm0, nu, 0);

		TABJR(it*2+1, :) = [nu, psipi0jr, psix0jr*4, psix1jr*4, psii1jr, psii2jr, Vpi0, Vx0, Vi0, EL0]; 
        
        % Estimated rule (psi, theta*)   
        [Vim, Vpim, Vxm, ELm, NUE] ...
            = opl3f(psipi0jr, psix0jr, psix1jr, psii1jr, psii2jr, km, sm, wm, rdm, rem, rmm, nu, 0);

		TABJR(it*2+2, :) = [nu, psipi0jr, psix0jr*4, psix1jr*4, psii1jr, psii2jr, Vpim, Vxm, Vim, ELm]; 

 		it = it+1;  % row of TAB
    %end
end

disp(TABJR)




% save table2.txt TABOP TABJR -ascii -tabs

