function [Vi, Vpi, Vx, EW, NUE] ...
	 = opl2f(psipi, psiDx, psii, k, s, w, rd, re, rm, nu, output);

% opl2f.m
%
% Program for:
% Marc Giannoni "Robust Optimal Monetary Policy in a Forward-Looking Model 
% with Parameter and Shock Uncertainty" in Journal of Applied Econometrics.
%
% Computes state space and statistics for given parameter and policy
%
% Needs: calib6f.m
%
% Last revision: December 2005.

% *****************************************************************
% [0] Input parameters
% *****************************************************************

% Parameters in baseline model:
b  = .99;       % beta
s0 = .1571;     % sigma
a  = .66;       % alpha
g  = .6314; 	% gamma
k0 = .0238; 	% kappa
w0  = .4729; 	% omega
t  = (1-a*b)*(1-a)*(w0+s0)/(a*k0*w0) - 1/w0;	% theta
lx = k0/t;	% LAMBDA, RW(97,98)
li = .236405;

clear a g t s0 k0 w0

Vshocks = calib6f(nu)/400^2;

% Define matrices of state space model for exog. disturbances e(t)
%	s(t+1) = T s(t) + e(t+1)
%	e(t)  = kk' s(t)
% where s(t) = [v(t), mu(t)]
T = [rd, 0, 0; 0, re, 0; 0, 0, rm];

kk = eye(3);
ns = size(T, 1);

% *****************************************************************
% [1] Construct M and m, and find eigenvalues and eigenvectors of M
% of dynamic system
%	[E(t)z(t+1)] = M [z(t)  ] + m rn(t)
%	[    k(t)  ]     [k(t-1)]
%
% where z(t) = [pi(t) x(t)]', k(t) = [x(t) i(t) i(t-1)]'.
% *****************************************************************

M = [1/b,  -k/b, 0, 0, 0;
    psipi/s-1/(b*s),  1+k/(b*s)+psiDx/s, -psiDx/s, psii/s, -1/(b*s);
    0, 1, 0, 0, 0;
    psipi,  psiDx, -psiDx, psii, -1/b;
    0, 0, 0, 1, 0];

m = [0, 0, -k/(b*(w+s));
     -w/((w+s)*s), -1/(w+s), k/(b*s*(w+s));
      zeros(3,3)];

% Matrices of Eigenvalues (GA0) and of left-eigenvectors (V0)
[V0, GA0] = eig(M');
V0 = V0';
GA0 = GA0';

NUE = sum((abs(diag(GA0))>1));

% check that eigenvalues correspond to eigenvectors
for i = 1:5
	if (sum(abs(V0(i, :)*M - GA0(i, i)*V0(i, :))) >= .000001)
		disp('eigenvect. and eigenval. don''t match')
	end
end

if NUE == 2

% *****************************************************************
% [3] Construct the matrices V, GAMMA, W, X, 
% *****************************************************************

	% construct V, GAMMA, W, and X
	V = zeros(2, 5);
	GAM = zeros(2);

	[SOR ind] = sort(abs(diag(GA0)));
	V = V0([ind(5) ind(4)], :);
	GAM(1, 1) = GA0(ind(5), ind(5));
	GAM(2, 2) = GA0(ind(4), ind(4));

	W = V(:, 1:2);
	X = V(:, 3:5);

% *****************************************************************
% [4] Construct matrices of:  	 z(t) = G k(t-1) + g s(t)
% 								 k(t) = N k(t-1) + n s(t)
% *****************************************************************

	E = W*m(1:2, :)*kk';
	Et = [E(1, :); zeros(1, ns)]*inv(eye(ns)-T/GAM(1, 1)) + ...
	     [zeros(1, ns); E(2, :)]*inv(eye(ns)-T/GAM(2, 2));
	g = -inv(W)*inv(GAM)*Et;
	G = -inv(W)*X;

	n = -M(3:5, 1:2)*inv(W)*inv(GAM)*Et;
	N = M(3:5, 1:2)*G+M(3:5, 3:5);

% *****************************************************************
% [5] Construct matrices Do, Go, and Ho of state space: 
%	y(t) = Do K(t)
%	K(t+1) = Go K(t) + Ho e(t+1)
%	
%	where: 	y(t) = [z(t)' k(t)']'
%		K(t) = [s(t)' z(t-1)' k(t-1)']'
% *****************************************************************

	Do = zeros(5, ns+5);
	Do(:, 1:ns) = [g; n];
	Do(:, ns+3:ns+5) = [G; N];

	D1o = Do(:, 1:ns);
	D2o = Do(:, ns+1:ns+5);

	Go = [T, zeros(ns, 5); Do];

	Ho = [eye(ns); zeros(5, ns)];

% *****************************************************************
% [6] Construct Jbar as in notes for OMPI
% *****************************************************************

	%vecJb11 = inv(eye(ns*ns) - kron(T, T))*OM(:);
	Jb11 = Vshocks;

	vec1 = b*T*Jb11*D1o';
	vecJb12 = inv(eye(ns*5) - kron(D2o, b*T))*vec1(:);
	Jb12 = reshape(vecJb12, ns, 5);

	vec2 = b*(D1o*Jb11*D1o'+D2o*Jb12'*D1o'+D1o*Jb12*D2o');
	vecJb22 = inv(eye(5*5) - kron(D2o, b*D2o))*vec2(:);
	Jb22 = reshape(vecJb22, 5, 5);

	clear vec1 vec2

   Jb = [Jb11, Jb12; Jb12', Jb22];

% *****************************************************************
% [7] Compute statistics:
%	V[q] = a' Jb22 a /b,  for q = a' y
%	E[W] = tr(Qo Jb22) / b
% *****************************************************************

	Vpi = Jb(ns+1, ns+1) /b;
	Vx  = Jb(ns+2, ns+2) /b;
	Vi  = Jb(ns+4, ns+4) /b;
 
	EW = Vpi + lx*Vx + li*Vi;

	% Express statistics in annual, % terms
	Vpi = Vpi*400^2;
	Vx  = Vx*100^2;
	Vi  = Vi*400^2;
	EW  = EW*400^2;

% *****************************************************************
% [9] If NUE <> 2, don't compute anything
% *****************************************************************

else
	Vpi = NaN;
	Vx  = NaN;
	Vi  = NaN;
	EW  = NaN;
end

if output == 1

	disp(' ')
	format compact
	disp(sprintf('b  = %6.4f', b))
	disp(sprintf('s  = %6.4f', s))
	disp(sprintf('k  = %6.4f', k))
	disp(sprintf('lx = %6.4f', lx*16))
	disp(sprintf('li = %6.4f', li))
	disp('Note: lx is multiplied by 16 for same time units')

	disp(' ')
	disp('Parameters of optimal policy rule')
	disp(sprintf('psipi = %6.4f', psipi))
	disp(sprintf('psiDx = %6.4f', psiDx*4))
	disp(sprintf('psii  = %6.4f', psii))
	disp('Note: psiDx is multiplied by 4 for same time units')

	disp(' ')
	disp(['Nb. of unstable eigenvalues of M = ' num2str(NUE)]);


	if NUE == 2

		disp(' ')
		disp('Statistics in %, annualized terms:')
		disp(' ')
		%disp(sprintf('Input: sd(rn) = %6.4f', sqrt(varrn)*4*100))
		%disp(' ')
		disp(sprintf('V[pi] = %6.4f', Vpi))
		disp(sprintf('V[x]  = %6.4f', Vx))
		disp(sprintf('V[i]  = %6.4f', Vi))
		disp(sprintf('E[W]  = %6.4f', EW))

	else
		disp('Nb. of unstable eigenvalues <> 2')
	end
end

format
