% fig3.m
%
% Program for:
% Marc Giannoni "Robust Optimal Monetary Policy in a Forward-Looking Model 
% with Parameter and Shock Uncertainty" in Journal of Applied Econometrics.
%
% Program generates figure 3.
% Plot of cumulated coefficients on pi and Dx, for:
%	- Optimal rule 
%		i(t) = a pi(t) + b Dx(t) + c1 i(t-1) - 1/b i(t-2)
%   - Robust optimal rule
%	- Rule estimated by Judd and Rudebusch (1999): Greenspan A
% Requires: irf.m
%
% Last revision: December 2005.

% ****************************************************************
% [0] Set Parameters
% ****************************************************************

% Parameters in baseline model:
b  = .99; 	% beta
s0 = .1571;	% sigma
a  = .66; 	% alpha
g  = .6314; % gamma
k0 = .0238; % kappa
w0 = .4729; % omega
t  = (1-a*b)*(1-a)*(w0+s0)/(a*k0*w0) - 1/w0;	% theta
lx = k0/t;	% LAMBDA, RW(97,98)
li = .236405;

clear a g t

% 95% intervals for sigma and kappa (Amato and Laubach)
ses = .0328;  % estimated standard error for sigma
sek = .0035;  % estimated standard error for kappa
sew = .0946;  % estimated standard error for omega (20%)

sl = max(s0-2*ses, 0);	% lower bound for s
sh = s0+2*ses;		% upper bound for s
kl = max(k0-2*sek, 0);	% lower bound for k
kh = k0+2*sek;		% upper bound for k
wl = max(w0-2*sew, 0);	% lower bound for w

% Minmax equilibrium:
km = kh;
sm = sl;

% ****************************************************************
% [1] Compute cumulated coefficients
% ****************************************************************

% x-axis 
xp_end = 6; %input('Max lags on pi (> 3) : ');
xy_end = 6; %input('Max lags on x  (> 3) : ');

xp = 0:xp_end;
xy = 0:xy_end;

% 1. Optimal Rule: i(t) = a pi(t) + b Dx(t) + c i(t-1) - 1/b i(t-2)
% ---------------------------------------------------------------

% Optimal policy coefficients

% Optimal policy rule
a0 = k0/(li*s0);
b0 = lx/(li*s0);
c0 = 1+k0/(b*s0)+1/b;

% Robust optimal policy rule
am = km/(li*sm);
bm = lx/(li*sm);
cm = 1+km/(b*sm)+1/b;

ORpcu0 = zeros(size(xp));
ORxcu0 = zeros(size(xy));
ORpcum = zeros(size(xp));
ORxcum = zeros(size(xy));

% Autocorrelation coefficient for pi and x
rhop = 1;
rhox = 1;

%  State Space System:
%	y(t)   = D k(t) + F e(t)
%	k(t+1) = G k(t) + H e(t)
%
% where: k(t) = [pi(t) x(t) x(t-1) r(t-1) r(t-2)]';
%	 y(t) = r(t)

% Construct the matrices D, F, G, H.
% baseline case
D0 = [a0  b0  -b0  c0 -1/b];
G0 = [rhop 0   0   0   0;
      0  rhox  0   0   0;
      0    1   0   0   0;
      a0  b0  -b0  c0 -1/b;
      0    0   0   1   0];

F = [0 0];
H = [eye(2); zeros(3, 2)];

% worst case
Dm = [am  bm  -bm  cm -1/b];
Gm = [rhop 0   0   0   0;
      0  rhox  0   0   0;
      0    1   0   0   0;
      am  bm  -bm  cm -1/b;
      0    0   0   1   0];
 
% Coefficients on pi
ORpcu00 = irf(1, xp_end+2, D0, F, G0, H);
ORpcu0 = ORpcu00(:,2:xp_end+2);

ORpcum0 = irf(1, xp_end+2, Dm, F, Gm, H);
ORpcum = ORpcum0(:,2:xp_end+2);

% Coefficients on Y
ORxcu00 = irf(2, xp_end+2, D0, F, G0, H);
ORxcu0 = ORxcu00(:,2:xp_end+2);

ORxcum0 = irf(2, xp_end+2, Dm, F, Gm, H);
ORxcum = ORxcum0(:,2:xp_end+2);

% 2. Rule estimated by Judd-Rudebusch (1999)
% i(t) = ajr pi(t) + bjr0 x(t) + bjr1 x(t-1) ...
%	+ cjr1 i(t-1) + cjr2 i(t-2)
% ------------------------------------------------

ajr = .42;
bjr0 = .30/4;
bjr1 = -.03/4;
cjr1 = 1.16;
cjr2 = -.43;


JRpcu = zeros(size(xp));
JRxcu = zeros(size(xy));

% Autocorrelation coefficient for pi and x
rhop = 1;
rhox = 1;

%  State Space System:
%	y(t)   = D k(t) + F e(t)
%	k(t+1) = G k(t) + H e(t)
%
% where: k(t) = [pi(t) x(t) x(t-1) r(t-1) r(t-2)]';
%	 y(t) = r(t)

% Construct the matrices D, F, G, H.
% baseline case
DJR = [ajr  bjr0  bjr1  cjr1 cjr2];
GJR = [rhop 0   0   0   0;
      0  rhox  0   0   0;
      0    1   0   0   0;
      ajr  bjr0  bjr1  cjr1 cjr2;
      0    0   0   1   0];

% Coefficients on pi
JRpcu0 = irf(1, xp_end+2, DJR, F, GJR, H);
JRpcu = JRpcu0(:,2:xp_end+2);

% Coefficients on Y
JRxcu0 = irf(2, xp_end+2, DJR, F, GJR, H);
JRxcu = JRxcu0(:,2:xp_end+2);

% ****************************************************************
% [2] Display Coefficients on pi and Y for various policy rules
% ****************************************************************

disp(' ')
disp('Cumulated Coefficients on pi:')
disp('-----------------------------')
disp(' ')
disp('       j    Opt. rule  Robust   Judd-Rud.')
disp('    ------- --------- --------- ---------')
disp(' ')
disp([xp' ORpcu0' ORpcum' JRpcu']);

disp(' ')
disp('Cumulated Coefficients on x:')
disp('-----------------------------')
disp(' ')
disp('       j    Opt. rule  Robust   Judd-Rud.')
disp('    ------- --------- --------- ---------')
disp(' ')
disp([xy' 4*ORxcu0' 4*ORxcum' 4*JRxcu']);


% ****************************************************************
% [3] Plot
% ****************************************************************
	
figure('Name', 'Response to Permanent Increase in pi and x')
subplot(211), 
	semilogy(xp, ORpcu0, '-', xp, ORpcum, '--', xp, JRpcu, '-.'),
	title('Response to Permanent Increase in Inflation')

	% options
	set(gca, 'Xlim', [0 xp_end])
	set(gca, 'Ylim', [0 5000])
	legend('baseline rule', 'robust rule', 'estimated historical ','Location', 'NorthWest');

subplot(212), 
	semilogy(xy, 4*ORxcu0, '-', xy, 4*ORxcum, '--', xy, 4*JRxcu, '-.'),
	title('Response to Permanent Increase in the Output Gap')

	% options
	set(gca, 'Xlim', [0 xy_end])
	set(gca, 'Ylim', [0 200])
	legend('baseline rule', 'robust rule', 'estimated historical ','Location', 'NorthWest');
% print fig03.eps -deps

