% fig1_2.m
%
% Program for:
% Marc Giannoni "Robust Optimal Monetary Policy in a Forward-Looking Model 
% with Parameter and Shock Uncertainty" in Journal of Applied Econometrics.
%
% Program generates figures 1 and 2.
% Computes and plots impulse response functions for optimal rule and
% robust optimal rule
% Requires: irf.m, calib6f.m
%
% Last revision: December 2005.

% *****************************************************************
% [0] Set parameters as inputs
% *****************************************************************

% Parameters in baseline model:
b  = .99; 	% beta
s0 = .1571;	% sigma
a  = .66; 	% alpha
g  = .6314; % gamma
k0 = .0238; % kappa
w0 = .4729; % omega
t  = (1-a*b)*(1-a)*(w0+s0)/(a*k0*w0) - 1/w0;	% theta
lx = k0/t;	% lambda, RW(97,98)
li = .236405;

clear a g t

rd = 0.35; % rho
re = rd; 
rm = rd;

% 95% intervals for sigma and kappa (Amato and Laubach)
ses = .0328;  % estimated standard error for sigma
sek = .0035;  % estimated standard error for kappa
sew = .0946;  % estimated standard error for omega (20%)

sl = max(s0-2*ses, 0);	% lower bound for s
sh = s0+2*ses;		% upper bound for s
kl = max(k0-2*sek, 0);	% lower bound for k
kh = k0+2*sek;		% upper bound for k
wl = max(w0-2*sew, 0);	% lower bound for w
wh = w0+2*sew;		% upper bound for w

nu = .5; % input('nu = ');
Vshocks = calib6f(nu)/(400^2);

nir = 11; %input('Nb. of impulse responses (incl. t=0) = ');

% Minmax equilibrium:
km = kh;
sm = sl;
wm = wl;

% Optimal policy rule
psipi = k0/(li*s0);
psiDx = lx/(li*s0);
psii = 1+k0/(b*s0)+1/b;

% Robust optimal policy rule
psipim = km/(li*sm);
psiDxm = lx/(li*sm);
psiim = 1+km/(b*sm)+1/b;

% Define matrices of state space model for exog. disturbances e(t)
%	s(t+1) = T s(t) + e(t+1)
%	e(t)  = kk' s(t)
% where s(t) = [v(t), mu(t)]
T = [rd, 0, 0; 0, re, 0; 0, 0, rm];

kk = eye(3);
ns = size(T, 1);

% disp('Evaluate at: (1) baseline parametrization')
% disp('             (2) minmax equilibrium')
calib = 1; %input('Choose 1 or 2: ');

if calib == 2
   k = km;
   s = sm;
   w = wm;
else   
   k = k0;
   s = s0;
   w = w0;
end

% *****************************************************************
% [1] Construct M and m, and find eigenvalues and eigenvectors of M
% of dynamic system
%	[E(t)z(t+1)] = M [z(t)  ] + m rn(t)
%	[    k(t)  ]     [k(t-1)]
%
% where z(t) = [pi(t) x(t)]', k(t) = [x(t) i(t) i(t-1)]'.
% *****************************************************************

M = [1/b,  -k/b, 0, 0, 0;
    psipi/s-1/(b*s),  1+k/(b*s)+psiDx/s, -psiDx/s, psii/s, -1/(b*s);
    0, 1, 0, 0, 0;
    psipi,  psiDx, -psiDx, psii, -1/b;
    0, 0, 0, 1, 0];
 
m = [0, 0, -k/(b*(w+s));
     -w/((w+s)*s), -1/(w+s), k/(b*s*(w+s));
      zeros(3,3)];
   
% with robust optimal policy rule
Mm = [1/b,  -k/b, 0, 0, 0;
    psipim/s-1/(b*s),  1+k/(b*s)+psiDxm/s, -psiDxm/s, psiim/s, -1/(b*s);
    0, 1, 0, 0, 0;
    psipim,  psiDxm, -psiDxm, psiim, -1/b;
    0, 0, 0, 1, 0];
 
% Matrices of Eigenvalues (GA0) and of left-eigenvectors (V0)
[V0, GA0] = eig(M');
V0 = V0';
GA0 = GA0';

[V0m, GA0m] = eig(Mm');
V0m = V0m';
GA0m = GA0m';

NUE = sum((abs(diag(GA0))>1));
NUEm = sum((abs(diag(GA0m))>1));

% check that eigenvalues correspond to eigenvectors
for i = 1:5
	if (sum(abs(V0(i, :)*M - GA0(i, i)*V0(i, :))) >= .000001)
		disp('eigenvect. and eigenval. don''t match')
	end
	if (sum(abs(V0m(i, :)*Mm - GA0m(i, i)*V0m(i, :))) >= .000001)
		disp('eigenvect. and eigenval. don''t match')
	end
end

if NUE == 2 & NUEm == 2

% *****************************************************************
% [3] Construct the matrices V, GAMMA, W, X, 
% *****************************************************************

	% construct V, GAMMA, W, and X
	V = zeros(2, 5);
	GAM = zeros(2);

	[SOR ind] = sort(abs(diag(GA0)));
	V = V0([ind(5) ind(4)], :);
	GAM(1, 1) = GA0(ind(5), ind(5));
	GAM(2, 2) = GA0(ind(4), ind(4));

	W = V(:, 1:2);
	X = V(:, 3:5);

   	% construct V, GAMMA, W, and X
	Vm = zeros(2, 5);
	GAMm = zeros(2);

	[SORm indm] = sort(abs(diag(GA0m)));
	Vm = V0m([indm(5) indm(4)], :);
	GAMm(1, 1) = GA0m(indm(5), indm(5));
	GAMm(2, 2) = GA0m(indm(4), indm(4));

	Wm = Vm(:, 1:2);
	Xm = Vm(:, 3:5);

   
% *****************************************************************
% [4] Construct matrices of:  	 z(t) = G k(t-1) + g s(t)
% 								 k(t) = N k(t-1) + n s(t)
% *****************************************************************

	E = W*m(1:2, :)*kk';
	Et = [E(1, :); zeros(1, ns)]*inv(eye(ns)-T/GAM(1, 1)) + ...
	     [zeros(1, ns); E(2, :)]*inv(eye(ns)-T/GAM(2, 2));
	g = -inv(W)*inv(GAM)*Et;
	G = -inv(W)*X;

	n = -M(3:5, 1:2)*inv(W)*inv(GAM)*Et;
	N = M(3:5, 1:2)*G+M(3:5, 3:5);

   	Em = Wm*m(1:2, :)*kk';
	Etm = [Em(1, :); zeros(1, ns)]*inv(eye(ns)-T/GAMm(1, 1)) + ...
	     [zeros(1, ns); Em(2, :)]*inv(eye(ns)-T/GAMm(2, 2));
	gm = -inv(Wm)*inv(GAMm)*Etm;
	Gm = -inv(Wm)*Xm;

	nm = -Mm(3:5, 1:2)*inv(Wm)*inv(GAMm)*Etm;
	Nm = Mm(3:5, 1:2)*Gm+Mm(3:5, 3:5);

% *****************************************************************
% [5] Construct matrices Do, Go, and Ho of state space: 
%	y(t) = Do K(t)
%	K(t+1) = Go K(t) + Ho e(t+1)
%	
%	where: 	y(t) = [z(t)' k(t)']'
%           K(t) = [s(t)' z(t-1)' k(t-1)']'
% *****************************************************************

	
    % State space with optimal policy
    Do = zeros(5, ns+5);
	Do(:, 1:ns) = [g; n];
	Do(:, ns+3:ns+5) = [G; N];

	D1o = Do(:, 1:ns);
	D2o = Do(:, ns+1:ns+5);

	Go = [T, zeros(ns, 5); Do];
    Ho = [eye(ns); zeros(5, ns)];
   
    % State space with robust optimal policy
   	Dm = zeros(5, ns+5);
	Dm(:, 1:ns) = [gm; nm];
	Dm(:, ns+3:ns+5) = [Gm; Nm];

	D1m = Dm(:, 1:ns);
	D2m = Dm(:, ns+1:ns+5);

	Gm = [T, zeros(ns, 5); Dm];
    Hm = [eye(ns); zeros(5, ns)];


% *****************************************************************
% [6] Construct Jbar as in notes for OMPI
% *****************************************************************

	%vecJb11 = inv(eye(ns*ns) - kron(T, T))*OM(:);
	Jb11 = Vshocks;

	vec1 = b*T*Jb11*D1o';
	vecJb12 = inv(eye(ns*5) - kron(D2o, b*T))*vec1(:);
	Jb12 = reshape(vecJb12, ns, 5);

	vec2 = b*(D1o*Jb11*D1o'+D2o*Jb12'*D1o'+D1o*Jb12*D2o');
	vecJb22 = inv(eye(5*5) - kron(D2o, b*D2o))*vec2(:);
	Jb22 = reshape(vecJb22, 5, 5);

	clear vec1 vec2
    
    Jb = [Jb11, Jb12; Jb12', Jb22];

% *****************************************************************
% [7] Compute statistics:
%	V[q] = a' Jb22 a /b,  for q = a' y
%	E[W] = tr(Qo Jb22) / b
% *****************************************************************

	Vpi = Jb(ns+1, ns+1) /b;
	Vx  = Jb(ns+2, ns+2) /b;
	Vi  = Jb(ns+4, ns+4) /b;
 
	EW = Vpi + lx*Vx + li*Vi;

	% Express statistics in annual, % terms
	Vpi = Vpi*400^2;
	Vx  = Vx*100^2;
	Vi  = Vi*400^2;
   	EW  = EW*400^2;
   
   	if imag(EW)> 1e-6
      disp('Imaginary component of E[L] > 1e-6')
   	else
      EW = real(EW);
   	end
    if imag(Vpi)> 1e-6
      disp('Imaginary component of V[pi] > 1e-6')
   	else
      Vpi = real(Vpi);
	end
   	if imag(Vx)> 1e-6
      disp('Imaginary component of V[x] > 1e-6')
   	else
      Vx = real(Vx);
	end
   	if imag(Vi)> 1e-6
      disp('Imaginary component of V[i] > 1e-6')
   	else
      Vi = real(Vi);
	end
  
% *****************************************************************
% [9] If NUE <> 2, don't compute anything
% *****************************************************************

else
	Vpi = NaN;
	Vx  = NaN;
	Vi  = NaN;
	EW  = NaN;
end

format

% *****************************************************************
% [4] Compute IRF's
% We write the system into the following state space
% representation, and compute the IRF's to a unit increase in e(t).
%
%	y(t) = Do K(t)
%	K(t+1) = Go K(t) + Ho e~(t)
%	
%	where: 	y(t) = [pi(t) x(t) x(t) i(t) i(t-1)]'
%		K(t) = [delta(t) eps(t), mu(t), y(t-1)]'
%		e~(t) = e(t+1)
% *****************************************************************

% IRF to shock to delta:
% ----------------------
Yod0 = irf(1, nir+1, Do, zeros(5, 3), Go, Ho);
% remove the first col of Yo0 since it contains the zero effect of 
% e~(t) = e(t+1) on y(t).
Yod = Yod0(:, 2:nir+1);

% IRF to shock to eps:
% --------------------
Yoe0 = irf(2, nir+1, Do, zeros(5, 3), Go, Ho);
% remove the first col of Yo0 since it contains the zero effect of 
% e~(t) = e(t+1) on y(t).
Yoe = Yoe0(:, 2:nir+1);

% IRF to shock to delta:
% ----------------------
Yom0 = irf(3, nir+1, Do, zeros(5, 3), Go, Ho);
% remove the first col of Yo0 since it contains the zero effect of 
% e~(t) = e(t+1) on y(t).
Yom = Yom0(:, 2:nir+1);


clear Yoe0 Yod0 Yom0

% IRF to shock to delta:
% ----------------------
Yodm0 = irf(1, nir+1, Dm, zeros(5, 3), Gm, Hm);
% remove the first col of Yo0 since it contains the zero effect of 
% e~(t) = e(t+1) on y(t).
Yodm = Yodm0(:, 2:nir+1);

% IRF to shock to eps:
% --------------------
Yoem0 = irf(2, nir+1, Dm, zeros(5, 3), Gm, Hm);
% remove the first col of Yo0 since it contains the zero effect of 
% e~(t) = e(t+1) on y(t).
Yoem = Yoem0(:, 2:nir+1);

% IRF to shock to delta:
% ----------------------
Yomm0 = irf(3, nir+1, Dm, zeros(5, 3), Gm, Hm);
% remove the first col of Yo0 since it contains the zero effect of 
% e~(t) = e(t+1) on y(t).
Yomm = Yomm0(:, 2:nir+1);


% *****************************************************************
% *****************************************************************


% IRF of rn = w/(w+s) delta(t) + s/(w+s) eps(t) = [w/(w+s), s/(w+s), 0] u(t)
% u(t+1) = T u(t) + e(t+1)
Yrnd0 = irf(1, nir+1, [w/(w+s), s/(w+s), 0], zeros(1,3), T, eye(3));
Yrnd = Yrnd0(:, 2:nir+1);

% IRF of rn = w/(w+s) delta(t) + s/(w+s) eps(t) = [w/(w+s), s/(w+s), 0] u(t)
% u(t+1) = T u(t) + e(t+1)
Yrne0 = irf(2, nir+1, [w/(w+s), s/(w+s), 0], zeros(1,3), T, eye(3));
Yrne = Yrne0(:, 2:nir+1);

%clear Yrnd0 Yrne0 

% *****************************************************************
% [11] Generate the figures 
% *****************************************************************

%Figure 1: Shock to \delta
figure
	subplot(311), plot(0:nir-1, Yod(4, :), '-', 0:nir-1, Yodm(4, :), '--', ...
			0:nir-1, Yrnd, '-.'), 
		title('Interest rate ')
		legend('baseline rule  ', 'robust rule  ', 'r^e  ',0);
	subplot(312), plot(0:nir-1, Yod(1, :), '-', 0:nir-1, Yodm(1, :), '--'), 
	      title('Inflation')
	subplot(313), plot(0:nir-1, Yod(2, :)/4, '-', 0:nir-1, Yodm(2, :)/4, '--'), 
	      title('Output gap')
% print fig01.eps -deps

%Figure 2: Shock to \mu
figure
	subplot(311), plot(0:nir-1, Yom(4, :), '-', 0:nir-1, Yomm(4, :), '--'), 
		title('Interest rate ')
		legend('baseline rule  ', 'robust rule  ',0);
	subplot(312), plot(0:nir-1, Yom(1, :), '-', 0:nir-1, Yomm(1, :), '--'), 
	      title('Inflation')
	subplot(313), plot(0:nir-1, Yom(2, :)/4, '-', 0:nir-1, Yomm(2, :)/4, '--'), 
	      title('Output gap')
% print fig02.eps -deps


 
% % Display numerical values
% disp(' ')
% disp('Opt. Monetary Policy: IRFs of pi, x, i')
% disp('--------------------------------------')
% disp(' ')
% disp('IRF to shock to delta:')
% disp(' ')
% disp('     time     pi(t)      x(t)      i(t) ')
% disp('   -------  --------   -------   -------')
% disp([(0:nir-1)' Yod(1, :)', Yod(2, :)'/4, Yod(4, :)'])
% disp(' ')
% disp('IRF to shock to eps:')
% disp(' ')
% disp('     time     pi(t)      x(t)      i(t)  ')
% disp('   -------  --------   -------   -------')
% disp([(0:nir-1)' Yoe(1, :)', Yoe(2, :)'/4, Yoe(4, :)'])
% disp(' ')
% disp('IRF to shock to mu:')
% disp(' ')
% disp('     time     pi(t)      x(t)      i(t)  ')
% disp('   -------  --------   -------   -------')
% disp([(0:nir-1)' Yom(1, :)', Yom(2, :)'/4, Yom(4, :)'])
% 
% disp(' ')
% disp('Robust Monetary Policy: IRFs of pi, x, i')
% disp('----------------------------------------')
% disp(' ')
% disp('IRF to shock to delta:')
% disp(' ')
% disp('     time     pi(t)      x(t)      i(t) ')
% disp('   -------  --------   -------   -------')
% disp([(0:nir-1)' Yodm(1, :)', Yodm(2, :)'/4, Yodm(4, :)'])
% disp(' ')
% disp('IRF to shock to eps:')
% disp(' ')
% disp('     time     pi(t)      x(t)      i(t)  ')
% disp('   -------  --------   -------   -------')
% disp([(0:nir-1)' Yoem(1, :)', Yoem(2, :)'/4, Yoem(4, :)'])
% disp(' ')
% disp('IRF to shock to mu:')
% disp(' ')
% disp('     time     pi(t)      x(t)      i(t)  ')
% disp('   -------  --------   -------   -------')
% disp([(0:nir-1)' Yomm(1, :)', Yomm(2, :)'/4, Yomm(4, :)'])
