function Vshocks = calib6f(nu)

% calib6f.m
%
% Program for:
% Marc Giannoni "Robust Optimal Monetary Policy in a Forward-Looking Model 
% with Parameter and Shock Uncertainty" in Journal of Applied Econometrics.
%
% Calibration of Variance-covariance matrix of shocks (Vshocks) for given nu.
%
% Sections [0]-[8] are taken from the programs par13.m and var13.m used for 
% the paper Rotemberg and Woodford (NBER Macroeconomics Annual 1997). See M.
% Woodford's webpage: http://www.columbia.edu/~mw2230
%
% Last revision: December 2005.



% ****************************************************************
% [0] Input parameters
% ****************************************************************

%par13.m
b = .99; % input('beta = ');
s = .1571; % input('sigma = ');
a = .66; % input('alpha = ');
g = .6314; % input('gamma = ');
k = .0238; % input('kappa = ');
w = .4729; % input('omega = ');
t = (1-a*b)*(1-a)*(w+s)/(a*k*w) - 1/w;

% var13
% ****************************************************************
% [1] Raw data input: 
%
% The following matrix contains 66 quarterly data points on:
%
% (output, inflation, interest rate)
%
% ****************************************************************

data = [
 1979.25      8.435679        0.088565      0.101800;
 1979.50      8.442211        0.079497      0.109467;
 1979.75      8.444838        0.077947      0.135767;
 1980.00      8.449835        0.090203      0.150467;
 1980.25      8.425648        0.088214      0.126867;
 1980.50      8.424990        0.099425      0.098367;
 1980.75      8.444859        0.097014      0.158533;
 1981.00      8.464067        0.100952      0.165700;
 1981.25      8.455594        0.074076      0.177800;
 1981.50      8.467667        0.066719      0.175767;
 1981.75      8.455020        0.077441      0.135867;
 1982.00      8.437912        0.064376      0.142267;
 1982.25      8.441909        0.051911      0.145133;
 1982.50      8.437132        0.045584      0.110067;
 1982.75      8.438756        0.039465      0.092867;
 1983.00      8.448871        0.039079      0.086533;
 1983.25      8.470374        0.044200      0.088033;
 1983.50      8.487867        0.032832      0.094600;
 1983.75      8.505849        0.043361      0.094300;
 1984.00      8.527817        0.048226      0.096833;
 1984.25      8.542822        0.031830      0.105600;
 1984.50      8.550048        0.036818      0.113900;
 1984.75      8.556933        0.020888      0.092667;
 1985.00      8.568133        0.041451      0.084767;
 1985.25      8.573725        0.030809      0.079233;
 1985.50      8.588024        0.025494      0.079000;
 1985.75      8.593950        0.035421      0.081033;
 1986.00      8.606193        0.020101      0.078267;
 1986.25      8.606961        0.020000      0.069200;
 1986.50      8.612067        0.029814      0.062067;
 1986.75      8.617400        0.034504      0.062667;
 1987.00      8.623605        0.029340      0.062200;
 1987.25      8.633517        0.029126      0.066500;
 1987.50      8.642239        0.033715      0.068433;
 1987.75      8.656851        0.033433      0.069167;
 1988.00      8.662661        0.028436      0.066633;
 1988.25      8.672777        0.042279      0.071567;
 1988.50      8.678717        0.051074      0.079833;
 1988.75      8.691264        0.036740      0.084700;
 1989.00      8.700964        0.045455      0.094433;
 1989.25      8.708375        0.044944      0.097267;
 1989.50      8.713779        0.031163      0.090833;
 1989.75      8.714896        0.030922      0.086133;
 1990.00      8.724874        0.048114      0.082500;
 1990.25      8.728167        0.051836      0.082433;
 1990.50      8.723427        0.038441      0.081600;
 1990.75      8.712924        0.042284      0.077433;
 1991.00      8.707466        0.050158      0.064267;
 1991.25      8.711789        0.028971      0.058633;
 1991.50      8.714288        0.028762      0.056433;
 1991.75      8.716765        0.024490      0.048167;
 1992.00      8.728313        0.032421      0.040233;
 1992.25      8.734592        0.028155      0.037700;
 1992.50      8.742079        0.016000      0.032567;
 1992.75      8.752629        0.027847      0.030367;
 1993.00      8.752581        0.035521      0.030400;
 1993.25      8.756793        0.019598      0.030000;
 1993.50      8.762552        0.019503      0.030600;
 1993.75      8.773988        0.023278      0.029900;
 1994.00      8.780265        0.026988      0.032133;
 1994.25      8.792018        0.019166      0.039400;
 1994.50      8.800792        0.022879      0.044867;
 1994.75      8.808563        0.022749      0.051667;
 1995.00      8.810102        0.030132      0.058100;
 1995.25      8.811265        0.022451      0.060200;
 1995.50      8.820005        0.022326      0.057967];

% ****************************************************************
% [2] Detrend output series. Now (data) is a 66x4 matrix whose
% second column is output minus linear trend plus intercept.
%
% We also turn inflation and interest rates from annual into
% quarterly rates, by dividing the corresponding columns of (data)
% by 4.
% ****************************************************************

constant = ones(66, 1);
trend = (1:66)';

coefficients = [constant trend]\data(:, 2);

data(:, 2) = data(:, 2) - trend*coefficients(2);
data(:, 3) = data(:, 3)/4;
data(:, 4) = data(:, 4)/4;

% ****************************************************************
% [3] Let:
%
% S(t) = [R(t), pi(t+1), Y(t+1), R(t-1), pi(t), Y(t), ...
%                                     ... R(t-2), pi(t-1), Y(t-1)]
%
% Specify the VAR:
%
% Tw*S(t) = Tcw + Aw*S(t-1) + e(t).
%
% Estimate the matrices Tw and Aw and the vector Tcw.
%
% Also construct the historical shock series e1(t), e2(t), e3(t)
% for t = 8000 to 9525 (each a row vector of length 62).
% ****************************************************************

constant = ones(62, 1);

Tw = eye(9);
Aw = zeros(9);
Aw(4:9, 1:6) = eye(6);
Tcw = zeros(9, 1);

regressand = data(4:65, 4);
regressors = [data(3:64, 4) data(4:65, 3) data(4:65, 2) ...
              data(2:63, 4) data(3:64, 3) data(3:64, 2) ...
              data(1:62, 4) data(2:63, 3) data(2:63, 2) ...
	      constant];

coefficients = (regressors\regressand)';

e1 = regressand' - coefficients*regressors';

Aw(1, :) = coefficients(1:9);
Tcw(1) = coefficients(10);

regressand = data(5:66, 3);
regressors = [regressors data(4:65, 4)];

coefficients = (regressors\regressand)';

e2 = regressand' - coefficients*regressors';

Aw(2, :) = coefficients(1:9);
Tcw(2) = coefficients(10);
Tw(2, 1) = - coefficients(11);

regressand = data(5:66, 2);
regressors = [regressors data(5:66, 3)];

coefficients = (regressors\regressand)';

e3 = regressand' - coefficients*regressors';

Aw(3, :) = coefficients(1:9);
Tcw(3) = coefficients(10);
Tw(3, 1:2) = - coefficients(11:12);

% ****************************************************************
% [4] Turn the specification of [3] above into:
%
% S(t) = cw + Bw*S(t-1) + Uw*e(t)
%
% From now on, e(t) will also be reduced to a 3x1 instead of a
% 9x1 vector, with Uw conformable (9x3).
% ****************************************************************

inverseTw = inv(Tw);

cw = inverseTw*Tcw;
Bw = inverseTw*Aw;

Uw = zeros(9, 3);
Uw(1:3, :) = inverseTw(1:3, 1:3);

% ****************************************************************
% [5] Compute and display the long-run values of (R, pi, y).
% ****************************************************************

slr = inv(eye(9) - Bw)*cw;

scale = [4 0 0; 0 4 0; 0 0 1];

%disp(' ')
%disp('Long-run values of (R, pi, y) = ')
%disp(' ')
%disp(slr(1:3)'*scale)

% ****************************************************************
% [6] Display estimated policy rule
% ****************************************************************

%disp(' ')
%disp('Estimated policy rule')
%disp('(mu1, mu2, mu3, phi0, phi1, phi2, theta0, theta1, theta2)')
%disp(' ')
%disp([Bw(1, 1) Bw(1, 4) Bw(1, 7) Bw(1, 2) Bw(1, 5) Bw(1, 8) ...
%	4*Bw(1, 3) 4*Bw(1, 6) 4*Bw(1, 9)]);

% ****************************************************************
% [7] Stack matrix e = [e1; e2; e3] and compute variance-
% covariance matrix of [e1(t), e2(t), e3(t)]. Compute also the
% corresponding stationary variance-covariance matrix of S(t) both
% with the three shocks and eliminating e1.
% ****************************************************************

shocks = [e1; e2; e3];

VCVe1 = shocks*shocks'/62;
VCVe2 = VCVe1;
VCVe2(1, 1) = 0;

VCVs1 = vcv(VCVe1, Bw, Uw);
VCVs2 = vcv(VCVe2, Bw, Uw);

% ****************************************************************
% [8] Construct series for S(t) deviations for t = 7975 to 9500:
% ****************************************************************

state = [data(3:64, 4)'-slr(1)
         data(4:65, 3)'-slr(2)
         data(4:65, 2)'-slr(3)
         data(2:63, 4)'-slr(4)
         data(3:64, 3)'-slr(5)
         data(3:64, 2)'-slr(6)
         data(1:62, 4)'-slr(7)
         data(2:63, 3)'-slr(8)
         data(2:63, 2)'-slr(9)];


% ****************************************************************
% [9] Choice of forcing process
% ****************************************************************

clear option

%disp(' ')
%disp('Choose forcing process from:')
%disp(' ')
%disp('(1) 3 real shocks')
%disp('(2) 2 real shocks')

option = 2; %input('Option # (1 or 2): ');

VCVe = VCVe1;
VCVs = VCVs1;

if option == 2;

	VCVe = VCVe2;
        VCVs = VCVs2;

end

% ****************************************************************
% [10] Construct the vectors (May 97 version) Mm, Nm, Pm, Rm, and 
%     the matrices Cm and Dm
% ****************************************************************

psi = (1-g)/(a*g);

Mm = zeros(9, 1);
Nm = zeros(9, 1);
Pm = zeros(9, 1);
Rm = zeros(9, 1);

Mm(3) = 1;

Nm(1) = -1/s;
Nm(2) =  1/s;

Pm(2) = psi/k;

Rm(1) = -1/(w+s);
Rm(2) = 1/(w+s) + b/k;
Rm(3) = s/(w+s);
Rm(5) = -(1+psi)/k;
Rm(6) = w/(w+s);

Cm = [Mm'-Nm'*Bw*inv(eye(9)-Bw); ...
      Pm'+Rm'*Bw-s/(s+w)*(Nm'*Bw-Mm'*(eye(9)-Bw))] * Bw;

Dm = [Mm'; ...
      Rm'*Bw+s/(s+w)*(Mm'*(eye(9)-Bw)+Nm'*Bw*Bw*inv(eye(9)-Bw))] * Uw;


% ****************************************************************
% [11] Compute the variance-covariance matrix of the shocks
% [delta(t), eps(t), mu(t)]
% using
%	c1m = 1st line of Cm
%	c2m = 2nd line of Cm.
% ****************************************************************

% Construct the processes of the shocks (cf notes calib 11/20/00)
c1m = [1 0]*Cm;
c2m = [0 1]*Cm;

hm = s*c1m-(w+s)*c2m;

% Compute VCV matrix of delta(t), eps(t), mu(t)
% (times 400^2, no real shock e1, no monetary shock)

% nu is proportion of variance of supply shocks 
% due to inefficient shocks

v6 = [s*c1m*(eye(9)-Bw); (1-nu)*hm*(eye(9)-Bw); nu*hm];

%disp('VCV of [delta, eps, mu] = ')
Vshocks = v6*VCVs2*v6'*400^2;

