#!/usr/bin/env python
""" This script provides a visual representation of the agent decisions over time based on the
exact solution.
"""
import warnings
warnings.filterwarnings("ignore")

import pickle as pkl
import numpy as np
import shutil
import sys
import os

np.random.seed(123)
import respy

# module wide variables
PROJECT_DIR = os.path.dirname(os.path.realpath(__file__))
PROJECT_DIR = PROJECT_DIR.replace('/recomputation/exact_solution', '')
sys.path.insert(0, PROJECT_DIR + '/_modules')

from auxiliary_shared import get_choice_probabilities
from auxiliary_shared import process_command_line
from auxiliary_shared import send_notification
from auxiliary_shared import enter_results_dir
from auxiliary_exact import plot_choices
from auxiliary_shared import SPEC_DIR
from auxiliary_shared import cleanup


def run(is_debug, which, num_procs):
    """ Run a single request.
    """
    # Create a fresh directory for each specification and step into the clean directory.
    spec_dir = 'data_' + which
    if os.path.exists(spec_dir):
        shutil.rmtree(spec_dir)
    os.mkdir(spec_dir), os.chdir(spec_dir)

    # Read the baseline specification. Several attributes will be modified later to fit the needs
    #  of the bootstrap exercise
    init_file = SPEC_DIR + '/data_' + which + '.ini'
    respy_obj = respy.RespyCls(init_file)

    # Ensure a speedy execution (if possible).
    respy_obj.unlock()
    respy_obj.set_attr('num_procs', num_procs)
    respy_obj.set_attr('is_parallel', (num_procs > 1))
    respy_obj.lock()

    # The agent decisions are determined by an exact solution to the dynamic programming problem.
    # I store the solved class instance as these exact results are used down the recomputation
    # pipeline.
    respy_obj.attr['is_interpolated'] = False
    respy_obj.attr['num_draws_emax'] = 100000
    respy_obj.attr['is_store'] = True

    if is_debug:
        respy_obj.attr['num_draws_emax'] = 10
        respy_obj.attr['is_interpolated'] = True

    # Print out modified request and simulate.
    respy_obj.write_out()
    respy.simulate(respy_obj)

    # Process choice probabilities, these will be used repeatedly to calculate goodness of fit
    # statistics.
    probs_choice = get_choice_probabilities('data.respy.info', False)

    # Return to specification directories.
    os.chdir('../')

    # Finishing
    return probs_choice


''' Execution of module as script.
'''

if __name__ == '__main__':

    # Process the command line arguments.
    description = 'Assess exact solution.'
    is_debug, num_procs = process_command_line(description)

    # Switch to RSLT_DIR. This separate the results form the source files and eases the updating
    # from the compute servers.
    source_dir = enter_results_dir('exact_solution')

    # Process request using multiple processors.
    cleanup()

    ret = list()
    for which in ['one', 'two', 'three']:
        ret += [run(is_debug, which, num_procs)]

    # Process results for future processing and create visual representation of the agent decision.
    rslts = dict()
    for i, task in enumerate(['one', 'two', 'three']):
        rslts[task] = ret[i]
    pkl.dump(rslts, open('exact.pkl', 'wb'))

    plot_choices(rslts, ['one', 'two', 'three'])
    send_notification('exact')

    os.chdir(source_dir)
