#!/usr/bin/env python
""" This script illustrates the effects of alternative smoothing parameters on estimation
performance.
"""
import warnings
warnings.filterwarnings("ignore")

from statsmodels.tools.eval_measures import rmse
from multiprocessing import Pool
from datetime import datetime
from functools import partial

import pickle as pkl
import numpy as np
import sys
import os

import respy

# module wide variables
PROJECT_DIR = os.path.dirname(os.path.realpath(__file__))
PROJECT_DIR = PROJECT_DIR.replace('/extension/smoothing', '')
sys.path.insert(0, PROJECT_DIR + '/_modules')

from auxiliary_smoothing import aggregate_smoothing_adaptive
from auxiliary_smoothing import aggregate_smoothing_fixed
from auxiliary_shared import get_choice_probabilities
from auxiliary_shared import write_bootstrap_sample
from auxiliary_shared import get_optimization_info
from auxiliary_shared import process_command_line
from auxiliary_smoothing import write_smoothing
from auxiliary_shared import send_notification
from auxiliary_shared import enter_results_dir
from auxiliary_shared import simulate_samples
from auxiliary_shared import estimate_static
from auxiliary_shared import get_seeds
from auxiliary_shared import to_string
from auxiliary_shared import EXACT_DIR
from auxiliary_shared import SPEC_DIR
from auxiliary_shared import mkdir_p
from auxiliary_shared import cleanup


def run_adaptive(is_debug, tau_start, start_vals, maxfun_dynamic, seed):
    """ Run adaptive smoothing exercise.
    """
    dirname = 'seed_' + to_string(seed)
    os.mkdir(dirname); os.chdir(dirname)

    respy_obj = respy.RespyCls(SPEC_DIR + '/data_one.ini')

    respy_obj.unlock()
    respy_obj.set_attr('file_sim', 'data.respy.dat')
    respy_obj.set_attr('file_est', '../data.respy.dat')
    respy_obj.set_attr('num_agents_est', 100)
    respy_obj.set_attr('maxfun', maxfun_dynamic)
    respy_obj.set_attr('num_points_interp', 200)
    respy_obj.set_attr('is_interpolated', True)
    respy_obj.lock()

    respy_obj.update_model_paras(start_vals)
    respy_obj.write_out()

    # I need to specify some details for the simulations to come.
    sim_args = None
    if not is_debug:
        sim_args = dict()
        sim_args['is_interpolated'] = False
        sim_args['num_draws_emax'] = 100000

    # I write out a random subsample of 100 individuals for the estimation.
    write_bootstrap_sample(respy_obj, 'one', seed)

    tau = tau_start

    stat = np.inf
    rslts = dict()

    while True:
        dirname = 'tau_' + to_string(tau)
        os.mkdir(dirname); os.chdir(dirname)

        respy_obj.write_out()

        simulate_samples('start', respy_obj, sim_args)

        open('.structRecomputation.tmp', 'a').close()
        rslt_vals, _ = respy.estimate(respy_obj)

        respy_obj.update_model_paras(rslt_vals)
        simulate_samples('finish', respy_obj, sim_args)

        # Now I update the class instance with the resulting parameters
        tau = float(tau * 0.75)

        respy_obj.unlock()
        respy_obj.set_attr('tau', tau)
        respy_obj.lock()

        # I check for termination in the case there is not an improvement in the RMSE.
        probs_true = get_choice_probabilities(EXACT_DIR + '/data_one/data.respy.info')
        probs_finish = get_choice_probabilities('finish/finish_sample.info')
        probs_start = get_choice_probabilities('start/start_sample.info')

        rslts[tau] = [rmse(probs_true, probs_finish), rmse(probs_true, probs_start)]

        os.chdir('../')

        if rmse(probs_true, probs_finish) >= stat:

            with open('smoothing_adaptive.txt', 'w') as outfile:
                string = '\n' + '{:>15}' * 3 + '\n'
                header = ['Tau', 'Start', 'Finish']
                outfile.write(string.format(*header))
                string = '\n' + '{:15.4f}' * 3
                for tau in sorted(rslts.keys()):
                    line = [tau] + rslts[tau]
                    outfile.write(string.format(*line))

            pkl.dump(rslts, open('rslts.smoothing_adaptive.pkl', 'wb'))
            break
        else:
            stat = rmse(probs_true, probs_finish)

    os.chdir('../')

    return stat


def run_fixed(is_debug, tau, start_vals, maxfun_dynamic, seed):
    """ This function runs an estimation on a single bootstrap sample determined by the seed.
    """
    # Prepare and change to subdirectory.
    dir_ = 'seed_' + to_string(seed)
    mkdir_p(dir_), os.chdir(dir_)

    # I need to specify some details for the simulations to come.
    sim_args = None
    if not is_debug:
        sim_args = dict()
        sim_args['is_interpolated'] = False
        sim_args['num_draws_emax'] = 100000

    respy_obj = respy.RespyCls(SPEC_DIR + '/data_one.ini')

    respy_obj.unlock()
    respy_obj.set_attr('file_est', 'data.respy.dat')
    respy_obj.set_attr('file_sim', 'data.respy.dat')
    respy_obj.set_attr('num_points_interp', 200)
    respy_obj.set_attr('maxfun', maxfun_dynamic)
    respy_obj.set_attr('is_interpolated', True)
    respy_obj.set_attr('num_agents_sim', 1000)
    respy_obj.set_attr('num_agents_est', 100)
    respy_obj.set_attr('tau', float(tau))
    respy_obj.lock()

    respy_obj.update_model_paras(start_vals)
    respy_obj.write_out()

    # I write out a random subsample of 100 individuals for the estimation.
    write_bootstrap_sample(respy_obj, 'one', seed)

    # Estimate the static model and simulate the results.
    simulate_samples('start', respy_obj, sim_args)

    start_time = datetime.now()
    open('.structRecomputation.tmp', 'a').close()
    rslt_vals, _ = respy.estimate(respy_obj)
    finish_time = datetime.now()

    duration = (finish_time - start_time).seconds / 60.00

    respy_obj.update_model_paras(rslt_vals)
    simulate_samples('finish', respy_obj, sim_args)

    probs_true = get_choice_probabilities(EXACT_DIR + '/data_one/data.respy.info')
    probs_finish = get_choice_probabilities('finish/finish_sample.info')
    probs_start = get_choice_probabilities('start/start_sample.info')

    rmse_finish = rmse(probs_finish, probs_true)
    rmse_start = rmse(probs_start, probs_true)

    num_evals, num_steps = get_optimization_info()

    os.chdir('../')

    return seed, rmse_start, rmse_finish, num_evals, num_steps, duration


''' Execution of module as script.
'''

if __name__ == '__main__':

    description = 'Assess effect of smoothing parameter.'
    is_debug, num_procs = process_command_line(description)

    # This sets the number of maximum function evaluations for the static and dynamic estimation
    # runs and the list of interpolation points to assess.
    maxfun_static, maxfun_dynamic, num_boots = 1000, 100000, 40
    parameters = [500, 400, 300, 200, 100]

    # Switch to RSLT_DIR. This separate the results form the source files and eases the updating
    # from the compute servers.
    source_dir = enter_results_dir('smoothing')

    if is_debug:
        maxfun_static, maxfun_dynamic, num_boots = 0, 0, 2
        parameters = [500, 100]

    cleanup()

    # Estimate a static economy with the true values at the start. The resulting estimate serves
    # as the input for the more challenging Monte Carlo exercises.
    start_vals = estimate_static(is_debug, maxfun_static)

    # Now I estimate a variety of dynamic economies with a variety of alternative smoothing setups.
    os.mkdir('dynamic'); os.chdir('dynamic')
    seeds = get_seeds(num_boots)

    os.mkdir('fixed'); os.chdir('fixed')

    for tau in parameters:

        dirname = 'tau_' + to_string(tau)
        os.mkdir(dirname), os.chdir(dirname)

        process_tasks = partial(run_fixed, is_debug, tau, start_vals, maxfun_dynamic)
        rslts = Pool(num_procs).map(process_tasks, seeds)

        write_smoothing(rslts)
        os.chdir('../')

    aggregate_smoothing_fixed(parameters)

    os.chdir('../')

    os.mkdir('adaptive'); os.chdir('adaptive')

    tau_start = 500.00
    process_tasks = partial(run_adaptive, is_debug, tau_start, start_vals,  maxfun_dynamic)
    if num_procs == 1:
        rslts = [process_tasks(seeds[0])]
    else:
        rslts = Pool(num_procs).map(process_tasks, seeds)

    aggregate_smoothing_adaptive(rslts)

    send_notification('smoothing')

    os.chdir(source_dir)
