#!/usr/bin/env python
""" This script allows to analyze the time to solve the model.
"""
import warnings
warnings.filterwarnings("ignore")

from multiprocessing import Pool
from functools import partial
from datetime import datetime
import pickle as pkl
import numpy as np
import shutil
import shlex
import copy
import sys
import os

np.random.seed(123)
import respy

# module wide variables
PROJECT_DIR = os.path.dirname(os.path.realpath(__file__))
PROJECT_DIR = PROJECT_DIR.replace('/extension/performance', '')
sys.path.insert(0, PROJECT_DIR + '/_modules')

from auxiliary_performance import plot_performance_model
from auxiliary_performance import write_performance
from auxiliary_shared import process_command_line
from auxiliary_shared import enter_results_dir
from auxiliary_shared import send_notification
from auxiliary_shared import to_string
from auxiliary_shared import SPEC_DIR
from auxiliary_shared import cleanup


def run(respy_obj, period):
    """ Run a single request.
    """
    # Encapsulation of changes to baseline specification.
    respy_period = copy.deepcopy(respy_obj)

    respy_period.unlock()
    respy_period.set_attr('num_periods', period)
    respy_period.lock()

    if os.path.exists(to_string(period)):
        shutil.rmtree(to_string(period))

    os.mkdir(to_string(period))
    os.chdir(to_string(period))

    respy_period.write_out()
    respy.simulate(respy_period)

    # Get results from the special output file that is integrated in the RESPY package just
    # for the purpose of referee's request.
    with open('.structRecomputation.performance') as in_file:
        rslt = []
        for line in in_file.readlines():
            list_ = shlex.split(line)
            str_ = list_[1] + ' ' + list_[2]
            rslt += [datetime.strptime(str_, "%d/%m/%Y %H:%M:%S")]

    rslt += [np.sum(respy_period.get_attr('states_number_period'))]

    os.chdir('../')

    return rslt

''' Execution of module as script.
'''

if __name__ == '__main__':

    description = 'Assess package performance.'
    is_debug, num_procs = process_command_line(description)

    # Switch to RSLT_DIR. This separate the results form the source files and eases the updating
    # from the compute servers.
    source_dir = enter_results_dir('performance')

    periods = [1] + list(range(5, 81, 5))

    if is_debug:
        periods = [1, 2]

    cleanup()

    # Initialize baseline specification and adjust it for the performance exercise.
    respy_obj = respy.RespyCls(SPEC_DIR + '/data_one.ini')

    respy_obj.unlock()
    respy_obj.set_attr('is_interpolated', False)
    respy_obj.set_attr('num_draws_emax', 100000)
    respy_obj.set_attr('is_parallel', False)

    if is_debug:
        respy_obj.set_attr('num_draws_emax', 10)
        respy_obj.set_attr('num_periods', 3)

    respy_obj.lock()

    # Process request using multiple processors.
    process_tasks = partial(run, respy_obj)
    ret = Pool(num_procs).map(process_tasks, periods)

    # Restructure results for further processing.
    rslts = dict()
    for i, period in enumerate(periods):
        rslts[period] = ret[i]

    pkl.dump(rslts, open('performance.pkl', 'wb'))

    write_performance(rslts)

    plot_performance_model(rslts)

    send_notification('performance')

    os.chdir(source_dir)
