#!/usr/bin/env python
""" This script illustrates the criterions in the criterion function.
"""
import warnings
warnings.filterwarnings("ignore")

import pickle as pkl
import numpy as np
import shutil
import sys
import os

np.random.seed(123)
import respy

# module wide variables
PROJECT_DIR = os.path.dirname(os.path.realpath(__file__))
PROJECT_DIR = PROJECT_DIR.replace('/extension/criterions', '')
sys.path.insert(0, PROJECT_DIR + '/_modules')

from auxiliary_shared import process_command_line
from auxiliary_criterions import write_criterions
from auxiliary_criterions import plot_criterions
from auxiliary_shared import send_notification
from auxiliary_shared import enter_results_dir
from auxiliary_shared import to_string
from auxiliary_shared import EXACT_DIR
from auxiliary_shared import SPEC_DIR
from auxiliary_shared import mkdir_p
from auxiliary_shared import cleanup


def run(is_debug, task, num_procs):
    """ Run a single request.
    """
    # Distribute the tasks
    solution, subsidy = task

    respy_obj = respy.RespyCls(SPEC_DIR + '/data_one.ini')

    # Ensure a speedy execution (if possible).
    respy_obj.unlock()
    respy_obj.set_attr('num_procs', num_procs)
    respy_obj.set_attr('is_parallel', (num_procs > 1))
    respy_obj.lock()

    model_paras = respy_obj.get_attr('model_paras')
    tuition = model_paras['coeffs_edu'][1]

    respy_obj.unlock()
    if solution == 'exact':
        respy_obj.set_attr('is_interpolated', False)
        respy_obj.set_attr('num_draws_emax', 100000)
    else:
        respy_obj.set_attr('is_interpolated', True)
    respy_obj.set_attr('maxfun', 0)
    respy_obj.lock()

    if is_debug:
        respy_obj.unlock()
        respy_obj.set_attr('num_draws_emax', 100)
        respy_obj.lock()

    # Step into a subdirectory for each single point.
    spec_dir = solution + '/subsidy_' + to_string(int(subsidy))
    mkdir_p(spec_dir), os.chdir(spec_dir)

    # Modify the baseline initialization file.
    model_paras['coeffs_edu'][1] = tuition + subsidy
    respy_obj.attr['model_paras'] = model_paras
    respy_obj.write_out()
    shutil.copy(EXACT_DIR + '/data_one/data.respy.dat', 'data.respy.dat')

    # Signal to the RESPY package that the limited interpolation model is used.
    open('.structRecomputation.tmp', 'a').close()

    # Collect the results for the specified level of the subsidy and write to disk.
    _, crit_val = respy.estimate(respy_obj)

    # Simulate the economy to check that the effects of the subsidy are as expected.
    if not is_debug:
        respy.simulate(respy_obj)

    os.chdir('../../')

    # Finishing
    return solution, subsidy, crit_val

''' Execution of module as script.
'''

if __name__ == '__main__':

    description = 'Assess implications of criterions in criterion function.'
    is_debug, num_procs = process_command_line(description)

    # Switch to RSLT_DIR. This separate the results form the source files and eases the updating
    # from the compute servers.
    source_dir = enter_results_dir('criterions')

    subsidies = np.linspace(-1000, 1000, 21)

    # Debugging setup
    if is_debug:
        subsidies = np.linspace(-1000, 1000, 3)

    # Create task grid
    solutions = ['approximate', 'exact']
    tasks = []
    for solution in solutions:
        for subsidy in subsidies:
            tasks += [(solution, subsidy)]

    # Process request using multiple processors.
    cleanup()

    ret = list()
    for task in tasks:
        ret += [run(is_debug, task, num_procs)]

    # Restructure return arguments for better interpretability and further processing. Write
    # information out to disk.
    rslts = dict()
    for solution in solutions:
        rslts[solution] = []
        for subsidy in subsidies:
            for candidate in ret:
                if candidate[:2] != (solution, subsidy):
                    continue
                rslts[solution] += [candidate[2]]
    pkl.dump(rslts, open('criterions.pkl', 'wb'))

    plot_criterions(rslts, subsidies)
    write_criterions(rslts, subsidies)

    send_notification('criterions')

    os.chdir(source_dir)
