from string import Formatter

import pandas as pd
import numpy as np
import subprocess
import argparse
import shutil
import socket
import errno
import shlex
import copy
import os

from clsMail import MailCls

from respy.python.simulate.simulate_auxiliary import write_out
import respy

PROJECT_DIR = os.path.dirname(os.path.realpath(__file__))
PROJECT_DIR = PROJECT_DIR.replace('/_modules', '')

SPEC_DIR = PROJECT_DIR + '/parametrizations'
RSLT_DIR = PROJECT_DIR + '/results'
EXACT_DIR = RSLT_DIR + '/exact_solution'


def estimate_static(is_debug, maxfun_static):
    """ Estimate a static economy as a baseline.
    """
    # First perform a static estimation on the exact baseline data
    os.mkdir('static'), os.chdir('static')
    respy_obj = respy.RespyCls(SPEC_DIR + '/data_one.ini')

    respy_obj.unlock()
    respy_obj.set_attr('file_est', EXACT_DIR + '/data_one/data.respy.dat')
    respy_obj.set_attr('maxfun', maxfun_static)
    respy_obj.set_attr('num_agents_sim', 100)
    respy_obj.set_attr('num_agents_est', 100)
    respy_obj.set_attr('delta', 0.0)
    respy_obj.lock()

    # Clarify some additional requests for the simulations fo the start and finish samples.
    sim_args = None
    if not is_debug:
        sim_args = dict()
        sim_args['is_interpolated'] = False
        sim_args['num_draws_emax'] = 100000

    respy_obj.write_out()

    # Simulate a sample with starting values and then start the estimation.
    simulate_samples('start', respy_obj, sim_args)

    start_vals, _ = respy.estimate(respy_obj)

    # Simulate a sample with the estimated samples. Estimate the static model and simulate the
    # results.
    respy_obj.update_model_paras(start_vals)
    simulate_samples('finish', respy_obj, sim_args)

    os.chdir('../')

    # Finishing
    return start_vals


def simulate_samples(dirname, respy_obj, kwargs=None):
    """ Simulate a sample in new subdirectory for the purposes of visual
    inspection,
    """

    respy_copy = copy.deepcopy(respy_obj)

    if os.path.exists(dirname):
        shutil.rmtree(dirname)

    # Update parameters for simulation.
    if kwargs is not None:
        respy_copy.unlock()
        for key_ in kwargs.keys():
            respy_copy.set_attr(key_, kwargs[key_])
        respy_copy.lock()

    os.mkdir(dirname), os.chdir(dirname)

    respy_copy.attr['file_sim'] = dirname + '_sample.dat'
    respy_copy.write_out()
    respy.simulate(respy_copy)


    del respy_copy

    os.chdir('../')


def enter_results_dir(which):
    """ This function creates the requested results directory and switches the working directory.
    """
    # Ensure that directory structure exists
    if not os.path.exists(RSLT_DIR):
        os.mkdir(RSLT_DIR)

    dirname = RSLT_DIR + '/' + which

    if os.path.exists(dirname):
        shutil.rmtree(dirname)

    os.mkdir(dirname)

    source_dir = os.getcwd()
    os.chdir(dirname)

    return source_dir


def write_bootstrap_sample(respy_obj, which, seed):
    """ Write out a bootstrap sample for estimation.
    """
    fname = EXACT_DIR + '/data_' + which + '/data.respy.dat'
    df = pd.read_csv(fname, delim_whitespace=True,
        header=-1, na_values='.', dtype={0: np.int, 1: np.int, 2: np.int,
        3: np.float, 4: np.int, 5: np.int, 6: np.int, 7: np.int}, nrows=40000)
    df.set_index([0, 1], drop=False, inplace=True)

    np.random.seed(seed)
    index_subsample = np.random.choice(range(1000), 100, replace=False).tolist()
    df_sample = df.loc[index_subsample]
    write_out(respy_obj, df_sample)


def get_seeds(num_draws, seed=789):
    """ Get a list of seeds to create the Monte Carlo datasets.
    """
    np.random.seed(seed)
    seeds = np.random.choice(range(1000), size=num_draws, replace=False).tolist()

    return seeds


def process_command_line(description):
    """ Distribute command line arguments.
    """
    parser = argparse.ArgumentParser(description=description)

    parser.add_argument('--debug', action='store_true', dest='is_debug',
        help='use debug specification')

    parser.add_argument('--procs', action='store', type=int, dest='num_procs',
        default=1, help='use multiple processors')

    # Process command line arguments
    args = parser.parse_args()

    # Extract arguments
    num_procs = args.num_procs
    is_debug = args.is_debug

    # Check arguments
    assert (is_debug in [True, False])
    assert (isinstance(num_procs, int))
    assert (num_procs > 0)

    # Finishing
    return is_debug, num_procs


def cleanup():

    # Cleanup
    if socket.gethostname() != 'vagrant':
        subprocess.call(['git', 'clean', '-f', '-d'])


def get_optimization_info():
    """ Get some additional information about the optimization.
    """
    with open('est.respy.info') as in_file:
        for line in in_file.readlines():
            # Split line
            list_ = shlex.split(line)
            # Skip empty lines
            if not list_:
                continue
            # Check for applicability
            if len(list_) < 4:
                continue
            if list_[2] == 'Steps':
                num_steps = int(list_[3])
            if list_[2] == 'Evaluations':
                num_evals = int(list_[3])

    # Finishing
    return num_evals, num_steps


def send_notification(which):
    """ Finishing up a run of the testing battery.
    """
    hostname = socket.gethostname()

    subject = ' STRUCT_RECOMPUTATION: '
    if which == 'exact':
        subject += 'Exact Solutions'
        message = ' Exact solutions'
    elif which == 'correct':
        subject += 'Correct Choices'
        message = ' Analysis of correct choices'
    elif which == 'monte':
        subject += 'Monte Carlo Investigation'
        message = ' Results from Monte Carlo exercise'
    elif which == 'criterions':
        subject += 'Criterions Investigation'
        message = ' Results from the criterions investigation'
    elif which == 'schemes':
        subject += 'Schemes Investigation'
        message = ' Results from the schemes investigation'
    elif which == 'smoothing':
        subject += 'Smoothing Investigation'
        message = ' Results from the smoothing investigation'
    elif which == 'performance':
        subject += 'Performance Investigation'
        message = ' Results from the performance investigation'

    else:
        raise AssertionError

    message += ' are available on @' + hostname + '.'

    mail_obj = MailCls()
    mail_obj.set_attr('subject', subject)
    mail_obj.set_attr('message', message)

    # This allows to use the same scripts for the recomputation material.
    try:
        mail_obj.lock()
        mail_obj.send()
    except:
        pass


def to_string(input):
    """ This function transfers the input to a string.
    """

    if isinstance(input, int):
        return '{0:05d}'.format(input)
    elif isinstance(input, float):
        return '{:.5f}'.format(input)
    else:
        raise NotImplementedError


def mkdir_p(path):
    """ Make directory, including parent directory (if required).
    """
    try:
        os.makedirs(path)
    except OSError as exc:
        if exc.errno == errno.EEXIST and os.path.isdir(path):
            pass
        else:
            raise


def get_choice_probabilities(fname, is_flatten=True):
    """ Get the choice probabilities.
    """
    # Initialize container.
    stats = np.tile(np.nan, (0, 4))

    with open(fname) as in_file:

        for line in in_file.readlines():

            # Split line
            list_ = shlex.split(line)

            # Skip empty lines
            if not list_:
                continue

            # If OUTCOMES is reached, then we are done for good.
            if list_[0] == 'Outcomes':
                break

            # Any lines that do not have an integer as their first element
            # are not of interest.
            try:
                int(list_[0])
            except ValueError:
                continue

            # All lines that make it down here are relevant.
            stats = np.vstack((stats, [float(x) for x in list_[1:]]))

    # Return all statistics as a flattened array.
    if is_flatten:
        stats = stats.flatten()

    # Finishing
    return stats


def strfdelta(tdelta, fmt):
    """ Get a string from a timedelta.
    """
    f, d = Formatter(), {}
    l = {'D': 86400, 'H': 3600, 'M': 60, 'S': 1}
    k = list(map(lambda x: x[1], list(f.parse(fmt))))
    rem = int(tdelta.total_seconds())
    for i in ('D', 'H', 'M', 'S'):
        if i in k and i in l.keys():
            d[i], rem = divmod(rem, l[i])
    return f.format(fmt, **d)


def formatting_duration(start_time, finish_time):
    """ This function formats the time objects to a pretty string that indicates the duration of
    the estimation.
    """
    duration_str = strfdelta(finish_time - start_time, "{H:02}:{M:02}:{S:02}")
    return duration_str