""" This module contains some auxiliary functions that are used to investigate the performance of
the package.
"""
import pandas as pd
import matplotlib

matplotlib.use('Agg')
import matplotlib.pyplot as plt

from auxiliary_shared import formatting_duration


def write_performance(rslts):
    """ Write out the results.
    """
    with open('performance.txt', 'w') as log_file:
        string = '{:>15}{:>15}{:>25}\n\n'
        line = ['Periods', 'Duration', 'States']
        log_file.write(string.format(*line))

        for period in sorted(rslts.keys()):
            line = (period, formatting_duration(*rslts[period][:2]), rslts[period][2])
            string = '{:15d}{:>15}{:25d}\n'
            log_file.write(string.format(*line))


def plot_performance_model(rslt):
    """ This graphs plots the development of computing power to solve the KW model.
    """
    performance = pd.DataFrame.from_dict(rslt, orient='index').sort_index()

    # Rename columns, index
    performance.reset_index(inplace=True)
    dict_columns = {'index': 'Periods', 0: 'Start', 1: 'End', 2: 'States'}
    performance = performance.rename(columns=dict_columns)

    # Create duration
    performance['Duration'] = performance.End - performance.Start

    # Convert the timedeltas to the prefered time unit: [s] for seconds, [m] for minutes
    performance.Duration = performance.Duration.astype('timedelta64[m]')

    # Performance graph
    ax1 = plt.figure(figsize=(12, 8)).add_subplot(111)

    # Add line for duration
    ax1.plot(performance.Periods, performance.Duration, label='Model Solution', c='red',
        linewidth=5, linestyle='--')

    # Add line for duration of KW1994
    ax1.plot((40, 40), (0, 50), label='KW1994 Solution', color='grey', linewidth=5, linestyle='-.')
    ax1.plot((-5, 40), (50, 50), color='grey', linewidth=5, linestyle='-.')
    ax1.plot(40, 50, 'o', color='black', linewidth=40)

    # Add a barplot chart for states which uses the same x axis
    ax2 = ax1.twinx()
    ax2.bar(performance.Periods, performance.States, 2, label='States', color='blue', alpha=0.8)

    # Remove spaces around 0
    ax1.set_xlim(-5, 85)
    ax1.set_ylim(0, 80)

    # Set labelsize of ticks
    tick_label_size = 16
    ax1.tick_params(labelsize=tick_label_size, direction='out', axis='both', top='off', right='off')
    ax2.tick_params(labelsize=tick_label_size, direction='out', axis='both', top='off', left='off')

    # Set size of axis labels and labels
    axis_label_size = 20
    ax1.set_xlabel('Periods', size=axis_label_size)
    ax1.set_ylabel('Duration (minutes)', size=axis_label_size)
    ax2.set_ylabel('States', size=axis_label_size)

    # Formatting with comma for thousands.
    func = matplotlib.ticker.FuncFormatter(lambda x, p: format(int(x), ','))
    ax2.get_yaxis().set_major_formatter(func)

    # Set legend
    handles1, labels1 = ax1.get_legend_handles_labels()
    handles2, labels2 = ax2.get_legend_handles_labels()
    plt.legend(handles1 + handles2, labels1 + labels2, loc='upper center', bbox_to_anchor=(0.5,
        -0.10), fancybox=False, frameon=False, shadow=False, ncol=4, fontsize=20)

    plt.savefig('graphs_performance_model.png', bbox_inches='tight', format='png')
