import numpy as np
import matplotlib

matplotlib.use('Agg')

import matplotlib.pyplot as plt


def write_criterions(rslts, subsidies):
    """ Write out the value of the criterion functions for varying subsidies.
    """
    with open('criterions.txt', 'w') as f:

        fmt_ = ' {:>15}' + ' {:>15}' * 2 + '\n\n'
        line = fmt_.format(*['Subsidy', 'Exact', 'Approximate'])
        f.write(line)

        fmt_ = ' {:>15}' + ' {:15.4f}' * 2 + '\n'
        for i, subsidy in enumerate(subsidies):
            list_ = [subsidy]
            list_ += [rslts['exact'][i]]
            list_ += [rslts['approximate'][i]]
            line = fmt_.format(*list_)
            f.write(line)


def plot_criterions(rslts, subsidies):
    """ Plot the criterions in the criterion function.
    """
    ax = plt.figure(figsize=(12, 8)).add_subplot(111)
    exact = rslts['exact']

    idx = 10
    if len(exact) < 21:
        idx = 1

    y_values = np.array(exact) - exact[idx]
    plt.plot(subsidies, y_values, '--', linewidth=5, color='blue',
        label='Exact')

    approximate = rslts['approximate']
    y_values = np.array(approximate) - approximate[idx]
    plt.plot(subsidies, y_values, '-.', linewidth=5, color='red',
        label='Approximate')

    plt.axvline(x=0.00, ymin=0.0, ymax=0.1, linewidth=1, color='grey',
        linestyle='--')

    plt.axhline(y=0.00, xmin=0.00, xmax=0.50, linewidth=1, color='grey',
        linestyle='--')

    plt.plot(0, 0, 'o', color='black', linewidth=40)

    # X Label
    ax.set_xlabel('Perturbation of Tuition Cost', fontsize=16)
    ax.set_xlim([-1100, 1100])

    # Y Label
    ax.set_ylabel('Value of Criterion Function', fontsize=16)
    ax.yaxis.get_major_ticks()[0].set_visible(False)
    ax.set_ylim([-0.005, 0.045])

    # Both Axes
    ax.tick_params(labelsize=16, direction='out', axis='both', top='off',
        right='off')

    # Formatting with comma for thousands.
    func = matplotlib.ticker.FuncFormatter(lambda x, p: format(int(x), ','))
    ax.get_xaxis().set_major_formatter(func)

    # Set up te legend nicely formatted.
    plt.legend(loc='upper center', bbox_to_anchor=(0.5, -0.10), fancybox=False,
        frameon=False, shadow=False, ncol=4, fontsize=20)

    # Store figure as *.png
    plt.savefig('graphs_criterions.png', bbox_inches='tight', format='png')

