/*
 * This program calculates p-values for the sharpened Diks-Panchenko statistic (sharpened assuming the Hall and Minotte (2002) method) for the 4-variate setting
 *
 * Diks, C. and Wolski, M. (JAE, 2015)
 * Nonlinear Granger Causality - Guidelines for Multivariate Analysis
 *
 * Copyright 2015, Marcin Wolski and Cees Diks
 * Software distributed under the GNU Public License 3.0. 
 * For details, see `LICENSE.txt'
 *
 * Cees Diks and Marcin Wolski -- May 2015
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>

#define max(a,b)	a>b?a:b

double C_EPS=2.4;

double rate_EPS=-0.153846; // -2/13

double C_eps2=4; 

double rate_eps2=-0.125; // -1/8

double EPS; // Calculated in main()

double eps2; // Calculated in main()

double C2;  // Calculated in main()

int Ndat=15000;
int Mmax=1;     /* maximum emb. dim. */
int n, K;
double T2,Q,**A,C[4],*h,*ohm,*cov;
double p_HJ, p_T2, HJ_TVAL, T2_TVAL;

/* sharpening function - bias reduction to order eps^4 */
double phihat(double *x, double *y, double *z, double *q, double *u, char *what, char *where, int point, int N, double epsilon, double c)
{
	int i;
	double nominator, denominator, new, w;
	
	nominator=denominator=new=0.0;

    for (i=0;i!=N;i++)
	{
	   w = exp((-(x[i]-x[point])*(x[i]-x[point])-(y[i]-y[point])*(y[i]-y[point])-(z[i]-z[point])*(z[i]-z[point])-(q[i]-q[point])*(q[i]-q[point])-(u[i]-u[point])*(u[i]-u[point]))/(2*epsilon*epsilon));

	   if(strcmp(what,"x")==0) 
           {		
	     nominator += (x[i]-x[point])*w;
	     denominator += w;
           }                

	   if(strcmp(what,"y")==0) 
           {		
	     nominator += (y[i]-y[point])*w;
	     denominator += w;
           }                

	   if(strcmp(what,"z")==0) 
           {		
	     nominator += (z[i]-z[point])*w;
	     denominator += w;
           }                

	   if(strcmp(what,"q")==0) 
           {		
	     nominator += (q[i]-q[point])*w;
	     denominator += w;
           }  
       if(strcmp(what,"u")==0) 
           {		
	     nominator += (u[i]-u[point])*w;
	     denominator += w;
           }              
    }		

	if (strcmp(what,"x")==0) 			
      	  new=x[point]+c*nominator/denominator;

	if (strcmp(what,"y")==0) 			
      	  new=y[point]+c*nominator/denominator;

	if (strcmp(what,"z")==0) 			
      	  new=z[point]+c*nominator/denominator;

	if (strcmp(what,"q")==0) 			
      	  new=q[point]+c*nominator/denominator;
      	  
    if (strcmp(what,"u")==0) 			
      	  new=u[point]+c*nominator/denominator;
		
	return(new);
}

void redun(double *x, double *y, double *z, double *q, double *u, int N, double epsilon)
{

  int i, j, s;
  double IYij, IXYij, IYZij, IXYZij;
  double dis2x, dis2y, dis2z, dis2q, dis2u, tCy=0, tCxy=0, tCyz=0, tCxyz=0, mu, C_DS;

  double *shx_xyq, *shx_xyzq, *shz_yzq, *shz_xyzq, *shy_yq, *shy_xyq, *shy_yzq, *shy_xyzq, *shq_yq, *shq_xyq, *shq_yzq, *shq_xyzq, *shu_yq, *shu_xyq, *shu_yzq, *shu_xyzq;
  double e, *Cyq, *Cxyq, *Cyzq, *Cxyzq, disx_xyq, disx_xyzq, disz_yzq, disz_xyzq, disy_yq, disy_xyq, disy_yzq, disy_xyzq, disq_yq, disq_xyq, disq_yzq, disq_xyzq;

  mu = 1.0;

  C_DS = C2; //Sharpening constant

  Cyq = (double *) malloc(N*sizeof(double));
  Cxyq = (double *) malloc(N*sizeof(double));
  Cyzq = (double *) malloc(N*sizeof(double));
  Cxyzq = (double *) malloc(N*sizeof(double));

  shx_xyq = (double *) malloc(N*sizeof(double));
  shx_xyzq = (double *) malloc(N*sizeof(double));
  shz_yzq = (double *) malloc(N*sizeof(double));
  shz_xyzq = (double *) malloc(N*sizeof(double));
  shy_yq = (double *) malloc(N*sizeof(double));
  shy_xyq = (double *) malloc(N*sizeof(double));
  shy_yzq = (double *) malloc(N*sizeof(double));
  shy_xyzq = (double *) malloc(N*sizeof(double));
  shq_yq = (double *) malloc(N*sizeof(double));
  shq_xyq = (double *) malloc(N*sizeof(double));
  shq_yzq = (double *) malloc(N*sizeof(double));
  shq_xyzq = (double *) malloc(N*sizeof(double));
  shu_yq = (double *) malloc(N*sizeof(double));
  shu_xyq = (double *) malloc(N*sizeof(double));
  shu_yzq = (double *) malloc(N*sizeof(double));
  shu_xyzq = (double *) malloc(N*sizeof(double));
  
  for (i=0;i!=N;i++)
  {
    h[i] = Cyq[i] = Cxyq[i] = Cyzq[i] = Cxyzq[i] = 0.0;

    shx_xyq[i] = shx_xyzq[i] = shz_yzq[i] = shz_xyzq[i] = shy_yq[i] = shy_xyq[i] = 0.0;
    shy_yzq[i] = shy_xyzq[i] = shq_yq[i] = shq_xyq[i] = shq_yzq[i] = shq_xyzq[i] = 0.0;
    shu_yq[i] = shu_xyq[i] = shu_yzq[i] = shu_xyzq[i];
    
    disx_xyq = disx_xyzq = disz_yzq = disz_xyzq = disy_yq = disy_xyq = 0.0;
    disy_yzq = disy_xyzq = disq_yq = disq_xyq = disq_yzq = disq_xyzq = 0.0; 
  }

  T2=Q=0.0;
  n = N; // was N - mmax

               
  //calculate the sharpened values
  //eps2 = C2*pow(n,DS_rate); //convergence at n^(-1/4)

  for (i=0;i!=n;i++)
  {
		//clean the variables
		shx_xyq[i] = shx_xyzq[i] = shz_yzq[i] = shz_xyzq[i] = shy_yq[i] = shy_xyq[i] = shy_yzq[i] = shy_xyzq[i] = shq_yq[i] = shq_xyq[i] = shq_yzq[i] = shq_xyzq[i] = 0.0;

		/* vector YQ */
		shy_yq[i] = phihat(x,y,z,q,u,"y","yq",i,n,eps2,C_DS);
		shq_yq[i] = phihat(x,y,z,q,u,"q","yq",i,n,eps2,C_DS);
		shu_yq[i] = phihat(x,y,z,q,u,"u","yq",i,n,eps2,C_DS);
		
		/* vector XYQ */
		shx_xyq[i] = phihat(x,y,z,q,u,"x","xyq",i,n,eps2,C_DS);
		shy_xyq[i] = phihat(x,y,z,q,u,"y","xyq",i,n,eps2,C_DS);
		shq_xyq[i] = phihat(x,y,z,q,u,"q","xyq",i,n,eps2,C_DS);
		shu_xyq[i] = phihat(x,y,z,q,u,"u","xyq",i,n,eps2,C_DS);
		
		/* vector YZQ */
		shy_yzq[i] = phihat(x,y,z,q,u,"y","yzq",i,n,eps2,C_DS);
		shz_yzq[i] = phihat(x,y,z,q,u,"z","yzq",i,n,eps2,C_DS);
		shq_yzq[i] = phihat(x,y,z,q,u,"q","yzq",i,n,eps2,C_DS);
		shu_yzq[i] = phihat(x,y,z,q,u,"u","yzq",i,n,eps2,C_DS);
		
		/* vector XYZQ */
		shx_xyzq[i] = phihat(x,y,z,q,u,"x","xyzq",i,n,eps2,C_DS);
		shy_xyzq[i] = phihat(x,y,z,q,u,"y","xyzq",i,n,eps2,C_DS);
		shz_xyzq[i] = phihat(x,y,z,q,u,"z","xyzq",i,n,eps2,C_DS);
		shq_xyzq[i] = phihat(x,y,z,q,u,"q","xyzq",i,n,eps2,C_DS);
		shu_xyzq[i] = phihat(x,y,z,q,u,"u","xyzq",i,n,eps2,C_DS);
  }

  
  for (i=0;i!=n;i++)
  {
    Cyq[i]=Cxyq[i]=Cyzq[i]=Cxyzq[i]=0.0;
    for (j=0;j!=n;j++)
    if (j!=i)

    {
      dis2y = (y[i]-shy_yq[j])*(y[i]-shy_yq[j]);
      dis2q = (q[i]-shq_yq[j])*(q[i]-shq_yq[j]);
      dis2u = (u[i]-shu_yq[j])*(u[i]-shu_yq[j]);
      
      Cyq[i] += exp(-(dis2y+dis2q+dis2u)/(2.0*epsilon*epsilon));
      A[3][i] += exp(-(dis2y+dis2q+dis2u)/(2.0*epsilon*epsilon));

      dis2x = (x[i]-shx_xyq[j])*(x[i]-shx_xyq[j]);
      dis2y = (y[i]-shy_xyq[j])*(y[i]-shy_xyq[j]);
      dis2q = (q[i]-shq_xyq[j])*(q[i]-shq_xyq[j]);
      dis2u = (u[i]-shu_xyq[j])*(u[i]-shu_xyq[j]);
      
      Cxyq[i] += exp(-(dis2x+dis2y+dis2q+dis2u)/(2.0*epsilon*epsilon));
      A[1][i] += exp(-(dis2x+dis2y+dis2q+dis2u)/(2.0*epsilon*epsilon));

      dis2y = (y[i]-shy_yzq[j])*(y[i]-shy_yzq[j]);
      dis2z = (z[i]-shz_yzq[j])*(z[i]-shz_yzq[j]);
      dis2q = (q[i]-shq_yzq[j])*(q[i]-shq_yzq[j]);
	  dis2u = (u[i]-shu_yzq[j])*(u[i]-shu_yzq[j]);
	  
      Cyzq[i] += exp(-(dis2y+dis2z+dis2q+dis2u)/(2.0*epsilon*epsilon));
      A[2][i] += exp(-(dis2y+dis2z+dis2q+dis2u)/(2.0*epsilon*epsilon));

      dis2x = (x[i]-shx_xyzq[j])*(x[i]-shx_xyzq[j]);
      dis2y = (y[i]-shy_xyzq[j])*(y[i]-shy_xyzq[j]);
      dis2z = (z[i]-shz_xyzq[j])*(z[i]-shz_xyzq[j]);
      dis2q = (q[i]-shq_xyzq[j])*(q[i]-shq_xyzq[j]);
	  dis2u = (u[i]-shu_xyzq[j])*(u[i]-shu_xyzq[j]);
	  
      Cxyzq[i] += exp(-(dis2x+dis2y+dis2z+dis2q+dis2u)/(2.0*epsilon*epsilon));
      A[0][i] += exp(-(dis2x+dis2y+dis2z+dis2q+dis2u)/(2.0*epsilon*epsilon));

    }   // end loop over j

    Cyq[i] /= (double)(n-1);
    Cxyq[i] /= (double)(n-1);
    Cyzq[i] /= (double)(n-1);
    Cxyzq[i] /= (double)(n-1);

    h[i] += 2.0/mu*(Cxyzq[i]*Cyq[i] - Cxyq[i]*Cyzq[i])/6.0;
  }

  for (i=0;i!=n;i++)
  {
    for (j=0;j!=n;j++)
    if (j!=i)
    {

      IYij = IXYij = IYZij = IXYZij = 0.0;

      dis2y = (y[i]-shy_yq[j])*(y[i]-shy_yq[j]);
      dis2q = (q[i]-shq_yq[j])*(q[i]-shq_yq[j]);
      dis2u = (u[i]-shu_yq[j])*(u[i]-shu_yq[j]);
      
      IYij= exp(-(dis2y+dis2q+dis2u)/epsilon/epsilon/2.0);

      dis2x = (x[i]-shx_xyq[j])*(x[i]-shx_xyq[j]);
      dis2y = (y[i]-shy_xyq[j])*(y[i]-shy_xyq[j]);
      dis2q = (q[i]-shq_xyq[j])*(q[i]-shq_xyq[j]);
	  dis2u = (u[i]-shu_xyq[j])*(u[i]-shu_xyq[j]);

      IXYij = exp(-(dis2x+dis2y+dis2q+dis2u)/epsilon/epsilon/2.0);

      dis2y = (y[i]-shy_yzq[j])*(y[i]-shy_yzq[j]);
      dis2z = (z[i]-shz_yzq[j])*(z[i]-shz_yzq[j]);
      dis2q = (q[i]-shq_yzq[j])*(q[i]-shq_yzq[j]);
      dis2u = (u[i]-shu_yzq[j])*(u[i]-shu_yzq[j]);
      
      IYZij = exp(-(dis2y+dis2z+dis2q+dis2u)/epsilon/epsilon/2.0);

      dis2x = (x[i]-shx_xyzq[j])*(x[i]-shx_xyzq[j]);
      dis2y = (y[i]-shy_xyzq[j])*(y[i]-shy_xyzq[j]);
      dis2z = (z[i]-shz_xyzq[j])*(z[i]-shz_xyzq[j]);
      dis2q = (q[i]-shq_xyzq[j])*(q[i]-shq_xyzq[j]);
	  dis2u = (u[i]-shu_xyzq[j])*(u[i]-shu_xyzq[j]);
	  
      IXYZij = exp(-(dis2x+dis2y+dis2z+dis2q+dis2u)/epsilon/epsilon/2.0);
      //h[j] += 2.0*(Cy[i]*IXYZij - Cyz[i]*IXYij)/(double)(6*n);
      //h[j] += 2.0*(Cxyz[i]*IYij - Cxy[i]*IYZij)/(double)(6*n);

      //h[i] += 2.0/mu*(Cxyzq[j]*IYij + IXYZij*Cyq[j] - Cxyq[j]*IYZij - IXYij*Cyzq[j])/(double)(6*n);

      h[j] += 2.0/mu*(Cxyzq[i]*IYij + IXYZij*Cyq[i] - Cxyq[i]*IYZij - IXYij*Cyzq[i])/(double)(6*(n-1));

    }   // end second loop over j

    tCy+=n*Cyq[i]; tCxy+=n*Cxyq[i], tCyz+=n*Cyzq[i], tCxyz+=n*Cxyzq[i];

     T2 += Cxyzq[i]*Cyq[i] - Cyzq[i]*Cxyq[i];
  } // end loop over i

  Q = 0;
  // Alternatively one might determine T2 as the sum of the h[i]
  for (i=0;i!=n;i++)
    Q += h[i];

  //fprintf(stderr,"T2 = %1.12f, Q=%1.12f, T2/Q = %1.12f\n", pVal,T2,Q,T2/Q);

  T2 /= (double)(n);
  for (i=0;i!=n;i++)
     h[i] -= T2;

  Q = (double) tCxyz/tCxy - (double) tCyz/tCy;

  C[0] = tCxyz/(double)(n*(n-1));
  C[1] = tCxy/(double)(n*(n-1));
  C[2] = tCyz/(double)(n*(n-1));
  C[3] = tCy/(double)(n*(n-1));

  for (i=0;i!=4;i++)
   for (j=0;j!=n;j++)
   {
     A[i][j] /= (double)(n-1);
     A[i][j] -= C[i];
   }

  free (Cyq);
  free (Cxyq);
  free (Cxyzq);
  free (Cyzq);

}


void InsertionSort(double *X, int *S, int M)
{
    int i, *I;
    int j;
    int r;
    double R;

    I= (int*) malloc (M*sizeof(int));

    for (i=0;i<M;i++)
      I[i]=i;

    for (i=1; i<M; i++)
      {
        R = X[i];
        r = i;
	for (j=i-1; (j>=0) && (X[j]>R); j--)
        {
	  X[j+1] = X[j];
          I[j+1] = I[j];
        }
	X[j+1] = R;
        I[j+1] = r;
      }
    for (i=0; i<M; i++)
      S[I[i]]=i;

}


void  uniform (double *X, int M)
{
  int *I, i;

  I = (int*) malloc (M*sizeof(int));
  InsertionSort(X, I, M);

  for (i=0;i<M;i++)
    X[i] = (double) I[i]/M*3.464101615;        // to make unit variance

}


/* normalize the time series to unit std. dev. */

void normalise(double *x, int N)
{
  int i;
  double mean=0.0, var=0.0;

  for (i=0;i!=N;i++)
  {
    mean += x[i];
    var += x[i]*x[i];
  }

  mean /= (double)(N);
  var /= (double)(N);
  var -= mean*mean;

  for (i=0;i!=N;i++)
    x[i] = (x[i]-mean)/sqrt(var);

  return;
}

void testcaus(double *z, double *x, double *y, double *q, double *u, int N, double epsilon, int U)
{

  int i, j, k, l, mmax;
  double S2, VT2, d[4], sigma[4][4];
  
  mmax=Mmax;
  
  if (U<=0)
  {
    normalise(z, N);
    normalise(x, N);
    normalise(y, N);
    normalise(q, N);
    normalise(u, N);
  }
  else
  {
    uniform(z, N);
    uniform(x, N);
    uniform(y, N);
    uniform(q, N);
    uniform(u, N);
  }
  
  redun(x,y,z,q,u,N,epsilon);
  
  for (i=0;i!=4;i++)
  {
    for (j=0;j!=4;j++)
    {
      sigma[i][j] = 0.0;
      for (k=0;k!=K;k++)
      {
        for (l=mmax+k;l!=N;l++)
          sigma[i][j] += 4.0*ohm[k]*(A[i][l]*A[j][l-k]+A[i][l-k]*A[j][l])/(double)(2*(n-k));
      }
    }
  }

  S2=0.0;
    
  d[0] = 1.0/C[1];
  d[1] = -C[0]/(C[1]*C[1]);
  d[2] = -1.0/C[3];
  d[3] = C[2]/(C[3]*C[3]);

  for (i=0;i!=4;i++)
    for (j=0;j!=4;j++)
      S2 += d[i]*sigma[i][j]*d[j];

   HJ_TVAL = Q*sqrt(n)/sqrt(S2);

  /* determine autocovariance of h[i] */
    
  for (k=0;k!=K;k++)
  {
    cov[k] = 0.0;
    
    for (i=mmax+k;i!=N;i++)
      cov[k] += h[i]*h[i-k];

    cov[k] /= (double)(n-k);
  } 
    
  /* variance of T2 */
	
  for (k=0;k!=K;k++)
    VT2 += 9.0*ohm[k]*cov[k];
  
  T2_TVAL = T2*sqrt(n)/sqrt(VT2);

  if (HJ_TVAL>0)
     p_HJ = 0.5 - 0.5*erf(HJ_TVAL/sqrt(2.0));
  else
     p_HJ = 0.5 + 0.5*erf(-HJ_TVAL/sqrt(2.0));

  if (T2_TVAL>0)
     p_T2 = 0.5 - 0.5*erf(T2_TVAL/sqrt(2.0));
  else
     p_T2 = 0.5 + 0.5*erf(-T2_TVAL/sqrt(2.0));
     
}


int main()
{
  char filename1[128], filename2[128], filename3[128], filename4[128], filename[128];	
  double z[Ndat], x[Ndat], y[Ndat], q[Ndat], u[Ndat], xstore[Ndat], ystore[Ndat], qstore[Ndat], ustore[Ndat], tmp, epsilon=EPS;
  int i, j, k, m, N;
  
  for (i=0;i!=N;i++)
    x[i]=0;
   for (i=0;i!=N;i++)
    y[i]=0;
   for (i=0;i!=N;i++)
    q[i]=0;
   for (i=0;i!=N;i++)
    z[i]=0;
   for (i=0;i!=N;i++)
    u[i]=0;
   for (i=0;i!=N;i++)
    xstore[i]=0;
   for (i=0;i!=N;i++)
    ystore[i]=0;
   for (i=0;i!=N;i++)
    qstore[i]=0;
   for (i=0;i!=N;i++)
    ustore[i]=0;

  /* reading files */

  FILE *infil, *outfil;

  printf("Input file (X): "); scanf("%s", filename1);

  if ( (infil=fopen(filename1,"r")) == NULL)
  {
    fprintf(stderr,"Error: unable to open file. Exiting...\n");
    exit(1);
  }

  i = 0;
  while (fscanf(infil,"%lf", &tmp) != EOF)
  {
      x[i] = tmp;
      xstore[i] = tmp;
      i++;
  }

  N = i;

  printf("%d data read\n", i);
 
  fclose(infil);

  printf("Input file (Y): "); scanf("%s", filename2);

  if ( (infil=fopen(filename2,"r")) == NULL)
  {
    fprintf(stderr,"Error: unable to open file. Exiting...\n");
    exit(1);
  }

  i = 0;

  while (fscanf(infil,"%lf", &tmp) != EOF)
  {
      y[i] = tmp;
      ystore[i] = tmp;
      i++;
  }

  printf("%d data read\n", i);

  fclose(infil);

  if (i!=N)
  {
    fprintf(stderr,"Error: data number mismatch (X and Y). Exiting...\n");
    exit(1);
  }

  printf("Input file (Q): "); scanf("%s", filename3);

  if ( (infil=fopen(filename3,"r")) == NULL)
  {
    fprintf(stderr,"Error: unable to open file. Exiting...\n");
    exit(1);
  }

  i = 0;

  while (fscanf(infil,"%lf", &tmp) != EOF)
  {
      q[i] = tmp;
      qstore[i] = tmp;
      i++;
  }

  printf("%d data read\n", i);

  printf("Input file (U): "); scanf("%s", filename4);

  if ( (infil=fopen(filename4,"r")) == NULL)
  {
    fprintf(stderr,"Error: unable to open file. Exiting...\n");
    exit(1);
  }

  i = 0;

  while (fscanf(infil,"%lf", &tmp) != EOF)
  {
      u[i] = tmp;
      ustore[i] = tmp;
      i++;
  }

  printf("%d data read\n", i);
  
  fclose(infil);
  
  if (i!=N)
  {
    fprintf(stderr,"Error: data number mismatch (Q and X/Y). Exiting...\n");
    exit(1);
  }
   
  /* calculating bandwidths */
  EPS = C_EPS*pow(N,rate_EPS);

  eps2 = C_eps2*pow(N,rate_eps2);

  C2 = EPS*EPS/(2*eps2*eps2);
  epsilon = EPS;
  
  A = (double **) malloc(4*sizeof(double *));
  for (i=0;i!=4;i++)
    A[i] = (double *) malloc(N*sizeof(double));
  
  h = (double *) malloc(N*sizeof(double));
  
  K = (int)(sqrt(sqrt(N)));
  ohm = (double *) malloc(K*sizeof(double));
  cov = (double *) malloc(K*sizeof(double));

  ohm[0] = 1.0;
  for (k=1;k<K;k++)
    ohm[k] = 2.0*(1.0-k/(double)(K));

  /* Generate output file names */

  i=0;

  while (filename1[i]!='\0' && filename1[i]!='.')
  {
    filename[i]=filename1[i];
    i++;
  }
  filename1[i]='\0';
  filename[i]='_';

  j=0;

  while (filename2[j]!='\0' && filename2[j]!='.')
  {
    filename[i+j+1]=filename2[j];
    j++;
  }

  filename2[j]='\0';
  filename[i+j+1]='.';  filename[i+j+2]='o'; filename[i+j+3]='u'; 
  filename[i+j+4]='t'; filename[i+j+5]='\0';


  if ( (outfil=fopen(filename,"w")) == NULL)
  {
    fprintf(stderr,"Error: unable to open file for writing.\
	Exiting...\n");
    exit(1);
  }
  
  /* write to the files */ 
  
  fprintf(outfil,"file 1: %s\n", filename1);
  fprintf(outfil,"file 2: %s\n", filename2);
  fprintf(outfil,"series length: %d\n",N);
  fprintf(outfil,"epsilon=%f\n",epsilon);

  fprintf(stderr,"epsilon=%f\n",epsilon);
  fprintf(outfil,"epsilon=%f\n",epsilon);


  /* generate vector z */
  for (i=0;i<N-1;i++)
   z[i] = y[i+1]; 
   
  //X -> Y (corrected for Q, normal)
  fprintf(stderr,"\n%s does not granger cause %s (corrected for the presence of %s and %s)\n", filename1, filename2, filename3, filename4);
  fprintf(outfil,"\n%s does not granger cause %s (corrected for the presence of %s and %s)\n", filename1, filename2, filename3, filename4);
  
  testcaus(z,x,y,q,u,N,epsilon,0); /*data sharpening normal distribution adjustment*/

  fprintf(stderr,"lX=lY=lQ=%d, p_T2=%1.6f T_T2=%1.6f, p_HJ=%1.6f T_HJ=%1.6f\n",Mmax,p_T2,T2_TVAL,p_HJ,HJ_TVAL);
  fprintf(outfil,"lX=lY=lQ=%d, p_T2=%1.6f T_T2=%1.6f, p_HJ=%1.6f T_HJ=%1.6f\n",Mmax,p_T2,T2_TVAL,p_HJ,HJ_TVAL);
  
  //X -> Y (corrected for Q, uniform)
  fprintf(stderr,"\n%s does not granger cause %s (corrected for the presence of %s and %s) - UNIF \n", filename1, filename2, filename3, filename4);
  fprintf(outfil,"\n%s does not granger cause %s (corrected for the presence of %s and %s) - UNIF \n", filename1, filename2, filename3, filename4);
  
  testcaus(z,x,y,q,u,N,epsilon,1); /*data sharpening uniform distribution adjustment*/

  fprintf(stderr,"lX=lY=lQ=%d, p_T2=%1.6f T_T2=%1.6f, p_HJ=%1.6f T_HJ=%1.6f\n",Mmax,p_T2,T2_TVAL,p_HJ,HJ_TVAL);
  fprintf(outfil,"lX=lY=lQ=%d, p_T2=%1.6f T_T2=%1.6f, p_HJ=%1.6f T_HJ=%1.6f\n",Mmax,p_T2,T2_TVAL,p_HJ,HJ_TVAL);
  
  
  /* clear the values  and load again */
  
   for (i=0;i!=N;i++)
    x[i]=0;
   for (i=0;i!=N;i++)
    y[i]=0;
   for (i=0;i!=N;i++)
    q[i]=0;
   for (i=0;i!=N;i++)
    z[i]=0;
   for (i=0;i!=N;i++)
    u[i]=0;
    
  A = (double **) malloc(4*sizeof(double *));
  for (i=0;i!=4;i++)
    A[i] = (double *) malloc(N*sizeof(double));
  
  h = (double *) malloc(N*sizeof(double));
  
  K = (int)(sqrt(sqrt(N)));
  ohm = (double *) malloc(K*sizeof(double));
  cov = (double *) malloc(K*sizeof(double));

  ohm[0] = 1.0;
  for (k=1;k<K;k++)
    ohm[k] = 2.0*(1.0-k/(double)(K));
    
  for (i=0;i!=N;i++)
   x[i] = xstore[i]; 
  for (i=0;i!=N;i++)
   y[i] = ystore[i]; 
  for (i=0;i!=N;i++)
   q[i] = qstore[i]; 
  for (i=0;i!=N;i++)
   u[i] = ustore[i];
   
  /* generate vector z */
  for (i=0;i<N-1;i++)
   z[i] = x[i+1];  
   
  //Y -> X (corrected for Q, normal)
  fprintf(stderr,"\n%s does not granger cause %s (corrected for the presence of %s and %s)\n", filename2, filename1, filename3, filename4);
  fprintf(outfil,"\n%s does not granger cause %s (corrected for the presence of %s and %s)\n", filename2, filename1, filename3, filename4);
 
  testcaus(z,y,x,q,u,N,epsilon,0); /*data sharpening normal distribution adjustment*/

  fprintf(stderr,"lX=lY=lQ=%d, p_T2=%1.6f T_T2=%1.6f, p_HJ=%1.6f T_HJ=%1.6f\n",Mmax,p_T2,T2_TVAL,p_HJ,HJ_TVAL);
  fprintf(outfil,"lX=lY=lQ=%d, p_T2=%1.6f T_T2=%1.6f, p_HJ=%1.6f T_HJ=%1.6f\n",Mmax,p_T2,T2_TVAL,p_HJ,HJ_TVAL);
  
  //Y -> X (corrected for Q, uniform)
  fprintf(stderr,"\n%s does not granger cause %s (corrected for the presence of %s and %s) - UNIF \n", filename2, filename1, filename3, filename4);
  fprintf(outfil,"\n%s does not granger cause %s (corrected for the presence of %s and %s) - UNIF \n", filename2, filename1, filename3, filename4);

  testcaus(z,y,x,q,u,N,epsilon,1); /*data sharpening uniform distribution adjustment*/

  fprintf(stderr,"lX=lY=lQ=%d, p_T2=%1.6f T_T2=%1.6f, p_HJ=%1.6f T_HJ=%1.6f\n",Mmax,p_T2,T2_TVAL,p_HJ,HJ_TVAL);
  fprintf(outfil,"lX=lY=lQ=%d, p_T2=%1.6f T_T2=%1.6f, p_HJ=%1.6f T_HJ=%1.6f\n",Mmax,p_T2,T2_TVAL,p_HJ,HJ_TVAL);
  
  fclose(outfil);

  return(0);
}
