%--------------------------------------------------------------------------
% Script to compute solution of the consumption model using OccBin 
%
% This code loops over a 100 point grid of values of the parameter GAMMA 
% in the consumpion model described in: "Likelihood Evaluation of DSGE Models 
% with Occassionally Binding Constraints" 
% 
% Pablo Cuba-Borda, Luca Guerrieri, Matteo Iacoviello and Molin Zhong
% Federal Reserve Board. Washington, D.C. 
%
% Created: 11/01/2017.
% Last modified: 04/23/2019.
%--------------------------------------------------------------------------

clear all
setpathdynare4
warning off

global oo00_  M00_ M10_  M01_  M11_ M_

global params_labels params

global cof cof10 cof01 cof11 ...
    Jbarmat Jbarmat10 Jbarmat01 Jbarmat11 ...
    Dbarmat10 Dbarmat01 Dbarmat11 ...
    decrulea decruleb

global filtered_errs_switch filtered_errs_init model_temp
global datavec irep xstory fstory

irep=1; datavec=[]; xstory=[]; fstory=[];

% 0: uses csolve; 1: csolve with gradient; 2: fsolve
solver=0;

set(0,'DefaultLineLineWidth',2)
randn('seed',1);
format compact

%-------------------------------
% INITIALIZE PARAMETERS
%-------------------------------
R    = 1.05;
BETA = 0.945;
RHO  = 0.90;
STD_U= 0.0100;
M = 1;
GAMMAC=1;

save PARAM_EXTRA_BABY R BETA RHO STD_U GAMMAC


% load grid for GAMMA
GAMMAVEC = dlmread('GAMMAVEC.txt');

% Grid points
gridpoints.nb=500;
gridpoints.ny=100;

% Grid points for debt
nb=gridpoints.nb;

% Income grid
nz=gridpoints.ny;
widthz=3.2;

% Get Tauchen Approximation
[ P, logz, ~, ~, ~ ] = markovappr(RHO,STD_U,widthz,nz) ;
Z=exp(logz);

% Bounds for discretized grid
bmin=0.75*M;
bmax=M*Z(end) ;

% Construct grid for debt/assets
B=linspace(bmin,bmax,nb);

%-----------------------------------
% Create scripts to speed up filtering
%-----------------------------------

modnam_00 = 'borrcon00'; % base model (constraint 1 and 2 below don't bind)
modnam_10 = 'borrcon10'; % first constraint is true
modnam_01 = 'borrcon00'; % second constraint is true
modnam_11 = 'borrcon00'; % both constraints bind
constraint1 = 'lb<-lb_ss';
constraint_relax1 = 'b>bnot';
constraint2 = 'lb<-1000000';
constraint_relax2 = 'lb>-1000000';
curb_retrench =0;
maxiter = 20;
obs_list = char('c');
err_list = char('eps_u');


call_pre_estimation_script

GAMMAVECTOR = GAMMAVEC;
for param_number = 1:numel(GAMMAVEC)
% for param_number = 1
    GAMMAC = GAMMAVEC(param_number);
    fprintf('\n Solving for GAMMAC = %4.4f \n',GAMMAC);
    Cdectemp = zeros(length(Z),length(B));
    Bdectemp = zeros(length(Z),length(B));
    
    save PARAM_EXTRA_CALIBRATED R BETA RHO STD_U GAMMAC M
    save PARAM_EXTRA_BABY.mat STD_U GAMMAC    
    
    eval(modnam_00)
    oo00_ = oo_;
    M00_ = M_;
    
    eval_param;
    
    zdatass = oo00_.dr.ys;
    
    [hm1,h,hl1,Jbarmat] = get_deriv(M00_,zdatass);
    cof = [hm1,h,hl1];
    
    
    M10_.params = M00_.params;
    [hm1,h,hl1,Jbarmat10,resid] = get_deriv(M10_,zdatass);
    cof10 = [hm1,h,hl1];
    Dbarmat10 = resid;
    
    if isempty(modnam_01)==0
        
        M01_.params = M00_.params;
        [hm1,h,hl1,Jbarmat01,resid] = get_deriv(M01_,zdatass);
        cof01 = [hm1,h,hl1];
        Dbarmat01 = resid;
        
        M11_.params = M00_.params;
        [hm1,h,hl1,Jbarmat11,resid] = get_deriv(M11_,zdatass);
        cof11 = [hm1,h,hl1];
        Dbarmat11 = resid;
        
    else
        
        cof01=[];
        cof11=[];
        
        Dbarmat01 = [];
        Dbarmat11 = [];
        
        Jbarmat01=[];
        Jbarmat11=[];
        
    end
    
    [decrulea,decruleb]=get_pq(oo00_.dr);
    
    nvars          = M00_.endo_nbr;
    ys_            = oo00_.dr.ys;
    endog_         = M00_.endo_names;
    exog_          = M00_.exo_names;
    
    
    % Evaluate model parameters
    % eval_param
    params(1)=M00_.params(1);   % RHO
    params(2)=M00_.params(2);   % BETA
    params(3)=M00_.params(3);   % M
    params(4)=M00_.params(4);   % R
    params(5)=M00_.params(5);   % STD_U
    params(6)=M00_.params(6);   % GAMMAC
    
    % Compute OccBin Related Objects
    [~, i1, ~]=intersect(M00_.endo_names,obs_list,'rows');
    [~, i2, ~]=intersect(M00_.exo_names,err_list,'rows');
    
    current_obs = [];
    
    
    %We need to change both of these grids into deviations from steady
    %state
    for ii = 1:length(Z)
        for jj = 1:length(B)
            
            init_val_old = zeros(7,1);
            init_val_old(1) = log(B(jj)/M);
            init_val_old(7) = log(Z(ii)/1)/RHO; %We want income to be equal to Z(ii) in the first period
            
            [outsim grad init_out E newviolvecbool relaxconstraint iter] = simulate_occbin_v2(...
                0,err_list,obs_list,current_obs,init_val_old,...
                constraint1_difference,constraint_relax1_difference,i1,i2,nvars,ys_,endog_,exog_,params,...
                decrulea,decruleb,cof,Jbarmat,cof10,Jbarmat10,Dbarmat10);
            
            Cdectemp(ii,jj) = outsim(3);
            Bdectemp(ii,jj) = outsim(1);
        end
    end
    Cdec{param_number} = exp(Cdectemp);
    Bdec{param_number} = exp(Bdectemp);
end

save('PF_OCCBIN_GRID_GAMMA1_01_45.mat','Bdec','Cdec','GAMMAVEC','R','BETA','RHO','STD_U','M','B','Z','P');