%==========================================================================
% Function to compute likelihood using Particle Filter + VFI Solution
%
% This code loops over 100 samples of model simulated data from the DGP
% described in the paper: "Likelihood Evaluation of DSGE Models with
% Occassionally Binding Constraints" 
% 
% Pablo Cuba-Borda, Luca Guerrieri, Matteo Iacoviello and Molin Zhong
% Federal Reserve Board. Washington, D.C. 
%
% Input: jobID = Sample path to be evaluated. Type: string. Values: 0-99
% Output:   text file = ['MC100GAM1_' jobId '_VFI_ME' num2str(me_size*100) '_sample' num2str(sampsize) '.txt']
%
% MEX File compilation: mex myparticlefilterv3_mex.F
%
% Created: 11/01/2017.
% Last modified: 04/23/2019.
%
%==========================================================================

function run_Filter_PFVFI_ME05_500(jobId)

%-------------------------------------------
% Housekeeping
%-------------------------------------------
disp(jobId);
% load(solutionfile);
jobNum = str2double(jobId);


%-------------------------------------------
% Solution File
%-------------------------------------------
% Discretized solution over a 15x200 grid of income x debt
% Policy function is Bdec{param_number} where each cell-element correspond
% to a solution for a different value of GAMMAC included in the vector
% GAMMAVEC.


% Load stored VFI solutions
load('PF_VFI_GRID_GAMMA1_01_45.mat')


ngrid =numel(GAMMAVEC);

%-------------------------------
% INITIALIZE PARAMETERS
%-------------------------------
fprintf(' R = %4.5f \n',R);
fprintf(' RHO = %4.5f \n',RHO);
fprintf(' STD_U = %4.5f \n',STD_U);
fprintf(' M = %4.5f \n',M);
fprintf(' GAMMAVEC(1)   = %4.5f \n',GAMMAVEC(1));
fprintf(' GAMMAVEC(end) = %4.5f \n',GAMMAVEC(ngrid));

% Attach matrices loaded from solution file to workspace
B = B;
P = P;
Z = Z;
GAMMAVEC=GAMMAVEC;
%==========================================================================
%                   Particle Filter + Global Solution
%==========================================================================

% LOAD DATA
simC = dlmread('simC.txt');

obs = simC(:,jobNum+1);
obs=obs(end-499:end,:);
sampsize = size(obs,1);

% CONFIGURATION OF PARTICLE FILTER
vardata = std(obs)^2;

% OPTIONS FOR FILTER
me_size = 0.05;
sigm  = sqrt(me_size*vardata);
np    = 5000000;
disp(np);
disp(me_size);
disp(sampsize);

% Initialize pool of workers 
% Note: if multicore-procession available use parfor in line 94.
% parpool(24);

%---------------------------------
% Loop over parameters
%---------------------------------
tic;

fprintf('\n *** Solving for true GAMMA=1 ***');

rng(13579);

for param_number = 1:numel(GAMMAVEC)


  GAMMAC = GAMMAVEC(param_number);
  PF_B = Bdec{param_number};
  PF_C = PF_B;

  fprintf('\n Evaluating for param_number = %i, GAMMAC = %4.4f',param_number,GAMMAC);

  fprintf('\n Drawing shocks for PF');


        seed = (123456789-123456).*rand(1,1) + 123456;
        ERRMAT    = zeros(1,np);
        like = myparticlefilterv3_mex(B',P,Z',PF_B,PF_C,obs,sigm,ERRMAT,seed);

        liki_vec = sum(like);

  out(param_number,:) = [param_number, GAMMAC, liki_vec];


end
toc;
dlmwrite(['MC100GAM1_' jobId '_VFI_ME' num2str(me_size*100) '_sample' num2str(sampsize) '.txt'],out,'precision','%4.6f')
