######################################
# Bayesian Model Averaging Program   #
######################################
# This version: 2008-11-05
# Martin Feldkircher
# martin.feldkircher@gzpace.net, http:feldkircher.gzpace.net
# Stefan Zeugner
# stefan.zeugner@gmail.com, http:www.zeugner.eu
#####################
# The main code starts at line 371 with the function "fls=function(....)" and is written
# by Martin Feldkircher and Stefan Zeugner as part of their work at the Institute for Advanced Studies (IHS),
# Vienna in 2006/2007. Descriptions of the algorithms and priors used can be found in
# Gary Koop ("Bayesian Econometrics", Wiley & Sons), Fernandez, C., E. Ley and M.F.J. Steel (2001b) 
# "Model Uncertainty in CrossCountry Growth Regressions," Journal of Applied Econometrics and
# and Fernandez, C., E. Ley and M.F.J. Steel (2001a) "Benchmark Priors for Bayesian Model Averaging,"
# Journal of Econometrics, 100: 381-427. 

####################
# USAGE            # 
###################################################################################################
# fls(X.data,burn=100,iter=1000,g="bric",start.value=41,theta="random", prior.msize=7,mcmc="bd",nmodel=100,
#     logfile=FALSE,logstep=100000,beta.save=TRUE,exact=FALSE,printRes=TRUE,int=FALSE,ask.set=FALSE)
#
#
####################
# PARAMETERS       #
#                  #
###################################################################################################
#X.data       submit a data frame or a matrix, where the first column corresponds to the dependent variable
#             followed by the covariates, including a constant term is not necessary since y and X is 
#             demeaned automatically.
#burn         is the number of burn-in draws
#iter         is the number of posterior draws
#g            is the hyperparameter on Zellner's g-prior for the regression coefficients. You can specify
#             g="bric" corresponding to the benchmark prior suggestion from FLS (2001), i.e   g=1/max(N, K^2)
#             with K denoting the total number of regressors and N the number of observations
#start.value  specifies the starting model. You can choose either a specific model by the corresponding
#             column indices (e.g. starting.model=numeric(ncol(X)) starts from the null model including 
#             solely a constant term) or you set a number (e.g. starting.model=20). In the latter case
#             randomly 20 covariates are chosen and the starting model is identified by those regressors
#             with a t-statistics>0.2.
#             start.value=0 or start.value=NULL starts from the null model
#             start.value=NA sets start.value=min(N-1,K)
#theta        regards the prior on model size. It can be either "random" or "fixed" and is based on the 
#             working paper "On the Effect of Prior Assumptions in Bayesian Model Averaging with Applications
#             to Growth Regression", by  Ley and Steel (2008). Theta denotes the a priori inclusion probability
#             of a regressor. Their suggestion is to use a binomial-beta hyperprior on theta (i.e. theta=random)
#             in order to be noninformative on model size. You can use theta=fix if you have strong prior
#             information on model size. 
#prior.msize  corresponds to the expected value of the model size prior. For theta=random there is little
#             impact on results by varying prior.msize. For fixed theta (i.e. informative) prior this is a
#             sensible choice. 
#mcmc         default is "bd" which corresponds to a birth / death MCMC alogrithm. You can choose
#             also "revjump" where we have implemented a true reversible jump algorithm where we have 
#             added a "move" step to the birth / death steps from "bd".
#nmodel       is the number of best models for which information is stored. Convergence analysis of
#             the sampler by means of the correlation between analytic posterior  model probabilities
#             (pmp's) and hat of the MCMC sampler is based on the number you  have set in nmodels. Also
#             if you want to save the regression coefficients (beta.save=T), they are taken from the
#             nmodel best models. Setting nmodel500 slows down the MCMC sampler. Note that posterior
#             inclusion probabilites, and mean calculations are based on the MCMC frequencies as opposed
#             to exact analytical calculations (as in fls).
#             Set nmodel=0 if you do not want to save this information.
#beta.save    can be either TRUE or FALSE, and specifies whether you want to save the regression coefficients
#             for the "nmodel" best models.         
#logfile      setting logfile=TRUE produces a logfile named "fls.log" in your current working directory,
#             in order to keep track of the sampling procedure.
#             setting logfile equal to some filepath (like "subfolder/bla.txt") puts the logfile 
#             into that specified position.
#logstep      specifies at which number of posterior draws information is written to the log file
#printRes     print out results to command line after ending the routine
#exact        ??? EXPLANATION PLEASE!
#int          TRUE / FALSE indicates whether the X.data matrix consists of interaction terms. In case you specify TRUE
#             interaction terms will only be sampled along with their component variables (e.g. AB+A+B, models
#             do not consist of all the three terms are not allowed due to interprational issues of the resulting coefficients
#ask.set      TRUE or FALSE whether R should pause between the graphs drawn at the end of the routine (only matters in interactive mode)
#hstep        is for using bma as a forecasting tool, hstep corresponds to the number of periods you want to forecast
#             hstep=NULL is the standard setting
##############################################################################################
#Output            #
#                  #
###################################################################################################
#estimates    a matrix....PIP denotint Posterior Inclusion Probabilities based on the relative frequencies
#             of the MCMC sampler, post mean = posterior mean, post SD = posterior standard deviation
#             and Idx is the column index of the specific regressor in the submitted data matrix
#info         contains information about the "Mean nr. of Regressors" (not counting the constant term),
#             "Draws"=posterior draws, "Burnins"=nr. of burnins taken, "Time" denotes total time elapsed
#             since calling the "fls" function, "Nr. of models visited" counts each time a model is accepted.
#             Note that we do not take into account the case of revisiting models by the sampler.
#             Modelspace is simply indicating the whole model space (2^K) and percentage visited is 
#             the nr. of models visited as a percentage of 2^K. "Corr PMP" is the correlation between
#             analytic and MCMC posterior model probabilites, where a correlation of 0.99 indicates 
#             excellent convergence. For "nmodel=100" the best 100 models are considered for the correlation
#             analysis. Finally, Nr. of Observations is given in the output as well.
# 
#             -----The following objects are saved in your output object-------
#topmodels    matrix of dimension K x nmodel and consists of the nmodel best models where zero stands for
#             exclusion of a regressor and 1 for inclusion
#k.vec        is a vector of length K+1 and indicates how many model werde drawn with a size of k.vec[i] regressors, 
#             where the first entry in k.vec corresponds to the nullmodel (hence K+1 entries)
#start.pos    is the chosen starting model
#beta.draws   matrix of stored regression coefficients if specified via beta.save=TRUE
#topmodprob   analytical posterior model probabilities of the best "nmodel" models
#pmp.10       Matrix where the first column represents analytical and the second MCMC model probabilities
#             they are normalized such that each column sums up to 1.
#topmodobject function to save pmp and betas
#theta        theta as specified by the user
#K            nr. of regressors
#prior.msize  prior model size as specified by the user
#             The plot of the model size shows the prior as compared to the posterior model size and is done automatically after the
#             routine finishes. To redo the plot use     modelsize.plot(X=result,line.width=2,sub="This is my plot"),
#             where comes from result=fls(......................)  
#fls.call     the function call as it was entered into the command line



###### FLS MAIN FUNCTION #########################


fls <-function(X.data,burn=100,iter=1000,g="bric",start.value=41,theta="random", prior.msize=7,
                mcmc="bd",nmodel=100,logfile=FALSE,logstep=100000,beta.save=TRUE,printRes=T,exact=F,int=F,ask.set=F){

  sPath=paste("C:/DATA/CRESPO/Work in Progress/BMA AFRICA/REPLICATION/R/",sep="")
  if (!exists("aux_bma")) {aux_bma=paste(sPath, "aux_bma_20081105.r",sep="")};source(aux_bma,local=T)
  #source(paste(sPath,"plotNformat.r",sep=""),local=T)
  y<-as.matrix(X.data[,1])
  X<-as.matrix(X.data[,2:ncol(X.data)])
  N<-nrow(X)
  K=ncol(X)
  g0=NA

  # User Checks:
  if(prior.msize>=K){
    cat("Submitted prior model size is >= than the nr. of   regressors\n, used K/2 instead\n\n")
    prior.msize=K/2
  }

  if(exact & !beta.save){
      cat("For exact posterior inference betas have to be saved\n, have set beta.save=TRUE instead\n\n")
      beta.save=TRUE
  }
  if (nmodel[1]<0|is.na(nmodel[1])) {nmodel=0}
  
 # subtract mean from all regressors as in FLS
  y.mean=mean(y)
  y<-y-matrix(y.mean,N,1,byrow=T)
  X.mean=colMeans(X)
  X<-X-matrix(X.mean,N,K,byrow=T)


 # multiply the whole matrix stuff out before going into the simulation loops
  XtX.big=crossprod(X)
  Xty.big=crossprod(X,y)
  yty = as.vector(crossprod(y))


######################################################################################################################################
#The function Starter selects randomly a start matrix and runs a
#regression. From this regression, the
#start Design matrix is that for which the t-stats are >0.2. So we
#can circumvent starting from a very bad start point.
  start.list=starter(K,start.value,y,N=N,XtX.big=XtX.big,Xty.big=Xty.big,X=X)
  molddraw=start.list$molddraw
  start.position=start.list$start.position
  kold=sum(molddraw)
  position=(1:K)[molddraw==1]

######################################################################################################################################
# for the case that X contains interaction terms
 if(int){
      if(length(grep("[.x.]",colnames(X.data),extended=T))==0){
          stop("Please separate column names of interaction terms by .x. (e.g. A.x.B)")
      }
      # this file constructs the molddraw matrix
      source(paste(sPath,"interaction_Sampler.r",sep=""),local=T)

      # overwrite matrix multiplication and  start model
      X=cbind(X.base,X.int)
      XtX.big=crossprod(X)
      Xty.big=crossprod(X,y)

      # we start with a model drawn from the pool of base models (without the interaction terms)
#      start.list=starter(K=k.base,start.value,y,N=N,XtX.big=XtX.big,Xty.big=Xty.big,X=X)
#      molddraw=c(start.list$molddraw,rep(0,k.int))
#      start.position=start.list$start.position
#      kold=sum(molddraw)
#      position=which(molddraw==1)
 }
 else{
  mMinus<-mPlus<-NA
 }

########################################################################################################################################
#specify g0 for the g-prior

# if user specifies a number for g, use this number instead of bric
  if(is.numeric(g)){
      g0=g
  }


if(g=="bric"){
  #else{
      if (N<=(K^2)){
          g0=1/(K^2)
      }
      else{
        g0=1/N
      }
}
  g2=1/(g0+1)

################################################################################################
#Initializing
###############################################################################################

  # calculate Likelihood for NullModel
  null.lik=((1-N)/2)*log(yty)
  null.count=0
  models.visited=0
  # initialize top 10 function
  lik.list=lprob(positions=(1:K)[molddraw==1],g0=g0,g2=g2,yty=yty,k=kold,N=N,null.lik=null.lik,
                     K=K,XtX.big=XtX.big,Xty=Xty.big)
  lprobold=lik.list$lprob
  b1=lik.list$b1new
  stdev=lik.list$stdevnew
  
  # specify model.odds function 1) "fixed theta", 2) or "uniform prior" or 3) for all other specifications use the random theta specification 
  pmp = switch(theta,"fix"=pmp.fix,"uniform"={prior.msize<<-K/2; pmp.unif},pmp.ran)
    
  # calculate the posterior model probability for the first model
  pmp2=pmp(ki=kold,K=K,m=prior.msize, lprobnew=lprobold)

  # initialize top 10 function
  topmods=top10(nmaxregressors=K,nbmodel=nmodel,bbeta=beta.save,
                inivec_lik=pmp(ki=0,K=K,m=prior.msize,lprobnew=null.lik))
  topmods$addmodel(mylik=pmp2,vec01=molddraw,vbeta=b1)





  # Initialize the rest
  mstart=molddraw
  lprobstart=pmp2
  inccount=rep(0,K)
  msize=0
  k.vec=rep(0,K)
  b1mo=numeric(K)          #calculate first and second moment of all coefficients
  ab=numeric(K)            #Initialize them here
  b2mo=numeric(K)
  bb=numeric(K)
  mnewdraw=numeric(K)
  int.terms.changed=FALSE

    


#######################################################################
#############################################################################################
    nrep=burn+iter                                #Burnins + Draws
    set.seed(as.numeric(Sys.time()))              #Set Seed randomly for number generator
    t1<-Sys.time()                                #Save time before going into the loop

   # generate logfile if desired
    if(logfile){
        if (is.character(logfile)) {
          sfilename=logfile}
        else {
          sfilename="fls.log"
        }
        file.create(sfilename)

      cat(as.character(Sys.time()),": starting loop ... \n",append=TRUE, file=sfilename)  #write one line
      fact=max(floor(nrep/100),logstep)
    }

 
 if(mcmc!="bd" & !int){
  sampling=rev.jump
 }
 if(mcmc!="bd" & int){
  sampling=rev.jump.int
 }
 if(mcmc=="bd" & int){
  sampling=fls.samp.int
 }
 if(mcmc=="bd" & !int){
    sampling=fls.samp
 }
 
###########################################################################################
#START MAIN LOOP
###########################################################################################
for (i in 1:nrep){
    if(logfile){
        if (i %% fact==0) {
        cat(as.character(Sys.time()),":",i,"current draw \n",
            "kold:",kold, "\n",append=TRUE, file=sfilename)  #write one line
        
        }
    }
##########################################################################################
#Start sampling program
###########################################################################################
      a=sampling(molddraw=molddraw,K=K,mMinus=mMinus,mPlus=mPlus)
      mnewdraw=a$mnewdraw
      positionnew=a$positionnew
      knew=length(positionnew) 
      if (int) {if (length(a$addi)>1|length(a$dropi)>1) {int.terms.changed=TRUE} else {int.terms.changed=FALSE}}
      #int.terms.changed = TRUE if there were multiple regs dropped or added due to interaction terms
      if(g=="EBL"){
        int.terms.changed=TRUE  # do this as long as zeugi does not fix the quick lik stuff for other g-priors
      }
    
      if (int.terms.changed) {
          lik.list.int=lprob(positions=positionnew,g0=g0,g2=g2,yty=yty,k=knew,N=N,null.lik=null.lik,
                     K=K,XtX.big=XtX.big,Xty=Xty.big) #in case of changing interaction terms, draw the big likelihood           
          lprobnew = lik.list.int$lprob
      } else {
          lprobnew=lik.list$child.lprob(a$addi,a$dropi,k=knew) #if standard sampling, use Frisch-Waugh to get the new lprob (faster)
      }
     
      pmp1=pmp(ki=knew,K=K,m=prior.msize, lprobnew=lprobnew)
      lratio=pmp1-pmp2
      #if (int.terms.changed) browser()
      
      #Now decide whether to accept candidate draw
      if(log(runif(1,min=0,max=1))< lratio){
          if (!int.terms.changed) {
              # in case one has used Frisch-Waugh and the new model got accepted,
              # calculate the 'real' inverse in order not to make copying mistakes
              lik.list=lprob(positions=positionnew,g0=g0,g2=g2,yty=yty,k=knew,N=N,null.lik=null.lik,
                     K=K,XtX.big=XtX.big,Xty=Xty.big)
          } else {
              lik.list=lik.list.int
          }
          
          position = positionnew
          lprobold=lik.list$lprob
          pmp2=pmp1 # get posterior odds for new model  if accepted
          molddraw=mnewdraw
          kold=knew   
          models.visited=models.visited+1 #does not account for revisiting models
      }


# Collect Posterior Draws
########################################################################
    if (i>burn){   
         stdev=lik.list$stdevnew; b1=lik.list$b1new                                      
         inccount = inccount + molddraw
          # compute average size of models
          msize=msize + kold
          # collect nr. of regressors in vector
          k.vec[kold]=k.vec[kold]+1
          if(!any(kold)){
            null.count=null.count+1
          }

          # add log(lik)*p(M) to topmodels
          topmods$addmodel(mylik=pmp2,vec01=molddraw,vbeta=b1)
          #calculating posterior properties of coefficients means
          b1mo[position]=b1mo[position]+b1;  
          b2mo[position]=b2mo[position]+stdev;
    }
}
###########################################################################################
#END MAIN LOOP
###########################################################################################

###########################################################################################
  t2<-Sys.time()
  timed<-difftime(t2,t1)
  
  post.inf=post.calc(inccount=inccount,iter=iter,msize=msize,b1mo=b1mo,b2mo=b2mo,y.mean=y.mean,X.mean=X.mean,
                      k.vec=k.vec,null.count=null.count,models.visited=models.visited,exact=exact,beta.save=beta.save,SAR=F,
                      topmods=topmods,X.data=X.data,burn=burn,timed=timed,theta=theta,m=prior.msize)

###########################################################################################

  # print results to console
  if(printRes){
    print(post.inf$post.mean)
    cat("\n")
    print(post.inf$info)
    cat("\n")
    print(timed)
  }

  result=list(estimates=post.inf$post.mean,time=timed,topmodels=post.inf$topmod,info=post.inf$info,
              k.vec=post.inf$k.vec,start.pos=sort(start.position),beta.draws=post.inf$beta.draws,
               topmodprob=post.inf$topmodprob,pmp.10=post.inf$pmp.10,topmodobject=topmods,theta=theta,
               K=K, prior.msize=prior.msize,cons=post.inf$cons,reg.names=post.inf$reg.names,exact=exact,fls.call=sys.call(0))



  class(result)=c("bma")
  # do modelsize plot
  #op=par(ask=ask.set)
  #try(modelsize.plot(X=result,line.width=2),silent=T)
  #try(plotConv(X=result,lwd=2))
  #par(op)
  return(result)
}


########################################################################################################################################

setwd("C:/DATA/CRESPO/Work in Progress/BMA AFRICA/REPLICATION/R/")

library(gmp)

dataFrame2=read.table("data.txt",header=T)
dataM2=dataFrame2
K=(ncol(dataM2)-1)

modelint1=fls(X.data=dataM2,burn=1000000,iter=2000000,g=1/(104),nmodel=1,theta="random", exact=F, prior.msize=K/2,logfile=F,mcmc="bd",start.value=0,beta.save=F,ask.set=F,int=T)
modelint2=fls(X.data=dataM2,burn=1000000,iter=2000000,g=1/(104),nmodel=1,theta="fixed", exact=F, prior.msize=K/2,logfile=F,mcmc="bd",start.value=0,beta.save=F,ask.set=F,int=T)

detach(package:gmp)
