/* nvssAnalysis.do v1.00           DC/SO/CQD               yyyy-mm-dd:2015-04-09
----|----1----|----2----|----3----|----4----|----5----|----6----|----7----|----8

This file uses NVSS (birth certificate) data on first and second births and runs
regressions on births by quarter allowing for additional controls, fixed effects,
and so forth.  All figures and tables from the paper and online appendix are pro
duced. In order to replicate all materials, simply replace the MAIN global below
with the location of this replication directory on your computer.

For additional details, refer to the README file in the replication directory.

*/

vers 11
clear all
set more off
cap log close

********************************************************************************
*** (1) globals and locals
********************************************************************************
global MAIN "/home/replication"

global DAT "$MAIN/data/NVSS"
global USW "$MAIN/data/auxiliary"
global OUT "$MAIN/results/NVSS"
global LOG "$MAIN/log"



log using "$LOG/nvssAnalysis.txt", text replace

#delimit ;
local estopt cells(b(star fmt(%-9.3f)) se(fmt(%-9.3f) par([ ]) )) stats        
             (N, fmt(%14.0gc) labels(Observations))
             collabels(none) label;
local Fnote  "F-test of age variables refers to the test that
              the coefficients on mother's age and age squared are jointly
              equal to zero. The critical value for rejection of joint
              insignificance is displayed below the F-statistic.";
local Xnote  "$ \chi^2 $ test of age variables refers to the test that
              the coefficients on mother's age and age squared are jointly
              equal to zero. The critical value for rejection of joint
              insignificance is displayed below the test statistic.";
local enote  "Heteroscedasticity robust standard errors are reported in
              parentheses.";
lab def mon 1 "Jan" 2 "Feb" 3 "Mar" 4 "Apr" 5 "May" 6 "Jun" 7 "Jul" 8 "Aug"
            9 "Sep" 10 "Oct" 11 "Nov" 12 "Dec";
#delimit cr

ssc install estout, replace

********************************************************************************
*** (2) Open data for descriptives
********************************************************************************
use          "$DAT/nvss2005_2013_all"
gen asian = mracerec == 4
gen other = mracerec==3|mracerec==4
keep if gestation!=.
gen expectMonth = conceptionMonth+9
replace expectMonth = expectMonth-12 if expectMonth>12
drop quarter2 quarter3 expectQuarter
gen quarter2 = expectMonth==4|expectMonth==5|expectMonth==6
gen quarter3 = expectMonth==7|expectMonth==8|expectMonth==9
gen expectQuarter = ceil(expectMonth/3)
lab var quarter2 "Quarter 2"
lab var quarter3 "Quarter 3"
keep if twin==1
replace twin=twin-1
gen birth = 1

********************************************************************************
*** (3a) Summary stats 
********************************************************************************
#delimit ;
local add `" "20-45 Married"   "20-45 unmarried, no paternity acnowledged"
             "20-45 Married"   "';
local nam Married noPaternity Married-b2;
#delimit cr
tokenize `nam'

gen Quarter1 = expectQuarter == 1 
gen Quarter2 = expectQuarter == 2 
gen Quarter3 = expectQuarter == 3 
gen Quarter4 = expectQuarter == 4 
lab var motherAge "\midrule\multicolumn{6}{l}{\bf Panel A: Mother} \\ Mother's Age (Years)"
lab var Quarter1    "\multicolumn{6}{l}{\bf Panel B: Child} \\ Quarter 1 Birth"
lab var Quarter2    "Quarter 2 Birth"
lab var Quarter3    "Quarter 3 Birth"
lab var Quarter4    "Quarter 4 Birth"
lab var college     "Some College +"
lab var ART         "Used ART$^{a}$ "
lab var WIC         "Received WIC food in Pregnancy$^{a}$ "
lab var BMI         "Pre-pregnancy BMI$^{a}$ "
lab var underweight "Pre-pregnancy Underweight (BMI $<$ 18.5)$^{a}$ "
lab var normalBMI   "Pre-pregnancy Normal Weight (18.5 $\leq$ BMI $<$ 25)$^{a}$ "
lab var overweight  "Pre-pregnancy Overweight (25 $\leq$ BMI $<$ 30)$^{a}$ "
lab var obese       "Pre-pregnancy Obese (BMI $\geq$ 30)$^{a}$ "
lab var female      "Female"
lab var asian       "Asian"
lab var other       "Other Race (Asian/Native American)"

local k=1
foreach type of local add {
    if `k'==1 local gg motherAge>=20&motherAge<=45&birthOrder==1&married==1
    if `k'==2 local gg motherAge>=20&motherAge<=45&birthOrder==1&mar_p=="N"
    if `k'==3 local gg motherAge>=20&motherAge<=45&birthOrder==2&married==1
    
    #delimit ;
    local Mum  motherAge black white other hispanic;
    local MumP college educCat smoker ART WIC BMI underwe normalBM overwe obese;
    local Kid  Quarter1 Quarter2 Quarter3 Quarter4 gestat premature female;
    #delimit cr

    preserve
    keep if smoker!=.&college!=.&twin==0&`gg'
    #delimit ;
    estpost tabstat `Mum' `MumP' `Kid', statistics(count mean sd min max)
    columns(statistics);
    esttab using "$OUT/sumStats_``k''.tex", replace label noobs
    cells("count(fmt(%12.0gc) label(N)) mean(fmt(3) label(Mean)) sd(fmt(2)
           label(Std. Dev.)) min(fmt(0) label(Min.)) max(fmt(0) label(Max.))")
    fragment tex nonumber nolines nomtitles nonotes;
    #delimit cr
    restore
    local ++k
}


********************************************************************************
*** (3b) Age plots by month (ART, no ART) [FIGURES 2, A2, A4]
********************************************************************************
*Footnote 13
corr highEd motherAge if married==1&birthOrder==1&motherAge>=20&motherAge<=45
corr smoke  motherAge if married==1&birthOrder==1&motherAge>=20&motherAge<=45
reg highEd motherAge if married==1&birthOrder==1&motherAge>=20&motherAge<=45
reg smoke  motherAge if married==1&birthOrder==1&motherAge>=20&motherAge<=45

local tL1  = string(sqrt((e(df_r)/1)*(e(N)^(1/e(N))-1)), "%5.3f")
dis `tL1'


#delimit ;
local add `" "20-45 Married" "20-45 unmarried, no paternity acnowledged"
             "20-45 Married" "';
local nam  Married noPaternity Married-b2;
#delimit cr

tokenize `nam'

local k=1
foreach type of local add {
    if `k'==1 local gg motherAge>=20&motherAge<=45&birthOrder==1&married==1
    if `k'==2 local gg motherAge>=20&motherAge<=45&birthOrder==1&mar_p=="N"
    if `k'==3 local gg motherAge>=20&motherAge<=45&birthOrder==2&married==1
    
    preserve
    keep if `gg'

    generat youngOld = 1 if motherAge>=28&motherAge<=31
    replace youngOld = 2 if motherAge>=40&motherAge<=45
    gen q2q3 = conceptionMonth>=4&conceptionMonth<=9
    reg q2q3 youngOld
    reg q2q3 youngOld if ART==1

    dis "Tests by quarter for All"
    reg Quarter1 youngOld
    reg Quarter2 youngOld
    reg Quarter3 youngOld
    reg Quarter4 youngOld
    
    dis "Tests by quarter for ART"
    reg Quarter1 youngOld if ART==1
    reg Quarter2 youngOld if ART==1
    reg Quarter3 youngOld if ART==1
    reg Quarter4 youngOld if ART==1    
    
    drop if youngOld==.|conceptionMonth==.
    count
    local NN = string(r(N),"%15.0fc")

    collapse (sum) birth, by(conceptionMonth youngOld)
    lab val conceptionMon mon
    bys youngOld: egen totalBirths = sum(birth)
    gen birthProportion = birth/totalBirths
    sort conceptionMonth youngOld

    local line1 lpattern(solid)    lcolor(black) lwidth(thick)
    local line2 lpattern(dash)     lcolor(black) lwidth(medium)

    #delimit ;
    twoway line birthProportion conceptionMonth if youngOld==1, `line1' ||
           line birthProportion conceptionMonth if youngOld==2, `line2'
    xaxis(1 2) scheme(s1mono) xtitle("Month of Conception", axis(2))
    xlabel(1(1)12, valuelabels axis(2)) ylabel(0.05(0.01)0.1)
    xlabel(1 "Oct" 2 "Nov" 3 "Dec" 4 "Jan" 5 "Feb" 6 "Mar" 7 "Apr" 8 "May" 9 "Jun"
           10 "Jul" 11 "Aug" 12 "Sep", axis(1)) xtitle("Expected Month")
    legend(label(1 "28-31 Year-olds") label(2 "40-45 Year-olds"))
    ytitle("Proportion of All Births")
    note("Number of observations = `NN'");
    graph export "$OUT/conceptionMonth_``k''.eps", as(eps) replace;
    #delimit cr
    restore

    preserve
    keep if `gg'
    generat youngOld = 1 if motherAge>=28&motherAge<=31
    replace youngOld = 2 if motherAge>=40&motherAge<=45

    drop if youngOld==.|conceptionMonth==.
    keep if ART==1
    count
    local NN = string(r(N),"%15.0fc")
    collapse (sum) birth, by(conceptionMonth youngOld)
    lab val conceptionMon mon

    bys youngOld: egen totalBirths = sum(birth)
    gen birthProportion = birth/totalBirths
    sort conceptionMonth youngOld

    local line1 lpattern(solid)    lcolor(black) lwidth(thick)
    local line2 lpattern(dash)     lcolor(black) lwidth(medium)

    #delimit ;
    twoway line birthProportion conceptionMonth if youngOld==1, `line1' ||
           line birthProportion conceptionMonth if youngOld==2, `line2'
    xaxis(1 2) scheme(s1mono) xtitle("Month of Conception", axis(2))
    xlabel(1(1)12, valuelabels axis(2)) ylabel(0.05(0.01)0.1)
    xlabel(1 "Oct" 2 "Nov" 3 "Dec" 4 "Jan" 5 "Feb" 6 "Mar" 7 "Apr" 8 "May" 9 "Jun"
           10 "Jul" 11 "Aug" 12 "Sep", axis(1)) xtitle("Expected Month")
    legend(label(1 "28-31 Year-olds") label(2 "40-45 Year-olds"))
    ytitle("Proportion of All Births") note("Number of observations = `NN'");
    graph export "$OUT/conceptionMonthART_``k''.eps", as(eps) replace;
    #delimit cr
    restore
    
    local ++k
}

********************************************************************************
*** (3c) Age plots by quarter [Figures 3, A3, A5]
********************************************************************************
tokenize `nam'
local k=1
foreach type of local add {
    if `k'==1 local gg motherAge>=20&motherAge<=45&birthOrder==1&married==1
    if `k'==2 local gg motherAge>=20&motherAge<=45&birthOrder==1&mar_p=="N"
    if `k'==3 local gg motherAge>=20&motherAge<=45&birthOrder==2&married==1
    
    preserve
    keep if `gg'
    tab motherAge, gen(_age)

    foreach Q in 2 3 {
        cap gen quarter`Q' = expectQuarter==`Q'
        lab var quarter`Q' "Quarter `Q'"
        reg quarter`Q' _age1-_age26 if motherAge>=20&motherAge<=45, nocons
        count if e(sample)==1
        local NN = string(r(N),"%15.0fc")
        local tL1  = sqrt((e(df_r)/1)*(e(N)^(1/e(N))-1))
        
        gen ageES`Q' = .
        gen ageLB`Q' = .
        gen ageUB`Q' = .
        gen ageNM`Q' = .
        foreach num of numlist 1(1)26 {
            replace ageES`Q' = _b[_age`num']                     in `num'
            replace ageLB`Q' = _b[_age`num']-`tL1'*_se[_age`num'] in `num'
            replace ageUB`Q' = _b[_age`num']+`tL1'*_se[_age`num'] in `num'
            replace ageNM`Q' = `num'+19                          in `num'
        }
    }
    
    local s1 lpattern(solid) lwidth(medthick)
    local s2 lpattern(dash)  lwidth(medium)    
    #delimit ;
    twoway connected ageES2 ageNM2 in 1/26, `s1' lcolor(red) mcolor(red) m(S)
    || line ageLB2 ageNM2 in 1/26,     `s2' lcolor(red)
    || line ageUB2 ageNM2 in 1/26,     `s2' lcolor(red)
    || connected ageES3 ageNM3 in 1/26,`s1' lcolor(blue) mcolor(blue) m(Oh)
    || line ageLB3 ageNM3 in 1/26,     `s2' lcolor(blue) xlabel(20(1)45) 
    || line ageUB3 ageNM3 in 1/26,     `s2' lcolor(blue) scheme(s1mono) 
    legend(order(1 "Point Estimate (Quarter 2)" 2 "95 % CI (Quarter 2)"
                 4 "Point Estimate (Quarter 3)" 6 "95 % CI (Quarter 3)"))
    xtitle("Mother's Age") ytitle("Proportion in Quarter" " ")
    note("Number of observations = `NN'");
    graph export "$OUT/quarter2-3Age_2045_``k''.eps", as(eps) replace;
    #delimit cr

    local ++k
    restore
}

********************************************************************************
*** (3d) Births per month [Figure 1a]
********************************************************************************
preserve
keep if motherAge>=20&motherAge<=45&birthOrder==1

foreach mr in 1 0 {
    foreach qu of numlist 1(1)4 {
        local dq=(`qu'-1)*3+1
        local uq=(`qu'*3)
        gen Quarter`qu'_`mr' = birthMonth>=`dq'&birthMonth<=`uq'
        reg Quarter`qu'_`mr' if married==`mr'
        local LcritT_`mr' = sqrt((e(df_r)/1)*(e(N)^(1/e(N))-1))
        local seQ`qu'_`mr'=_se[_cons]
    }
}

count if married==1
local NNm = string(r(N),"%15.0fc")
count if married==0
local NNu = string(r(N),"%15.0fc")

append using "$USW/influenza"
lab var influenza "Influenza Activity"
lab var week "Week (Influenza)"
format birth %04.2f

gen quarter = ceil(birthMonth/3)
tab birthMonth quarter 
replace quarter = ceil(week/13) if week!=.
tab week quarter 
collapse influenza (sum) birth, by(quarter married)
bys married: egen totalBirth = total(birth)
replace birth = birth/totalBirth
lab var influenza "Influenza Activity"
gen birthUB=.
gen birthLB=.
foreach mr in 1 0 {    
    foreach qu of numlist 1(1)4 {
        sum birth if quarter==`qu'&married==`mr'
        replace birthUB=r(mean)+`LcritT_`mr''*`seQ`qu'_`mr'' if quarter==`qu'&married==`mr'
        replace birthLB=r(mean)-`LcritT_`mr''*`seQ`qu'_`mr'' if quarter==`qu'&married==`mr'
    }
}
sort married quarter
#delimit ;
twoway connected birth quarter if married==1, lcolor(black) lwidth(thick)
msymbol(S) mcolor(black) ||
       connected birth quarter if married==0, lcolor(gs4) lwidth(thick)
msymbol(D) mcolor(gs4) lpattern(dash) ||
  rcap birthLB birthUB quarter if married==1, lcolor(gs12) ||
  rcap birthLB birthUB quarter if married==0, lcolor(gs12) ||
   connected influenza quarter, yaxis(2) lcolor(red) mcolor(red)
lpattern(solid) msymbol(o) msize(small)
xlabel(1 "Quarter 1" 2 "Quarter 2" 3 "Quarter 3" 4 "Quarter 4")
scheme(s1mono) ylabel(0.2(0.02)0.30, angle(0))
legend(order(1 "Proportion Births (Married)"  5 "Influenza Activity"
             2 "Proportion Births (Unmarried)" 3 "95% CI") size(small))
ytitle("Proportion Births") xtitle("Quarter of Occurrence")
note("Number of married mothers = `NNm'.  Number of unmarried mothers = `NNu'");
graph export "$OUT/births-influenza-quarter.eps", as(eps) replace;
#delimit cr

restore    



********************************************************************************
*** (3e) Calculations for Table A13
********************************************************************************
keep if twin==0&motherAge>=20&motherAge<=45&birthOrder==1

gen class1 = motherAge>=20&motherAge<=45&birthOrder==1&mar_p=="N"
gen class2 = motherAge>=20&motherAge<=45&birthOrder==1&married==1
gen class3 = class1==0&class2==0

count if class1==1
local t1 = round(`r(N)'/9)
count if class2==1
local t2 = round(`r(N)'/9)
count if class3==1
local t3 = round(`r(N)'/9)

foreach bq of numlist 1(1)4 {
    qui count if class1==1&birthQuarter==`bq'
    local n1x`bq'=round(`r(N)'/9)
    local p1x`bq'=`n1x`bq''/`t1'

    qui count if class2==1&birthQuarter==`bq'
    local n2x`bq'=round(`r(N)'/9)
    local r2x`bq'=round(`p1x`bq''*`t2')
    local d2x`bq'=round(`n2x`bq''-`r2x`bq'')

    qui count if class3==1&birthQuarter==`bq'
    local n3x`bq'=round(`r(N)'/9)
    local r3x`bq'=round(`p1x`bq''*`t3')
    local d3x`bq'=round(`n3x`bq''-`r3x`bq'')

    dis "Births c1: `n1x`bq''.  Births c2: `n2x`bq''.  Projected births `r2x`bq''.  Delta births `d2x`bq''"
    dis "Births c1: `n1x`bq''.  Births c2: `n3x`bq''.  Projected births `r3x`bq''.  Delta births `d3x`bq''"
    dis " "
    
}
dis "Totals: `t1', `t2', `t3'"


********************************************************************************
*** (4) Open data for regressions
********************************************************************************
use          "$DAT/nvss2005_2013_all", clear
gen asian = mracerec == 4
gen other = mracerec==3|mracerec==4
lab var asian       "Asian"
lab var asian       "Other Race"
keep if gestation!=.
keep if twin<3
gen expectMonth = conceptionMonth+9
replace expectMonth = expectMonth-12 if expectMonth>12
drop quarter2 quarter3
gen quarter2 = expectMonth==4|expectMonth==5|expectMonth==6
gen quarter3 = expectMonth==7|expectMonth==8|expectMonth==9
lab var quarter2 "Quarter 2"
lab var quarter3 "Quarter 3"

********************************************************************************
*** (5a) Run for quarter 2 and 3, Married Sample [Tables 1, 2]
********************************************************************************
local age motherAge motherAge2
local edu highEd
local c2  WIC underweight overweight obese noART
local yab abs(fips)
local con smoker black white hispanic i.gestation


preserve
keep if twin==1&liveBirth==1&motherAge>=20&motherAge<=45&birthOrder==1&married==1
foreach Q in 2 3 {
    local season Spring
    if `Q'==3 local season Summer
    
    eststo: areg quarter`Q' `age' `edu' `con' _year*, `se' `yab'
    test `age'
    local F1a= string(r(F), "%5.3f")
    local opt1 = round((-_b[motherAge]/(0.02*_b[motherAge2]))*100)/100
    local L1   = string((e(df_r)/2)*(e(N)^(2/e(N))-1), "%5.3f")
    local tL1  = string(sqrt((e(df_r)/1)*(e(N)^(1/e(N))-1)), "%5.3f")
    local pvL  = ttail(e(N),sqrt((e(df_r)/1)*(e(N)^(1/e(N))-1)))*2
    
    eststo: areg quarter`Q' `age'       _year* if e(sample) , `se' `yab'
    test `age'
    local F2a= string(r(F), "%5.3f")
    local opt2 = round((-_b[motherAge]/(0.02*_b[motherAge2]))*100)/100
    
    eststo:  reg quarter`Q' `age'              if e(sample) , `se'
    test `age'
    local F3a= string(r(F), "%5.3f")
    local opt3 = round((-_b[motherAge]/(0.02*_b[motherAge2]))*100)/100
    
    local ysm if year>=2009&ART!=.&WIC!=.&underweight!=.
    eststo: areg quarter`Q' `age' `edu' `con' _year* `ysm', `se' `yab'
    test `age'
    local F4a= string(r(F), "%5.3f")
    local opt4 = round((-_b[motherAge]/(0.02*_b[motherAge2]))*100)/100
    local L4   = string((e(df_r)/2)*(e(N)^(2/e(N))-1), "%5.3f")
    local tL4  = string(sqrt((e(df_r)/1)*(e(N)^(1/e(N))-1)), "%5.3f")
    
    eststo: areg quarter`Q' `age' `edu' `con' `c2' _year* `ysm', `se' `yab'
    test `age'
    local F5a= string(r(F), "%5.3f")
    local opt5 = round((-_b[motherAge]/(0.02*_b[motherAge2]))*100)/100
    
    #delimit ;
    local onote  "`season' Birth Maximizing Age calculates the turning point 
                  of the mother's age quadratic.";
    local not "All singleton, first-born children of married 20-45 year old
    mothers are included.  All births occuring from 2005-2013 are included
    unless otherwise specified in column notes. The omitted baseline race
    in each case is Asian/Native American.
    `Fnote' Leamer critical values refer to Leamer/Schwartz/Deaton 
    critical 5\% values adjusted for sample size. The Leamer critical value 
    for a t-statistic is `tL1' in columns 1-3 and `tL4' in columns 4 and 5.
    `onote' `enote' $^{\ddagger}$ Significance based on Leamer criterion at 5\%.";

    esttab est3 est2 est1 est4 est5 using "$OUT/NVSSBinaryQ`Q'_Married.tex",
    replace `estopt' keep(`age' `edu' smoker black white hispanic `c2') 
    title("Season of Birth Correlates: Quarter `Q' (Married Mothers, 20--45)"
          \label{Q`Q'-Married})
    style(tex) mlabels(, depvar) booktabs 
    starlevel ("$ ^{\ddagger} $" `pvL')
    postfoot("F-test of Age Variables  &`F3a'&`F2a'&`F1a'&`F4a'&`F5a' \\ "
             "Leamer Critical Value (F)&`L1'&`L1'&`L1'&`L4'&`L4' \\      "
             "`season' Birth Maximizing Age &`opt3'&`opt2'&`opt1'&`opt4'&`opt5' \\         "
             "State and Year FE&&Y&Y&Y&Y\\ Gestation FE &&&Y&Y&Y\\       "
             "2009-2013 Only&&&&Y&Y\\ \bottomrule                        "
             "\multicolumn{6}{p{16.2cm}}{\begin{footnotesize} `not'"
             "\end{footnotesize}}\end{tabular}\end{table}");
    #delimit cr
    estimates clear
}
restore


********************************************************************************
*** (5b) Run for quarter 2 and 3, Unmarried no paternity Sample [Tables A8, A9]
********************************************************************************
preserve
keep if twin==1&liveBirth==1&motherAge>=20&motherAge<=45&birthOrder==1&mar_p=="N"
count
    
foreach Q in 2 3 {
    local season Spring
    if `Q'==3 local season Summer
    eststo: areg quarter`Q' `age' `edu' `con' _year*, `se' `yab'
    test `age'
    local F1a= string(r(F), "%5.3f")
    local opt1 = round((-_b[motherAge]/(0.02*_b[motherAge2]))*100)/100
    local L1   = string((e(df_r)/2)*(e(N)^(2/e(N))-1), "%5.3f")
    local tL1  = string(sqrt((e(df_r)/1)*(e(N)^(1/e(N))-1)), "%5.3f")
    local pvL  = ttail(e(N),sqrt((e(df_r)/1)*(e(N)^(1/e(N))-1)))*2

    eststo: areg quarter`Q' `age'       _year* if e(sample) , `se' `yab'
    test `age'
    local F2a= string(r(F), "%5.3f")
    local opt2 = round((-_b[motherAge]/(0.02*_b[motherAge2]))*100)/100
    
    eststo:  reg quarter`Q' `age'              if e(sample) , `se'
    test `age'
    local F3a= string(r(F), "%5.3f")
    local opt3 = round((-_b[motherAge]/(0.02*_b[motherAge2]))*100)/100
    
    local ysm if year>=2009&ART!=.&WIC!=.&underweight!=.
    eststo: areg quarter`Q' `age' `edu' `con' _year* `ysm', `se' `yab'
    test `age'
    local F4a= string(r(F), "%5.3f")
    local opt4 = round((-_b[motherAge]/(0.02*_b[motherAge2]))*100)/100
    local L4   = string((e(df_r)/2)*(e(N)^(2/e(N))-1), "%5.3f")
    local tL4  = string(sqrt((e(df_r)/1)*(e(N)^(1/e(N))-1)), "%5.3f")
    
    eststo: areg quarter`Q' `age' `edu' `con' `c2' _year* `ysm', `se' `yab'
    test `age'
    local F5a= string(r(F), "%5.3f")
    local opt5 = round((-_b[motherAge]/(0.02*_b[motherAge2]))*100)/100

    #delimit ;
    local onote  "`season' Birth Maximizing Age calculates the turning point 
                  of the mother's age quadratic.";
    local not "All singleton, first-born children born to unmarried 20-45
    year-old mothers without paternity acknowledgement on the child's birth
    certificate are included.  All births occuring from 2005-2013 are included
    unless otherwise specified in column notes. The omitted baseline race
    in each case is Asian/Native American.
    `Fnote' Leamer critical values refer to Leamer/Schwartz/Deaton 
    critical 5\% values adjusted for sample size. The Leamer critical value 
    for a t-statistic is `tL1' in columns 1-3 and `tL4' in columns 4 and 5.
    `onote' `enote' $^{\ddagger}$ Significance based on Leamer criterion at 5\%.";

    esttab est3 est2 est1 est4 est5 using "$OUT/NVSSBinaryQ`Q'_noPaternity.tex",
    replace `estopt' keep(`age' `edu' smoker black white hispanic `c2') 
    title("Season of Birth Correlates: Quarter `Q'
           (Unmarried Mothers with no Paternity Acknowledgement, 20--45)"
          \label{Q`Q'-noPaternity})
    style(tex) mlabels(, depvar) booktabs 
    starlevel ("$ ^{\ddagger} $" `pvL')
    postfoot("F-test of Age Variables  &`F3a'&`F2a'&`F1a'&`F4a'&`F5a' \\ "
             "Leamer Critical Value (F)&`L1'&`L1'&`L1'&`L4'&`L4' \\      "
             "`season' Birth Maximizing Age &`opt3'&`opt2'&`opt1'&`opt4'&`opt5' \\         "
             "State and Year FE&&Y&Y&Y&Y\\ Gestation FE &&&Y&Y&Y\\       "
             "2009-2013 Only&&&&Y&Y\\ \bottomrule                        "
             "\multicolumn{6}{p{16.2cm}}{\begin{footnotesize} `not'"
             "\end{footnotesize}}\end{tabular}\end{table}");
    #delimit cr
    estimates clear
}
restore

********************************************************************************
*** (5c) Run for quarter 2 and 3: Second births
********************************************************************************
local age motherAge motherAge2
local edu highEd
local c2  WIC underweight overweight obese noART
local yab abs(fips)
local con smoker i.gestation black white hispanic

preserve
keep if twin==1&liveBirth==1&motherAge>=20&motherAge<=45&birthOrder==2&married==1
foreach Q in 2 3 {
    local season Spring
    if `Q'==3 local season Summer
    eststo: areg quarter`Q' `age' `edu' `con' _year*, `se' `yab'
    test `age'
    local F1a= string(r(F), "%5.3f")
    local opt1 = round((-_b[motherAge]/(0.02*_b[motherAge2]))*100)/100
    local L1   = string((e(df_r)/2)*(e(N)^(2/e(N))-1), "%5.3f")
    local tL1  = string(sqrt((e(df_r)/1)*(e(N)^(1/e(N))-1)), "%5.3f")
    local pvL  = ttail(e(N),sqrt((e(df_r)/1)*(e(N)^(1/e(N))-1)))*2
    
    eststo: areg quarter`Q' `age'       _year* if e(sample) , `se' `yab'
    test `age'
    local F2a= string(r(F), "%5.3f")
    local opt2 = round((-_b[motherAge]/(0.02*_b[motherAge2]))*100)/100
    
    eststo:  reg quarter`Q' `age'              if e(sample) , `se'
    test `age'
    local F3a= string(r(F), "%5.3f")
    local opt3 = round((-_b[motherAge]/(0.02*_b[motherAge2]))*100)/100
    
    local ysm if year>=2009&ART!=.&WIC!=.&underweight!=.
    eststo: areg quarter`Q' `age' `edu' `con' _year* `ysm', `se' `yab'
    test `age'
    local F4a= string(r(F), "%5.3f")
    local opt4 = round((-_b[motherAge]/(0.02*_b[motherAge2]))*100)/100
    local L4   = string((e(df_r)/2)*(e(N)^(2/e(N))-1), "%5.3f")
    local tL4  = string(sqrt((e(df_r)/1)*(e(N)^(1/e(N))-1)), "%5.3f")
    
    eststo: areg quarter`Q' `age' `edu' `con' `c2' _year* `ysm', `se' `yab'
    test `age'
    local F5a= string(r(F), "%5.3f")
    local opt5 = round((-_b[motherAge]/(0.02*_b[motherAge2]))*100)/100
    
    #delimit ;
    local onote  "`season' Birth Maximizing Age calculates the turning point 
                  of the mother's age quadratic.";
    local not "All singleton, second-born children of married 20-45 year old
    mothers are included.  All births occuring from 2005-2013 are included
    unless otherwise specified in column notes. The omitted baseline race
    in each case is Asian/Native American.
    `Fnote' Leamer critical values refer to Leamer/Schwartz/Deaton 
    critical 5\% values adjusted for sample size. The Leamer critical value 
    for a t-statistic is `tL1' in columns 1-3 and `tL4' in columns 4 and 5.
    `onote' `enote' $^{\ddagger}$ Significance based on Leamer criterion at 5\%.";

    esttab est3 est2 est1 est4 est5 using "$OUT/NVSSBinaryQ`Q'_Married-b2.tex",
    replace `estopt' keep(`age' `edu' smoker black white hispanic `c2') 
    title("Season of Birth Correlates for Second Births: Quarter `Q'
          (Married Mothers, 20--45)" \label{Q`Q'-Married-b2})
    style(tex) mlabels(, depvar) booktabs 
    starlevel ("$ ^{\ddagger} $" `pvL')
    postfoot("F-test of Age Variables  &`F3a'&`F2a'&`F1a'&`F4a'&`F5a' \\ "
             "Leamer Critical Value (F)&`L1'&`L1'&`L1'&`L4'&`L4' \\      "
             "`season' Birth Maximizing Age &`opt3'&`opt2'&`opt1'&`opt4'&`opt5' \\         "
             "State and Year FE&&Y&Y&Y&Y\\ Gestation FE &&&Y&Y&Y\\       "
             "2009-2013 Only&&&&Y&Y\\ \bottomrule                        "
             "\multicolumn{6}{p{16.2cm}}{\begin{footnotesize} `not'"
             "\end{footnotesize}}\end{tabular}\end{table}");
    #delimit cr
    estimates clear
}
restore

********************************************************************************
*** (5d) Alternative Regressions [Table A2, A6 (top block), A5 (bottom block)]
********************************************************************************
#delimit ;
local add `"
    "with state-specific linear trends and unemployment rate at conception" "';
local nam  StateT;
#delimit cr
tokenize `nam'

local k=1
foreach type of local add {
    if `k'==1 local gg twin==1&liveBirth==1&birthOrder==1&married==1
    
    local c3
    if `k'==1 local c3  i.fips#c.year value
    local con smoker i.gestation black white hispanic
    preserve
    keep if motherAge>=20&motherAge<=45&`gg'
    
    foreach Q in 2 3 {    
        local season Spring
        if `Q'==3 local season Summer
        eststo: areg quarter`Q' `age' `edu' `con' _year* `c3', `se' `yab'
        test `age'
        local F1a= string(r(F), "%5.3f")
        local opt1 = round((-_b[motherAge]/(0.02*_b[motherAge2]))*100)/100
        local L1   = string((e(df_r)/2)*(e(N)^(2/e(N))-1), "%5.3f")
        local tL1  = string(sqrt((e(df_r)/1)*(e(N)^(1/e(N))-1)), "%5.3f")
        local pvL  = ttail(e(N),sqrt((e(df_r)/1)*(e(N)^(1/e(N))-1)))*2

        eststo: areg quarter`Q' `age' _year*  `c3' if e(sample) , `se' `yab'
        test `age'
        local F2a= string(r(F), "%5.3f")
        local opt2 = round((-_b[motherAge]/(0.02*_b[motherAge2]))*100)/100

        eststo:  reg quarter`Q' `age'              if e(sample) , `se'
        test `age'
        local F3a= string(r(F), "%5.3f")
        local opt3 = round((-_b[motherAge]/(0.02*_b[motherAge2]))*100)/100

        local ysm if year>=2009&ART!=.&WIC!=.&underweight!=.
        eststo: areg quarter`Q' `age' `edu' `con' _year* `c3' `ysm', `se' `yab'
        test `age'
        local F4a= string(r(F), "%5.3f")
        local opt4 = round((-_b[motherAge]/(0.02*_b[motherAge2]))*100)/100
        local L4   = string((e(df_r)/2)*(e(N)^(2/e(N))-1), "%5.3f")
        local tL4  = string(sqrt((e(df_r)/1)*(e(N)^(1/e(N))-1)), "%5.3f")

        eststo: areg quarter`Q' `age' `edu' `con' `c2' `c3' _year* `ysm', `se' `yab'
        test `age'
        local F5a= string(r(F), "%5.3f")
        local opt5 = round((-_b[motherAge]/(0.02*_b[motherAge2]))*100)/100

        #delimit ;
        local onote  "`season' Birth Maximizing Age calculates the turning point 
                      of the mother's age quadratic.";
        local not "All singleton first births occurring to married women aged 20-45
        are included. All births occuring from 2005-2013 are included unless otherwise
        specified in column notes. The omitted baseline race in each case is Asian/Native
        American. `Fnote' Leamer critical values refer
        to Leamer/Schwartz/Deaton critical 5\% values adjusted for sample
        size. The Leamer critical value for a t-statistic is `tL1' in
        columns 1-3 and `tL4' in columns 4 and 5.  `onote' `enote'
        $^{\ddagger}$ Significance based on Leamer criterion at 5\%.";

        esttab est3 est2 est1 est4 est5 using "$OUT/NVSSBinaryQ`Q'_``k''.tex",
        replace `estopt' keep(`age' `edu' smoker black white hispanic `c2') 
        title("Season of Birth Correlates `type': Quarter `Q' (Married Mothers, 20--45)"
              \label{Q`Q'-``k''}) style(tex) mlabels(, depvar) booktabs 
        starlevel ("$ ^{\ddagger} $" `pvL')
        postfoot("F-test of Age Variables  &`F3a'&`F2a'&`F1a'&`F4a'&`F5a' \\ "
                 "Leamer Critical Value (F)&`L1'&`L1'&`L1'&`L4'&`L4' \\      "
                 "`season' Birth Maximizing Age &`opt3'&`opt2'&`opt1'&`opt4'&`opt5' \\         "
                 "State and Year FE&&Y&Y&Y&Y\\ Gestation FE &&&Y&Y&Y\\       "
                 "2009-2013 Only&&&&Y&Y\\ \bottomrule                        "
                 "\multicolumn{6}{p{16.2cm}}{\begin{footnotesize} `not'"
                 "\end{footnotesize}}\end{tabular}\end{table}");
        #delimit cr
        estimates clear
    }
    restore
    local ++k
}


#delimit ;
local add `" " Excluding December Conceptions" "';
local nam NoDec;
#delimit cr
tokenize `nam'

local k=1
foreach type of local add {
    if `k'==1 local gg twin==1&liveBirth==1&birthOrder==1&expectMonth!=9

    local c3
    local con smoker i.gestation black white hispanic
    preserve
    keep if motherAge>=20&motherAge<=45&married==1&`gg'
    count
    
    foreach Q in 3 {    
        local season Spring
        if `Q'==3 local season Summer
        eststo: areg quarter`Q' `age' `edu' `con' _year* `c3', `se' `yab'
        test `age'
        local F1a= string(r(F), "%5.3f")
        local opt1 = round((-_b[motherAge]/(0.02*_b[motherAge2]))*100)/100
        local L1   = string((e(df_r)/2)*(e(N)^(2/e(N))-1), "%5.3f")
        local tL1  = string(sqrt((e(df_r)/1)*(e(N)^(1/e(N))-1)), "%5.3f")
        local pvL  = ttail(e(N),sqrt((e(df_r)/1)*(e(N)^(1/e(N))-1)))*2

        eststo: areg quarter`Q' `age' _year*  `c3' if e(sample) , `se' `yab'
        test `age'
        local F2a= string(r(F), "%5.3f")
        local opt2 = round((-_b[motherAge]/(0.02*_b[motherAge2]))*100)/100

        eststo:  reg quarter`Q' `age'              if e(sample) , `se'
        test `age'
        local F3a= string(r(F), "%5.3f")
        local opt3 = round((-_b[motherAge]/(0.02*_b[motherAge2]))*100)/100

        local ysm if year>=2009&ART!=.&WIC!=.&underweight!=.
        eststo: areg quarter`Q' `age' `edu' `con' _year* `c3' `ysm', `se' `yab'
        test `age'
        local F4a= string(r(F), "%5.3f")
        local opt4 = round((-_b[motherAge]/(0.02*_b[motherAge2]))*100)/100
        local L4   = string((e(df_r)/2)*(e(N)^(2/e(N))-1), "%5.3f")
        local tL4  = string(sqrt((e(df_r)/1)*(e(N)^(1/e(N))-1)), "%5.3f")

        eststo: areg quarter`Q' `age' `edu' `con' `c2' `c3' _year* `ysm', `se' `yab'
        test `age'
        local F5a= string(r(F), "%5.3f")
        local opt5 = round((-_b[motherAge]/(0.02*_b[motherAge2]))*100)/100

        eststo: areg quarter`Q' motherAge `edu' `con' `c2' `c3' _year* `ysm', `se' `yab'
        test motherAge
        local F6a= string(sqrt(r(F)), "%5.3f")

        #delimit ;
        local onote  "`season' Birth Maximizing Age calculates the turning point 
                      of the mother's age quadratic.";
        local not "All singleton, first births occurring to married
        women aged 20-45 are included. All births occuring from 2005-2013
        are included unless otherwise specified in column notes. `Fnote'
        Leamer critical values refer to Leamer/Schwartz/Deaton critical 5\%
        values adjusted for sample size. The Leamer critical value for a
        t-statistic is `tL1' in columns 1-3 and `tL4' in columns 4 and 5.
        `onote' `enote' $^{\ddagger}$ Significance based on Leamer criterion at 5\%.";

        esttab est3 est2 est1 est4 est5 est6 using "$OUT/NVSSLinAgeQ`Q'_``k''.tex",
        replace `estopt' keep(`age' `edu' smoker black white hispanic `c2') 
        title("Season of Birth Correlates`type': Quarter `Q'
               (Married Mothers, 20--45)" \label{LinQ`Q'-``k''}) 
        style(tex) mlabels(, depvar) booktabs 
        starlevel ("$ ^{\ddagger} $" `pvL')
        postfoot("F-test of Age Variables  &`F3a'&`F2a'&`F1a'&`F4a'&`F5a'&`F6a' \\ "
                 "Leamer Critical Value (F)&`L1'&`L1'&`L1'&`L4'&`L4'&`tL4' \\     "
                 "`season' Birth Maximizing Age &`opt3'&`opt2'&`opt1'&`opt4'&`opt5'&-- \\ "
                 "State and Year FE&&Y&Y&Y&Y&Y\\ Gestation FE &&&Y&Y&Y&Y\\       "
                 "2009-2013 Only&&&&Y&Y&Y\\ \bottomrule                        "
                 "\multicolumn{7}{p{18.3cm}}{\begin{footnotesize} `not'"
                 "\end{footnotesize}}\end{tabular}\end{table}");
        #delimit cr
        estimates clear
    }
    restore
    local ++k
}

********************************************************************************
*** (6) Logit regressions
********************************************************************************
local age motherAge motherAge2
local edu highEd
local c2  WIC underweight overweight obese noART
local se
local nc


local con smoker i.gestation black white hispanic
preserve
keep if twin==1&liveBirth==1&birthOrder==1&motherAge>=20&motherAge<=45&married==1

foreach Q in 2 3 {
    local season Spring
    if `Q'==3 local season Summer
    logit quarter`Q' `age' `edu' `con' _year* i.fips, `se' 
    test `age'
    local F1a= string(r(chi2), "%5.3f")
    local opt1 = round((-_b[motherAge]/(0.02*_b[motherAge2]))*100)/100
    local rdf  = e(N)-e(rank)
    local L1   = string(2*((`rdf'/2)*(e(N)^(2/e(N))-1)), "%5.3f")
    local tL1  = string(sqrt((`rdf'/1)*(e(N)^(1/e(N))-1)), "%5.3f")
    local pvL  = ttail(e(N),sqrt((`rdf'/1)*(e(N)^(1/e(N))-1)))*2
    margins, dydx(`age' `edu' smoker black white hispanic `nc') post
    estimates store m1
    
    logit quarter`Q' `age'       _year* i.fips if e(sample) , `se' 
    test `age'
    local F2a= string(r(chi2), "%5.3f")
    local opt2 = round((-_b[motherAge]/(0.02*_b[motherAge2]))*100)/100
    margins, dydx(`age') post
    estimates store m2
    
    logit quarter`Q' `age'              if e(sample) , `se'
    test `age'
    local F3a= string(r(chi2), "%5.3f")
    local opt3 = round((-_b[motherAge]/(0.02*_b[motherAge2]))*100)/100
    margins, dydx(`age') post
    estimates store m3
    
    local ysm if year>=2009&ART!=.&WIC!=.&underweight!=.
    logit quarter`Q' `age' `edu' `con' _year* i.fips `ysm', `se' 
    test `age'
    local F4a= string(r(chi2), "%5.3f")
    local opt4 = round((-_b[motherAge]/(0.02*_b[motherAge2]))*100)/100
    local rdf  = e(N)-e(rank)
    local L4   = string(2*((`rdf'/2)*(e(N)^(2/e(N))-1)), "%5.3f")
    local tL4  = string(sqrt((`rdf'/1)*(e(N)^(1/e(N))-1)), "%5.3f")
    margins, dydx(`age' `edu' smoker black white hispanic `nc') post
    estimates store m4
    
    logit quarter`Q' `age' `edu' `con' `c2' _year* i.fips `ysm', `se'
    test `age'
    local F5a= string(r(chi2), "%5.3f")
    local opt5 = round((-_b[motherAge]/(0.02*_b[motherAge2]))*100)/100
    margins, dydx(`age' `edu' smoker black white hispanic `nc' `c2') post
    estimates store m5

    local Anot
    #delimit ;
    local onote  "`season' Birth Maximizing Age calculates the turning point 
                  of the mother's age quadratic.";
    if `Q'==2 local Anot "A small number of observations are removed from the
    regression sample as there is no variation in quarter 2 births (all births
    occur in quarters 1, 3 and 4) for a specific state by year cell,
    conditional on controls and fixed effects.  In total 0.03\% of observations
    are removed from the sample in order to estimate the logit specification.";
    local not "`Anot' Average marginal effects of logit parameters are reported.
    All singleton, first born children of married mothers are included.
    All births occuring from 2005-2013 are included unless otherwise specified
    in column notes. `Xnote' Leamer critical values refer to Leamer/Schwartz/Deaton 
    critical 5\% values adjusted for sample size. The Leamer critical value 
    for a t-statistic is `tL1' in columns 1-3 and `tL4' in columns 4 and 5.
   `onote' `enote' $^{\ddagger}$ Significance based on Leamer criterion at 5\%.";

    esttab m3 m2 m1 m4 m5 using "$OUT/NVSSLogitQ`Q'_Married.tex", margin
    replace `estopt' keep(`age' `edu' smoker black white hispanic `c2' `nc') style(tex) 
    title("Season of Birth Correlates Logit: Quarter `Q' (Married Mothers, 20--45)"
          \label{LogQ`Q'-Married}) 
    mlabels("Quarter `Q'" "Quarter `Q'" "Quarter `Q'" "Quarter `Q'" "Quarter `Q'")
    starlevel ("$ ^{\ddagger} $" `pvL') booktabs 
    postfoot("$ \chi^2$ test of Age Variables  &`F3a'&`F2a'&`F1a'&`F4a'&`F5a' \\ "
             "Leamer Critical Value (Age)&`L1'&`L1'&`L1'&`L4'&`L4' \\      "
             "`season' Birth Maximizing Age &`opt3'&`opt2'&`opt1'&`opt4'&`opt5' \\         "
             "State and Year FE&&Y&Y&Y&Y\\ Gestation FE &&&Y&Y&Y\\       "
             "2009-2013 Only&&&&Y&Y\\ \bottomrule                        "
             "\multicolumn{6}{p{16.2cm}}{\begin{footnotesize} `not'"
             "\end{footnotesize}}\end{tabular}\end{table}");
    #delimit cr
    estimates clear
}
restore



********************************************************************************
*** (7) Including fetal deaths
********************************************************************************
use          "$DAT/nvss2005_2013_all", clear
append using "$DAT/nvssFD2005_2013_all"
keep if twin==1 & motherAge>=20 & motherAge <= 45 & birthOrder==1
gen state2 = subinstr(state," ", "", .)
bys state: egen minTemp = mean(cold)
replace motherAge2 = motherAge2/100 if liveBirth==0

keep if liveBirth==0|(liveBirth==1&highEd!=.&smoker!=.&gestation!=.)
keep if twin<3

gen expectMonth = conceptionMonth+9
replace expectMonth = expectMonth-12 if expectMonth>12
drop quarter2 quarter3
gen quarter2 = expectMonth==4|expectMonth==5|expectMonth==6
gen quarter3 = expectMonth==7|expectMonth==8|expectMonth==9
lab var quarter2 "Quarter 2"
lab var quarter3 "Quarter 3"
gen births = liveBirth==1
gen fetalDeath = liveBirth==0

#delimit ;
local add `" "20-45 Married"
             "20-45 Unmarried" "';
local nam  Married Unmarried;
#delimit cr
tokenize `nam'


local k=1
foreach type of local add {
    if `k'==1 local gg motherAge>=20&motherAge<=45&married==1
    if `k'==2 local gg motherAge>=20&motherAge<=45&married==0
    local min = 0.002
    local max = 0.0034
    local del = 0.0002
    if `k'==2     local min = 0.0058
    if `k'==2     local del = 0.0004
    if `k'==2     local max = 0.0088

    
    preserve
    keep if `gg'
    count
    local NN = r(N)
    local NN = string(`NN',"%12.0gc")
    count if fetalDeath==1
    local fN = r(N)
    collapse (sum) births fetalDeath, by(birthMonth)
    egen totalFdeath = total(fetalDeath)
    egen totalBirths = total(births)
    gen  total = births+fetalDeath
    gen  birthsTotal = births/total
    gen  fetalDeathTotal = fetalDeath/total
    replace fetalDeath = fetalDeath/totalFdeath

    #delimit ;
    twoway line fetalDeathTotal birthMonth, lcolor(black) lwidth(thick) scheme(s1mono)
    ylabel(`min'(`del')`max')
    ytitle("Fetal Deaths/(Fetal Deaths+Births)") xtitle("Month of Occurrence")
    xlabel(1 "Jan" 2 "Feb" 3 "Mar" 4 "Apr" 5 "May" 6 "Jun" 7 "Jul" 8 "Aug"
           9 "Sep" 10 "Oct" 11 "Nov" 12 "Dec") note("Number of observations=`NN'"); 
    graph export "$OUT/birthsFdeaths-``k''.eps", as(eps) replace;
    #delimit cr
    restore
    
    local ++k
}


********************************************************************************
*** (8) Clear
********************************************************************************
log close
