/* mturkAnalysis.do v0.00          DC/SO/CQD               yyyy-mm-dd:2018-10-07
----|----1----|----2----|----3----|----4----|----5----|----6----|----7----|----8

This script undertakes all analysis using Mechanical Turk surveys implemented in
2016 and 2018.  In order to replicate these results, simply change the global
"MAIN" below to the location of the "replication" directory on your computer.

Refer to the README in the main directory for additional details.
*/

vers 11
clear all
set more off
cap log close



*-------------------------------------------------------------------------------
*--- (0) Globals, Locals, Setup (check for required external ado)
*-------------------------------------------------------------------------------
global MAIN "/home/replication"

global DAT "$MAIN/data/MTurk"
global LOG "$MAIN/log"
global OUT "$MAIN/results/MTurk"
global NVS "$MAIN/data/NVSS"
global ACS "$MAIN/data/ACS"
global GEO "$MAIN/data/auxiliary"

#delimit ;
local snames `" "Alabama" "Alaska" "Arizona" "Arkansas" "California"
"Colorado" "Connecticut" "Delaware" "District of Columbia" "Florida" "Georgia"
"Idaho" "Illinois" "Indiana" "Iowa" "Kansas" "Kentucky" "Louisiana" "Maine"
"Maryland" "Massachusetts" "Michigan" "Minnesota" "Mississippi" "Missouri"
"Montana" "Nebraska" "Nevada" "New Hampshire" "New Jersey" "New Mexico"
"New York" "North Carolina" "North Dakota" "Ohio" "Oklahoma" "Oregon"
"Pennsylvania" "Rhode Island" "South Carolina" "South Dakota" "Tennessee"
"Texas" "Utah" "Virginia" "Washington" "West Virginia" "Wisconsin" "Hawaii"
"Vermont" "Wyoming" "';
local sprop 151 23 212 93 1218 170 112 29 21 631 318 51 400 206 97 91 138 145
41 187 211 309 171 93 189 32 59 90 41 279 65 616 312 24 361 122 125 398 33 152
27 205 855 93 261 223 57 180 45 19 18;
#delimit cr

log using "$LOG/mturkAnalysis.txt", text replace


foreach ado in estout spmap mixlogit {
     ssc install `ado', replace
}

*-------------------------------------------------------------------------------
*--- (1) Append and Generate Variables
*-------------------------------------------------------------------------------
insheet using "$DAT/mturk_ANONYMYSED_JITTERED.csv", clear names comma case
saveold "$DAT/mturk_ANONYMYSED_JITTERED.dta", replace version(11)

*DOB and BWT replace missings as separate indicator
replace dob        ="missing" if dob        ==""
replace birthweight="missing" if birthweight==""
replace birthweight_p=5 if birthweight_p==.
replace dob_position =5 if dob_position ==.

tab gender     , gen(_gend)
tab cost       , gen(_cost)
tab birthweight, gen(_bwt)
tab sob        , gen(_sob)
tab dob        , gen(_dob)

drop _gend1 _cost5 _bwt2 _sob4 _dob1
rename _cost1 _costx
rename _cost4 _cost1
rename _cost3 _cost4
rename _costx _cost3
rename _cost2 _cost11
rename _cost10 _cost2
rename _cost11 _cost10
rename _bwt1 _bwt2
rename _bwt4 _bwtx
rename _bwt5 _bwt4
rename _bwt3 _bwt5
rename _bwtx _bwt3
rename _bwt6 _bwtx
rename _bwt7 _bwt6
rename _bwt8 _bwt7
rename _bwtx _bwt8
rename _bwt9 _bwtx
rename _bwt10 _bwt9
rename _bwt11 _bwt10
rename _bwtx _bwt11
rename _sob1 _sob4
gen goodSeason=_sob2==1|_sob3==1
gen     costNumerical = subinstr(cost,"$","",1)
replace costNumerical = subinstr(costNumerical,",","",1)
destring costNumerical, replace
replace costNumerical = costNumerical/1000
gen spring = _sob2
gen summer = _sob3
gen all = 1
lab var age "Age"
lab var age2 "Age Squared"
lab var someCollege "Some College +"
lab var hispanic "Hispanic"
lab var teacher "Teacher"
lab var parent  "Parent"
lab var _bwt2  "5lbs, 13oz"
lab var _bwt3  "6lbs, 3oz"
lab var _bwt4  "6lbs, 8oz"
lab var _bwt5  "6lbs, 13oz"
lab var _bwt6  "7lbs, 3oz"
lab var _bwt7  "7lbs, 8oz"
lab var _bwt8  "7lbs, 13oz"
lab var _bwt9  "8lbs, 3oz"
lab var _bwt10 "8lbs, 8oz"
lab var _bwt11 "8lbs, 13oz"
lab var _dob2  "Weekend Day"
lab var _gend2 "Girl"

local oFEs i.round i.option i.surveyV

bys ID: gen N=_n

bys RespState: gen statePop = _N
count
gen surveyProportion = statePop/r(N)
gen censusProportion = .
tokenize `sprop'
local total = 0
foreach state of local snames {
    dis "State: `state', pop: `1'"
    qui replace censusProportion = `1' if RespState=="`state'"
    local total = `total'+`1'
    macro shift
}
dis `total'
replace censusProportion = censusProportion/10000
gen weight = surveyProportion/censusProportion
replace weight=1/weight

*-------------------------------------------------------------------------------
*-- (2) Summary Statistics [TABLE A18 AND FIGURE A7]
*-------------------------------------------------------------------------------
preserve
keep if N==1

lab var sex       "Female"
lab var birthyr   "Year of Birth"
lab var age       "Age"
lab var educY     "Years of Education"
lab var nchild    "Number of Children"
lab var pregnant1 "Currently Pregnant"
lab var married   "Married"
lab var black     "Black"
lab var white     "White"
lab var otherRac  "Other Race (Asian/Native American)"
lab var hispanic  "Hispanic"
lab var employed  "Employed"
lab var unemploy  "Unemployed"
lab var highEduc  "Some College +"
lab var parent    "Parent"
lab var teacher   "Education, Training, and Library occupation"
lab var ftotinc   "Total Family Income (1000s)"
lab var mturkSal  "Hourly earnings on MTurk"
#delimit ;
estpost sum sex age black white otherRace hispanic married highEduc educY
employed ftotinc teacher parent nchild mturkSal;

estout using "$OUT/MTurkSum.tex", replace label style(tex)
cells("count(fmt(%7.0gc) label(N)) mean(fmt(2) label(Mean))
sd(fmt(2) label(Std.\ Dev.)) min(fmt(2) label(Min)) max(fmt(2) label(Max))");
#delimit cr
restore


gen statename=RespState
count
bys statename: gen stateProportion = _N/r(N)
preserve
collapse stateProportion, by(statename)
rename statename NAME
merge 1:1 NAME using "$GEO/US_db"
format stateProportion %5.3f
#delimit ;
spmap stateProportion if NAME!="Alaska"&NAME!="Hawaii"&NAME!="Puerto Rico"
using "$GEO/US_coord_mercator",
point(data($DAT/mturk_ANONYMYSED_JITTERED) xcoord(long3) ycoord(lat3)
      select(drop if (latitude<24.39|latitude>49.38)|(longitude<-124.84|longitude>-66.9))
      size(*0.5) fcolor(red))
id(_ID) osize(thin) legtitle("Proportion of Respondents") legstyle(2) fcolor(Greens)
legend(symy(*1.2) symx(*1.2) size(*1.4) rowgap(1));
graph export "$OUT/surveyCoverage.eps", as(eps) replace;
#delimit cr
restore

preserve
insheet using "$GEO/population2015.csv", delim(";") names clear
replace state=subinstr(state,".","",1)
rename state NAME
merge 1:1 NAME using "$GEO/US_db"
format proportion %5.3f
#delimit ;
spmap proportion if NAME!="Alaska"&NAME!="Hawaii"&NAME!="Puerto Rico"
using "$GEO/US_coord_mercator", id(_ID) osize(thin)
legtitle("Proportion of Respondents (Census)") legstyle(2) fcolor(Greens)
legend(symy(*1.2) symx(*1.2) size(*1.4) rowgap(1));
graph export "$OUT/usaCoverage.eps", as(eps) replace;
#delimit cr
restore


replace age    = childBYear-RespYOB if parent==1
gen osample = RespSex=="Female"&married==1&parent==1&age>=20&age<=45

*-------------------------------------------------------------------------------
*--- (3A) Comparison with NVSS [TABLE A19: PANEL A]
*-------------------------------------------------------------------------------
preserve
gen ageBirth=age
gen race=11 if white==1
gen marst=married
gen cbirthmonth     = 1  if RespKidBMonth=="January"
replace cbirthmonth = 2  if RespKidBMonth=="February"
replace cbirthmonth = 3  if RespKidBMonth=="March"
replace cbirthmonth = 4  if RespKidBMonth=="April"
replace cbirthmonth = 5  if RespKidBMonth=="May"
replace cbirthmonth = 6  if RespKidBMonth=="June"
replace cbirthmonth = 7  if RespKidBMonth=="July"
replace cbirthmonth = 8  if RespKidBMonth=="August"
replace cbirthmonth = 9  if RespKidBMonth=="September"
replace cbirthmonth = 10 if RespKidBMonth=="October"
replace cbirthmonth = 11 if RespKidBMonth=="November"
replace cbirthmonth = 12 if RespKidBMonth=="December"
keep  if ageBirth>=25&ageBirth<=45&sex==1
keep if nchild!=0
keep if N==1
gen Q1 = cbirthmonth >= 1 & cbirthmonth <=3
gen Q2 = cbirthmonth >= 4 & cbirthmonth <=6
gen Q3 = cbirthmonth >= 7 & cbirthmonth <=9
gen Q4 = cbirthmonth >=10 & cbirthmonth <=12
gen     sexchild = 1 if RespKidGender=="Girl"
replace sexchild = 0 if RespKidGender=="Boy"
#delimit ;
collapse (sum) N (mean) nchild sexchild Q1 Q2 Q3 Q4 ageBirth hispanic
  black white otherRace highEduc married (sd) sd_nchild=nchild sd_sexchild=sexchild
  sd_Q1=Q1 sd_Q2=Q2 sd_Q3=Q3 sd_Q4=Q4 sd_ageBirth=ageBirth sd_black=black
  sd_hispanic=hispanic sd_white=white sd_otherRace=otherRace sd_highEduc=highEduc
  sd_married=married;
#delimit cr

expand 13
gen mean  = .
gen stdev = .
gen var   = ""
local i = 1
#delimit ;
local comvars nchild sexchild Q1 Q2 Q3 Q4 ageBirth black white
              otherRace hispanic highEduc married;
#delimit cr
foreach var of varlist `comvars' {
    replace mean  = `var' in `i'
    replace stdev = sd_`var' in `i'
    replace var = "`var'" in `i'
    local ++i
}
gen data = "MTurk"
keep mean stdev var data N
tempfile MTurkSum
save `MTurkSum'
restore
preserve
insheet using "$NVS/natl2013.csv", comma names clear
gen oneChild = lbo_rec==1
sum oneChild

gen N_NV = 1
gen nchild = lbo_rec
replace nchild = 6 if nchild>6&nchild<20
replace nchild = . if nchild>=20
gen sexchild = sex=="F"
gen Q1 = dob_mm >= 1 & dob_mm <=3
gen Q2 = dob_mm >= 4 & dob_mm <=6
gen Q3 = dob_mm >= 7 & dob_mm <=9
gen Q4 = dob_mm >=10 & dob_mm <=12
gen ageBirth = mager
gen highEduc = meduc>=4 if meduc!=9&meduc!=.
gen hispanic = umhisp!=0
gen black    = mracerec==2
gen white    = mracerec==1
gen otherRace= mracerec==3|mracerec==4
gen married  = mar==1
#delimit ;
collapse (sum) N (mean) nchild sexchild Q1 Q2 Q3 Q4 ageBirth highEduc hispanic
  black white otherRace married (sd) sd_nchild=nchild sd_sexchild=sexchild sd_Q1=Q1
  sd_Q2=Q2 sd_Q3=Q3 sd_Q4=Q4 sd_ageBirth=ageBirth sd_hispanic=hispanic
  sd_black=black sd_white=white sd_otherRace=otherRace sd_highEduc=highEduc
  sd_married=married;
#delimit cr
expand 13
gen meanNV  = .
gen stdevNV = .
gen var   = ""
local i = 1
foreach var of varlist `comvars' {
    replace meanNV  = `var' in `i'
    replace stdevNV = sd_`var' in `i'
    replace var = "`var'" in `i'
    local ++i
}
keep meanNV stdevNV var N_NV
tempfile NVSSSum
save `NVSSSum'
merge 1:1 var using `MTurkSum'
local i = 1
#delimit ;
local vnames `""Number of Children" "Age at First Birth" "Female Child"
               "Some College +" "Born January-March" "Born April-June"
               "Born July-September" "Born October-December"
               "Black" "White" "Other Race" "Hispanic" "Married" "';
local variables nchild ageBirth sexchild highEduc Q1 Q2 Q3 Q4
                black white otherRace hispanic married;
#delimit cr

tokenize `variables'
file open bstats using "$OUT/NVSScomp.txt", write replace
foreach var of local vnames {
    foreach stat in N mean stdev N_NV meanNV stdevNV {
        qui sum `stat' if var=="`1'"
        local val`stat'=r(mean)
    }
    qui ttesti `valN' `valmean' `valstdev' `valN_NV' `valmeanNV' `valstdevNV'
    foreach val in mu_1 sd_1 mu_2 sd_2 t {
        local `val'=string(r(`val'), "%5.3f")
        *local `val'=round(r(`val')*1000)/1000
        *if ``val''<1&``val''>0 local `val' = "0``val''"
    }
    local dif = string((`mu_1'-`mu_2'),"%5.3f")
    *if `dif'<1&`dif'>0 local dif = "0`dif'"
    file write bstats "`var'&`mu_1'&(`sd_1')&`mu_2'&(`sd_2')&`dif'&`t'\\ " _n
    macro shift
}
file close bstats
restore


*-------------------------------------------------------------------------------
*--- (3B) Comparison with ACS [TABLE A19: PANEL B]
*-------------------------------------------------------------------------------
preserve
keep if N==1
keep if RespSex=="Female" &age>=20&age<=45
gen Q1=RespMOB=="January"|RespMOB=="February"|RespMOB=="March"     if RespMOB!=""
gen Q2=RespMOB=="April"  |RespMOB=="May"     |RespMOB=="June"      if RespMOB!=""
gen Q3=RespMOB=="July"   |RespMOB=="August"  |RespMOB=="September" if RespMOB!=""
gen Q4=RespMOB=="October"|RespMOB=="November"|RespMOB=="December"  if RespMOB!=""
#delimit ;
collapse (sum) N (mean) ftotinc highEduc someCollege married employed hispanic
black white otherRace age educY teacher Q1 Q2 Q3 Q4 parent
  (sd) sd_ftotinc=ftotinc
  sd_highEduc=highEduc sd_someCollege=someCollege sd_married=married
  sd_employed=employed sd_hispanic=hispanic sd_black=black sd_white=white
  sd_otherRace=otherRace sd_age=age sd_educY=educY sd_teacher=teacher
  sd_Q1=Q1 sd_Q2=Q2 sd_Q3=Q3 sd_Q4=Q4 sd_parent=parent;
#delimit cr
expand 17
gen mean  = .
gen stdev = .
gen var   = ""
local i = 1
#delimit ;
local comvars ftotinc highEduc someCollege married employed hispanic black white
              otherRace age educY teacher Q1 Q2 Q3 Q4 parent;
#delimit cr
foreach var of varlist `comvars' {
    replace mean  = `var' in `i'
    replace stdev = sd_`var' in `i'
    replace var = "`var'" in `i'
    local ++i
}
gen data = "MTurk"
keep mean stdev var data N
tempfile MTurkSum2
save `MTurkSum2'
restore

preserve
insheet using "$ACS/ACS2014_20-45yrs.csv", clear names comma case
gen oneChild=nchild==1

gen parent = nchild!=0
keep if race<7
drop if occ2010 == 9920
gen N_ACS = 1
replace ftotinc = ftotinc/1000
gen educY     = 0  if educ==0
replace educY = 4  if educ==1
replace educY = 8  if educ==2
replace educY = 9  if educ==3
replace educY = 10 if educ==4
replace educY = 11 if educ==5
replace educY = 12 if educ==6
replace educY = 13 if educ==7
replace educY = 14 if educ==8
replace educY = 16 if educ==10
replace educY = 17 if educ==11
gen teacher = twoLevelOcc=="Education, Training, and Library Occupations"
gen hispanic=hispan!=0
gen age=motherAge
gen white = race==1
gen black = race==2
gen someCollege = educ>=7
gen otherRace = race!=1&race!=2
gen employed  = empstat==1
gen Q1 = birthQuarter==1
gen Q2 = birthQuarter==2
gen Q3 = birthQuarter==3
gen Q4 = birthQuarter==4

*rename hispan hispanic
#delimit ;
collapse (sum) N_ACS (mean) ftotinc highEduc someCollege married employed
  hispanic black white otherRace age educY teacher Q1 Q2 Q3 Q4 parent (sd)
  sd_ftotinc=ftotinc sd_highEduc=highEduc sd_someCollege=someCollege
  sd_married=married sd_employed=employed sd_hispanic=hispanic sd_black=black
  sd_white=white sd_otherRace=otherRace sd_age=age sd_educY=educY
  sd_teacher=teacher sd_Q1=Q1 sd_Q2=Q2 sd_Q3=Q3 sd_Q4=Q4 sd_parent=parent;
#delimit cr
expand 17
gen meanACS  = .
gen stdevACS = .
gen var      = ""
local i = 1
foreach var of varlist `comvars' {
    replace mean  = `var' in `i'
    replace stdev = sd_`var' in `i'
    replace var = "`var'" in `i'
    local ++i
}
keep meanACS stdevACS var N_ACS
tempfile ACSSum
save `ACSSum'
merge 1:1 var using `MTurkSum2'
keep if _merge==3
local i = 1
#delimit ;
local vnames `" "Family Income (1000s)" "Age" "Some College +"
                "Respondent Born January-March" "Respondent Born April-June"
                "Respondent Born July-September" "Respondent Born October-December"
                "Parent" "Employed" "Education, Training, Library Occ." "Black" "White"
                "Other Race" "Hispanic" "Married"  "';
local variables ftotinc age highEduc Q1 Q2 Q3 Q4 parent employed teacher black white
                otherRace hispanic married;
#delimit cr
tokenize `variables'
file open mstats using "$OUT/ACScomp.txt", write replace
foreach var of local vnames {
    dis "`var'"
    foreach stat in N mean stdev N_ACS meanACS stdevACS {
        qui sum `stat' if var=="`1'"
        local val`stat'=r(mean)
    }
    qui ttesti `valN' `valmean' `valstdev' `valN_ACS' `valmeanACS' `valstdevACS'
    foreach val in mu_1 sd_1 mu_2 sd_2 t {
        local `val'=string(r(`val'), "%5.3f")
        *local `val'=round(r(`val')*1000)/1000
        *if ``val''<1&``val''>0 local `val' = "0``val''"
    }
    local dif = round((`mu_1'-`mu_2')*1000)/1000
    if `dif'<1&`dif'>0 local dif = "0`dif'"
    if `dif'>-1&`dif'<0  {
        local adif = abs(`dif')
        local dif = "-0`adif'"
    }
    file write mstats "`var'&`mu_1'&(`sd_1')&`mu_2'&(`sd_2')&`dif'&`t'\\ " _n
    macro shift
}
file close mstats
restore

*-------------------------------------------------------------------------------
*-- (4) Main analysis  [TABLE 4, TABLE A20]
*-------------------------------------------------------------------------------
** GENERATE GREEDY LEAMER CRITERION LEVEL
reg chosen `oFEs' _sob* _cost* _gend* _bwt* _dob*
local tvL  = sqrt((e(df_r)/1)*(e(N)^(1/e(N))-1))
local pvL  = ttail(e(N),sqrt((e(df_r)/1)*(e(N)^(1/e(N))-1)))*2
dis `pvL'

local nvar1 _bwt2 _bwt3 _bwt4 _bwt5 _bwt6 _bwt7 _bwt8 _bwt9 _bwt10 _bwt11
order `nvar1'
local nvar2 _dob2
cap drop nkids
gen     nkids = 1 if RespNumKids=="1"
replace nkids = 2 if RespNumKids!="0"&nkids==.

local ctrl `oFEs' _gend* _bwt* _dob*
local se cluster(ID)
foreach spec in main wt {
    if `"`spec'"'=="main" {
        local wnote
        #delimit ;
        local ttitle "Birth Characteristics and Willingness to Pay
                      for Season of Birth";
        #delimit cr
        local wt
    }
    else if `"`spec'"'=="wt" {
        local wt [pw=weight]
        #delimit ;
        local ttitle "Birth Characteristics and WTP for Season of Birth
                      (reweighted sample)";
        local wnote "Observations are re-weighted based on Census Bureau
        state population cells.  Each respondent is weighted such that the
        frequency of observations by state in MTurk data is identical to
        that in Census Bureau data.  In practice this is calculated as
        $ Pwt=(Pr(Census)_s/Pr(MTurk)_s)$ for each state $ s$.";
        #delimit cr
    }    
    eststo: logit chosen spring summer _sob4 costNumerical `ctrl' s2018 `wt', `se'
    local numR = e(N_clust)
    margins, dydx(spring summer costNumerical  _gend2 `nvar1' `nvar2' _sob4) post
    est store n1
    estadd scalar numR  = `numR'
    estadd scalar wtpSp = -1000*_b[spring]/_b[costNumerical]
    estadd scalar wtpSu = -1000*_b[summer]/_b[costNumerical]
    nlcom ratio:_b[spring]/_b[costNumerical], post
    local lb = string(-1000*(_b[ratio]-`tvL'*_se[ratio]), "%5.1f")
    local ub = string(-1000*(_b[ratio]+`tvL'*_se[ratio]), "%5.1f")
    estadd local conf95sp "[`ub';`lb']": n1
    est restore n1
    nlcom ratio:_b[summer]/_b[costNumerical], post
    local lb = string(-1000*(_b[ratio]-`tvL'*_se[ratio]), "%5.1f")
    local ub = string(-1000*(_b[ratio]+`tvL'*_se[ratio]), "%5.1f")
    estadd local conf95su "[`ub';`lb']": n1

    **NVSS main sample
    local c osample==1
    eststo: logit chosen spring summer _sob4 costNumerical `ctrl' s2018 `wt' if `c', `se'
    local numR = e(N_clust)
    margins, dydx(spring summer costNumerical  _gend2 `nvar1' `nvar2' _sob4) post
    est store nmain1
    estadd scalar numR  = `numR'
    estadd scalar wtpSp = -1000*_b[spring]/_b[costNumerical]
    estadd scalar wtpSu = -1000*_b[summer]/_b[costNumerical]
    nlcom ratio:_b[spring]/_b[costNumerical], post
    local lb = string(-1000*(_b[ratio]-`tvL'*_se[ratio]), "%5.1f")
    local ub = string(-1000*(_b[ratio]+`tvL'*_se[ratio]), "%5.1f")
    estadd local conf95sp "[`ub';`lb']": nmain1
    est restore nmain1
    nlcom ratio:_b[summer]/_b[costNumerical], post
    local lb = string(-1000*(_b[ratio]-`tvL'*_se[ratio]), "%5.1f")
    local ub = string(-1000*(_b[ratio]+`tvL'*_se[ratio]), "%5.1f")
    estadd local conf95su "[`ub';`lb']": nmain1

    **1 child sample
    local c osample==1&nkids==1
    eststo: logit chosen spring summer _sob4 costNumerical `ctrl' s2018 `wt' if `c', `se'
    local numR = e(N_clust)
    margins, dydx(spring summer costNumerical  _gend2 `nvar1' `nvar2' _sob4) post
    est store nkids1
    estadd scalar numR  = `numR'
    estadd scalar wtpSp = -1000*_b[spring]/_b[costNumerical]
    estadd scalar wtpSu = -1000*_b[summer]/_b[costNumerical]
    nlcom ratio:_b[spring]/_b[costNumerical], post
    local lb = string(-1000*(_b[ratio]-`tvL'*_se[ratio]), "%5.1f")
    local ub = string(-1000*(_b[ratio]+`tvL'*_se[ratio]), "%5.1f")
    estadd local conf95sp "[`ub';`lb']": nkids1
    est restore nkids1
    nlcom ratio:_b[summer]/_b[costNumerical], post
    local lb = string(-1000*(_b[ratio]-`tvL'*_se[ratio]), "%5.1f")
    local ub = string(-1000*(_b[ratio]+`tvL'*_se[ratio]), "%5.1f")
    estadd local conf95su "[`ub';`lb']": nkids1

    **2 child sample
    local c osample==1&nkids==2
    eststo: logit chosen spring summer _sob4 costNumerical `ctrl' s2018 `wt' if `c', `se'
    local numR = e(N_clust)
    margins, dydx(spring summer costNumerical  _gend2 `nvar1' `nvar2' _sob4) post
    est store nkids2
    estadd scalar numR  = `numR'
    estadd scalar wtpSp = -1000*_b[spring]/_b[costNumerical]
    estadd scalar wtpSu = -1000*_b[summer]/_b[costNumerical]
    nlcom ratio:_b[spring]/_b[costNumerical], post
    local lb = string(-1000*(_b[ratio]-`tvL'*_se[ratio]), "%5.1f")
    local ub = string(-1000*(_b[ratio]+`tvL'*_se[ratio]), "%5.1f")
    estadd local conf95sp "[`ub';`lb']": nkids2
    est restore nkids2
    nlcom ratio:_b[summer]/_b[costNumerical], post
    local lb = string(-1000*(_b[ratio]-`tvL'*_se[ratio]), "%5.1f")
    local ub = string(-1000*(_b[ratio]+`tvL'*_se[ratio]), "%5.1f")
    estadd local conf95su "[`ub';`lb']": nkids2

    **Intended childless sample
    local c planning==0&parent!=1&age>=20&age<=45
    eststo: logit chosen spring summer _sob4 costNumerical `ctrl' s2018 `wt' if `c', `se'
    local numR = e(N_clust)
    margins, dydx(spring summer costNumerical  _gend2 `nvar1' `nvar2' _sob4) post
    est store nplan1
    estadd scalar numR  = `numR'
    estadd scalar wtpSp = -1000*_b[spring]/_b[costNumerical]
    estadd scalar wtpSu = -1000*_b[summer]/_b[costNumerical]
    nlcom ratio:_b[spring]/_b[costNumerical], post
    local lb = string(-1000*(_b[ratio]-`tvL'*_se[ratio]), "%5.1f")
    local ub = string(-1000*(_b[ratio]+`tvL'*_se[ratio]), "%5.1f")
    estadd local conf95sp "[`ub';`lb']": nplan1
    est restore nplan1
    nlcom ratio:_b[summer]/_b[costNumerical], post
    local lb = string(-1000*(_b[ratio]-`tvL'*_se[ratio]), "%5.1f")
    local ub = string(-1000*(_b[ratio]+`tvL'*_se[ratio]), "%5.1f")
    estadd local conf95su "[`ub';`lb']": nplan1

    lab var _dob2 "Weekend Day"
    lab var _sob4 "Fall"
    lab var spring "Spring"
    lab var summer "Summer"
    lab var costNumerical "Cost (in 1000s)"
    lab var goodSeason "Quarter 2 or Quarter 3"
    #delimit ;
    esttab n1 nmain1 nkids1 nkids2 nplan1  using "$OUT/WTP-seasons-`spec'.tex", replace
    cells(b(star fmt(%-9.3f)) se(fmt(%-9.3f) par([ ]) )) stats
    (wtpSp conf95sp N numR, fmt(%5.1f %5.1f %9.0gc %9.0gc)
        label("WTP for Spring (USD)" "95\% CI" Observations "Number of Respondents"))
    starlevel ("$ ^{\ddagger} $" `pvL') collabels(,none)
    mgroups("Full Sample" "Married Mothers 20-45" "Intended Childless", pattern(1 1 0 0 1)
        prefix(\multicolumn{@span}{c}{) suffix(}) span erepeat(\cmidrule(lr){@span}))
    mlabels("All" "All" "1 child" "$\geq$ 2 children" "20-45") booktabs
    label title(`ttitle'\label{conjointWTP`spec'})
    keep(spring summer _sob4 costNumerical _gend2 `nvar1' `nvar2') style(tex)
    postfoot("\bottomrule           "
             "\multicolumn{6}{p{17.0cm}}{\begin{footnotesize} Each estimation sample  "
             "consists of US-based respondents to a Mechanical Turk survey with waves "
             "completed in September 2016 and May 2018.  Any subsets of this sample   "
             "are listed in column headings. Average marginal effects from a logit    "
             "regression are displayed. All columns include option order fixed effects"
             "and round fixed effects, plus an indicator for the survey wave (2016 or "
             "2018). The sample in each column is indicated in column headings.       "
             "Standard errors are clustered by respondent. `wnote' Willingness to pay "
             "and its 95\% confidence interval is estimated based on the ratio of     "
             "costs to the probability of choosing a spring birth. The 95\% confidence"
             "interval is calculated using the delta method for the (non-linear)      "
             "ratio, with confidence levels based on Leamer values.                   "
             "$^{\ddagger}$ Significance based on Leamer criterion at 5\%."
             "\end{footnotesize}}\end{tabular}\end{table}");
    #delimit cr
    estimates clear
}

*-------------------------------------------------------------------------------
*-- (5) Bootstrap WTP predictors for NVSS analysis [TABLE A22]
*-------------------------------------------------------------------------------
cap gen price = costNumerical
cap gen group = 1000*ID+round
#delimit ;
cap gen highEd  = RespEduc!="Eighth Grade or Less" &
                  RespEduc!="High School Degree/GED" &
                  RespEduc!="Some High School";
#delimit cr
local bwts _bwt2 _bwt3 _bwt4 _bwt5 _bwt6 _bwt7 _bwt8 _bwt9 _bwt10 _bwt11


gen educYrs     = 8 if RespEduc=="Eighth Grade or Less"
replace educYrs = 10 if RespEduc=="Eighth Grade or Less"
replace educYrs = 12 if RespEduc=="High School Degree/GED"
replace educYrs = 13 if RespEduc=="Some College"
replace educYrs = 14 if RespEduc=="2-year College Degree"
replace educYrs = 16 if RespEduc=="4-year College Degree"
replace educYrs = 17 if RespEduc=="Master's Degree"
replace educYrs = 17 if RespEduc=="Doctoral Degree"
replace educYrs = 17 if RespEduc=="Professional Degree (JD,MD,MBA)"
local wt

mixlogit chosen price `wt' if osample==1, id(ID) group(group) rand(_sob* _gend* `bwts')
local price = _b[price]
tempfile betas
set seed 1704
mixlbeta _sob2 if osample==1, saving(`betas') replace
rename _sob2 __sob2
merge m:1 ID using `betas'
drop _merge
rename _sob2 betaSOB
rename __sob2 _sob2
replace betaSOB=-1000*betaSOB/`price'
gen educYrsSq = educYrs^2
gen female = _gend2
cap gen ageSq=age*age

local educ educYrs educYrsSq
local age  i.age
local educ highEd
local age  age ageSq

reg betaSOB `age' `educ' white black hispanic `wt' if osample==1
estimates store WTPfull

gen WTPests1  = .
gen WTPests2  = .
preserve
insheet using "$NVS/nvss2005_2013_BSAMP.csv", comma names clear case
save "$NVS/nvss2005_2013_BSAMP.dta", replace
gen ageSq = age*age
local controls female birthweight weekend
gen birthweight = bweightLin


estimates restore WTPfull
predict WTPhatFull, xb

sum WTPhatFull
local mean1=r(mean)
local mean2=r(mean)

reg quarter2 WTPhatFull if smoker!=.&gest!=.
local tL1  = sqrt((e(df_r)/1)*(e(N)^(1/e(N))-1))
local wtpF `=_b[WTPhat]'
local N1 = e(N)

reg quarter2 WTPhatFull `controls' if smoker!=.&gest!=.
local wtpC `=_b[WTPhat]'
local N2 = e(N)

local beta1 = 1000*`wtpF'
local beta2 = 1000*`wtpC'

restore
replace WTPests1 = `wtpF'    in 1
replace WTPests2 = `wtpC'    in 1
drop betaSOB


*-- Bootstrap N=100 --------------------------------------------------------
set seed 1307

local j = 2
local N = 100
foreach num of numlist 1(1)`N' {
    dis "BOOTSTRAP REPLICATION `num'"
    preserve

    use "$NVS/nvss2005_2013_BSAMP.dta", clear 
    gen ageSq = age*age
    gen birthweight = bweightLin
    bsample
    local controls smoker WIC underweight overweight obese noART hispanic
    local controls female birthweight weekend

    estimates restore WTPfull
    predict WTPhatFull, xb
    
    qui reg quarter2 WTPhatFull if smoker!=.&gest!=.
    local wtpF `=_b[WTPhatFull]'

    qui reg quarter2 WTPhatFull `controls' if smoker!=.&gest!=.
    local wtpC `=_b[WTPhatFull]'

    restore
    replace WTPests1  = `wtpF'    in `j'
    replace WTPests2  = `wtpC'    in `j'
    
    local ++j
}
foreach num of numlist 1(1)2 {
    sum WTPests`num'
    local se`num' = 1000*r(sd)
}

local betas
local stdes
file open table8 using "$OUT/WTPspringBirth.tex", replace write
file write table8 "WTP (1000 USD)"

foreach num of numlist 1(1)2 {
    local se = `se`num''
    local t = `beta`num''/`se'
    local beta `=string(`beta`num'', "%5.3f")'
    file write table8 "&" `"`macval(beta)'"'
    if abs(`t')>`tL1' file write table8 "$^{\ddagger}$"
}
file write table8 "\\" _n 
foreach num of numlist 1(1)2 {
    local se `=string(`se`num'', "%5.3f")'
    file write table8 "& [" `"`macval(se)'"' "]"
}
file write table8 "\\ \\" _n
file write table8 "Mean WTP (NVSS data) "
foreach num of numlist 1(1)2 {
    local mean `=string(`mean`num'', "%7.2f")'
    file write table8 "& \\$" `"`macval(mean)'"'
}
file write table8 "\\" _n "Observations "
foreach num of numlist 1(1)2 {
    local NN `=string(`N`num'', "%12.0gc")'
    file write table8 "&" `"`macval(NN)'"' 
}
file write table8 "\\" _n
file close table8

*-------------------------------------------------------------------------------
*-- (6) Heterogeneity using mixed logit [TABLE A21]
*-------------------------------------------------------------------------------
gen bweightLin = 2500 if birthweight=="5 pounds 8 ounces"
replace bweightLin = 2637 if birthweight=="5 pounds 13 ounces"
replace bweightLin = 2807 if birthweight=="6 pounds 3 ounces"
replace bweightLin = 2948 if birthweight=="6 pounds 8 ounces"
replace bweightLin = 3090 if birthweight=="6 pounds 13 ounces"
replace bweightLin = 3260 if birthweight=="7 pounds 3 ounces"
replace bweightLin = 3402 if birthweight=="7 pounds 8 ounces"
replace bweightLin = 3544 if birthweight=="7 pounds 13 ounces"
replace bweightLin = 3714 if birthweight=="8 pounds 3 ounces"
replace bweightLin = 3856 if birthweight=="8 pounds 8 ounces"
replace bweightLin = 4000 if birthweight=="8 pounds 13 ounces"

qui reg chosen `oFEs' _sob* _cost* _gend* _bwt* _dob*
local tvL  = sqrt((e(df_r)/1)*(e(N)^(1/e(N))-1))
local pvL  = ttail(e(N),sqrt((e(df_r)/1)*(e(N)^(1/e(N))-1)))*2
dis `pvL'

cap gen all=1 
local conds osample==1 all==1

local j = 1
foreach name in married all {
    if `j'==1 local cond osample==1
    if `j'==2 local cond all==1
    
    cap gen price = costNumerical
    cap gen group = 1000*ID+round
    cap tab round, gen(_rr)
    cap tab option, gen(_oo)
    local bwts _bwt2 _bwt3 _bwt4 _bwt5 _bwt6 _bwt7 _bwt8 _bwt9 _bwt10 _bwt11
    
    mixlogit chosen price if `cond', id(ID) group(group) rand(_sob* _gend* `bwts' _dob2)
    estimates store ml`name'
    levelsof ID if e(sample)==1
    local numR = r(r)
    estadd scalar numR = `numR'
    estadd scalar pcb = 100*normal(_b[Mean:_sob2]/abs(_b[SD:_sob2]))
    local price = _b[price]
    
    estadd scalar wtp = -1000*(_b[_sob2]/_b[price])
    nlcom ratio:_b[_sob2]/_b[price], post
    local lb = string(-1000*(_b[ratio]-1.96*_se[ratio]), "%5.1f")
    local ub = string(-1000*(_b[ratio]+1.96*_se[ratio]), "%5.1f")
    estadd local conf95 "[`ub';`lb']": ml`name'

    mixlogit chosen price if `cond', id(ID) group(group) rand(_sob* _gend* bweightLin)
    estimates store mlbw`name'
    levelsof ID if e(sample)==1
    local numR = r(r)
    estadd scalar numR = `numR'
    estadd scalar pcb = 100*normal(_b[Mean:_sob2]/abs(_b[SD:_sob2]))

    estadd scalar wtp = -1000*(_b[_sob2]/_b[price])
    nlcom ratio:_b[_sob2]/_b[price], post
    local lb = string(-1000*(_b[ratio]-1.96*_se[ratio]), "%5.1f")
    local ub = string(-1000*(_b[ratio]+1.96*_se[ratio]), "%5.1f")
    estadd local conf95 "[`ub';`lb']": mlbw`name'
    local ++j
}

lab var _sob2 "Spring"
lab var _sob3 "Summer"
lab var _sob4 "Fall"
lab var price "Cost (in 1000s)"
lab var bweightLin "Birthweight (in grams)"

#delimit ;
esttab mlall mlbwall mlmarried mlbwmarried using "$OUT/WTP-mixedlogit-both.tex", replace
cells(b(star fmt(%-9.3f)) se(fmt(%-9.3f) par([ ]) )) stats
(wtp conf95 pcb N numR, fmt(%5.1f %5.1f %5.1f %9.0gc %9.0gc)
    label("WTP for Spring Birth" "95\% CI"
          "\% Positively Impacted by Spring Birth" Observations
          "Number of Respondents")) starlevel ("$ ^{\ddagger} $" `pvL')
mgroups("All" "Married Mothers, 20-45", pattern(1 0 1 0)
        prefix(\multicolumn{@span}{c}{) suffix(}) span erepeat(\cmidrule(lr){@span}))
collabels(,none) style(tex) booktabs label
title("Allowing for Preference Heterogeneity with Mixed Logit"\label{WTPmix})
postfoot("\bottomrule           "
         "\multicolumn{5}{p{17.1cm}}{\begin{footnotesize} Panel A displays   "
         "mean coefficients from the mixed logit, and panel B displays the   "
         "estimated standard deviation of each coefficient.  All coefficients"
         " with the exception of Cost are allowed to vary randomly throughout"
         " the sample.  The WTP is calculated as the ratio of the coefficient"
         " on spring birth tso that on costs, and confidence intervals are    "
         "calculated by the delta method. The \% of respondents who value    "
         "a spring birth positively based on individual coefficients is      "
         "displayed at the foot of the table.  Standard errors are clustered "
         "by respondent."
         "\end{footnotesize}}\end{tabular}\end{table}");
#delimit cr
estimates clear


*-------------------------------------------------------------------------------
*-- (7) Close
*-------------------------------------------------------------------------------
log close
