function b = est(H,D,X,Z,type)
    % Simulations based on DTUS for:
    % Cheng Chou, Ruoyao Shi, "What Time Use Surveys Can (And Cannot) Tell Us
    % About Labor Supply", Journal of Applied Economics, 2021, forthcoming
    % Estimation routine for 7 days (Table 1)
    % Ruoyao Shi
    
    % H - hours, weekly or daily, depending on type
    % D - diary day
    % X - regressor(s)
    % Z - IV
    % type:
    %   1 - week estimator, does not use D
    %   2 - impute estimator
    %   3 - pool estimator
    %   4 - day estimator
    [n,~]   = size(H);
    Reg     = [ones(n,1),X];
    IV      = [ones(n,1),Z];
    
    if type == 1 % week estimator (CPS & DTUS)
        Ainv    = inv(Reg' * IV * inv(IV' * IV) * IV' * Reg);
        b       = Ainv * Reg' * IV * inv(IV' * IV) * IV'  * H;
        u       = H - Reg * b;
        sigmau  = u' * u / n;
    elseif type == 2 % impute estimator (ATUS & DTUS)
        X_hat       = IV*inv(IV'*IV) * IV'*X;
        X_hat       = [X_hat(D==1,:); X_hat(D==2,:); X_hat(D==3,:); X_hat(D==4,:); X_hat(D==5,:); X_hat(D==6,:); X_hat(D==7,:)]; % fitted X
        nt      = [sum(D==1) sum(D==2) sum(D==3) sum(D==4) sum(D==5) sum(D==6) sum(D==7)]; % subsample size
        H1 = H(D==1,1);   IV1 = IV(D==1,:);
        H2 = H(D==2,1);   IV2 = IV(D==2,:);
        H3 = H(D==3,1);   IV3 = IV(D==3,:);
        H4 = H(D==4,1);   IV4 = IV(D==4,:);
        H5 = H(D==5,1);   IV5 = IV(D==5,:);
        H6 = H(D==6,1);   IV6 = IV(D==6,:);
        H7 = H(D==7,1);   IV7 = IV(D==7,:);
        IV_hat      = [IV1; IV2; IV3; IV4; IV5; IV6; IV7];
        H1_hat      = IV_hat * inv(IV1'*IV1) * IV1' * H1;
        H2_hat      = IV_hat * inv(IV2'*IV2) * IV2' * H2;
        H3_hat      = IV_hat * inv(IV3'*IV3) * IV3' * H3;
        H4_hat      = IV_hat * inv(IV4'*IV4) * IV4' * H4;
        H5_hat      = IV_hat * inv(IV5'*IV5) * IV5' * H5;
        H6_hat      = IV_hat * inv(IV6'*IV6) * IV6' * H6;
        H7_hat      = IV_hat * inv(IV7'*IV7) * IV7' * H7;
        H_hat       = H1_hat + H2_hat + H3_hat + H4_hat + H5_hat + H6_hat + H7_hat;
        b           = inv([ones(n,1) X_hat]'*[ones(n,1) X_hat]) * [ones(n,1) X_hat]'*H_hat;
    elseif type == 3 % day estimator (ATUS & DTUS)
        nt      = [sum(D==1) sum(D==2) sum(D==3) sum(D==4) sum(D==5) sum(D==6) sum(D==7)]; % subsample size
        Ainv    = inv(Reg' * IV * inv(IV' * IV) * IV' * Reg);
        R       = n * eye(n);
        for i = 1:n
            R(i,i)  = R(i,i) / nt(D(i));
        end
        b       = Ainv * Reg' * IV * inv(IV' * IV) * IV'  * (R * H);
    elseif type == 4 % within estimator (ATUS & DTUS)
        nt      = [sum(D==1) sum(D==2) sum(D==3) sum(D==4) sum(D==5) sum(D==6) sum(D==7)]; % subsample size
        H1 = H(D==1,1);   Reg1 = Reg(D==1,:);   IV1 = IV(D==1,:);
        H2 = H(D==2,1);   Reg2 = Reg(D==2,:);   IV2 = IV(D==2,:);
        H3 = H(D==3,1);   Reg3 = Reg(D==3,:);   IV3 = IV(D==3,:);
        H4 = H(D==4,1);   Reg4 = Reg(D==4,:);   IV4 = IV(D==4,:);
        H5 = H(D==5,1);   Reg5 = Reg(D==5,:);   IV5 = IV(D==5,:);
        H6 = H(D==6,1);   Reg6 = Reg(D==6,:);   IV6 = IV(D==6,:);
        H7 = H(D==7,1);   Reg7 = Reg(D==7,:);   IV7 = IV(D==7,:);
        b1      = inv(Reg1' * IV1 * inv(IV1' * IV1) * IV1' * Reg1) * Reg1' * IV1 * inv(IV1' * IV1) * IV1' * H1;
        b2      = inv(Reg2' * IV2 * inv(IV2' * IV2) * IV2' * Reg2) * Reg2' * IV2 * inv(IV2' * IV2) * IV2' * H2;
        b3      = inv(Reg3' * IV3 * inv(IV3' * IV3) * IV3' * Reg3) * Reg3' * IV3 * inv(IV3' * IV3) * IV3' * H3;
        b4      = inv(Reg4' * IV4 * inv(IV4' * IV4) * IV4' * Reg4) * Reg4' * IV4 * inv(IV4' * IV4) * IV4' * H4;
        b5      = inv(Reg5' * IV5 * inv(IV5' * IV5) * IV5' * Reg5) * Reg5' * IV5 * inv(IV5' * IV5) * IV5' * H5;
        b6      = inv(Reg6' * IV6 * inv(IV6' * IV6) * IV6' * Reg6) * Reg6' * IV6 * inv(IV6' * IV6) * IV6' * H6;
        b7      = inv(Reg7' * IV7 * inv(IV7' * IV7) * IV7' * Reg7) * Reg7' * IV7 * inv(IV7' * IV7) * IV7' * H7;
        b       = b1 + b2 + b3 + b4 + b5 + b6 + b7;
        u1      = H1 - Reg1 * b1;   sigmau1     = u1' * u1 / nt(1);
        u2      = H2 - Reg2 * b2;   sigmau2     = u2' * u2 / nt(2);
        u3      = H3 - Reg3 * b3;   sigmau3     = u3' * u3 / nt(3);
        u4      = H4 - Reg4 * b4;   sigmau4     = u4' * u4 / nt(4);
        u5      = H5 - Reg5 * b5;   sigmau5     = u5' * u5 / nt(5);
        u6      = H6 - Reg6 * b6;   sigmau6     = u6' * u6 / nt(6);
        u7      = H7 - Reg7 * b7;   sigmau7     = u7' * u7 / nt(7);
        Ainv    = inv(Reg' * IV * inv(IV' * IV) * IV' * Reg);
    end
    
end