%% ABG Rep - MC Simulation Experiment
% for asymmetric case with HET
% Dependence for sim DGP:
% Y(t) = X0 + a1 X1(t-1) + a2 X2(t) 
% X1 and u are iid std normals, X0 = 0
% X2(t) = Y(t-1)

clear all
tic
figflag = 1 ;   % set to 1 to produce plots
bw = 0.1 ;      % bandwidth for simulation from DGP
T = 200 ;       % length of sim 100, 200, 300, 400, 500
rng(80) ;
MCr = 10000 ;   % Number of reps
bb = 50000  ;   % burnin 
sw = bb+T ; 

%% Load data

load ABGRep.mat 
g = g1GDP(2:end) ;
n = NFCI(1:end-1) ; 
gl = g1GDP(1:end-1) ;

Tg = length(g) ;

zg = zscore(g) ;
zn = zscore(n) ;

rg = tiedrank(g)/(Tg+0.01) ;
rn = tiedrank(n)/(Tg+0.01) ;
rgl = tiedrank(gl)/(Tg+0.01) ;

% Create output matrices

ocCOP33 = zeros(MCr,2) ;
ocCOP50 = zeros(MCr,2) ;
ocCOP67 = zeros(MCr,2) ;

ocCON33 = zeros(MCr,2) ;
ocCON50 = zeros(MCr,2) ;
ocCON67 = zeros(MCr,2) ;

%% Monte Carlo

for j = 1:MCr ;

    %  draws
    X1 = randn(bb+sw,1) ;
    u = randn(bb+sw,1) ;

    % discard burnins
    X1 = zscore(X1(end-T:end,1)) ;
    u = u(end-T:end,1) ;

    % initialize + parameters
    
    X0 = 0 ;
    a1 = -0.25 ;    % dependence coef on lagged NFCI
    a2 = 0.1 ;      % dependence coef on lagged DV 

    X10 = 0 ; 
    X20 = 0 ; 

    Y = zeros(T+1,1) ;
    Y(1) = X0 + a1*X10 + a2*X20 + u(1) ; 

    % sim dependence

    for t = 2:T+1 ;
        Y(t) = X0 + a1*X1(t-1) + a2*Y(t-1) + u(t) ;  
    end ;

    X1 = X1(1:T) ;
    X2 = Y(1:T) ;
    Y = Y(2:T+1) ;

    rY = tiedrank(Y)/(1.01*T) ;
    rX1 = tiedrank(X1)/(1.01*T) ;
    rX2 = tiedrank(X2)/(1.01*T) ;

    % calibrate HET to QR on pseudo data

    ng  = norminv(rg, 0,1) ;
    nn = norminv(rn, 0, 1) ;
    ngl  = norminv(rgl, 0,1) ;

    cg = zscore(ng) ;
    cn = zscore(nn) ;
    cgl = zscore(ngl) ;
    
    y = cg ;
    x =  [ones(Tg,1)  cn cgl] ;

    % QR on pseudo data
    q33 = rq(x, y, 0.33) ;
    q50 = rq(x, y, 0.50) ; 
    
    % Cop QR lines on pseudo data  
    xp =  [ones(Tg,1) cn] ;
    p33 = xp*q33(1:2) ; 
    p50 = xp*q50(1:2) ; 

    stde = 2*(p50-p33) ;
    h = regress(stde, [ones(Tg,1) cn]) ;        % <-- stdev = h(1) + h(2) * n
    sX1 = norminv(rX1, 0, 1) ;
    sY = norminv(rY, 0, h(1)+h(2)*sX1) ;

    % swarm to ranks
    rsY = tiedrank(sY)/(1.01*length(sY)) ;     

    % Sim marginal - asym distrib uses pearson, calib to be close to that of target
    [pearr, type] = pearsrnd(0,1,-0.75,5,5000,1) ;      % set 4th para, kurtosis to 5 to get closer to sample
    aY = ksdensity(pearr, rsY, 'function', 'icdf', 'width', bw) ;

    % add some outliers - last 15 percent of sample uses ECDF of last 15 percent of sample 
    oaY = aY ;
    moaY = ksdensity(zg(round(0.85*Tg)+1:end), rsY, 'function', 'icdf', 'width', bw) ;
    oaY(round(0.85*T)+1:end) = moaY(round(0.85*T)+1:end) ;

    % std data, with and without outliers -- ay is asym dist target, oaY is asym with outliers
    oaY = zscore(oaY) ;
    aY = zscore(aY) ;

    X1L = zscore(X1(1:T)) ;
    X2L = zscore(X2(1:T)) ;
    
    % QR and copQR

    for i = 1:2 ;
        lhs = [aY oaY] ;
        ggg = lhs(:,i) ; 
        nnn = X1L ;
        gggl= X2L ;

        ggr = tiedrank(ggg)/(T+0.01) ;     
        nnr = tiedrank(nnn)/(T+0.01) ;
        gglr = tiedrank(gggl)/(T+0.01) ; 

        gps = norminv(ggr, 0, 1) ;
        nps = norminv(nnr, 0, 1) ; 
        glps = norminv(gglr, 0, 1) ;

        gps = zscore(gps) ;
        nps = zscore(nps) ;
        glps = zscore(glps) ;

        % COP QR on pseudo data, with two RHS variables

        y = gps ;
        x =  [ones(T,1)  nps glps] ;

        q33 = rq(x, y, 0.33) ;
        q50 = rq(x, y, 0.50) ; 
        q67 = rq(x, y, 0.67) ; 

        % CON QR, rep ABG with two RHS variables in regs on observed std normal
 
        nABG = zscore(nnn) ;
        gABG = zscore(ggg) ;
        glABG = zscore(gggl) ;

        xABG =  [ones(T,1)  nABG, glABG] ;
        q33ABG = rq(xABG, gABG, 0.33) ;
        q50ABG = rq(xABG, gABG, 0.50) ; 
        q67ABG = rq(xABG, gABG, 0.67) ; 

        % store coefs -- i toggles between aY (no outliers) and oaY (w/ outliers)
        % oc -- first row is no outliers, second row is outliers, cop QR is first col, con QR is second col 
        % (in q* first coef is intercept, second is for lagged NFCI, third lagged DV)

        oc33(i,:) = [q33(2) q33ABG(2)] ;
        oc50(i,:) = [q50(2) q50ABG(2)] ;
        oc67(i,:) = [q67(2) q67ABG(2)] ;

    end ;

    % j indexes reps as rows -- first col no outliers, second with outliers

    ocCOP33(j,:) = [oc33(1,1) oc33(2,1)] ; 
    ocCOP50(j,:) = [oc50(1,1) oc50(2,1)] ;   
    ocCOP67(j,:) = [oc67(1,1) oc67(2,1)] ;

    ocCON33(j,:) = [oc33(1,2) oc33(2,2)] ;
    ocCON50(j,:) = [oc50(1,2) oc50(2,2)] ;
    ocCON67(j,:) = [oc67(1,2) oc67(2,2)] ;
end

%% plots

if figflag == 1 ;

    % 33 coef, COPQR
    figure()
    ksdensity(ocCOP33(:,1)) ;   % blue is no outliers
    hold
    ksdensity(ocCOP33(:,2)) ;   % red is outliers
    
    % 50 coef, COPQR
    figure()
    ksdensity(ocCOP50(:,1)) ;   % blue is no outliers
    hold
    ksdensity(ocCOP50(:,2)) ;   % red is outliers

    % 67 coef, COPQR
    figure()
    ksdensity(ocCOP67(:,1)) ;   % blue is no outliers
    hold
    ksdensity(ocCOP67(:,2)) ;   % red is outliers

    % 33 coef, CON QR
    figure()
    ksdensity(ocCON33(:,1)) ;   % blue is no outliers
    hold
    ksdensity(ocCON33(:,2)) ;   % red is outliers

    % 50 coef, CON QR
    figure()
    ksdensity(ocCON50(:,1)) ;   % blue is no outliers
    hold
    ksdensity(ocCON50(:,2)) ;   % red is outliers

    % 67 coef, CON QR
    figure()
    ksdensity(ocCON67(:,1)) ;   % blue is no outliers
    hold
    ksdensity(ocCON67(:,2)) ;   % red is outliers

    %  plot for 50s, true is -0.25
    figure()
    ksdensity(ocCOP50(:,1)) ;   % no outliers COP QR
    hold
    ksdensity(ocCOP50(:,2)) ;   % red is outliers COP QR
    ksdensity(ocCON50(:,1)) ;   % no outliers CON QR
    ksdensity(ocCON50(:,2)) ;   %  outliers CON QR

    %  plot for 33s, 50s true is -0.25
    figure()
    ksdensity(ocCOP33(:,1)) ;   % no outliers COP QR
    hold
    ksdensity(ocCOP33(:,2)) ;   % red is outliers COP QR
    ksdensity(ocCON33(:,1)) ;   % no outliers CON QR
    ksdensity(ocCON33(:,2)) ;   %  outliers CON QR

end ;

%% Prob of coef gt than or equal to -0.2, for 33

CONoutPrGtTrueMed = 1- ksdensity(ocCON33(:,2), -0.25, 'function', 'cdf')        % outliers
COPQRp = 1- ksdensity(ocCOP33(:,2), -0.25, 'function', 'cdf')

%% Save output
save('MCSim.mat','ocCON33','ocCOP33','ocCON50','ocCOP50')

toc



