%% ABG replication -- Heteroskedasticity tests using observed and pseudo data
% Heteroskedasticity assumed to take the form sig2 = h0 + h1*NFCI

%% Pick growth measure and sample
clear all 
g_flag = 3 ;                     % Set to 1 = ABG's series (1dp), 2 = 1 qtr growth from GDPC1, 3 = 4 qtr growth from GDPC1
s_flag = 1 ;                     % Set to 0 for ABG's sample (ends 2015Q4); 1 for extended (ends 2021Q3)

%% Load Data 1973Q1 to 2021Q3 - see ABGRep_Data.m for details

load ABGRep

% Choose output growth series
if g_flag == 1 ;                % ABGs output growth data    
    g = ABGgGDP(2:end) ;        % output growth      
    gl = ABGgGDP(1:end-1) ;     % lagged
    n = NFCI(1:end-1) ;         % (lagged) NFCI 
    tt = QTR(2:end) ;           % dates, start 1973Q2
elseif g_flag == 2 ;            % 1-qtr output growth data
    g = g1GDP(2:end) ;          % output growth
    gl = g1GDP(1:end-1) ;       % lagged
    n = NFCI(1:end-1) ;         % (lagged) NFCI 
    tt = QTR(2:end) ;           % dates, start 1973Q2
elseif g_flag == 3 ;            % 4-qtr output growth data
    g = g4GDP(5:end) ;          % output growth
    gl = g4GDP(1:end-4) ;       % lagged
    n = NFCI(1:end-4) ;         % (lagged) NFCI 
    tt = QTR(5:end) ;           % dates, start 1974Q1
end ;

% If using ABG sample remove observations after 2015Q4
if s_flag == 0 ;
    num_ext = 20 ;             % Data extended to 2021Q3, 23 observations past end of ABG sample
    g = g(1:end-num_ext) ;
    gl = gl(1:end-num_ext) ;
    n = n(1:end-num_ext) ;
    tt = tt(1:end-num_ext) ;
    clear num_ext 
end ;

T = length(tt);                 % sample size

clear ABFgGDP g1GDP g4GDP NFCI QTR

%% Standardized, rank and generate pseudo data as standard normal

% Standardized observed data for standard QR
gm = zscore(g) ;                % output growth
nm = zscore(n) ;                % lagged NFCI
glm = zscore(gl) ;              % lagged output growth

% Ranks
gr = tiedrank(g)/(T+1) ;        % output growth
nr = tiedrank(n)/(T+1) ;        % lagged NFCI
glr = tiedrank(gl)/(T+1) ;      % lagged output growth

% Standardized normal pseudo data
gps = norminv(gr,0,1) ;         % output growth
nps = norminv(nr,0,1) ;         % lagged NFCI
glps = norminv(glr,0,1) ;       % lagged output growth

clear gr nr glr

%%  White Test - observed data

white_obs = zeros(1,2) ;                            % store test statistic and p-value here
y = gm ;                                            % dependent variable ABG regression
x = [ones(T,1) nm glm ] ;                           % rhs variables ABG regression
h = ols(y,x) ;                                      % ABG regression
u = h.resid ;                                       % residuals
u2 = u.^2 ;                                         % squared residuals
xhet = [ ones(T,1) nps] ;                           % rhs for het equation
hhet = ols(u2,xhet) ;                               % het regression
white_obs(1) = T*hhet.rsqr ;                        % statistic 
white_obs(2) = 1 - chi2cdf(white_obs(1),1) ;        % p-value   
clear y x h u2 xhet hhet 

%% Machado-Santos Silva Test - observed data

tau = [ 0.05 0.10 0.33 0.50 0.67 0.90 0.95 ] ;      % QR quantiles
mss_obs = zeros(7,2) ;                              % store test statistics and p-values here
y = gm ;                                            % dependent variable ABG regression 
x = [ ones(T,1) nm glm ] ;                          % rhs variables ABG regression
for q = 1:7 ;                                       % begin do loop through quantiles
    b = rq(x,y,tau(q)) ;                            % QR at current quantile
    u = y - x*b ;                                   % residuals  
    Ineg = zeros(T,1) ;                             % vector for indicator variable     
    Ineg(u < 0) = 1 ;                               % indicator for negative residuals
    rho = u.*(tau(q)-Ineg) ;                        % check function (u)                 
    h = ols(rho,[ones(T,1) nps]) ;                  % estimate het
    mss_obs(q,1) = T*h.rsqr ;                       % chi-squared stat
    mss_obs(q,2) = 1 - chi2cdf(mss_obs(q,1),1) ;    % p-value
    clear b u Ineg rho h 
end ;
clear y x q
mss_obs = [tau' mss_obs ] ;


%%  White Test - pseudo data

white_ps = zeros(1,2) ;                             % store test statistic and p-value here
y = gps ;                                           % dependent variable ABG regression
x = [ones(T,1) nps glps ] ;                         % rhs variables ABG regression
h = ols(y,x) ;                                      % ABG regression
u = h.resid ;                                       % residuals
u2 = u.^2 ;                                         % squared residuals
xhet = [ ones(T,1) nps] ;                           % rhs for het equation
hhet = ols(u2,xhet) ;                               % het regression
white_ps(1) = T*hhet.rsqr ;                         % statistic 
white_ps(2) = 1 - chi2cdf(white_ps(1),1) ;          % p-value   
clear y x h u2 xhet hhet 

%% Machado-Santos Silva Test

tau = [ 0.05 0.10 0.33 0.50 0.67 0.90 0.95 ] ;      % QR quantiles
mss_ps = zeros(7,2) ;                               % store test statistics and p-values here
y = gps ;                                           % dependent variable ABG regression 
x = [ ones(T,1) nps glps ] ;                        % rhs variables ABG regression
for q = 1:7 ;                                       % begin do loop through quantiles
    b = rq(x,y,tau(q)) ;                            % QR at current quantile
    u = y - x*b ;                                   % residuals  
    Ineg = zeros(T,1) ;                             % vector for indicator variable     
    Ineg(u < 0) = 1 ;                               % indicator for negative residuals
    rho = u.*(tau(q)-Ineg) ;                        % check function (u)                 
    h = ols(rho,[ones(T,1) nps]) ;                  % estimate het
    mss_ps(q,1) = T*h.rsqr ;                        % chi-squared stat
    mss_ps(q,2) = 1 - chi2cdf(mss_ps(q,1),1) ;      % p-value
    clear b u Ineg rho h 
end ;
clear y x q
mss_ps = [tau' mss_ps ] ;

%% Save objects to files to produce figures / tables

if g_flag == 1 && s_flag == 0                           % ABG's 1 dp growth measure & ABG's sample
    save('HET_1dp_abg.mat','white_obs','mss_obs','white_ps','mss_ps')
elseif g_flag == 1 && s_flag == 1 ;                     % ABG's 1 dp growth measure & Extended sample 
    save('HET_1dp_ext.mat','white_obs','mss_obs','white_ps','mss_ps')
elseif g_flag == 2 && s_flag == 0 ;                     % 1 qtr growth measure & ABG's sample
    save('HET_1qtr_abg.mat','white_obs','mss_obs','white_ps','mss_ps')
elseif g_flag == 2 && s_flag == 1 ;                     % 1 qtr growth measure & Extended sample 
    save('HET_1qtr_ext.mat','white_obs','mss_obs','white_ps','mss_ps')
elseif g_flag == 3 && s_flag == 0 ;                     % 4 qtr growth measure & ABG's sample
    save('HET_4qtr_abg.mat','white_obs','mss_obs','white_ps','mss_ps')
elseif g_flag == 3 && s_flag == 1 ;                     % 4 qtr growth measure & Extended sample 
    save('HET_4qtr_ext.mat','white_obs','mss_obs','white_ps','mss_ps')
end ;


