%% Block Bootstrap for QR coefficients confidence intervals -- Covid-dummy added

%% Pick various cases / parameters
clear all 
tic
Blength = 6 ;                           % Block length for bootstrap
nsam = 10000 ;                          % Number of bootstrap samples
rng(2)                                  % Seeds random number generator

%% Load Data 1973Q1 to 2020Q4 - see ABGRep_Data.m for details

load ABGRep

% Use GDPC1 based output growth series

g = g1GDP(2:end) ;          % output growth
gl = g1GDP(1:end-1) ;       % lagged
n = NFCI(1:end-1) ;         % (lagged) NFCI 
tt = QTR(2:end) ;           % dates for plots, start 1973Q2
T = length(tt) ;            % sample size

% Save Covid block

gcv = g(end-6:end) ;            
glcv = gl(end-6:end) ;
ncv = n(end-6:end) ;

% Remove Covid block

g = g(1:end-7) ;
gl = gl(1:end-7) ;
gn = n(1:end-7) ;

Tpc = length(g);                % sample size - pre covid

%% Create overlapping bootstrap blocks

% Matrices to store blocks of pre-covid data - each column to contain a block of Blength observations
gBB = NaN(Blength,Tpc-Blength+1) ;      % output growth 
nBB = NaN(Blength,Tpc-Blength+1) ;      % lagged ncfi 
glBB = NaN(Blength,Tpc-Blength+1) ;     % lagged output growth

% Create (Tpc - Blength) overlapping blocks from pre-covid data
for t = 1:Tpc-Blength+1 ;
    gBB(:,t) = g(t:t+Blength-1) ;       % output growth
    nBB(:,t) = n(t:t+Blength-1) ;       % lagged ncfi
    glBB(:,t) = gl(t:t+Blength-1) ;     % lagged output growth
end ;

numB = ceil(Tpc/Blength) ;              % number of blocks reqd for bootstrap sample of length T 

%% Draw bootstrap samples and estimate QR specs

% Store bootstrap estimates here
qOBSbs = NaN(4,7,nsam) ;                % Observed
qPSbs = NaN(4,7,nsam) ;                 % Pseudo

for j = 1:nsam ;
	draw_j = ceil(length(gBB)*rand(numB,1)) ;       % vector of locations for start of blocks for sample j
    gj = [] ;                                       % store bootstrap sample for output here
    nj = [] ;                                       % store bootstrap sample for ncfi here
    glj = [] ;                                      % store bootstrap sample for lagged output here
    cdj = [] ;                                      % store bootstrap sample for covid dummy here
    for b = 1:numB ;                                % begin drawing blocks for bootstrap sample
        gj = [ gj ; gBB(:,draw_j(b)) ] ;            % add next block for output growth
        nj = [ nj ; nBB(:,draw_j(b)) ] ;            % add next block for ncgi growth
        glj = [ glj ; glBB(:,draw_j(b)) ] ;         % add next block for lagged output growth
    end ;
    
    % Shorten series to match sample data (removes part of last block)
    gj = gj(1:Tpc) ;                                % output growth
    nj = nj(1:Tpc) ;                                % lagged nfci
    glj = glj(1:Tpc) ;                              % lagged output growth
    cdj = zeros(Tpc,1) ;                            % covid dummy
    
    % Add Covid observations to end of sample
    
    gj = [ gj ; gcv ] ;
    nj = [ nj ; ncv ] ;
    glj = [ glj ; glcv ] ;
    cdj = [ cdj ; ones(7,1) ] ;

    % Standardize 
    gmj = zscore(gj) ;                              % output growth
    nmj = zscore(nj) ;                              % nfci
    glmj = zscore(glj) ;                            % lagged output growth

    % Ranks
    grj = tiedrank(gmj)/(T+1) ;                     % output growth   
    nrj = tiedrank(nmj)/(T+1) ;                     % nfci
    glrj = tiedrank(glmj)/(T+1) ;                   % lagged output growth

    % Pass through inverse normal to get pseudo data
    gpsj = norminv(grj,0,1) ;                       % output growth
    npsj = norminv(nrj,0,1) ;                       % nfci
    glpsj = norminv(glrj,0,1) ;                     % lagged output growth

    % Std QR on obs data
    yOBSj = gmj ;                                   % dependent variable
    xOBSj = [ ones(T,1) nmj glmj cdj ] ;            % rhs variables
    q05OBSj = rq(xOBSj, yOBSj, 0.05) ;
    q10OBSj = rq(xOBSj, yOBSj, 0.10) ;
    q33OBSj = rq(xOBSj, yOBSj, 0.33) ;
    q50OBSj = rq(xOBSj, yOBSj, 0.50) ; 
    q67OBSj = rq(xOBSj, yOBSj, 0.67) ; 
    q90OBSj = rq(xOBSj, yOBSj, 0.90) ;
    q95OBSj = rq(xOBSj, yOBSj, 0.95) ;
    % Store estimates for bootstrap sample j
    qOBSbs(:,:,j) = [ q05OBSj q10OBSj q33OBSj q50OBSj q67OBSj q90OBSj q95OBSj ] ;
    clear yOBSj xOBSj q05OBSj q10OBSj q33OBSj q50OBSj q67OBSj q90OBSj q95OBSj 

    % QR on pseudo data
    yPSj = gpsj ;                                   % dependent variable
    xPSj = [ ones(T,1) npsj glpsj cdj ] ;           % rhs variables
    q05PSj = rq(xPSj, yPSj, 0.05) ;  
    q10PSj = rq(xPSj, yPSj, 0.10) ;       
    q33PSj = rq(xPSj, yPSj, 0.33) ;
    q50PSj = rq(xPSj, yPSj, 0.50) ; 
    q67PSj = rq(xPSj, yPSj, 0.67) ; 
    q90PSj = rq(xPSj, yPSj, 0.90) ;
    q95PSj = rq(xPSj, yPSj, 0.95) ;  
    % Store estimates for bootstrap sample j
    qPSbs(:,:,j) = [ q05PSj q10PSj q33PSj q50PSj q67PSj q90PSj q95PSj ] ;
    %clear yPSj xPSj q05PSj q10PSj q33PSj q50PSj q67PSj q90PSj q95PSj 
     
end ;
toc

% Caclulate percentiles from BB samples for each quantile

OBS_BB = NaN(7,5,4) ;
PS_BB = NaN(7,5,4) ;

for q = 1:7 ;
    
    % OBS QR, parameters for current quantile
    OBS1q = reshape(qOBSbs(1,q,:),nsam,1) ;         % intercept
    OBS2q = reshape(qOBSbs(2,q,:),nsam,1) ;         % lagged nfci
    OBS3q = reshape(qOBSbs(3,q,:),nsam,1) ;         % lagged output growth
    OBS4q = reshape(qOBSbs(4,q,:),nsam,1) ;         % covid dummy
    
    % QR on pseudo data, parameters for current quantile
    PS1q = reshape(qPSbs(1,q,:),nsam,1) ;           % intercept
    PS2q = reshape(qPSbs(2,q,:),nsam,1) ;           % lagged nfci
    PS3q = reshape(qPSbs(3,q,:),nsam,1) ;           % lagged output growth
    PS4q = reshape(qPSbs(4,q,:),nsam,1) ;           % covid dummy

    % Percentiles for OBS's QR - intercept
    OBS_BB(q,1,1) = prctile(OBS1q,5) ;
    OBS_BB(q,2,1) = prctile(OBS1q,33) ;
    OBS_BB(q,3,1) = prctile(OBS1q,50) ;
    OBS_BB(q,4,1) = prctile(OBS1q,67) ;
    OBS_BB(q,5,1) = prctile(OBS1q,95) ;

    % Percentiles for OBS's QR - nfci
    OBS_BB(q,1,2) = prctile(OBS2q,5) ;
    OBS_BB(q,2,2) = prctile(OBS2q,33) ;
    OBS_BB(q,3,2) = prctile(OBS2q,50) ;
    OBS_BB(q,4,2) = prctile(OBS2q,67) ;
    OBS_BB(q,5,2) = prctile(OBS2q,95) ;
    
    % Percentiles for OBS's QR - lagged output
    OBS_BB(q,1,3) = prctile(OBS3q,5) ;
    OBS_BB(q,2,3) = prctile(OBS3q,33) ;
    OBS_BB(q,3,3) = prctile(OBS3q,50) ;
    OBS_BB(q,4,3) = prctile(OBS3q,67) ;
    OBS_BB(q,5,3) = prctile(OBS3q,95) ;

    % Percentiles for OBS's QR - covid dummy
    OBS_BB(q,1,4) = prctile(OBS4q,5) ;
    OBS_BB(q,2,4) = prctile(OBS4q,33) ;
    OBS_BB(q,3,4) = prctile(OBS4q,50) ;
    OBS_BB(q,4,4) = prctile(OBS4q,67) ;
    OBS_BB(q,5,4) = prctile(OBS4q,95) ;

    % Percentiles for QR on pseudo data - intercept
    PS_BB(q,1,1) = prctile(PS1q,5) ;
    PS_BB(q,2,1) = prctile(PS1q,33) ;
    PS_BB(q,3,1) = prctile(PS1q,50) ;
    PS_BB(q,4,1) = prctile(PS1q,67) ;
    PS_BB(q,5,1) = prctile(PS1q,95) ;

    % Percentiles for QR on pseudo data - nfci
    PS_BB(q,1,2) = prctile(PS2q,5) ;
    PS_BB(q,2,2) = prctile(PS2q,33) ;
    PS_BB(q,3,2) = prctile(PS2q,50) ;
    PS_BB(q,4,2) = prctile(PS2q,67) ;
    PS_BB(q,5,2) = prctile(PS2q,95) ;

    % Percentiles for QR on pseudo data - lagged output
    PS_BB(q,1,3) = prctile(PS3q,5) ;
    PS_BB(q,2,3) = prctile(PS3q,33) ;
    PS_BB(q,3,3) = prctile(PS3q,50) ;
    PS_BB(q,4,3) = prctile(PS3q,67) ;
    PS_BB(q,5,3) = prctile(PS3q,95) ;

    % Percentiles for QR on pseudo data - covid dummy
    PS_BB(q,1,4) = prctile(PS4q,5) ;
    PS_BB(q,2,4) = prctile(PS4q,33) ;
    PS_BB(q,3,4) = prctile(PS4q,50) ;
    PS_BB(q,4,4) = prctile(PS4q,67) ;
    PS_BB(q,5,4) = prctile(PS4q,95) ;

end ;

%% Save objects to files to produce tables

save('BB_Dummy.mat','OBS_BB','PS_BB','qOBSbs','qPSbs')

