%% ABG replication -- Estimation with Dummy for Covid Pandemic (2020Q1-2021Q3)
% (1) Estimates ABG's QR on observed data amd produces QR lines
% (2) Estimates QR on pseudo data and produces QR line
% (3) Produce copula QR lines using ECDF fitted marginal
% (4) Writes output to file to produce tables / figures

clear all

%% Pick various cases / parameters

clear all 
f_flag = 0 ;                    % Set to 1 to produce figures and 0 to supress figures

%% Load Data 1973Q1 to 2021Q3 - see ABGRep_Data.m for details

load ABGRep

% Use GDPC1 based output growth series

g = g1GDP(2:end) ;              % output growth
gl = g1GDP(1:end-1) ;           % lagged
n = NFCI(1:end-1) ;             % (lagged) NFCI 
tt = QTR(2:end) ;               % dates, start 1973Q2
T = length(tt);                 % sample size

% Create dummy variable for Covid Pandemic
covid = [ zeros(187,1) ; ones(7,1) ] ;      % 187 obs 1973Q2-2019Q4 & 7 obs 2020Q1-2021Q3

clear ABGgGDP g1GDP g4GDP NFCI QTR

%% Standardized, rank and generate pseudo data as standard normal & fit marginal to output growth

% Standardized observed data for standard QR
gm = zscore(g) ;                % output growth
nm = zscore(n) ;                % lagged NFCI
glm = zscore(gl) ;              % lagged output growth

% Ranks
gr = tiedrank(g)/(T+1) ;        % output growth
nr = tiedrank(n)/(T+1) ;        % lagged NFCI
glr = tiedrank(gl)/(T+1) ;      % lagged output growth

% Standardized normal pseudo data
gps = norminv(gr,0,1) ;         % output growth
nps = norminv(nr,0,1) ;         % lagged NFCI
glps = norminv(glr,0,1) ;       % lagged output growth

clear gr nr glr

%% Std QR on (standardized) obs data, rep ABG with two RHS variables plus Covid dummy in regs

yObs = gm ;                             % dependent variable
xObs = [ ones(T,1) nm glm covid ] ;     % rhs variables

% q01 etc are vectors of parameters, qObs stores all vectors
q05Obs = rq(xObs, yObs, 0.05) ;
q10Obs = rq(xObs, yObs, 0.10) ;
q33Obs = rq(xObs, yObs, 0.33) ;
q50Obs = rq(xObs, yObs, 0.50) ; 
q67Obs = rq(xObs, yObs, 0.67) ; 
q90Obs = rq(xObs, yObs, 0.90) ;
q95Obs = rq(xObs, yObs, 0.95) ;
qObs = [ q05Obs q10Obs q33Obs q50Obs q67Obs q90Obs q95Obs ] ;

% QR lines on observed data, ie from parameters of QR to implied values of target variable using 
% expected value of glG (equals zero) to proxy third variable and covid = 1, pObs0 stores all QR lines
p05Obs = xObs(:,1:2)*q05Obs(1:2) ;
p10Obs = xObs(:,1:2)*q10Obs(1:2) ;
p33Obs = xObs(:,1:2)*q33Obs(1:2) ; 
p50Obs = xObs(:,1:2)*q50Obs(1:2) ;
p67Obs = xObs(:,1:2)*q67Obs(1:2) ;
p90Obs = xObs(:,1:2)*q90Obs(1:2) ;
p95Obs = xObs(:,1:2)*q95Obs(1:2) ;
pObs0 = [ p05Obs p10Obs p33Obs p50Obs p67Obs p90Obs p95Obs ] ;
clear p05Obs p10Obs p33Obs p50Obs p67Obs p90Obs p95Obs

% QR lines on observed data, ie from parameters of QR to implied values of target variable using 
% expected value of glG (equals zero) to proxy third variable and covid = 1, pObs1 stores all QR lines
p05Obs = xObs(:,1:2)*q05Obs(1:2) + q05Obs(4) ;
p10Obs = xObs(:,1:2)*q10Obs(1:2) + q10Obs(4) ;
p33Obs = xObs(:,1:2)*q33Obs(1:2) + q33Obs(4) ; 
p50Obs = xObs(:,1:2)*q50Obs(1:2) + q50Obs(4) ;
p67Obs = xObs(:,1:2)*q67Obs(1:2) + q67Obs(4) ;
p90Obs = xObs(:,1:2)*q90Obs(1:2) + q90Obs(4) ;
p95Obs = xObs(:,1:2)*q95Obs(1:2) + q95Obs(4) ;
pObs1 = [ p05Obs p10Obs p33Obs p50Obs p67Obs p90Obs p95Obs ] ;
clear p05Obs p10Obs p33Obs p50Obs p67Obs p90Obs p95Obs

% Plot figure

if f_flag == 1 ;
    figure('position', [100, 100, 850, 550]) ;
    plot(nm,pObs0,'LineWidth',2)
    hold
    scatter(nm(1:end-7),gm(1:end-7), 50, 'o','r','filled')
    scatter(nm(end-6:end),gm(end-6:end), 50, 'o','g','filled')
    legend('5th','10th','33rd','50th','67th','90th','95th','location','SW')
    title('QR Observed Data - Covid = 0')
    axis([-2 5 -5 2]) 
    grid on 
    box on
end ;


if f_flag == 1 ;
    figure('position', [100, 100, 850, 550]) ;
    plot(nm,pObs1,'LineWidth',2)
    hold
    scatter(nm(1:end-7),gm(1:end-7), 50, 'o','r','filled')
    scatter(nm(end-6:end),gm(end-6:end), 50, 'o','g','filled')
    legend('5th','10th','33rd','50th','67th','90th','95th','location','SW')
    title('QR Observed Data - Covid = 1')
    grid on 
    box on
end ;

clear yObs xObs q05Obs q10Obs q33Obs q50Obs q67Obs q90Obs q95Obs 
%% QR on pseudo data, with two RHS variables

yPs = gps ;                             % depenedent variable
xPs = [ ones(T,1) nps glps covid ] ;    % rhs variables

% q01 etc are vectors of parameters, qPs stores all vectors
q05Ps = rq(xPs, yPs, 0.05) ;  
q10Ps = rq(xPs, yPs, 0.10) ;       
q33Ps = rq(xPs, yPs, 0.33) ;
q50Ps = rq(xPs, yPs, 0.50) ; 
q67Ps = rq(xPs, yPs, 0.67) ; 
q90Ps = rq(xPs, yPs, 0.90) ;
q95Ps = rq(xPs, yPs, 0.95) ;  
qPs = [ q05Ps q10Ps q33Ps q50Ps q67Ps q90Ps q95Ps ] ;
clear p05Ps p10Ps p33Ps p50Ps p67Ps p90Ps p95Ps

% QR lines for pseudo data, ie from parameters of QR to implied values of target variable using
% expected value of glG equals zero to proxy third variable and covid = 0, pPs0 stores all QR lines
p05Ps = xPs(:,1:2)*q05Ps(1:2) ;
p10Ps = xPs(:,1:2)*q10Ps(1:2) ;
p33Ps = xPs(:,1:2)*q33Ps(1:2) ; 
p50Ps = xPs(:,1:2)*q50Ps(1:2) ; 
p67Ps = xPs(:,1:2)*q67Ps(1:2) ;
p90Ps = xPs(:,1:2)*q90Ps(1:2) ;
p95Ps = xPs(:,1:2)*q95Ps(1:2) ;
pPs0 = [ p05Ps p10Ps p33Ps p50Ps p67Ps p90Ps p95Ps ] ;
clear p05Ps p10Ps p33Ps p50Ps p67Ps p90Ps p95Ps

% QR lines on pseudo data, ie from parameters of QR to implied values of target variable using 
% expected value of glG (equals zero) to proxy third variable and covid = 1, pPs1 stores all QR lines
p05Ps = xPs(:,1:2)*q05Ps(1:2) + q05Ps(4) ;
p10Ps = xPs(:,1:2)*q10Ps(1:2) + q10Ps(4) ;
p33Ps = xPs(:,1:2)*q33Ps(1:2) + q33Ps(4) ; 
p50Ps = xPs(:,1:2)*q50Ps(1:2) + q50Ps(4) ;
p67Ps = xPs(:,1:2)*q67Ps(1:2) + q67Ps(4) ;
p90Ps = xPs(:,1:2)*q90Ps(1:2) + q90Ps(4) ;
p95Ps = xPs(:,1:2)*q95Ps(1:2) + q95Ps(4) ;
pPs1 = [ p05Ps p10Ps p33Ps p50Ps p67Ps p90Ps p95Ps ] ;
clear p05Ps p10Ps p33Ps p50Ps p67Ps p90Ps p95Ps

% Plot figure
if f_flag == 1 ;
    figure('position', [100, 100, 850, 550]) ;
    plot(nps,pPs0,'LineWidth',2)
    hold
    scatter(nps(1:end-7),gps(1:end-7), 50, 'o','b','filled')
    scatter(nps(end-6:end),gps(end-6:end), 50, 'o','g','filled')
    legend('5th','10th','33rd','50th','67th','90th','95th','location','SW')
    title('QR Pseudo Data - Covid = 0')
    grid on 
    box on
end ;

if f_flag == 1 ;
    figure('position', [100, 100, 850, 550]) ;
    plot(nps,pPs1,'LineWidth',2)
    hold
    scatter(nps(1:end-7),gps(1:end-7), 50, 'o','b','filled')
    scatter(nps(end-6:end),gps(end-6:end), 50, 'o','g','filled')
    legend('5th','10th','33rd','50th','67th','90th','95th','location','SW')
    title('QR Pseudo Data - Covid = 1')
    grid on 
    box on
end ;

clear yPs xPs q05Ps q10Ps q33Ps q67Ps q90Ps q95Ps

%% Save objects to file to produce figures / tables

save('EST_Dummy.mat','qObs','pObs0','pObs1','qPs','pPs0','pPs1')

