%% ABG replication -- Simulates from copula to produce swarm for NCFI and output growth

clear all

%% Pick various cases / parameters

clear all 
f_flag = 1 ;                    % Set to 1 to produce figures and 0 to supress figures
nsim = 10000 ;              	% Number of simulations
rng(26) ;                     	% Seed for random number generator

%% Load Data 1973Q1 to 2021Q3 - see ABGRep_Data.m for details

load ABGRep

g = g1GDP(2:end) ;          % output growth
gl = g1GDP(1:end-1) ;       % lagged
n = NFCI(1:end-1) ;         % (lagged) NFCI 
tt = QTR(2:end) ;           % dates for plots, start 1973Q2

T = length(tt);                 % sample size

clear ABGgGDP g1GDP g4GDP NFCI QTR

%% Standardized, rank and pseudo data as standard normal

% Standardized observed data for standard QR
gm = zscore(g) ;                % output growth
nm = zscore(n) ;                % lagged NFCI
glm = zscore(gl) ;              % lagged output growth

% Ranks
gr = tiedrank(g)/(T+1) ;        % output growth
nr = tiedrank(n)/(T+1) ;        % lagged NFCI
glr = tiedrank(gl)/(T+1) ;      % lagged output growth

% Standardized normal pseudo data
gps = norminv(gr,0,1) ;         % output growth
nps = norminv(nr,0,1) ;         % lagged NFCI
glps = norminv(glr,0,1) ;       % lagged output growth

clear gr nr glr

%% Copula QR estimation on pseudo data to get parameters for simulation

yPS = gps ;                             % depenedent variable
xPS = [ ones(T,1) nps glps ] ;          % rhs variables
q50PS = rq(xPS, yPS, 0.50) ;            % QR at median
p50PS = xPS(:,1:2)*q50PS(1:2) ;         % QR lines at median      
beta = q50PS(2) ;                       % save correlation for copula simulation as slope in pseu space of QR 50th percentile

% Adjust pseudo output series by median QR lines, standardize
gpsst = gps - p50PS ;                   % remove median QR line
gpss = zscore(gpsst) ;                  % standardize

% Estimate QR at 33rd and 67th using adjusted pseudo data for output and only pseudo NFCI as RHS variable
yPSA = gpss ;                           % dependent variable
xPSA =  [ones(T,1)  nps] ;              % rhs variables
q33PSA = rq(xPSA, yPSA, 0.33) ;         % QR estimates at 33rd
q67PSA = rq(xPSA, yPSA, 0.67) ;         % QR estimates at 67th
p33PSA = xPSA*q33PSA ;                  % QR line at 33rd
p67PSA = xPSA*q67PSA ;                  % QR line at 67th
std6733 = p67PSA - p33PSA ;             % 67th - 33rd percentile (roughly 1sd)
sd_adjust = mean(std6733)/std(gpss) ;   % Adjustment so sample average matches stdev of adjusted pseudo output series 
stdAdj= std6733/sd_adjust ;             % Adjusted standard deviation           

%clear yPS xPS q50PS p50PS gpsst gpss yPSA xPSA q33PSA q67PSA p33PSA p67PSA std6733 sd_adjust t q 

%% Simulate from copula model to generate unconditional swarm on obs scale

Rho = [ 1.00 beta ; beta 1.00 ] ;               % correlation matrix
reps = copularnd('Gaussian',Rho,nsim+5000) ;    % swarm with 5k burn in
v1 = reps(5001:nsim+5000,1) ;                   % output draws
v2 = reps(5001:nsim+5000,2) ;                   % lagged nfci draws

% Swarm to pseudo space
h = regress(stdAdj, [ones(T,1) nps]) ;          % Estimate HET in pseudo space
sw_nps = norminv(v2,0,std(nps)) ;               % Pseudo NFCI swarm
sw_gps = norminv(v1,0,h(1)+h(2)*sw_nps) ;       % Pseudo output swarm -- with HET

% Take ranks of pseudo swarm
rs_nps = tiedrank(sw_nps)/(length(sw_nps)+1);   % lagged ncfi  
rs_gps = tiedrank(sw_gps)/(length(sw_gps)+1);   % output growth
  
% Take swarm to observed scale 
sw_nm = ksdensity(nm,rs_gps,'function','icdf') ; 
sw_gm = ksdensity(gm,rs_nps,'function','icdf') ; 

if f_flag == 1 ;
    figure('position', [100, 100, 850, 550]) ;
    scatter_kde(sw_nps,sw_gps,'filled', 'MarkerSize', 25);
    hold
    scatter(nps,gps,25,'o','r','filled') 
    title('Copula Simulation - Pseudo Space')
    grid on 
    box on
    cb = colorbar();

    figure('position', [100, 100, 850, 550]) ;
    scatter_kde(sw_nm,sw_gm,'filled', 'MarkerSize', 25);
    hold
    scatter(nm,gm,25,'o','r','filled') 
    title('Copula Simulation - Observed Space')
    grid on 
    box on
    cb = colorbar();
end ;

clear beta q50ps Rho reps v1 v2 stdCHT h rs_nps rs_gps 

%% Save objects to files to produce figures / tables

save('CopSim.mat','sw_nps','sw_gps','sw_nm','sw_gm')

