/*
** This is the GAUSS code for computing the parametric test statistics for the GARCH-type models in the empirical studies of
** Chen(2006), A Unified Approach to Standardized-Residuals-Based Correlation Tests for GARCH-type models
*/

library pgraph,cml;
#include cml.ext;
graphset;

format /m1 /rdn 8,2;
output file=e:\Research\Unified_Qtests\Revision\JAE_Revise2\empirical\Parametric_Tests.result reset; 
m=5;
output off;

@******************************@
@  DEFINE STOCK INDEX RETURNS  @
@******************************@
/* The daily close price of S&P500 and NASDAQ: Jan 1990-Sep 2006 */  
load SP5D[4224,1]=e:\Research\Unified_Qtests\Revision\JAE_Revise2\Empirical\Data\SP5D.dat; @ S&P500 @
load NACD[4224,1]=e:\Research\Unified_Qtests\Revision\JAE_Revise2\Empirical\Data\NACD.dat; @ NASDAQ @
rSP5D=100*(ln(SP5D[2:4224])-ln(SP5D[1:4223]));
rNACD=100*(ln(NACD[2:4224])-ln(NACD[1:4223]));
/*
xy(seqa(1,1,rows(NACD)),NACD~SP5D);
xy(seqa(1,1,rows(NACD)-1),rNACD~rSP5D);
*/

DGP=1;
do while DGP<=2;
if DGP==1; y=rSP5D; yt="S&P500 "; endif;
if DGP==2; y=rNACD; yt="NASADQ "; endif;

model=1;
do while model<=6;

if DGP==1 and model==1; modelt="AR-GARCH    "; QMLE=0.047731246|0.012177275|0.0056414457|0.93779707|0.057130940; endif;
if DGP==1 and model==2; modelt="AR-EGARCH   "; QMLE=0.023518221|0.019292239|-0.092396999|0.98362883|-0.087671238|0.11416469; endif;
if DGP==1 and model==3; modelt="SETAR-GARCH "; QMLE=0.0086659284|-0.048381415|0.035396692|0.044631896|-0.038121226|0.0055721694|0.93833045|0.056642411; endif;
if DGP==1 and model==4; modelt="SETAR-EGARCH"; QMLE=-0.022379975|-0.050756029|0.017116895|0.046039388|-0.016263131|-0.090608475|0.98256197|-0.087644799|0.11079198; endif;
if DGP==1 and model==5; modelt="GARCH-M     "; QMLE=0.063361548|0.012061616|0.022875499|0.0057859346|0.93716055|0.057598563; endif;
if DGP==1 and model==6; modelt="EGARCH-M    "; QMLE=0.019372999|0.018930582|-0.0058715850|-0.091755736|0.98424406|-0.087838724|0.11371846; endif;

if DGP==2 and model==1; modelt=" AR-GARCH     "; QMLE=0.066010057|0.097485818|0.014505260|0.90832197|0.085118173; endif;
if DGP==2 and model==2; modelt=" AR-EGARCH    "; QMLE=0.035244751|0.10402000|-0.12440633|0.98706345|-0.063387797|0.16535316; endif;
if DGP==2 and model==3; modelt=" SETAR-GARCH  "; QMLE=-0.048331202|-0.010562317|0.11268812|0.081630101|0.037387113|0.013991855|0.90967479|0.084016283; endif;
if DGP==2 and model==4; modelt=" SETAR-EGARCH "; QMLE=-0.13859358|-0.049577148|0.081756423|0.088949858|-0.088717643|-0.12218367|0.98684477|-0.061699559|0.16146374; endif;
if DGP==2 and model==5; modelt=" GARCH-M      "; QMLE=0.076241680|0.098113637|0.042407755|0.014837837|0.90659755|0.086738801; endif;
if DGP==2 and model==6; modelt=" EGARCH-M     "; QMLE=0.039537870|0.10532255|0.015030952|-0.12527745|0.98653333|-0.062788095|0.16639469; endif;


output on;

if model==1; {mu,h,u,e,w,z}=WZ_AR_GARCH(y,QMLE); endif;
if model==2; {mu,h,u,e,w,z}=WZ_AR_EGARCH(y,QMLE); endif;
if model==3; {mu,h,u,e,w,z}=WZ_SETAR_GARCH(y,QMLE); endif;
if model==4; {mu,h,u,e,w,z}=WZ_SETAR_EGARCH(y,QMLE); endif;
if model==5; {mu,h,u,e,w,z}=WZ_AR_GARCH_M(y,QMLE); endif;
if model==6; {mu,h,u,e,w,z}=WZ_AR_EGARCH_M(y,QMLE); endif;

T=rows(y);
@*******************************@
@  THE PROPOSED TEST STATISTICS @
@*******************************@

/********************************/
/* THE SEPARATE TEST STATISTICS */
/********************************/
{corr11,Bart11,C11,Q11,Psi11}=Correlation_Test(e,w,z,m,1,1);
{corr12,Bart12,C12,Q12,Psi12}=Correlation_Test(e,w,z,m,1,2);
{corr21,Bart21,C21,Q21,Psi21}=Correlation_Test(e,w,z,m,2,1);
{corr22,Bart22,C22,Q22,Psi22}=Correlation_Test(e,w,z,m,2,2);
{corrss,Bartss,Css,Qss,Psiss}=Correlation_Test(e,w,z,m,3,3);
{corrsc,Bartsc,Csc,Qsc,Psisc}=Correlation_Test(e,w,z,m,3,4);
{corrcs,Bartcs,Ccs,Qcs,Psics}=Correlation_Test(e,w,z,m,4,3);
{corrcc,Bartcc,Ccc,Qcc,Psicc}=Correlation_Test(e,w,z,m,4,4);

/*******************************/
/* THE OMNIBUS TEST STATISTICS */
/*******************************/
PTStestK=zeros(m,1);
SCStestK=zeros(m,1);
k=1;
do while k<=m;
corrPTS=corr11[k]|corr12[k]|corr21[k]|corr22[k];
corrSCS=corrss[k]|corrsc[k]|corrcs[k]|corrcc[k];
PsiPTS=Psi11[.,k]~Psi12[.,k]~Psi21[.,k]~Psi22[.,k];
PsiSCS=Psiss[.,k]~Psisc[.,k]~Psics[.,k]~Psicc[.,k];
PTStestK[k]=T*corrPTS'inv(PsiPTS'PsiPTS./T)*corrPTS;
SCStestK[k]=T*corrSCS'inv(PsiSCS'PsiSCS./T)*corrSCS;
k=k+1;
endo;
corrPTS=corr11|corr12|corr21|corr22;
corrSCS=corrss|corrsc|corrcs|corrcc;
PsiPTSJ=Psi11~Psi12~Psi21~Psi22;
PsiSCSJ=Psiss~Psisc~Psics~Psicc;
PTStestJ=T*corrPTS'inv(PsiPTSJ'PsiPTSJ./T)*corrPTS;
SCStestJ=T*corrSCS'inv(PsiSCSJ'PsiSCSJ./T)*corrSCS;

@***************************@
@  EXISTING TEST STATISTICS @
@***************************@

/*********************************/
/* THE BOX-PIERCE TEST STATISTIC */
/*********************************/
{C11o,Q11o}=BoxPierce_McLeodLi(e,m,1,1);

/********************************/
/* THE MCLOED-LI TEST STATISTIC */
/********************************/
{C22o,Q22o}=BoxPierce_McLeodLi(e,m,2,2);

/*****************************/
/* THE LI-MAK TEST STATISTIC */
/*****************************/
{CLM,QLM}=Li_Mak(e,w,z,m);

/**************************/
/* THE BHK TEST STATISTIC */
/**************************/
{CBH,QBH}=BHK(e,z,m);

/*************************/
/* THE LM TEST STATISTIC */
/*************************/
{LM1,pv_LM1}=LM_Test(e,h,z,1); @ Lundergh and Terasvirta(2002): Volatility clustering @
{LM2,pv_LM2}=LM_Test(e,h,z,2); @ Lundergh and Terasvirta(2002): Nonlinearity @
{LM3,pv_LM3}=LM_Test(e,h,z,3); @ Engle and Ng(1993): Negative size-bias @
{LM4,pv_LM4}=LM_Test(e,h,z,4); @ Engle and Ng(1993): Joint @

print "----------------------------------------------";
datestr(0) "        " timestr(0);
print "----------------------------------------------";
print "DGP=" yt "Model=" modelt;
print "----------------------------------------------";
print " THE PROPOSED TESTS: k=1   k=3   k=5   m=5    ";
print "----------------------------------------------";
print " (11)" C11[1] " " C11[3] " " C11[5] " " Q11;
print " (12)" C12[1] " " C12[3] " " C12[5] " " Q12;
print " (21)" C21[1] " " C21[3] " " C21[5] " " Q21;
print " (22)" C22[1] " " C22[3] " " C22[5] " " Q22;
print "(PTS)" PTStestk[1]  " " PTStestk[3]  " " PTStestk[5]  " " PTStestJ;
print " ";
print " (ss)" Css[1] " " Css[3] " " Css[5] " " Qss;
print " (sc)" Csc[1] " " Csc[3] " " Csc[5] " " Qsc;
print " (cs)" Ccs[1] " " Ccs[3] " " Ccs[5] " " Qcs;
print " (cc)" Ccc[1] " " Ccc[3] " " Ccc[5] " " Qcc;
print "(SCS)" SCStestk[1]  " " SCStestk[3]  " " SCStestk[5]  " " SCStestJ;
print " ";
print "--------------------------------------------------------------";
print " THE EXISTING AUTOCORRELATION TESTS: k=1   k=3   k=5   m=5    ";
print "--------------------------------------------------------------";
print "  ";
print " (BP)" C11o[1] " " C11o[3] " " C11o[5] " " Q11o;
print " (ML)" C22o[1] " " C22o[3] " " C22o[5] " " Q22o;
print " (LM)" CLM[1] " " CLM[3] " " CLM[5] " " QLM;
print " (BH)" CBH[1] " " CBH[3] " " CBH[5] " " QBH;
print "  ";
print "--------------------------------------------------------------";
print " THE EXISTING LM TESTS                                        ";
print "--------------------------------------------------------------";
print "(V)  " LM1;
print "(N)  " LM2;
print "(NSB)" LM3;
print "(JB) " LM4;
print "----------------------------------------------";


model=model+1;
endo;

DGP=DGP+1;
endo;


/*
** This GAUSS code computes the conditional mean and variance derivatives of the EGARCH-type model.
**
** Command:
** {MD,var}=var_alpha(e,w,z); 
** 
** Inputs:
** 1) e: standardized residuals
** 2) w: the matrix of "standardized" conditional mean derivatives (\nabla_{\theta}m_t)/h_t^{1/2}
** 3) z: the matrix of "standardized" conditional variance derivatives (\nabla_{\theta}z_t)/h_t
**
** Outputs:
** 1)  MD: the martingale difference series
** 2) var: the asymptotic variance-covariance matrix estimator 
**
** Written by 
** Yi-Ting Chen, Academia Sinica
** November 10, 2004
*/

proc(2)=var_alpha(e,w,z);
local T,MD,var;
T=rows(e);
MD=(w.*e+0.5.*z.*(e.^2-1))*inv((w'w)./T+0.5.*(z'z)./T);
var=(MD'MD)./T;     @ This is the asymptotic variance estimator  of $\sqrt{T}(\hat{\alpha}_T-\alpha_o)$ @
retp(MD,var./T);    @ The reported asymptotic variance estimator is that of $\hat{\alpha}_T-\alpha_o$ @
endp;




@*************************************************@
@  THE CONDITIONAL MEAN AND VARIANCE DERIVATIVES  @
@*************************************************@
/**********************************/
/* I. THE AR(1)-GARCH(1,1) MODEL  */
/**********************************/
proc(6)=WZ_AR_GARCH(y,QMLE);
local T,p,r0,r1,d0,d1,d2,m,h,u,v,w,z,i,h_coef;
T=rows(y);
p=5;   
r0=QMLE[1];
r1=QMLE[2];
d0=QMLE[3];
d1=QMLE[4];
d2=QMLE[5];
m=zeros(T,1);
h=zeros(T,1);
u=zeros(T,1);
v=zeros(T,1);
w=zeros(T,p);
z=zeros(T,p);
w[.,1]=ones(T,1);
m[1]=meanc(y);
u[1]=y[1]-m[1];
h[1]=stdc(y)^2;
v[1]=u[1]/h[1]^0.5;
i=2;
do while i<=T;
m[i]=r0+r1*y[i-1];
u[i]=y[i]-m[i];
h[i]=d0+d1*h[i-1]+d2*u[i-1]^2;
v[i]=u[i]/h[i]^0.5;
w[i,2]=y[i-1];
z[i,1]=-2*d2*u[i-1]*w[i-1,1]+d1*z[i-1,1];
z[i,2]=-2*d2*u[i-1]*w[i-1,2]+d1*z[i-1,2];
z[i,3]=1+d1*z[i-1,3];
z[i,4]=h[i-1]+d1*z[i-1,4];
z[i,5]=u[i-1]^2+d1*z[i-1,5];
i=i+1;
endo;
w=w./(h.^0.5);
z=z./h;
retp(m,h,u,v,w,z);
endp;

/************************************/
/* II. THE AR(1)-EGARCH(1,1) MODEL  */
/************************************/
proc(6)=WZ_AR_EGARCH(y,QMLE);
local T,p,r0,r1,d0,d1,d2,d3,m,h,u,av,s,v,w,z,i,h_coef,lambda;
lambda=10^(-3);
T=rows(y);
p=6;   
r0=QMLE[1];
r1=QMLE[2];
d0=QMLE[3];
d1=QMLE[4];
d2=QMLE[5];
d3=QMLE[6];
m=zeros(T,1);
h=zeros(T,1);
u=zeros(T,1);
v=zeros(T,1);
w=zeros(T,p);
z=zeros(T,p);
av=zeros(T,1);
w[.,1]=ones(T,1);
m[1]=meanc(y);
u[1]=y[1]-m[1];
h[1]=stdc(y)^2;
v[1]=u[1]/h[1]^0.5;
av[1]=(lambda^2+v[1]^2)^0.5;
i=2;
do while i<=T;
m[i]=r0+r1*y[i-1];
u[i]=y[i]-m[i];
h[i]=exp(d0+d1*ln(h[i-1])+d2*v[i-1]+d3*av[i-1]);
v[i]=u[i]/h[i]^0.5;
av[i]=(lambda^2+v[i]^2)^0.5;
w[i,2]=y[i-1];
s=v[i-1]/av[i-1];
z[i,1]=-(d2+d3*s)*w[i-1,1]/h[i-1]^0.5+(d1-0.5*(d2+d3*s).*v[i-1])*z[i-1,1];
z[i,2]=-(d2+d3*s)*w[i-1,2]/h[i-1]^0.5+(d1-0.5*(d2+d3*s).*v[i-1])*z[i-1,2];
z[i,3]=1+(d1-0.5*(d2+d3*s).*v[i-1])*z[i-1,3];
z[i,4]=ln(h[i-1])+(d1-0.5*(d2+d3*s).*v[i-1])*z[i-1,4];
z[i,5]=v[i-1]+(d1-0.5*(d2+d3*s).*v[i-1])*z[i-1,5];
z[i,6]=av[i-1]+(d1-0.5*(d2+d3*s).*v[i-1])*z[i-1,6];
i=i+1;
endo;
retp(m,h,u,v,w./(h.^0.5),z);
endp;

/***************************************/
/* III. THE SETAR(1)-GARCH(1,1) MODEL  */
/***************************************/
proc(6)=WZ_SETAR_GARCH(y,QMLE);
local T,p,r01,r11,r02,r12,r3,d0,d1,d2,m,h,u,v,w,z,i,h_coef,Tran,lambda;
T=rows(y);
p=8;   
r01=QMLE[1];
r11=QMLE[2];
r02=QMLE[3];
r12=QMLE[4];
r3=QMLE[5];
d0=QMLE[6];
d1=QMLE[7];
d2=QMLE[8];
m=zeros(T,1);
h=zeros(T,1);
u=zeros(T,1);
v=zeros(T,1);
w=zeros(T,p);
z=zeros(T,p);
Tran=zeros(T,1);
m[1]=meanc(y);
u[1]=y[1]-m[1];
h[1]=stdc(y)^2;
v[1]=u[1]/h[1]^0.5;
lambda=1/1000;
i=2;
do while i<=T;
Tran[i]=1/(1+exp((y[i-1]-r3)/lambda));
m[i]=Tran[i].*(r01+r11*y[i-1])+(1-Tran[i]).*(r02+r12*y[i-1]);
u[i]=y[i]-m[i];
h[i]=d0+d1*h[i-1]+d2*u[i-1]^2;
v[i]=u[i]/h[i]^0.5;
w[i,1]=Tran[i];
w[i,2]=Tran[i]*y[i-1];
w[i,3]=1-Tran[i];
w[i,4]=(1-Tran[i])*y[i-1];
w[i,5]=(r01-r02+r11*y[i-1]-r12*y[i-1])*(1/lambda)/(exp(-(y[i-1]-r3)/lambda)+2+exp((y[i-1]-r3)/lambda));
z[i,1]=-2*d2*u[i-1]*w[i-1,1]+d1*z[i-1,1];
z[i,2]=-2*d2*u[i-1]*w[i-1,2]+d1*z[i-1,2];
z[i,3]=-2*d2*u[i-1]*w[i-1,3]+d1*z[i-1,3];
z[i,4]=-2*d2*u[i-1]*w[i-1,4]+d1*z[i-1,4];
z[i,5]=-2*d2*u[i-1]*w[i-1,5]+d1*z[i-1,5];
z[i,6]=1+d1*z[i-1,6];
z[i,7]=h[i-1]+d1*z[i-1,7];
z[i,8]=u[i-1]^2+d1*z[i-1,8];
i=i+1;
endo;
w=w./(h.^0.5);
z=z./h;
retp(m,h,u,v,w,z);
endp;

/***************************************/
/* IV. THE SETAR(1)-EGARCH(1,1) MODEL  */
/***************************************/
proc(6)=WZ_SETAR_EGARCH(y,QMLE);
local T,p,r01,r11,r02,r12,r3,d0,d1,d2,d3,m,h,u,av,avp,v,w,z,i,h_coef,lambda,Tran,s;
lambda=10^(-3);
T=rows(y);
p=9;   
r01=QMLE[1];
r11=QMLE[2];
r02=QMLE[3];
r12=QMLE[4];
r3=QMLE[5];
d0=QMLE[6];
d1=QMLE[7];
d2=QMLE[8];
d3=QMLE[9];
m=zeros(T,1);
h=zeros(T,1);
u=zeros(T,1);
v=zeros(T,1);
Tran=zeros(T,1);
w=zeros(T,p);
z=zeros(T,p);
av=zeros(T,1);
m[1]=meanc(y);
u[1]=y[1]-m[1];
h[1]=stdc(y)^2;
v[1]=u[1]/h[1]^0.5;
av[1]=(lambda^2+v[1]^2)^0.5;
i=2;
do while i<=T;
Tran[i]=1/(1+exp((y[i-1]-r3)/lambda));
m[i]=Tran[i].*(r01+r11*y[i-1])+(1-Tran[i]).*(r02+r12*y[i-1]);
u[i]=y[i]-m[i];
h[i]=exp(d0+d1*ln(h[i-1])+d2*v[i-1]+d3*av[i-1]);
v[i]=u[i]/h[i]^0.5;
av[i]=(lambda^2+v[i]^2)^0.5;
s=v[i-1]/av[i-1];
w[i,1]=Tran[i];
w[i,2]=Tran[i]*y[i-1];
w[i,3]=1-Tran[i];
w[i,4]=(1-Tran[i])*y[i-1];
w[i,5]=(r01-r02+r11*y[i-1]-r12*y[i-1])*(1/lambda)/(exp(-(y[i-1]-r3)/lambda)+2+exp((y[i-1]-r3)/lambda));
z[i,1]=-(d2+d3*s)*w[i-1,1]/h[i-1]^0.5+(d1-0.5*(d2+d3*s).*v[i-1])*z[i-1,1];
z[i,2]=-(d2+d3*s)*w[i-1,2]/h[i-1]^0.5+(d1-0.5*(d2+d3*s).*v[i-1])*z[i-1,2];
z[i,3]=-(d2+d3*s)*w[i-1,3]/h[i-1]^0.5+(d1-0.5*(d2+d3*s).*v[i-1])*z[i-1,3];
z[i,4]=-(d2+d3*s)*w[i-1,4]/h[i-1]^0.5+(d1-0.5*(d2+d3*s).*v[i-1])*z[i-1,4];
z[i,5]=-(d2+d3*s)*w[i-1,5]/h[i-1]^0.5+(d1-0.5*(d2+d3*s).*v[i-1])*z[i-1,5];
z[i,6]=1+(d1-0.5*(d2+d3*s).*v[i-1])*z[i-1,6];
z[i,7]=ln(h[i-1])+(d1-0.5*(d2+d3*s).*v[i-1])*z[i-1,7];
z[i,8]=v[i-1]+(d1-0.5*(d2+d3*s).*v[i-1])*z[i-1,8];
z[i,9]=av[i-1]+(d1-0.5*(d2+d3*s).*v[i-1])*z[i-1,9];
i=i+1;
endo;
retp(m,h,u,v,w./(h.^0.5),z);
endp;


/******************************/
/* V. THE GARCH(1,1)-M MODEL  */
/******************************/
proc(6)=WZ_AR_GARCH_M(y,QMLE);
local T,p,r0,r1,r2,d0,d1,d2,m,h,u,v,w,z,i,h_coef;
T=rows(y);
p=6;   
r0=QMLE[1];
r1=QMLE[2];
r2=QMLE[3];
d0=QMLE[4];
d1=QMLE[5];
d2=QMLE[6];
m=zeros(T,1);
h=zeros(T,1);
u=zeros(T,1);
v=zeros(T,1);
w=ones(T,p);
z=ones(T,p);
m[1]=meanc(y);
u[1]=y[1]-m[1];
h[1]=stdc(y)^2;
v[1]=u[1]/h[1]^0.5;
i=2;
do while i<=T;
h[i]=d0+d1*h[i-1]+d2*u[i-1]^2;
m[i]=r0+r1*y[i-1]+r2*ln(h[i]);
u[i]=y[i]-m[i];
v[i]=u[i]/h[i]^0.5;
z[i,1]=-2*d2*u[i-1]*w[i-1,1]+d1*z[i-1,1];
z[i,2]=-2*d2*u[i-1]*w[i-1,2]+d1*z[i-1,2];
z[i,3]=-2*d2*u[i-1]*w[i-1,3]+d1*z[i-1,3];
z[i,4]=-2*d2*u[i-1]*w[i-1,4]+d1*z[i-1,4];
z[i,5]=-2*d2*u[i-1]*w[i-1,5]+h[i-1]+d1*z[i-1,5];
z[i,6]=-2*d2*u[i-1]*w[i-1,6]+u[i-1]^2+d1*z[i-1,6];
w[i,1]=1+r2*z[i,1];
w[i,2]=y[i-1]+r2*z[i,2];
w[i,3]=ln(h[i])+r2*z[i,3];
w[i,4]=r2*z[i,4];
w[i,5]=r2*z[i,5];
w[i,6]=r2*z[i,6];
i=i+1;
endo;
w=w./(h.^0.5);
z=z./h;
retp(m,h,u,v,w,z);
endp;



/********************************/
/* VI. THE EGARCH(1,1)-M MODEL  */
/********************************/
proc(6)=WZ_AR_EGARCH_M(y,QMLE);
local T,p,r0,r1,r2,d0,d1,d2,d3,m,h,u,av,s,v,w,z,i,h_coef,lambda;
lambda=10^(-3);
T=rows(y);
p=7;   
r0=QMLE[1];
r1=QMLE[2];
r2=QMLE[3];
d0=QMLE[4];
d1=QMLE[5];
d2=QMLE[6];
d3=QMLE[7];
m=zeros(T,1);
h=zeros(T,1);
u=zeros(T,1);
v=zeros(T,1);
w=ones(T,p);
z=ones(T,p);
av=zeros(T,1);
w[.,1]=ones(T,1);
m[1]=meanc(y);
u[1]=y[1]-m[1];
h[1]=stdc(y)^2;
v[1]=u[1]/h[1]^0.5;
av[1]=(lambda^2+v[1]^2)^0.5;
i=2;
do while i<=T;
m[i]=r0+r1*y[i-1];
u[i]=y[i]-m[i];
h[i]=exp(d0+d1*ln(h[i-1])+d2*v[i-1]+d3*av[i-1]);
v[i]=u[i]/h[i]^0.5;
av[i]=(lambda^2+v[i]^2)^0.5;
s=v[i-1]/av[i-1];
z[i,1]=-(d2+d3*s)/h[i-1]^0.5+(d1-0.5*(d2+d3*s).*v[i-1])*z[i-1,1];
z[i,2]=-(d2+d3*s)*w[i-1,2]/h[i-1]^0.5+(d1-0.5*(d2+d3*s).*v[i-1])*z[i-1,2];
z[i,3]=-(d2+d3*s)*w[i-1,3]/h[i-1]^0.5+(d1-0.5*(d2+d3*s).*v[i-1])*z[i-1,3];
z[i,4]=1+(d1-0.5*(d2+d3*s).*v[i-1])*z[i-1,4];
z[i,5]=ln(h[i-1])+(d1-0.5*(d2+d3*s).*v[i-1])*z[i-1,5];
z[i,6]=v[i-1]+(d1-0.5*(d2+d3*s).*v[i-1])*z[i-1,6];
z[i,7]=av[i-1]+(d1-0.5*(d2+d3*s).*v[i-1])*z[i-1,7];
w[i,1]=1+r2*z[i,1];
w[i,2]=y[i-1]+r2*z[i,2];
w[i,3]=ln(h[i])+r2*z[i,3];
w[i,4]=r2*z[i,4];
w[i,5]=r2*z[i,5];
w[i,6]=r2*z[i,6];
w[i,7]=r2*z[i,7];
i=i+1;
endo;
retp(m,h,u,v,w./(h.^0.5),z);
endp;


@******************************@
@ THE PROPOSED TEST STATISTICS @
@******************************@
/*
** This is the GAUSS code for the PTS and SCS correlation test statistics.
** Reference:
** Chen(2006), "A Unified Approach to Standardized-Residuals-Based Correlation Tests for GARCH-type Models",
** submitted to the Journal of Applied Econometrics
**
** INPUTS: 
**
** 1. v: a T x 1 vector, the standardized residuals of the model being tested
** 2. w: a T x p matrix, the standardized conditional mean derivatives
** 3. z: a T x p matrix, the standardized conditional variance derivatives 
** 4. m: an integer, the maximum lag 
** 5. a: an integer, (a,b): (1,1)=>Q11, (1,2)=>Q12, (2,1)=>Q21, (2,2)=>Q22 
** 6. b: an integer, (a,b): (3,3)=>Qss, (3,4)=>Qsc, (4,3)=>Qcs, (4,4)=>Qcc
**
** OUTPUTS:
**
** 1. corr:     a m x 1 matrix, sample correlations
** 2. Bart:     a m x 1 matrix, the adjusted Bartlett confidence interval
** 3. C_stat:   a m x 1 vector, rho_ab[k]-based individual test statistic for k=1,2,...,m 
** 4. Q_stat:   a scalar, portmanteau test statistic
** 5. Psi:      the martingale-difference transformation
**
** COMMAND:
**
** {corr,Bart,C_stat,Q_stat,Psi}=Correlation_Test(v,w,z,m,a,b);
**
** Written by
** Yi-Ting Chen, Academia Sinica.
** October 5, 2006.
**
*/
proc(5)=Correlation_Test(v,w,z,m,a,b);
local T,f,g,fp,sf,sg,sfp,corr,Lambda,Upsilon,C_stat,C_var,Bartlett,var,wzv,i,Psi,k,Lambda_k,Psi_k,l,Lambda_l,Psi_l,Q_stat;
T=rows(v);   
f=v.^a;               
g=v.^b;    
fp=a*(v.^(a-1));
if a==3 and b==3; 
f=sin(v);               
g=sin(v);               
fp=cos(v);
endif;
if a==3 and b==4; 
f=sin(v);               
g=cos(v);               
fp=cos(v);
endif;
if a==4 and b==3; 
f=cos(v);               
g=sin(v);               
fp=-sin(v);
endif;
if a==4 and b==4; 
f=cos(v);               
g=cos(v);               
fp=-sin(v);
endif;
sf=(f-meanc(f))./stdc(f);
sg=(g-meanc(g))./stdc(g);
sfp=fp./stdc(f);
corr=zeros(m,1);
Lambda=zeros(m,cols(w));
Upsilon=(w'w)./T+(0.5).*(z'z)./T;
C_stat=zeros(m,1);
C_var=zeros(m,1);
Bartlett=zeros(m,1);
var=zeros(m,m);
wzv=zeros(T,cols(w));
i=1;
do while i<=T;
wzv[i,.]=w[i,.].*v[i]+0.5.*z[i,.].*(v[i]^2-1);
i=i+1;
endo;
Psi=zeros(T,m);
k=1;
do while k<=m;
corr[k]=(sf[k+1:T]'sg[1:T-k])./(T-k);
Lambda_k=meanc(sfp).*(sg[1:T-k]'w[k+1:T,.])./T+0.5*meanc(sfp.*v).*(sg[1:T-k]'z[k+1:T,.])./T;
Psi_k=sf[k+1:T].*sg[1:T-k]-wzv[k+1:T,.]*inv(Upsilon)*Lambda_k';
var[k,k]=Psi_k'Psi_k./(T-k);
Psi[k+1:T,k]=Psi_k;
l=k+1;
do while l<=m;
Lambda_l=meanc(sfp).*(sg[1:T-l]'w[l+1:T,.])./T+0.5*meanc(sfp.*v).*(sg[1:T-l]'z[l+1:T,.])./T;
Psi_l=sf[l+1:T].*sg[1:T-l]-wzv[1:T-l,.]*inv(Upsilon)*Lambda_l';
Psi_k=Psi_k[1:T-l];
var[k,l]=Psi_k'Psi_l./(T-l);
var[l,k]=var[k,l];
l=l+1;
endo;
Bartlett[k]=(var[k,k]^0.5)/(T^0.5);
C_stat[k]=(T-k)*corr[k]*inv(var[k,k])*corr[k];
k=k+1;
endo;
Q_stat=T*corr'inv(var)*corr;
Psi=Psi[m+1:T,.];
retp(corr,Bartlett.*1.96,C_stat,Q_stat,Psi);
endp;

@**********************************************@
@ THE BOX-PIERCE and MCLEOD-LI TEST STATISTICS @
@**********************************************@
proc(2)=BoxPierce_McLeodLi(v,m,a,b);
local T,f,g,sf,sg,corr,C_stat,k,Q_stat;
T=rows(v);   
f=v.^a;               
g=v.^b;    
sf=(f-meanc(f))./stdc(f);
sg=(g-meanc(g))./stdc(g);
corr=zeros(m,1);
C_stat=zeros(m,1);
k=1;
do while k<=m;
corr[k]=(sf[k+1:T]'sg[1:T-k])./(T-k);
C_stat[k]=T*(corr[k]^2);
k=k+1;
endo;
Q_stat=T*sumc(corr.^2);
retp(C_stat,Q_stat);
endp;

@***************************@
@ THE LI-MAK TEST STATISTIC @
@***************************@
proc(2)=Li_Mak(v,w,z,m);
local T,f,g,sf,sg,zeta_am1,zeta_a,zeta_ap1,zeta_ap2,zeta_2a,zeta_b,zeta_2b,zeta_3,zeta_4,
      Gamm,GammLB,Lambda,Delta,adj_A,adj_B,C_var,C_stat,Upsilon,Omega,k,l,Lam_wk,Lam_zk,Q_var,Q_stat,GammT;

T=rows(v);                 @ sample size @
f=v.^2;                    @ transformations of v @                
g=v.^2;   
sf=(f-meanc(f))./stdc(f);  @ standardization of f @
sg=(g-meanc(g))./stdc(g);  @ standardization of g @
Gamm=zeros(m,1);
Lambda=zeros(m,cols(z));
Delta=zeros(m,cols(z));
adj_A=zeros(m,m);
C_var=zeros(m,1);
C_stat=zeros(m,1);
Upsilon=(w'w)./T+(0.5).*(z'z)./T;
k=1;
do while k<=m;
Gamm[k]=(sf[k+1:T]'sg[1:T-k])./T; @ the lag-k Q_power based on f and g @
Lambda[k,.]=((v[1:T-k].^2-1)'z[k+1:T,.])./T;
adj_A[k,k]=-0.25.*Lambda[k,.]*inv(Upsilon)*Lambda[k,.]';
C_var[k]=1+adj_A[k,k];
C_stat[k]=(T-k)*Gamm[k]*inv(C_var[k])*Gamm[k];
k=k+1;
endo;
k=1;
do while k<=m-1;
l=k+1;
do while l<=m;
adj_A[k,l]=-0.25.*Lambda[k,.]*inv(Upsilon)*Lambda[l,.]';
adj_A[l,k]=adj_A[k,l];
l=l+1;
endo;
k=k+1;
endo;
Q_var=eye(m)+adj_A;
Q_stat=T*(Gamm'inv(Q_var)*Gamm);
retp(C_stat,Q_stat);
endp;

@************************@
@ THE BHK TEST STATISTIC @
@************************@
proc(2)=BHK(v,z,m);
local T,f,g,sf,sg,zeta_am1,zeta_a,zeta_ap1,zeta_ap2,zeta_2a,zeta_b,zeta_2b,zeta_3,zeta_4,
      Gamm,GammLB,Lambda,Delta,adj_A,adj_B,C_var,C_stat,Upsilon,Omega,k,l,Lam_wk,Lam_zk,Q_var,Q_stat,GammT;
T=rows(v);                 @ sample size @
f=v.^2;                    @ transformations of v @                
g=v.^2;   
sf=(f-meanc(f))./stdc(f);  @ standardization of f @
sg=(g-meanc(g))./stdc(g);  @ standardization of g @
Gamm=zeros(m,1);
Lambda=zeros(m,cols(z));
Delta=zeros(m,cols(z));
adj_A=zeros(m,m);
C_var=zeros(m,1);
C_stat=zeros(m,1);
Upsilon=(z'z)./T;
k=1;
do while k<=m;
Gamm[k]=(sf[k+1:T]'sg[1:T-k])./T; @ the lag-k Q_power based on f and g @
Lambda[k,.]=((v[1:T-k].^2-1)'z[k+1:T,.])./T;
adj_A[k,k]=-(1/(meanc(v.^4)-1)).*Lambda[k,.]*inv(Upsilon)*Lambda[k,.]';
C_var[k]=1+adj_A[k,k];
C_stat[k]=(T-k)*Gamm[k]*inv(C_var[k])*Gamm[k];
k=k+1;
endo;
k=1;
do while k<=m-1;
l=k+1;
do while l<=m;
adj_A[k,l]=-(1/(meanc(v.^4)-1)).*Lambda[k,.]*inv(Upsilon)*Lambda[l,.]';
adj_A[l,k]=adj_A[k,l];
l=l+1;
endo;
k=k+1;
endo;
Q_var=eye(m)+adj_A;
Q_stat=T*(Gamm'inv(Q_var)*Gamm);
retp(C_stat,Q_stat);
endp;

@***********************@
@ THE LM TEST STATISTIC @
@***********************@
proc(2)=LM_Test(v,h,z,eta_type);
local T,v2,v3,u,u3,m,eta,i,dum,p,q,sr,var,LM,p_v;
T=rows(v);   
v2=v.^2;
v3=v.^3;
u=v.*(h.^0.5);
u3=u.^3;
dum=dummy(u,0);
dum=dum[.,1];
if eta_type==1;   @ Lundergh and Terasvirta(2002): Volatility clustering @
m=5; 
eta=zeros(T,m);
i=m+1;
do while i<=T;
eta[i,.]=v2[i-m:i-1]';
i=i+1;
endo;
eta=eta[m+1:T,.];
endif;
if eta_type==2;   @ Lundergh and Terasvirta(2002): Nonlinearity @
m=1; 
eta=zeros(T,2*m);
i=m+1;
do while i<=T;
eta[i,.]=u[i-m:i-1]'~u3[i-m:i-1]';
i=i+1;
endo;
eta=eta[m+1:T,.];
endif;
if eta_type==3;   @ Engle and Ng(1993): Negative size-bias @
eta=u[1:T-1].*dum[1:T-1];
endif;
if eta_type==4;   @ Engle and Ng(1993): Joint size-bias @
eta=dum[1:T-1]~u[1:T-1].*dum[1:T-1]~u[1:T-1].*(1-dum[1:T-1]);
endif;
p=rows(eta);
q=cols(eta);
sr=v2[T-p+1:T]-1;
z=z[T-p+1:T,.];
var=(sr'sr./T).*(eta'eta-(eta'z)*inv(z'z)*(z'eta))./T;
LM=T*(eta'sr./T)'inv(var)*(eta'sr./T);
p_v=cdfchic(LM,q);
retp(LM,p_v);
endp;
@+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++@
